////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of HE trigger based calculator module.
//                    Compute the PSDU length of the HE-TB frame from the
//                    paramters received in the trigger frame.
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module tbCalculator( 
   //$port_g Clock and reset
   input  wire          clk,                     // Clock
   input  wire          hardRst_n,               // Hardware Reset
   input  wire          softRst_n,               // Software Reset

   //$port_g Control interface
   input  wire          psduLenStart_p,          // BSDU Length Computation Start pulse
   output wire          psduLenValid_p,          // PSDU Length Valid pulse
   output wire          psduLenValid,            // PSDU Length Valid flag
   output reg    [19:0] psduLen,                 // PSDU Length

   //$port_g Parameters
   input  wire   [11:0] rxTriggerULLength,
   input  wire    [1:0] rxTriggerGIAndLTFType,
   input  wire    [2:0] rxTriggerNumHELTFAndMidPeriod,
   input  wire          rxTriggerULSTBC,
   input  wire    [2:0] rxTriggerULPacketExtension,
   input  wire          rxTriggerDoppler,
   input  wire    [7:0] rxTriggerRUAllocation,
   input  wire          rxTriggerULFECCoding,
   input  wire    [3:0] rxTriggerULMCS,
   input  wire          rxTriggerULDCM,
   input  wire    [2:0] rxTriggerNumSS,
   input  wire          rxTriggerLDPCExtraSymbol,

   //$port_g Debug Port
   output wire   [15:0] debug                    // Debug Port
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
localparam WIDTH          = 24;
localparam DIVIDEND_WIDTH = 17;
localparam DIVISOR_WIDTH  = 10;


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
reg                    [4:0] compPhase;          // Computation Phase
reg                          started;

reg              [WIDTH-1:0] multIn1;
reg              [WIDTH-1:0] multIn2;
wire             [WIDTH-1:0] multOut;
                 
reg              [WIDTH-1:0] addIn1;
reg              [WIDTH-1:0] addIn2;
wire             [WIDTH-1:0] addOut;
                 
reg              [WIDTH-1:0] reg1;
reg              [WIDTH-1:0] reg2;

reg                          divStart_p;         // Divider start pulse
reg                          divStarted;         // Divider start flag
wire                         divDone_p;          // Divider done pulse
reg     [DIVIDEND_WIDTH-1:0] dividend;           // Divider divident
reg      [DIVISOR_WIDTH-1:0] divisor;            // Divider divisor
reg     [DIVIDEND_WIDTH-1:0] quotient;           // Divider quotient
reg     [DIVIDEND_WIDTH-1:0] divRemainder;       // Divider remainder
reg     [DIVISOR_WIDTH+14:0] extnDivisor;        // Extension of the divisor
reg                    [4:0] divCnt;             // Divider counter
reg     [DIVIDEND_WIDTH-1:0] quotientComb;       // Divider quotient
reg     [DIVIDEND_WIDTH-1:0] divRemainderComb;   // Divider remainder

reg                          rxTriggerMMA;       //
reg                    [3:0] rxTriggerNumHELTF;  //
reg                    [4:0] theltf_dot8;        //
reg                    [2:0] tgiheltf_dot8;      //
wire                   [4:0] theltfsym_dot8;     //
wire                   [4:0] tsym_dot8;          //
wire                         peDisambiguity;     //
wire                   [1:0] preFECPaddingFactor;//
reg                   [10:0] nsd;                //
reg                    [8:0] nsdshort;           //
reg                    [3:0] nbpscs;             //
reg                    [2:0] rmul;               //
reg                    [2:0] rdiv;               //


////////////////////////////////////////////////////////////////////////////////
// Function Definitions
////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

//******************************************************************************
// Trigger parameters decoder
//******************************************************************************

// Number of HE-LTF & Midamble periodicity
always @*
begin
   if (rxTriggerDoppler)
   begin
      rxTriggerMMA = rxTriggerNumHELTFAndMidPeriod[2]; // 0: 10, 1: 20
      case (rxTriggerNumHELTFAndMidPeriod[1:0])
      2'd0:   rxTriggerNumHELTF = 4'd1;
      2'd1:   rxTriggerNumHELTF = 4'd2;
      default rxTriggerNumHELTF = 4'd4;
      endcase
   end
   else
   begin
      rxTriggerMMA = 1'b0; // N/A
      case (rxTriggerNumHELTFAndMidPeriod[2:0])
      3'd0:   rxTriggerNumHELTF = 4'd1;
      3'd1:   rxTriggerNumHELTF = 4'd2;
      3'd2:   rxTriggerNumHELTF = 4'd4;
      3'd3:   rxTriggerNumHELTF = 4'd6;
      default rxTriggerNumHELTF = 4'd8;
      endcase
   end
end


// HE-LTF Type and GI Type
always @*
begin
   case (rxTriggerGIAndLTFType)
   2'd0:    {theltf_dot8, tgiheltf_dot8} = {5'd04, 3'd2};
   2'd1:    {theltf_dot8, tgiheltf_dot8} = {5'd08, 3'd2};
   default: {theltf_dot8, tgiheltf_dot8} = {5'd16, 3'd4};
   endcase
end

assign theltfsym_dot8 = theltf_dot8 | {2'd0,tgiheltf_dot8}; // Theltf  + Tgi,heltf
assign tsym_dot8      = 5'd16       | {2'd0,tgiheltf_dot8}; // Tdft,he + Tgi,heltf


// Packet Extension & disambiguity
assign peDisambiguity      = rxTriggerULPacketExtension[2];
assign preFECPaddingFactor = rxTriggerULPacketExtension[1:0];


// Nsd
always @*
begin
   if      (rxTriggerRUAllocation[7:1]<=36) nsd = 11'd24;
   else if (rxTriggerRUAllocation[7:1]<=52) nsd = 11'd48;
   else if (rxTriggerRUAllocation[7:1]<=60) nsd = 11'd102;
   else if (rxTriggerRUAllocation[7:1]<=64) nsd = 11'd234;
   else if (rxTriggerRUAllocation[7:1]<=66) nsd = 11'd468;
   else if (rxTriggerRUAllocation[7:1]<=67) nsd = 11'd980;
   else                                     nsd = 11'd1960;
   
   if (rxTriggerULDCM)                      nsd = {1'b0,nsd[10:1]};
end


// Nsd,short
always @*
begin
   if      (rxTriggerRUAllocation[7:1]<=36) nsdshort = 9'd6;
   else if (rxTriggerRUAllocation[7:1]<=52) nsdshort = 9'd12;
   else if (rxTriggerRUAllocation[7:1]<=60) nsdshort = 9'd24;
   else if (rxTriggerRUAllocation[7:1]<=64) nsdshort = 9'd60;
   else if (rxTriggerRUAllocation[7:1]<=66) nsdshort = 9'd120;
   else if (rxTriggerRUAllocation[7:1]<=67) nsdshort = 9'd240;
   else                                     nsdshort = 9'd492;
   
   if (rxTriggerULDCM)                      nsdshort = {1'b0,nsdshort[8:2],1'b0};
end


// Nbpscs
always @*
begin
   case (rxTriggerULMCS)
   4'd0 :   nbpscs = 4'd1;
   4'd1 :   nbpscs = 4'd2;
   4'd2 :   nbpscs = 4'd2;
   4'd3 :   nbpscs = 4'd4;
   4'd4 :   nbpscs = 4'd4;
   4'd5 :   nbpscs = 4'd6;
   4'd6 :   nbpscs = 4'd6;
   4'd7 :   nbpscs = 4'd6;
   4'd8 :   nbpscs = 4'd8;
   4'd9 :   nbpscs = 4'd8;
   4'd10:   nbpscs = 4'd10;
   default: nbpscs = 4'd10;
   endcase
end


// R
always @*
begin
   case (rxTriggerULMCS)
   4'd0 :   {rmul,rdiv}= {3'd1 , 3'd2};
   4'd1 :   {rmul,rdiv}= {3'd1 , 3'd2};
   4'd2 :   {rmul,rdiv}= {3'd3 , 3'd4};
   4'd3 :   {rmul,rdiv}= {3'd1 , 3'd2};
   4'd4 :   {rmul,rdiv}= {3'd3 , 3'd4};
   4'd5 :   {rmul,rdiv}= {3'd2 , 3'd3};
   4'd6 :   {rmul,rdiv}= {3'd3 , 3'd4};
   4'd7 :   {rmul,rdiv}= {3'd5 , 3'd6};
   4'd8 :   {rmul,rdiv}= {3'd3 , 3'd4};
   4'd9 :   {rmul,rdiv}= {3'd5 , 3'd6};
   4'd10:   {rmul,rdiv}= {3'd3 , 3'd4};
   default: {rmul,rdiv}= {3'd5 , 3'd6};
   endcase
end


//******************************************************************************
// PSDU Length computation
//******************************************************************************

//  theltf_dot8   = 4<<heltftype
//  tgiheltf_dot8 = 1<<giType
//  tsym_dot8     = 16 + tgiheltf_dot8 #12.8 + tgiheltf
//  m             = 2 #for HE-TB
//
//  # Table 27-12
//  theltfsym_dot8 = theltf_dot8 + tgiheltf_dot8
//
//  if (feccoding):
//     ntail+nservice = 16 #LDPC
//  else:
//     ntail+nservice = 22 #BCC
//
//  # 27-121
//  #thepreamble = trlsig + thesiga + thestft + nheltf*theltfsym
//
//  nheltf_theltfsym_dot8 = nheltf * theltfsym_dot8
//  thepreamble_dot8      = 25 + nheltf_theltfsym_dot8
//
//  txtime_dot8                = ((ullength + 5) *5) / 3
//  txtimewithoutpreamble_dot8 = txtime_dot8 - thepreamble_dot8
//
//  if (doppler==0):
//     nma = 0
//  else:
//     tma_dot8 = mma * tsym_dot8 + nheltf_theltfsym_dot8
//     nma      = (txtimewithoutpreamble_dot8 - (pedisambiguity + 2) * tsym_dot8) / tma_dot8
//
//  if (nma<0):
//     nma = 0
//
//  nsym = floor((txtimewithoutpreamble_dot8 - nma * nheltf_theltfsym_dot8) / tsym_dot8) - pedisambiguity
//
//
//  ncbpsinit  = nbpscs * nss * nsd
//  ncbpsshort = nbpscs * nss * nsdshort
//
//  ndbpsinit  = ncbpsinit  * r
//  ndbpsshort = ncbpsshort * r
//
//  if (feccoding==0 or ldpcextra==0):
//     ainit    = a
//     nsyminit = nsym
//  elif (a==1):
//     ainit    = 0 # 4
//     nsyminit = nsym-1-ulstbc # mstbc=2
//  else:
//     ainit    = a-1
//     nsyminit = nsym
//
//  if (ainit==0):
//     ndbpslast = ndbpsinit
//  else:
//     ndbpslast = ndbpsshort * ainit
//
//  ndb = (nsyminit-1-ulstbc) * ndbpsinit + (1+ulstbc) * ndbpslast
//
//  psdulen = (ndb-ntailnservice)/8
//
always @(posedge clk or negedge hardRst_n)
begin
   if (~hardRst_n)
      started <= 1'd0;
   else if (~softRst_n)
      started <= 1'd0;
   else if (psduLenStart_p)
      started <= 1'd1;
   else if (psduLenValid_p)
      started <= 1'd0;
end


// PSDU Length Computation phase 
always @(posedge clk or negedge hardRst_n)
begin
   if (hardRst_n == 1'b0) // Asynchronous Reset
      compPhase <= 5'd0;
   else if (softRst_n == 1'b0) // Synchronous Reset
      compPhase <= 5'd0;
   else if (~started | psduLenValid_p)
      compPhase <= 5'd0;
   else if (~divStart_p & (~divStarted | divDone_p))
      compPhase <= compPhase + 5'd1;
end
assign psduLenValid_p = (compPhase == 5'd18) ? 1'b1 : 1'b0;


always @(posedge clk or negedge hardRst_n)
begin
   if (hardRst_n == 1'b0) // Asynchronous Reset
   begin
      reg1       <= {WIDTH{1'b0}};
      reg2       <= {WIDTH{1'b0}};
                 
      multIn1    <= {WIDTH{1'b0}};
      multIn2    <= {WIDTH{1'b0}};
                 
      addIn1     <= {WIDTH{1'b0}};
      addIn2     <= {WIDTH{1'b0}};

      dividend   <= {WIDTH{1'b0}};
      divisor    <= {WIDTH{1'b0}};
      divStart_p <= 1'b0;
   end
   else if (softRst_n == 1'b0) // Synchronous Reset
   begin
      reg1       <= {WIDTH{1'b0}};
      reg2       <= {WIDTH{1'b0}};
                 
      multIn1    <= {WIDTH{1'b0}};
      multIn2    <= {WIDTH{1'b0}};
                 
      addIn1     <= {WIDTH{1'b0}};
      addIn2     <= {WIDTH{1'b0}};

      dividend   <= {WIDTH{1'b0}};
      divisor    <= {WIDTH{1'b0}};
      divStart_p <= 1'b0;
   end
   else if (started & ~divStart_p & (~divStarted | divDone_p))
   begin
      // Default Value
      divStart_p <= 1'b0;

      multIn1    <= {WIDTH{1'b0}};
      multIn2    <= {WIDTH{1'b0}};
                 
      addIn1     <= {WIDTH{1'b0}};
      addIn2     <= {WIDTH{1'b0}};

      // Computation
      case (compPhase)
      5'd0: 
      begin
         // Nheltf * Theltfsym
         multIn1 <= {{WIDTH-5{1'b0}},theltfsym_dot8};
         multIn2 <= {{WIDTH-4{1'b0}},rxTriggerNumHELTF};
 
         // LLength + 5
         addIn1  <= {{WIDTH-12{1'b0}},rxTriggerULLength};
         addIn2  <= 'd6;
      end
      5'd1: 
      begin
         //nHEltf * tHEltfsym
         reg1 <= multOut;

         // (LLength + 5) * 5
         multIn1 <= addOut;
         multIn2 <= 'd5;

         // thepreamble = -25 - Nheltf * Theltfsym
         addIn1  <= 'd24;
         addIn2  <= multOut;
      end
      5'd2:
      begin
         // thepreamble
         reg2    <= ~addOut;

         // ((LLength + 5) * 5) / 3
         dividend   <= multOut[DIVIDEND_WIDTH-1:0];
         divisor    <= 'd3;
         divStart_p <= 1'b1;
      end
      5'd3:
      begin
         // (pedisambiguity + 2) * tsym_dot8
         multIn1 <= peDisambiguity ? -'sd3 : -'sd2;
         multIn2 <= {{WIDTH-5{1'b0}},tsym_dot8};

         // txtimewithoutpreamble = txtime - thepreamble
         addIn1  <= {{WIDTH-DIVIDEND_WIDTH{1'b0}},quotient};
         addIn2  <= reg2;
      end
      5'd4:
      begin
         // txtimewithoutpreamble
         reg2    <= addOut[WIDTH-1] ? 'd0 : addOut; // Negative time detection

         // mma * tsym
         multIn1 <= rxTriggerMMA ? 'd20 : 'd10;
         multIn2 <= {{WIDTH-5{1'b0}},tsym_dot8};

         // txtimewithoutpreamble - (pedisambiguity + 2) * tsym
         addIn1  <= addOut[WIDTH-1] ? 'd0 : addOut; // Negative time detection
         addIn2  <= multOut;
      end
      5'd5:
      begin
         // tma = mma * tsym + Nheltf * Theltfsym
         addIn1  <= multOut;
         addIn2  <= reg1; //nHEltf_tHEltfsym;

         // txtimewithoutpreamble - (pedisambiguity + 2) * tsym
         dividend   <= addOut[WIDTH-1] ? 'd0 : addOut[DIVIDEND_WIDTH-1:0]; // Negative time detection
      end
      5'd6:
      begin
         // nma = (txtime - thepreamble - (pedisambiguity + 2) * tsym ) / tma
         divisor    <= addOut[DIVISOR_WIDTH-1:0];
         divStart_p <= 1'b1;
      end
      5'd7:
      begin
         // nma * nheltf * theltfsym
         multIn1 <= reg1; //nHEltf_tHEltfsym;
         multIn2 <= rxTriggerDoppler ? {{WIDTH-DIVIDEND_WIDTH{1'b0}},quotient} : 'd0;

         // txtimewithoutpreamble + 1
         addIn1  <= reg2;
         addIn2  <= 'd1;
      end
      5'd8:
      begin
         // txtimewithoutpreamble_dot8 - nma * nheltf * theltfsym_dot8
         addIn1  <= ~multOut;
         addIn2  <= addOut;
      end
      5'd9:
      begin
         // (txtimewithoutpreamble_dot8 - nma * nheltf * theltfsym_dot8) / tsym_dot8
         dividend   <= addOut[WIDTH-1] ? 'd0 : addOut[DIVIDEND_WIDTH-1:0]; // Negative time detection
         divisor    <= {{DIVISOR_WIDTH-5{1'b0}},tsym_dot8};
         divStart_p <= 1'b1;
      end
      5'd10:
      begin
         // nbpscs * nss * nsd 
         multIn1 <= rxTriggerNumSS[0] ? {{WIDTH-5{1'b0}},nbpscs,1'b0} : {{WIDTH-4{1'b0}},nbpscs};
         multIn2 <= {{WIDTH-11{1'b0}},nsd};
      end
      5'd11:
      begin
         // Nsym = Nsym -1 when disambiguity bit is set
         addIn1  <= {{WIDTH-DIVIDEND_WIDTH{1'b0}},quotient};
         addIn2  <= peDisambiguity ? -'sd1 : 'sd0;

         // nbpscs * nss * nsd * r_mul
         multIn1 <= multOut;
         multIn2 <= {{WIDTH-3{1'b0}},rmul};
      end
      5'd12:
      begin
         // nbpscs * nss * nsdshort
         multIn1 <= rxTriggerNumSS[0] ? {{WIDTH-5{1'b0}},nbpscs,1'b0} : {{WIDTH-4{1'b0}},nbpscs};
         multIn2 <= {{WIDTH-9{1'b0}},nsdshort};

         // Nsym,init-1-ulstbc
         // if (feccoding==0 or ldpcextra==0)
         //    nsyminit = nsym
         // elif (a==1)
         //    nsyminit = nsym-1-ulstbc # mstbc=2
         // else
         //    nsyminit = nsym
         addIn1  <= addOut;//{{WIDTH-DIVIDEND_WIDTH{1'b0}},quotient};
         addIn2  <= rxTriggerULFECCoding & rxTriggerLDPCExtraSymbol & preFECPaddingFactor==2'd1 & rxTriggerULSTBC ? -'sd4 :
                    rxTriggerULFECCoding & rxTriggerLDPCExtraSymbol & preFECPaddingFactor==2'd1 | rxTriggerULSTBC ? -'sd2 : -'sd1;

         // ndbps = nbpscs * nss * nsd * r
         dividend   <= multOut[DIVIDEND_WIDTH-1:0];
         divisor    <= {{DIVISOR_WIDTH-3{1'b0}},rdiv};
         divStart_p <= 1'b1;
      end
      5'd13:
      begin
         // ndbps
         reg1    <= {{WIDTH-DIVIDEND_WIDTH{1'b0}},quotient};

         // Nsym,init-1-ulstbc
         reg2    <= addOut;

         // nbpscs * nss * nsdshort * r_mul
         multIn1 <= multOut;
         multIn2 <= {{WIDTH-3{1'b0}},rmul};
      end
      5'd14:
      begin
         // (nsyminit-1-ulstbc) * ndbpsinit
         multIn1 <= reg1;                        //ndbpsinit
         multIn2 <= {{WIDTH-9{1'b0}},reg2[8:0]}; //Nsym,init-1-ulstbc (always < 2^9)

         //  if (feccoding==0 or ldpcextra==0) ainit = preFec
         //  else                              ainit = preFec-1
         addIn1  <= {{WIDTH-2{1'b0}},preFECPaddingFactor};
         addIn2  <= rxTriggerULFECCoding & rxTriggerLDPCExtraSymbol ? -'sd1 : 'sd0;

         // ndbpsshort = nbpscs * nss * nsdshort * r
         dividend   <= multOut[DIVIDEND_WIDTH-1:0];
         divisor    <= {{DIVISOR_WIDTH-3{1'b0}},rdiv};
         divStart_p <= 1'b1;
      end
      5'd15:
      begin
         // if ainit==0: ndbpsinit
         // else:        ndbpsshort * ainit
         multIn1 <= addOut[1:0]=='d0 ? reg1 : {{WIDTH-DIVIDEND_WIDTH{1'b0}},quotient};
         multIn2 <= addOut[1:0]=='d0 ? 'd1  : {{WIDTH-2{1'b0}},addOut[1:0]};

         //  (nsyminit-1-ulstbc) * ndbpsinit - ntail - nservice
         addIn1  <= multOut;
         addIn2  <= rxTriggerULFECCoding ? -'sd16 : -'sd22;
      end
      5'd16:
      begin
         // ndb = (nsyminit-1-ulstbc) * ndbpsinit + (1+ulstbc) * ndbpslast - ntail - nservice
         addIn1  <= rxTriggerULSTBC ? {multOut[WIDTH-2:0],1'b0}: multOut;
         addIn2  <= addOut;
      end
      endcase
   end
   else
      divStart_p <= 1'b0;
end

// Multiplier
assign multOut = multIn1 * multIn2;

// Adder
assign addOut  = addIn1  + addIn2;


// PSDU length = (ndb - ntailnservice) / 8
always @(posedge clk or negedge hardRst_n)
begin
   if (~hardRst_n)
      psduLen <= 20'd0;
   else if (~softRst_n)
      psduLen <= 20'd0;
   else if (compPhase==5'd17)
   begin
      if (addOut[WIDTH-1] | // Invalid PSDU Length : Negative PSDU Length
          reg2[WIDTH-1]   ) // Invalid PSDU Length : (Nsym,init - 1 - STBC) < 0
         psduLen <= 20'd0;
      else 
         psduLen <= addOut[WIDTH-2:3];
   end
end

assign psduLenValid = ~started;


//******************************************************************************
// Divider
//******************************************************************************
always @(posedge clk or negedge hardRst_n)
begin
   if (~hardRst_n)
      divStarted <= 1'd0;
   else if (~softRst_n)
      divStarted <= 1'd0;
   else if (divStart_p)
      divStarted <= 1'd1;
   else if (divDone_p)
      divStarted <= 1'd0;
end


assign divDone_p = divStarted & divCnt==5'd0;


always @ (posedge clk or negedge hardRst_n)
begin
   if (~hardRst_n)
   begin
      divCnt          <= 5'd16;
      divRemainder    <= {{DIVIDEND_WIDTH  }{1'b0}};
      quotient        <= {{DIVIDEND_WIDTH  }{1'b0}}; 
      extnDivisor     <= {{DIVISOR_WIDTH+15}{1'b0}};
   end  
   else 
   begin
      if (divStart_p)
      begin
         quotient     <= {{DIVIDEND_WIDTH}{1'b0}}; 
         divRemainder <= dividend;

         if (dividend[DIVIDEND_WIDTH-1:DIVIDEND_WIDTH-4]==4'd0)
         begin
            divCnt       <= 5'd12;
            extnDivisor  <= {4'b0,divisor,11'b0};
         end
         else
         begin
            divCnt       <= 5'd16;
            extnDivisor  <= {divisor,15'b0};
         end
      end
      else if (divCnt!=5'd0)
      begin
         quotient     <= quotientComb;
         divRemainder <= divRemainderComb;
         extnDivisor  <= {2'd0,extnDivisor[DIVISOR_WIDTH+14:2]};
         divCnt       <= divCnt - 5'd2;
      end
   end
end


always @*
begin
   divRemainderComb = divRemainder;
   quotientComb     = quotient;

   // 1st Step
   if ({8'b0,divRemainderComb} >= extnDivisor)
   begin
      divRemainderComb = divRemainderComb - extnDivisor[DIVIDEND_WIDTH-1:0];
      quotientComb     = {quotientComb[DIVIDEND_WIDTH-2:0], 1'b1};
   end
   else
   begin
      quotientComb     = {quotientComb[DIVIDEND_WIDTH-2:0], 1'b0};
   end

   // 2nd Step
   if ({8'b0,divRemainderComb} >= {1'b0,extnDivisor[DIVISOR_WIDTH+14:1]})
   begin
      divRemainderComb = divRemainderComb - {1'b0,extnDivisor[DIVIDEND_WIDTH-1:1]};
      quotientComb     = {quotientComb[DIVIDEND_WIDTH-2:0], 1'b1};
   end
   else
   begin
      quotientComb     = {quotientComb[DIVIDEND_WIDTH-2:0], 1'b0};
   end
end


//******************************************************************************
// Debug Port
//******************************************************************************
assign debug = {divCnt[4:0],
                divDone_p,
                divStarted,
                divStart_p,
                compPhase[4:0],
                psduLenValid_p,
                psduLenValid,
                psduLenStart_p};


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

`ifdef  RW_SIMU_ON
reg [15:0] debug_txtimewithoutpreamble;
reg [15:0] debug_nsym;
reg [15:0] debug_ndbpsinit;
reg [15:0] debug_ndbpslast;
reg  [1:0] debug_ainit;

// Capture TxTime without preamble
always @(posedge clk or negedge hardRst_n)
begin
   if (~hardRst_n)
      debug_txtimewithoutpreamble <= 'd0;
   else if (~softRst_n)
      debug_txtimewithoutpreamble <= 'd0;
   else if (compPhase=='d4 & addOut[WIDTH-1])
      // Negative txtime, forced to '0'
      debug_txtimewithoutpreamble <= 'd0;
   else if (compPhase=='d4)
      debug_txtimewithoutpreamble <= addOut;
end

// Capture Number of symbols
always @(posedge clk or negedge hardRst_n)
begin
   if (~hardRst_n)
      debug_nsym <= 'd0;
   else if (~softRst_n)
      debug_nsym <= 'd0;
   else if (compPhase=='d10 & divDone_p)
      debug_nsym <= quotient;
end

// Capture Ndbps,init
always @(posedge clk or negedge hardRst_n)
begin
   if (~hardRst_n)
      debug_ndbpsinit <= 'd0;
   else if (~softRst_n)
      debug_ndbpsinit <= 'd0;
   else if (compPhase=='d13 & divDone_p)
      debug_ndbpsinit <= quotient;
end

// Capture Ndbps,last
always @(posedge clk or negedge hardRst_n)
begin
   if (~hardRst_n)
      debug_ndbpslast <= 'd0;
   else if (~softRst_n)
      debug_ndbpslast <= 'd0;
   else if (compPhase=='d16)
      debug_ndbpslast <= multOut;
end

// Capture a,init
always @(posedge clk or negedge hardRst_n)
begin
   if (~hardRst_n)
      debug_ainit <= 'd0;
   else if (~softRst_n)
      debug_ainit <= 'd0;
   else if (compPhase=='d15)
      debug_ainit <= addOut[1:0];
end


`endif//RW_SIMU_ON


///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef  RW_ASSERT_ON
`endif//RW_ASSERT_ON

endmodule
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
