////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : Module which creates and formats the compressed
//                    beamforming frame.
//
// Simulation Notes :
//
//    For simulation, one define is available
//
//    RW_SIMU_ON   : which creates string signals to display the FSM states on
//                the waveform viewers
//
// Some pragmas for code coverage have been defined.
//  - The implicite default state has been excluded
//  - The some default states have been excluded because not reachable by design
// pragma coverage implicit_default=off
//
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
//
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
module formatBFR (
          //$port_g Clock and Reset interface
          input  wire        macCoreTxClk,          // MAC Core Transmit Clock
          input  wire        macCoreClkHardRst_n,   // Hard Reset of the MAC Core Clock domain
                                                    // active low
          input  wire        macCoreTxClkSoftRst_n, // Soft Reset of the MAC Core Clock domain
                                                    // active low
          output reg         macCoreBfrRxClkEn,     // Indicates that the RX Clock must be enabled
                                                    // for the BFR Report Read on MPIF
          //$port_g MAC Controller interface
          input  wire [47:0] destAddr,              // Receiver address
          input  wire [47:0] srcAddr,               // Source address
          input  wire [15:0] duration,              // Duration
          input  wire  [5:0] ulPwrHeadroom,         // Uplink power headroom
          input  wire        txUPHControl,          // Uplink power headroom HT control field is
                                                    // present

          //$port_g BFR Controller interface
          input  wire [14:0] bfrLength,             // Beamforming report frame length
          input  wire  [2:0] bfrNr,                 // Beamforming report Nr Index
          input  wire  [2:0] bfrNc,                 // Beamforming report Nc Index
          input  wire  [1:0] bfrChBW,               // Beamforming report Channel Width
          input  wire  [1:0] bfrFeedbackType,       // Beamforming report Feedback Type
          input  wire        bfrHE,                 // Beamforming report Format (VHT or HE)
          input  wire  [6:0] bfrRUStartIndex,       // Beamforming report RU Start Index
                                                    // (Only for HE BFR)
          input  wire  [6:0] bfrRUEndIndex,         // Beamforming report RU End Index
                                                    // (Only for HE BFR)
          input  wire  [5:0] bfrSoundingDialToken,  // Beamforming report Sounding Dialog Token
                                                    // Number
          input  wire  [1:0] bfrGrouping,           // Beamforming report Grouping
          input  wire        bfrCodebook,           // Beamforming report Codebook Information

          //$port_g Tx Controller Main FSM interface
          input  wire        bfrStart_p,            // indicates that a BFR packet has to be sent
          output reg         bfrDone_p,             // indicates that the BFR packet has been sent

          //$port_g FCS Block interface
          input  wire        fcsEnd_p,              // indicates the end of the transmission
          input  wire        fcsBusy,               // indicates that the FCS block is busy cannot
                                                    // accept new data.

          output wire        bfrFCSEnable,          // Enable the FCS Block
          output wire        bfrFCSStart_p,         // indicates the begining of the transmission
                                                    // and enable the FCS block
          output wire        bfrFCSShiftTx,         // indicates FCS engine to append FCS calculated
                                                    // on data
          output reg   [7:0] bfrFCSDInTx,           // Data to the FCS block
          output wire        bfrFCSDInValidTx,      // indicates that the data on fcsDInTx is valid
                                                    // and can be processed.

          //$port_g Beamforming Report Controller
          input  wire  [7:0] bfrData,               // Beamforming report byte
          input  wire        bfrDataValid,          // Beamforming report byte valid
          output wire        bfrDataReady,          // Beamforming report byte request

          //$port_g CSReg Block interface
          input  wire        pwrMgt,                // Power Management is enabled for the current
          input  wire [47:0] bssID,                 // BSS ID

          //$port_g Debug interface
          output reg   [3:0] formatBFRFSMCs         // formatBFR FSM Current State
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
// formatBFR FSM states definition
//$fsm_sd formatBFRFSM
localparam
   BFR_IDLE             = 4'd0,
   BFR_STARTTX          = 4'd1,
   BFR_FRMCNTRL         = 4'd2,
   BFR_DURATIONID       = 4'd3,
   BFR_RA               = 4'd4,
   BFR_TA               = 4'd5,
   BFR_BSSID            = 4'd6,
   BFR_SEQCNTRL         = 4'd7,
   BFR_CATEGORY         = 4'd8,
   BFR_ACTION           = 4'd9,
   BFR_MIMOCNTRL        = 4'd10,
   BFR_REPORT           = 4'd11,
   BFR_FCS              = 4'd12,
   BFR_HTCNTRL          = 4'd13;

localparam
   FRMCNTRL_BYTECNT     = 15'd2,   // Number of bytes of Frame Control field
   DURATIONID_BYTECNT   = 15'd2,   // Number of bytes of Duration ID field
   RA_BYTECNT           = 15'd6,   // Number of bytes of RA field
   TA_BYTECNT           = 15'd6,   // Number of bytes of TA field
   BSSID_BYTECNT        = 15'd6,   // Number of bytes of TA field
   SEQCNTRL_BYTECNT     = 15'd2,   // Number of bytes of Sequence Control field
   HTCNTRL_BYTECNT      = 15'd4,   // Number of bytes of HT Control field
   VHTMIMOCNTRL_BYTECNT = 15'd3,   // Number of bytes of VHT MIMO Control field
   HEMIMOCNTRL_BYTECNT  = 15'd5,   // Number of bytes of HE  MIMO Control field
   PROTOCOL_VERSION     = 2'b00,   // Protocol version
   MNGTFRM_TYPE         = 2'b00,   // Management Frame Type
   ACTIONNOACK_SUBTYPE  = 4'b1110; // Action No Ack Frame Sub-Type


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////

// formatBFR FSM signals definition
reg                      [3:0] formatBFRFSMNs;                  // formatBFRFSM FSM Next State


wire                           countDone_p;                     // Pulse to indicate the end of a
                                                                // multiple bytes field
reg                     [14:0] byteCnt;                         // Counts the number of bytes
wire                           byteCntEn;                       // Enable the byteCnt counter
reg                     [14:0] nBytes;                          // Number of bytes of the current
                                                                // field

wire                           bfrActive;                       // Indicate that formatBFR is active

reg                     [11:0] bfrSeqNbr;                       // Beamforming Report Frame Sequence
                                                                // Number
wire                    [23:0] vhtMIMOCtrl;                     // VHT MIMO Control field
wire                    [39:0] heMIMOCtrl;                      // HE MIMO Control field
wire                    [31:0] htCtrl;                          // HT control field


////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

// formatBFR FSM Current State Logic
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)         // Asynchronous Reset
    formatBFRFSMCs <= BFR_IDLE;
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    formatBFRFSMCs <= BFR_IDLE;
  else
    formatBFRFSMCs <= formatBFRFSMNs;
end


// formatBFR FSM Next State Logic.
always @*
begin
  case(formatBFRFSMCs)

    BFR_IDLE:
      //$fsm_s In BFR_IDLE state, the state machine waits until it is triggered by
      //the MAC Controller
      if (bfrStart_p)
        //$fsm_t When bfrStart_p is received, the state machine goes to BFR_STARTTX state
        formatBFRFSMNs = BFR_STARTTX;
      else
        //$fsm_t While bfrStart_p is not received, the state machine stays in BFR_IDLE state
        formatBFRFSMNs = BFR_IDLE;

    BFR_STARTTX:
      //$fsm_s In BFR_STARTTX state, the state machine start the FCS engine and the MAC-PHY
      //Interface

      //$fsm_t One clock cycle after, the state machine moves to BFR_FRMCNTRL state
        formatBFRFSMNs = BFR_FRMCNTRL;

    BFR_FRMCNTRL:
      //$fsm_s In BFR_FRMCNTRL state, the state machine sends the first byte of the BFR frame
      //Control
      if (countDone_p)
        //$fsm_t When countDone_p is received meaning that the Frame Control has been completely
        //sent trough the FCS,
        //the state machine goes to BFR_DURATIONID state
        formatBFRFSMNs = BFR_DURATIONID;
      else
        //$fsm_t When countDone_p is low meaning that the Frame Control has not been yet completely
        //sent trough the FCS,
        //the state machine stays in BFR_FRMCNTRL state
        formatBFRFSMNs = BFR_FRMCNTRL;

    BFR_DURATIONID:
      //$fsm_s In BFR_DURATIONID state, the state machine sends the Duration ID to FCS block
      if (countDone_p)
        //$fsm_t When countDone_p is received meaning that the Duration ID has been completely sent
        //through the FCS,
        //the state machine goes to BFR_RA state
        formatBFRFSMNs = BFR_RA;
      else
        //$fsm_t When countDone_p is low meaning that the Duration ID has not been yet completely
        //sent through the FCS block,
        //the state machine stays in BFR_DURATIONID state
        formatBFRFSMNs = BFR_DURATIONID;

    BFR_RA:
      //$fsm_s In BFR_RA state, the state machine sends the RA to FCS block
      if (countDone_p)
        //$fsm_t When countDone_p is received meaning that the RA has been completely sent through
        //the FCS,
        //the state machine goes to BFR_TA state
        formatBFRFSMNs = BFR_TA;
      else
        //$fsm_t When countDone_p is low meaning that the RA has not been yet completely sent
        //through the FCS block,
        //the state machine stays in BFR_RA state
        formatBFRFSMNs = BFR_RA;

    BFR_TA:
      //$fsm_s In BFR_TA state, the state machine sends the TA to FCS block
      if (countDone_p)
        //$fsm_t When countDone_p is received meaning that the TA has been completely sent through
        //the FCS,
        //the state machine goes to BFR_BSSID state
        formatBFRFSMNs = BFR_BSSID;
      else
        //$fsm_t When countDone_p is low meaning that the TA has not been yet completely sent
        //through the FCS block,
        //the state machine stays in BFR_TA state
        formatBFRFSMNs = BFR_TA;

    BFR_BSSID:
      //$fsm_s In BFR_BSSID state, the state machine sends the BSSID to FCS block
      if (countDone_p)
        //$fsm_t When countDone_p is received meaning that the BSSID has been completely sent
        //through the FCS,
        //the state machine goes to BFR_SEQCNTRL state
        formatBFRFSMNs = BFR_SEQCNTRL;
      else
        //$fsm_t When countDone_p is low meaning that the BSSID has not been yet completely sent
        //through the FCS block,
        //the state machine stays in BFR_BSSID state
        formatBFRFSMNs = BFR_BSSID;

    BFR_SEQCNTRL:
      //$fsm_s In BFR_SEQCNTRL state, the state machine sends the BFR Sequence Control field to
      //FCS block
      if (countDone_p && txUPHControl)
        //$fsm_t When countDone_p is received meaning that the BFR Sequence Control field has been
        //completely sent through the FCS and UPH is present,
        //the state machine goes to BFR_HTCNTRL state
        formatBFRFSMNs = BFR_HTCNTRL;
      else if (countDone_p && ~txUPHControl)
        //$fsm_t When countDone_p is received meaning that the BFR Sequence Control field has been
        //completely sent through the FCS and UPH is not present,
        //the state machine goes to BFR_CATEGORY state
        formatBFRFSMNs = BFR_CATEGORY;
      else
        //$fsm_t When countDone_p is low meaning that the BFR Sequence Control field has not been
        //yet completely sent through the FCS block,
        //the state machine stays in BFR_SEQCNTRL state
        formatBFRFSMNs = BFR_SEQCNTRL;

    BFR_HTCNTRL:
      //$fsm_s In BFR_HTCNTRL state, the state machine sends the BFR Sequence Control field to
      //FCS block
      if (countDone_p)
        //$fsm_t When countDone_p is received meaning that the BFR HT Control field has been
        //completely sent through the FCS,
        //the state machine goes to BFR_CATEGORY state
        formatBFRFSMNs = BFR_CATEGORY;
      else
        //$fsm_t When countDone_p is low meaning that the BFR HT Control field has not been
        //yet completely sent through the FCS block,
        //the state machine stays in BFR_HTCNTRL state
        formatBFRFSMNs = BFR_HTCNTRL;

    BFR_CATEGORY:
      //$fsm_s In BFR_CATEGORY state, the state machine sends the BFR category field to FCS block
      if (!fcsBusy)
        //$fsm_t When fcsBusy is low meaning that the FCS is free, the BFR Category field has been
        //completely sent through the FCS,
        //the state machine goes to BFR_ACTION state
        formatBFRFSMNs = BFR_ACTION;
      else
        //$fsm_t When fcsBusy is high meaning that the FCS block is busy, the BFR Category field
        //has not been yet completely sent through the FCS block,
        //the state machine stays in BFR_CATEGORY state
        formatBFRFSMNs = BFR_CATEGORY;

    BFR_ACTION:
      //$fsm_s In BFR_ACTION state, the state machine sends the BFR Action field to FCS block
      if (!fcsBusy)
        //$fsm_t When fcsBusy is low meaning that the FCS is free, the BFR Action field has been
        //completely sent through the FCS,
        //the state machine goes to BFR_MIMOCNTRL state
        formatBFRFSMNs = BFR_MIMOCNTRL;
      else
        //$fsm_t When fcsBusy is high meaning that the FCS block is busy, the BFR Action field has
        //not been yet completely sent through the FCS block,
        //the state machine stays in BFR_ACTION state
        formatBFRFSMNs = BFR_ACTION;

    BFR_MIMOCNTRL:
      //$fsm_s In BFR_MIMOCNTRL state, the state machine sends the BFR MIMO Control field to FCS
      //block
      if (countDone_p)
        //$fsm_t When countDone_p is received meaning that the BFR MIMO Control field has been
        //completely sent through the FCS,
        //the state machine goes to BFR_REPORT state
        formatBFRFSMNs = BFR_REPORT;
      else
        //$fsm_t When countDone_p is low meaning that the BFR MIMO Control field has not been yet
        //completely sent through the FCS block,
        //the state machine stays in BFR_MIMOCNTRL state
        formatBFRFSMNs = BFR_MIMOCNTRL;

    BFR_REPORT:
      //$fsm_s In BFR_REPORT state, the state machine sends the BFR compressed report &
      //the MU Exclusive Beamforming report field to FCS block
      if (countDone_p)
        //$fsm_t When countDone_p is received meaning that the BFR compressed report &
        //the MU Exclusive Beamforming report field has been completely sent through the FCS,
        //the state machine goes to BFR_FCS state
        formatBFRFSMNs = BFR_FCS;
      else
        //$fsm_t When countDone_p is low meaning that the BFR compressed report & the MU Exclusive
        //Beamforming report field has not been yet completely sent through the FCS block,
        //the state machine stays in BFR_REPORT state
        formatBFRFSMNs = BFR_REPORT;

    BFR_FCS:
      //$fsm_s In BFR_FCS state, the state machine trigs the FCS block to shift out the FCS value
      //and wait until its completion.
      if (fcsEnd_p)
        //$fsm_t When fcsEnd_p is received meaning that the FCS has been completely sent,
        //the state machine goes back to BFR_IDLE state
        formatBFRFSMNs = BFR_IDLE;
      else
        //$fsm_t While fcsEnd_p is not received, the state machine stays in BFR_FCS state
        formatBFRFSMNs = BFR_FCS;

     // Disable coverage on the default state because it cannot be reached.
     // pragma coverage block = off
     default:
        formatBFRFSMNs = BFR_IDLE;
     // pragma coverage block = on
  endcase
end


// byteCnt counter generation
// This counter allows to count the number of bytes per field
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)         // Asynchronous Reset
    byteCnt     <= 15'b0;
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    byteCnt     <= 15'b0;
  else
  begin
    // The counter is incremented only if it is started and when the FCS block is not busy
    if (!fcsBusy && byteCntEn)
    begin
      // While the value of the counter is not equal to nBytes, the byteCnt counter is incremented
      if (byteCnt < nBytes)
        byteCnt <= byteCnt + 15'd1;
      else
      begin
        byteCnt <= 15'b0;
      end
    end
  end
end


// This combinational process assigns the number of the current field.
// Note that nBytes is 1 byte less than the number of byte of the field because
// it is used as reference for the byteCnt counter which starts at 0.
always @*
begin
  case (formatBFRFSMCs)
    BFR_FRMCNTRL :
      nBytes = FRMCNTRL_BYTECNT - 15'd1;

    BFR_DURATIONID :
      nBytes = DURATIONID_BYTECNT - 15'd1;

    BFR_RA :
      nBytes = RA_BYTECNT - 15'd1;

    BFR_TA :
      nBytes = TA_BYTECNT - 15'd1;

    BFR_BSSID :
      nBytes = BSSID_BYTECNT - 15'd1;

    BFR_SEQCNTRL :
      nBytes = SEQCNTRL_BYTECNT - 15'd1;

    BFR_HTCNTRL :
      nBytes = HTCNTRL_BYTECNT - 15'd1;

    BFR_MIMOCNTRL :
      if (bfrHE)
        nBytes = HEMIMOCNTRL_BYTECNT  - 15'd1;
      else
        nBytes = VHTMIMOCNTRL_BYTECNT - 15'd1;

    BFR_REPORT :
    begin
      // Beamforming Report field length is calculated from the Length of the PPDU
      //
      // report Length = MPDULength - FrmCntrl - DurationId - RA - TA - BSSID -
      //                 SeqCntrl - Category - Action - MIMOCtrl - FCS
      if (bfrHE==1'b0)
        nBytes = bfrLength - 15'd33 - 15'd1;
      else
        nBytes = bfrLength - 15'd35 - 15'd1;
    end

    default
      nBytes = 15'd0;
  endcase
end


// Start the counter in the start which have more that 1 byte length as for DURATIONID and RA.
assign byteCntEn = ( (formatBFRFSMCs == BFR_FRMCNTRL  ) ||
                     (formatBFRFSMCs == BFR_DURATIONID) ||
                     (formatBFRFSMCs == BFR_RA        ) ||
                     (formatBFRFSMCs == BFR_TA        ) ||
                     (formatBFRFSMCs == BFR_BSSID     ) ||
                     (formatBFRFSMCs == BFR_SEQCNTRL  ) ||
                     (formatBFRFSMCs == BFR_HTCNTRL   ) ||
                     (formatBFRFSMCs == BFR_MIMOCNTRL ) ||
                    ((formatBFRFSMCs == BFR_REPORT    ) && bfrDataValid && !fcsBusy))
                    ? 1'b1 : 1'b0;


// The countDone_p indication used to change the state is generated when the byteCnt counter reach
// the nBytes value.
// Due to the handshaking, this pulse is generated only when the fcs is not busy.
// Note also that this pulse is generated only when the counter is started.
assign countDone_p = (byteCntEn && (byteCnt == nBytes) && !fcsBusy) ? 1'b1 : 1'b0;


// bfrDone_p pulses generation indicating the end of the BFR transmission
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)         // Asynchronous Reset
    bfrDone_p       <= 1'b0;
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    bfrDone_p       <= 1'b0;
  else
  begin
    bfrDone_p       <= 1'b0;
    if ((formatBFRFSMCs == BFR_FCS) && fcsEnd_p)
    // The bfrDone_p pulse is generated when the fcsEnd_p is received
      bfrDone_p     <= 1'b1;
  end
end


// bfrSeqNbr counter generation
// This counter is used as Sequence Number for the BFR Frame, it is incremented after each BFR
// Transmission.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)         // Asynchronous Reset
    bfrSeqNbr     <= 12'h0;
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    bfrSeqNbr     <= 12'h0;
  else
  begin
    // The counter is incremented after each BFR Transmission
    if (bfrDone_p)
      bfrSeqNbr   <= bfrSeqNbr + 12'd1;
  end
end


// VHT MIMO Control field
assign vhtMIMOCtrl = 
   {bfrSoundingDialToken,  // SoundingDialogToken     
    2'b00,                 // Reserved                
    1'b1,                  // FirstFeedbackSegment    
    3'b000,                // RemainingFeedbackSegment
    bfrFeedbackType[0],    // FeedbackType            
    bfrCodebook,           // CodebookInformantion    
    bfrGrouping,           // Grouping                
    bfrChBW,               // ChannelWidth            
    bfrNr,                 // NrIndex                 
    bfrNc};                // NcIndex                 


// HE MIMO Control field
assign heMIMOCtrl = 
   {4'b0000,               // reserved
    bfrSoundingDialToken,  // Sounding Dialog Token Number
    bfrRUEndIndex,         // RU End Index
    bfrRUStartIndex,       // RU Start Index
    1'b1,                  // First Feedback Segment
    3'b000,                // Remaining Feedback Segment
    bfrFeedbackType,       // Feedback Type
    bfrCodebook,           // Codebook Informantion
    bfrGrouping[0],        // Grouping
    bfrChBW,               // BW
    bfrNr,                 // Nr Index
    bfrNc};                // Nc Index

// Uplink Power Headroom (UPH) HE Variant HT Control
assign htCtrl =
   {18'd0,                  // Padding
    2'd0,                   // UPH Control Information: Reserved
    ulPwrHeadroom[5:0],     // UPH Control Information: Uplink Power Headroom
    4'd4,                   // UPH Control ID
    2'b11};                 // HE Variant HT Control


// FCS Block Control
////////////////////

// FCS data assignment
always @*
begin
  case(formatBFRFSMCs)

    BFR_FRMCNTRL :
      begin
        // frame control for BFR will be {+HTC = txUPHControl, protectedFrame = 1'b0, moreData = 1'b0,
        //   pwrMgt = pwrMgt, retry = 1'b0, morefrg = 1'b0,  frmDs = 1'b0, toDs = 1'b0,
        //   SubType = ActionNoAck, Type = ManagementFrame, ProtocolVersion = 2'b00}
        case(byteCnt[0])
          1'b0       : bfrFCSDInTx = {ACTIONNOACK_SUBTYPE, MNGTFRM_TYPE, PROTOCOL_VERSION};
          1'b1       : bfrFCSDInTx = {txUPHControl, 2'b0, pwrMgt, 4'b0};
        endcase
      end

    BFR_DURATIONID :
      begin
        case(byteCnt[0])
          1'b0       : bfrFCSDInTx = duration[7:0];
          1'b1       : bfrFCSDInTx = duration[15:8];
        endcase
      end

    BFR_RA  :
      begin
        case(byteCnt[2:0])
          3'd0       : bfrFCSDInTx = destAddr[7:0];
          3'd1       : bfrFCSDInTx = destAddr[15:8];
          3'd2       : bfrFCSDInTx = destAddr[23:16];
          3'd3       : bfrFCSDInTx = destAddr[31:24];
          3'd4       : bfrFCSDInTx = destAddr[39:32];
          3'd5       : bfrFCSDInTx = destAddr[47:40];
          // Disable coverage because in BFR_RA state, byteCnt cannot be bigger than 5
          // pragma coverage block = off
          default    : bfrFCSDInTx = 8'b0;
          // pragma coverage block = on
        endcase
      end

    BFR_TA  :
      begin
        case(byteCnt[2:0])
          3'd0       : bfrFCSDInTx = srcAddr[7:0];
          3'd1       : bfrFCSDInTx = srcAddr[15:8];
          3'd2       : bfrFCSDInTx = srcAddr[23:16];
          3'd3       : bfrFCSDInTx = srcAddr[31:24];
          3'd4       : bfrFCSDInTx = srcAddr[39:32];
          3'd5       : bfrFCSDInTx = srcAddr[47:40];
          // Disable coverage because in BFR_TA state, byteCnt cannot be bigger than 5
          // pragma coverage block = off
          default    : bfrFCSDInTx = 8'b0;
          // pragma coverage block = on
        endcase
      end

    BFR_BSSID :
      begin
        case(byteCnt[2:0])
          3'd0       : bfrFCSDInTx = bssID[7:0];
          3'd1       : bfrFCSDInTx = bssID[15:8];
          3'd2       : bfrFCSDInTx = bssID[23:16];
          3'd3       : bfrFCSDInTx = bssID[31:24];
          3'd4       : bfrFCSDInTx = bssID[39:32];
          3'd5       : bfrFCSDInTx = bssID[47:40];
          // Disable coverage because in BFR_BSSID state, byteCnt cannot be bigger than 5
          // pragma coverage block = off
          default    : bfrFCSDInTx = 8'b0;
          // pragma coverage block = on
        endcase
      end

    BFR_SEQCNTRL :
      begin
        case(byteCnt[0])
          // Sequence control for BFR will be {SequenceNumber = bfrSeqNbr, FragmentNumber = 4'b0}
          1'b0       : bfrFCSDInTx = {bfrSeqNbr[3:0],4'h0};
          1'b1       : bfrFCSDInTx =  bfrSeqNbr[11:4];
        endcase
      end

    BFR_HTCNTRL :
      begin
        case(byteCnt[1:0])
          2'd0       : bfrFCSDInTx = htCtrl[0  +: 8];
          2'd1       : bfrFCSDInTx = htCtrl[8  +: 8];
          2'd2       : bfrFCSDInTx = htCtrl[16 +: 8];
          2'd3       : bfrFCSDInTx = htCtrl[24 +: 8];
        endcase
      end

    BFR_CATEGORY :
      begin
        if (bfrHE==1'b0)
          // Category subfield of Action field for VHT BFR will be {Category = VHT}
          bfrFCSDInTx = 8'd21;
        else
          // Category subfield of Action field for HE BFR will be {Category = HE}
          bfrFCSDInTx = 8'd30;
      end

    BFR_ACTION :
      begin
        // Action subfield of Action field for VHT BFR will be {Action = VHT Compressed Beamforming}
        // Action subfield of Action field for HE  BFR will be {Action = HE  Compressed Beamforming}
        // Both equal to '0'
        bfrFCSDInTx = 8'd0;
      end

    BFR_MIMOCNTRL :
      begin
        case ({bfrHE,byteCnt[2:0]})
        // VHT MIMO control
        {1'b0,3'd0} : bfrFCSDInTx = vhtMIMOCtrl[0 +: 8];
        {1'b0,3'd1} : bfrFCSDInTx = vhtMIMOCtrl[8 +: 8];
        {1'b0,3'd2} : bfrFCSDInTx = vhtMIMOCtrl[16+: 8];
        // HE MIMO control
        {1'b1,3'd0} : bfrFCSDInTx = heMIMOCtrl[0  +: 8];
        {1'b1,3'd1} : bfrFCSDInTx = heMIMOCtrl[8  +: 8];
        {1'b1,3'd2} : bfrFCSDInTx = heMIMOCtrl[16 +: 8];
        {1'b1,3'd3} : bfrFCSDInTx = heMIMOCtrl[24 +: 8];
        {1'b1,3'd4} : bfrFCSDInTx = heMIMOCtrl[32 +: 8];
        // Disable coverage because in BFR_MIMOCNTRL state, byteCnt cannot be bigger than
        // 2 in VHT BFR or 4 for HE BFR
        // pragma coverage block = off
        default     : bfrFCSDInTx = 8'b0;
        // pragma coverage block = on
        endcase
      end

    BFR_REPORT  :
    begin
      // BFR report field for BFR is provided by the Beamforming Report Controller
      bfrFCSDInTx = bfrData;
    end

    default :
      bfrFCSDInTx = 8'b0;

  endcase
end


// When the state machine is in BFR_STARTTX, the FCS Engine is started
assign bfrFCSStart_p   = (formatBFRFSMCs == BFR_STARTTX) ? 1'b1 : 1'b0;


// When the state machine is in BFR_FCS state, the FCS engine is requested to shift out the
// computed signature
assign bfrFCSShiftTx   = ((formatBFRFSMNs == BFR_FCS) && !fcsEnd_p) ? 1'b1 : 1'b0;


// Data Valid generation
// The data valid is high when the consumer (here the FCS engine) is not busy.
// In this specific case, the data are valid only when the FSM is in an active state with data
assign bfrFCSDInValidTx = (formatBFRFSMCs == BFR_REPORT) ? (bfrDataValid && !fcsBusy) :
                                                           (bfrActive    && !fcsBusy) ;


// Indicate when the formatBFR module is active and shift data to the FCS
// The BFR_STARTTX states does not provide data and the BFR_FCS neither.
assign bfrActive = ((formatBFRFSMCs != BFR_IDLE   ) &&
                    (formatBFRFSMCs != BFR_STARTTX) &&
                    (formatBFRFSMCs != BFR_FCS    )) ? 1'b1 : 1'b0;


// Enable the FCS module.
assign bfrFCSEnable = (formatBFRFSMCs != BFR_IDLE) ? 1'b1 : 1'b0;


// bfr Controller Interface management
//////////////////////////

// Beamforming Data is requested when FCS is not busy.
assign bfrDataReady = ((formatBFRFSMCs == BFR_REPORT) && !fcsBusy) ? 1'b1 : 1'b0;

// Beamforming Report
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)         // Asynchronous Reset
    macCoreBfrRxClkEn <= 1'b0;
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    macCoreBfrRxClkEn <= 1'b0;
  else if (formatBFRFSMCs == BFR_REPORT)
    macCoreBfrRxClkEn <= 1'b1;
  else
    macCoreBfrRxClkEn <= 1'b0;
end



////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// String definition to display formatBFRFSM current state
reg [22*8:0] formatBFRFSMCs_str;

// Disable coverage the RW_SIMU_ON part because this code is not functional but
// here to ease the simulation
// pragma coverage block = off

// formatBFRFSM FSM states displayed in a string to easy simulation and debug
always @*
begin
  case (formatBFRFSMCs)
  BFR_IDLE           :  formatBFRFSMCs_str = {"BFR_IDLE"};
  BFR_STARTTX        :  formatBFRFSMCs_str = {"BFR_STARTTX"};
  BFR_FRMCNTRL       :  formatBFRFSMCs_str = {"BFR_FRMCNTRL"};
  BFR_DURATIONID     :  formatBFRFSMCs_str = {"BFR_DURATIONID"};
  BFR_RA             :  formatBFRFSMCs_str = {"BFR_RA"};
  BFR_TA             :  formatBFRFSMCs_str = {"BFR_TA"};
  BFR_BSSID          :  formatBFRFSMCs_str = {"BFR_BSSID"};
  BFR_SEQCNTRL       :  formatBFRFSMCs_str = {"BFR_SEQCNTRL"};
  BFR_HTCNTRL        :  formatBFRFSMCs_str = {"BFR_HTCNTRL"};
  BFR_CATEGORY       :  formatBFRFSMCs_str = {"BFR_CATEGORY"};
  BFR_ACTION         :  formatBFRFSMCs_str = {"BFR_ACTION"};
  BFR_MIMOCNTRL      :  formatBFRFSMCs_str = {"BFR_MIMOCNTRL"};
  BFR_REPORT         :  formatBFRFSMCs_str = {"BFR_REPORT"};
  BFR_FCS            :  formatBFRFSMCs_str = {"BFR_FCS"};
  default            :  formatBFRFSMCs_str = {"XXX"};
  endcase
end
// pragma coverage block = on
`endif // RW_SIMU_ON


// System Verilog Assertions
////////////////////////////

`ifdef RW_ASSERT_ON
`endif // RW_ASSERT_ON
endmodule

