//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company: RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : Module which creates and formats the MPDU frames. 
//                    
// Simulation Notes : 
//                    
//    For simulation, one define is available
//
//    RW_SIMU_ON   : which creates string signals to display the FSM states on  
//                the waveform viewers
//
// Some pragmas for code coverage have been defined.
//  - The implicite default state has been excluded
//  - The some default states have been excluded because not reachable by design
// pragma coverage implicit_default=off 
//
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements : 
//   This version does not include yet the optimization found if the two first 
//   byte of the MAC Header Descriptor which are marked as "Reserved" are not 
//   provided by the txFIFO. This modification requires a change in the DMA 
//   Engine
//   
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//module txController (
module formatMPDU ( 
          //$port_g Clock and Reset interface
          input  wire        macCoreTxClk,          // MAC Core Transmit Clock
          input  wire        macCoreClkHardRst_n,   // Hard Reset of the MAC Core Clock domain 
                                                    // active low
          input  wire        macCoreTxClkSoftRst_n, // Soft Reset of the MAC Core Clock domain 
                                                    // active low
          //$port_g MAC Controller interface
          input  wire        skipMPDU_p,            // Indication that MPDU in TX FIFO has to be discarded
          input  wire [15:0] duration,              // Duration
          input  wire        retry,                 // Retry Bit
          input  wire [15:0] tsfDuration,           // Duration from the preamble until the first bit of TSF
          input  wire        sendOnSIFS,            // If set, data is sent on tickSIFS else on tickSlot
          output reg   [3:0] txTID,                 // TID of the transmitted frame
          output reg   [1:0] typeTx,                // TYPE of the transmitted frame
          output reg         typeTxValid,           // Indicates the typeTx signal is valid
          output reg         txAMSDUPresent,        // Indicates than the transmitted frame is an A-MSDU
          output reg         txBcMc,                // Indicates than the transmitted frame has a group address.
          output reg         sentCFEND,             // Indicates that the transmitted frame is a CF-END
          output reg         sentRTS,               // Indicates that the transmitted frame is a RTS
          output reg         sentBAReq,             // Indicates that the transmitted frame is a BA-Request
          output wire        sentNDP,               // Indicates that the frame is a NDP
`ifdef RW_BFMER_EN
          output reg         sentBFRPOLL,           // Indicates that the frame is a BFR POLL
`endif // RW_BFMER_EN
          output reg         skipMPDUDone_p,        // Indicates that MPDU contained in the TXFIFO has been discarded.

          //$port_g MAC-PHY-IF Block interface
          input  wire        mpIfTxEn,              // Transmit on-going indication 

          //$port_g Tx Controller Main FSM interface
          input  wire        sendMPDU_p,            // Indicates that a DATA packet has to be sent
          input  wire        sendSWTB,              // Indicates that the frame is a SW TB frame
          output reg         currentMPDUDone_p,     // Indicates that the DATA packet has been sent
          output wire        mpduTxStart_p,         // Indicates to the MAC PHY IF that the TX start
          
          //$port_g Tx Parameters Cache interface
          input  wire [15:0] MPDUFrameLengthTx,     // Gives the length of the MPDU.
          input  wire        dontGenerateMH,        // Indicates that HW should not generate the MAC Header by itself
          input  wire        dontEncrypt,           // Indicates that HW should bypassed the encryption operation during tx.
          input  wire        dontTouchFC,           // Indicates that HW should not update the Frame Control field,
          input  wire        dontTouchDur,          // Indicates that HW should not update the Duration field
          input  wire        dontTouchQoS,          // Indicates that HW should not update the QoS field 
          input  wire        dontTouchHTC,          // Indicates that HW should not update the HTC field 
          input  wire        dontTouchTSF,          // Indicates that HW should not update the TSF field 
          input  wire        dontTouchDTIM,         // Indicates that HW should not update the DTIM Count field
          input  wire        dontTouchFCS,          // Indicates that HW should not update the FCS field
          input  wire        aMPDU,                 // Indicates whether this Transmit Header Descriptor belong to an A-MPDU
          input  wire        txSMPDU,               // Detect S-MPDU

          //$port_g Tx FIFO interface
          input  wire        txFIFOEmpty,           // Indicates that a TX FIFO is empty
          input  wire  [7:0] txFIFORdData,          // Read data from TX FIFO
          input  wire  [1:0] txFIFOMPDUDelimiters,  // MPDU delimiter coming from the TX Tag FIFO
          input  wire        txFIFODataValid,       // Indicates when the txFIFORdData is Valid

          output wire        txFIFORead,            // Read data request from TX FIFO

          //$port_g Key Search Engine interface
          input  wire  [2:0] cTypeKSR,              // Indicates the type of encryption
                                                    // 0 : No Key
                                                    // 1 : WEP
                                                    // 2 : TKIP
                                                    // 3 : CCMP
                                                    // 5 : GCMP
          input  wire  [1:0] cLenKSR,               // Indicates the cypher length of encryption

          //$port_g TxVector Interface
          input  wire  [3:0] formatMod,             // 4'b0000: NON-HT
                                                    // 4'b0001: NON-HT-DUP-OFDM
                                                    // 4'b0010: HT-MF
                                                    // 4'b0011: HT-GF
                                                    // 4'b0100: VHT
                                                    // 4'b0101: HE-SU
                                                    // 4'b0110: HE-MU
                                                    // 4'b0111: HE-EXT-SU
                                                    // 4'b1000: HE-TB

          //$port_g Encryption Engine interface
          // Shared by all encrypt mode
          output wire        txCsIsIdle,            // Indicates the FSM is out of IDLE
          output reg  [15:0] txDataLength,          // Indicate the Tx Data to encrypted Length
          output wire [47:0] txCCMPTKIPSeqCnt,      // Packet Number for TKIP and CCMP

          // Plain Data
          output reg   [7:0] txPlainData,           // Plain data out
          output reg         txPlainDataValid,      // Plain data valid
          output reg         txPlainDataEnd,        // Last byte valid
          input  wire        txPlainDataReady,      // Plain data ready

          // Encrypted Data
          input  wire  [7:0] txEncrData,            // Encrypted data in
          input  wire        txEncrDataValid,       // Encrypted data in Valid
          input  wire        txEncrDataEnd_p,       // Encrypted data in the last byte
          output wire        txEncrDataReady,       // Encrypted data Ready

          // WEP and TKIP SBOX Init
          output wire        initEncrTxCntrl_p,     // Pulse to init the Encryption Engine 
          input  wire        cryptoInitDone_p,      // Indicates crypto Engine has finished initialization  

          // WEP Only
          output reg [23:0]  txWEPIV,               // WEP IV

          // TKIP Only

          // CCMP Only
          output wire        txAddress4Pres,        // Address 4 is present
          output wire        txQoSFrame,             // Frame is QoS
          output reg [15:0]  txFrameControl,        // MAC Header Frame Control Field
          output reg  [47:0] txAddr1,               // Address 1 Value
          output reg  [47:0] txAddr2,               // Address 2 Value
          output reg [47:0]  txAddr3,               // MAC Header Address 3 Field
          output reg [47:0]  txAddr4,               // MAC Header Address 4 Field
          output reg [15:0]  txQoSCF,               // MAC Header QoS Control Field
          output reg [15:0]  txSeqControl,          // MAC Header Seauence Control Field
          output reg         txHTMode,              // Indicate HT mode for CCMP
          
`ifdef RW_WAPI_EN
          output reg [7:0]   txKeyIdx,              // WAPI Header Key Index
          output reg [127:0] txPN,                  // WAPI Header Packet Number (PN) Field
`endif //RW_WAPI_EN
          

          //$port_g FCS Block interface
          input  wire        fcsEnd_p,              // Indicates the end of the transmission
          input  wire        fcsBusy,               // Indicates that the FCS block is busy cannot accept new data.

          output wire        mpduFCSEnable,         // Enable the FCS Block
          output wire        mpduFCSStart_p,        // Indicates the begining of the transmission and enable the FCS block
          output wire        mpduFCSShiftTx,        // Indicates FCS engine to append FCS calculated on data
          output reg   [7:0] mpduFCSDInTx,          // Data to the FCS block
          output wire        mpduFCSDInValidTx,     // Indicates that the data on fcsDInTx is valid and can be processed.

          //$port_g Timers Block interface
          input  wire        tickSIFS,              // A pulse to indicate the end of SIFS period
          input  wire        tickSlot,              // A pulse to indicate the end of Slot period
          input  wire  [7:0] dtimCnt,               // Current DTIM count
          input  wire [63:0] tsf,                   // TSF current value

          //$port_g CSReg Block interface
          input  wire  [7:0] timOffset,             // Indicates to the HW the offset, from the first byte of the Beacon frame, 
                                                    // at which the first byte of the TIM field is located
          input  wire        pwrMgt,                // Power Management is enabled for the current
 
          //$port_g Backoff Block interface
          input  wire  [2:0] activeAC,              // Indicates the current AC which has won contention

          //$port_g Debug interface
          output reg   [4:0] formatMPDUFSMCs,       // formatMPDUFSM FSM Current State
          output wire        mpduSeqCtrlDone_p,     // Indicates that the sequence number has been transmitted
          output wire [14:0] formatMPDUFrameDebug1, // frame Debug 1 from formatMPDU module
          output wire [14:0] formatMPDUFrameDebug2  // frame Debug 2 from formatMPDU module
                 );

//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

// formatMPDUFSM FSM states definition
//$fsm_sd formatMPDUFSM
localparam 
             FD_IDLE  =  5'd0,  
       FD_INITCRYPTO  =  5'd1,  
          FD_STARTTX  =  5'd2,  
   FD_FRMCNTRL_BYTE1  =  5'd3,  
   FD_FRMCNTRL_BYTE2  =  5'd4,  
       FD_DURID_DATA  =  5'd5,  
            FD_ADDR1  =  5'd6,  
            FD_ADDR2  =  5'd7,  
            FD_ADDR3  =  5'd8, 
         FD_SQNCNTRL  =  5'd9, 
            FD_ADDR4  =  5'd10, 
         FD_QOSCNTRL  =  5'd11, 
          FD_CFCNTRL  =  5'd12, 
          FD_HTCNTRL  =  5'd13, 
              FD_TSF  =  5'd14, 
               FD_IV  =  5'd15, 
            FD_EXTIV  =  5'd16, 
      FD_WAPI_KEYIDX  =  5'd17, 
          FD_WAPI_PN  =  5'd18, 
        FD_MPDUSTART  =  5'd19, 
         FD_FRM_BODY  =  5'd20, 
 FD_WAIT_ICVMIC_DONE  =  5'd22, 
              FD_FCS  =  5'd25,  
      FD_FLUSHTXFIFO  =  5'd26;  

localparam
FRMCNTRL_BYTE1_OFFSET = 16'd0,   // Offset of the first byte of the frame control 
FRMCNTRL_BYTE2_OFFSET = 16'd1,   // Offset of the second byte of the frame control

        DURID_BYTECNT = 16'd2,   // Number of bytes of Duration ID field
        DURID_OFFSET  = 16'd2,   // Offset of the Duration ID field

         ADDR_BYTECNT = 16'd6,   // Number of bytes of ADDR field
         ADDR1_OFFSET = 16'd4,   // Offset of the ADDR1 field
         ADDR2_OFFSET = 16'd10,  // Offset of the ADDR2 field
         ADDR3_OFFSET = 16'd16,  // Offset of the ADDR3 field
         ADDR4_OFFSET = 16'd24,  // Offset of the ADDR4 field

     SQNCNTRL_BYTECNT = 16'd2,   // Number of bytes of Sequence Control field
      SQNCNTRL_OFFSET = 16'd22,  // Offset of the Sequence Control field

     QOSCNTRL_BYTECNT = 16'd2,   // Number of bytes of QoS Control field
      QOSCNTRL_OFFSET = 16'd30,  // Offset of the QoS Control field

      CFCNTRL_BYTECNT = 16'd2,   // Number of bytes of Carried Frame Control field
       CFCNTRL_OFFSET = 16'd32,  // Offset of the Carried Frame Control field

      HTCNTRL_BYTECNT = 16'd4,   // Number of bytes of HT Control field
       HTCNTRL_OFFSET = 16'd34,  // Offset of the HT Control field

           IV_BYTECNT = 16'd4,   // Number of bytes of IV field
           IV_OFFSET  = 16'd38,  // Offset of the IV field

        EXTIV_BYTECNT = 16'd4,   // Number of bytes of Extended IV field
         EXTIV_OFFSET = 16'd42,  // Offset of the Extended IV field


          ICV_BYTECNT = 16'd4,   // Number of bytes of ICV field

   CCMP128MIC_BYTECNT = 16'd8,   // Number of bytes of CCMPMIC field
   CCMP256MIC_BYTECNT = 16'd16,  // Number of bytes of CCMPMIC field
    
      GCMPMIC_BYTECNT = 16'd16,  // Number of bytes of GCMPMIC field
    
    `ifdef RW_WAPI_EN
   WAPI_KEYIDX_BYTECNT = 16'd2,   // Number of bytes of KeyIdx field 
   WAPI_KEYIDX_OFFSET  = 16'd38,  // Offset of the KeyIdx field
       
       WAPI_PN_BYTECNT = 16'd16,  // Number of bytes of PN field 
       WAPI_PN_OFFSET  = 16'd40,  // Offset of the PN field
       
      WAPIMIC_BYTECNT  = 16'd16,  // Number of bytes of WAPIMIC field
    `endif //RW_WAPI_EN

      FRM_BODY_OFFSET = 16'd46,  // Offset of the Extended IV field

          FCS_BYTECNT = 16'd4,   // Number of bytes of FCS field
           FCS_OFFSET = 16'd46,  // Offset from the end of the frame of the FCS field        

          TSF_BYTECNT = 16'd8,   // Number of bytes of TSF field
           TSF_OFFSET = 16'd46;  // Offset from the end of the frame of the TSF field        

localparam 
   ENCRYPTION_TYPE_NONE = 3'd0,   // None encryption is selected
   ENCRYPTION_TYPE_WEP  = 3'd1,   // WEP encryption is selected
   ENCRYPTION_TYPE_TKIP = 3'd2,   // TKIP encryption is selected
   ENCRYPTION_TYPE_CCMP = 3'd3,   // CCMP encryption is selected
`ifdef RW_WAPI_EN
   ENCRYPTION_TYPE_WAPI = 3'd4,   // WAPI encryption is selected
`endif // RW_WAPI_EN
   ENCRYPTION_TYPE_GCMP = 3'd5;   // GCMP encryption is selected

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

// formatMPDUFSM FSM signals definition
reg [4:0] formatMPDUFSMNs;  // formatMPDUFSM FSM Next State
reg [4:0] formatMPDUFSMPs;  // formatMPDUFSM FSM Next State

`ifdef RW_SIMU_ON
// String definition to display formatMPDUFSM current state
reg [18*8-1:0] formatMPDUFSMCs_str;
// String definition to display the type of frame
reg [23*8-1:0] txFrameType_str;
// String definition to display the type of encryption
reg [8*8-1:0] encryptionType_str;
`endif // RW_SIMU_ON

wire        countDone_p;          // Pulse to indicate the end of a multiple bytes field
reg  [15:0] txByteCnt;            // Counts the number of transmitted bytes
reg  [15:0] readByteCnt;          // Counts the number of bytes read from the TxFIFO
reg  [15:0] readByteCnt_ff1;      // Counts the number of bytes read from the TxFIFO
reg  [15:0] readGoodByteCnt;      // Counts the number of bytes read from the TxFIFO
reg  [15:0] readFieldByteCnt;     // Counts the number of bytes read in the current field
reg  [15:0] fieldBytes;           // Number of bytes of the current field
reg  [15:0] fieldOffset;          // Offset in byte of the current field in the Transmit Header Descriptor
reg  [15:0] frameBodyLength;      // Length of the frame body computed based on the frame length and 
                                  // the length of the mac header fields transmitted
reg         txFIFOReadEn;         // Enable the read accesses from the Tx FIFO 
reg         toDS;                 // Indicates that the toDs bit extracted from the MAC Header is set 
reg         fromDS;               // Indicates that the fromDs bit extracted from the MAC Header is set 
reg         protectedFrame;       // Indicates that the Protected bit extracted from the MAC Header is set and a valid encryption type has been selected
reg         order;                // Indicates that the order bit extracted from the MAC Header is set 
reg   [3:0] subtypeTx;            // Frame SubType extracted from the MAC Header.
wire        extIV;                // Indicates that the MAC Header should contained extIV because the encryption type 

reg   [7:0] addr1MSB;             // Stores MSB of the Transmitted ADDR1
reg   [5:0] seqNumber;            // Stores Transmitted Sequence Number
reg   [3:0] fragNumber;           // Stores Transmitted Fragment Number
reg         txPwrMgt;             // Stores Transmitted PwrMgt bit
reg         moreFrag;             // Stores Transmitted more Frag bit
reg         moreData;             // Stores Transmitted more Data bit
reg         txRetry;              // Stores Transmitted Retry bit

wire        endOfMPDUinTxFifo;    // Indicates when the last byte of the MPDU has been popped from the TX FIFO
reg         endOfMPDUinTxFifo_fcs;// Save the pulse when it is detected outside of FLUSH fifo
reg   [7:0] txData;               // Bytes which are transmitted.    
reg   [1:0] writePointer;         // Write Pointer of the ping-pong buffer
reg   [1:0] readPointer;          // Read Pointer of the ping-pong buffer
reg   [8:0] readDataBank0;        // Bank 0 of the ping-pong buffer
reg   [8:0] readDataBank1;        // Bank 1 of the ping-pong buffer
reg   [8:0] readDataBank2;        // Bank 2 of the ping-pong buffer

reg   [1:0] carriedFrameType;     // Type of carried frame
reg   [3:0] carriedFrameSubType;  // Subtype of carried frame

wire        controlWrapperFrame;  // Indicates that the frame is a Control Wrapper Frame
wire        qosFrame;             // Indicates that the frame is a QoS Frame
wire        htcFrame;             // Indicates that the frame is a +HTC Frame
wire        isBeacon;             // Indicates that the frame is a Beacon
wire        isProbeResponse;      // Indicates that the frame is a Probe Response
wire        isTimingAdvertisement;// Indicates that the frame is a Timing Advertisement
wire        isNDP;                // Indicates that the frame is a NDP
wire        endOfFrame;           // Indicates the end of the frame.
reg  [63:0] tsfAntenna;           // TSF value at the antenna taken into account the PHY and RF TX delays
wire [15:0] dtimOffset;           // Indicates the offset, from the first byte of the Beacon frame, 
                                  // at which the DTIM field is located

reg [47:0]  txCCMPSeqCnt;         // Capture CCMP pkt number
reg [47:0]  txTKIPSeqCnt;         // Capture TKIP pkt number

reg         CSIsIdle;             // Pulse to indicate that the CS is idle
wire        enable;               // Signal to indicate the block is no idle

reg         wait4Slot;            // Flag indicating that the formatMPDU is waiting for the slot boundary

reg         skipMPDU;             // Flag indicating a skip MPDU request from txcontroller

reg [15:0]  readByteOffset;       
reg  [5:0]  readByteOffset_ff1;   
wire        nextcountdone_p;      
wire        lastPayloadByte;


reg         txWEP;                 // WEP Mode
reg         txTKIP;                // TKIP Mode
reg         txCCMP;                // CCMP Mode
reg         txGCMP;                // GCMP Mode
`ifdef RW_WAPI_EN
reg         txWAPI;                // WAPI Mode
`endif // RW_WAPI_EN
wire        formatMPDUBusy;        // Indicate that the formatMPDU FSM cannot accept new byte because it is busy in internal state (or shift the FCS).
wire        cryptoEngineBusy;      // Indicate that the crypto Engine is currently busy and cannot accept new data.
wire        encryptionOnGoing;
reg [1:0]   bufferCnt;

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// signal to enable most of registers blocks
assign enable       = (((formatMPDUFSMCs == FD_IDLE) && (formatMPDUFSMNs != FD_IDLE)) || (formatMPDUFSMCs != FD_IDLE));

// formatMPDUFSM FSM Current State Logic 
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    formatMPDUFSMCs <= FD_IDLE; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    formatMPDUFSMCs <= FD_IDLE; 
  else if(enable)
    formatMPDUFSMCs <= formatMPDUFSMNs; 
end

// formatMPDUFSM FSM Previous State Logic 
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    formatMPDUFSMPs <= FD_IDLE; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    formatMPDUFSMPs <= FD_IDLE; 
  else if(enable)
    formatMPDUFSMPs <= formatMPDUFSMCs; 
end


// formatMPDUFSM FSM Next State Logic.
always @* 
begin
  case(formatMPDUFSMCs)

    FD_IDLE:
      //$fsm_s In FD_IDLE state, the state machine waits until it is triggered by 
      //the MAC Controller. 
      if (sendMPDU_p)
      begin 
        //$fsm_t When sendMPDU_p is received, the state machine goes to FD_STARTTX state.
        formatMPDUFSMNs = FD_STARTTX;
      end
      else if (skipMPDU_p)
        //$fsm_t When skipMPDU_p is received, the state machine moves to FD_FLUSHTXFIFO state to discard the current MPDU.
        formatMPDUFSMNs = FD_FLUSHTXFIFO;
      else
        //$fsm_t While sendMPDU_p or skipMPDU_p is not received, the state machine stays in FD_IDLE state.
        formatMPDUFSMNs = FD_IDLE;

    FD_STARTTX:
      //$fsm_s In FD_STARTTX state, the state machine start the FCS engine and the MAC-PHY Interface.
      if (MPDUFrameLengthTx == 16'd0)
        //$fsm_t When MPDUFrameLengthTx is null, the state machine moves to FD_MPDUSTART.
        formatMPDUFSMNs = FD_MPDUSTART;
      else if (dontGenerateMH)
        //$fsm_t One clock cycle after and if the SW has requested to do not generate the MAC Header (dontGenerateMH set), 
        //the state machine moves to FD_MPDUSTART state.
        formatMPDUFSMNs = FD_MPDUSTART; 
      else
        //$fsm_t One clock cycle after and if the SW has requested to generate the MAC Header (dontGenerateMH reset), 
        //the state machine moves to FD_FRMCNTRL_BYTE1 state.
        formatMPDUFSMNs = FD_FRMCNTRL_BYTE1; 

    FD_FRMCNTRL_BYTE1:
      //$fsm_s In FD_FRMCNTRL_BYTE1 state, the Tx Controller sends the first byte of the DATA frame Control.
      if (countDone_p)
        //$fsm_t When countDone_p is low meaning that the TX Controller can continus the processing, 
        //the state machine goes to FD_FRMCNTRL_BYTE2 state.
        formatMPDUFSMNs = FD_FRMCNTRL_BYTE2;
      else
        //$fsm_t While countDone_p is high meaning that the TX Controller has to wait, 
        //the state machine stays in FD_FRMCNTRL_BYTE1 state.
        formatMPDUFSMNs = FD_FRMCNTRL_BYTE1;

    FD_FRMCNTRL_BYTE2:
      //$fsm_s In FD_FRMCNTRL_BYTE2 state, the Tx Controller sends the second byte of the DATA frame Control.
      if (countDone_p) 
        //$fsm_t When countDone_p is low meaning that the TX Controller can continus the processing, 
        //the state machine goes to FD_DURID_DATA state.
        formatMPDUFSMNs = FD_DURID_DATA; 
      else
        //$fsm_t While countDone_p is high meaning that the TX Controller has to wait, 
        //the state machine stays in FD_FRMCNTRL_BYTE2 state.
        formatMPDUFSMNs = FD_FRMCNTRL_BYTE2;

    FD_DURID_DATA:
      //$fsm_s In FD_DURID_DATA state, the Tx Controller sends the durationID of the DATA frame.
      if (countDone_p) 
        //$fsm_t When countDone_p is received meaning that the Duration ID has been completely sent, 
        //the state machine goes to FD_ADDR1 state.
        formatMPDUFSMNs = FD_ADDR1; 
      else
        //$fsm_t While countDone_p is low meaning that the DurationID has not been yet completely sent, 
        //the state machine stays in FD_DURID_DATA state.
        formatMPDUFSMNs = FD_DURID_DATA;

    FD_ADDR1:
      //$fsm_s In FD_ADDR1 state, the Tx Controller sends the ADDR1 field of the DATA frame.
      if (countDone_p) 
      begin
        // If countDone_p is received, the state machine moves to another state. However, this next state
        // depends on the type of frame.
        // 
        //  1- If the frame is a Control Wrapper frame, the next state is FD_CFCNTRL to process the 
        //     Carried Frame Control field.
        //
        //  2- If the frame is an ACK or CTS, the next state is FD_MPDUSTART.
        //
        //  3- If the frame is neither an ACK, CTS nor Control wrapper, the next state is FD_ADDR2 
        //     to process the ADDR2 field.
        //
        if (controlWrapperFrame)
          //$fsm_t When countDone_p is received meaning that the ADDR1 field has been completely sent,
          // and the frame is a Control Wrapper, the state machine goes to FD_CFCNTRL state.
          formatMPDUFSMNs = FD_CFCNTRL; 

        else if ((typeTx == 2'b01) && ((subtypeTx == 4'b1100) || (subtypeTx == 4'b1101)))
          //$fsm_t When countDone_p is received meaning that the ADDR1 field has been completely sent,
          // and the frame is a ACK or a CTS, the state machine goes to FD_MPDUSTART state.
          formatMPDUFSMNs = FD_MPDUSTART;
        else  
          //$fsm_t When countDone_p is received meaning that the ADDR1 field has been completely sent,
          //and the frame is neither a ACK, a CTS nor a Control Wrapper, 
          //the state machine goes to FD_ADDR2 state.
          formatMPDUFSMNs = FD_ADDR2; 
      end
      else
        //$fsm_t While countDone_p is low meaning that the ADDR1 has not been yet completely sent, 
        //the state machine stays in FD_ADDR1 state.
        formatMPDUFSMNs = FD_ADDR1;

    FD_ADDR2:
      //$fsm_s In FD_ADDR2 state, the Tx Controller sends the ADDR2 field of the DATA frame.
      if (countDone_p) 
      begin
        // If countDone_p is received, the state machine moves to another state. However, this next state depends on
        // the type of frame.
        // 
        //  1- If the frame is a BlockAck or BlockAckReq frame, the next state is FD_MPDUSTART 
        //     to process the ADDR2 field.
        //
        //  2- If the frame is a RTS, CF-END, CF-END+CF-ACK or a PS-POLL, the next state is FD_MPDUSTART.
        //
        //  3- If the frame is a Management or Data frame, the next state is FD_ADDR3 
        //     to process the ADDR3 field.
        //
        if ((typeTx == 2'b01) && ((subtypeTx == 4'b1000) || (subtypeTx == 4'b1001)))
          //$fsm_t When countDone_p is received meaning that the ADDR2 field has been completely sent,
          //and the frame is a BlockAck or BlockAckReq, the state machine goes to FD_MPDUSTART state.
          formatMPDUFSMNs = FD_MPDUSTART; 
        else if (typeTx == 2'b01)
          //$fsm_t When countDone_p is received meaning that the ADDR2 field has been completely sent,
          //and the frame is a RTS, CF-END, CF-END+CF-ACK or a PS-POLL, the state machine goes to FD_MPDUSTART state.
          formatMPDUFSMNs = FD_MPDUSTART;
        else
          //$fsm_t When countDone_p is received meaning that the ADDR2 field has been completely sent,
          //and the frame is a Management of Data frame, the state machine goes to FD_ADDR3 state.
          formatMPDUFSMNs = FD_ADDR3; 
      end
      else
        //$fsm_t While countDone_p is low meaning that the ADDR2 has not been yet completely sent, 
        //the state machine stays in FD_ADDR2 state.
        formatMPDUFSMNs = FD_ADDR2;

    FD_ADDR3:
      //$fsm_s In FD_ADDR3 state, the Tx Controller sends the ADDR3 field of the DATA frame.
      if (countDone_p) 
        //$fsm_t When countDone_p is received meaning that the ADDR3 field has been completely sent, 
        //the state machine goes to FD_SQNCNTRL state.
        formatMPDUFSMNs = FD_SQNCNTRL; 
      else
        //$fsm_t While countDone_p is low meaning that the ADDR3 has not been yet completely sent, 
        //the state machine stays in FD_ADDR3 state.
        formatMPDUFSMNs = FD_ADDR3;

    FD_SQNCNTRL:
      //$fsm_s In FD_SQNCNTRL state, the Tx Controller sends the Sequence Number field of the DATA frame.
      if (countDone_p) 
      begin
        // If countDone_p is received, the state machine moves to another state. However, this next state depends on
        // the type of frame.
        // 
        //  1- If the frame has both fromDS and toDS bit sets, the next state is FD_ADDR4 
        //     to process the ADDR4 field.
        //
        //  2- If the frame does not have fromDS and toDS bit sets but is a QoS Dataframe (qosFrame = 1),  
        //     the next state is FD_QOSCNTRL to process the QoS Control field. 
        //
        //  3- If the frame does not have fromDS or toDS bit sets, is not a QoS Dataframe (qosFrame = 0),
        //     but is a HTC frame (htcFrame set), the next state is FD_HTC to process the HT Control field. 
        //
        //  4- If the frame does not have fromDS or toDS bit sets, is neither a QoS Dataframe (qosFrame = 0) nor an 
        //     HTC frame but is protected (protectedFrame bit set), the next state is FD_IV to process the IV field.
        //     If WAPI encryption is supported and active the next state is FD_WAPI_KEYIDX.
        //
        //  5- If the frame does not have fromDS or toDS bit sets, is neither a QoS Dataframe (qosFrame = 0) nor an 
        //     HTC frame and is not protected (protectedFrame bit reset), the next state is FD_MPDUSTART.
        //
        if (fromDS && toDS)
          // Case 1
          //$fsm_t When countDone_p is received meaning that the Sequence Control field has been completely sent,
          //and the frame has both fromDS and toDS bit sets, the state machine goes to FD_ADDR4 state.
          formatMPDUFSMNs = FD_ADDR4; 

        else
        begin
          if (qosFrame)
            // Case 2
            //$fsm_t When countDone_p is received meaning that the Sequence Control field has been completely sent,
            //and the Data frame does not have fromDS and toDS bit sets but is a QoS Dataframe (qosFrame = 1), 
            //the state machine goes to FD_QOSCNTRL state.
            formatMPDUFSMNs = FD_QOSCNTRL; 

          else 
          begin
            if (htcFrame)
              // Case 3
              //$fsm_t When countDone_p is received meaning that the Sequence Control field has been completely sent,
              //and the Data frame does not have fromDS and toDS bit sets and is not a QoS Dataframe but is a HTC Frame (htcFrame = 1),
              //the state machine goes to FD_HTCNTRL state.
              formatMPDUFSMNs = FD_HTCNTRL; 

            else
            begin 
              if (txWEP || txTKIP || txCCMP || txGCMP)
                // Case 4
                //$fsm_t When countDone_p is received meaning that the Sequence Control field has been completely sent,
                //and the frame does not have fromDS and toDS bit sets, is not a QoS nor an HT frame but is encrypted 
                //(protectedFrame = 1), the state machine goes to FD_IV state.
                //If WAPI mode is active the next state is FD_WAPI_KEYIDX
                formatMPDUFSMNs = FD_IV; 
              `ifdef RW_WAPI_EN
              else if ( txWAPI )
                formatMPDUFSMNs = FD_WAPI_KEYIDX;
              `endif // RW_WAPI_EN
              else
                // Case 5        
                //$fsm_t If the frame does not have fromDS or toDS bit sets, is neither a QoS Dataframe (qosFrame = 0) nor an 
                //HTC frame and is not protected (protectedFrame bit reset), the next state is FD_MPDUSTART.
                formatMPDUFSMNs = FD_MPDUSTART;
            end  
          end  
        end  
      end  
      else
        //$fsm_t While countDone_p is low meaning that the Sequence Number field has not been yet completely sent, 
        //the state machine stays in FD_SQNCNTRL state.
        formatMPDUFSMNs = FD_SQNCNTRL;

    FD_ADDR4:
      //$fsm_s In FD_ADDR4 state, the Tx Controller sends the ADDR4 field of the DATA frame
      // Note that a Management frame is not supposed to reach this state!
      if (countDone_p) 
      begin
        // If countDone_p is received, the state machine moves to another state. However, this next state depends on
        // the type of frame.
        // 
        //  1- If the frame is a QoS Dataframe (qosFrame = 1),  
        //     the next state is FD_QOSCNTRL to process the QoS Control field. 
        //
        //  2- If the frame is not a QoS Dataframe (qosFrame = 0),
        //     but is an HTC frame (htcFrame set), the next state is FD_HTC to process the HT Control field. 
        //
        //  3- If the frame is neither a QoS Dataframe (qosFrame = 0) nor an 
        //     HTC frame but is protected (protectedFrame bit set), the next state is FD_IV to process the IV field.
        //
        //  4- If the frame is neither a QoS Dataframe (qosFrame = 0) nor an 
        //     HTC frame and is not protected (protectedFrame bit reset), the next state is FD_MPDUSTART to process 
        //     the Frame body field.
        //
        //
        if (qosFrame)
          // Case 1
          //$fsm_t When countDone_p is received meaning that the ADDR4 field has been completely sent,
          //and the Data frame is a QoS Dataframe (qosFrame = 1), the state machine 
          //goes to FD_QOSCNTRL state.
          formatMPDUFSMNs = FD_QOSCNTRL; 

        else 
        begin
          if (htcFrame)
            // Case 2
            //$fsm_t When countDone_p is received meaning that the ADDR4 field has been completely sent,
            //and the Data frame is not a QoS Dataframe but is an HT Frame (htcFrame = 1), the state machine 
            //goes to FD_HTCNTRL state.
            formatMPDUFSMNs = FD_HTCNTRL; 

          else
          begin 
            if (txWEP || txTKIP || txCCMP || txGCMP)
            begin
              // Case 3
              //$fsm_t When countDone_p is received meaning that the ADDR4 field has been completely sent,
              //and the frame is neither a QoS nor an HT frame but is encrypted (protectedFrame = 1), the 
              //state machine goes to FD_IV state.
              //If WAPI mode is active the next state is FD_WAPI_KEYIDX
              formatMPDUFSMNs = FD_IV; 
            end
            `ifdef RW_WAPI_EN
            else if (txWAPI)
              formatMPDUFSMNs = FD_WAPI_KEYIDX;
            `endif // RW_WAPI_EN
            else
              // Case 4        
              //$fsm_t When countDone_p is received meaning that the ADDR4 field has been completely sent,
              //and the frame is neither a QoS nor an HT frame and is not encrypted, the state machine goes to FD_MPDUSTART state.
              formatMPDUFSMNs = FD_MPDUSTART;
          end  
        end  
      end  

      else
        //$fsm_t While countDone_p is low meaning that the ADDR4 field has not been yet completely sent, 
        //the state machine stays in FD_ADDR4 state.
        formatMPDUFSMNs = FD_ADDR4;

    FD_QOSCNTRL:
      //$fsm_s FD_QOSCNTRL state, the Tx Controller sends the QoS Control field of the DATA frame
      if (countDone_p) 
      begin
        // If countDone_p is received, the state machine moves to another state. However, this next state depends on
        // the type of frame.
        // 
        //  1- If the frame is an HTC frame (htcFrame set), the next state is FD_HTC to process the HT Control field. 
        //
        //  2- If the frame is not an HTC frame (htcFrame reset) but is protected (protectedFrame bit set), 
        //     the next state is FD_IV to process the IV field.
        //
        //  3- If the frame does not an HTC frame and is not protected (protectedFrame bit reset) , the next state is FD_MPDUSTART 
        //     to process the Frame body field.
        //
        if (htcFrame)
          // Case 1
          //$fsm_t When countDone_p is received meaning that the QoS Control field has been completely sent,
          //and the Data frame is an HT Frame (htcFrame = 1), the state machine goes to FD_HTCNTRL state.
          formatMPDUFSMNs = FD_HTCNTRL; 

        else
        begin 
            if(protectedFrame)
            begin
              `ifdef RW_WAPI_EN
              if(txWAPI)
                formatMPDUFSMNs = FD_WAPI_KEYIDX;
              else
                formatMPDUFSMNs = FD_IV;
              `else
                formatMPDUFSMNs = FD_IV;
              `endif //RW_WAPI_EN
            end
            else
            // Case 3       
            //$fsm_t When countDone_p is received meaning that the QoS Control field has been completely sent,
            //and the frame is not an HT frame and is not encrypted, the state machine goes to FD_MPDUSTART state.
            formatMPDUFSMNs = FD_MPDUSTART;

        end  
      end  
      else
        //$fsm_t While countDone_p is low meaning that the QoS Control field has not been yet completely sent, 
        //the state machine stays in FD_QOSCNTRL state.
        formatMPDUFSMNs = FD_QOSCNTRL;

    FD_CFCNTRL:
      //$fsm_s FD_CFCNTRL state, the Tx Controller sends the CF Control field of the DATA frame.
      if (countDone_p) 
        //$fsm_t If countDone_p is received  meaning that the CF Control field has been completely sent, 
        //the state machine goes to FD_HTCNTRL state.
        formatMPDUFSMNs = FD_HTCNTRL; 
      else
        //$fsm_t While countDone_p is low meaning that the CF Control field has not been yet completely sent, 
        //the state machine stays in FD_CFCNTRL state.
        formatMPDUFSMNs = FD_CFCNTRL;

    FD_HTCNTRL:
      //$fsm_s FD_HTCNTRL state, the Tx Controller sends the HT Control field of the DATA frame
      if (countDone_p) 
      begin
        // If countDone_p is received, the state machine moves to another state. However, this next state depends on
        // the type of frame.
        // 
        //  1- If the frame is protected (protectedFrame bit set), the next state is FD_IV to process the IV field.
        //
        //  2- If the frame is not protected (protectedFrame bit reset), the next state is FD_MPDUSTART.
        //
        
        //TODO : Check if WAPI Frame include MAC header with HT Control Field
        if (protectedFrame)
        begin
          // Case 1
          //$fsm_t When countDone_p is received meaning that the HT Control field has been completely sent,
          //and the frame is encrypted (protectedFrame = 1), the state machine goes to FD_IV state.
          `ifdef RW_WAPI_EN
            if(txWAPI)
              formatMPDUFSMNs = FD_WAPI_KEYIDX;
            else
              formatMPDUFSMNs = FD_IV;
          `else
              formatMPDUFSMNs = FD_IV;
          `endif //RW_WAPI_EN
        end
        else
        begin
          // Case 2        
          //$fsm_t If the frame is not protected (protectedFrame bit reset), the next state is FD_MPDUSTART.
          formatMPDUFSMNs = FD_MPDUSTART;
        end
      end
      else
        //$fsm_t While countDone_p is low meaning that the HT Control field has not been yet completely sent, 
        //the state machine stays in FD_HTCNTRL state.
        formatMPDUFSMNs = FD_HTCNTRL;

    FD_TSF:
      //$fsm_s FD_TSF state, the Tx Controller sends the TSF field of the Beacon, Probe Response or Timing Advertisement frame.
      if (countDone_p) 
      begin
        //$fsm_t If countDone_p is received meaning that the TSF field has been completely sent, 
        //the state machine goes to FD_FRM_BODY state.
        //Note that only a Beacon, a Probe Response or a Timing Advertisement frame can reach this state.
          formatMPDUFSMNs = FD_FRM_BODY;
      end
      else
        //$fsm_t While countDone_p is low meaning that the TSF field has not been yet completely sent, 
        //the state machine stays in FD_TSF state.
        formatMPDUFSMNs = FD_TSF;

    FD_IV:
      // If countDone_p is received, the state machine moves to another state. However, this next state depends on
      // the encryption type.
      // 
      //  1- If the frame has an Extended IV field (TKIP and CCMP encryption), the next state is FD_EXTIV 
      //     to process the Extended IV field.
      //
      //  2- If the frame does not have Extended IV field (WEP encryption), the next state is FD_MPDUSTART
      //     to process the Frame body field.
      //
      //$fsm_s FD_IV state, the Tx Controller sends the IV field of the DATA frame.
      if (countDone_p) 
      begin
        if(txWEP)
          //$fsm_t When countDone_p and cipher type is WEP but SBOX Init is not done then the FSM has to wait, 
          //the state machine goes to FD_INITCRYPTO state
          formatMPDUFSMNs = FD_INITCRYPTO;
        else if (extIV)
          //$fsm_t If countDone_p is received  meaning that the IV field has been completely sent, 
          //and the frame has an Extended IV field (TKIP or CCMP) (extIV bit = 1), the state machine goes to FD_EXTIV state.
          formatMPDUFSMNs = FD_EXTIV;
        else
          //$fsm_t If countDone_p is received  meaning that the IV field has been completely sent, 
          //and the frame does not an Extended IV field (WEP) (extIV bit = 0), the state machine goes to FD_MPDUSTART state.
          formatMPDUFSMNs = FD_MPDUSTART; 
      end
      else
        //$fsm_t While countDone_p is low meaning that the IV field has not been yet completely sent, 
        //the state machine stays in FD_IV state.
        formatMPDUFSMNs = FD_IV;

    FD_EXTIV:
      //$fsm_s FD_EXTIV state, the Tx Controller sends the Extended IV field of the DATA frame.
      if (countDone_p)
        //$fsm_t When countDone_p then the FSM has to wait for the initialization of the crypto Engine,
        //the state machine goes to FD_INITCRYPTO state
        formatMPDUFSMNs = FD_INITCRYPTO;
      else
        //$fsm_t While countDone_p is low meaning that the Extended IV field has not been yet completely sent, 
        //the state machine stays in FD_EXTIV state.
        formatMPDUFSMNs = FD_EXTIV;

    FD_INITCRYPTO:
      //$fsm_s In FD_INITCRYPTO state, the state machine waits completion of crypto Engine initialization
      if (cryptoInitDone_p && !fcsBusy) 
        //$fsm_t When cryptoInitDone_p is received, the state machine goes to FD_MPDUSTART state.
        formatMPDUFSMNs = FD_MPDUSTART;
      else
        //$fsm_t While cipher init is not done, the state machine stays in FD_INITCRYPTO state.
        formatMPDUFSMNs = FD_INITCRYPTO;
        
        
    `ifdef RW_WAPI_EN
    FD_WAPI_KEYIDX:
    //$fsm_s In FD_WAPI_KEYIDX state, the Tx Controller sends the Key Index of the WAPI encrypted DATA Frame 
      if (countDone_p) 
        //$fsm_t When countDone_p is received, the state machine moves to FW_WAPI_PN state 
        formatMPDUFSMNs = FD_WAPI_PN;
      else
        //$fsm_t While countDone_p is not received, the state machine stays in FD_WAPI_KEYIDX state
        formatMPDUFSMNs = FD_WAPI_KEYIDX;
        
    FD_WAPI_PN:
      //TODO check against other encryption protocol for going into MPDUSTART state
      //$fsm_s FD_WAPI_PN state, the Tx Controller sends the Packet Number of the WAPI encrypted DATA frame.
      if (countDone_p)
        //$fsm_t When countDone_p is received, the state machine moves to FD_INITCRYPTO state 
        formatMPDUFSMNs = FD_INITCRYPTO;
      else
        //$fsm_t While countDone_p is low the state machine stays in FD_WAPI_PN state.
        formatMPDUFSMNs = FD_WAPI_PN;
    `endif //RW_WAPI_EN
    
    FD_MPDUSTART:
      //$fsm_s In FD_MPDUSTART state, the FSM waits for mpduTxStart_p,
      //thus avoid the tx end pulse before the rising edge of tx req,
      //ampdu can ignored the pulse.

      if ((aMPDU && !txSMPDU) || dontGenerateMH || mpIfTxEn)
      begin
         if (isBeacon || isProbeResponse || isTimingAdvertisement)
            //$fsm_t If mpIfTxEn and is a Beacon, a Probe Response or a Timing Advertisement then goes to TSF state.
            formatMPDUFSMNs = FD_TSF;
         else if ((typeTx == 2'b01) && ((subtypeTx == 4'b1010) ||  // PS-POLL
                                        (subtypeTx == 4'b1011) ||  // RTS
                                        (subtypeTx == 4'b1100) ||  // CTS
                                        (subtypeTx == 4'b1101) ||  // ACK
                                        (subtypeTx == 4'b1110) ||  // CF-END
                                        (subtypeTx == 4'b1111)))   // CF-END + CF-ACK
         //$fsm_t If mpIfTxEn and the frame is control frame part of (PS-POLL, RTS, CTS, ACK, CF-END or CF-END + CF-ACK) then goes to FCS state.
            formatMPDUFSMNs = FD_FCS;
         else if(isNDP)
            //$fsm_t If mpIfTxEn and is a NDP then goes to IDLE state.
            formatMPDUFSMNs = FD_IDLE;            
         else if (readGoodByteCnt >= MPDUFrameLengthTx - 16'd4)
            //$fsm_t If mpIfTxEn and others cases then goes to FD_FRM_BODY
            formatMPDUFSMNs = FD_FCS;
         else
            //$fsm_t If mpIfTxEn and others cases then goes to FD_FRM_BODY
            formatMPDUFSMNs = FD_FRM_BODY;
      end
      else
         //$fsm_t If not mpIfTxEn, then wait in  FD_MPDUSTART State
         formatMPDUFSMNs = FD_MPDUSTART;

    FD_FRM_BODY:
      //$fsm_s FD_FRM_BODY state, the Tx Controller sends the Frame Body field of the DATA frame.
      if (countDone_p) 
      begin
        // If countDone_p is received, the state machine moves to another state. However, this next state depends on
        // the encryption type.
        // 
        //  1- If encryption type is WEP, TKIP, CCMP, GCMP or WAPI, 
        //     the next state is FD_WAIT_ICVMIC_DONE to process the ICV or MIC field.
        //
        //  2- Else the next state is FD_FCS to process the FCS field.
        //
        if(protectedFrame && (fieldBytes >= 16'd0))
        begin
          case (cTypeKSR)
            ENCRYPTION_TYPE_WEP,
            ENCRYPTION_TYPE_TKIP,
            ENCRYPTION_TYPE_CCMP,
          `ifdef RW_WAPI_EN
            ENCRYPTION_TYPE_WAPI,
          `endif //RW_WAPI_EN
            ENCRYPTION_TYPE_GCMP:
              //$fsm_t If countDone_p is received  meaning that the Frame Body field has been completely sent, 
              //and the frame is encrypted in WEP/TKIP,CCMP,WAPI or GCMP, the state machine goes to FD_WAIT_ICVMIC_DONE state.
              formatMPDUFSMNs = FD_WAIT_ICVMIC_DONE; 

            default :  
              //$fsm_t If countDone_p is received  meaning that the Frame Body field has been completely sent,  
              //and the cipher type is neither WEP, TKIP, CCMP, GCMP nor WAPI,then the state machine goes to FD_FCS state.
              formatMPDUFSMNs = FD_FCS; 
          endcase
        end
        else
          //$fsm_t If countDone_p is received  meaning that the Frame Body field has been completely sent,  
          //and the frame is not ecnrypted,then the state machine goes to FD_FCS state.
          formatMPDUFSMNs = FD_FCS;
      end
      else
        //$fsm_t While countDone_p is low meaning that the Frame Body field has not been yet completely sent, 
        //the state machine stays in FD_FRM_BODY state.
        formatMPDUFSMNs = FD_FRM_BODY;

    FD_WAIT_ICVMIC_DONE:
      //$fsm_s In FD_WAIT_ICVMIC_DONE state, the Tx Controller wait for Encryption Enginge to finish computing the ICV or MIC.
      if (txEncrDataEnd_p && txEncrDataValid && txEncrDataReady)
        //$fsm_t If txEncrDataEnd_p && txEncrDataValid is received from the encryptionEngine 
        // meaning that the ICV or MIC field has been completely sent, the state machine goes to FD_FCS state.
        formatMPDUFSMNs = FD_FCS;
      else
        //$fsm_t While txEncrDataEnd_p is not received, then the state waits in FD_WAIT_ICVMIC_DONE state.
        formatMPDUFSMNs = FD_WAIT_ICVMIC_DONE;

    FD_FCS:
      //$fsm_s In FD_FCS state, the state machine trigs the FCS block to shift out the FCS value and wait until its completion.
      if (fcsEnd_p) 
        if (endOfMPDUinTxFifo || endOfMPDUinTxFifo_fcs)
          //$fsm_t When countDone_p is received meaning that the FCS has been completely sent and the complete MPDU has been read from the TX FIFO
          //, the state machine goes back to FD_IDLE state
          formatMPDUFSMNs = FD_IDLE; 
        else  
          //$fsm_t When countDone_p is received but the last byte of MPDU indication has not been received,the state machine goes to FD_FLUSHTXFIFO state
          // to flush the TX FIFO.
          formatMPDUFSMNs = FD_FLUSHTXFIFO; 
      else
        //$fsm_t While countDone_p is not received, the state machine stays in FD_FCS state
        formatMPDUFSMNs = FD_FCS;

    FD_FLUSHTXFIFO:
      //$fsm_s In FD_FLUSHTXFIFO state, the state machine reads the TX Fifo until it gets the last byte of MPDU indication. This state is 
      // reached  in two cases
      //   1- Request from MAC Controller to skip the MPDU stored in TX FIFO. 
      //   2- length mistmatch between the number of bytes in the TX FIFO and the MPDU Length defined.
      if (endOfMPDUinTxFifo) 
        //$fsm_t When endOfMPDUinTxFifo is received meaning that the TX FIFO has been completely flushed, the state machine goes back to FD_IDLE state
        formatMPDUFSMNs = FD_IDLE; 
      else
        //$fsm_t While endOfMPDUinTxFifo is not received, the state machine stays in FD_FLUSHTXFIFO state
        formatMPDUFSMNs = FD_FLUSHTXFIFO;

     // Disable coverage on the default state because it cannot be reached.
     // pragma coverage block = off 
     default:   
        formatMPDUFSMNs = FD_IDLE; 
     // pragma coverage block = on 

  endcase
end


// Pulse indication CS is IDLE
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    CSIsIdle <= 1'b0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
    CSIsIdle <= 1'b0; 
  else if(enable) 
    if((formatMPDUFSMCs != FD_IDLE) && (formatMPDUFSMNs == FD_IDLE))
      CSIsIdle <= 1'b1; 
end

// Transmitted frame Type and SubType parsing.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    typeTx      <= 2'b0;
    subtypeTx   <= 4'b0;
    typeTxValid <= 1'b0;
  end
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
  begin
    typeTx      <= 2'b0;
    subtypeTx   <= 4'b0;
    typeTxValid <= 1'b0;
  end
  else if(enable)
  begin
    if (formatMPDUFSMCs == FD_FRMCNTRL_BYTE1)
      if (txFIFODataValid)
      begin
        typeTx      <= txFIFORdData[3:2];
        subtypeTx   <= txFIFORdData[7:4];
        typeTxValid <= 1'b1;
      end  
  end
end  

// Indicates that the transmitted frame is a CF-END or a CF-END+CF-ACK
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    sentCFEND <= 1'b0;
  else
  begin
    if ({typeTx,subtypeTx[3:1]} == 5'b01111)
      sentCFEND <= 1'b1;
    else  
      sentCFEND <= 1'b0;
  end
end  

// Indicates that the transmitted frame is a RTS
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    sentRTS <= 1'b0;
  else
  begin
    if ({typeTx,subtypeTx} == 6'b011011)
      sentRTS <= 1'b1;
    else if (({typeTx,subtypeTx} == 6'b010111) && ({carriedFrameType,carriedFrameSubType} == 6'b011011))
      sentRTS <= 1'b1;
    else  
      sentRTS <= 1'b0;
  end
end  

`ifdef RW_BFMER_EN
// Indicates that the transmitted frame is a BFR-POLL
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    sentBFRPOLL <= 1'b0;
  else
  begin
    if ({typeTx,subtypeTx} == 6'b010100)
      sentBFRPOLL <= 1'b1;
    else if (({typeTx,subtypeTx} == 6'b010100) && ({carriedFrameType,carriedFrameSubType} == 6'b010100))
      sentBFRPOLL <= 1'b1;
    else  
      sentBFRPOLL <= 1'b0;
  end
end  
`endif // RW_BFMER_EN

// Indicates that the transmitted frame is a BA request
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    sentBAReq <= 1'b0;
  else
  begin
    if ({typeTx,subtypeTx} == 6'b011000)
      sentBAReq <= 1'b1;
    else if (({typeTx,subtypeTx} == 6'b010111) && ({carriedFrameType,carriedFrameSubType} == 6'b011000))
      sentBAReq <= 1'b1;
    else  
      sentBAReq <= 1'b0;
  end
end  

// Transmitted frame toDS, fromDS, protectedFrame and order parsing.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    toDS            <= 1'b0;
    fromDS          <= 1'b0;
    moreFrag        <= 1'b0;
    txRetry         <= 1'b0;
    txPwrMgt        <= 1'b0;
    moreData        <= 1'b0;
    order           <= 1'b0;
    protectedFrame  <= 1'b0;
  end
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
  begin
    toDS            <= 1'b0;
    fromDS          <= 1'b0;
    order           <= 1'b0;
    moreFrag        <= 1'b0;
    txRetry         <= 1'b0;
    txPwrMgt        <= 1'b0;
    moreData        <= 1'b0;
    protectedFrame  <= 1'b0;
  end
  else if(enable)
  begin
    if (formatMPDUFSMCs == FD_FRMCNTRL_BYTE2)
      if (txFIFODataValid)
      begin
        toDS            <= txFIFORdData[0];
        fromDS          <= txFIFORdData[1];
        moreFrag        <= txFIFORdData[2];
        moreData        <= txFIFORdData[5];
        order           <= txFIFORdData[7];
        if ((cTypeKSR == ENCRYPTION_TYPE_NONE) || dontEncrypt)
          protectedFrame  <= 1'b0;
        else
        begin
          case({typeTx,subtypeTx})
             6'b100000  :  protectedFrame  <= txFIFORdData[6];//DATA
             6'b100001  :  protectedFrame  <= txFIFORdData[6];//DATA+CF-ACK
             6'b100010  :  protectedFrame  <= txFIFORdData[6];//DATA+CF-POLL
             6'b100011  :  protectedFrame  <= txFIFORdData[6];//DATA+CF-ACK+CF-POLL
//           6'b100100  :  protectedFrame  <= txFIFORdData[6];//Null
             6'b101000  :  protectedFrame  <= txFIFORdData[6];//QoS DATA
             6'b101001  :  protectedFrame  <= txFIFORdData[6];//QoS DATA+CF-ACK
             6'b101010  :  protectedFrame  <= txFIFORdData[6];//QoS DATA+CF-POLL
             6'b101011  :  protectedFrame  <= txFIFORdData[6];//QoS DATA+CF-ACK+CF-POLL
//           6'b101100  :  protectedFrame  <= txFIFORdData[6];//QoS Null
                
             6'b001010  :  protectedFrame  <= txFIFORdData[6];//Disassociation only CCMP
             6'b001011  :  protectedFrame  <= txFIFORdData[6];//Authentification only WEP
             6'b001100  :  protectedFrame  <= txFIFORdData[6];//Deauthentification only CCMP
             6'b001101  :  protectedFrame  <= txFIFORdData[6];//Action only CCMP
             6'b001110  :  protectedFrame  <= txFIFORdData[6];//Action No Ack only CCMP
             default    :  protectedFrame  <= 1'b0;
           endcase
         end

        if (dontGenerateMH)
        begin
          txRetry   <= txFIFORdData[3];
          txPwrMgt  <= txFIFORdData[4];
        end  
        else  
        begin
          // The retry bit of the Frame Control is forced to 0 in case of Control Frame
          // otherwise it comes from the macController. 
          if (typeTx == 2'b01)
            txRetry   <= 1'b0;
          else  
            txRetry   <= retry;

          // The pwrMgt bit of the Frame Control is set 
          // the pwrMgt register is set or 
          // the MAC Header formatted by the SW has the pwrMgt bit set
          txPwrMgt  <= pwrMgt || txFIFORdData[4];
        end  
      end  
  end
end  

// Transmitted carried frame type and subtype
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    carriedFrameType    <= 2'd0;
    carriedFrameSubType <= 4'd0;
  end
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
  begin
    carriedFrameType    <= 2'd0;
    carriedFrameSubType <= 4'd0;
  end
  else if(enable)
  begin
    if (formatMPDUFSMCs == FD_CFCNTRL)
      if (txFIFODataValid && (readByteCnt_ff1 == fieldOffset))
      begin
        carriedFrameType    <= txFIFORdData[3:2];
        carriedFrameSubType <= txFIFORdData[7:4];
      end  
  end
end  


// Indicate the presence of extIV field in the MAC Header
assign extIV = ((cTypeKSR == ENCRYPTION_TYPE_TKIP) || (cTypeKSR == ENCRYPTION_TYPE_CCMP) || (cTypeKSR == ENCRYPTION_TYPE_GCMP)) ? 1'b1 : 1'b0;

// Transmitted frame TID parsing.
// This process extract the TID information from the last byte of the QoS field.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    txTID            <= 4'b0;
    txAMSDUPresent   <= 1'b0;
  end
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
  begin
    txTID            <= 4'b0;
    txAMSDUPresent   <= 1'b0;
  end  
  else if(enable)
  begin
    if (formatMPDUFSMCs == FD_STARTTX)
    begin
      case(activeAC)
        3'b011: txTID    <= 4'd1;
        3'b010: txTID    <= 4'd0;
        3'b001: txTID    <= 4'd4;
        3'b000: txTID    <= 4'd6;
       default: txTID    <= 4'd0;
      endcase
      txAMSDUPresent    <= 1'b0;
    end    
    else if (formatMPDUFSMCs == FD_QOSCNTRL)
    begin
      if (txFIFODataValid && (readByteCnt_ff1 == fieldOffset))
      begin
          txTID           <= txFIFORdData[3:0];
          txAMSDUPresent  <= txFIFORdData[7];
      end
    end
  end
end  

// Transmitted address type.
// This process extract the BroadCast/MultiCast UniCast information from the first byte of the ADDR1 field.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txBcMc           <= 1'b0;
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    txBcMc           <= 1'b0;
  else if(enable)
  begin
    if (formatMPDUFSMCs == FD_STARTTX)
      txBcMc           <= 1'b0;
    else if (formatMPDUFSMCs == FD_ADDR1)
    begin
      if (txFIFODataValid && (readByteCnt_ff1 == fieldOffset))
        txBcMc  <= txFIFORdData[0];
    end
  end
end  

// Type of transmitted frames
// The frame is a Control Wrapper Frame when typeTx is 1 (Control) and sybtypeTx is 7
assign controlWrapperFrame = ((typeTx == 2'b01) && (subtypeTx == 4'b0111)) ? 1'b1 : 1'b0;

// The frame is a QoS Frame when typeTx is 2 (Data) and sybtypeTx bigger or equal to 8
assign qosFrame = ((typeTx == 2'b10) && subtypeTx[3]) ? 1'b1 : 1'b0;

// The frame is an HTC Frame is this is a ControlWrapperFrame or a QoS or management frame with the order bit set.
assign htcFrame = (((qosFrame || (typeTx == 2'b00)) && order) || (controlWrapperFrame)) ? 1'b1 : 1'b0;

// The frame is a Beacon Frame when typeTx is 0 (Management) and sybtypeTx is 8
assign isBeacon = ((typeTx == 2'b00) && (subtypeTx == 4'b1000)) ? 1'b1 : 1'b0;

// The frame is a Probe Response Frame when typeTx is 0 (Management) and sybtypeTx is 5
assign isProbeResponse = ((typeTx == 2'b00) && (subtypeTx == 4'b0101)) ? 1'b1 : 1'b0;

// The frame is a Timing Advertisement Frame when typeTx is 0 (Management) and sybtypeTx is 6
assign isTimingAdvertisement = ((typeTx == 2'b00) && (subtypeTx == 4'b0110)) ? 1'b1 : 1'b0;

// The frame is a Null Data Packet when frame length is 0
assign isNDP = (MPDUFrameLengthTx == 16'd0) ? 1'b1 : 1'b0;

assign sentNDP = isNDP;

// This combinational process assigns the number of bytes contained into the current field as well as the 
// offset of the current field in the TX Header Descriptor.
// Note that fieldBytes is 1 byte less than the number of byte of the field because
// it is used as reference for the txByteCnt counter which starts at 0.
always @ *
begin
  case (formatMPDUFSMCs)
  
    FD_FRMCNTRL_BYTE1 :
      begin 
        fieldBytes      = 16'd0;
        fieldOffset     = FRMCNTRL_BYTE1_OFFSET;
      end
  
    FD_FRMCNTRL_BYTE2 :
      begin 
        fieldBytes      = 16'd0;
        fieldOffset     = FRMCNTRL_BYTE2_OFFSET;
      end
  
    FD_DURID_DATA :
      begin 
        fieldBytes      = DURID_BYTECNT - 16'd1;
        fieldOffset     = DURID_OFFSET;
      end
      
    FD_ADDR1 : 
      begin 
        fieldBytes      = ADDR_BYTECNT - 16'd1;
        fieldOffset     = ADDR1_OFFSET;
      end
    
    FD_ADDR2 : 
      begin 
        fieldBytes      = ADDR_BYTECNT - 16'd1;
        fieldOffset     = ADDR2_OFFSET;
      end

    FD_ADDR3 : 
      begin 
        fieldBytes      = ADDR_BYTECNT - 16'd1;
        fieldOffset     = ADDR3_OFFSET;
      end

    FD_ADDR4 : 
      begin 
        fieldBytes      = ADDR_BYTECNT - 16'd1;
        fieldOffset     = ADDR4_OFFSET;
      end

    FD_SQNCNTRL : 
      begin 
        fieldBytes      = SQNCNTRL_BYTECNT - 16'd1;
        fieldOffset     = SQNCNTRL_OFFSET;
      end

    FD_QOSCNTRL : 
      begin 
        fieldBytes      = QOSCNTRL_BYTECNT - 16'd1;
        fieldOffset     = QOSCNTRL_OFFSET;
      end

    FD_CFCNTRL : 
      begin 
        fieldBytes      = CFCNTRL_BYTECNT - 16'd1;
        fieldOffset     = CFCNTRL_OFFSET;
      end

    FD_HTCNTRL : 
      begin 
        fieldBytes      = HTCNTRL_BYTECNT - 16'd1;
        fieldOffset     = HTCNTRL_OFFSET;
      end

    FD_TSF : 
      begin 
        fieldBytes      = TSF_BYTECNT - 16'd1;
        fieldOffset     = TSF_OFFSET;
      end

    FD_IV : 
      begin 
        fieldBytes      = IV_BYTECNT - 16'd1;
        fieldOffset     = IV_OFFSET;
      end

    FD_EXTIV : 
      begin 
        fieldBytes      = EXTIV_BYTECNT - 16'd1;
        fieldOffset     = EXTIV_OFFSET;
      end

     `ifdef RW_WAPI_EN
     FD_WAPI_KEYIDX :
       begin
         fieldBytes      = WAPI_KEYIDX_BYTECNT - 16'd1;
         fieldOffset     = WAPI_KEYIDX_OFFSET;
       end 
     FD_WAPI_PN :
      begin
         fieldBytes      = WAPI_PN_BYTECNT - 16'd1;
         fieldOffset     = WAPI_PN_OFFSET;
      end 
    `endif //RW_WAPI_EN       
   
    FD_FRM_BODY :
      begin 
        fieldBytes = (frameBodyLength > 16'd0) ? frameBodyLength - 16'd1 : 16'd0;
        if (dontGenerateMH)
          fieldOffset   = 16'b0;
        else
        begin
          if (isBeacon || isProbeResponse || isTimingAdvertisement)
            fieldOffset = FRM_BODY_OFFSET + 16'd8;
          else  
          `ifdef RW_WAPI_EN
            fieldOffset = (txWAPI) ? FRM_BODY_OFFSET + 16'd10 : FRM_BODY_OFFSET;
          `else
            fieldOffset = FRM_BODY_OFFSET;
          `endif
        end  
      end
  
    FD_WAIT_ICVMIC_DONE : 
      begin 
        if (txWEP || txTKIP)
          fieldBytes    = ICV_BYTECNT - 16'd1;
        else if (txCCMP && cLenKSR==2'b00)
          fieldBytes    = CCMP128MIC_BYTECNT - 16'd1;
        else if (txCCMP)
          fieldBytes    = CCMP256MIC_BYTECNT - 16'd1;
    `ifdef RW_WAPI_EN
        else if (txWAPI)
          fieldBytes    = WAPIMIC_BYTECNT - 16'd1;
    `endif //RW_WAPI_EN
        else
          fieldBytes    = GCMPMIC_BYTECNT - 16'd1;
        fieldOffset   = frameBodyLength + FRM_BODY_OFFSET;
      end
    
    FD_FCS : 
      begin 
        fieldBytes    = FCS_BYTECNT - 16'd1;    
        if (dontGenerateMH)
          fieldOffset = frameBodyLength;
        else
        begin
          if(txWEP || txTKIP)
            fieldOffset = frameBodyLength + ICV_BYTECNT + FCS_OFFSET;
            else if(txCCMP && cLenKSR==2'b00)
              fieldOffset = frameBodyLength + CCMP128MIC_BYTECNT + FCS_OFFSET;
            else if(txCCMP && cLenKSR==2'b00)
              fieldOffset = frameBodyLength + CCMP128MIC_BYTECNT + FCS_OFFSET;
            else if(txCCMP)
              fieldOffset = frameBodyLength + CCMP256MIC_BYTECNT + FCS_OFFSET;
            else if(txGCMP)
              fieldOffset = frameBodyLength + GCMPMIC_BYTECNT + FCS_OFFSET;
          `ifdef RW_WAPI_EN
          else if(txWAPI)
            fieldOffset = frameBodyLength + WAPIMIC_BYTECNT + FCS_OFFSET;
          `endif // RW_WAPI_EN
          else
            fieldOffset = frameBodyLength + FCS_OFFSET;
        end
      end

    default :
      begin
        fieldBytes  = 16'd0;
        fieldOffset = 16'd0;
      end  

  endcase   
end

// This process computes the the frame body field length.
// When the first byte of the frame Body is popped from the TX FIFO, the Frame Body length is:
// MPDU Frame Length - number of byte already transmitted - the FCS byte count(4).
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    frameBodyLength      <= 16'b0; 
  end
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
  begin
    frameBodyLength      <= 16'b0; 
  end
  else if(enable)
  begin
    if((formatMPDUFSMCs == FD_MPDUSTART) && (formatMPDUFSMNs != FD_MPDUSTART))
    begin
      if(txWEP || txTKIP)
        frameBodyLength <= MPDUFrameLengthTx - readGoodByteCnt - 16'd8;  // WEP : ICV (4 bytes) + FCS (4 bytes)
      else if(txCCMP && cLenKSR==2'b0 && (txDataLength > 16'h0))
        frameBodyLength <= MPDUFrameLengthTx - readGoodByteCnt - 16'd12; // CCMP: MIC (8 bytes) + FCS (4 bytes)
      else if(txGCMP && protectedFrame && (txDataLength > 16'h0))
        frameBodyLength <= MPDUFrameLengthTx - readGoodByteCnt - 16'd20; // GCMP: MIC (16 bytes) + FCS (4 bytes)
      else if(txCCMP && (txDataLength > 16'h0))
        frameBodyLength <= MPDUFrameLengthTx - readGoodByteCnt - 16'd20; // CCMP: MIC (16 bytes) + FCS (4 bytes)
      else if(txGCMP && (txDataLength > 16'h0))
        frameBodyLength <= MPDUFrameLengthTx - readGoodByteCnt - 16'd20; // GCMP: MIC (16 bytes) + FCS (4 bytes)
      `ifdef RW_WAPI_EN
      else if(txWAPI && (txDataLength > 16'h0))
        frameBodyLength <= MPDUFrameLengthTx - readGoodByteCnt - 16'd20; // WAPI: MIC (16 bytes) + FCS (4 bytes)
      `endif
      else if (isBeacon || isProbeResponse || isTimingAdvertisement)
        frameBodyLength <= MPDUFrameLengthTx - readGoodByteCnt - 16'd12; // TSF + FCS (4 bytes) 
      else
        frameBodyLength <= MPDUFrameLengthTx - readGoodByteCnt - 16'd4; // FCS (4 bytes) 
    end
  end
end

// Enable the Read process
// When the state machine reaches the FD_STARTTX state, the TX FIFO Read accesses are enabled.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
   txFIFOReadEn       <= 1'b0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
    txFIFOReadEn      <= 1'b0; 
  else if(enable)
  begin
    if ((formatMPDUFSMNs == FD_STARTTX) || (formatMPDUFSMNs == FD_FLUSHTXFIFO)) 
      txFIFOReadEn    <= 1'b1;
  end  
end

// Assign TX FIFO Read Enable
// The TX FIFO is read when :
// - The TX FIFO Read is enabled
// - and the end of MPDU is not yet reach in the TX FIFO (MPDU Delimiter != 2)
// - and the TX FIFO is not empty (!txFIFOEmpty)
// - and the FCS Engine is not busy or we are in Flush mode or we are shifting out the FCS signature.
assign txFIFORead = (txFIFOReadEn && !endOfMPDUinTxFifo && !endOfMPDUinTxFifo_fcs && !txFIFOEmpty && !currentMPDUDone_p && (MPDUFrameLengthTx != 16'd0)) &&
                    (!(((formatMPDUFSMCs != FD_FRM_BODY) || !protectedFrame) && fcsBusy) || (formatMPDUFSMCs == FD_FLUSHTXFIFO))                         &&
                    (formatMPDUFSMCs != FD_FCS)                                       &&
                    !formatMPDUBusy                                                   &&
                    !cryptoEngineBusy;
        
// Indicate that the formatMPDU FSM cannot accept new byte because it is busy in internal state (or shift the FCS).
assign formatMPDUBusy = (formatMPDUFSMNs == FD_MPDUSTART)                                  ||
                        (formatMPDUFSMCs == FD_MPDUSTART)                                  ||
                        (formatMPDUFSMCs == FD_WAIT_ICVMIC_DONE)                           ||
                        ((formatMPDUFSMCs != FD_FRM_BODY) && bufferCnt == 2'd2 && fcsBusy) ||                          
                        ((formatMPDUFSMCs == FD_FRM_BODY) && countDone_p);                          

// Indicate that the crypto Engine is currently busy and cannot accept new data.
// The Crypto Engine is busy 
//   - during FRAME Body encryption if txPlainDataReady is low
//   - During initialization phase.
assign cryptoEngineBusy = ((formatMPDUFSMCs == FD_FRM_BODY) && protectedFrame && !txPlainDataReady) ||
                          (formatMPDUFSMCs == FD_INITCRYPTO)                                        ||
                          ((formatMPDUFSMCs == FD_EXTIV) &&  countDone_p)                           ||
                          ((formatMPDUFSMCs == FD_IV) && txWEP && countDone_p)                      ||
                          ((formatMPDUFSMCs == FD_WAPI_PN) && countDone_p);


// Indicate that the last byte of the payload has been fetched
// This flag is written in the readDataBank Buffers along with the associated fetched data from the txFIFO.
// Then, when the crypto Engine will read the data, the lastPayloadByte flag will be provided to generate 
// the txPlainDataEnd indication. 
assign lastPayloadByte = (formatMPDUFSMCs == FD_FRM_BODY) && countDone_p;


always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    readDataBank2     <= 9'b0; 
    readDataBank1     <= 9'b0; 
    readDataBank0     <= 9'b0; 
  end
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
  begin
    readDataBank2     <= 9'b0; 
    readDataBank1     <= 9'b0; 
    readDataBank0     <= 9'b0; 
  end
  else if(enable)
  begin
    if (txFIFODataValid)
    begin
      case (writePointer)
            2'd0 : readDataBank0 <= {lastPayloadByte,txData};
            2'd1 : readDataBank1 <= {lastPayloadByte,txData};
         default : readDataBank2 <= {lastPayloadByte,txData};
      endcase   
    end    
  end    
end

// Add the TX Delay in PHY and RF to transmit the TSF 
// The timestamp in the beacon should be the TSF contents at the point
// of transmission of the first bit of TSF in the air.
//assign tsfAntenna = tsf + tsfDuration
// tsfDuration is the duration between the beginning of the transmission until the first bit of the TSF field
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    tsfAntenna     <= 64'b0; 
  end
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (formatMPDUFSMCs == FD_IDLE))  // Synchronous Reset
  begin
    tsfAntenna     <= 64'b0; 
  end
  else
  begin
    if (mpduTxStart_p)
      tsfAntenna <= tsf + {48'b0,tsfDuration};
  end    
end

// Compute the offset, from the first byte of the Beacon frame, 
// at which the DTIM field is located.
assign dtimOffset = {8'h0,timOffset} + 16'd2;

// This combinational process patches the MAC Header fields, the TSF and DTIM fields is required.
always @ *
begin
  case (formatMPDUFSMCs)

    FD_FRMCNTRL_BYTE2 :
      begin 

        if (dontTouchFC || dontGenerateMH)
          txData      = txFIFORdData;
        else 
        begin
          // use +THC, Protected and More Data bits from Tx Fifo
          txData[7:5]  = txFIFORdData[7:5]; 

          // If pwrMgt register is set, overwrite the Power Management bit
          // Else use the content of Tx Fifo
          if (pwrMgt)
            txData[4] = 1'b1; 
          else
            txData[4] = txFIFORdData[4];
          
          // If frame type is Control, force the retry bit to 0
          // Else use the retry information coming from macController.
          if (typeTx == 2'b01)
            txData[3] = 1'b0;
          else
            txData[3] = retry;

          // Use More-Frag, From Ds and To DS bits from Tx Fifo
          txData[2:0]  = txFIFORdData[2:0]; 
        end  
      end
  
    FD_DURID_DATA :
      begin 
        if (dontTouchDur)
          txData      = txFIFORdData;
        else if (sentCFEND)
          txData      = 8'b0;
        else
        begin
          case (readFieldByteCnt[0])
            1'b0 : txData = duration[7:0];
            1'b1 : txData = duration[15:8]; 
          endcase
        end
      end

    FD_TSF :
      begin 
        if (dontTouchTSF)
          txData      = txFIFORdData;
        else
        begin
          case (readFieldByteCnt)
            16'h00     : txData = tsfAntenna[7:0];
            16'h01     : txData = tsfAntenna[15:8]; 
            16'h02     : txData = tsfAntenna[23:16]; 
            16'h03     : txData = tsfAntenna[31:24]; 
            16'h04     : txData = tsfAntenna[39:32];
            16'h05     : txData = tsfAntenna[47:40]; 
            16'h06     : txData = tsfAntenna[55:48]; 
            16'h07     : txData = tsfAntenna[63:56]; 
            // Disable coverage on the default state because it cannot be reached.
            // pragma coverage block = off 
            default    : txData = txFIFORdData;
            // pragma coverage block = on
          endcase
        end
      end

    FD_QOSCNTRL :
      begin
        txData = txFIFORdData;
      end

    FD_FRM_BODY :
      begin 
        // If the dontTouchDTIM bit is not see and the frame is a Beacon, the DTIM field has to be updated
        if (!dontTouchDTIM && isBeacon)
        begin
          // When the number of read good bytes reaches the dtimOffset (i.e. timOffset from register plus 2) 
          // the txData is updated by the dtimCnt comming from the Timers module.
          if (readGoodByteCnt == dtimOffset)
            txData = dtimCnt;
          else  
            txData = txFIFORdData;
        end
        else
          txData = txFIFORdData;
            
      end

    default:
      txData  = txFIFORdData;

  endcase
end  



// Detects the end of the MPDU in the TX FIFO based on the MPDU Delimiters.
// When the last byte of the MPDU has been read from the TX FIFO, the flag endOfMPDUinTxFifo is set
// and the read accesses from the TX FIFO are stopped.
assign endOfMPDUinTxFifo = (txFIFODataValid && (txFIFOMPDUDelimiters == 2'b10));

always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)
    endOfMPDUinTxFifo_fcs <= 1'b0;
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))
    endOfMPDUinTxFifo_fcs <= 1'b0;
  else if(enable)
  begin
    if((formatMPDUFSMCs != FD_FLUSHTXFIFO) && endOfMPDUinTxFifo)
      endOfMPDUinTxFifo_fcs <= 1'b1;
  end
end

always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    writePointer     <= 2'b0; 
    readPointer      <= 2'b0; 
  end
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
  begin
    writePointer     <= 2'b0; 
    readPointer      <= 2'b0; 
  end
  else if(enable)
  begin
    if (txFIFODataValid)
    begin
      if (writePointer == 2'd2)
        writePointer      <= 2'd0; 
      else
        writePointer      <= writePointer + 2'd1; 
    end
    if ((!encryptionOnGoing && mpduFCSDInValidTx) || (txPlainDataValid && txPlainDataReady))
    begin
      if (readPointer == 2'd2)
        readPointer      <= 2'd0; 
      else
        readPointer      <= readPointer + 2'd1; 
    end
  end
end

// Count the number of byte stored in the Buffer
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    bufferCnt <= 2'b0;
//  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1) || (formatMPDUFSMCs == FD_INITCRYPTO) || (formatMPDUFSMCs == FD_FLUSHTXFIFO)) // Synchronous Reset
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1) || (formatMPDUFSMCs == FD_FLUSHTXFIFO)) // Synchronous Reset
    bufferCnt <= 2'b0;
  else if (txFIFODataValid && !((!encryptionOnGoing && mpduFCSDInValidTx) || (txPlainDataValid && txPlainDataReady)))
    bufferCnt <= bufferCnt + 2'b1;
  else if (!txFIFODataValid && ((!encryptionOnGoing && mpduFCSDInValidTx) || (txPlainDataValid && txPlainDataReady)) && (bufferCnt != 2'b0))
    bufferCnt <= bufferCnt - 2'b1;
end


assign nextcountdone_p = ((readByteCnt_ff1 == fieldBytes + fieldOffset - 16'd1) && (txFIFODataValid)  && (!fcsBusy)) ||
                         ((readByteCnt_ff1 == fieldBytes + fieldOffset)         && (!txFIFODataValid) && (!fcsBusy));

always @(*)
begin
  case(formatMPDUFSMCs)
    FD_ADDR1:
    begin
      if(nextcountdone_p)
      begin
        if(controlWrapperFrame)
          // jump to CFCNTRL
          readByteOffset = CFCNTRL_OFFSET - ADDR1_OFFSET - ADDR_BYTECNT; // 32 - 10 = 12
        else if((typeTx == 2'b01) && ((subtypeTx == 4'b1100) || (subtypeTx == 4'b1101)))
          // jump to MPDU START
          readByteOffset = 16'd46 - ADDR1_OFFSET + ADDR_BYTECNT;
        else
          // jump to ADDR2
          readByteOffset = 16'd0;
      end
      else
        readByteOffset = 16'd0;
    end

    FD_ADDR2:
    begin
      if(nextcountdone_p && (typeTx == 2'b01))
        // jump to MPDU START
        readByteOffset = 16'd46 - ADDR2_OFFSET - ADDR_BYTECNT;
      else
        // jump to ADDR3
        readByteOffset = 16'd0;
    end

    FD_SQNCNTRL:
    begin
      if(nextcountdone_p)
      begin
        if(toDS && fromDS)
          // jump to ADDR4
          readByteOffset = 16'd0;
        else if(qosFrame)
           // jump to QoS
          readByteOffset = QOSCNTRL_OFFSET - SQNCNTRL_OFFSET - SQNCNTRL_BYTECNT; // 30 - 24 = 4
        else if (htcFrame)
          // jump to HTC
          readByteOffset = HTCNTRL_OFFSET  - SQNCNTRL_OFFSET - SQNCNTRL_BYTECNT; // 34 - 24 = 6
        else if (protectedFrame)
          // jump to IV
          readByteOffset = IV_OFFSET       - SQNCNTRL_OFFSET - SQNCNTRL_BYTECNT; // 38 - 24 = 10
        else
          // jump to MPDU START
          readByteOffset = 16'd46          - SQNCNTRL_OFFSET - SQNCNTRL_BYTECNT;
      end
      else
        readByteOffset = 16'd0;
    end

    FD_ADDR4:
    begin
      if(nextcountdone_p)
      begin
        if (qosFrame)
          // jump to QoS
          readByteOffset = 16'd0;
        else if (htcFrame)
          // jump to HTC
          readByteOffset = HTCNTRL_OFFSET  - ADDR4_OFFSET - ADDR_BYTECNT; // 34 - 30 = 4
        else if (protectedFrame)
          // jump to IV
          readByteOffset = IV_OFFSET       - ADDR4_OFFSET - ADDR_BYTECNT; // 38 - 30 = 8
        else
          // jump to MPDU START
          readByteOffset = 16'd46          - ADDR4_OFFSET - ADDR_BYTECNT;
      end
      else
        readByteOffset = 16'd0;
    end

    FD_QOSCNTRL:
    begin
      if (nextcountdone_p)
      begin
        if(htcFrame)
          // jump to HTC
          readByteOffset = HTCNTRL_OFFSET  - QOSCNTRL_OFFSET - QOSCNTRL_BYTECNT;
        else if(protectedFrame)
          // jump to IV
          readByteOffset = IV_OFFSET       - QOSCNTRL_OFFSET - QOSCNTRL_BYTECNT; // 38 - 34 = 4
        else
          // jump to MPDU START
          readByteOffset = 16'd46          - QOSCNTRL_OFFSET - QOSCNTRL_BYTECNT;
      end
      else
        readByteOffset = 16'd0;
    end

    FD_HTCNTRL:
    begin
      if (nextcountdone_p)
      begin
        if(protectedFrame)
          // jump to IV
          readByteOffset = 16'd0;
        else
          // jump to MPDU START
          readByteOffset = 16'd46          - HTCNTRL_OFFSET - HTCNTRL_BYTECNT;
      end
      else
        readByteOffset = 16'd0;
    end

    FD_IV:
    begin
      if (nextcountdone_p)
      begin
        if(txCCMP || txGCMP || txTKIP)
          // jump to Ext IV or Search Key
          readByteOffset = 16'd0;
        else
          // jump to INITCRYPTO or MPDUSTART
          readByteOffset = 16'd46          - IV_OFFSET - IV_BYTECNT;
      end
      else
        readByteOffset = 16'd0;
    end

    default
    begin
      readByteOffset = 16'd0;
    end
  endcase
end

always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    readByteOffset_ff1       <= 6'd0;
  end
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
  begin
    readByteOffset_ff1       <= 6'd0;
  end
  else if(enable)
  begin
    if(nextcountdone_p)
      readByteOffset_ff1     <= readByteOffset[5:0];
    else
      readByteOffset_ff1     <= 6'd0;
  end
end

// delayed the readByteCnt of one clock cycle
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    readByteCnt_ff1       <= 16'd0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
    readByteCnt_ff1       <= 16'd0; 
  else if(enable)
  begin
    if (txFIFODataValid)
    begin
      readByteCnt_ff1       <= readByteCnt_ff1 + 16'd1 + {10'd0,readByteOffset_ff1[5:0]};
    end
  end
end

// Increment the number of bytes read from the TX FIFO.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    readByteCnt       <= 16'b0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
    readByteCnt       <= 16'b0; 
  else if(enable)
  begin
    if (txFIFORead)
    begin
      readByteCnt       <= readByteCnt + 16'd1 + readByteOffset[15:0];
    end
  end
end

// Increment the number of good bytes read from the TX FIFO.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    readGoodByteCnt     <= 16'b0; 
  end
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
  begin
    readGoodByteCnt     <= 16'b0; 
  end
  else if(enable)
  begin
    if (txFIFODataValid)
      if ((readByteCnt_ff1 >= fieldOffset) && (readGoodByteCnt < MPDUFrameLengthTx))
        readGoodByteCnt  <= readGoodByteCnt + 16'd1;

  end
end

// This counter allows to count the number of transmitted bytes
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txByteCnt       <= 16'b0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
    txByteCnt       <= 16'b0; 
  else if(enable)
  begin
    if (fcsEnd_p) 
      txByteCnt       <= txByteCnt + 16'd4;  
    else if (mpduFCSDInValidTx) 
      txByteCnt       <= txByteCnt + 16'd1;  
  end
end

// This counter allows to count the number of bytes read per field
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    readFieldByteCnt       <= 16'b0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
    readFieldByteCnt       <= 16'b0; 
  else if(enable)
  begin
    if (txFIFODataValid) 
      if (readByteCnt_ff1 >= fieldOffset)
        readFieldByteCnt   <= readFieldByteCnt + 16'd1;  
    if (countDone_p) 
      readFieldByteCnt     <= 16'b0;  
  end
end


// The countDone_p indication used to change the state is generated when the readByteCnt_ff1 counter reaches 
// the end of the corresponding field position in the Transmit Header Descriptor. This position is given by 
// the following formula :
//  fieldBytes  : Number of bytes of the current field.
//  fieldOffset : Number of bytes between the begining of the THD and the current field
// Due to the handshaking, this pulse is generated only when the txFIFODataValid is valid.
// At the end of the transmission (FCS field), the countDone_p pulse is not generated when the last byte of the FCS 
// has been read from the Tx FIFO but when this last bytes has been processed by the FCS Engine.
assign countDone_p = (((formatMPDUFSMCs != FD_FCS) && (txFIFODataValid && (readByteCnt_ff1 == fieldOffset + fieldBytes))) || 
                     endOfFrame || fcsEnd_p) ? 1'b1 : 1'b0;

assign endOfFrame = (txByteCnt == MPDUFrameLengthTx) ? 1'b1 : 1'b0;


// currentMPDUDone_p pulses generation indicating the end of the DATA frame transmission
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    currentMPDUDone_p       <= 1'b0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    currentMPDUDone_p       <= 1'b0; 
  else if(enable)
  begin
    // The mpduDone_p pulse is generated when the fcsEnd_p is received
    if ((formatMPDUFSMCs == FD_FCS || formatMPDUFSMCs == FD_FLUSHTXFIFO) && (formatMPDUFSMNs == FD_IDLE))
      currentMPDUDone_p    <= 1'b1; 
    else if((MPDUFrameLengthTx == 16'd0) && (formatMPDUFSMCs == FD_MPDUSTART) && (formatMPDUFSMNs == FD_IDLE))
      currentMPDUDone_p    <= 1'b1; 
  end
end

// SkipMPU request from tx controler logic
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    skipMPDU <= 1'b0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    skipMPDU <= 1'b0; 
  else
  begin
     if(skipMPDU_p)
       skipMPDU <= 1'b1;
     else if(skipMPDUDone_p)
       skipMPDU <= 1'b0;
  end 
end

// When the FSM moves from FD_FLUSHTXFIFO to FD_IDLE, the skipMPDUDone_p pulse indicating the completion fo the MPDU Skip fro the TX FIFO
// is generated.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    skipMPDUDone_p   <= 1'b0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    skipMPDUDone_p   <= 1'b0; 
  else if(enable)
  begin
    if ((formatMPDUFSMCs == FD_FLUSHTXFIFO) && (formatMPDUFSMNs == FD_IDLE) && skipMPDU)
      skipMPDUDone_p <= 1'b1; 
  end
end


// FCS Block Control
////////////////////

// When the state machine is in FD_STARTTX, the FCS Engine is started
assign mpduFCSStart_p   = (formatMPDUFSMCs == FD_STARTTX) ? 1'b1 : 1'b0;


// If dontTouchFCS is not set, the FCS field comes from the FCS engine and not from the TX FIFO. In this case,
// when the number of transmitted bytes reaches the MPDU Frame Length minus 4 (which is the the full frame except the FCS), 
// the FCS engine is requested to shift out the computed signature.
// Note that when mpduFCSShiftTx is set, the Tx Controller does not stop fetching the TX FIFO but stop to send the data 
// to the FCS engine
//assign mpduFCSShiftTx   = ((formatMPDUFSMCs != FD_IDLE) && !dontTouchFCS && (txByteCnt >= MPDUFrameLengthTx - 16'd5) && !fcsEnd_p && ((formatMPDUFSMCs != FD_FRM_BODY))) ? 1'b1 : 1'b0;

//assign mpduFCSShiftTx   = ((formatMPDUFSMCs != FD_IDLE) && !dontTouchFCS && (txByteCnt >= MPDUFrameLengthTx - 16'd5) && !fcsEnd_p && (formatMPDUFSMCs == FD_FCS)) ? 1'b1 : 1'b0;
assign mpduFCSShiftTx   = ((formatMPDUFSMCs != FD_IDLE) && !dontTouchFCS && (bufferCnt == 2'd0) && !fcsEnd_p && (formatMPDUFSMCs == FD_FCS)) ? 1'b1 : 1'b0;



// Generation of encryptionOnGoing flag indicating that the data to be provided to the FCS block comes from the crypto engine and not from
// the internal formatMPDU.v
assign encryptionOnGoing = (protectedFrame                                  &&
`ifdef RW_WAPI_EN
                            (txWEP || txTKIP || txCCMP || txGCMP || txWAPI) &&
`else
                            (txWEP || txTKIP || txCCMP || txGCMP )          &&
`endif
                            ((formatMPDUFSMCs == FD_FRM_BODY) || (formatMPDUFSMCs == FD_WAIT_ICVMIC_DONE))) ? 1'b1 : 1'b0;

// The mpduFCSInValid indicates when the mpduFCSDInTx can be captured by the FCS Engine.
// The mpduFCSDInTx are valid if the number of Good byte read from the TxFifo is bigger than the number of transmitted bytes, 
// the FCS Engine is not busy (fcsBusy low) and we are not shifting the FCS from the FCS Engine.
assign mpduFCSDInValidTx = (encryptionOnGoing) ? txEncrDataValid && !fcsBusy :
                                                 ((txByteCnt < readGoodByteCnt) && (formatMPDUFSMCs != FD_FLUSHTXFIFO) && !fcsBusy) ? 1'b1 : 
                                                                                                                                    1'b0; 
// The mpduFCSDInTx data provided to the FCS Engine comes fom one of the two banks depending on the readPointer.
always @*
begin
  if (encryptionOnGoing)
    mpduFCSDInTx  = txEncrData;
  else
    case (readPointer)
         2'd0 : mpduFCSDInTx  = readDataBank0[7:0];
         2'd1 : mpduFCSDInTx  = readDataBank1[7:0];
      default : mpduFCSDInTx  = readDataBank2[7:0];
    endcase
end 
    
assign mpduFCSEnable = (formatMPDUFSMCs != FD_IDLE);


// MAC-PHY Interface management
// ////////////////////////////

always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    wait4Slot       <= 1'b0; 
  end  
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
  begin
    wait4Slot       <= 1'b0; 
  end  
  else
    if (sendMPDU_p && !sendSWTB && (!aMPDU || txSMPDU))
      wait4Slot      <= 1'b1; 
    else if (mpduTxStart_p)
      wait4Slot      <= 1'b0; 
end

// When the state machine is in FD_WAIT state and the SIFS or SLOT tick indication is received,
// the Transmission is requested. That guaranty to request the transmission on the sifs boundary
assign mpduTxStart_p    = (wait4Slot && (tickSlot || (tickSIFS && sendOnSIFS))) ? 1'b1 : 1'b0;



// Debug Interface management
// ////////////////////////////


always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    fragNumber       <= 4'b0; 
    seqNumber        <= 6'b0; 
  end  
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
  begin
    fragNumber       <= 4'b0; 
    seqNumber        <= 6'b0; 
  end  
  else if(enable)
  begin
    if (formatMPDUFSMCs == FD_SQNCNTRL)
    begin
      case (readFieldByteCnt[0])
        1'b0 : 
          begin
            fragNumber      <= txFIFORdData[3:0];
            seqNumber[3:0]  <= txFIFORdData[7:4]; 
          end  
        1'b1 : 
          seqNumber[5:4]    <= txFIFORdData[1:0]; 
      endcase
    end
  end
end

always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    addr1MSB        <= 8'b0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    addr1MSB        <= 8'b0; 
  else if(enable)
    if ((formatMPDUFSMCs == FD_SQNCNTRL) && (readFieldByteCnt == 16'h05))
      addr1MSB <= txFIFORdData;
end

assign formatMPDUFrameDebug1 = { fragNumber, protectedFrame, txPwrMgt, moreFrag, moreData, txRetry, subtypeTx, typeTx};
assign formatMPDUFrameDebug2 = { seqNumber, 1'b0, addr1MSB};

assign mpduSeqCtrlDone_p = (formatMPDUFSMCs == FD_FRM_BODY) || (formatMPDUFSMCs == FD_FCS);


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// CryptoEngine Control

assign initEncrTxCntrl_p = ((formatMPDUFSMPs != FD_INITCRYPTO) && (formatMPDUFSMCs == FD_INITCRYPTO));

assign txEncrDataReady = !fcsBusy;

// Tx IDLE Flag
assign txCsIsIdle      = (formatMPDUFSMCs == FD_IDLE);

// Plain Data to Encrypt Engine
always @*
begin
  case (readPointer)
       2'd0 : txPlainData  = readDataBank0[7:0];
       2'd1 : txPlainData  = readDataBank1[7:0];
    default : txPlainData  = readDataBank2[7:0];
  endcase
end

// Plain Data end pulse
always @*
begin
  case (readPointer)
       2'd0 : txPlainDataEnd  = readDataBank0[8];
       2'd1 : txPlainDataEnd  = readDataBank1[8];
    default : txPlainDataEnd  = readDataBank2[8];
  endcase
end

reg inPayload;

always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txPlainDataValid <= 1'b0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1) || !inPayload) // Synchronous Reset
    txPlainDataValid <= 1'b0; 
  else if (txPlainDataReady)
  begin
    if ((bufferCnt != 2'b0) && protectedFrame)
      txPlainDataValid <= 1'b1;
    else  
      txPlainDataValid <= 1'b0;
  end
end

always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    inPayload <= 1'b0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    inPayload <= 1'b0; 
  else if (txPlainDataReady && txPlainDataEnd)
    inPayload <= 1'b0;
  else if (formatMPDUFSMCs == FD_MPDUSTART)
    inPayload <= 1'b1;
end
//assign txPlainDataValid = txPlainDataReady;


// CCMP Tx Data length calculation before the CCMP start pulse
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txDataLength  <= 16'h0; 
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1))  // Synchronous Reset
    txDataLength  <= 16'h0;
  else if(enable)
  begin
    if(txGCMP && (formatMPDUFSMCs == FD_EXTIV) && (readByteCnt_ff1 == fieldOffset))
    // encrDataLength     MPDU Length                  Header            KeyIdx   FCS     EXT IV
      txDataLength <= (MPDUFrameLengthTx[15:0] - readGoodByteCnt[15:0] - 16'd16 - 16'd4 - 16'd4);
    else if(txCCMP && cLenKSR!=2'b0 && (formatMPDUFSMCs == FD_EXTIV) && (readByteCnt_ff1 == fieldOffset))
    // encrDataLength     MPDU Length                  Header            KeyIdx   FCS     EXT IV
      txDataLength <= (MPDUFrameLengthTx[15:0] - readGoodByteCnt[15:0] - 16'd16 - 16'd4 - 16'd4);
    else if((formatMPDUFSMCs == FD_EXTIV) && (readByteCnt_ff1 == fieldOffset))
    // encrDataLength     MPDU Length                  Header            KeyIdx   FCS     EXT IV
      txDataLength <= (MPDUFrameLengthTx[15:0] - readGoodByteCnt[15:0] - 16'd8  - 16'd4 - 16'd4);
`ifdef RW_WAPI_EN
    else if((formatMPDUFSMCs == FD_WAPI_KEYIDX) && (readByteCnt_ff1 == fieldOffset))
    // encrDataLength     MPDU Length                  Header            KeyIdx   FCS     PN
      txDataLength <= (MPDUFrameLengthTx[15:0] - readGoodByteCnt[15:0] - 16'd18 - 16'd4 - 16'd16);
`endif //RW_WAPI_EN
  end
end

// RSR to store the Packet number for CCMP
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txCCMPSeqCnt   <= 48'h0;
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    txCCMPSeqCnt   <= 48'h0;
  else if(enable)
  begin
    if (txFIFODataValid && ((formatMPDUFSMCs == FD_EXTIV) || ((formatMPDUFSMCs == FD_IV) && ((readByteCnt_ff1 == fieldOffset) || (readByteCnt_ff1 == fieldOffset+16'h1)))))
      txCCMPSeqCnt <= {txCCMPSeqCnt[39:0],txFIFORdData};
  end
end

// RSR to store the Packet number for TKIP
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txTKIPSeqCnt   <= 48'h0;
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    txTKIPSeqCnt   <= 48'h0;
  else if(enable)
  begin
    if(txFIFODataValid && ((formatMPDUFSMCs == FD_EXTIV) || ((formatMPDUFSMCs == FD_IV) && ((readByteCnt_ff1 == fieldOffset) || (readByteCnt_ff1 == fieldOffset+16'h2)))))
    begin
      if((formatMPDUFSMCs == FD_IV) && (readByteCnt_ff1 == fieldOffset))
        txTKIPSeqCnt[7:0]  <= txFIFORdData;
      else if((formatMPDUFSMCs == FD_IV) && (readByteCnt_ff1 == (fieldOffset + 16'h2)))
        txTKIPSeqCnt[15:8] <= txFIFORdData;
      else
        txTKIPSeqCnt       <= {txTKIPSeqCnt[39:0],txFIFORdData};
    end
  end
end

assign txCCMPTKIPSeqCnt = (txTKIP) ? txTKIPSeqCnt :
                          (txCCMP) ? txCCMPSeqCnt :
                          (txGCMP) ? txCCMPSeqCnt :
                                     48'h0;

// Frame Control Field 
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txFrameControl <= 16'h0;
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    txFrameControl <= 16'h0;
  else if(enable)
  begin
    if((formatMPDUFSMCs == FD_FRMCNTRL_BYTE1) && (txFIFODataValid == 1'b1))
      txFrameControl[7:0]  <= txFIFORdData;
    else if((formatMPDUFSMCs == FD_FRMCNTRL_BYTE2) && (txFIFODataValid == 1'b1))
      txFrameControl[15:8]  <= txFIFORdData;    
  end
end

// ADD1
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txAddr1   <= 48'h0;
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    txAddr1   <= 48'h0;
  else if(enable)
    if((formatMPDUFSMCs == FD_ADDR1) && (txFIFODataValid == 1'b1))
      txAddr1 <= {txFIFORdData,txAddr1[47:8]};
end

// ADD2
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txAddr2   <= 48'h0;
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    txAddr2   <= 48'h0;
  else if(enable)
    if((formatMPDUFSMCs == FD_ADDR2) && (txFIFODataValid == 1'b1))
      txAddr2   <= {txFIFORdData,txAddr2[47:8]};
end

// ADDR3 LSR
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txAddr3   <= 48'h0;
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    txAddr3   <= 48'h0;
  else if(enable)
    if((formatMPDUFSMCs == FD_ADDR3) && (txFIFODataValid == 1'b1))
      txAddr3   <= {txFIFORdData,txAddr3[47:8]};
end

// ADDR4 LSR
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txAddr4   <= 48'h0;
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    txAddr4   <= 48'h0;
  else if(enable)
    if((formatMPDUFSMCs == FD_ADDR4) && (txFIFODataValid == 1'b1))
      txAddr4   <= {txFIFORdData,txAddr4[47:8]};
end

// QoS Control Field 
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txQoSCF <= 16'h0;
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    txQoSCF <= 16'h0;
  else if(enable)
  begin
    if((formatMPDUFSMCs == FD_QOSCNTRL) && (txFIFODataValid == 1'b1))
      txQoSCF <= {txFIFORdData,txQoSCF[15:8]};
  end
end

// ADD4 Present Flag
assign txAddress4Pres  = (toDS && fromDS);

// QoS field Present Flag
assign txQoSFrame      = qosFrame;

// Sequence Control Field 
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txSeqControl <= 16'h0;
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    txSeqControl <= 16'h0;
  else if(enable)
  begin
    if((formatMPDUFSMCs == FD_SQNCNTRL) && (txFIFODataValid == 1'b1))
      txSeqControl <= {txFIFORdData,txSeqControl[15:8]};
  end
end

`ifdef RW_WAPI_EN

// Key index Field 
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txKeyIdx <= 8'h0;
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    txKeyIdx <= 8'h0;
  else if(enable)
  begin
    // register only the first byte since second byte is the reserved field
    if((formatMPDUFSMCs == FD_WAPI_KEYIDX) && (txFIFODataValid == 1'b1) && ~countDone_p)
      txKeyIdx <= txFIFORdData;
  end
end

// WAPI Packet Number (PN) Field 
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txPN <= 128'h0;
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
    txPN <= 128'h0;
  else if(enable)
  begin
    if((formatMPDUFSMCs == FD_WAPI_PN) && (txFIFODataValid == 1'b1))
      txPN <= {txFIFORdData,txPN[127:8]};
  end
end

`endif //RW_WAPI_EN

// WEP IV Capture and WEP/TKIP IV Valid
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    txWEPIV      <= 24'h0;
  end
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
  begin
    txWEPIV      <= 24'h0;
  end
  else if(enable)
  begin

    if((formatMPDUFSMCs == FD_IV) && txFIFODataValid && (readByteCnt_ff1 < fieldBytes + fieldOffset))
      txWEPIV <= {txFIFORdData,txWEPIV[23:8]};

  end
end


// WEP    Flag
// TKIP   Flag
// CCMP   Flag
// CCMPHT Flag
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    txWEP                   <= 1'b0;
    txTKIP                  <= 1'b0;
    txCCMP                  <= 1'b0;
    txGCMP                  <= 1'b0;
`ifdef RW_WAPI_EN
    txWAPI                  <= 1'b0;
`endif 
    txHTMode                <= 1'b0;
  end
  else if ((macCoreTxClkSoftRst_n == 1'b0) || (CSIsIdle == 1'b1)) // Synchronous Reset
  begin
    txWEP                   <= 1'b0;
    txTKIP                  <= 1'b0;
    txCCMP                  <= 1'b0;
    txGCMP                  <= 1'b0;
`ifdef RW_WAPI_EN
    txWAPI                  <= 1'b0;
`endif 
    txHTMode                <= 1'b0;
  end
  else if(enable)
  begin
    if(protectedFrame)
    begin
      if(formatMPDUFSMCs == FD_DURID_DATA)
      begin
        if(cTypeKSR == ENCRYPTION_TYPE_WEP)
          txWEP                 <= 1'b1;
        else if(cTypeKSR == ENCRYPTION_TYPE_TKIP)
          txTKIP                <= 1'b1;
        else if(cTypeKSR == ENCRYPTION_TYPE_CCMP)
        begin
          txCCMP                  <= 1'b1;
          txHTMode                <= (formatMod >= 4'b0010); // HT, VHT or HE
        end
        else if(cTypeKSR == ENCRYPTION_TYPE_GCMP)
        begin
          txGCMP                  <= 1'b1;
          txHTMode                <= (formatMod >= 4'b0010); // HT, VHT or HE
        end
`ifdef RW_WAPI_EN
        else if(cTypeKSR == ENCRYPTION_TYPE_WAPI)
        begin
          txWAPI                  <= 1'b1;/*((formatMod == 3'b010) || (formatMod == 3'b011) || (formatMod == 3'b100))*/
        end
`endif //RW_WAPI_EN
      end
    end
  end
end

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// Disable coverage the RW_SIMU_ON part because this code is not functional but 
// here to ease the simulation
// pragma coverage block = off 

// formatMPDUFSM FSM states displayed in a string to easy simulation and debug
always @*
begin
  case (formatMPDUFSMCs)
              FD_IDLE  :  formatMPDUFSMCs_str = {"FD_IDLE"};
        FD_INITCRYPTO  :  formatMPDUFSMCs_str = {"FD_INITCRYPTO"};
           FD_STARTTX  :  formatMPDUFSMCs_str = {"FD_STARTTX"};
    FD_FRMCNTRL_BYTE1  :  formatMPDUFSMCs_str = {"FD_FRMCNTRL_BYTE1"};
    FD_FRMCNTRL_BYTE2  :  formatMPDUFSMCs_str = {"FD_FRMCNTRL_BYTE2"};
        FD_DURID_DATA  :  formatMPDUFSMCs_str = {"FD_DURID_DATA"};
             FD_ADDR1  :  formatMPDUFSMCs_str = {"FD_ADDR1"};
             FD_ADDR2  :  formatMPDUFSMCs_str = {"FD_ADDR2"};
             FD_ADDR3  :  formatMPDUFSMCs_str = {"FD_ADDR3"};
          FD_SQNCNTRL  :  formatMPDUFSMCs_str = {"FD_SQNCNTRL"};
             FD_ADDR4  :  formatMPDUFSMCs_str = {"FD_ADDR4"};
          FD_QOSCNTRL  :  formatMPDUFSMCs_str = {"FD_QOSCNTRL"};
           FD_CFCNTRL  :  formatMPDUFSMCs_str = {"FD_CFCNTRL"};
           FD_HTCNTRL  :  formatMPDUFSMCs_str = {"FD_HTCNTRL"};
               FD_TSF  :  formatMPDUFSMCs_str = {"FD_TSF"};
                FD_IV  :  formatMPDUFSMCs_str = {"FD_IV"};
             FD_EXTIV  :  formatMPDUFSMCs_str = {"FD_EXTIV"};
       FD_WAPI_KEYIDX  :  formatMPDUFSMCs_str = {"FD_WAPI_KEYIDX"};
           FD_WAPI_PN  :  formatMPDUFSMCs_str = {"FD_WAPI_PN"};
         FD_MPDUSTART  :  formatMPDUFSMCs_str = {"FD_MPDUSTART"};
          FD_FRM_BODY  :  formatMPDUFSMCs_str = {"FD_FRM_BODY"};
  FD_WAIT_ICVMIC_DONE  :  formatMPDUFSMCs_str = {"FD_WAIT_ICVMIC_DONE"};
               FD_FCS  :  formatMPDUFSMCs_str = {"FD_FCS"};
       FD_FLUSHTXFIFO  :  formatMPDUFSMCs_str = {"FD_FLUSHTXFIFO"};
              default  :  formatMPDUFSMCs_str = {"XXX"};
   endcase
end

// Type of transmitted frame displayed in a string to easy simulation and debug
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if ((macCoreClkHardRst_n == 1'b0))  // Asynchronous Reset
    txFrameType_str = {"XXX"};
  else
    if ((MPDUFrameLengthTx == 16'd0) && (formatMPDUFSMCs == FD_STARTTX))
      txFrameType_str = {"NDP"};
    else if (formatMPDUFSMCs == FD_FRMCNTRL_BYTE2)
      casex ({subtypeTx,typeTx})
           6'b000000  :  txFrameType_str = {"ASSOCIATION REQUEST"};
           6'b000100  :  txFrameType_str = {"ASSOCIATION RESPONSE"};
           6'b001000  :  txFrameType_str = {"REASSOCIATION REQUEST"};
           6'b001100  :  txFrameType_str = {"REASSOCIATION RESPONSE"};
           6'b010000  :  txFrameType_str = {"PROBE RESPONSE"};
           6'b010100  :  txFrameType_str = {"PROBE RESPONSE"};
           6'b011000  :  txFrameType_str = {"TIMING ADVERTISEMENT"};
           6'b100000  :  txFrameType_str = {"BEACON"};
           6'b100100  :  txFrameType_str = {"ATIM"};
           6'b101000  :  txFrameType_str = {"DISSASSOCIATION"};
           6'b101100  :  txFrameType_str = {"AUTHENTICATION"};
           6'b110000  :  txFrameType_str = {"DEAUTHENTICATION"};
           6'b110100  :  txFrameType_str = {"ACTION"};
           6'b111000  :  txFrameType_str = {"ACTION No Ack"};
           6'b001001  :  txFrameType_str = {"TRIGGER"};
           6'b010001  :  txFrameType_str = {"BEAMFORMING REPORT POLL"};
           6'b010101  :  txFrameType_str = {"VHT/HE NDP ANNOUNCEMENT"};
           6'b011001  :  txFrameType_str = {"CONTROL FRAME EXTENSION"};
           6'b011101  :  txFrameType_str = {"CONTROL WRAPPER"};
           6'b100001  :  txFrameType_str = {"BA REQUEST"};
           6'b100101  :  txFrameType_str = {"BA"};
           6'b101001  :  txFrameType_str = {"PS-POLL"};
           6'b101101  :  txFrameType_str = {"RTS"};
           6'b110001  :  txFrameType_str = {"CTS"};
           6'b110101  :  txFrameType_str = {"ACK"};
           6'b111001  :  txFrameType_str = {"CF-END"};
           6'b111101  :  txFrameType_str = {"CF-END+CF-ACK"};
           6'b000010  :  txFrameType_str = {"DATA"};
           6'b000110  :  txFrameType_str = {"DATA+CF-ACK"};
           6'b001010  :  txFrameType_str = {"DATA+CF-POLL"};
           6'b001110  :  txFrameType_str = {"DATA+CF-ACK+CF-POLL"};
           6'b010010  :  txFrameType_str = {"Null"};
           6'b010110  :  txFrameType_str = {"CF-ACK"};
           6'b011010  :  txFrameType_str = {"CF-POLL"};
           6'b011110  :  txFrameType_str = {"CF-ACK+CF-POLL"};
           6'b100010  :  txFrameType_str = {"QoS DATA"};
           6'b100110  :  txFrameType_str = {"QoS DATA+CF-ACK"};
           6'b101010  :  txFrameType_str = {"QoS DATA+CF-POLL"};
           6'b101110  :  txFrameType_str = {"QoS DATA+CF-ACK+CF-POLL"};
           6'b110010  :  txFrameType_str = {"QoS Null"};
           6'b111010  :  txFrameType_str = {"QoS CF-POLL"};
           6'b111110  :  txFrameType_str = {"QoS CF-ACK+CF-POLL"};
             default  :  txFrameType_str = {"UNKNOWN"};
       endcase
end

// Type of encryption displayed in a string to easy simulation and debug
always @*
begin
  casex ({cLenKSR,cTypeKSR})
    {2'bxx,ENCRYPTION_TYPE_NONE} : encryptionType_str = {"NONE"};
    {2'b00,ENCRYPTION_TYPE_WEP}  : encryptionType_str = {"WEP-64"};
    {2'b01,ENCRYPTION_TYPE_WEP}  : encryptionType_str = {"WEP-128"};
    {2'bxx,ENCRYPTION_TYPE_TKIP} : encryptionType_str = {"TKIP"};
    {2'b00,ENCRYPTION_TYPE_CCMP} : encryptionType_str = {"CCMP-128"};
    {2'b10,ENCRYPTION_TYPE_CCMP} : encryptionType_str = {"CCMP-256"};
`ifdef RW_WAPI_EN
    {2'bxx,ENCRYPTION_TYPE_WAPI} : encryptionType_str = {"WAPI"};
`endif // RW_WAPI_EN
    {2'b00,ENCRYPTION_TYPE_GCMP} : encryptionType_str = {"GCMP-128"};
    {2'b10,ENCRYPTION_TYPE_GCMP} : encryptionType_str = {"GCMP-256"};
    default                      : encryptionType_str = {"XXXX"};
  endcase
end


// pragma coverage block = on 
`endif // RW_SIMU_ON

`ifdef RW_ASSERT_ON

// System Verilog Assertions
////////////////////////////


// Cover Points Definitions
///////////////////////////

//$rw_sva This cover point checks if we are transmitting a CTS Frame
countCTSFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx} == 6'b011100)));

//$rw_sva This cover point checks if we are transmitting a RTS Frame
countRTSFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx} == 6'b011011)));

//$rw_sva This cover point checks if we are transmitting a ACK Frame
countACKFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx} == 6'b011101)));

//$rw_sva This cover point checks if we are transmitting a BA Frame
countBAFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx} == 6'b011001)));

//$rw_sva This cover point checks if we are transmitting a BA_REQ Frame
countBAREQFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx} == 6'b011000)));

//$rw_sva This cover point checks if we are transmitting a CONTROL WRAPPER Frame
countCONTROLWRAPPERFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx} == 6'b010111)));

//$rw_sva This cover point checks if we are transmitting a CF-END Frame
countCFENDFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx} == 6'b011110)));

//$rw_sva This cover point checks if we are transmitting a CF-END+CF-ACK Frame
countCFENDPCFACKFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx} == 6'b011111)));

//$rw_sva This cover point checks if we are transmitting a PS-POLL Frame
countPSPOLLFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx} == 6'b011010)));

//$rw_sva This cover point checks if we are transmitting a Beacon Frame
countBeaconFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx} == 6'b001000)));

//$rw_sva This cover point checks if we are transmitting a Probe Response Frame
countProbeResponseFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx} == 6'b000101)));

//$rw_sva This cover point checks if we are transmitting a Timing Advertisement Frame
countTimingAdvertisementFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx} == 6'b000110)));

//$rw_sva This cover point checks if we are transmitting a QoS Data Frame
countQoSDataFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx[3]} == 3'b101)));

//$rw_sva This cover point checks if we are transmitting a Data Frame
countDataFromFifo: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx[3]} == 3'b100)));

//$rw_sva This cover point checks if we are transmitting a QoS +HTC Frame
countHTCFrame: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && ({typeTx,subtypeTx[3]} == 3'b101) && htcFrame));

//$rw_sva This cover point checks if we are transmitting a Management Frame
countManagementFrame: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && (typeTx == 3'b00)));

//$rw_sva This cover point checks if we are transmitting an encrypted Frame
countEncryptedFrame: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && protectedFrame));

//$rw_sva This cover point checks if we are transmitting a WDS Frame
countWDSFrame: cover property (@(posedge macCoreTxClk) (currentMPDUDone_p && toDS && fromDS));

`endif // RW_ASSERT_ON


endmodule
                 
