////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : Module which creates and formats the QoS Null frames.
//
// Simulation Notes :
//
//    For simulation, one define is available
//
//    RW_SIMU_ON   : which creates string signals to display the FSM states on
//                the waveform viewers
//
// Some pragmas for code coverage have been defined.
//  - The implicite default state has been excluded
//  - The some default states have been excluded because not reachable by design
// pragma coverage implicit_default=off
//
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
//
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

module formatQOSNULL (
          //$port_g Clock and Reset interface
          input  wire        macCoreTxClk,          // MAC Core Transmit Clock
          input  wire        macCoreClkHardRst_n,   // Hard Reset of the MAC Core Clock domain
                                                    // active low
          input  wire        macCoreTxClkSoftRst_n, // Soft Reset of the MAC Core Clock domain
                                                    // active low

          //$port_g MAC Controller interface
          input  wire [47:0] destAddr,              // Receiver address
          input  wire [47:0] srcAddr,               // Source address
          input  wire [47:0] bssID,                 // BSS ID
          input  wire [15:0] duration,              // Duration
          input  wire  [7:0] acBitmap,              // Available Channel Bitmap field
          input  wire  [5:0] ulPwrHeadroom,         // Uplink power headroom
          input  wire        txUPHControl,          // Uplink power headroom HT control field is
                                                    // present

          //$port_g Tx Controller Main FSM interface
          input  wire        qosnullStart_p,        // Indicates that a QoS Null packet has to be
                                                    // sent
          output reg         qosnullDone_p,         // Indicates that the Qos Null packet has been
                                                    // sent

          //$port_g FCS Block interface
          input  wire        fcsEnd_p,              // indicates the end of the transmission
          input  wire        fcsBusy,               // indicates that the FCS block is busy cannot
                                                    // accept new data.
          output wire        qosnullFCSEnable,      // Enable the FCS Block
          output wire        qosnullFCSStart_p,     // indicates the begining of the transmission
                                                    // and enable the FCS block
          output wire        qosnullFCSShiftTx,     // indicates FCS engine to append FCS calculated
                                                    // on data
          output reg   [7:0] qosnullFCSDInTx,       // Data to the FCS block
          output wire        qosnullFCSDInValidTx,  // indicates that the data on fcsDInTx is valid
                                                    // and can be processed.

          //$port_g CSReg Block interface
          input  wire        pwrMgt,                // Power Management is enabled for the current

          //$port_g Debug interface
          output reg   [3:0] formatQOSNULLFSMCs     // formatQOSNULLFSM FSM Current State
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
// formatQOSNULLFSM FSM states definition
//$fsm_sd formatQOSNULLFSM
localparam
   QOSNULL_IDLE           =  4'd0,
   QOSNULL_STARTTX        =  4'd1,
   QOSNULL_FRMCNTRL_BYTE1 =  4'd2,
   QOSNULL_FRMCNTRL_BYTE2 =  4'd3,
   QOSNULL_DURATIONID     =  4'd4,
   QOSNULL_RA             =  4'd5,
   QOSNULL_TA             =  4'd6,
   QOSNULL_BSSID          =  4'd7,
   QOSNULL_SEQCONTROL     =  4'd8,
   QOSNULL_QOSCONTROL     =  4'd9,
   QOSNULL_HTCONTROL      =  4'd10,
   QOSNULL_FCS            =  4'd11;

localparam
   DURATIONID_BYTECNT     = 4'd2,    // Number of bytes of Duration ID field
   RA_BYTECNT             = 4'd6,    // Number of bytes of RA field
   TA_BYTECNT             = 4'd6,    // Number of bytes of TA field
   BSSID_BYTECNT          = 4'd6,    // Number of bytes of BSS ID field
   SEQCONTROL_BYTECNT     = 4'd2,    // Number of bytes of Sequence Control field
   QOSCONTROL_BYTECNT     = 4'd2,    // Number of bytes of QoS Control field
   HTCONTROL_BYTECNT      = 4'd4,    // Number of bytes of HT Control field
   DATAFRM_TYPE           = 2'b10,   // DATA Frame Type
   QOSNULL_SUBTYPE        = 4'b1100, // QoS Null Frame Sub-Type
   PROTOCOL_VERSION       = 2'b00;   // Protocol version


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////

// formatQOSNULLFSM FSM signals definition
reg   [3:0] formatQOSNULLFSMNs;   // QoS NULL FSM Next State

wire        countDone_p;          // Pulse to indicate the end of a multiple bytes field
reg   [3:0] byteCnt;              // Counts the number of bytes
wire        byteCntEn;            // Enable the byteCnt counter
reg   [3:0] nBytes;               // Number of bytes of the current field

wire        qosnullActive;        // Indicate that formatQOSNULL is active

reg  [11:0] qosnullSeqNbr;        // QoS Null Frame Sequence Number

wire [13:0] bqrControl;           // Bandwidth Query Report Control
wire [11:0] uphControl;           // Uplink Power Headroom Control
reg  [31:0] htControl;            // HE Variant HT Control

////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

// formatQOSNULLFSM FSM Current State Logic
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)         // Asynchronous Reset
    formatQOSNULLFSMCs <= QOSNULL_IDLE;
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    formatQOSNULLFSMCs <= QOSNULL_IDLE;
  else
    formatQOSNULLFSMCs <= formatQOSNULLFSMNs;
end


// formatQOSNULLFSM FSM Next State Logic.
always @*
begin
   case(formatQOSNULLFSMCs)

   QOSNULL_IDLE:
      //$fsm_s In QOSNULL_IDLE state, the state machine waits until it is triggered by the MAC
      //Controller
      if (qosnullStart_p)
         //$fsm_t When qosnullStart_p is received, the state machine goes to QOSNULL_STARTTX state
         formatQOSNULLFSMNs = QOSNULL_STARTTX;
      else
         //$fsm_t While qosnullStart_p is not received, the state machine stays in QOSNULL_IDLE
         //state
         formatQOSNULLFSMNs = QOSNULL_IDLE;

   QOSNULL_STARTTX:
      //$fsm_s In QOSNULL_STARTTX state, the state machine start the FCS engine and the MAC-PHY
      //Interface

      //$fsm_t After one clock cycle, the state machine moves to QOSNULL_FRMCNTRL_BYTE1 state
      formatQOSNULLFSMNs = QOSNULL_FRMCNTRL_BYTE1;

   QOSNULL_FRMCNTRL_BYTE1:
      //$fsm_s In QOSNULL_FRMCNTRL_BYTE1 state, the state machine sends the first byte of the
      //QoS NULL frame control field
      if (!fcsBusy)
         //$fsm_t When fcsBusy is low meaning that the FCS is free, the state machine goes to
         //QOSNULL_FRMCNTRL_BYTE2 state
         formatQOSNULLFSMNs = QOSNULL_FRMCNTRL_BYTE2;
      else
         //$fsm_t When fcsBusy is high meaning that the FCS block is busy, the state machine stays
         //in QOSNULL_FRMCNTRL_BYTE1 state
         formatQOSNULLFSMNs = QOSNULL_FRMCNTRL_BYTE1;

   QOSNULL_FRMCNTRL_BYTE2:
      //$fsm_s In QOSNULL_FRMCNTRL_BYTE2 state, the state machine sends the second byte of the
      //QoS Null frame control field
      if (!fcsBusy)
         //$fsm_t When fcsBusy is low meaning that the FCS is free, the state machine goes to
         //QOSNULL_DURATIONID state
         formatQOSNULLFSMNs = QOSNULL_DURATIONID;
      else
         //$fsm_t When fcsBusy is high meaning that the FCS block is busy, the state machine stays
         //in QOSNULL_FRMCNTRL_BYTE2 state
         formatQOSNULLFSMNs = QOSNULL_FRMCNTRL_BYTE2;

   QOSNULL_DURATIONID:
      //$fsm_s In QOSNULL_DURATIONID state, the state machine sends the Duration ID to FCS block
      if (countDone_p)
         //$fsm_t When countDone_p is received meaning that the Duration ID has been completely sent
         //through the FCS, the state machine goes to QOSNULL_RA state
         formatQOSNULLFSMNs = QOSNULL_RA;
      else
         //$fsm_t When countDone_p is low meaning that the Duration ID has not been yet completely
         //sent through the FCS block, the state machine stays in QOSNULL_DURATIONID state
         formatQOSNULLFSMNs = QOSNULL_DURATIONID;

   QOSNULL_RA:
      //$fsm_s In QOSNULL_RA state, the state machine sends the RA to FCS block
      if (countDone_p)
         //$fsm_t When countDone_p is received meaning that the RA has been completely sent through
         //the FCS, the state machine goes to QOSNULL_TA state
         formatQOSNULLFSMNs = QOSNULL_TA;
      else
         //$fsm_t When countDone_p is low meaning that the RA has not been yet completely sent
         //through the FCS block, the state machine stays in QOSNULL_RA state
         formatQOSNULLFSMNs = QOSNULL_RA;

   QOSNULL_TA:
      //$fsm_s In QOSNULL_TA state, the state machine sends the TA to FCS block
      if (countDone_p)
         //$fsm_t When countDone_p is received meaning that the TA has been completely sent through
         //the FCS, the state machine goes to QOSNULL_BSSID state
         formatQOSNULLFSMNs = QOSNULL_BSSID;
      else
         //$fsm_t When countDone_p is low meaning that the TA has not been yet completely sent
         //through the FCS block, the state machine stays in QOSNULL_TA state
         formatQOSNULLFSMNs = QOSNULL_TA;

   QOSNULL_BSSID:
      //$fsm_s In QOSNULL_BSSID state, the state machine sends the BSSID to FCS block
      if (countDone_p)
         //$fsm_t When countDone_p is received meaning that the BSSID has been completely sent
         //through the FCS, the state machine goes to QOSNULL_SEQCONTROL state
         formatQOSNULLFSMNs = QOSNULL_SEQCONTROL;
      else
         //$fsm_t When countDone_p is low meaning that the BSSID has not been yet completely sent
         //through the FCS block, the state machine stays in QOSNULL_BSSID state
         formatQOSNULLFSMNs = QOSNULL_BSSID;

   QOSNULL_SEQCONTROL:
      //$fsm_s In QOSNULL_SEQCONTROL state, the state machine sends the Sequence Control to FCS
      //block
      if (countDone_p)
         //$fsm_t When countDone_p is received meaning that the Sequence Control has been completely
         //sent through the FCS, the state machine goes to QOSNULL_QOSCONTROL state
         formatQOSNULLFSMNs = QOSNULL_QOSCONTROL;
      else
         //$fsm_t When countDone_p is low meaning that the Sequence Control has not been yet
         //completely sent through the FCS block, the state machine stays in QOSNULL_SEQCONTROL
         //state
         formatQOSNULLFSMNs = QOSNULL_SEQCONTROL;

   QOSNULL_QOSCONTROL:
      //$fsm_s In QOSNULL_QOSCONTROL state, the state machine sends the QoS Control to FCS block
      if (countDone_p)
         //$fsm_t When countDone_p is received meaning that the QoS Control has been completely sent
         //through the FCS, the state machine goes to QOSNULL_HTCONTROL state
         formatQOSNULLFSMNs = QOSNULL_HTCONTROL;
      else
         //$fsm_t When countDone_p is low meaning that the QoS Control has not been yet completely
         //sent through the FCS block, the state machine stays in QOSNULL_QOSCONTROL state
         formatQOSNULLFSMNs = QOSNULL_QOSCONTROL;

   QOSNULL_HTCONTROL:
      //$fsm_s In QOSNULL_HTCONTROL state, the state machine sends the HT Control to FCS block
      if (countDone_p)
         //$fsm_t When countDone_p is received meaning that the HT Control has been completely sent
         //through the FCS, the state machine goes to QOSNULL_FCS state
         formatQOSNULLFSMNs = QOSNULL_FCS;
      else
         //$fsm_t When countDone_p is low meaning that the HT Control has not been yet completely
         //sent through the FCS block, the state machine stays in QOSNULL_HTCONTROL state
         formatQOSNULLFSMNs = QOSNULL_HTCONTROL;

   QOSNULL_FCS:
      //$fsm_s In QOSNULL_FCS state, the state machine trigs the FCS block to shift out the FCS
      //value and wait until its completion.
      if (fcsEnd_p)
         //$fsm_t When fcsEnd_p is received meaning that the FCS has been completely sent,
         //the state machine goes to QOSNULL_IDLE state
         formatQOSNULLFSMNs = QOSNULL_IDLE;
      else
         //$fsm_t While fcsEnd_p is not received, the state machine stays in QOSNULL_FCS state
         formatQOSNULLFSMNs = QOSNULL_FCS;

   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off
   default:
      formatQOSNULLFSMNs = QOSNULL_IDLE;
   // pragma coverage block = on
   endcase
end


// byteCnt counter generation
// This counter allows to count the number of bytes per field
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    byteCnt     <= 4'b0;
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    byteCnt     <= 4'b0;
  else
  begin
    // The counter is incremented only if it is started and when the FCS block is not busy
    if (!fcsBusy && byteCntEn)
    begin
      // While the value of the counter is not equal to nBytes, the byteCnt counter is incremented
      if (byteCnt < nBytes)
        byteCnt <= byteCnt + 4'd1;
      else
      begin
        byteCnt  <= 4'b0;
      end
    end
  end
end


// This combinational process assigns the number of the current field.
// Note that nBytes is 1 byte less than the number of byte of the field because
// it is used as reference for the byteCnt counter which starts at 0.
always @*
begin
  case (formatQOSNULLFSMCs)
  QOSNULL_DURATIONID : nBytes = DURATIONID_BYTECNT - 4'd1;
  QOSNULL_RA         : nBytes = RA_BYTECNT         - 4'd1;
  QOSNULL_TA         : nBytes = TA_BYTECNT         - 4'd1;
  QOSNULL_BSSID      : nBytes = BSSID_BYTECNT      - 4'd1;
  QOSNULL_SEQCONTROL : nBytes = SEQCONTROL_BYTECNT - 4'd1;
  QOSNULL_QOSCONTROL : nBytes = QOSCONTROL_BYTECNT - 4'd1;
  QOSNULL_HTCONTROL  : nBytes = HTCONTROL_BYTECNT  - 4'd1;
  default            : nBytes = 4'd0;
  endcase
end


// Start the counter in the start which have more that 1 byte length as for DURATIONID and RA.
assign byteCntEn = (formatQOSNULLFSMCs == QOSNULL_DURATIONID) |
                   (formatQOSNULLFSMCs == QOSNULL_RA        ) |
                   (formatQOSNULLFSMCs == QOSNULL_TA        ) |
                   (formatQOSNULLFSMCs == QOSNULL_BSSID     ) |
                   (formatQOSNULLFSMCs == QOSNULL_SEQCONTROL) |
                   (formatQOSNULLFSMCs == QOSNULL_QOSCONTROL) |
                   (formatQOSNULLFSMCs == QOSNULL_HTCONTROL );


// The countDone_p indication used to change the state is generated when the byteCnt counter reach
// the nBytes value.
// Due to the handshaking, this pulse is generated only when the fcs is not busy.
// Note also that this pulse is generated only when the counter is started.
assign countDone_p = (byteCntEn && (byteCnt == nBytes) && !fcsBusy) ? 1'b1 : 1'b0;


// qosnullDone_p pulses generation indicating the end of the QoS Null transmission
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n)
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    qosnullDone_p       <= 1'b0;
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    qosnullDone_p       <= 1'b0;
  else
  begin
    qosnullDone_p       <= 1'b0;
    if ((formatQOSNULLFSMCs == QOSNULL_FCS) && fcsEnd_p)
      // The qosnullDone_p pulse is generated when the fcsEnd_p is received
      qosnullDone_p    <= 1'b1;
  end
end


// qosnullSeqNbr counter generation
// This counter is used as Sequence Number for the QoS Null Frame, it is incremented
// after each QoS Null Transmission.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)         // Asynchronous Reset
    qosnullSeqNbr     <= 12'h0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    qosnullSeqNbr     <= 12'h0; 
  else
  begin
    // The counter is incremented after each QoS Null Transmission
    if (qosnullDone_p)
      qosnullSeqNbr   <= qosnullSeqNbr + 12'd1; 
  end
end


// FCS Block Control
////////////////////

// FCS data assignment
always @*
begin
   case(formatQOSNULLFSMCs)

   QOSNULL_FRMCNTRL_BYTE1:
      qosnullFCSDInTx = {QOSNULL_SUBTYPE, DATAFRM_TYPE, PROTOCOL_VERSION};

   QOSNULL_FRMCNTRL_BYTE2:
      // frame control for QoS NULL will be {+HTC/order = 1'b1, wep = 1'b0, moreData = 1'b0,
      // pwrMgt = pwrMgt, retry = 1'b0, morefrg = 1'b0, frmDs = 1'b0, toDs = 1'b0}
      qosnullFCSDInTx = {1'b1, 2'b0, pwrMgt, 4'b0};

   QOSNULL_DURATIONID:
      begin
        case(byteCnt[0])
        1'b0    : qosnullFCSDInTx = duration[7:0];
        1'b1    : qosnullFCSDInTx = duration[15:8];
        endcase
      end

   QOSNULL_RA:
      begin
        case(byteCnt)
        4'd0    : qosnullFCSDInTx = destAddr[7:0];
        4'd1    : qosnullFCSDInTx = destAddr[15:8];
        4'd2    : qosnullFCSDInTx = destAddr[23:16];
        4'd3    : qosnullFCSDInTx = destAddr[31:24];
        4'd4    : qosnullFCSDInTx = destAddr[39:32];
        default : qosnullFCSDInTx = destAddr[47:40];
        endcase
      end

   QOSNULL_TA:
      begin
        case(byteCnt)
        4'd0    : qosnullFCSDInTx = srcAddr[7:0];
        4'd1    : qosnullFCSDInTx = srcAddr[15:8];
        4'd2    : qosnullFCSDInTx = srcAddr[23:16];
        4'd3    : qosnullFCSDInTx = srcAddr[31:24];
        4'd4    : qosnullFCSDInTx = srcAddr[39:32];
        default : qosnullFCSDInTx = srcAddr[47:40];
        endcase
      end

   QOSNULL_BSSID:
      begin
        case(byteCnt)
        4'd0    : qosnullFCSDInTx = bssID[7:0];
        4'd1    : qosnullFCSDInTx = bssID[15:8];
        4'd2    : qosnullFCSDInTx = bssID[23:16];
        4'd3    : qosnullFCSDInTx = bssID[31:24];
        4'd4    : qosnullFCSDInTx = bssID[39:32];
        default : qosnullFCSDInTx = bssID[47:40];
        endcase
      end

   QOSNULL_SEQCONTROL:
      begin
        case(byteCnt[0])
        // Sequence control for QoS Null will be
        // {SequenceNumber = qosnullSeqNbr, FragmentNumber = 4'b0}
        1'b0    : qosnullFCSDInTx = {qosnullSeqNbr[3:0],4'h0};
        1'b1    : qosnullFCSDInTx =  qosnullSeqNbr[11:4];
        endcase
      end

   QOSNULL_QOSCONTROL:
      begin
        case(byteCnt[0])
        1'b0:     qosnullFCSDInTx = {1'b0,  //Reserved
                                     2'd0,  //AckPolicy
                                     1'b0,  //
                                     4'd0}; //TID
        1'b1:     qosnullFCSDInTx =  8'd0;  //TXOP Duration Requested
        endcase
      end

   QOSNULL_HTCONTROL:
      begin
        case(byteCnt)
        4'd0    : qosnullFCSDInTx = htControl[7:0];
        4'd1    : qosnullFCSDInTx = htControl[15:8];
        4'd2    : qosnullFCSDInTx = htControl[23:16];
        default : qosnullFCSDInTx = htControl[31:24];
        endcase
      end

   default :
      qosnullFCSDInTx = 8'b0;

   endcase
end

// Bandwidth Query Report (BQR) Control
assign bqrControl = {
   2'd0,               // BQR Control Information: Reserved
   acBitmap[7:0],      // BQR Control Information: Available Channel Bitmap
   4'd5                // BQR Control ID
};

// Uplink Power Headroom (UPH) Control
assign uphControl = {
   2'd0,               // UPH Control Information: Reserved
   ulPwrHeadroom[5:0], // UPH Control Information: Uplink Power Headroom
   4'd4                // UPH Control ID
};

// Bandwidth Query Report (BQR) HE Variant HT Control
always @*
begin
   if (txUPHControl)
      htControl = {4'h0,               // Padding
                   uphControl[11:0],   // UPH Control
                   bqrControl[13:0],   // BQR Control
                   2'b11};             // HE Variant HT Control
   else
      htControl = {16'h0,              // Padding
                   bqrControl[13:0],   // BQR Control
                   2'b11};             // HE Variant HT Control
end

// When the state machine is in QOSNULL_STARTTX, the FCS Engine is started
assign qosnullFCSStart_p    = (formatQOSNULLFSMCs == QOSNULL_STARTTX) ? 1'b1 : 1'b0;

// When the state machine is in QOSNULL_FCS state, the FCS engine is requested to
// shift out the computed signature
assign qosnullFCSShiftTx    = ((formatQOSNULLFSMNs == QOSNULL_FCS) && !fcsEnd_p) ? 1'b1 : 1'b0;

// Data Valid generation
// The data valid is high when the consumer (here the FCS engine) is not busy.
// In this specific case, the data are valid only when the FSM is in an active state with data
assign qosnullFCSDInValidTx = (qosnullActive && !fcsBusy) ? 1'b1 : 1'b0;

// Indicate when the formatQOSNULL module is active and shift data to the FCS
// The QOSNULL_STARTTX states does not provide data and the QOSNULL_FCS neither.
assign qosnullActive = (formatQOSNULLFSMCs != QOSNULL_IDLE   ) &
                       (formatQOSNULLFSMCs != QOSNULL_STARTTX) &
                       (formatQOSNULLFSMCs != QOSNULL_FCS    );

// Enable the FCS module.
assign qosnullFCSEnable = (formatQOSNULLFSMCs != QOSNULL_IDLE);


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// Disable coverage the RW_SIMU_ON part because this code is not functional but
// here to ease the simulation
// pragma coverage block = off

// String definition to display formatQOSNULLFSM current state
reg [22*8:0] formatQOSNULLFSMCs_str;

// formatQOSNULLFSM FSM states displayed in a string to easy simulation and debug
always @*
begin
   case (formatQOSNULLFSMCs)
   QOSNULL_IDLE            :  formatQOSNULLFSMCs_str = {"QOSNULL_IDLE"};
   QOSNULL_STARTTX         :  formatQOSNULLFSMCs_str = {"QOSNULL_STARTTX"};
   QOSNULL_FRMCNTRL_BYTE1  :  formatQOSNULLFSMCs_str = {"QOSNULL_FRMCNTRL_BYTE1"};
   QOSNULL_FRMCNTRL_BYTE2  :  formatQOSNULLFSMCs_str = {"QOSNULL_FRMCNTRL_BYTE2"};
   QOSNULL_DURATIONID      :  formatQOSNULLFSMCs_str = {"QOSNULL_DURATIONID"};
   QOSNULL_RA              :  formatQOSNULLFSMCs_str = {"QOSNULL_RA"};
   QOSNULL_TA              :  formatQOSNULLFSMCs_str = {"QOSNULL_TA"};
   QOSNULL_BSSID           :  formatQOSNULLFSMCs_str = {"QOSNULL_BSSID"};
   QOSNULL_SEQCONTROL      :  formatQOSNULLFSMCs_str = {"QOSNULL_SEQCONTROL"};
   QOSNULL_QOSCONTROL      :  formatQOSNULLFSMCs_str = {"QOSNULL_QOSCONTROL"};
   QOSNULL_HTCONTROL       :  formatQOSNULLFSMCs_str = {"QOSNULL_HTCONTROL"};
   QOSNULL_FCS             :  formatQOSNULLFSMCs_str = {"QOSNULL_FCS"};
   default                 :  formatQOSNULLFSMCs_str = {"XXX"};
   endcase
end
// pragma coverage block = on
`endif // RW_SIMU_ON

endmodule
