//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : 
//                    
// Simulation Notes : 
//                    
//    For simulation, one define is available
//
//    RW_SIMU_ON   : which creates string signals to display the FSM states on  
//                the waveform viewers
//
// Some pragmas for code coverage have been defined.
//  - The implicite default state has been excluded
//  - The some default states have been excluded because not reachable by design
// pragma coverage implicit_default=off 
//
// Synthesis Notes  :
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module txControllerMainFSM( 
          //$port_g Clock and Reset interface
          input  wire        macCoreTxClk,          // MAC Core Transmit Clock
          input  wire        macCoreClkHardRst_n, // Hard Reset of the MAC Core Clock domain 
                                                    // active low
          input  wire        macCoreTxClkSoftRst_n, // Soft Reset of the MAC Core Clock domain 
                                                    // active low
          //$port_g MAC Controller interface
          input  wire        sendData_p,            // indicates that data packet has to be sent
          input  wire        sendRTS_p,             // indicates that an RTS packet has to be sent
          input  wire        sendCTS_p,             // indicates that a CTS packet has to be sent
          input  wire        sendACK_p,             // indicates that an ACK packet has to be sent
          input  wire        sendCFEND_p,           // indicates that a CF-END packet has to be sent
          input  wire        sendBA_p,              // indicates that an Block ACK packet has to be sent
`ifdef RW_BFMEE_EN
          input  wire        sendBFR_p,             // indicates that an Beamforming Report packet has to be sent
`endif //RW_BFMEE_EN
          input  wire        sendBQR_p,             // Indication that an BQR packet has to be sent
          input  wire        sendSWTB_p,            // Indicates that an SW Trigger based frame has
                                                    // to be sent
          input  wire        sendOnSIFS,            // If set, data is sent on tickSIFS else on tickSlot
          output reg         txDone_p,              // indicates the end of the transmission

          input  wire  [3:0] respTxFormatMod,       // Response Transmission with Format and Modulation
          input  wire        respTxAMPDU,           // Response Transmission with A-MPDU
          input  wire        respTxSMPDU,           // Response Transmission with S-MPDU
          input  wire [14:0] respMPDUFrameLengthTx, // MPDU Length of the A-MPDU response

          
          //$port_g TX Parameter interface
          input  wire        txParameterHDReady_p,  // Indicates if the Header Descriptor fields are usable 
          input  wire        txParameterPTReady_p,  // Indicates if the Policy Table fields are usable 
          input  wire        aMPDU,                 // indicates that the current frame is an A-MPDU
          input  wire        txSMPDU,               // Flag of S-MPDU
          input  wire [15:0] MPDUFrameLengthTx,     // Gives the length of the current MPDU
          input  wire  [3:0] formatModTx,           // Format and Modulation of PPDU for Transmission
          input  wire  [9:0] nBlankMDelimiters,     // Indicates the number of blank MPDU delimiters that are inserted
          input  wire  [1:0] whichDescriptor,       // Indicates what kind of a descriptor this is

          //$port_g Timers Block interface
          input  wire        tickSlot,            // A pulse to indicate the end of Slot period
          input  wire        tickSIFS,            // A pulse to indicate the end of SIFS period

          //$port_g formatXXX  submodule interface
          input  wire        rtsDone_p,             // indicates that the RTS packet has been sent
          input  wire        ctsDone_p,             // indicates that the CTS packet has been sent
          input  wire        ackDone_p,             // indicates that the ACK packet has been sent
          input  wire        cfendDone_p,           // indicates that the CF-END packet has been sent
          input  wire        baDone_p,              // indicates that the BA packet has been sent
`ifdef RW_BFMEE_EN
          input  wire        bfrDone_p,             // indicates that the BFR packet has been sent
`endif //RW_BFMEE_EN
          input  wire        qosnullDone_p,         // indicates that the QoS-Null packet has been sent
          input  wire        currentMPDUDone_p,     // indicates that the Data packet has been sent
          output reg         mpduDone_p,            // indicates that the MPDU packet has been sent (including padding and blank delimiter if any)
          input  wire        rtsTxStart_p,          // indicates that the RTS packet starts on the MAC-PHY Interface
          input  wire        ctsTxStart_p,          // indicates that the CTS packet starts on the MAC-PHY Interface
          input  wire        cfendTxStart_p,        // indicates that the CF-END packet starts on the MAC-PHY Interface
          //input  wire        qosnullTxStart_p,      // indicates that the QoS-Null packet starts on the MAC-PHY Interface
          input  wire        mpduTxStart_p,         // indicates that the Data packet starts on the MAC-PHY Interface

          input  wire        aMPDUDelimiterDone_p,  // indicates that the A-MPDU Delimiter has been sent
          input  wire        aMPDUPaddingDone_p,    // indicates that the padding bytes have been sent
          input  wire        aMPDUBlankDelimiterDone_p,// indicates that the Blank Delimiters have to be generated

          output reg         aMPDUPaddingStart_p,   // indicates that some pad  byte have to be added 
          output reg         aMPDUDelimiterStart_p, // indicates that the AMPDU Delimiters have to be generated
          output reg         aMPDUBlankDelimiterStart_p,// indicates if blank delimiters shall be added
          output reg   [3:0] aMPDUFormatMod,        // indicates the AMPDU Format and Modulation
          output reg  [13:0] aMPDUDelMPDULength,    // indicates the MPDU Length of the AMPDU Delimiter
          output reg         aMPDUDelEOF,           // indicates the End Of Frame of the AMPDU Delimiter
          output reg         sendBQR,               // indicates that QoS Null Packet should have an
                                                    // BQR Control field.
          output wire        sendMPDU_p,            // Start the transmission of a MPDU.
          output reg         sendSWTB,              // Indicates that the frame is a SW TB frame
`ifdef RW_BFMEE_EN
          output wire        bfrStart_p,            // Start the transmission of a BFR.
`endif //RW_BFMEE_EN
          output wire        ackStart_p,            // Start the transmission of a ACK.
          output wire        baStart_p,             // Start the transmission of a BA.
          output wire        qosnullStart_p,        // Start the transmission of a QoS Null frame.

          //$port_g MAC-PHY IF Module interface
          input  wire        mpIfTxErr_p,           // Transmit error 
          input  wire        mpIfTxEn,              // Transmit on-going Indication
          output wire        startTx_p,             // Start Tx trigger                 
          output wire        stopTx_p,              // Stop Tx trigger                  
          output wire        swtbReady_p,           // Indicates that the Header Descriptor of the
                                                    // software TB frame is ready
          
          //$port_g Key Search Engine interface
          output wire        txKeySearchIndexTrig_p,// Trigs the Key Search Engine for a Key
                                                    // Research based on index.
          input  wire        keyStorageValid_p,     // Indicates when the Key Search Engine has
                                                    // returned the result 

          //$port_g Debug interface
          output reg   [4:0] txControlMainFSMCs     // txControlMainFSM FSM Current State
                 );


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

// txControlMainFSM FSM states definition
//$fsm_sd txControlMainFSM
localparam 
                 IDLE  =  5'd0,  // 
WAIT_FOR_SIFS_OR_SLOT  =  5'd1,  // 
         TX_AMPDU_DLM  =  5'd2,  // 
             TX_FRAME  =  5'd3,  // 
       WAIT_HD_TOGGLE  =  5'd4,  // 
         TX_AMPDU_PAD  =  5'd5,  // 
             TX_CFEND  =  5'd6,  // 
               TX_RTS  =  5'd7,  // 
               TX_CTS  =  5'd8,  // 
               TX_ACK  =  5'd9,  // 
              TX_BACK  =  5'd10, // 
           TX_QOSNULL  =  5'd11, // 
TX_RESP_WAIT_FOR_SIFS  =  5'd12, // 
    TX_RESP_AMPDU_DLM  =  5'd13, // 
`ifdef RW_BFMEE_EN
               TX_BFR  =  5'd14, // 
`endif //RW_BFMEE_EN
          TX_WAIT_END  =  5'd15, // 
    TX_RESP_AMPDU_PAD  =  5'd16, // 
         TX_BLANK_DLM  =  5'd17, // 
      TX_RESP_WAIT_PT  =  5'd18, // 
     TX_RESP_WAIT_KSE  =  5'd19; // 



//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

// txControlMainFSM FSM signals definition
reg [4:0] txControlMainFSMNs;  // txControlMainFSM FSM Next State

reg lastMPDU;
reg newHDValid;         // Indicates that the new Header Descriptor is available 
reg aMPDUHdr;           // Indicates that the new Header Descriptor is an aMPDU Header
reg txParameterPTReady; // Capture txParameterPTReady_p pulse
reg keyStorageValid;    // Capture keyStorageValid_p pulse

wire ampduTxStart_p;
wire respTxStart_p;   // indicates that the response packet starts on the MAC-PHY Interface

reg sendACK;    // Capture sendACK_p pulse
`ifdef RW_BFMEE_EN
reg sendBFR;    // Capture sendBFR_p pulse
`endif //RW_BFMEE_EN

`ifdef RW_SIMU_ON
// String definition to display txControlMainFSM current state
reg [21*8:0] txControlMainFSMCs_str;
`endif // RW_SIMU_ON


wire notNullBlankDel; // indicate that the blank delimiter needs to be inserted after the current MPDU 

reg  swtbReady;  // Indicated that the SW Trigger based descriptor is ready
reg  swtbWait;   // Indicates that the SW Trigger based descriptor is waited
                 // after TX start


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

// txControlMainFSM FSM Current State Logic 
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txControlMainFSMCs <= IDLE; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    txControlMainFSMCs <= IDLE; 
  else
    txControlMainFSMCs <= txControlMainFSMNs; 
end



// txControlMainFSM FSM Next State Logic.
always @* 
begin
  if (mpIfTxErr_p && (txControlMainFSMCs != IDLE))
    txControlMainFSMNs = TX_WAIT_END; 
  else
  begin
    case(txControlMainFSMCs)

      IDLE:
        //$fsm_s In IDLE state, the state machine waits for a trigger from the MAC Controller to launch the 
        //Transmission process from the TX FIFO or by generating a protection framem, or a CF-END or a 
        //response frame.
        
        case ({sendRTS_p, sendCTS_p,  sendACK_p, sendCFEND_p,
`ifdef RW_BFMEE_EN
               sendBA_p,  sendData_p, sendBFR_p, sendSWTB_p , sendBQR_p})
`else //RW_BFMEE_EN
               sendBA_p,  sendData_p, 1'b0     , sendSWTB_p , sendBQR_p})
`endif //RW_BFMEE_EN
          9'b100000000 : 
            //$fsm_t When sendRTS_p is received, the state machine goes to TX_RTS state
            txControlMainFSMNs = TX_RTS; 
          9'b010000000 : 
            //$fsm_t When sendCTS_p is received, the state machine goes to TX_CTS state
            txControlMainFSMNs = TX_CTS; 
          9'b001000000 : 
            //$fsm_t When sendACK_p is received,
            //the state machine goes to TX_RESP_WAIT_FOR_SIFS state
            txControlMainFSMNs = TX_RESP_WAIT_FOR_SIFS;
          9'b000100000 : 
            //$fsm_t When sendCFEND_p is received, the state machine goes to TX_CFEND state
            txControlMainFSMNs = TX_CFEND; 
          9'b000010000 : 
            //$fsm_t When sendBA_p is received, 
            //the state machine goes to TX_RESP_WAIT_FOR_SIFS state
            txControlMainFSMNs = TX_RESP_WAIT_FOR_SIFS;
          9'b000001000 :
            begin 
              if (txSMPDU)
                //$fsm_t When sendData_p is received and the transmission is a singletonVHTMPDU, 
                // the state machine goes directly to TX_AMPDU_DLM state (because the HD is already present)
                txControlMainFSMNs = TX_AMPDU_DLM; 
              else if (aMPDU)
                //$fsm_t When sendData_p is received and the frame is an A-MPDU, the state machine goes to WAIT_FOR_SIFS_OR_SLOT state
                txControlMainFSMNs = WAIT_FOR_SIFS_OR_SLOT; 
              else  
                //$fsm_t When sendData_p is received and the frame is a singleton MPDU, the state machine goes to TX_FRAME state
                txControlMainFSMNs = TX_FRAME; 
            end  
`ifdef RW_BFMEE_EN
          9'b000000100 : 
            //$fsm_t When sendBFR_p is received, the state machine goes to TX_RESP_WAIT_FOR_SIFS state
            txControlMainFSMNs = TX_RESP_WAIT_FOR_SIFS; 
`endif //RW_BFMEE_EN
          9'b000000010 : 
            //$fsm_t When sendSWTB_p is received,
            //the state machine goes to TX_RESP_WAIT_FOR_SIFS state
            txControlMainFSMNs = TX_RESP_WAIT_FOR_SIFS; 

          9'b000000001 : 
            //$fsm_t When sendBQR_p is received, 
            //the state machine goes to TX_RESP_WAIT_FOR_SIFS state
            txControlMainFSMNs = TX_RESP_WAIT_FOR_SIFS;

          default :
            //$fsm_t While no tx trigger is received, the state machine stays in IDLE state
            txControlMainFSMNs = IDLE;
        endcase    

      WAIT_FOR_SIFS_OR_SLOT:
        //$fsm_s In case on A-MPDU, the state machines waits in WAIT_FOR_SIFS_OR_SLOT until the Slot or SIFS boundary.
        if ((sendOnSIFS && tickSIFS) || tickSlot) 
          //$fsm_t When tickSlot_p or tickSIFS in case of sendOnSIFS mode is received and AMPDU, the state machine goes to
          // WAIT_HD_TOGGLE state to wait for the availability of the 1st MDPU THD information 
          txControlMainFSMNs = WAIT_HD_TOGGLE; 
        else
          //$fsm_t While no Slot/SIFS boundary indication is received, the state machine stays in WAIT_FOR_SIFS_OR_SLOT state
          txControlMainFSMNs = WAIT_FOR_SIFS_OR_SLOT;

      WAIT_HD_TOGGLE:
        //$fsm_s In WAIT_HD_TOGGLE state, the state machine waits for the availability of the next Header Descriptor from txParametersCache.
        if (newHDValid && ~aMPDUHdr) 
          //$fsm_t When newHDValid is received meaning that the new Header Descriptor is available, the state machine goes to TX_AMPDU_DLM state
          txControlMainFSMNs = TX_AMPDU_DLM;
        else
          //$fsm_t While newHDValid is not received, the state machine stays in WAIT_HD_TOGGLE state
          txControlMainFSMNs = WAIT_HD_TOGGLE;


      TX_AMPDU_DLM:
        //$fsm_s In TX_AMPDU_DLM state, the state machine launches the AMPDU delimiter generation and waits until its completion.
        if (aMPDUDelimiterDone_p) 
          //$fsm_t When aMPDUDelimiterDone_p is received meaning that the AMPDU delimiter has been completely transmitted to 
          //the MAC-PHY Interface, the state machine goes to TX_FRAME state
          txControlMainFSMNs = TX_FRAME; 
        else
          //$fsm_t While aMPDUDelimiterDone_p is not received, the state machine stays in TX_AMPDU_DLM state
          txControlMainFSMNs = TX_AMPDU_DLM;

      TX_FRAME:
        //$fsm_s In TX_FRAME state, the state machine launches the DATA frame transmission from the TX FIFO and waits until its completion.
        if (currentMPDUDone_p)
        begin
          if (!aMPDU && !sendSWTB)
            //$fsm_t When mpduDone_p is received and the frame was not an A-MPDU, the state machine goes back to IDLE state.
            txControlMainFSMNs = TX_WAIT_END; 
          else
          begin
            if (MPDUFrameLengthTx[1:0] != 2'b0)
              //$fsm_t When mpduDone_p is received and the MPDU length is not a multiple of 4, 
              // the state machine goes to TX_AMPDU_PAD state to add the padding bytes.
              txControlMainFSMNs = TX_AMPDU_PAD; 
            else
            begin
              if (notNullBlankDel) 
                //$fsm_t While aMPDUPaddingDone_p is not received, the state machine stays in TX_BLANK_DLM state
                txControlMainFSMNs = TX_BLANK_DLM;
              else if (lastMPDU)
                //$fsm_t When mpduDone_p is received and the MPDU length is a multiple of 4 and 
                // the transmitted MPDU was the last one, the state machine goes back to IDLE state.
                txControlMainFSMNs = TX_WAIT_END; 
              else  
                //$fsm_t When mpduDone_p is received and the MPDU lenght is a multiple of 4 and 
                // the transmitted MPDU was not the last one, the state machine goes back to WAIT_HD_TOGGLE state.
                txControlMainFSMNs = WAIT_HD_TOGGLE; 
            end    
          end
        end
        else
          //$fsm_t While currentMPDUDone_p is not received meaning that the MPDU transmission is not completed yet, 
          //the state machine stays in TX_FRAME state.
          txControlMainFSMNs = TX_FRAME;

      TX_AMPDU_PAD:
        //$fsm_s In TX_AMPDU_PAD state, the state machine launches the MPDU PADDING generation and waits until its completion.
        if (aMPDUPaddingDone_p)
        begin
          if (notNullBlankDel) 
            //$fsm_t While aMPDUPaddingDone_p is not received, the state machine stays in TX_BLANK_DLM state
            txControlMainFSMNs = TX_BLANK_DLM;
          else if (lastMPDU)
            //$fsm_t When aMPDUPaddingDone_p is received and the transmitted MPDU was the last one, 
            // the state machine goes back to IDLE state.
            txControlMainFSMNs = TX_WAIT_END; 
          else  
            //$fsm_t When aMPDUPaddingDone_p is received and the transmitted MPDU was not the last one, 
            // the state machine goes back to WAIT_HD_TOGGLE state.
            txControlMainFSMNs = WAIT_HD_TOGGLE; 
        end
        else
          //$fsm_t While aMPDUPaddingDone_p is not received, the state machine stays in TX_AMPDU_PAD state
          txControlMainFSMNs = TX_AMPDU_PAD;

      TX_BLANK_DLM:
        //$fsm_s In TX_BLANK_DLM state, the state machine launches the Blank Delimiter insertion and waits until its completion.
        if (aMPDUBlankDelimiterDone_p) 
            if (lastMPDU)
              //$fsm_t When aMPDUBlankDelimiterDone_p is received and the MPDU length is a multiple of 4 and 
              // the transmitted MPDU was the last one, the state machine goes back to IDLE state.
              txControlMainFSMNs = TX_WAIT_END; 
            else  
              //$fsm_t When aMPDUBlankDelimiterDone_p is received and the MPDU lenght is a multiple of 4 and 
              // the transmitted MPDU was not the last one, the state machine goes back to WAIT_HD_TOGGLE state.
              txControlMainFSMNs = WAIT_HD_TOGGLE; 
        else
          //$fsm_t While aMPDUBlankDelimiterDone_p is not received, the state machine stays in TX_BLANK_DLM state
          txControlMainFSMNs = TX_BLANK_DLM;

      TX_CFEND:
        //$fsm_s In TX_CFEND state, the state machine launches the CF-END frame generation and waits 
        //until its complete transmission.
        if (cfendDone_p) 
          //$fsm_t When cfendDone_p is received, the state machine goes back to IDLE state
          txControlMainFSMNs = TX_WAIT_END;
        else
          //$fsm_t While cfendDone_p is not received, the state machine stays in TX_CFEND state
          txControlMainFSMNs = TX_CFEND;

      TX_RTS:
        //$fsm_s In TX_RTS state, the state machine launches the RTS frame generation and waits 
        //until its complete transmission.
        if (rtsDone_p) 
          //$fsm_t When rtsDone_p is received, the state machine goes back to IDLE state
          txControlMainFSMNs = TX_WAIT_END;
        else
          //$fsm_t While rtsDone_p is not received, the state machine stays in TX_RTS state
          txControlMainFSMNs = TX_RTS;

      TX_CTS:
        //$fsm_s In TX_CTS state, the state machine launches the CTS frame generation and waits 
        //until its complete transmission.
        if (ctsDone_p) 
          //$fsm_t When ctsDone_p is received, the state machine goes back to IDLE state
          txControlMainFSMNs = TX_WAIT_END; 
        else
          //$fsm_t While ctsDone_p is not received, the state machine stays in TX_CTS state
          txControlMainFSMNs = TX_CTS;

      TX_ACK:
        //$fsm_s In TX_ACK state, the state machine launches the ACK frame generation and waits 
        //until its complete transmission.
        if (ackDone_p && respTxAMPDU && respMPDUFrameLengthTx[1:0]!=2'd0) 
          //$fsm_t When ackDone_p is received, the response is an aMPDU and the MPDU length is
          //not a multiple of 4,
          //the state machine goes to TX_RESP_AMPDU_PAD state
          txControlMainFSMNs = TX_RESP_AMPDU_PAD; 
        else if (ackDone_p)
          //$fsm_t When ackDone_p is received, the response is not an aMPDU or the MPDU length
          //is a multiple of 4,
          //the state machine goes to TX_WAIT_END state
          txControlMainFSMNs = TX_WAIT_END; 
        else
          //$fsm_t While ackDone_p is not received,
          //the state machine stays in TX_ACK state
          txControlMainFSMNs = TX_ACK;

      TX_BACK:
        //$fsm_s In TX_BACK state, the state machine launches the BLOCK ACK frame generation and waits 
        //until its complete transmission.
        if (baDone_p) 
          //$fsm_t When baDone_p is received, the state machine goes back to IDLE state
          txControlMainFSMNs = TX_WAIT_END; 
        else
          //$fsm_t While baDone_p is not received, the state machine stays in TX_BACK state
          txControlMainFSMNs = TX_BACK;

      TX_QOSNULL:
        //$fsm_s In TX_QOSNULL state, the state machine launches the QoS Null frame generation and
        //waits until its complete transmission.
        if (qosnullDone_p && respTxAMPDU && respMPDUFrameLengthTx[1:0]!=2'd0) 
          //$fsm_t When qosnullDone_p is received, the response is an aMPDU and the MPDU length is
          //not a multiple of 4,
          //the state machine goes to TX_RESP_AMPDU_PAD state
          txControlMainFSMNs = TX_RESP_AMPDU_PAD; 
        else if (qosnullDone_p)
          //$fsm_t When qosnullDone_p is received, the response is not an aMPDU or the MPDU length
          //is a multiple of 4,
          //the state machine goes to TX_WAIT_END state
          txControlMainFSMNs = TX_WAIT_END; 
        else
          //$fsm_t While qosnullDone_p is not received,
          //the state machine stays in TX_QOSNULL state
          txControlMainFSMNs = TX_QOSNULL;

      TX_RESP_WAIT_FOR_SIFS:
        begin
          //$fsm_s In TX_RESP_WAIT_FOR_SIFS, the state machines waits in TX_RESP_WAIT_FOR_SIFS until the SIFS boundary.
          if (tickSIFS)
          begin
            if (sendSWTB)
               //$fsm_t When tickSIFS is received, a Software trigger based frame should be sent,
               //the state machine goes to TX_RESP_WAIT_PT state
               txControlMainFSMNs = TX_RESP_WAIT_PT; 
            else if (respTxAMPDU)
               //$fsm_t When tickSIFS is received and the frame is an aMPDU,
               //the state machine goes to TX_RESP_AMPDU_DLM state
               txControlMainFSMNs = TX_RESP_AMPDU_DLM; 
`ifdef RW_BFMEE_EN
            else if (sendBFR)
               //$fsm_t When tickSIFS is received, the frame is not an aMPDU and beamforming report 
               //should be sent, the state machine goes to TX_BFR state
               txControlMainFSMNs = TX_BFR; 
`endif //RW_BFMEE_EN
            else if (sendACK)
               //$fsm_t When tickSIFS is received, the frame is not an aMPDU and ack should be sent,
               //the state machine goes to TX_ACK state
               txControlMainFSMNs = TX_ACK; 
            else//if (sendBA)
               //$fsm_t When tickSIFS is received, the frame is not VHT and block ack should be
               //sent, the state machine goes to TX_BACK state
               txControlMainFSMNs = TX_BACK; 
          end
          else
            //$fsm_t While tickSIFS is not received,
            //the state machine stays in TX_RESP_WAIT_FOR_SIFS state
            txControlMainFSMNs = TX_RESP_WAIT_FOR_SIFS;
        end

      TX_RESP_AMPDU_DLM:
        //$fsm_s In TX_RESP_AMPDU_DLM state, the state machine launches the AMPDU delimiter generation and waits until its completion.
`ifdef RW_BFMEE_EN
        if (aMPDUDelimiterDone_p && sendBFR)
          //$fsm_t When aMPDUDelimiterDone_p is received meaning that the AMPDU delimiter has been
          //completely transmitted to the MAC-PHY Interface and beamforming report should be sent,
          //the state machine goes to TX_BFR state
          txControlMainFSMNs = TX_BFR;
        else
`endif //RW_BFMEE_EN
        if (aMPDUDelimiterDone_p && sendBQR)
          //$fsm_t When aMPDUDelimiterDone_p is received meaning that the AMPDU delimiter has been
          //completely transmitted to the MAC-PHY Interface and Bandwidth Query Report should be
          //sent, the state machine goes to TX_QOSNULL state
          txControlMainFSMNs = TX_QOSNULL;
        else if (aMPDUDelimiterDone_p && sendACK)
          //$fsm_t When aMPDUDelimiterDone_p is received meaning that the AMPDU delimiter has been
          //completely transmitted to the MAC-PHY Interface and Ack should be sent,
          //the state machine goes to TX_ACK state
          txControlMainFSMNs = TX_ACK;
        else if (aMPDUDelimiterDone_p)
          //$fsm_t When aMPDUDelimiterDone_p is received meaning that the AMPDU delimiter has been
          //completely transmitted to the MAC-PHY Interface and block ack should be sent, 
          //the state machine goes to TX_BACK state
          txControlMainFSMNs = TX_BACK;
        else
          //$fsm_t While aMPDUDelimiterDone_p is not received,
          //the state machine stays in TX_RESP_AMPDU_DLM state
          txControlMainFSMNs = TX_RESP_AMPDU_DLM;

      TX_RESP_AMPDU_PAD:
        //$fsm_s In TX_RESP_AMPDU_PAD state, the state machine launches the MPDU PADDING generation
        //and waits until its completion.
        if (aMPDUPaddingDone_p) 
          //$fsm_t When aMPDUPaddingDone_p is received and the MPDU length is a multiple of 4, 
          //the state machine goes to TX_WAIT_END state.
          txControlMainFSMNs = TX_WAIT_END; 
        else
          //$fsm_t While aMPDUPaddingDone_p is not received,
          //the state machine stays in TX_RESP_AMPDU_PAD state
          txControlMainFSMNs = TX_RESP_AMPDU_PAD;

`ifdef RW_BFMEE_EN
      TX_BFR:
        //$fsm_s In TX_BFR state, the state machine launches the Beamforming report frame generation
        //and waits until its complete transmission.
        if (bfrDone_p && respTxAMPDU && respMPDUFrameLengthTx[1:0]!=2'd0) 
          //$fsm_t When bfrDone_p is received, the response is an aMPDU and the MPDU length is not
          //a multiple of 4,
          //the state machine goes to TX_RESP_AMPDU_PAD state
          txControlMainFSMNs = TX_RESP_AMPDU_PAD; 
        else if (bfrDone_p)
          //$fsm_t When bfrDone_p is received, the response is not an aMPDU or the MPDU length is
          //a multiple of 4,
          //the state machine goes to TX_WAIT_END state
          txControlMainFSMNs = TX_WAIT_END; 
        else
          //$fsm_t While bfrDone_p is not received,
          //the state machine stays in TX_BFR state
          txControlMainFSMNs = TX_BFR;
`endif //RW_BFMEE_EN

      TX_WAIT_END:
        //$fsm_s In TX_WAIT_END state, the state machine waits for the end of the transmission on the MAC-PHY Interface.
        if (!mpIfTxEn) 
          //$fsm_t When mpIfTxEn is low meaning that the transmission is finished at 
          //the MAC-PHY Interface, the state machine goes to IDLE state
          txControlMainFSMNs = IDLE; 
        else
          //$fsm_t While mpIfTxEn is set, the state machine stays in TX_WAIT_END state
          txControlMainFSMNs = TX_WAIT_END;

      TX_RESP_WAIT_PT:
        begin
          //$fsm_s In TX_RESP_WAIT_PT, the state machines waits in TX_RESP_WAIT_PT until the policy
          //table is available.
          if (txParameterPTReady || txParameterPTReady_p)
            //$fsm_t When policy table is availabe,
            //the state machine goes to TX_RESP_WAIT_KSE state
            txControlMainFSMNs = TX_RESP_WAIT_KSE; 
          else
            //$fsm_t While policy table is not available,
            //the state machine stays in TX_RESP_WAIT_PT state
            txControlMainFSMNs = TX_RESP_WAIT_PT;
        end

      TX_RESP_WAIT_KSE:
        begin
          //$fsm_s In TX_RESP_WAIT_KSE, the state machines waits in TX_RESP_WAIT_KSE until the
          //Encryption parameters are available.
          if (keyStorageValid || keyStorageValid_p)
            //$fsm_t When encryption parameters are availabe,
            //the state machine goes to TX_RESP_WAIT_KSE state
            txControlMainFSMNs = WAIT_HD_TOGGLE; 
          else
            //$fsm_t While encryption parameters are not available,
            //the state machine stays in TX_RESP_WAIT_KSE state
            txControlMainFSMNs = TX_RESP_WAIT_KSE;
        end

       // Disable coverage on the default state because it cannot be reached.
       // pragma coverage block = off 
       default:   
          txControlMainFSMNs = IDLE; 
       // pragma coverage block = on 
    endcase
  end  
end



// 
assign startTx_p = ctsTxStart_p  || rtsTxStart_p   || cfendTxStart_p ||
                   mpduTxStart_p || ampduTxStart_p || respTxStart_p;

assign swtbReady_p = respTxStart_p        & swtbReady |
                     swtbReady            & swtbWait  |
                     txParameterHDReady_p & swtbWait;


// In case of underrun detected by the MAC, the txReq is deasserted.
assign stopTx_p  = mpIfTxErr_p;

assign ampduTxStart_p = ((txControlMainFSMNs == TX_AMPDU_DLM) && ((txControlMainFSMCs == WAIT_HD_TOGGLE) && (whichDescriptor == 2'b01) && ~sendSWTB || (txControlMainFSMCs == WAIT_FOR_SIFS_OR_SLOT))) ? 1'b1 : 1'b0;

assign respTxStart_p  = ((txControlMainFSMCs == TX_RESP_WAIT_FOR_SIFS) && tickSIFS) ? 1'b1 : 1'b0;

// mpduDone_p generation
// The mpduDone_p is generated when the txControlMainFSMCs FSM moves from TX_FRAME, TX_AMPDU_PAD or TX_BLANK_DLM
// to TX_WAIT_END or WAIT_HD_TOGGLE.
// This is used to indicated the completion of MPDU transmission (including padding and balnk delimiters if any)
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    mpduDone_p <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    mpduDone_p <= 1'b0; 
  else if (((txControlMainFSMCs == TX_FRAME)    || (txControlMainFSMCs == TX_AMPDU_PAD) || (txControlMainFSMCs == TX_BLANK_DLM)) &&
           ((txControlMainFSMNs == TX_WAIT_END) ||  (txControlMainFSMNs == WAIT_HD_TOGGLE)))
    mpduDone_p <= 1'b1;
  else
    mpduDone_p <= 1'b0;
end

always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    aMPDUPaddingStart_p <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    aMPDUPaddingStart_p <= 1'b0; 
  else if ((txControlMainFSMNs == TX_AMPDU_PAD     )||
           (txControlMainFSMNs == TX_RESP_AMPDU_PAD))
    aMPDUPaddingStart_p <= 1'b1;
  else
    aMPDUPaddingStart_p <= 1'b0;
end

always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    aMPDUDelimiterStart_p <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    aMPDUDelimiterStart_p <= 1'b0; 
  else if ((txControlMainFSMNs == TX_AMPDU_DLM     )||
           (txControlMainFSMNs == TX_RESP_AMPDU_DLM))
    aMPDUDelimiterStart_p <= 1'b1;
  else
    aMPDUDelimiterStart_p <= 1'b0;
end


assign sendMPDU_p            = ((txControlMainFSMCs != TX_FRAME    ) && (txControlMainFSMNs == TX_FRAME         )) ? 1'b1 : 1'b0;
`ifdef RW_BFMEE_EN
assign bfrStart_p            = ((txControlMainFSMCs != TX_BFR      ) && (txControlMainFSMNs == TX_BFR           )) ? 1'b1 : 1'b0;
`endif //RW_BFMEE_EN
assign ackStart_p            = ((txControlMainFSMCs != TX_ACK) &&
                                (txControlMainFSMNs == TX_ACK)) ? 1'b1 : 1'b0;

assign baStart_p             = ((txControlMainFSMCs != TX_BACK) &&
                                (txControlMainFSMNs == TX_BACK)) ? 1'b1 : 1'b0;

assign qosnullStart_p        = ((txControlMainFSMCs != TX_QOSNULL) &&
                                (txControlMainFSMNs == TX_QOSNULL)) ? 1'b1 : 1'b0;

always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    aMPDUBlankDelimiterStart_p <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    aMPDUBlankDelimiterStart_p <= 1'b0; 
  else if (txControlMainFSMNs == TX_BLANK_DLM)
    aMPDUBlankDelimiterStart_p <= 1'b1;
  else
    aMPDUBlankDelimiterStart_p <= 1'b0;
end


assign notNullBlankDel    = (nBlankMDelimiters != 10'b0);

always @*
begin
   if (txControlMainFSMNs==TX_RESP_AMPDU_DLM ||
       txControlMainFSMNs==TX_RESP_AMPDU_PAD )
   begin
      aMPDUFormatMod     = respTxFormatMod;
      aMPDUDelMPDULength = respMPDUFrameLengthTx[13:0];
      aMPDUDelEOF        = respTxSMPDU;
   end
   else if (txControlMainFSMNs==TX_AMPDU_DLM && sendSWTB)
   begin
      // Note: aMPDU is not set by the txParameterCache in case of single MPDU
      // from DMA TB Channel
      aMPDUFormatMod     = 4'd8;
      aMPDUDelMPDULength = MPDUFrameLengthTx[13:0];
      aMPDUDelEOF        = ~aMPDU;
   end
   else
   begin
      aMPDUFormatMod     = formatModTx;
      aMPDUDelMPDULength = MPDUFrameLengthTx[13:0];
      aMPDUDelEOF        = txSMPDU;
   end
end

// newHDValid indication generation
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    newHDValid <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    newHDValid <= 1'b0; 
  else
    if (((sendData_p || sendSWTB_p) && aMPDU && !txSMPDU)  || sendMPDU_p || mpduDone_p)
      newHDValid <= 1'b0;
    else if(txParameterHDReady_p)
      newHDValid <= 1'b1;
end


// A-MPDU Header descriptor flag
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    aMPDUHdr <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    aMPDUHdr <= 1'b0; 
  else if(txParameterHDReady_p)
  begin
    if (aMPDU && whichDescriptor==2'b00)
      aMPDUHdr <= 1'b1;
    else
      aMPDUHdr <= 1'b0;
  end
end

// When Policy Table is ready, trig the Key Search Engine
assign txKeySearchIndexTrig_p = txControlMainFSMCs==TX_RESP_WAIT_PT &
                                (txParameterPTReady | txParameterPTReady_p);


// policy table Valid Flag, used in case of Software trigger based transmit.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txParameterPTReady <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    txParameterPTReady <= 1'b0; 
  else if (sendSWTB_p)
    txParameterPTReady <= 1'b0; 
  else if (txParameterPTReady_p)
    txParameterPTReady <= 1'b1; 
end


// key Storage Valid Flag, used in case of Software trigger based transmit.
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    keyStorageValid <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    keyStorageValid <= 1'b0; 
  else if (sendSWTB_p)
    keyStorageValid <= 1'b0; 
  else if (keyStorageValid_p)
    keyStorageValid <= 1'b1; 
end


// LastMPDU indication generation
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    lastMPDU <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    lastMPDU <= 1'b0; 
  else
    if (sendMPDU_p)
    begin
      if (aMPDU  && (whichDescriptor[1:0] == 2'b11) ||
          aMPDU  && txSMPDU                         ||
          ~aMPDU && sendSWTB) // Note: aMPDU is not set by the txParameterCache
                              // in case of single MPDU from DMA TB Channel
        lastMPDU <= 1'b1; 
      else  
        lastMPDU <= 1'b0; 
    end    
end

// txDone_p indication generation
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    txDone_p <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    txDone_p <= 1'b0; 
  else
    if ((mpIfTxEn == 1'b0) && (txControlMainFSMCs == TX_WAIT_END))
      txDone_p <= 1'b1; 
    else  
      txDone_p <= 1'b0; 
end

// sendACK_p pulse capture
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    sendACK <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    sendACK <= 1'b0; 
  else
    if (sendACK_p == 1'b1)
      sendACK <= 1'b1; 
    else if (txControlMainFSMCs == TX_WAIT_END)
      sendACK <= 1'b0; 
end

`ifdef RW_BFMEE_EN
// sendBFR_p pulse capture
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    sendBFR <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    sendBFR <= 1'b0; 
  else
    if (sendBFR_p == 1'b1)
      sendBFR <= 1'b1; 
    else if (txControlMainFSMCs == TX_WAIT_END)
      sendBFR <= 1'b0; 
end
`endif //RW_BFMEE_EN

// sendBQR_p pulse capture
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    sendBQR <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    sendBQR <= 1'b0; 
  else
    if (sendBQR_p == 1'b1)
      sendBQR <= 1'b1; 
    else if (txControlMainFSMCs == TX_WAIT_END)
      sendBQR <= 1'b0; 
end

// sendSWTB_p pulse capture
always @ (posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    sendSWTB <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    sendSWTB <= 1'b0; 
  else
    if (sendSWTB_p == 1'b1)
      sendSWTB <= 1'b1; 
    else if (txControlMainFSMCs == TX_WAIT_END)
      sendSWTB <= 1'b0; 
end

// SW Trigger Based header Descriptor Ready flag
always @(posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    swtbReady <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    swtbReady <= 1'b0; 
  else if (sendSWTB_p || (txControlMainFSMCs==IDLE))
    swtbReady <= 1'b0;
  else if (sendSWTB && txParameterHDReady_p)
    swtbReady <= 1'b1;
end

// SW Trigger Based header Descriptor wait flag
always @(posedge macCoreTxClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    swtbWait <= 1'b0; 
  else if (macCoreTxClkSoftRst_n == 1'b0)  // Synchronous Reset
    swtbWait <= 1'b0; 
  else if (swtbReady || (txControlMainFSMCs==IDLE))
    swtbWait <= 1'b0;
  else if (respTxStart_p && sendSWTB)
    swtbWait <= 1'b1;
end


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

`ifdef RW_SIMU_ON
// Disable coverage the RW_SIMU_ON part because this code is not functional but 
// here to ease the simulation
// pragma coverage block = off 

// txControlMainFSM FSM states displayed in a string to easy simulation and debug
always @*
begin
  case (txControlMainFSMCs)
                 IDLE  :  txControlMainFSMCs_str = {"IDLE"};
WAIT_FOR_SIFS_OR_SLOT  :  txControlMainFSMCs_str = {"WAIT_FOR_SIFS_OR_SLOT"};
       WAIT_HD_TOGGLE  :  txControlMainFSMCs_str = {"WAIT_HD_TOGGLE"};
         TX_AMPDU_DLM  :  txControlMainFSMCs_str = {"TX_AMPDU_DLM"};
             TX_FRAME  :  txControlMainFSMCs_str = {"TX_FRAME"};
         TX_AMPDU_PAD  :  txControlMainFSMCs_str = {"TX_AMPDU_PAD"};
             TX_CFEND  :  txControlMainFSMCs_str = {"TX_CFEND"};
               TX_RTS  :  txControlMainFSMCs_str = {"TX_RTS"};
               TX_CTS  :  txControlMainFSMCs_str = {"TX_CTS"};
               TX_ACK  :  txControlMainFSMCs_str = {"TX_ACK"};
              TX_BACK  :  txControlMainFSMCs_str = {"TX_BACK"};
           TX_QOSNULL  :  txControlMainFSMCs_str = {"TX_QOSNULL"};
TX_RESP_WAIT_FOR_SIFS  :  txControlMainFSMCs_str = {"TX_RESP_WAIT_FOR_SIFS"};
    TX_RESP_AMPDU_DLM  :  txControlMainFSMCs_str = {"TX_RESP_AMPDU_DLM"};
`ifdef RW_BFMEE_EN
               TX_BFR  :  txControlMainFSMCs_str = {"TX_BFR"};
`endif // RW_BFMEE_EN
           TX_WAIT_END :  txControlMainFSMCs_str = {"TX_WAIT_END"};
    TX_RESP_AMPDU_PAD  :  txControlMainFSMCs_str = {"TX_RESP_AMPDU_PAD"};
         TX_BLANK_DLM  :  txControlMainFSMCs_str = {"TX_BLANK_DLM"};
      TX_RESP_WAIT_PT  :  txControlMainFSMCs_str = {"TX_RESP_WAIT_PT"};
     TX_RESP_WAIT_KSE  :  txControlMainFSMCs_str = {"TX_RESP_WAIT_KSE"};
              default  :  txControlMainFSMCs_str = {"XXX"};
   endcase
end
// pragma coverage block = on 
`endif // RW_SIMU_ON

// System Verilog Assertions
////////////////////////////

`ifdef RW_ASSERT_ON
//$rw_sva This cover point checks if we are transmitting an aMPDU
countIfAMPDUFrame: cover property (@(posedge macCoreTxClk) (mpduDone_p && lastMPDU));



`endif // RW_ASSERT_ON



endmodule
                 
