//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : txParametersCache compressed policy table management
//                    module
//                    
// Simulation Notes : 
//    For simulation, two defines are available
//
//    RW_SIMU_ON   : which creates string signals to display the FSM states on  
//                the waveform viewers
//
//    RW_ASSERT_ON : which enables System Verilog Assertions.
//
// Synthesis Notes  :
//
// Application Note : 
//
// Simulator        :
//     For simulation with RW_ASSERT_ON, the simulator must support System 
//     Verilog Assertions
//
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

// pragma coverage pragmas = on
// pragma coverage implicit_default=off 

module txParametersCacheCPT (
          //$port_g Clock and Reset interface
          input wire         macPITxClk,            // MAC Platform Transmit Clock
          input wire         macPIClkHardRst_n,     // Hard Reset for the MAC Platform Clock domain 
                                                    // active low
          input wire         macPIClkSoftRst_n,     // Soft Reset for the MAC Platform Clock domain 
                                                    // active low
          //$port_g txCtrlReg Interface with the DMA Engine
          input wire         txCtrlRegWr,           // TX parameter Cache module signals
                                                    // Write signal to the Tx Parameter Cache.
          input wire  [31:0] txCtrlRegData,         // TX parameter Cache module signals
                                                    // Write signal to the Tx Parameter Cache.
          input wire         txCtrlRegPT,           // Indicates currently Policy table information is
                                                    // being passed.
          //$port_g Controls
          input  wire        clearSets_p,           // Indicates that the Sets have to be cleared.
          input  wire  [1:0] nFilledPTSets,         // Indicates the number of PT Sets already filled.
          input  wire        currentReadPTSet,      // Indicates if the Policy Table Set0 is free
          input  wire        currentWritePTSet,     // Indicates if the Policy Table Set1 is free
          output wire        pTableManagementBusy,  // Indicates that the P-table FSM is in busy state
          output wire        pTableLastEntry_p,     // Indicates that the received data is the last P-Table Entry

          //$port_g P-Table Fields
          output wire        fecCodingPT,           // FEC Coding
          output wire  [7:0] smmIndexPT,            // Spatial Map Matrix Index
          output wire  [9:0] keySRAMIndex,          // Key Storage RAM Index
          output wire  [6:0] mcsIndexTxRC1          // MCS index for PPDU frame Tx
);

//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

// pTableManagement FSM states definition
//$fsm_sd cPTableManagement
localparam 
              CPT_IDLE    =  2'd0,
              CPT_SECUSER =  2'd1,
              CPT_BUSY    =  2'd2;
  

//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

// cPTableManagement FSM signals definition
reg     [1:0] cPTableManagementCs;     // cPTableManagement FSM Current State
reg     [1:0] cPTableManagementNs;     // cPTableManagement FSM Next State

wire          txCtrlRegPT_p;           // Indicates a rising edge on txCtrlRegPT
reg           txCtrlRegPT_ff1;         // txCtrlRegPT delayed of one clock cycle

wire          cPTableSecUserCntrlInf_p;// Indicates that the received data is P-Table Secondary User Control Information


//Compressed Policy Table Secondary User Information Set 1
reg    [25:0] cPTableSecUserCntrlInfSet1;

//Compressed Policy Table Secondary User Information Set 2
reg    [25:0] cPTableSecUserCntrlInfSet0;


//////////////////////////////////////////////////////////////////////////////
// Beginning of Logic part
//////////////////////////////////////////////////////////////////////////////

// This process generates pulses indicating when the 
// DMA Engine starts a Policy Table transfer
always @ (posedge macPITxClk or negedge macPIClkHardRst_n) 
begin
  if (macPIClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    txCtrlRegPT_ff1 <= 1'b0;
  end
  else if ((macPIClkSoftRst_n == 1'b0) || (clearSets_p == 1'b1))  // Synchronous Reset
  // In case of retry, the Tx Parameters Cache is cleared (synchronous reset).
  begin
    txCtrlRegPT_ff1 <= 1'b0; 
  end
  else
  begin
    txCtrlRegPT_ff1 <= txCtrlRegPT; 
  end
end

assign txCtrlRegPT_p = !txCtrlRegPT_ff1 && txCtrlRegPT;


// Policy Table Management FSM
//////////////////////////////
// cPTableManagement FSM Current State Logic 
always @ (posedge macPITxClk or negedge macPIClkHardRst_n) 
begin
  if (macPIClkHardRst_n == 1'b0)  // Asynchronous Reset
    cPTableManagementCs <= CPT_IDLE; 
  else if ((macPIClkSoftRst_n == 1'b0) || (clearSets_p == 1'b1))  // Synchronous Reset
  // In case of retry, the Tx Parameters Cache is cleared (synchronous reset).
    cPTableManagementCs <= CPT_IDLE;
  else
    cPTableManagementCs <= cPTableManagementNs; 
end


// cPTableManagement FSM Next State Logic.
always @* 
begin
  case(cPTableManagementCs)

    CPT_IDLE:
      //$fsm_s In CPT_IDLE state, the state machine waits for a 
      //Policy table fields transfer from the DMA Engine
      if (txCtrlRegPT_p) 
        //$fsm_t When txCtrlRegPT is received, the state machine goes to CPT_SECUSER state
        cPTableManagementNs = CPT_SECUSER;
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in CPT_IDLE state
        cPTableManagementNs = CPT_IDLE;

    CPT_SECUSER:
      //$fsm_s In CPT_SECUSER state, the state machine waits for the reception of 
      //P-Table CPT_SECUSER Information from the DMA Engine
      if (txCtrlRegWr) 
          //$fsm_t When txCtrlRegWr is received, the state machine moves to CPT_BUSY state
          cPTableManagementNs = CPT_BUSY; 
      else if (!txCtrlRegPT) 
        //$fsm_t If txCtrlRegPT goes low, the state machine moves to CPT_IDLE state
        cPTableManagementNs = CPT_IDLE; 
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in CPT_SECUSER state
        cPTableManagementNs = CPT_SECUSER; 

    CPT_BUSY:
      //$fsm_s In CPT_BUSY state, the state machine waits and the txCtrlReg is busy
      if (nFilledPTSets != 2'd2)
        //$fsm_t If one of the Policy Table Set becomes free, 
        //the state machine goes back to CPT_IDLE state and is ready to receive 
        //another Policy table
        cPTableManagementNs = CPT_IDLE; 
      else
        //$fsm_t If none of the Policy Table Set is free, the state machine stays in 
        //CPT_BUSY state
        cPTableManagementNs = CPT_BUSY; 

    // Disable coverage on the default state because it cannot be reached.
    // pragma coverage block = off 
    default:   
        cPTableManagementNs = CPT_IDLE; 
    // pragma coverage block = on 
  endcase
end


// Policy Table Triggers
////////////////////////
// When the DMA Engine transfer a new part of the Policy Table, a pulse according to the 
// type of data currently transfered (based on the pTableManagement FSM current state) is generated. 
// These pulses are used to stored the coming data to the right location into the PT Sets. 
assign cPTableSecUserCntrlInf_p = ((cPTableManagementCs == CPT_SECUSER) && txCtrlRegWr) ? 1'b1 : 1'b0;

assign pTableManagementBusy     =  (cPTableManagementCs == CPT_BUSY) ? 1'b1 : 1'b0;
assign pTableLastEntry_p        = cPTableSecUserCntrlInf_p;


// Policy Table Sets Update
///////////////////////////
// This process fills the PT Sets with the data comming from the DMA Engine
always @ (posedge macPITxClk or negedge macPIClkHardRst_n) 
begin
  if (macPIClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    cPTableSecUserCntrlInfSet1   <= 26'b0;

    cPTableSecUserCntrlInfSet0   <= 26'b0;
  end
  else if ((macPIClkSoftRst_n == 1'b0) || (clearSets_p == 1'b1))  // Synchronous Reset
  // In case of retry, the Tx Parameters Cache is cleared (synchronous reset).
  begin
    cPTableSecUserCntrlInfSet1   <= 26'b0;

    cPTableSecUserCntrlInfSet0   <= 26'b0;
  end
  else
    if (currentWritePTSet)
    // If the current PT Set write pointer is 1, the PT Set 1 fields
    // are updated with the data comming from the DMA Engine. The field of
    // the PT Set 1 is selected based on the pulses generated by the FSM
    begin
      // Received P-Table Secondary User Control Information data
      if (cPTableSecUserCntrlInf_p)
        cPTableSecUserCntrlInfSet1 <= txCtrlRegData[25:0];
    end  
    else
    // If the current PT Set write pointer is 0, the PT Set 0 fields
    // are updated following the same mechanism than for PT Set 1 previously
    // described
    begin
      // Received P-Table Secondary User Control Information data
      if (cPTableSecUserCntrlInf_p)
        cPTableSecUserCntrlInfSet0 <= txCtrlRegData[25:0];
    end  
end


// Policy Table Ouptuts Assigments
//////////////////////////////////
// Secondary User Control Information output assignment
assign keySRAMIndex       = (currentReadPTSet) ? cPTableSecUserCntrlInfSet1[25:16] :  cPTableSecUserCntrlInfSet0[25:16];
assign smmIndexPT         = (currentReadPTSet) ? cPTableSecUserCntrlInfSet1[15:8]  :  cPTableSecUserCntrlInfSet0[15:8];
assign fecCodingPT        = (currentReadPTSet) ? cPTableSecUserCntrlInfSet1[7]     :  cPTableSecUserCntrlInfSet0[7];
assign mcsIndexTxRC1      = (currentReadPTSet) ? cPTableSecUserCntrlInfSet1[6:0]   :  cPTableSecUserCntrlInfSet0[6:0];



////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON

  // String definition to display cPTableManagement current state
  reg [10*8:0] cPTableManagementCs_str;

  // cPTableManagement FSM states displayed in a string to easy simulation and debug
  always @*
  begin
    case (cPTableManagementCs)
                CPT_IDLE : cPTableManagementCs_str = {"CPT_IDLE"};
             CPT_SECUSER : cPTableManagementCs_str = {"CPT_SECUSER"};
                CPT_BUSY : cPTableManagementCs_str = {"CPT_BUSY"};
                 default : cPTableManagementCs_str = {"XXXX"};
     endcase
  end
`endif // RW_SIMU_ON


// System Verilog Assertions
////////////////////////////

endmodule
