//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author          : $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : txParametersCache policy table management module
//                    
// Simulation Notes : 
//    For simulation, two defines are available
//
//    RW_SIMU_ON   : which creates string signals to display the FSM states on  
//                the waveform viewers
//
//    RW_ASSERT_ON : which enables System Verilog Assertions.
//
// Synthesis Notes  :
//
// Application Note : 
//
// Simulator        :
//     For simulation with RW_ASSERT_ON, the simulator must support System 
//     Verilog Assertions
//
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

// pragma coverage pragmas = on
// pragma coverage implicit_default=off 

module txParametersCachePT ( 
          //$port_g Clock and Reset interface
          input wire         macPITxClk,            // MAC Platform Transmit Clock
          input wire         macPIClkHardRst_n,     // Hard Reset for the MAC Platform Clock domain 
                                                    // active low
          input wire         macPIClkSoftRst_n,     // Soft Reset for the MAC Platform Clock domain 
                                                    // active low
          //$port_g txCtrlReg Interface with the DMA Engine
          input wire         txCtrlRegWr,           // TX parameter Cache module signals
                                                    // Write signal to the Tx Parameter Cache.
          input wire  [31:0] txCtrlRegData,         // TX parameter Cache module signals
                                                    // Write signal to the Tx Parameter Cache.
          input wire         txCtrlRegPT,           // Indicates currently Policy table information is
                                                    // being passed.
          //$port_g Controls
          input  wire        clearSets_p,           // Indicates that the Sets have to be cleared.
          input  wire  [1:0] nFilledPTSets,         // Indicates the number of PT Sets already filled.
          input  wire        currentReadPTSet,      // Indicates if the Policy Table Set0 is free
          input  wire        currentWritePTSet,     // Indicates if the Policy Table Set1 is free
          output wire        pTableManagementBusy,  // Indicates that the P-table FSM is in busy state
          output wire        pTableLastEntry_p,     // Indicates that the received data is the last P-Table Entry

          //$port_g P-Table Fields
          output wire  [3:0] spatialReusePT,        // Spatial Reuse Information
          output wire        midamblePT,            // Midamble periodicity
          output wire        dopplerPT,             // Doppler
          output wire  [2:0] nTxProtPT,             // Number of Transmit Chains for Protection Frame
          output wire  [2:0] nTxPT,                 // Number of Transmit Chains for PPDU
          output wire  [1:0] stbcPT,                // Space Time Block Coding
          output wire        fecCodingPT,           // FEC Coding
          output wire  [1:0] numExtnSSPT,           // Number of Extension Spatial Streams
          output wire        bfFrmExPT,             // Beam Forming Frame Exchange
          output wire        smoothingProtTxPT,     // Smoothing of Protection frame for Transmission
          output wire        smoothingTxPT,         // Smoothing of PPDU for Transmission
          output wire        soundingTxPT,          // Sounding Protection frame for Transmission
          output wire  [2:0] packetExtensionPT,     // Packet Extension Value
          output wire  [5:0] bssColorPT,            // BSS Color
          output wire        upLinkFlagPT,          // UP Link Flag
          output wire        beamChangePT,          // Beam Change
          output wire        beamFormedPT,          // BeamFormed frame
          output wire  [7:0] smmIndexPT,            // Spatial Map Matrix Index
          output wire  [7:0] antennaSetPT,          // Antenna Set
          output wire  [9:0] keySRAMIndex,          // Key Storage RAM Index
          output wire  [9:0] keySRAMIndexRA,        // Key Storage RAM Index for Receiver Address
          output wire [11:0] rtsThreshold,          // RTS Threshold
          output wire  [7:0] shortRetryLimit,       // Short Retry Limit
          output wire  [7:0] longRetryLimit,        // Long Retry Limit
          output wire  [2:0] nRetryRC1,             // Number of trial for this rate control
          output wire  [2:0] formatModProtTxRC1,    // Format and Modulation for Tx Protection frame (RTS/CTS)
          output wire  [1:0] bwProtTxRC1,           // Bandwidth for protection frame
          output wire  [6:0] mcsIndexProtTxRC1,     // MCS index forProtection frame Tx
          output wire  [2:0] navProtFrmExRC1,       // NAV Protection Frame Exchange
          output wire  [2:0] formatModTxRC1,        // Format and Modulation for Tx PPDU
          output wire  [1:0] giTypePreTypeTxRC1,    // Guard Interval of PPDU and Preamble Type for Transmission
          output wire  [1:0] bwTxRC1,               // Bandwidth for PPDU
          output wire  [6:0] mcsIndexTxRC1,         // MCS index for PPDU frame Tx
          output wire        dcmRC1,                // Dual Carrier Modulation
          output wire  [1:0] heLTFTypeRC1,          // HE-LTF Type
          output wire  [7:0] txPwrLevelProtRC1,     // Transmit Power Level of protection
          output wire  [7:0] txPwrLevelRC1,         // Transmit Power Level
          output wire  [2:0] nRetryRC2,             // Number of trial for this rate control
          output wire  [2:0] formatModProtTxRC2,    // Format and Modulation for Tx Protection frame (RTS/CTS)
          output wire  [1:0] bwProtTxRC2,           // Bandwidth for protection frame
          output wire  [6:0] mcsIndexProtTxRC2,     // MCS index forProtection frame Tx
          output wire  [2:0] navProtFrmExRC2,       // NAV Protection Frame Exchange
          output wire  [2:0] formatModTxRC2,        // Format and Modulation for Tx PPDU
          output wire  [1:0] giTypePreTypeTxRC2,    // Guard Interval of PPDU and Preamble Type for Transmission
          output wire  [1:0] bwTxRC2,               // Bandwidth for PPDU
          output wire  [6:0] mcsIndexTxRC2,         // MCS index for PPDU frame Tx
          output wire        dcmRC2,                // Dual Carrier Modulation
          output wire  [1:0] heLTFTypeRC2,          // HE-LTF Type
          output wire  [7:0] txPwrLevelProtRC2,     // Transmit Power Level of protection
          output wire  [7:0] txPwrLevelRC2,         // Transmit Power Level
          output wire  [2:0] nRetryRC3,             // Number of trial for this rate control
          output wire  [2:0] formatModProtTxRC3,    // Format and Modulation for Tx Protection frame (RTS/CTS)
          output wire  [1:0] bwProtTxRC3,           // Bandwidth for protection frame
          output wire  [6:0] mcsIndexProtTxRC3,     // MCS index forProtection frame Tx
          output wire  [2:0] navProtFrmExRC3,       // NAV Protection Frame Exchange
          output wire  [2:0] formatModTxRC3,        // Format and Modulation for Tx PPDU
          output wire  [1:0] giTypePreTypeTxRC3,    // Guard Interval of PPDU and Preamble Type for Transmission
          output wire  [1:0] bwTxRC3,               // Bandwidth for PPDU
          output wire  [6:0] mcsIndexTxRC3,         // MCS index for PPDU frame Tx
          output wire        dcmRC3,                // Dual Carrier Modulation
          output wire  [1:0] heLTFTypeRC3,          // HE-LTF Type
          output wire  [7:0] txPwrLevelProtRC3,     // Transmit Power Level of protection
          output wire  [7:0] txPwrLevelRC3,         // Transmit Power Level
          //output wire  [2:0] nRetryRC4,             // Number of trial for this rate control
          output wire  [2:0] formatModProtTxRC4,    // Format and Modulation for Tx Protection frame (RTS/CTS)
          output wire  [1:0] bwProtTxRC4,           // Bandwidth for protection frame
          output wire  [6:0] mcsIndexProtTxRC4,     // MCS index forProtection frame Tx
          output wire  [2:0] navProtFrmExRC4,       // NAV Protection Frame Exchange
          output wire  [2:0] formatModTxRC4,        // Format and Modulation for Tx PPDU
          output wire  [1:0] giTypePreTypeTxRC4,    // Guard Interval of PPDU and Preamble Type for Transmission
          output wire  [1:0] bwTxRC4,               // Bandwidth for PPDU
          output wire  [6:0] mcsIndexTxRC4,         // MCS index for PPDU frame Tx
          output wire        dcmRC4,                // Dual Carrier Modulation
          output wire  [1:0] heLTFTypeRC4,          // HE-LTF Type
          output wire  [7:0] txPwrLevelProtRC4,     // Transmit Power Level of protection
          output wire  [7:0] txPwrLevelRC4          // Transmit Power Level
);

//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

// pTableManagement FSM states definition
//$fsm_sd pTableManagement
localparam 
              PT_IDLE     =  4'd0,
              PT_PHY1     =  4'd1,
              PT_PHY2     =  4'd2,
              PT_MAC1     =  4'd3,
              PT_MAC2     =  4'd4,
              PT_RATE1    =  4'd5,
              PT_RATE2    =  4'd6,
              PT_RATE3    =  4'd7,
              PT_RATE4    =  4'd8,
              PT_POWER1   =  4'd9,
              PT_POWER2   =  4'd10,
              PT_POWER3   =  4'd11,
              PT_POWER4   =  4'd12,
              PT_BUSY     =  4'd13;


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

// pTableManagement FSM signals definition
reg     [3:0] pTableManagementCs;      // pTableManagement FSM Current State
reg     [3:0] pTableManagementNs;      // pTableManagement FSM Next State

wire          txCtrlRegPT_p;           // Indicates a rising edge on txCtrlRegPT
reg           txCtrlRegPT_ff1;         // txCtrlRegPT delayed of one clock cycle

wire          pTablePHYCntrlInf1_p;    // Indicates that the received data is P-Table PHY Control Information 1
wire          pTablePHYCntrlInf2_p;    // Indicates that the received data is P-Table PHY Control Information 2
wire          pTableMACCntrlInf1_p;    // Indicates that the received data is P-Table MAC Control Information 1
wire          pTableMACCntrlInf2_p;    // Indicates that the received data is P-Table MAC Control Information 2
wire          pTableRATECntrlInf1_p;   // Indicates that the received data is P-Table RATE Control Information 1
wire          pTableRATECntrlInf2_p;   // Indicates that the received data is P-Table RATE Control Information 2
wire          pTableRATECntrlInf3_p;   // Indicates that the received data is P-Table RATE Control Information 3
wire          pTableRATECntrlInf4_p;   // Indicates that the received data is P-Table RATE Control Information 4
wire          pTablePOWERCntrlInf1_p;  // Indicates that the received data is P-Table POWER Control Information 1
wire          pTablePOWERCntrlInf2_p;  // Indicates that the received data is P-Table POWER Control Information 2
wire          pTablePOWERCntrlInf3_p;  // Indicates that the received data is P-Table POWER Control Information 3
wire          pTablePOWERCntrlInf4_p;  // Indicates that the received data is P-Table POWER Control Information 4
              
//Policy Table Information Set 1
reg    [31:0] pTablePHYCntrlInf1Set1;  // P-Table PHY Control Information 1 Set 1
reg    [31:0] pTablePHYCntrlInf2Set1;  // P-Table PHY Control Information 2 Set 1
reg    [31:0] pTableMACCntrlInf1Set1;  // P-Table MAC Control Information 1 Set 1
reg    [31:0] pTableMACCntrlInf2Set1;  // P-Table MAC Control Information 2 Set 1
reg    [31:0] pTableRateCntrlInf1Set1; // P-Table Rate Information Set 1
reg    [31:0] pTableRateCntrlInf2Set1; // P-Table Rate Information Set 1
reg    [31:0] pTableRateCntrlInf3Set1; // P-Table Rate Information Set 1
reg    [31:0] pTableRateCntrlInf4Set1; // P-Table Rate Information Set 1
reg    [31:0] pTablePowerCntrlInf1Set1;// P-Table Power Control Information 1 Set 1
reg    [31:0] pTablePowerCntrlInf2Set1;// P-Table Power Control Information 2 Set 1
reg    [31:0] pTablePowerCntrlInf3Set1;// P-Table Power Control Information 3 Set 1
reg    [31:0] pTablePowerCntrlInf4Set1;// P-Table Power Control Information 4 Set 1

//Policy table Information Set 2
reg    [31:0] pTablePHYCntrlInf1Set0;  // P-Table PHY Control Information 1 Set 2
reg    [31:0] pTablePHYCntrlInf2Set0;  // P-Table PHY Control Information 2 Set 2
reg    [31:0] pTableMACCntrlInf1Set0;  // P-Table MAC Control Information 1 Set 2
reg    [31:0] pTableMACCntrlInf2Set0;  // P-Table MAC Control Information 2 Set 2
reg    [31:0] pTableRateCntrlInf1Set0; // P-Table Rate Information 1 Set 2
reg    [31:0] pTableRateCntrlInf2Set0; // P-Table Rate Information 2 Set 2
reg    [31:0] pTableRateCntrlInf3Set0; // P-Table Rate Information 3 Set 2
reg    [31:0] pTableRateCntrlInf4Set0; // P-Table Rate Information 4 Set 2
reg    [31:0] pTablePowerCntrlInf1Set0;// P-Table Power Control Information 1 Set 2
reg    [31:0] pTablePowerCntrlInf2Set0;// P-Table Power Control Information 2 Set 2
reg    [31:0] pTablePowerCntrlInf3Set0;// P-Table Power Control Information 3 Set 2
reg    [31:0] pTablePowerCntrlInf4Set0;// P-Table Power Control Information 4 Set 2


//////////////////////////////////////////////////////////////////////////////
// Beginning of Logic part
//////////////////////////////////////////////////////////////////////////////

// This process generates pulses indicating when the 
// DMA Engine starts a Policy Table transfer
always @ (posedge macPITxClk or negedge macPIClkHardRst_n) 
begin
  if (macPIClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    txCtrlRegPT_ff1 <= 1'b0;
  end
  else if ((macPIClkSoftRst_n == 1'b0) || (clearSets_p == 1'b1))  // Synchronous Reset
  // In case of retry, the Tx Parameters Cache is cleared (synchronous reset).
  begin
    txCtrlRegPT_ff1 <= 1'b0; 
  end
  else
  begin
    txCtrlRegPT_ff1 <= txCtrlRegPT; 
  end
end

assign txCtrlRegPT_p = !txCtrlRegPT_ff1 && txCtrlRegPT;


// Policy Table Management FSM
//////////////////////////////
// pTableManagement FSM Current State Logic 
always @ (posedge macPITxClk or negedge macPIClkHardRst_n) 
begin
  if (macPIClkHardRst_n == 1'b0)  // Asynchronous Reset
    pTableManagementCs <= PT_IDLE; 
  else if ((macPIClkSoftRst_n == 1'b0) || (clearSets_p == 1'b1))  // Synchronous Reset
  // In case of retry, the Tx Parameters Cache is cleared (synchronous reset).
    pTableManagementCs <= PT_IDLE;
  else
    pTableManagementCs <= pTableManagementNs; 
end


// pTableManagement FSM Next State Logic.
always @* 
begin
  case(pTableManagementCs)

    PT_IDLE:
      //$fsm_s In PT_IDLE state, the state machine waits for a 
      //Policy table fields transfer from the DMA Engine
      if (txCtrlRegPT_p) 
        //$fsm_t When txCtrlRegPT is received, the state machine goes to PT_PHY1 state
        pTableManagementNs = PT_PHY1;
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in PT_IDLE state
        pTableManagementNs = PT_IDLE;

    PT_PHY1:
      //$fsm_s In PT_PHY1 state, the state machine waits for the reception of 
      //P-Table PHY Control Information 1 from the DMA Engine
      if (txCtrlRegWr) 
        //$fsm_t When txCtrlRegWr is received, the state machine goes to PT_PHY2 state
        pTableManagementNs = PT_PHY2; 
      else if (!txCtrlRegPT) 
        //$fsm_t If txCtrlRegPT goes low, the state machine moves to PT_IDLE state
        pTableManagementNs = PT_IDLE; 
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in PT_PHY1 state
        pTableManagementNs = PT_PHY1; 

    PT_PHY2:
      //$fsm_s In PT_PHY2 state, the state machine waits for the reception of 
      //P-Table PHY Control Information 2 from the DMA Engine
      if (txCtrlRegWr) 
        //$fsm_t When txCtrlRegWr is received, the state machine goes to PT_MAC1 state
        pTableManagementNs = PT_MAC1; 
      else if (!txCtrlRegPT) 
        //$fsm_t If txCtrlRegPT goes low, the state machine moves to PT_IDLE state
        pTableManagementNs = PT_IDLE; 
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in PT_PHY2 state
        pTableManagementNs = PT_PHY2; 

    PT_MAC1:
      //$fsm_s In PT_MAC1 state, the state machine waits for the reception of 
      //P-Table MAC Control Information 1 from the DMA Engine
      if (txCtrlRegWr) 
        //$fsm_t When txCtrlRegWr is received, the state machine goes to PT_MAC2 state
        pTableManagementNs = PT_MAC2;
      else if (!txCtrlRegPT) 
        //$fsm_t If txCtrlRegPT goes low, the state machine moves to PT_IDLE state
        pTableManagementNs = PT_IDLE; 
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in PT_MAC1 state
        pTableManagementNs = PT_MAC1; 

    PT_MAC2:
      //$fsm_s In PT_MAC2 state, the state machine waits for the reception of 
      //P-Table MAC Control Information 2 from the DMA Engine
      if (txCtrlRegWr) 
        //$fsm_t When txCtrlRegWr is received, the state machine goes to PT_RATE1 state
        pTableManagementNs = PT_RATE1;
      else if (!txCtrlRegPT) 
        //$fsm_t If txCtrlRegPT goes low, the state machine moves to PT_IDLE state
        pTableManagementNs = PT_IDLE; 
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in PT_MAC2 state
        pTableManagementNs = PT_MAC2; 

    PT_RATE1:
      //$fsm_s In PT_RATE1 state, the state machine waits for the reception of 
      //P-Table PT_RATE1 Information from the DMA Engine
      if (txCtrlRegWr) 
          //$fsm_t When txCtrlRegWr is received, the state machine moves to PT_RATE2 state
          pTableManagementNs = PT_RATE2; 
      else if (!txCtrlRegPT) 
        //$fsm_t If txCtrlRegPT goes low, the state machine moves to PT_IDLE state
        pTableManagementNs = PT_IDLE; 
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in PT_RATE1 state
        pTableManagementNs = PT_RATE1; 

    PT_RATE2:
      //$fsm_s In PT_RATE2 state, the state machine waits for the reception of 
      //P-Table PT_RATE2 Information from the DMA Engine
      if (txCtrlRegWr) 
          //$fsm_t When txCtrlRegWr is received, the state machine moves to PT_RATE3 state
          pTableManagementNs = PT_RATE3; 
      else if (!txCtrlRegPT) 
        //$fsm_t If txCtrlRegPT goes low, the state machine moves to PT_IDLE state
        pTableManagementNs = PT_IDLE; 
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in PT_RATE2 state
        pTableManagementNs = PT_RATE2; 

    PT_RATE3:
      //$fsm_s In PT_RATE3 state, the state machine waits for the reception of 
      //P-Table PT_RATE3 Information from the DMA Engine
      if (txCtrlRegWr) 
          //$fsm_t When txCtrlRegWr is received, the state machine moves to PT_RATE3 state
          pTableManagementNs = PT_RATE4; 
      else if (!txCtrlRegPT) 
        //$fsm_t If txCtrlRegPT goes low, the state machine moves to PT_IDLE state
        pTableManagementNs = PT_IDLE; 
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in PT_RATE3 state
        pTableManagementNs = PT_RATE3; 

    PT_RATE4:
      //$fsm_s In PT_RATE4 state, the state machine waits for the reception of 
      //P-Table PT_RATE4 Information from the DMA Engine
      if (txCtrlRegWr) 
          //$fsm_t When txCtrlRegWr is received, the state machine moves to PT_BUSY state
          pTableManagementNs = PT_POWER1; 
      else if (!txCtrlRegPT) 
        //$fsm_t If txCtrlRegPT goes low, the state machine moves to PT_IDLE state
        pTableManagementNs = PT_IDLE; 
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in PT_RATE4 state
        pTableManagementNs = PT_RATE4; 

    PT_POWER1:
      //$fsm_s In PT_POWER1 state, the state machine waits for the reception of 
      //P-Table PT_POWER1 Information from the DMA Engine
      if (txCtrlRegWr) 
          //$fsm_t When txCtrlRegWr is received, the state machine moves to PT_POWER2 state
          pTableManagementNs = PT_POWER2; 
      else if (!txCtrlRegPT) 
        //$fsm_t If txCtrlRegPT goes low, the state machine moves to PT_IDLE state
        pTableManagementNs = PT_IDLE; 
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in PT_POWER1 state
        pTableManagementNs = PT_POWER1; 

    PT_POWER2:
      //$fsm_s In PT_POWER2 state, the state machine waits for the reception of 
      //P-Table PT_POWER2 Information from the DMA Engine
      if (txCtrlRegWr) 
          //$fsm_t When txCtrlRegWr is received, the state machine moves to PT_POWER3 state
          pTableManagementNs = PT_POWER3; 
      else if (!txCtrlRegPT) 
        //$fsm_t If txCtrlRegPT goes low, the state machine moves to PT_IDLE state
        pTableManagementNs = PT_IDLE; 
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in PT_POWER2 state
        pTableManagementNs = PT_POWER2; 

    PT_POWER3:
      //$fsm_s In PT_POWER3 state, the state machine waits for the reception of 
      //P-Table PT_POWER3 Information from the DMA Engine
      if (txCtrlRegWr) 
          //$fsm_t When txCtrlRegWr is received, the state machine moves to PT_POWER3 state
          pTableManagementNs = PT_POWER4; 
      else if (!txCtrlRegPT) 
        //$fsm_t If txCtrlRegPT goes low, the state machine moves to PT_IDLE state
        pTableManagementNs = PT_IDLE; 
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in PT_POWER3 state
        pTableManagementNs = PT_POWER3; 

    PT_POWER4:
      //$fsm_s In PT_POWER4 state, the state machine waits for the reception of 
      //P-Table PT_POWER4 Information from the DMA Engine
      if (txCtrlRegWr) 
          //$fsm_t When txCtrlRegWr is received, the state machine moves to PT_BUSY state
          pTableManagementNs = PT_BUSY; 
      else if (!txCtrlRegPT) 
        //$fsm_t If txCtrlRegPT goes low, the state machine moves to PT_IDLE state
        pTableManagementNs = PT_IDLE; 
      else
        //$fsm_t If txCtrlRegWr is low, the state machine stays in PT_POWER4 state
        pTableManagementNs = PT_POWER4; 

    PT_BUSY:
      //$fsm_s In PT_BUSY state, the state machine waits and the txCtrlReg is busy
      if (nFilledPTSets != 2'd2)
        //$fsm_t If one of the Policy Table Set becomes free, 
        //the state machine goes back to PT_IDLE state and is ready to receive 
        //another Policy table
        pTableManagementNs = PT_IDLE; 
      else
        //$fsm_t If none of the Policy Table Set is free, the state machine stays in 
        //PT_BUSY state
        pTableManagementNs = PT_BUSY; 

    // Disable coverage on the default state because it cannot be reached.
    // pragma coverage block = off 
    default:   
        pTableManagementNs = PT_IDLE; 
    // pragma coverage block = on 
  endcase
end


// Policy Table Triggers
////////////////////////
// When the DMA Engine transfer a new part of the Policy Table, a pulse according to the 
// type of data currently transfered (based on the pTableManagement FSM current state) is generated. 
// These pulses are used to stored the coming data to the right location into the PT Sets. 
assign pTablePHYCntrlInf1_p    = ((pTableManagementCs == PT_PHY1)    && txCtrlRegWr) ? 1'b1 : 1'b0;
assign pTablePHYCntrlInf2_p    = ((pTableManagementCs == PT_PHY2)    && txCtrlRegWr) ? 1'b1 : 1'b0;
assign pTableMACCntrlInf1_p    = ((pTableManagementCs == PT_MAC1)    && txCtrlRegWr) ? 1'b1 : 1'b0;
assign pTableMACCntrlInf2_p    = ((pTableManagementCs == PT_MAC2)    && txCtrlRegWr) ? 1'b1 : 1'b0;
assign pTableRATECntrlInf1_p   = ((pTableManagementCs == PT_RATE1)   && txCtrlRegWr) ? 1'b1 : 1'b0;
assign pTableRATECntrlInf2_p   = ((pTableManagementCs == PT_RATE2)   && txCtrlRegWr) ? 1'b1 : 1'b0;
assign pTableRATECntrlInf3_p   = ((pTableManagementCs == PT_RATE3)   && txCtrlRegWr) ? 1'b1 : 1'b0;
assign pTableRATECntrlInf4_p   = ((pTableManagementCs == PT_RATE4)   && txCtrlRegWr) ? 1'b1 : 1'b0;
assign pTablePOWERCntrlInf1_p  = ((pTableManagementCs == PT_POWER1)  && txCtrlRegWr) ? 1'b1 : 1'b0;
assign pTablePOWERCntrlInf2_p  = ((pTableManagementCs == PT_POWER2)  && txCtrlRegWr) ? 1'b1 : 1'b0;
assign pTablePOWERCntrlInf3_p  = ((pTableManagementCs == PT_POWER3)  && txCtrlRegWr) ? 1'b1 : 1'b0;
assign pTablePOWERCntrlInf4_p  = ((pTableManagementCs == PT_POWER4)  && txCtrlRegWr) ? 1'b1 : 1'b0;

assign pTableManagementBusy    =  (pTableManagementCs == PT_BUSY) ? 1'b1 : 1'b0;
assign pTableLastEntry_p       = pTablePOWERCntrlInf4_p;


// Policy Table Sets Update
///////////////////////////
// This process fills the PT Sets with the data comming from the DMA Engine
always @ (posedge macPITxClk or negedge macPIClkHardRst_n) 
begin
  if (macPIClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    pTablePHYCntrlInf1Set1   <= 32'b0;
    pTablePHYCntrlInf2Set1   <= 32'b0;
    pTableMACCntrlInf1Set1   <= 32'b0;
    pTableMACCntrlInf2Set1   <= 32'b0;
    pTableRateCntrlInf1Set1  <= 32'b0;
    pTableRateCntrlInf2Set1  <= 32'b0;
    pTableRateCntrlInf3Set1  <= 32'b0;
    pTableRateCntrlInf4Set1  <= 32'b0;
    pTablePowerCntrlInf1Set1 <= 32'b0;
    pTablePowerCntrlInf2Set1 <= 32'b0;
    pTablePowerCntrlInf3Set1 <= 32'b0;
    pTablePowerCntrlInf4Set1 <= 32'b0;

    pTablePHYCntrlInf1Set0   <= 32'b0;
    pTablePHYCntrlInf2Set0   <= 32'b0;
    pTableMACCntrlInf1Set0   <= 32'b0;
    pTableMACCntrlInf2Set0   <= 32'b0;
    pTableRateCntrlInf1Set0  <= 32'b0;
    pTableRateCntrlInf2Set0  <= 32'b0;
    pTableRateCntrlInf3Set0  <= 32'b0;
    pTableRateCntrlInf4Set0  <= 32'b0;
    pTablePowerCntrlInf1Set0 <= 32'b0;
    pTablePowerCntrlInf2Set0 <= 32'b0;
    pTablePowerCntrlInf3Set0 <= 32'b0;
    pTablePowerCntrlInf4Set0 <= 32'b0;
  end
  else if ((macPIClkSoftRst_n == 1'b0) || (clearSets_p == 1'b1))  // Synchronous Reset
  // In case of retry, the Tx Parameters Cache is cleared (synchronous reset).
  begin
    pTablePHYCntrlInf1Set1   <= 32'b0;
    pTablePHYCntrlInf2Set1   <= 32'b0;
    pTableMACCntrlInf1Set1   <= 32'b0;
    pTableMACCntrlInf2Set1   <= 32'b0;
    pTableRateCntrlInf1Set1  <= 32'b0;
    pTableRateCntrlInf2Set1  <= 32'b0;
    pTableRateCntrlInf3Set1  <= 32'b0;
    pTableRateCntrlInf4Set1  <= 32'b0;
    pTablePowerCntrlInf1Set1 <= 32'b0;
    pTablePowerCntrlInf2Set1 <= 32'b0;
    pTablePowerCntrlInf3Set1 <= 32'b0;
    pTablePowerCntrlInf4Set1 <= 32'b0;

    pTablePHYCntrlInf1Set0   <= 32'b0;
    pTablePHYCntrlInf2Set0   <= 32'b0;
    pTableMACCntrlInf1Set0   <= 32'b0;
    pTableMACCntrlInf2Set0   <= 32'b0;
    pTableRateCntrlInf1Set0  <= 32'b0;
    pTableRateCntrlInf2Set0  <= 32'b0;
    pTableRateCntrlInf3Set0  <= 32'b0;
    pTableRateCntrlInf4Set0  <= 32'b0;
    pTablePowerCntrlInf1Set0 <= 32'b0;
    pTablePowerCntrlInf2Set0 <= 32'b0;
    pTablePowerCntrlInf3Set0 <= 32'b0;
    pTablePowerCntrlInf4Set0 <= 32'b0;
  end
  else
    if (currentWritePTSet)
    // If the current PT Set write pointer is 1, the PT Set 1 fields
    // are updated with the data comming from the DMA Engine. The field of
    // the PT Set 1 is selected based on the pulses generated by the FSM
    begin
      // Received P-Table PHY Control Information 1 data
      if (pTablePHYCntrlInf1_p)
        pTablePHYCntrlInf1Set1 <= txCtrlRegData;

      // Received P-Table PHY Control Information 2 data
      else if (pTablePHYCntrlInf2_p)
        pTablePHYCntrlInf2Set1 <= txCtrlRegData;

      // Received P-Table MAC Control Information 1 data
      else if (pTableMACCntrlInf1_p)
        pTableMACCntrlInf1Set1 <= txCtrlRegData;

      // Received P-Table MAC Control Information 2 data
      else if (pTableMACCntrlInf2_p)
        pTableMACCntrlInf2Set1 <= txCtrlRegData;

      // Received P-Table RATE Control Information 1 data
      else if(pTableRATECntrlInf1_p)
        pTableRateCntrlInf1Set1 <= txCtrlRegData;

      // Received P-Table RATE Control Information 2 data
      else if(pTableRATECntrlInf2_p)
        pTableRateCntrlInf2Set1 <= txCtrlRegData;

      // Received P-Table RATE Control Information 3 data
      else if(pTableRATECntrlInf3_p)
        pTableRateCntrlInf3Set1 <= txCtrlRegData;

      // Received P-Table RATE Control Information 4 data
      else if(pTableRATECntrlInf4_p)
        pTableRateCntrlInf4Set1 <= txCtrlRegData;

      // Received P-Table POWER Control Information 1 data
      else if(pTablePOWERCntrlInf1_p)
        pTablePowerCntrlInf1Set1 <= txCtrlRegData;

      // Received P-Table POWER Control Information 2 data
      else if(pTablePOWERCntrlInf2_p)
        pTablePowerCntrlInf2Set1 <= txCtrlRegData;

      // Received P-Table POWER Control Information 3 data
      else if(pTablePOWERCntrlInf3_p)
        pTablePowerCntrlInf3Set1 <= txCtrlRegData;

      // Received P-Table POWER Control Information 4 data
      else if(pTablePOWERCntrlInf4_p)
        pTablePowerCntrlInf4Set1 <= txCtrlRegData;

    end  
    else
    // If the current PT Set write pointer is 0, the PT Set 0 fields
    // are updated following the same mechanism than for PT Set 1 previously
    // described
    begin
      // Received P-Table PHY Control Information 1 data
      if (pTablePHYCntrlInf1_p)
        pTablePHYCntrlInf1Set0 <= txCtrlRegData;

      // Received P-Table PHY Control Information 2 data
      else if (pTablePHYCntrlInf2_p)
        pTablePHYCntrlInf2Set0 <= txCtrlRegData;

      // Received P-Table MAC Control Information 1 data
      else if (pTableMACCntrlInf1_p)
        pTableMACCntrlInf1Set0 <= txCtrlRegData;

      // Received P-Table MAC Control Information 2 data
      else if (pTableMACCntrlInf2_p)
        pTableMACCntrlInf2Set0 <= txCtrlRegData;

      // Received P-Table RATE Control Information 1 data
      else if(pTableRATECntrlInf1_p)
        pTableRateCntrlInf1Set0 <= txCtrlRegData;

      // Received P-Table RATE Control Information 2 data
      else if(pTableRATECntrlInf2_p)
        pTableRateCntrlInf2Set0 <= txCtrlRegData;

      // Received P-Table RATE Control Information 3 data
      else if(pTableRATECntrlInf3_p)
        pTableRateCntrlInf3Set0 <= txCtrlRegData;

      // Received P-Table RATE Control Information 4 data
      else if(pTableRATECntrlInf4_p)
        pTableRateCntrlInf4Set0 <= txCtrlRegData;

      // Received P-Table POWER Control Information 1 data
      else if(pTablePOWERCntrlInf1_p)
        pTablePowerCntrlInf1Set0 <= txCtrlRegData;

      // Received P-Table POWER Control Information 2 data
      else if(pTablePOWERCntrlInf2_p)
        pTablePowerCntrlInf2Set0 <= txCtrlRegData;

      // Received P-Table POWER Control Information 3 data
      else if(pTablePOWERCntrlInf3_p)
        pTablePowerCntrlInf3Set0 <= txCtrlRegData;

      // Received P-Table POWER Control Information 4 data
      else if(pTablePOWERCntrlInf4_p)
        pTablePowerCntrlInf4Set0 <= txCtrlRegData;

    end  
end


// Policy Table Ouptuts Assigments
//////////////////////////////////
// PHY Control Information 1 output assignment
assign spatialReusePT     = (currentReadPTSet) ? pTablePHYCntrlInf1Set1[27:24]    :  pTablePHYCntrlInf1Set0[27:24];
assign midamblePT         = (currentReadPTSet) ? pTablePHYCntrlInf1Set1[21]       :  pTablePHYCntrlInf1Set0[21];
assign dopplerPT          = (currentReadPTSet) ? pTablePHYCntrlInf1Set1[20]       :  pTablePHYCntrlInf1Set0[20];
assign nTxProtPT          = (currentReadPTSet) ? pTablePHYCntrlInf1Set1[19:17]    :  pTablePHYCntrlInf1Set0[19:17];
assign nTxPT              = (currentReadPTSet) ? pTablePHYCntrlInf1Set1[16:14]    :  pTablePHYCntrlInf1Set0[16:14];
assign stbcPT             = (currentReadPTSet) ? pTablePHYCntrlInf1Set1[8:7]      :  pTablePHYCntrlInf1Set0[8:7];
assign fecCodingPT        = (currentReadPTSet) ? pTablePHYCntrlInf1Set1[6]        :  pTablePHYCntrlInf1Set0[6];
assign numExtnSSPT        = (currentReadPTSet) ? pTablePHYCntrlInf1Set1[5:4]      :  pTablePHYCntrlInf1Set0[5:4];
assign bfFrmExPT          = (currentReadPTSet) ? pTablePHYCntrlInf1Set1[3]        :  pTablePHYCntrlInf1Set0[3];
assign smoothingProtTxPT  = (currentReadPTSet) ? pTablePHYCntrlInf1Set1[2]        :  pTablePHYCntrlInf1Set0[2];
assign smoothingTxPT      = (currentReadPTSet) ? pTablePHYCntrlInf1Set1[1]        :  pTablePHYCntrlInf1Set0[1];
assign soundingTxPT       = (currentReadPTSet) ? pTablePHYCntrlInf1Set1[0]        :  pTablePHYCntrlInf1Set0[0];

// PHY Control Information 2
assign packetExtensionPT  = (currentReadPTSet) ? pTablePHYCntrlInf2Set1[28:26]    :  pTablePHYCntrlInf2Set0[28:26];
assign bssColorPT         = (currentReadPTSet) ? pTablePHYCntrlInf2Set1[25:20]    :  pTablePHYCntrlInf2Set0[25:20];
assign upLinkFlagPT       = (currentReadPTSet) ? pTablePHYCntrlInf2Set1[18]       :  pTablePHYCntrlInf2Set0[18];
assign beamChangePT       = (currentReadPTSet) ? pTablePHYCntrlInf2Set1[17]       :  pTablePHYCntrlInf2Set0[17];
assign beamFormedPT       = (currentReadPTSet) ? pTablePHYCntrlInf2Set1[16]       :  pTablePHYCntrlInf2Set0[16];
assign smmIndexPT         = (currentReadPTSet) ? pTablePHYCntrlInf2Set1[15:8]     :  pTablePHYCntrlInf2Set0[15:8];
assign antennaSetPT       = (currentReadPTSet) ? pTablePHYCntrlInf2Set1[7:0]      :  pTablePHYCntrlInf2Set0[7:0];

// MAC Control Information 1
assign keySRAMIndexRA     = (currentReadPTSet) ? pTableMACCntrlInf1Set1[19:10]    :  pTableMACCntrlInf1Set0[19:10];
assign keySRAMIndex       = (currentReadPTSet) ? pTableMACCntrlInf1Set1[9:0]      :  pTableMACCntrlInf1Set0[9:0];

// MAC Control Information 2
assign rtsThreshold       = (currentReadPTSet) ? pTableMACCntrlInf2Set1[27:16]    :  pTableMACCntrlInf2Set0[27:16];
assign shortRetryLimit    = (currentReadPTSet) ? pTableMACCntrlInf2Set1[15:8]     :  pTableMACCntrlInf2Set0[15:8];
assign longRetryLimit     = (currentReadPTSet) ? pTableMACCntrlInf2Set1[7:0]      :  pTableMACCntrlInf2Set0[7:0];

//RATE Control Information 1
assign nRetryRC1          = (currentReadPTSet) ? pTableRateCntrlInf1Set1[31:29]   :  pTableRateCntrlInf1Set0[31:29];
assign formatModProtTxRC1 = (currentReadPTSet) ? pTableRateCntrlInf1Set1[28:26]   :  pTableRateCntrlInf1Set0[28:26];
assign bwProtTxRC1        = (currentReadPTSet) ? pTableRateCntrlInf1Set1[25:24]   :  pTableRateCntrlInf1Set0[25:24];
assign mcsIndexProtTxRC1  = (currentReadPTSet) ? pTableRateCntrlInf1Set1[23:17]   :  pTableRateCntrlInf1Set0[23:17];
assign navProtFrmExRC1    = (currentReadPTSet) ? pTableRateCntrlInf1Set1[16:14]   :  pTableRateCntrlInf1Set0[16:14];
assign formatModTxRC1     = (currentReadPTSet) ? pTableRateCntrlInf1Set1[13:11]   :  pTableRateCntrlInf1Set0[13:11];
assign giTypePreTypeTxRC1 = (currentReadPTSet) ? pTableRateCntrlInf1Set1[10:9]    :  pTableRateCntrlInf1Set0[10:9];
assign bwTxRC1            = (currentReadPTSet) ? pTableRateCntrlInf1Set1[8:7]     :  pTableRateCntrlInf1Set0[8:7];
assign mcsIndexTxRC1      = (currentReadPTSet) ? pTableRateCntrlInf1Set1[6:0]     :  pTableRateCntrlInf1Set0[6:0];

//POWER Control Information 1
assign dcmRC1             = (currentReadPTSet) ? pTablePowerCntrlInf1Set1[18]     :  pTablePowerCntrlInf1Set0[18];
assign heLTFTypeRC1       = (currentReadPTSet) ? pTablePowerCntrlInf1Set1[17:16]  :  pTablePowerCntrlInf1Set0[17:16];
assign txPwrLevelProtRC1  = (currentReadPTSet) ? pTablePowerCntrlInf1Set1[15:8]   :  pTablePowerCntrlInf1Set0[15:8];
assign txPwrLevelRC1      = (currentReadPTSet) ? pTablePowerCntrlInf1Set1[7:0]    :  pTablePowerCntrlInf1Set0[7:0];

//RATE Control Information 2
assign nRetryRC2          = (currentReadPTSet) ? pTableRateCntrlInf2Set1[31:29]   :  pTableRateCntrlInf2Set0[31:29];
assign formatModProtTxRC2 = (currentReadPTSet) ? pTableRateCntrlInf2Set1[28:26]   :  pTableRateCntrlInf2Set0[28:26];
assign bwProtTxRC2        = (currentReadPTSet) ? pTableRateCntrlInf2Set1[25:24]   :  pTableRateCntrlInf2Set0[25:24];
assign mcsIndexProtTxRC2  = (currentReadPTSet) ? pTableRateCntrlInf2Set1[23:17]   :  pTableRateCntrlInf2Set0[23:17];
assign navProtFrmExRC2    = (currentReadPTSet) ? pTableRateCntrlInf2Set1[16:14]   :  pTableRateCntrlInf2Set0[16:14];
assign formatModTxRC2     = (currentReadPTSet) ? pTableRateCntrlInf2Set1[13:11]   :  pTableRateCntrlInf2Set0[13:11];
assign giTypePreTypeTxRC2 = (currentReadPTSet) ? pTableRateCntrlInf2Set1[10:9]    :  pTableRateCntrlInf2Set0[10:9];
assign bwTxRC2            = (currentReadPTSet) ? pTableRateCntrlInf2Set1[8:7]     :  pTableRateCntrlInf2Set0[8:7];
assign mcsIndexTxRC2      = (currentReadPTSet) ? pTableRateCntrlInf2Set1[6:0]     :  pTableRateCntrlInf2Set0[6:0];

//POWER Control Information 2
assign dcmRC2             = (currentReadPTSet) ? pTablePowerCntrlInf2Set1[18]     :  pTablePowerCntrlInf2Set0[18];
assign heLTFTypeRC2       = (currentReadPTSet) ? pTablePowerCntrlInf2Set1[17:16]  :  pTablePowerCntrlInf2Set0[17:16];
assign txPwrLevelProtRC2  = (currentReadPTSet) ? pTablePowerCntrlInf2Set1[15:8]   :  pTablePowerCntrlInf2Set0[15:8];
assign txPwrLevelRC2      = (currentReadPTSet) ? pTablePowerCntrlInf2Set1[7:0]    :  pTablePowerCntrlInf2Set0[7:0];

//RATE Control Information 3
assign nRetryRC3          = (currentReadPTSet) ? pTableRateCntrlInf3Set1[31:29]   :  pTableRateCntrlInf3Set0[31:29];
assign formatModProtTxRC3 = (currentReadPTSet) ? pTableRateCntrlInf3Set1[28:26]   :  pTableRateCntrlInf3Set0[28:26];
assign bwProtTxRC3        = (currentReadPTSet) ? pTableRateCntrlInf3Set1[25:24]   :  pTableRateCntrlInf3Set0[25:24];
assign mcsIndexProtTxRC3  = (currentReadPTSet) ? pTableRateCntrlInf3Set1[23:17]   :  pTableRateCntrlInf3Set0[23:17];
assign navProtFrmExRC3    = (currentReadPTSet) ? pTableRateCntrlInf3Set1[16:14]   :  pTableRateCntrlInf3Set0[16:14];
assign formatModTxRC3     = (currentReadPTSet) ? pTableRateCntrlInf3Set1[13:11]   :  pTableRateCntrlInf3Set0[13:11];
assign giTypePreTypeTxRC3 = (currentReadPTSet) ? pTableRateCntrlInf3Set1[10:9]    :  pTableRateCntrlInf3Set0[10:9];
assign bwTxRC3            = (currentReadPTSet) ? pTableRateCntrlInf3Set1[8:7]     :  pTableRateCntrlInf3Set0[8:7];
assign mcsIndexTxRC3      = (currentReadPTSet) ? pTableRateCntrlInf3Set1[6:0]     :  pTableRateCntrlInf3Set0[6:0];

//POWER Control Information 3
assign dcmRC3             = (currentReadPTSet) ? pTablePowerCntrlInf3Set1[18]     :  pTablePowerCntrlInf3Set0[18];
assign heLTFTypeRC3       = (currentReadPTSet) ? pTablePowerCntrlInf3Set1[17:16]  :  pTablePowerCntrlInf3Set0[17:16];
assign txPwrLevelProtRC3  = (currentReadPTSet) ? pTablePowerCntrlInf3Set1[15:8]   :  pTablePowerCntrlInf3Set0[15:8];
assign txPwrLevelRC3      = (currentReadPTSet) ? pTablePowerCntrlInf3Set1[7:0]    :  pTablePowerCntrlInf3Set0[7:0];

//RATE Control Information 4
//assign nRetryRC4          = (currentReadPTSet) ? pTableRateCntrlInf4Set1[31:29]   :  pTableRateCntrlInf4Set0[31:29];
assign formatModProtTxRC4 = (currentReadPTSet) ? pTableRateCntrlInf4Set1[28:26]   :  pTableRateCntrlInf4Set0[28:26];
assign bwProtTxRC4        = (currentReadPTSet) ? pTableRateCntrlInf4Set1[25:24]   :  pTableRateCntrlInf4Set0[25:24];
assign mcsIndexProtTxRC4  = (currentReadPTSet) ? pTableRateCntrlInf4Set1[23:17]   :  pTableRateCntrlInf4Set0[23:17];
assign navProtFrmExRC4    = (currentReadPTSet) ? pTableRateCntrlInf4Set1[16:14]   :  pTableRateCntrlInf4Set0[16:14];
assign formatModTxRC4     = (currentReadPTSet) ? pTableRateCntrlInf4Set1[13:11]   :  pTableRateCntrlInf4Set0[13:11];
assign giTypePreTypeTxRC4 = (currentReadPTSet) ? pTableRateCntrlInf4Set1[10:9]    :  pTableRateCntrlInf4Set0[10:9];
assign bwTxRC4            = (currentReadPTSet) ? pTableRateCntrlInf4Set1[8:7]     :  pTableRateCntrlInf4Set0[8:7];
assign mcsIndexTxRC4      = (currentReadPTSet) ? pTableRateCntrlInf4Set1[6:0]     :  pTableRateCntrlInf4Set0[6:0];

//POWER Control Information 4
assign dcmRC4             = (currentReadPTSet) ? pTablePowerCntrlInf4Set1[18]     :  pTablePowerCntrlInf4Set0[18];
assign heLTFTypeRC4       = (currentReadPTSet) ? pTablePowerCntrlInf4Set1[17:16]  :  pTablePowerCntrlInf4Set0[17:16];
assign txPwrLevelProtRC4  = (currentReadPTSet) ? pTablePowerCntrlInf4Set1[15:8]   :  pTablePowerCntrlInf4Set0[15:8];
assign txPwrLevelRC4      = (currentReadPTSet) ? pTablePowerCntrlInf4Set1[7:0]    :  pTablePowerCntrlInf4Set0[7:0];


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
  // String definition to display pTableManagement current state
  reg [10*8:0] pTableManagementCs_str;

  // pTableManagement FSM states displayed in a string to easy simulation and debug
  always @*
  begin
    case (pTableManagementCs)
                PT_IDLE  :  pTableManagementCs_str = {"PT_IDLE"};
                PT_PHY1  :  pTableManagementCs_str = {"PT_PHY1"};
                PT_PHY2  :  pTableManagementCs_str = {"PT_PHY2"};
                PT_MAC1  :  pTableManagementCs_str = {"PT_MAC1"};
                PT_MAC2  :  pTableManagementCs_str = {"PT_MAC2"};
               PT_RATE1  :  pTableManagementCs_str = {"PT_RATE1"};
               PT_RATE2  :  pTableManagementCs_str = {"PT_RATE2"};
               PT_RATE3  :  pTableManagementCs_str = {"PT_RATE3"};
               PT_RATE4  :  pTableManagementCs_str = {"PT_RATE4"};
              PT_POWER1  :  pTableManagementCs_str = {"PT_POWER1"};
              PT_POWER2  :  pTableManagementCs_str = {"PT_POWER2"};
              PT_POWER3  :  pTableManagementCs_str = {"PT_POWER3"};
              PT_POWER4  :  pTableManagementCs_str = {"PT_POWER4"};
                PT_BUSY  :  pTableManagementCs_str = {"PT_BUSY"};
                default  :  pTableManagementCs_str = {"XXXX"};
     endcase
  end
`endif // RW_SIMU_ON


// System Verilog Assertions
////////////////////////////

endmodule
