//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: jvanthou $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 30461 $
// $Date: 2017-05-22 16:39:46 +0200 (Mon, 22 May 2017) $
// ---------------------------------------------------------------------------
// Dependencies     : None                                                      
// Description      : Top level of txTimeCalculator module
//                    
// Simulation Notes : 
// Synthesis Notes  : 
//   This design contains two fully asynchronuous clock domains 
//   (macPIClk and macCoreClk). The clock domain crossing is handled in two
//   different ways. 
//     1- All the triggers signals are resynchronized using the 
//        risingEdgeDetector module (resynchronization ff are named _resync).
//     2- All the others signals are not resynchronized because are as per 
//        design, static and can be safely captured when the signals 
//        resynchronized by risingEdgeDetector are high.
//  
//  As a consequence of that, a flash path can be fined between the two clock 
//  domain.
//
// Application Note :                                                       
// Simulator        :                                                       
// Parameters       :                                                       
// Terms & concepts :                                                       
// Bugs             :                                                       
// Open issues and future enhancements :                                    
// References       :                                                       
// Revision History :                                                       
// ---------------------------------------------------------------------------
//                                                                          
// 
// 
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

module txTimeCalculator ( 
          //$port_g Clock and Reset interface
          input  wire        macPIClk,                // MAC Platform Interface Transmit Clock
          input  wire        macPIClkHardRst_n,       // Hard Reset of the MAC Platform Interface Clock domain 
                                                      // active low

          input  wire        macCoreClk,              // MAC Platform Interface Transmit Clock
          input  wire        macCoreClkHardRst_n,     // Hard Reset of the MAC Platform Interface Clock domain 
                                                      // active low

          //$port_g Tx Controller interface
          output  reg        disambiguityMC,          // If VHT and the nSymb modulo 10 is 9 then the flag is high

          //$port_g MAC Controller interface
          input  wire [19:0] ppduLengthMC,            // PPDU Length from MAC Controller
                                                      // This field gives the length of the PPDU for computing time on air.
          input  wire  [3:0] ppduPreTypeMC,           // PPDU Preamble Type from MAC Controller
                                                      // This field indicates what type of preamble is transmitted for the frame.
                                                      // The encoding is as follows:
                                                      // 4'b0000: Non-HT-Short
                                                      // 4'b0001: Non-HT-Long
                                                      // 4'b0010: HT-MF
                                                      // 4'b0011: HT-GF
                                                      // 4'b0100: VHT
                                                      // 4'b0101: HE-SU
                                                      // 4'b0110: HE-MU
                                                      // 4'b0111: HE-ER-SU
                                                      // 4'b1000: HE-TB
          input  wire  [6:0] ppduMCSIndexMC,          // PPDU LegRate or MCS Index from MAC Controller
                                                      // This field indicates the rate at which this PPDU is transmitted.
          input  wire  [1:0] ppduChBwMC,              // This field indicates the channel bandwidth
          input  wire  [1:0] ppduNumExtnSSMC,         // Number of Extension Spatial Streams      
          input  wire  [1:0] ppduSTBCMC,              // Enable Space Time Block Coding          
          input  wire  [1:0] ppduGITypeMC,            // PPDU Guard Interval type from MAC Controller
                                                      // Indicates that the frame should be transmitted with Short GI.
`ifdef  RW_MUMIMO_TX_EN
          input  wire  [2:0] ppduSecNSTSMC,           // Indicates the number of nSTS used by the
                                                      // secondary TX Path in case of muMIMO frame
                                                      // . '0' in case of single user frame
`endif//RW_MUMIMO_TX_EN
          input  wire  [1:0] ppduHELTFTypeMC,         // Indicates the type of HE-LTF
          input  wire  [2:0] ppduNumHeLtfMC,          // Indicates the number of HE-LTF symbols
                                                      // minus 1
          input  wire        ppduDCMMC,               // indicate that dual carrier modulation is 
                                                      // used for the HE-Data field
          input  wire        ppduTriggerMethodMC,     // Indicates the method used to trigger this
                                                      // HE TB PPDU transmission
          input  wire        ppduDopplerMC,           // Indicates whether the doppler effect
                                                      // should be considered for the PPDU or not
          input  wire        ppduMmaMC,               // Indicates the midamble periodicity
                                                      // 1'b0: 10
                                                      // 1'b1: 20
          input  wire  [2:0] ppduRuTypeMC,            // Indicates the type of RU
          input  wire  [2:0] ppduPacketExtensionMC,   // Indicates the packet extension parameters.
          input  wire  [4:0] ppduHeTbLengthMC,        // Indicates the length of the HE TB PPDU 
                                                      // response to an TRS trigger and is set to
                                                      // the number of OFDM symbols in the Data
                                                      // field of the HE TB PPDU minus 1.
          input  wire        woPreambleMC,            // Indicates that total duration should not include the preamble duration 
          input  wire        woSignExtMC,             // Indicates that total duration should not include the signal Extension if any 
          input  wire        startComputationMC_p,    // Start Time on Air computation from MAC Controller

          output  reg [15:0] timeOnAirMC,             // Time On Air result to MAC Controller
                                                      // This field gives the time taken for frame transmission in microseconds (us)
                                                      // computed from parameters given by the MAC Controller.
          output  reg        timeOnAirValidMC,        // Time On Air Valid to MAC Controller
                                                      // This field is set when the air time computation is completed.

          //$port_g CSReg interface
          input  wire        band5G,                  // Indicate the current operating band (0: 2.4GHz, 1:5GHz)
          input  wire [19:0] ppduLength,              // PPDU Length
                                                      // This field gives the length of the PPDU for computing time on air.
          input  wire  [1:0] ppduChBw,                // This field indicates the channel bandwidth
          input  wire  [3:0] ppduPreType,             // PPDU Preamble Type
                                                      // This field indicates what type of preamble is transmitted for the frame.
                                                      // The encoding is as follows:
                                                      // 4'b0000: Non-HT-Short
                                                      // 4'b0001: Non-HT-Long
                                                      // 4'b0010: HT-MF
                                                      // 4'b0011: HT-GF
                                                      // 4'b0100: VHT
                                                      // 4'b0101: HE-SU
                                                      // 4'b0110: HE-MU
                                                      // 4'b0111: HE-ER-SU
                                                      // 4'b1000: HE-TB
          input  wire  [1:0] ppduNumExtnSS,           // Start Transmission with Number of Extension Spatial Streams      
          input  wire  [1:0] ppduSTBC,                // Start Transmission with Space Time Block Coding          
          input  wire  [1:0] ppduGIType,              // PPDU Guard Interval Type
                                                      // Indicates that the frame should be transmitted with Short GI.
          input  wire  [6:0] ppduMCSIndex,            // PPDU LegRate or MCS Index
                                                      // This field indicates the rate at which this PPDU is transmitted. 
          input  wire  [1:0] ppduHELTFType,           // Indicates the type of HE-LTF
          input  wire  [2:0] ppduNumHeLtf,            // Indicates the number of HE-LTF symbols
                                                      // minus 1
          input  wire        ppduDCM,                 // indicate that dual carrier modulation is 
                                                      // used for the HE-Data field
          input  wire        ppduDoppler,             // Indicates whether the doppler effect
                                                      // should be considered for the PPDU or not
          input  wire        ppduMma,                 // Indicates the midamble periodicity
                                                      // 1'b0: 10
                                                      // 1'b1: 20
          input  wire  [2:0] ppduRuType,              // Indicates the type of RU
          input  wire  [2:0] ppduPacketExtension,     // Indicates the packet extension parameters.

          input  wire        computeDuration,         // Request to perform a Time on Air computation
          output wire        computeDurationIn,       // Clear the computeDuration
          output wire        computeDurationInValid,  // Clear the computeDuration valid

          output wire [15:0] timeOnAir,               // Time On Air
          output  reg        timeOnAirValid,          // Time on Air is valid
                                                      // This field gives the time taken for frame transmission in microseconds (us)
                                                      // computed from parameters programmed in timeOnAirParamReg register.
          //$port_g Debug port
          output wire [15:0] debugPortTxTimeCalculator// Debug port
  );


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////

localparam DCCK_SHORT_PREAMBLE  = 16'd96; // Duration of DSSS/CCK Short preamble
localparam DCCK_LONG_PREAMBLE   = 16'd192;// Duration of DSSS/CCK Long preamble
localparam NON_HT_OFDM_PREAMBLE = 16'd20; // Duration of NON-HT OFDM preamble
localparam MM_PREAMBLE          = 16'd32; // Duration of Mixed-Mode preamble
localparam GF_PREAMBLE          = 16'd24; // Duration of GreenField preamble
localparam VHT_PREAMBLE         = 16'd36; // Duration of VHT preamble
localparam HE_SU_PREAMBLE       = 16'd36; // Duration of HE-SU preamble without HE-LTF
localparam HE_ER_SU_PREAMBLE    = 16'd44; // Duration of HE-ER-SU preamble without HE-LTF
localparam HE_TB_PREAMBLE       = 16'd40; // Duration of HE-TB-SU preamble without HE-LTF

// dividerControlFSM FSM states definition
//$fsm_sd dividerControlFSM
localparam
   IDLE            =  3'd0,
   COMPUTE_NSYMB   =  3'd1,
   COMPUTE_SHORTGI =  3'd2,
   COMPUTE_NMA     =  3'd3,
   COMPUTE_DOT8    =  3'd4;


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////

// dividerControlFSM FSM signals definition
reg   [2:0] dividerControlFSMCs;  // dividerControlFSM FSM Current State
reg   [2:0] dividerControlFSMNs;  // dividerControlFSM FSM Next State

reg   [6:0] mcsIndex_ff1;              // Indicates the rate at which this PPDU is transmitted.
reg         swProcessing;              // Indicates that a SW request is being processed.

reg         startDiv_p;                // Pulse to launch the divider
reg         startDivDly_p;             // Pulse to launch the divider

reg  [14:0] nBitPSymb_comb;            // Combinational number of bits per symbole
reg  [10:0] nBitPSymbShort1ss;         // Number of bits in one segment of the last OFDM symbol for
                                       // Nss=1
reg  [11:0] nBitPSymbShort;            // Number of bits in one segment of the last OFDM symbol
reg   [3:0] nSS_comb;                  // Combination number of spacial stream
wire  [3:0] nSTSTotal;                 // Total number of space time stream
reg         busy;                      // Busy indication
reg         busyDly;                   // Busy indication
wire        quickDivEnable;            // Indicates that the duration computation
                                       // has to be speed up, the divider is not used
reg         quickDivStart_p;           // Indicates the start of the computation when the divider is not used
reg         quickDivDone_p;            // Indicates the end of the computation when the divider is not used
reg         quickDivDoneDly_p;         // Indicates the end of the computation when the divider is not used (Delay of 1cc)
reg         startComputationMC;        // Indicates that a computation is pending for the MC 
reg         startComputation;          // Indicates that a computation is pending for the Register 
reg  [15:0] preambleDuration;          // Computed preamble duration
reg  [15:0] data14Duration;            // Computed duration for DATA of 14 bytes
reg  [15:0] data20Duration;            // Computed duration for DATA of 20 bytes
reg  [14:0] nBitPSymb;                 // Stored number of bits per symbol
reg  [14:0] divisor;                   // Divisor of the divider
reg  [23:0] dividend;                  // Dividend of the divider
wire        roundUp;                   // Round Up the result of the division
reg         computeDurationInValidInt; // computeDurationInValid signal coming from the macCoreClk domain

wire  [3:0] preambleType;              // Indicates what type of preamble is transmitted for the frame.
wire  [1:0] giType;                    // Indicates what type of guard interval is used for the frame
wire  [1:0] chBw;                      // Channel Bandwidth
wire  [6:0] mcsIndex;                  // Indicates the rate at which this PPDU is transmitted.
wire [19:0] length;                    // Indicates the length of the PPDU for computing time on air.
wire  [1:0] numExtnSS;                 // Number of extended spacial stream
wire  [3:0] priNSTS;                   // Number of nSTS used by the primary user
`ifdef  RW_MUMIMO_TX_EN
wire  [2:0] secNSTS;                   // Number of nSTS used by the secondary users
                                       // '0' for single user frame.
`endif//RW_MUMIMO_TX_EN
wire  [1:0] stbc;                      // Indicates that STBC has been selected.

wire [23:0] nBit;                      // Number of bits in ppdu
reg  [23:0] nES;                       // Number of bits in ppdu
wire [23:0] computednSymb;             // Number of Symb computed by the divider
reg  [13:0] computednSymbCapt;         // Number of Symb computed by the divider
reg  [15:0] dataDuration;              // Computed Data Duration
wire [15:0] totalDuration;             // Total duration (preamble + ppdu + signalExt)
wire [15:0] totalDurationWithoutPreamble;// Total duration without preamble (ppdu + signalExt)
wire [15:0] totalDurationWithPreamble; // Total duration with preamble (preamble + ppdu + signalExt)
wire        divDone_p;                 // Indicates the end of the computation when the divider is used

wire  [3:0] ndltf;                     // Number of DLTFs required for data space-time streams          
wire  [3:0] neltf;                     // Number of ELTFs required for extension spatial streams
wire  [3:0] vhtltf;                    // Number of VHT-LTFs required for space-time streams

wire  [3:0] nltf;                      // Number of LTFs required for extension spatial streams
wire        isDCCK;                    // Indicates that the modulation is DSSS/CCK
wire  [3:0] initCnt;                   // Initial value of the divider step.
wire        startComputation_p;        // Pulse generated in macCoreClk domain when a rising edge of computeDuration is detected

wire        isLegacy;
wire        isHT;
wire        isVHT;
wire        isHE;
wire        isHETB;
wire        is40MHz;
wire        is80MHz;
wire        is160MHz;

wire [13:0] remainderOut;              // Remainder from divider (used in VHT for disambiguity 
                                       // and in HE for Nexcess)
reg   [3:0] remainderOutExtraSGIus;    // Convertion of remainderOut into us
wire        endOfProcessing;
  
  
reg  [19:0] ppduLengthResync;        // PPDU Length
                                     // This field gives the length of the PPDU for computing time on air.
reg   [1:0] ppduChBwResync;          // This field indicates the channel bandwidth
reg   [3:0] ppduPreTypeResync;       // PPDU Preamble Type
                                     // This field indicates what type of preamble is transmitted for the frame.
                                     // The encoding is as follows:
                                     // 4'b0000: Non-HT-Short
                                     // 4'b0001: Non-HT-Long
                                     // 4'b0010: HT-MF
                                     // 4'b0011: HT-GF
                                     // 4'b0100: VHT
                                     // 4'b0101: HE-SU
                                     // 4'b0110: HE-MU
                                     // 4'b0111: HE-ER-SU
                                     // 4'b1000: HE-TB
reg   [1:0] ppduNumExtnSSResync;     // Start Transmission with Number of Extension Spatial Streams      
reg   [1:0] ppduSTBCResync;          // Start Transmission with Space Time Block Coding          
reg   [1:0] ppduGITypeResync;        // PPDU Guard Interval Type
                                     // Indicates that the frame should be transmitted with Short GI.
reg   [6:0] ppduMCSIndexResync;      // PPDU LegRate or MCS Index
                                     // This field indicates the rate at which this PPDU is transmitted. 
reg   [1:0] ppduHELTFTypeResync;     // Indicates the type of HE-LTF
reg   [2:0] ppduNumHeLtfResync;      // Indicates the number of HE-LTF symbols
                                     // minus 1
reg         ppduDCMResync;           // indicate that dual carrier modulation is 
                                     // used for the HE-Data field
reg         ppduDopplerResync;       // Indicates whether the doppler effect
                                     // should be considered for the PPDU or not
reg         ppduMmaResync;           // Indicates the midamble periodicity
                                     // 1'b0: 10
                                     // 1'b1: 20
reg   [2:0] ppduRuTypeResync;        // Indicates the type of RU
reg   [2:0] ppduPacketExtensionResync;// Indicates the packet extension parameters.

reg [15:0] timeOnAirReg;             // Time On Air computed based on register parameters (on macCoreclk)
reg [15:0] timeOnAirResync;          // timeOnAirReg resynchronized on macPIClk

wire [15:0] signalExtDur;            // Duration of signalExtension if any.
  
wire  [1:0] heLtfType;               // Indicates the type of HE-LTF
wire  [2:0] numHeLtf;                // Indicates the number of HE-LTF symbols in preamble minus 1
wire        dcm;                     // Indicate that dual carrier modulation is used for the
                                     // HE-Data field
wire        doppler;                 // Indicates whether the doppler effect should be considered
                                     // for the PPDU or not.
wire        mma;                     // Indicates the midamble periodicity
wire  [2:0] ruType;                  // Indicates the type of RU.
wire  [2:0] packetExtension;         // Indicates the packet extension parameters.
wire        triggerMethod;           // Indicates the method used to trigger this HE TB PPDU
                                     // transmission
wire        isHeTbTrs;               // Indicates a HE-TB PPDU frame with TRS trigger method
reg   [3:0] nHeLtf;                  // Indicates the number of HE-LTF symbols in preamble
reg   [5:0] nHeLtfField;             // Number of HE-LTF field (preamble & midamble)
reg   [8:0] nHeLtfField1x;           // nHeLtfField * 1 
reg   [8:0] nHeLtfField2x;           // nHeLtfField * 2
reg   [8:0] nHeLtfField4x;           // nHeLtfField * 4
reg   [8:0] nHeLtfField8x;           // nHeLtfField * 8
reg   [8:0] nHeLtfSymb;              // Number of HE-LTF symbol (preamble & midamble)
reg  [12:0] nHeLtfSymbTheLtfDot8;    // Duration of all HE-LTF symbols without GI (in 0.8us unit).
reg  [10:0] nHeLtfSymbTgiDot8;       // Duration of all HE-LTF GI (in 0.8us unit).
wire [12:0] preMidHeLtfDot8;         // Duration of all HE-LTF symbols with GI (in 0.8us unit).
reg  [12:0] nSymTDftHeDot8;          // Duration of all data symbol without GI (in 0.8us unit).
reg  [12:0] nSymTgiDot8;             // Duration of all data GI (in 0.8us unit).
wire [12:0] nSymTSymDot8;            // Duration of all data symbol with GI (in 0.8us unit).
wire [12:0] heLtfAndDataSymbDot8;    // Duration of all HE-LTF and all data symbol (in 0.8us unit).
wire        fastNumHeLtf;            // Number of HE LTF minus one when fast divider is used.
reg   [3:0] fastHeNSym;              // Number of data symbol when fast divider is used.
reg   [7:0] fastHeLtfAndData;        // Dutation of all HE-LTF and all data symbol (in 1us unit), 
                                     // when fast divider is used.
reg   [1:0] fastHeAInit;             // preFEC coding factor when fast divider is used.
reg   [1:0] slowHeAInit;             // preFEC coding factor when fast divider is not used.
wire  [1:0] aInit;                   // preFEC coding factor
reg   [4:0] packetExtensionDur;      // Duration of the packet extension.
wire [13:0] nBitPSymbShort3x;        // mSTBC*nBitPSymbShort*3
wire [13:0] nBitPSymbShort2x;        // mSTBC*nBitPSymbShort*2
wire [13:0] nBitPSymbShort1x;        // mSTBC*nBitPSymbShort*1
wire        woPreamble;              // Indicates that total duration should not include the
                                     // preamble duration 
wire  [5:0] heTbLength;              // Indicates the length of the HE TB PPDU response to an TRS
                                     // trigger and is set to the number of OFDM symbols in the Data
                                     // field of the HE TB PPDU
wire  [8:0] heNSymCapt;              // Indicates the number of OFDM symbols in the HE Data field
`ifdef RW_SIMU_ON
// String definition to display dividerControlFSMCs current state
reg [15*8-1:0] dividerControlFSMCs_str;
`endif // RW_SIMU_ON

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////////
// Resynchronization with macPIClk for register interface
//////////////////////////////////////////////////////////////////////////////

// Instanciation of risingEdgeDetector
// Name of the instance : U_computeDurationRisingEdgeDetector
// Name of the file containing this module : risingEdgeDetector.v
risingEdgeDetector U_computeDurationRisingEdgeDetector (
    .dstclk       (macCoreClk),
    .dstresetn    (macCoreClkHardRst_n),
    .srcdata      (computeDuration),
    .dstdata      (startComputation_p)
    );

// Instanciation of risingEdgeDetector
// Name of the instance : U_computeDurationInValidRisingEdgeDetector
// Name of the file containing this module : risingEdgeDetector.v
risingEdgeDetector U_computeDurationInValidRisingEdgeDetector (
    .dstclk       (macPIClk),
    .dstresetn    (macPIClkHardRst_n),
    .srcdata      (computeDurationInValidInt),
    .dstdata      (computeDurationInValid)
    );


// Resynchronization of the parameter coming from register (macPIClk)
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    ppduLengthResync          <= 20'b0;
    ppduChBwResync            <= 2'b0;
    ppduPreTypeResync         <= 4'b0;
    ppduNumExtnSSResync       <= 2'b0;
    ppduSTBCResync            <= 2'b0;
    ppduGITypeResync          <= 2'b0;
    ppduMCSIndexResync        <= 7'b0;
    ppduHELTFTypeResync       <= 2'b0;
    ppduNumHeLtfResync        <= 3'b0;
    ppduDCMResync             <= 1'b0;
    ppduDopplerResync         <= 1'b0;
    ppduMmaResync             <= 1'b0;
    ppduRuTypeResync          <= 3'b0;
    ppduPacketExtensionResync <= 3'b0;
  end
  else if (startComputation_p)
  begin
    ppduLengthResync          <= ppduLength;
    ppduChBwResync            <= ppduChBw;
    ppduPreTypeResync         <= ppduPreType;
    ppduNumExtnSSResync       <= ppduNumExtnSS;
    ppduSTBCResync            <= ppduSTBC;
    ppduGITypeResync          <= ppduGIType;
    ppduMCSIndexResync        <= ppduMCSIndex;
    ppduHELTFTypeResync       <= ppduHELTFType;
    ppduNumHeLtfResync        <= ppduNumHeLtf;
    ppduDCMResync             <= ppduDCM;
    ppduDopplerResync         <= ppduDoppler;
    ppduMmaResync             <= ppduMma;
    ppduRuTypeResync          <= ppduRuType;
    ppduPacketExtensionResync <= ppduPacketExtension;
  end
end


// Generate the timeOnAirValid indication for register interface
always @ (posedge macPIClk or negedge macPIClkHardRst_n) 
begin
  if (macPIClkHardRst_n == 1'b0)  // Asynchronous Reset
    timeOnAirValid     <= 1'b0;
  else
  begin
    if (computeDurationInValid)
      timeOnAirValid   <= 1'b1;
    else if (computeDuration)
      timeOnAirValid   <= 1'b0;
  end
end

// Resynchronization of timeOnAirReg
always @ (posedge macPIClk or negedge macPIClkHardRst_n) 
begin
  if (macPIClkHardRst_n == 1'b0)  // Asynchronous Reset
    timeOnAirResync     <= 16'b0;
  else if (computeDurationInValid)
    timeOnAirResync   <= timeOnAirReg;
end

// Assignment of timeOnAir. This is to keep the Resync suffix used for Clock Domain crossing check
// and keep the same signal name (eg timeOnAir) through the hierarchy
assign timeOnAir = timeOnAirResync;

//////////////////////////////////////////////////////////////////////////////
// Control from MAC Controller interface
//////////////////////////////////////////////////////////////////////////////
assign endOfProcessing = (divDone_p && ((giType==2'b00 && (dividerControlFSMCs == COMPUTE_NSYMB) && ~isHE) ||
                                        (dividerControlFSMCs == COMPUTE_SHORTGI) |
                                        (dividerControlFSMCs == COMPUTE_DOT8)));

// Generate of the startComputationMC signal which indicates
// a pending request from the MAC Controller
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    startComputationMC <= 1'b0;
  else
    if (startComputationMC_p)
      startComputationMC <= 1'b1;
    else if (!swProcessing && (quickDivDoneDly_p || endOfProcessing))
      startComputationMC <= 1'b0;
end 
      

// Generate of the startComputation signal which indicates
// a pending request from the Registers
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    startComputation <= 1'b0;
  else
    if (startComputation_p)
      startComputation <= 1'b1;
    else if (swProcessing && (quickDivDoneDly_p || endOfProcessing))
      startComputation <= 1'b0;
end 


// Capture the computed timeOnAir duration requested by the MAC Controller.
// This value is stable when timeOnAirValidMC is set.
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    timeOnAirMC <= 16'b0;
    timeOnAirValidMC <= 1'b0;
  end
  else
  begin
    if (startComputationMC_p)
      timeOnAirValidMC <= 1'b0;
    else
    begin
      if (!swProcessing && (quickDivDoneDly_p || endOfProcessing))
      begin
        timeOnAirMC      <= totalDuration;
        timeOnAirValidMC <= 1'b1;
      end  
    end  
  end
end

      
// Capture the computed timeOnAirReg duration requested by the registers.
// This value is stable until the next request from the registers.
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    timeOnAirReg              <= 16'b0;
    computeDurationInValidInt <= 1'b0;
  end
  else
  begin
    computeDurationInValidInt <= 1'b0;
    if (swProcessing && (quickDivDoneDly_p || endOfProcessing))
    begin
      timeOnAirReg              <= totalDuration;
      computeDurationInValidInt <= 1'b1;
    end  
  end
end


// Logic to clear the startComputation bit in the CSReg.
assign computeDurationIn = 1'b0;


// Indicate if the on-going computing has been requested by the MAC Controller
// (swProcessing = 0) or the register (swProcessing = 1)
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    swProcessing <= 1'b0;
  else
  begin
    if (!busy)
    begin
      if (startComputationMC || startComputationMC_p)
        swProcessing <= 1'b0;
      else if (startComputation || startComputation_p)
        swProcessing <= 1'b1;
    end    
  end    
end


// dividerControlFSM FSM Current State Logic 
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    dividerControlFSMCs <= IDLE; 
  else
    dividerControlFSMCs <= dividerControlFSMNs; 
end


// dividerControlFSM FSM Next State Logic.
always @* 
begin
  case(dividerControlFSMCs)

    IDLE:
      //$fsm_s In IDLE state, the state machine waits for a trigger.
      if (startDiv_p & isHeTbTrs & doppler) 
        //$fsm_t When startDiv_p is received and HE-TB PPDU frame with TRS 
        //trigger method and with doppler, the state machine goes to
        //COMPUTE_NMA state.
        dividerControlFSMNs = COMPUTE_NMA; 
      else if (startDiv_p & isHeTbTrs & ~doppler) 
        //$fsm_t When startDiv_p is received and HE-TB PPDU frame with TRS 
        //trigger method and without doppler, the state machine goes to
        //COMPUTE_DOT8 state.
        dividerControlFSMNs = COMPUTE_DOT8; 
      else if (startDiv_p)
        //$fsm_t When startDiv_p is received, the state machine goes to COMPUTE_NSYMB state.
        dividerControlFSMNs = COMPUTE_NSYMB; 
      else
        //$fsm_t While startDiv_p is not received, the state machine stays in IDLE state.
        dividerControlFSMNs = IDLE;

    COMPUTE_NSYMB:
      //$fsm_s In COMPUTE_NSYMB state, the state machine controls the divider to compute the number of Symbol.
      if (divDone_p & isHE & ~doppler)
        //$fsm_t When divDone_p is received and HE frame without midamble, the state machine goes
        //to COMPUTE_DOT8 state.
        dividerControlFSMNs = COMPUTE_DOT8; 
      else if (divDone_p & isHE & doppler)
        //$fsm_t When divDone_p is received and HE frame with midamble, the state machine goes to
        //COMPUTE_NMA state.
        dividerControlFSMNs = COMPUTE_NMA; 
      else if (divDone_p &  giType[0])
        //$fsm_t When divDone_p is received, not HE frame and shortGI is set, the state machine
        //goes to COMPUTE_SHORTGI state.
        dividerControlFSMNs = COMPUTE_SHORTGI; 
      else if (divDone_p & ~giType[0]) 
        //$fsm_t When divDone_p is received, not HE frame and shortGI is not set, the state machine
        //goes back to IDLE state.
        dividerControlFSMNs = IDLE; 
      else
        //$fsm_t While divDone_p is not received, the state machine stays in COMPUTE_NSYMB state.
        dividerControlFSMNs = COMPUTE_NSYMB;

    COMPUTE_SHORTGI:
      //$fsm_s In COMPUTE_SHORTGI state, the state machine controls the divider to divide the number of Symbol by 10.
      if (divDone_p) 
        //$fsm_t When divDone_p is received, the state machine goes back to IDLE state.
        dividerControlFSMNs = IDLE; 
      else
        //$fsm_t While divDone_p is not received, the state machine stays in COMPUTE_SHORTGI state.
        dividerControlFSMNs = COMPUTE_SHORTGI;

    COMPUTE_NMA:
      //$fsm_s In COMPUTE_NMA state, the state machine controls the divider to compute the number of HE midamble.
      if (divDone_p)
        //$fsm_t When divDone_p is received, the state machine goes to COMPUTE_DOT8 state.
        dividerControlFSMNs = COMPUTE_DOT8; 
      else
        //$fsm_t While divDone_p is not received, the state machine stays in COMPUTE_NMA state.
        dividerControlFSMNs = COMPUTE_NMA;

    COMPUTE_DOT8:
      //$fsm_s In COMPUTE_DOT8 state, the state machine controls the divider to convert time in 0.8
      //unit to microsecond (*8/10).
      if (divDone_p)
        //$fsm_t When divDone_p is received, the state machine goes to IDLE state.
        dividerControlFSMNs = IDLE; 
      else
        //$fsm_t While divDone_p is not received, the state machine stays in COMPUTE_DOT8 state.
        dividerControlFSMNs = COMPUTE_DOT8;

    // Disable coverage on the default state because it cannot be reached.
    // pragma coverage block = off 
    default:   
        dividerControlFSMNs = IDLE; 
    // pragma coverage block = on 
  endcase
end


// Generation of the quickDivEnable signal. Indicates that the duration computation
// has to be speed up, the divider is not used.
// set if legacy frame with 1Mbps or 2Mbps datarate or frame with length 0, 14 or 20 bytes
// unset if HE-TB-PPDU frame with TRS trigger method, in this case the length signal 
// indicates the numbers of OFDM Symbol in the data field minus 1.
assign quickDivEnable = ((mcsIndex<=7'd1 && preambleType[3:1]==3'b0) ||
                        (length==20'd0)  ||
                        (length==20'd14) ||
                        (length==20'd20)) && !isHETB;//!isHeTbTrs;


// Generation of the startDiv_p pulse which trigs the divider.
// The divider is used for all the ppdu length value except 14 and 20.
// For those values, the computing has been speed up.
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    startDiv_p    <= 1'b0;
    startDivDly_p <= 1'b0;
  end
  else
  begin
    startDivDly_p <= startDiv_p;
    if (!busyDly)
    begin
      if (!swProcessing && startComputationMC && !quickDivEnable)
        startDiv_p <= 1'b1;
      else if (swProcessing && startComputation && !quickDivEnable) 
        startDiv_p <= 1'b1;
      else
        startDiv_p <= 1'b0;
    end
    else if (divDone_p && (dividerControlFSMCs == COMPUTE_NSYMB) && giType==2'b01)
      startDiv_p <= 1'b1;
    else if (divDone_p && (dividerControlFSMCs == COMPUTE_NSYMB) && isHE)
      startDiv_p <= 1'b1;
    else if (divDone_p && (dividerControlFSMCs == COMPUTE_NMA))
      startDiv_p <= 1'b1;
    else
      startDiv_p <= 1'b0;
  end
end


// Generate the quickDivDone_p pulse. This pulse indicates the completion of a 
// duration computing which did not used the divider.
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    quickDivStart_p   <= 1'b0;
    quickDivDone_p    <= 1'b0;
    quickDivDoneDly_p <= 1'b0;
  end
  else
  begin
    quickDivDoneDly_p <= quickDivDone_p;
    quickDivDone_p    <= quickDivStart_p;
    if (!busyDly)
    begin
      if (!swProcessing && startComputationMC && quickDivEnable) 
        quickDivStart_p <= 1'b1;
      else if (swProcessing && startComputation && quickDivEnable) 
        quickDivStart_p <= 1'b1;
      else    
        quickDivStart_p <= 1'b0;
    end
    else    
      quickDivStart_p <= 1'b0;
  end    
end

// Generation of the busy signal which indicates an on-going processing
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
  begin
    busy    <= 1'b0;
    busyDly <= 1'b0;
  end
  else
  begin
    busyDly <= busy;
    if (endOfProcessing || quickDivDoneDly_p || computeDurationInValidInt)
      busy <= 1'b0;
    else if (!busy && (startComputation || startComputationMC || startComputation_p || startComputationMC_p))
      busy <= 1'b1;
  end    
end

// In case of OFDM frame on 2.4GHz band,
assign signalExtDur = (!band5G && !isDCCK) ? 16'd6 : 16'd0;


// totalDurationWithoutPreamble = dataDuration + signalExtension.
assign totalDurationWithoutPreamble = (!swProcessing && woSignExtMC) ? dataDuration[15:0] : dataDuration[15:0] + signalExtDur;

// totalDurationWithPreamble = preambleDuration + dataDuration + signalExtension.
assign totalDurationWithPreamble = preambleDuration + totalDurationWithoutPreamble;

// totalDuration = preambleDuration + dataDuration + signalExtension.
assign totalDuration = (woPreamble) ? totalDurationWithoutPreamble : totalDurationWithPreamble;


// Assign the different parameter depending of the source of the request 
//(MAC Controller or Registers)
assign mcsIndex     = (swProcessing) ? ppduMCSIndexResync   : ppduMCSIndexMC;
assign length       = (swProcessing) ? ppduLengthResync     : ppduLengthMC;
assign preambleType = (swProcessing) ? ppduPreTypeResync    : ppduPreTypeMC;

assign isLegacy     = (preambleType[3:1] == 3'b000);
assign isHT         = (preambleType[3:1] == 3'b001);
assign isVHT        = (preambleType      == 4'b0100);
assign isHE         = (preambleType      >= 4'b0101);
assign isHETB       = (preambleType      == 4'b1000);

assign stbc         = (isLegacy) ? 2'b0 : (swProcessing) ? ppduSTBCResync       : ppduSTBCMC;
assign giType       = (isLegacy) ? 2'b0 : (swProcessing) ? ppduGITypeResync     : ppduGITypeMC;
assign chBw         = (isLegacy) ? 2'b0 : (swProcessing) ? ppduChBwResync       : ppduChBwMC;

assign numExtnSS    = (isLegacy || isVHT) ? 2'b0 : (swProcessing) ? ppduNumExtnSSResync  : ppduNumExtnSSMC;

`ifdef  RW_MUMIMO_TX_EN
assign secNSTS      = (swProcessing) ? 3'b0 : ppduSecNSTSMC;
`endif//RW_MUMIMO_TX_EN
assign woPreamble   = (swProcessing) ? 1'b0 : woPreambleMC;

assign is40MHz      = (chBw == 2'd1);
assign is80MHz      = (chBw == 2'd2);
assign is160MHz     = (chBw == 2'd3);

assign heLtfType       = (swProcessing) ? ppduHELTFTypeResync       : ppduHELTFTypeMC;
assign numHeLtf        = (swProcessing) ? ppduNumHeLtfResync        : ppduNumHeLtfMC;
assign dcm             = (~isHE)        ? 1'b0 :                    
                         (swProcessing) ? ppduDCMResync             : ppduDCMMC;
assign triggerMethod   = (swProcessing) ? 1'b0                      : ppduTriggerMethodMC;
assign doppler         = (swProcessing) ? ppduDopplerResync         : ppduDopplerMC;
assign mma             = (swProcessing) ? ppduMmaResync             : ppduMmaMC;
assign ruType          = (swProcessing) ? ppduRuTypeResync          : ppduRuTypeMC;
assign packetExtension = (swProcessing) ? ppduPacketExtensionResync : ppduPacketExtensionMC;

// Indicates if the modulation is DSSS/CCK or not.
assign isDCCK = ((mcsIndex<7'd4) && (preambleType[3:1] == 3'b000)) ? 1'b1 : 1'b0;

// Indicates a HE-TB PPDU frame with TRS trigger method
assign isHeTbTrs = isHETB & triggerMethod;


// For DSSS/CCK modulation (MCSIndex < 4 and PreType < 2)
// 
// duration = preambleduration + length*8/dataRate
// If preambleType = 0 (Short preamble), preambleduration = 96
// If preambleType = 1 (Long preamble),  preambleduration = 192
//
// If MCSIndex = 0, datarate = 1
// If MCSIndex = 1, datarate = 2
// If MCSIndex = 2, datarate = 5.5
// If MCSIndex = 3, datarate = 11
//
//
// For OFDM modulation (Legacy and HT and VHT)
// 
// duration = preambleduration + dataduration
//
// preambleduration = If LEGACY then 20us
//                    If HT MixMode then 32us + NLTF*4
//                    If HT GreendField then 24us + (NLTF-1)*4
//                    If VHT then 36us + NLTF*4
//
// NLTF             = depends on HT or VHT NSS, and extNSS for HT only
//
// dataduration     = ceil( NumberOfSymbols * SymbolDuration)
//
// SymbolDuration   = is 4 us for Normal Guard Interval and 3.6 for Short Guard Interval
//
// NumberOfSymbols  = If no STBC then  ceil((length*8+22)/NDBPS) else 2*ceil((length*8+22)/NDBPS)
//
// NDBPS            = Number of data bits per symbol (it is defined according to the MCSINDEX and the operating mode LEGACY or HT or VHT)
// Note: The 22 bits in the formula come from the 16 bits of service field and 6 tail bits.
//
//
//
// For HE OFDM modulation
//
// duration = preambleduration + dataduration + midambleduration + tpe + signalextension
//
//
// preamble duration = if HE-SU    then 36us + NHE-LTF * THE-LTF-SYM 
//                     if HE-ER-SU then 44us + NHE-LTF * THE-LTF-SYM
//                     if HE-TB    then 40us + NHE-LTF * THE-LTF-SYM
//
// NHE-LTF can be 1,2,4,6, or 8, provided by the numHeLtf input for HE-TB format
//                               Same as vhtltf for HE-SU & HE-ER-SU format
//
// THE-LTF-SYM      = THE-LTF + TGI,HE-LTF
// THE-LTF          = THE-LTF-1X, THE-LTF-2X or THE-LTF-4X depending upon the LTF duration used
//                    (HE_LTF_TYPE)
// THE-LTF-1X       =  3.2 µs
// THE-LTF-2X       =  6.4 µs
// THE-LTF-4X       = 12.8 µs
//
// TGI,HE-LTF       = TGI1,Data, TGI2,Data or TGI4,Data depending on the GI used for data
// TGI,Data         = TGI1,Data, TGI2,Data or TGI4,Data depending on the GI used for data
// TGI1,Data        =  0.8 µs
// TGI2,Data        =  1.6 µs
// TGI4,Data        =  3.2 µs
//
//
// dataduration     = Nsym * Tsym
//
// Nsym             = if HE-TB with TRS trigger then indicated by the ppduHeTbLengthMC input
//                    else ceil( (8*APEP_LENGTH + NTail + Nservice) / NDBPS )
//
// TSYM             = TSYM1, TSYM2, or TSYM4 depending on the GI used
// TSYM1            = 13.6 µs = TDFT,HE + TGI1,Data
// TSYM2            = 14.4 µs = TDFT,HE + TGI2,Data
// TSYM4            = 16   µs = TDFT,HE + TGI4,Data
// TDFT,HE          = 12.8 µs
//
// NDBPS            = Number of data bits per symbol
//                    (it is defined according to the MCSINDEX, RUTYPE and DCM)
//
//
// midambleduration = NMA * NHE-LTFT * HE-LTF-SYM
//
// NMA              = max( 0 , ceil( (Nsym-1)/Mma) )-1 )
// MMA              = the midamble periodicity in number of OFDM symbols, whose value is either
//                    10 or 20
//
//
// tpe              = if HE-TB with TRS trigger then default_pe_duration
//                    else is function of preFec padding factor and Nominal_packet_padding
// 
// aInit            = preFec padding factor
//                    if Nexcess=0 then 4
//                    else min( ceil( Nexcess / Ndbpsshort ), 4)
// Ndbpsshort       = Ncbpsshort * R
// R                = The coding rate
// Ncbpsshort       = Nsdshort * Nss * Nbpscs
//
// Table 28-28 Nsdshort values
// -------------------------------------
// | RU Size    |      NSD,short       |
// -------------------------------------
// |            |   DCM = 0 | DCM = 1  |
// -------------------------------------
// |    26-tone |         6 |      2   |
// |    52-tone |        12 |      6   |
// |   106-tone |        24 |     12   |
// |   242-tone |        60 |     30   |
// |   484-tone |       120 |     60   |
// |   996-tone |       240 |    120   |
// | 2*996-tone |       492 |    246   |
// -------------------------------------
// 
// Table 28-41 Tpe,Nominal:
// -------------------------------
// |       | NOMINAL_PACKET_PADDING
// -------------------------------
// | aInit | 0 µs | 8 µs | 16 µs |
// -------------------------------
// |     1 | 0 µs | 0 µs |  4 µs |
// |     2 | 0 µs | 0 µs |  8 µs |
// |     3 | 0 µs | 4 µs | 12 µs |
// |     4 | 0 µs | 8 µs | 16 µs |
// -------------------------------
//
// Ntail            =  6 for BCC encoder, 0 for LDPC encoder
// Nservice         = 16

// Instanciation of divider
// Name of the instance : U_divider
// Name of the file containing this module : divider.v
divider #(.NBIT_WIDTH(24),.NBITPSYMB_WIDTH(15)) U_divider (
    .macCoreClk             (macCoreClk),
    .macCoreClkHardRst_n    (macCoreClkHardRst_n),
    .startDiv_p             (startDivDly_p),
    .roundUp                (roundUp),
    .nBit                   (dividend),
    .nBitPSymb              (divisor),
    .initCnt                (initCnt),
    .nSymb                  (computednSymb),
    .remainderOut           (remainderOut),
    .divDone_p              (divDone_p)
    );

always @*
begin
   case (dividerControlFSMNs)
   COMPUTE_SHORTGI: divisor = 15'd10;
   COMPUTE_NMA:     divisor = (mma) ? 15'd20 : 15'd10;
   COMPUTE_DOT8:    divisor = 15'd10;
   default:         divisor = nBitPSymb;
   endcase
end

always @*
begin
   case (dividerControlFSMNs)
   COMPUTE_SHORTGI: dividend = (stbc!=2'd0) ? computednSymb<<24'd1     : computednSymb<<24'd0;
   COMPUTE_NMA:     dividend = (isHeTbTrs)  ? {19'b0,ppduHeTbLengthMC} : {10'd0,computednSymbCapt}-24'd1;
   COMPUTE_DOT8:    dividend = {8'd0,heLtfAndDataSymbDot8,3'd0};
   default:         dividend = nBit;
   endcase
end

//assign divisor  = (dividerControlFSMNs == COMPUTE_SHORTGI) ? 15'ha : nBitPSymb;
//assign dividend = (dividerControlFSMNs == COMPUTE_SHORTGI) ? ((stbc > 2'd0) ? computednSymb<<24'd1 : computednSymb<<24'd0) : nBit;
assign roundUp  = (dividerControlFSMCs == COMPUTE_SHORTGI) ? 1'b0 : 1'b1;

// Generation of the disambiguity
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    disambiguityMC <= 1'b0;
  else if (startDiv_p)
    disambiguityMC <= 1'b0;
  else if (divDone_p && (dividerControlFSMCs == COMPUTE_SHORTGI) && (remainderOut[3:0] == 4'd9) && (preambleType == 4'b0100) && !swProcessing)
    disambiguityMC <= 1'b1;
end

// Generation of the busy signal which indicates an on-going processing
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    computednSymbCapt <= 14'b0;
  else
    if (divDone_p && (dividerControlFSMCs == COMPUTE_NSYMB))
    begin
      if (stbc > 2'd0)
        computednSymbCapt <= {computednSymb[12:0],1'b0};
      else  
        computednSymbCapt <= computednSymb[13:0];
    end    
end

//
always @(*)
begin
  if(giType==2'b01)
  begin
    case(remainderOut[3:0])
      4'd3: remainderOutExtraSGIus = 4'd1; // (3*4)/10 = 1.2
      4'd4: remainderOutExtraSGIus = 4'd1; // (4*4)/10 = 1.6
      4'd5: remainderOutExtraSGIus = 4'd2; // (5*4)/10 = 2.0
      4'd6: remainderOutExtraSGIus = 4'd2; // (6*4)/10 = 2.4
      4'd7: remainderOutExtraSGIus = 4'd2; // (7*4)/10 = 2.8
      4'd8: remainderOutExtraSGIus = 4'd3; // (8*4)/10 = 3.2
      4'd9: remainderOutExtraSGIus = 4'd3; // (9*4)/10 = 3.6
      default: remainderOutExtraSGIus = 4'd0; // Others remainder value is not modulo 10 or less than 3
    endcase
  end
  else
  begin
    remainderOutExtraSGIus = 4'd0;
  end
end


// assign dataDuration with the computed duration 
always @ *
begin
  if (length == 20'd0 & ~isHE)
    dataDuration = 16'd0;
  //else if (length == 20'd0 & isHETB & ~isHeTbTrs)
  //  dataDuration = {8'd0,fastHeLtfAndData};
  else if (length == 20'd0 & quickDivEnable & isHE)
    dataDuration = {8'd0,fastHeLtfAndData} + 16'd4/*PacketExtension*/;
  else if (length == 20'd14 & quickDivEnable)
    dataDuration = data14Duration;
  else if (length == 20'd20 & quickDivEnable)
    dataDuration = data20Duration;
  else if(isLegacy)
  begin
    if (mcsIndex_ff1 == 7'd0) // 1 Mbps
      dataDuration = {length[12:0],3'b0};
    else if (mcsIndex_ff1 == 7'd1) // 2 Mbps
      dataDuration = {length[13:0],2'b0};
    else if (isDCCK)
      dataDuration = computednSymb[15:0];
    else
      dataDuration = {computednSymb[13:0],2'b0};
  end
  else if (isHT | isVHT) // HT or VHT
  begin
    if (giType==2'b01)
      // MM-HT or VHT
      // In GF, the TXTIME formula is different than for HT-MM or VHT. 
      // However, if the packet alignment is added, both formula becomes equivalent.
      dataDuration = {computednSymbCapt[13:0] - computednSymb[13:0],2'b00};
    else
    begin
      if (stbc > 2'd0) 
        dataDuration = {computednSymb[12:0],3'b0};
      else 
        dataDuration = {computednSymb[13:0],2'b0};
    end
  end
  else // HE
  begin
     dataDuration = computednSymb[15:0] + {11'b0,packetExtensionDur};
  end
end

// Set the number of steps done by the divider.
// As the divider does 5 steps per clock cycle, this number must be a multiple of 5.
// The max value is 15.
assign initCnt = (dividerControlFSMNs == COMPUTE_NMA)  ? 4'd15 : // ToDo: Can be 10
                                                                 // (not supported by the divider)
                 (dividerControlFSMNs == COMPUTE_DOT8) ? 4'd15 :
                 (nBit < 24'd128)                      ? 4'd5  : 4'd15;
//assign initCnt = 15;


////    Table 20-11 -- Determining the number of space time streams
////      +---------------------+------------+-----------------+
////      |  Number of Spatial  | STBC field | Number of space |
////      |  Streams (from MCS) |            | time streams    |
////      +---------------------+------------+-----------------+
////      |          1          |      0     |        1        | 
////      |          1          |      1     |        2        | 
////      |          2          |      0     |        2        | 
////      |          2          |      1     |        3        | 
////      |          2          |      2     |        4        | 
////      |          3          |      0     |        3        | 
////      |          3          |      1     |        4        | 
////      |          4          |      0     |        4        | 
////      +---------------------+------------+-----------------+
////
////    Table 20-12 -- Number of DLTFs required for data space-time streams
////      +-------+-------+
////      |  NSTS | NDLTF |
////      +-------+-------+
////      |  1    |   1   | 
////      |  2    |   2   | 
////      |  3    |   4   | 
////      |  4    |   4   |
////      +-------+-------+
assign ndltf = ((nSS_comb + {2'b00,stbc}) == 4'd3) ? 4'd4 : (nSS_comb+{2'b00,stbc});

////assign ndltf = nSS_comb;
////assign ndltf = nSS_comb+stbc;
//
////    Table 20-13 -- Number of ELTFs required for extension spatial streams
////      +-------+-------+
////      |  NESS | NELTF |
////      +-------+-------+
////      |  0    |   0   | 
////      |  1    |   1   | 
////      |  2    |   2   | 
////      |  3    |   4   |
////      +-------+-------+
assign neltf = (numExtnSS == 2'd3) ? 4'd4 : {2'b00,numExtnSS};

////    Table 22-13 -- Number of VHT LTFs required for differents number of spatial streams
////      +-------------+---------+
////      |  NSTS,total | NVHTLTF |
////      +-------------+---------+
////      |      1      |   1     |
////      |      2      |   2     |
////      |      3      |   4     |
////      |      4      |   4     |
////      |      5      |   6     |
////      |      6      |   6     |
////      |      7      |   8     |
////      |      8      |   8     |
////      +-------------+---------+
// if stbc == 1 then NSTS = NSS * 2 else NSTS = NSS
assign vhtltf  = (nSTSTotal == 4'd1) ? 4'd1             : 
                 (nSTSTotal[0])      ? nSTSTotal + 4'd1 : 
                                       nSTSTotal;

// For DSSS/CCK modulation (MCSIndex < 4)
//
// If preambleType = 0 (Short preamble), preambleduration = 96
// If preambleType = 1 (Long preamble),  preambleduration = 192
//
// For non-HT OFDM modulation (4 <= MCSIndex <= 11)
// 
// preambleduration = 20
//
// For HT OFDM modulation
//
// If preambleType = 2 (HT MixedMode)
// If preambleType = 3 (HT GreenField)
// 
// duration = preambleduration + dataduration
// preambleduration :
//    If Greenfield mode :
//        preambleduration = 8+8+8+(NLTF-1)*4
//        preambleduration = 24+(NLTF-1)*4
//    If MixedMode mode :
//        preambleduration = 8+8+4+8+4+(NLTF)*4
//        preambleduration = 32+(NLTF)*4
//
// The total number of HT-LTFs is:
// NLTF = NDLTF + NELTF       (20-22)
// The number of Data HT-LTFs is denoted NDLTF. The number of extension HT-LTFs is denoted NELTF. 
// NLTF shall not exceed 5. 
// Table 20-11 shows the determination of the number of space time streams from the
// MCS and STBC fields in the HT-SIG. Table 20-12 shows the number of Data HT-LTFs as a function of the
// number of space time streams (NSTS). Table 20-13 shows the number of extension HT-LTFs as a function of
// the number of extension spatial streams (NESS).NSTS plus NESS is less than or equal to 4. In the case where
// NSTS equals 3, NESS cannot exceed one; if NESS equals one in this case then NLTF equals 5.
//
assign nltf = (preambleType == 4'b0010) ? ndltf+neltf       :
              (preambleType == 4'b0011) ? ndltf+neltf-4'd1  :
                                          vhtltf            ;

// Set the preambleDuration depending on the MCS and the preamble type.
always @*
begin
  if ((mcsIndex_ff1 < 7'd4) && (preambleType == 4'b0001)) 
    preambleDuration = DCCK_LONG_PREAMBLE;
  else if ((mcsIndex_ff1 < 7'd4) && (preambleType == 4'b0000))
    preambleDuration = DCCK_SHORT_PREAMBLE;
  else if (preambleType[3:1] == 3'b000)
    preambleDuration = NON_HT_OFDM_PREAMBLE;
  else
  begin
    if (preambleType == 4'b0010) // MixedMode
      preambleDuration = MM_PREAMBLE+{10'b0,nltf,2'b0};
    else if(preambleType == 4'b0011) // GreenField
      preambleDuration = GF_PREAMBLE+{10'b0,nltf,2'b0};
    else if(preambleType == 4'b0100)  // VHT
      preambleDuration = VHT_PREAMBLE+{10'b0,vhtltf,2'b0};
    else if(preambleType == 4'b0101)  // HE-SU
      preambleDuration = HE_SU_PREAMBLE;
    else if(preambleType == 4'b0111)  // HE-ER-SU
      preambleDuration = HE_ER_SU_PREAMBLE;
    else if(preambleType == 4'b1000)  // HE-TB
      preambleDuration = HE_TB_PREAMBLE;
    else
      preambleDuration = 0; // Unsupported Frame
  end
end    

// Set nBitPSymb_comb with the number of bits per symbole.
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    nBitPSymb_comb <= 15'd0;
  else if (preambleType[3:1] == 3'b000)
  begin
    case (mcsIndex[3:0])
             4'd0 : nBitPSymb_comb <= 15'd2;   // This value have been muplitplied by 2
             4'd1 : nBitPSymb_comb <= 15'd4;   // This value have been muplitplied by 2
             4'd2 : nBitPSymb_comb <= 15'd11;  // This value have been muplitplied by 2
             4'd3 : nBitPSymb_comb <= 15'd22;  // This value have been muplitplied by 2
             4'd4 : nBitPSymb_comb <= 15'd24;  //  6Mbps
             4'd5 : nBitPSymb_comb <= 15'd36;  //  9Mbps
             4'd6 : nBitPSymb_comb <= 15'd48;  // 12Mbps 
             4'd7 : nBitPSymb_comb <= 15'd72;  // 18Mbps 
             4'd8 : nBitPSymb_comb <= 15'd96;  // 24Mbps 
             4'd9 : nBitPSymb_comb <= 15'd144; // 36Mbps 
            4'd10 : nBitPSymb_comb <= 15'd192; // 48Mbps 
            4'd11 : nBitPSymb_comb <= 15'd216; // 54Mbps 
    // Disable coverage on the default state because it cannot be reached.
    // pragma coverage block = off 
       default : nBitPSymb_comb <= 15'd1;    
    // pragma coverage block = on
    endcase                           
  end
  else if(preambleType[3:1] == 3'b001)
  // In case of HT rate.
  begin
    case (mcsIndex[6:0])
             7'd0 : nBitPSymb_comb <= (is40MHz) ? 15'd54    : 15'd26   ;
             7'd1 : nBitPSymb_comb <= (is40MHz) ? 15'd108   : 15'd52   ;
             7'd2 : nBitPSymb_comb <= (is40MHz) ? 15'd162   : 15'd78   ;
             7'd3 : nBitPSymb_comb <= (is40MHz) ? 15'd216   : 15'd104  ;
             7'd4 : nBitPSymb_comb <= (is40MHz) ? 15'd324   : 15'd156  ;
             7'd5 : nBitPSymb_comb <= (is40MHz) ? 15'd432   : 15'd208  ;
             7'd6 : nBitPSymb_comb <= (is40MHz) ? 15'd486   : 15'd234  ;
             7'd7 : nBitPSymb_comb <= (is40MHz) ? 15'd540   : 15'd260  ;
             7'd8 : nBitPSymb_comb <= (is40MHz) ? 15'd108   : 15'd52   ;
             7'd9 : nBitPSymb_comb <= (is40MHz) ? 15'd216   : 15'd104  ;
            7'd10 : nBitPSymb_comb <= (is40MHz) ? 15'd324   : 15'd156  ;
            7'd11 : nBitPSymb_comb <= (is40MHz) ? 15'd432   : 15'd208  ;
            7'd12 : nBitPSymb_comb <= (is40MHz) ? 15'd648   : 15'd312  ;
            7'd13 : nBitPSymb_comb <= (is40MHz) ? 15'd864   : 15'd416  ;
            7'd14 : nBitPSymb_comb <= (is40MHz) ? 15'd972   : 15'd468  ;
            7'd15 : nBitPSymb_comb <= (is40MHz) ? 15'd1080  : 15'd520  ;
            7'd16 : nBitPSymb_comb <= (is40MHz) ? 15'd162   : 15'd78   ;
            7'd17 : nBitPSymb_comb <= (is40MHz) ? 15'd324   : 15'd156  ;
            7'd18 : nBitPSymb_comb <= (is40MHz) ? 15'd486   : 15'd234  ;
            7'd19 : nBitPSymb_comb <= (is40MHz) ? 15'd648   : 15'd312  ;
            7'd20 : nBitPSymb_comb <= (is40MHz) ? 15'd972   : 15'd468  ;
            7'd21 : nBitPSymb_comb <= (is40MHz) ? 15'd1296  : 15'd624  ;
            7'd22 : nBitPSymb_comb <= (is40MHz) ? 15'd1458  : 15'd702  ;
            7'd23 : nBitPSymb_comb <= (is40MHz) ? 15'd1620  : 15'd780  ;
            7'd24 : nBitPSymb_comb <= (is40MHz) ? 15'd216   : 15'd104  ;
            7'd25 : nBitPSymb_comb <= (is40MHz) ? 15'd432   : 15'd208  ;
            7'd26 : nBitPSymb_comb <= (is40MHz) ? 15'd648   : 15'd312  ;
            7'd27 : nBitPSymb_comb <= (is40MHz) ? 15'd864   : 15'd416  ;
            7'd28 : nBitPSymb_comb <= (is40MHz) ? 15'd1296  : 15'd624  ;
            7'd29 : nBitPSymb_comb <= (is40MHz) ? 15'd1728  : 15'd832  ;
            7'd30 : nBitPSymb_comb <= (is40MHz) ? 15'd1944  : 15'd936  ;
            7'd31 : nBitPSymb_comb <= (is40MHz) ? 15'd2160  : 15'd1040 ;
            7'd32 : nBitPSymb_comb <=                         15'd24   ;
            7'd33 : nBitPSymb_comb <= (is40MHz) ? 15'd324   : 15'd156  ;
            7'd34 : nBitPSymb_comb <= (is40MHz) ? 15'd432   : 15'd208  ;
            7'd35 : nBitPSymb_comb <= (is40MHz) ? 15'd540   : 15'd260  ;
            7'd36 : nBitPSymb_comb <= (is40MHz) ? 15'd486   : 15'd234  ;
            7'd37 : nBitPSymb_comb <= (is40MHz) ? 15'd648   : 15'd312  ;
            7'd38 : nBitPSymb_comb <= (is40MHz) ? 15'd810   : 15'd390  ;
            7'd39 : nBitPSymb_comb <= (is40MHz) ? 15'd432   : 15'd208  ;
            7'd40 : nBitPSymb_comb <= (is40MHz) ? 15'd540   : 15'd260  ;
            7'd41 : nBitPSymb_comb <= (is40MHz) ? 15'd540   : 15'd260  ;
            7'd42 : nBitPSymb_comb <= (is40MHz) ? 15'd648   : 15'd312  ;
            7'd43 : nBitPSymb_comb <= (is40MHz) ? 15'd756   : 15'd364  ;
            7'd44 : nBitPSymb_comb <= (is40MHz) ? 15'd756   : 15'd364  ;
            7'd45 : nBitPSymb_comb <= (is40MHz) ? 15'd864   : 15'd416  ;
            7'd46 : nBitPSymb_comb <= (is40MHz) ? 15'd648   : 15'd312  ;
            7'd47 : nBitPSymb_comb <= (is40MHz) ? 15'd810   : 15'd390  ;
            7'd48 : nBitPSymb_comb <= (is40MHz) ? 15'd810   : 15'd390  ;
            7'd49 : nBitPSymb_comb <= (is40MHz) ? 15'd972   : 15'd468  ;
            7'd50 : nBitPSymb_comb <= (is40MHz) ? 15'd1134  : 15'd546  ;
            7'd51 : nBitPSymb_comb <= (is40MHz) ? 15'd1134  : 15'd546  ;
            7'd52 : nBitPSymb_comb <= (is40MHz) ? 15'd1296  : 15'd624  ;
            7'd53 : nBitPSymb_comb <= (is40MHz) ? 15'd540   : 15'd260  ;
            7'd54 : nBitPSymb_comb <= (is40MHz) ? 15'd648   : 15'd312  ;
            7'd55 : nBitPSymb_comb <= (is40MHz) ? 15'd756   : 15'd364  ;
            7'd56 : nBitPSymb_comb <= (is40MHz) ? 15'd648   : 15'd312  ;
            7'd57 : nBitPSymb_comb <= (is40MHz) ? 15'd756   : 15'd364  ;
            7'd58 : nBitPSymb_comb <= (is40MHz) ? 15'd864   : 15'd416  ;
            7'd59 : nBitPSymb_comb <= (is40MHz) ? 15'd972   : 15'd468  ;
            7'd60 : nBitPSymb_comb <= (is40MHz) ? 15'd864   : 15'd416  ;
            7'd61 : nBitPSymb_comb <= (is40MHz) ? 15'd972   : 15'd468  ;
            7'd62 : nBitPSymb_comb <= (is40MHz) ? 15'd1080  : 15'd520  ;
            7'd63 : nBitPSymb_comb <= (is40MHz) ? 15'd1080  : 15'd520  ;
            7'd64 : nBitPSymb_comb <= (is40MHz) ? 15'd1188  : 15'd572  ;
            7'd65 : nBitPSymb_comb <= (is40MHz) ? 15'd810   : 15'd390  ;
            7'd66 : nBitPSymb_comb <= (is40MHz) ? 15'd972   : 15'd468  ;
            7'd67 : nBitPSymb_comb <= (is40MHz) ? 15'd1134  : 15'd546  ;
            7'd68 : nBitPSymb_comb <= (is40MHz) ? 15'd972   : 15'd468  ;
            7'd69 : nBitPSymb_comb <= (is40MHz) ? 15'd1134  : 15'd546  ;
            7'd70 : nBitPSymb_comb <= (is40MHz) ? 15'd1296  : 15'd624  ;
            7'd71 : nBitPSymb_comb <= (is40MHz) ? 15'd1458  : 15'd702  ;
            7'd72 : nBitPSymb_comb <= (is40MHz) ? 15'd1296  : 15'd624  ;
            7'd73 : nBitPSymb_comb <= (is40MHz) ? 15'd1458  : 15'd702  ;
            7'd74 : nBitPSymb_comb <= (is40MHz) ? 15'd1620  : 15'd780  ;
            7'd75 : nBitPSymb_comb <= (is40MHz) ? 15'd1620  : 15'd780  ;
            7'd76 : nBitPSymb_comb <= (is40MHz) ? 15'd1782  : 15'd858  ;
    // Disable coverage on the default state because it cannot be reached.
    // pragma coverage block = off 
       default : nBitPSymb_comb <= 15'd1;
    // pragma coverage block = on 
    endcase
  end
  else if(preambleType[3:0] == 4'b0100)
  // VHT - Table 22-30 to 22-61
  begin
    case(nSS_comb)
      4'd1:
      begin
        case(mcsIndex[3:0])
          4'd0: nBitPSymb_comb <= (is160MHz) ? 15'd234 : (is80MHz) ? 15'd117 : (is40MHz) ? 15'd54 : 15'd26  ;
          4'd1: nBitPSymb_comb <= (is160MHz) ? 15'd468 : (is80MHz) ? 15'd234 : (is40MHz) ? 15'd108: 15'd52  ;
          4'd2: nBitPSymb_comb <= (is160MHz) ? 15'd702 : (is80MHz) ? 15'd351 : (is40MHz) ? 15'd162: 15'd78  ;
          4'd3: nBitPSymb_comb <= (is160MHz) ? 15'd936 : (is80MHz) ? 15'd468 : (is40MHz) ? 15'd216: 15'd104 ;
          4'd4: nBitPSymb_comb <= (is160MHz) ? 15'd1404: (is80MHz) ? 15'd702 : (is40MHz) ? 15'd324: 15'd156 ;
          4'd5: nBitPSymb_comb <= (is160MHz) ? 15'd1872: (is80MHz) ? 15'd936 : (is40MHz) ? 15'd432: 15'd208 ;
          4'd6: nBitPSymb_comb <= (is160MHz) ? 15'd2106: (is80MHz) ? 15'd1053: (is40MHz) ? 15'd486: 15'd234 ;
          4'd7: nBitPSymb_comb <= (is160MHz) ? 15'd2340: (is80MHz) ? 15'd1170: (is40MHz) ? 15'd540: 15'd260 ;
          4'd8: nBitPSymb_comb <= (is160MHz) ? 15'd2808: (is80MHz) ? 15'd1404: (is40MHz) ? 15'd648: 15'd312 ;
          4'd9: nBitPSymb_comb <= (is160MHz) ? 15'd3120: (is80MHz) ? 15'd1560: (is40MHz) ? 15'd720: 15'd1   ;
          // Disable coverage on the default state because it cannot be reached.
          // pragma coverage block = off 
          default: nBitPSymb_comb <= 15'd1;
          // pragma coverage block = on
        endcase
      end
      4'd2:
      begin
        case(mcsIndex[3:0])
          4'd0: nBitPSymb_comb <= (is160MHz) ? 15'd468 : (is80MHz) ? 15'd234 : (is40MHz) ? 15'd108 : 15'd52   ;
          4'd1: nBitPSymb_comb <= (is160MHz) ? 15'd936 : (is80MHz) ? 15'd468 : (is40MHz) ? 15'd216 : 15'd104  ;
          4'd2: nBitPSymb_comb <= (is160MHz) ? 15'd1404: (is80MHz) ? 15'd702 : (is40MHz) ? 15'd324 : 15'd156  ;
          4'd3: nBitPSymb_comb <= (is160MHz) ? 15'd1872: (is80MHz) ? 15'd936 : (is40MHz) ? 15'd432 : 15'd208  ;
          4'd4: nBitPSymb_comb <= (is160MHz) ? 15'd2808: (is80MHz) ? 15'd1404: (is40MHz) ? 15'd648 : 15'd312  ;
          4'd5: nBitPSymb_comb <= (is160MHz) ? 15'd3744: (is80MHz) ? 15'd1872: (is40MHz) ? 15'd864 : 15'd416  ;
          4'd6: nBitPSymb_comb <= (is160MHz) ? 15'd4212: (is80MHz) ? 15'd2106: (is40MHz) ? 15'd972 : 15'd468  ;
          4'd7: nBitPSymb_comb <= (is160MHz) ? 15'd4680: (is80MHz) ? 15'd2340: (is40MHz) ? 15'd1080: 15'd520  ;
          4'd8: nBitPSymb_comb <= (is160MHz) ? 15'd5616: (is80MHz) ? 15'd2808: (is40MHz) ? 15'd1296: 15'd624  ;
          4'd9: nBitPSymb_comb <= (is160MHz) ? 15'd6240: (is80MHz) ? 15'd3120: (is40MHz) ? 15'd1440: 15'd1    ;
          // Disable coverage on the default state because it cannot be reached.
          // pragma coverage block = off 
          default: nBitPSymb_comb <= 15'd1;
          // pragma coverage block = on
        endcase
      end
      4'd3:
      begin
        case(mcsIndex[3:0])
          4'd0: nBitPSymb_comb <= (is160MHz) ? 15'd702 : (is80MHz) ? 15'd351 : (is40MHz) ? 15'd162 : 15'd78   ;
          4'd1: nBitPSymb_comb <= (is160MHz) ? 15'd1404: (is80MHz) ? 15'd702 : (is40MHz) ? 15'd324 : 15'd156  ;
          4'd2: nBitPSymb_comb <= (is160MHz) ? 15'd2106: (is80MHz) ? 15'd1053: (is40MHz) ? 15'd486 : 15'd234  ;
          4'd3: nBitPSymb_comb <= (is160MHz) ? 15'd2808: (is80MHz) ? 15'd1404: (is40MHz) ? 15'd648 : 15'd312  ;
          4'd4: nBitPSymb_comb <= (is160MHz) ? 15'd4212: (is80MHz) ? 15'd2106: (is40MHz) ? 15'd972 : 15'd468  ;
          4'd5: nBitPSymb_comb <= (is160MHz) ? 15'd5616: (is80MHz) ? 15'd2808: (is40MHz) ? 15'd1296: 15'd624  ;
          4'd6: nBitPSymb_comb <= (is160MHz) ? 15'd6318: (is80MHz) ? 15'd1   : (is40MHz) ? 15'd1458: 15'd702  ;
          4'd7: nBitPSymb_comb <= (is160MHz) ? 15'd7020: (is80MHz) ? 15'd3510: (is40MHz) ? 15'd1620: 15'd780  ;
          4'd8: nBitPSymb_comb <= (is160MHz) ? 15'd8424: (is80MHz) ? 15'd4212: (is40MHz) ? 15'd1944: 15'd936  ;
          4'd9: nBitPSymb_comb <= (is160MHz) ? 15'd1   : (is80MHz) ? 15'd4680: (is40MHz) ? 15'd2160: 15'd1040 ;
          // Disable coverage on the default state because it cannot be reached.
          // pragma coverage block = off 
          default: nBitPSymb_comb <= 15'd1;
          // pragma coverage block = on
        endcase
      end
      4'd4:
      begin
        case(mcsIndex[3:0])
          4'd0: nBitPSymb_comb <= (is160MHz) ? 15'd936  : (is80MHz) ? 15'd468 : (is40MHz) ? 15'd216 : 15'd104  ;
          4'd1: nBitPSymb_comb <= (is160MHz) ? 15'd1872 : (is80MHz) ? 15'd936 : (is40MHz) ? 15'd432 : 15'd208  ;
          4'd2: nBitPSymb_comb <= (is160MHz) ? 15'd2808 : (is80MHz) ? 15'd1404: (is40MHz) ? 15'd648 : 15'd312  ;
          4'd3: nBitPSymb_comb <= (is160MHz) ? 15'd3744 : (is80MHz) ? 15'd1872: (is40MHz) ? 15'd864 : 15'd416  ;
          4'd4: nBitPSymb_comb <= (is160MHz) ? 15'd5616 : (is80MHz) ? 15'd2808: (is40MHz) ? 15'd1296: 15'd624  ;
          4'd5: nBitPSymb_comb <= (is160MHz) ? 15'd7488 : (is80MHz) ? 15'd3744: (is40MHz) ? 15'd1728: 15'd832  ;
          4'd6: nBitPSymb_comb <= (is160MHz) ? 15'd8424 : (is80MHz) ? 15'd4212: (is40MHz) ? 15'd1944: 15'd936  ;
          4'd7: nBitPSymb_comb <= (is160MHz) ? 15'd9360 : (is80MHz) ? 15'd4680: (is40MHz) ? 15'd2160: 15'd1040 ;
          4'd8: nBitPSymb_comb <= (is160MHz) ? 15'd11232: (is80MHz) ? 15'd5616: (is40MHz) ? 15'd2592: 15'd1248 ;
          4'd9: nBitPSymb_comb <= (is160MHz) ? 15'd12480: (is80MHz) ? 15'd6240: (is40MHz) ? 15'd2880: 15'd1    ;
          // Disable coverage on the default state because it cannot be reached.
          // pragma coverage block = off 
          default: nBitPSymb_comb <= 15'd1;
          // pragma coverage block = on
        endcase
      end
      4'd5:
      begin
        case(mcsIndex[3:0])
          4'd0: nBitPSymb_comb <= (is160MHz) ? 15'd1170 : (is80MHz) ? 15'd585 : (is40MHz) ? 15'd270 : 15'd130  ;
          4'd1: nBitPSymb_comb <= (is160MHz) ? 15'd2340 : (is80MHz) ? 15'd1170: (is40MHz) ? 15'd540 : 15'd260  ;
          4'd2: nBitPSymb_comb <= (is160MHz) ? 15'd3510 : (is80MHz) ? 15'd1755: (is40MHz) ? 15'd810 : 15'd390  ;
          4'd3: nBitPSymb_comb <= (is160MHz) ? 15'd4680 : (is80MHz) ? 15'd2340: (is40MHz) ? 15'd1080: 15'd520  ;
          4'd4: nBitPSymb_comb <= (is160MHz) ? 15'd7020 : (is80MHz) ? 15'd3510: (is40MHz) ? 15'd1620: 15'd780  ;
          4'd5: nBitPSymb_comb <= (is160MHz) ? 15'd9360 : (is80MHz) ? 15'd4680: (is40MHz) ? 15'd2160: 15'd1040 ;
          4'd6: nBitPSymb_comb <= (is160MHz) ? 15'd10530: (is80MHz) ? 15'd5265: (is40MHz) ? 15'd2430: 15'd1170 ;
          4'd7: nBitPSymb_comb <= (is160MHz) ? 15'd11700: (is80MHz) ? 15'd5850: (is40MHz) ? 15'd2700: 15'd1300 ;
          4'd8: nBitPSymb_comb <= (is160MHz) ? 15'd14040: (is80MHz) ? 15'd7020: (is40MHz) ? 15'd3240: 15'd1560 ;
          4'd9: nBitPSymb_comb <= (is160MHz) ? 15'd15600: (is80MHz) ? 15'd7800: (is40MHz) ? 15'd3600: 15'd1    ;
          // Disable coverage on the default state because it cannot be reached.
          // pragma coverage block = off 
          default: nBitPSymb_comb <= 15'd1;
          // pragma coverage block = on
        endcase
      end
      4'd6:
      begin
        case(mcsIndex[3:0])
          4'd0: nBitPSymb_comb <= (is160MHz) ? 15'd1404 : (is80MHz) ? 15'd702 : (is40MHz) ? 15'd324 : 15'd156  ;
          4'd1: nBitPSymb_comb <= (is160MHz) ? 15'd2808 : (is80MHz) ? 15'd1404: (is40MHz) ? 15'd648 : 15'd312  ;
          4'd2: nBitPSymb_comb <= (is160MHz) ? 15'd4212 : (is80MHz) ? 15'd2106: (is40MHz) ? 15'd972 : 15'd468  ;
          4'd3: nBitPSymb_comb <= (is160MHz) ? 15'd5616 : (is80MHz) ? 15'd2808: (is40MHz) ? 15'd1296: 15'd624  ;
          4'd4: nBitPSymb_comb <= (is160MHz) ? 15'd8424 : (is80MHz) ? 15'd4212: (is40MHz) ? 15'd1944: 15'd936  ;
          4'd5: nBitPSymb_comb <= (is160MHz) ? 15'd11232: (is80MHz) ? 15'd5616: (is40MHz) ? 15'd2592: 15'd1248 ;
          4'd6: nBitPSymb_comb <= (is160MHz) ? 15'd12636: (is80MHz) ? 15'd6318: (is40MHz) ? 15'd2916: 15'd1404 ;
          4'd7: nBitPSymb_comb <= (is160MHz) ? 15'd14040: (is80MHz) ? 15'd7020: (is40MHz) ? 15'd3240: 15'd1560 ;
          4'd8: nBitPSymb_comb <= (is160MHz) ? 15'd16848: (is80MHz) ? 15'd8424: (is40MHz) ? 15'd3888: 15'd1872 ;
          4'd9: nBitPSymb_comb <= (is160MHz) ? 15'd18720: (is80MHz) ? 15'd1   : (is40MHz) ? 15'd4320: 15'd2080 ;
          // Disable coverage on the default state because it cannot be reached.
          // pragma coverage block = off 
          default: nBitPSymb_comb <= 15'd1;
          // pragma coverage block = on
        endcase
      end
      4'd7:
      begin
        case(mcsIndex[3:0])
          4'd0: nBitPSymb_comb <= (is160MHz) ? 15'd1638 : (is80MHz) ? 15'd819  : (is40MHz) ? 15'd378 : 15'd182  ;
          4'd1: nBitPSymb_comb <= (is160MHz) ? 15'd3276 : (is80MHz) ? 15'd1638 : (is40MHz) ? 15'd756 : 15'd364  ;
          4'd2: nBitPSymb_comb <= (is160MHz) ? 15'd4914 : (is80MHz) ? 15'd2457 : (is40MHz) ? 15'd1134: 15'd546  ;
          4'd3: nBitPSymb_comb <= (is160MHz) ? 15'd6552 : (is80MHz) ? 15'd3276 : (is40MHz) ? 15'd1512: 15'd728  ;
          4'd4: nBitPSymb_comb <= (is160MHz) ? 15'd9828 : (is80MHz) ? 15'd4914 : (is40MHz) ? 15'd2268: 15'd1092 ;
          4'd5: nBitPSymb_comb <= (is160MHz) ? 15'd13104: (is80MHz) ? 15'd6552 : (is40MHz) ? 15'd3024: 15'd1456 ;
          4'd6: nBitPSymb_comb <= (is160MHz) ? 15'd14742: (is80MHz) ? 15'd1    : (is40MHz) ? 15'd3402: 15'd1638 ;
          4'd7: nBitPSymb_comb <= (is160MHz) ? 15'd16380: (is80MHz) ? 15'd8190 : (is40MHz) ? 15'd3780: 15'd1820 ;
          4'd8: nBitPSymb_comb <= (is160MHz) ? 15'd19656: (is80MHz) ? 15'd9828 : (is40MHz) ? 15'd4536: 15'd2184 ;
          4'd9: nBitPSymb_comb <= (is160MHz) ? 15'd21840: (is80MHz) ? 15'd10920: (is40MHz) ? 15'd5040: 15'd1    ;
          // Disable coverage on the default state because it cannot be reached.
          // pragma coverage block = off 
          default: nBitPSymb_comb <= 15'd1;
          // pragma coverage block = on
        endcase
      end
      4'd8:
      begin
        case(mcsIndex[3:0])
          4'd0: nBitPSymb_comb <= (is160MHz) ? 15'd1872 : (is80MHz) ? 15'd936  : (is40MHz) ? 15'd432 : 15'd208  ;
          4'd1: nBitPSymb_comb <= (is160MHz) ? 15'd3744 : (is80MHz) ? 15'd1872 : (is40MHz) ? 15'd864 : 15'd416  ;
          4'd2: nBitPSymb_comb <= (is160MHz) ? 15'd5616 : (is80MHz) ? 15'd2808 : (is40MHz) ? 15'd1296: 15'd624  ;
          4'd3: nBitPSymb_comb <= (is160MHz) ? 15'd7488 : (is80MHz) ? 15'd3744 : (is40MHz) ? 15'd1728: 15'd832  ;
          4'd4: nBitPSymb_comb <= (is160MHz) ? 15'd11232: (is80MHz) ? 15'd5616 : (is40MHz) ? 15'd2592: 15'd1248 ;
          4'd5: nBitPSymb_comb <= (is160MHz) ? 15'd14976: (is80MHz) ? 15'd7488 : (is40MHz) ? 15'd3456: 15'd1664 ;
          4'd6: nBitPSymb_comb <= (is160MHz) ? 15'd16848: (is80MHz) ? 15'd8424 : (is40MHz) ? 15'd3888: 15'd1872 ;
          4'd7: nBitPSymb_comb <= (is160MHz) ? 15'd18720: (is80MHz) ? 15'd9360 : (is40MHz) ? 15'd4320: 15'd2080 ;
          4'd8: nBitPSymb_comb <= (is160MHz) ? 15'd22464: (is80MHz) ? 15'd11232: (is40MHz) ? 15'd5184: 15'd2496 ;
          4'd9: nBitPSymb_comb <= (is160MHz) ? 15'd24960: (is80MHz) ? 15'd12480: (is40MHz) ? 15'd5760: 15'd1    ;
          // Disable coverage on the default state because it cannot be reached.
          // pragma coverage block = off 
          default: nBitPSymb_comb <= 15'd1;
          // pragma coverage block = on
        endcase
      end
      // Disable coverage on the default state because it cannot be reached.
      // pragma coverage block = off 
      default: nBitPSymb_comb <= 15'd1;
      // pragma coverage block = on
    endcase
  end
  else if(preambleType[3:0] == 4'b0101 | // HE-SU
          preambleType[3:0] == 4'b0111 | // HE-ER-SU
          preambleType[3:0] == 4'b0110 | // HE-MU
          preambleType[3:0] == 4'b1000 ) // HE-TB
  // HE - 802.11ax D3.1 - Table 28-51 to 28-106
  begin
    case({ruType,nSS_comb,mcsIndex[3:0]})
      //// ru=26, Nss=1
      {3'd0, 4'd1, 4'd0 }: nBitPSymb_comb <= 15'd12;
      {3'd0, 4'd1, 4'd1 }: nBitPSymb_comb <= 15'd24;
      {3'd0, 4'd1, 4'd2 }: nBitPSymb_comb <= 15'd36;
      {3'd0, 4'd1, 4'd3 }: nBitPSymb_comb <= 15'd48;
      {3'd0, 4'd1, 4'd4 }: nBitPSymb_comb <= 15'd72;
      {3'd0, 4'd1, 4'd5 }: nBitPSymb_comb <= 15'd96;
      {3'd0, 4'd1, 4'd6 }: nBitPSymb_comb <= 15'd108;
      {3'd0, 4'd1, 4'd7 }: nBitPSymb_comb <= 15'd120;
      {3'd0, 4'd1, 4'd8 }: nBitPSymb_comb <= 15'd144;
      {3'd0, 4'd1, 4'd9 }: nBitPSymb_comb <= 15'd160;
      {3'd0, 4'd1, 4'd10}: nBitPSymb_comb <= 15'd180;
      {3'd0, 4'd1, 4'd11}: nBitPSymb_comb <= 15'd200;
      //// ru=26, Nss=2
      {3'd0, 4'd2, 4'd0 }: nBitPSymb_comb <= 15'd24;
      {3'd0, 4'd2, 4'd1 }: nBitPSymb_comb <= 15'd48;
      {3'd0, 4'd2, 4'd2 }: nBitPSymb_comb <= 15'd72;
      {3'd0, 4'd2, 4'd3 }: nBitPSymb_comb <= 15'd96;
      {3'd0, 4'd2, 4'd4 }: nBitPSymb_comb <= 15'd144;
      {3'd0, 4'd2, 4'd5 }: nBitPSymb_comb <= 15'd192;
      {3'd0, 4'd2, 4'd6 }: nBitPSymb_comb <= 15'd216;
      {3'd0, 4'd2, 4'd7 }: nBitPSymb_comb <= 15'd240;
      {3'd0, 4'd2, 4'd8 }: nBitPSymb_comb <= 15'd288;
      {3'd0, 4'd2, 4'd9 }: nBitPSymb_comb <= 15'd320;
      {3'd0, 4'd2, 4'd10}: nBitPSymb_comb <= 15'd360;
      {3'd0, 4'd2, 4'd11}: nBitPSymb_comb <= 15'd400;
      //// ru=52, Nss=1
      {3'd1, 4'd1, 4'd0 }: nBitPSymb_comb <= 15'd24;
      {3'd1, 4'd1, 4'd1 }: nBitPSymb_comb <= 15'd48;
      {3'd1, 4'd1, 4'd2 }: nBitPSymb_comb <= 15'd72;
      {3'd1, 4'd1, 4'd3 }: nBitPSymb_comb <= 15'd96;
      {3'd1, 4'd1, 4'd4 }: nBitPSymb_comb <= 15'd144;
      {3'd1, 4'd1, 4'd5 }: nBitPSymb_comb <= 15'd192;
      {3'd1, 4'd1, 4'd6 }: nBitPSymb_comb <= 15'd216;
      {3'd1, 4'd1, 4'd7 }: nBitPSymb_comb <= 15'd240;
      {3'd1, 4'd1, 4'd8 }: nBitPSymb_comb <= 15'd288;
      {3'd1, 4'd1, 4'd9 }: nBitPSymb_comb <= 15'd320;
      {3'd1, 4'd1, 4'd10}: nBitPSymb_comb <= 15'd360;
      {3'd1, 4'd1, 4'd11}: nBitPSymb_comb <= 15'd400;
      //// ru=52, Nss=2
      {3'd1, 4'd2, 4'd0 }: nBitPSymb_comb <= 15'd48;
      {3'd1, 4'd2, 4'd1 }: nBitPSymb_comb <= 15'd96;
      {3'd1, 4'd2, 4'd2 }: nBitPSymb_comb <= 15'd144;
      {3'd1, 4'd2, 4'd3 }: nBitPSymb_comb <= 15'd192;
      {3'd1, 4'd2, 4'd4 }: nBitPSymb_comb <= 15'd288;
      {3'd1, 4'd2, 4'd5 }: nBitPSymb_comb <= 15'd384;
      {3'd1, 4'd2, 4'd6 }: nBitPSymb_comb <= 15'd432;
      {3'd1, 4'd2, 4'd7 }: nBitPSymb_comb <= 15'd480;
      {3'd1, 4'd2, 4'd8 }: nBitPSymb_comb <= 15'd576;
      {3'd1, 4'd2, 4'd9 }: nBitPSymb_comb <= 15'd640;
      {3'd1, 4'd2, 4'd10}: nBitPSymb_comb <= 15'd720;
      {3'd1, 4'd2, 4'd11}: nBitPSymb_comb <= 15'd800;
      //// ru=106, Nss=1
      {3'd2, 4'd1, 4'd0 }: nBitPSymb_comb <= 15'd51;
      {3'd2, 4'd1, 4'd1 }: nBitPSymb_comb <= 15'd102;
      {3'd2, 4'd1, 4'd2 }: nBitPSymb_comb <= 15'd153;
      {3'd2, 4'd1, 4'd3 }: nBitPSymb_comb <= 15'd204;
      {3'd2, 4'd1, 4'd4 }: nBitPSymb_comb <= 15'd306;
      {3'd2, 4'd1, 4'd5 }: nBitPSymb_comb <= 15'd408;
      {3'd2, 4'd1, 4'd6 }: nBitPSymb_comb <= 15'd459;
      {3'd2, 4'd1, 4'd7 }: nBitPSymb_comb <= 15'd510;
      {3'd2, 4'd1, 4'd8 }: nBitPSymb_comb <= 15'd612;
      {3'd2, 4'd1, 4'd9 }: nBitPSymb_comb <= 15'd680;
      {3'd2, 4'd1, 4'd10}: nBitPSymb_comb <= 15'd765;
      {3'd2, 4'd1, 4'd11}: nBitPSymb_comb <= 15'd850;
      //// ru=106, Nss=2
      {3'd2, 4'd2, 4'd0 }: nBitPSymb_comb <= 15'd102;
      {3'd2, 4'd2, 4'd1 }: nBitPSymb_comb <= 15'd204;
      {3'd2, 4'd2, 4'd2 }: nBitPSymb_comb <= 15'd306;
      {3'd2, 4'd2, 4'd3 }: nBitPSymb_comb <= 15'd408;
      {3'd2, 4'd2, 4'd4 }: nBitPSymb_comb <= 15'd612;
      {3'd2, 4'd2, 4'd5 }: nBitPSymb_comb <= 15'd816;
      {3'd2, 4'd2, 4'd6 }: nBitPSymb_comb <= 15'd918;
      {3'd2, 4'd2, 4'd7 }: nBitPSymb_comb <= 15'd1020;
      {3'd2, 4'd2, 4'd8 }: nBitPSymb_comb <= 15'd1224;
      {3'd2, 4'd2, 4'd9 }: nBitPSymb_comb <= 15'd1360;
      {3'd2, 4'd2, 4'd10}: nBitPSymb_comb <= 15'd1530;
      {3'd2, 4'd2, 4'd11}: nBitPSymb_comb <= 15'd1700;
      // ru=242, Nss=1
      {3'd3, 4'd1, 4'd0 }: nBitPSymb_comb <= 15'd117;
      {3'd3, 4'd1, 4'd1 }: nBitPSymb_comb <= 15'd234;
      {3'd3, 4'd1, 4'd2 }: nBitPSymb_comb <= 15'd351;
      {3'd3, 4'd1, 4'd3 }: nBitPSymb_comb <= 15'd468;
      {3'd3, 4'd1, 4'd4 }: nBitPSymb_comb <= 15'd702;
      {3'd3, 4'd1, 4'd5 }: nBitPSymb_comb <= 15'd936;
      {3'd3, 4'd1, 4'd6 }: nBitPSymb_comb <= 15'd1053;
      {3'd3, 4'd1, 4'd7 }: nBitPSymb_comb <= 15'd1170;
      {3'd3, 4'd1, 4'd8 }: nBitPSymb_comb <= 15'd1404;
      {3'd3, 4'd1, 4'd9 }: nBitPSymb_comb <= 15'd1560;
      {3'd3, 4'd1, 4'd10}: nBitPSymb_comb <= 15'd1755;
      {3'd3, 4'd1, 4'd11}: nBitPSymb_comb <= 15'd1950;
      // ru=242, Nss=2
      {3'd3, 4'd2, 4'd0 }: nBitPSymb_comb <= 15'd234;
      {3'd3, 4'd2, 4'd1 }: nBitPSymb_comb <= 15'd468;
      {3'd3, 4'd2, 4'd2 }: nBitPSymb_comb <= 15'd702;
      {3'd3, 4'd2, 4'd3 }: nBitPSymb_comb <= 15'd936;
      {3'd3, 4'd2, 4'd4 }: nBitPSymb_comb <= 15'd1404;
      {3'd3, 4'd2, 4'd5 }: nBitPSymb_comb <= 15'd1872;
      {3'd3, 4'd2, 4'd6 }: nBitPSymb_comb <= 15'd2106;
      {3'd3, 4'd2, 4'd7 }: nBitPSymb_comb <= 15'd2340;
      {3'd3, 4'd2, 4'd8 }: nBitPSymb_comb <= 15'd2808;
      {3'd3, 4'd2, 4'd9 }: nBitPSymb_comb <= 15'd3120;
      {3'd3, 4'd2, 4'd10}: nBitPSymb_comb <= 15'd3510;
      {3'd3, 4'd2, 4'd11}: nBitPSymb_comb <= 15'd3900;
      // ru=484, Nss=1
      {3'd4, 4'd1, 4'd0 }: nBitPSymb_comb <= 15'd234;
      {3'd4, 4'd1, 4'd1 }: nBitPSymb_comb <= 15'd468;
      {3'd4, 4'd1, 4'd2 }: nBitPSymb_comb <= 15'd702;
      {3'd4, 4'd1, 4'd3 }: nBitPSymb_comb <= 15'd936;
      {3'd4, 4'd1, 4'd4 }: nBitPSymb_comb <= 15'd1404;
      {3'd4, 4'd1, 4'd5 }: nBitPSymb_comb <= 15'd1872;
      {3'd4, 4'd1, 4'd6 }: nBitPSymb_comb <= 15'd2106;
      {3'd4, 4'd1, 4'd7 }: nBitPSymb_comb <= 15'd2340;
      {3'd4, 4'd1, 4'd8 }: nBitPSymb_comb <= 15'd2808;
      {3'd4, 4'd1, 4'd9 }: nBitPSymb_comb <= 15'd3120;
      {3'd4, 4'd1, 4'd10}: nBitPSymb_comb <= 15'd3510;
      {3'd4, 4'd1, 4'd11}: nBitPSymb_comb <= 15'd3900;
      // ru=484, Nss=2
      {3'd4, 4'd2, 4'd0 }: nBitPSymb_comb <= 15'd468;
      {3'd4, 4'd2, 4'd1 }: nBitPSymb_comb <= 15'd936;
      {3'd4, 4'd2, 4'd2 }: nBitPSymb_comb <= 15'd1404;
      {3'd4, 4'd2, 4'd3 }: nBitPSymb_comb <= 15'd1872;
      {3'd4, 4'd2, 4'd4 }: nBitPSymb_comb <= 15'd2808;
      {3'd4, 4'd2, 4'd5 }: nBitPSymb_comb <= 15'd3744;
      {3'd4, 4'd2, 4'd6 }: nBitPSymb_comb <= 15'd4212;
      {3'd4, 4'd2, 4'd7 }: nBitPSymb_comb <= 15'd4680;
      {3'd4, 4'd2, 4'd8 }: nBitPSymb_comb <= 15'd5616;
      {3'd4, 4'd2, 4'd9 }: nBitPSymb_comb <= 15'd6240;
      {3'd4, 4'd2, 4'd10}: nBitPSymb_comb <= 15'd7020;
      {3'd4, 4'd2, 4'd11}: nBitPSymb_comb <= 15'd7800;
      // ru=996, Nss=1
      {3'd5, 4'd1, 4'd0 }: nBitPSymb_comb <= 15'd490;
      {3'd5, 4'd1, 4'd1 }: nBitPSymb_comb <= 15'd980;
      {3'd5, 4'd1, 4'd2 }: nBitPSymb_comb <= 15'd1470;
      {3'd5, 4'd1, 4'd3 }: nBitPSymb_comb <= 15'd1960;
      {3'd5, 4'd1, 4'd4 }: nBitPSymb_comb <= 15'd2940;
      {3'd5, 4'd1, 4'd5 }: nBitPSymb_comb <= 15'd3920;
      {3'd5, 4'd1, 4'd6 }: nBitPSymb_comb <= 15'd4410;
      {3'd5, 4'd1, 4'd7 }: nBitPSymb_comb <= 15'd4900;
      {3'd5, 4'd1, 4'd8 }: nBitPSymb_comb <= 15'd5880;
      {3'd5, 4'd1, 4'd9 }: nBitPSymb_comb <= 15'd6533;
      {3'd5, 4'd1, 4'd10}: nBitPSymb_comb <= 15'd7350;
      {3'd5, 4'd1, 4'd11}: nBitPSymb_comb <= 15'd8166;
      // ru=996, Nss=2
      {3'd5, 4'd2, 4'd0 }: nBitPSymb_comb <= 15'd980;
      {3'd5, 4'd2, 4'd1 }: nBitPSymb_comb <= 15'd1960;
      {3'd5, 4'd2, 4'd2 }: nBitPSymb_comb <= 15'd2940;
      {3'd5, 4'd2, 4'd3 }: nBitPSymb_comb <= 15'd3920;
      {3'd5, 4'd2, 4'd4 }: nBitPSymb_comb <= 15'd5880;
      {3'd5, 4'd2, 4'd5 }: nBitPSymb_comb <= 15'd7840;
      {3'd5, 4'd2, 4'd6 }: nBitPSymb_comb <= 15'd8820;
      {3'd5, 4'd2, 4'd7 }: nBitPSymb_comb <= 15'd9800;
      {3'd5, 4'd2, 4'd8 }: nBitPSymb_comb <= 15'd11760;
      {3'd5, 4'd2, 4'd9 }: nBitPSymb_comb <= 15'd13066;
      {3'd5, 4'd2, 4'd10}: nBitPSymb_comb <= 15'd14700;
      {3'd5, 4'd2, 4'd11}: nBitPSymb_comb <= 15'd16333;
      // Disable coverage on the default state because it cannot be reached.
      // pragma coverage block = off 
      default:             nBitPSymb_comb <= 15'd1;
      // pragma coverage block = on
    endcase
  end
end


// Set nBitPSymbShort1ss with the short number of bits per symbol for Nss=1.
always @*
begin
   case({ruType,mcsIndex[3:0],dcm})
   // ru=26
   {3'd0, 4'd0 , 1'b1}: nBitPSymbShort1ss = 11'd1;
   {3'd0, 4'd0 , 1'b0}: nBitPSymbShort1ss = 11'd3;
   {3'd0, 4'd1 , 1'b1}: nBitPSymbShort1ss = 11'd2;
   {3'd0, 4'd1 , 1'b0}: nBitPSymbShort1ss = 11'd6;
   {3'd0, 4'd2 , 1'b0}: nBitPSymbShort1ss = 11'd9;
   {3'd0, 4'd3 , 1'b1}: nBitPSymbShort1ss = 11'd4;
   {3'd0, 4'd3 , 1'b0}: nBitPSymbShort1ss = 11'd12;
   {3'd0, 4'd4 , 1'b1}: nBitPSymbShort1ss = 11'd6; 
   {3'd0, 4'd4 , 1'b0}: nBitPSymbShort1ss = 11'd18;
   {3'd0, 4'd5 , 1'b0}: nBitPSymbShort1ss = 11'd24;
   {3'd0, 4'd6 , 1'b0}: nBitPSymbShort1ss = 11'd27;
   {3'd0, 4'd7 , 1'b0}: nBitPSymbShort1ss = 11'd30;
   {3'd0, 4'd8 , 1'b0}: nBitPSymbShort1ss = 11'd36;
   {3'd0, 4'd9 , 1'b0}: nBitPSymbShort1ss = 11'd40;
   {3'd0, 4'd10, 1'b0}: nBitPSymbShort1ss = 11'd45;
   {3'd0, 4'd11, 1'b0}: nBitPSymbShort1ss = 11'd50;
   // ru=52
   {3'd1, 4'd0 , 1'b1}: nBitPSymbShort1ss = 11'd3; 
   {3'd1, 4'd0 , 1'b0}: nBitPSymbShort1ss = 11'd6; 
   {3'd1, 4'd1 , 1'b1}: nBitPSymbShort1ss = 11'd6; 
   {3'd1, 4'd1 , 1'b0}: nBitPSymbShort1ss = 11'd12;
   {3'd1, 4'd2 , 1'b0}: nBitPSymbShort1ss = 11'd18;
   {3'd1, 4'd3 , 1'b1}: nBitPSymbShort1ss = 11'd12;
   {3'd1, 4'd3 , 1'b0}: nBitPSymbShort1ss = 11'd24;
   {3'd1, 4'd4 , 1'b1}: nBitPSymbShort1ss = 11'd18;
   {3'd1, 4'd4 , 1'b0}: nBitPSymbShort1ss = 11'd36;
   {3'd1, 4'd5 , 1'b0}: nBitPSymbShort1ss = 11'd48;
   {3'd1, 4'd6 , 1'b0}: nBitPSymbShort1ss = 11'd54;
   {3'd1, 4'd7 , 1'b0}: nBitPSymbShort1ss = 11'd60;
   {3'd1, 4'd8 , 1'b0}: nBitPSymbShort1ss = 11'd72;
   {3'd1, 4'd9 , 1'b0}: nBitPSymbShort1ss = 11'd80;
   {3'd1, 4'd10, 1'b0}: nBitPSymbShort1ss = 11'd90;
   {3'd1, 4'd11, 1'b0}: nBitPSymbShort1ss = 11'd100;
   // ru=106
   {3'd2, 4'd0 , 1'b1}: nBitPSymbShort1ss = 11'd6; 
   {3'd2, 4'd0 , 1'b0}: nBitPSymbShort1ss = 11'd12;
   {3'd2, 4'd1 , 1'b1}: nBitPSymbShort1ss = 11'd12;
   {3'd2, 4'd1 , 1'b0}: nBitPSymbShort1ss = 11'd24;
   {3'd2, 4'd2 , 1'b0}: nBitPSymbShort1ss = 11'd36;
   {3'd2, 4'd3 , 1'b1}: nBitPSymbShort1ss = 11'd24;
   {3'd2, 4'd3 , 1'b0}: nBitPSymbShort1ss = 11'd48;
   {3'd2, 4'd4 , 1'b1}: nBitPSymbShort1ss = 11'd36;
   {3'd2, 4'd4 , 1'b0}: nBitPSymbShort1ss = 11'd72;
   {3'd2, 4'd5 , 1'b0}: nBitPSymbShort1ss = 11'd96;
   {3'd2, 4'd6 , 1'b0}: nBitPSymbShort1ss = 11'd108;
   {3'd2, 4'd7 , 1'b0}: nBitPSymbShort1ss = 11'd120;
   {3'd2, 4'd8 , 1'b0}: nBitPSymbShort1ss = 11'd144;
   {3'd2, 4'd9 , 1'b0}: nBitPSymbShort1ss = 11'd160;
   {3'd2, 4'd10, 1'b0}: nBitPSymbShort1ss = 11'd180;
   {3'd2, 4'd11, 1'b0}: nBitPSymbShort1ss = 11'd200;
   // ru=242
   {3'd3, 4'd0 , 1'b1}: nBitPSymbShort1ss = 11'd15; 
   {3'd3, 4'd0 , 1'b0}: nBitPSymbShort1ss = 11'd30; 
   {3'd3, 4'd1 , 1'b1}: nBitPSymbShort1ss = 11'd30; 
   {3'd3, 4'd1 , 1'b0}: nBitPSymbShort1ss = 11'd60; 
   {3'd3, 4'd2 , 1'b0}: nBitPSymbShort1ss = 11'd90; 
   {3'd3, 4'd3 , 1'b1}: nBitPSymbShort1ss = 11'd60; 
   {3'd3, 4'd3 , 1'b0}: nBitPSymbShort1ss = 11'd120;
   {3'd3, 4'd4 , 1'b1}: nBitPSymbShort1ss = 11'd90; 
   {3'd3, 4'd4 , 1'b0}: nBitPSymbShort1ss = 11'd180;
   {3'd3, 4'd5 , 1'b0}: nBitPSymbShort1ss = 11'd240;
   {3'd3, 4'd6 , 1'b0}: nBitPSymbShort1ss = 11'd270;
   {3'd3, 4'd7 , 1'b0}: nBitPSymbShort1ss = 11'd300;
   {3'd3, 4'd8 , 1'b0}: nBitPSymbShort1ss = 11'd360;
   {3'd3, 4'd9 , 1'b0}: nBitPSymbShort1ss = 11'd400;
   {3'd3, 4'd10, 1'b0}: nBitPSymbShort1ss = 11'd450;
   {3'd3, 4'd11, 1'b0}: nBitPSymbShort1ss = 11'd500;
   // ru=484
   {3'd4, 4'd0 , 1'b1}: nBitPSymbShort1ss = 11'd30;
   {3'd4, 4'd0 , 1'b0}: nBitPSymbShort1ss = 11'd60;
   {3'd4, 4'd1 , 1'b1}: nBitPSymbShort1ss = 11'd60;
   {3'd4, 4'd1 , 1'b0}: nBitPSymbShort1ss = 11'd120;
   {3'd4, 4'd2 , 1'b0}: nBitPSymbShort1ss = 11'd180;
   {3'd4, 4'd3 , 1'b1}: nBitPSymbShort1ss = 11'd120;
   {3'd4, 4'd3 , 1'b0}: nBitPSymbShort1ss = 11'd240;
   {3'd4, 4'd4 , 1'b1}: nBitPSymbShort1ss = 11'd180;
   {3'd4, 4'd4 , 1'b0}: nBitPSymbShort1ss = 11'd360;
   {3'd4, 4'd5 , 1'b0}: nBitPSymbShort1ss = 11'd480;
   {3'd4, 4'd6 , 1'b0}: nBitPSymbShort1ss = 11'd540;
   {3'd4, 4'd7 , 1'b0}: nBitPSymbShort1ss = 11'd600;
   {3'd4, 4'd8 , 1'b0}: nBitPSymbShort1ss = 11'd720;
   {3'd4, 4'd9 , 1'b0}: nBitPSymbShort1ss = 11'd800;
   {3'd4, 4'd10, 1'b0}: nBitPSymbShort1ss = 11'd900;
   {3'd4, 4'd11, 1'b0}: nBitPSymbShort1ss = 11'd1000;
   // ru=996
   {3'd5, 4'd0 , 1'b1}: nBitPSymbShort1ss = 11'd60;
   {3'd5, 4'd0 , 1'b0}: nBitPSymbShort1ss = 11'd120;
   {3'd5, 4'd1 , 1'b1}: nBitPSymbShort1ss = 11'd120;
   {3'd5, 4'd1 , 1'b0}: nBitPSymbShort1ss = 11'd240;
   {3'd5, 4'd2 , 1'b0}: nBitPSymbShort1ss = 11'd360;
   {3'd5, 4'd3 , 1'b1}: nBitPSymbShort1ss = 11'd240;
   {3'd5, 4'd3 , 1'b0}: nBitPSymbShort1ss = 11'd480;
   {3'd5, 4'd4 , 1'b1}: nBitPSymbShort1ss = 11'd360;
   {3'd5, 4'd4 , 1'b0}: nBitPSymbShort1ss = 11'd720;
   {3'd5, 4'd5 , 1'b0}: nBitPSymbShort1ss = 11'd960;
   {3'd5, 4'd6 , 1'b0}: nBitPSymbShort1ss = 11'd1080;
   {3'd5, 4'd7 , 1'b0}: nBitPSymbShort1ss = 11'd1200;
   {3'd5, 4'd8 , 1'b0}: nBitPSymbShort1ss = 11'd1440;
   {3'd5, 4'd9 , 1'b0}: nBitPSymbShort1ss = 11'd1600;
   {3'd5, 4'd10, 1'b0}: nBitPSymbShort1ss = 11'd1800;
   {3'd5, 4'd11, 1'b0}: nBitPSymbShort1ss = 11'd2000;
   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default:             nBitPSymbShort1ss = 11'd1;
   // pragma coverage block = on
   endcase
end

// Set nBitPSymbShort with the short number of bits per symbol.
// NDBPS,SHORT = NSD,SHORT * Nss * NBPSCS * R
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    nBitPSymbShort <= 12'd0;
  else if (preambleType > 4'b100) // HE Frames
  begin
      if (nSS_comb==4'd2)
         nBitPSymbShort <= {nBitPSymbShort1ss,1'b0};
      else
         nBitPSymbShort <= {1'b0,nBitPSymbShort1ss};
  end
end


//  NSS : Number of Spacial Stream
//      +-------+-----+
//      |  MCS  | NSS |
//      +-------+-----+
//      |  0-7  |  1  |
//      +-------+-----+
//      |  8-15 |  2  |
//      +-------+-----+
//      | 16-20 |  3  |
//      +-------+-----+
//      | 21-23 |  3  |
//      +-------+-----+
//      | 23-27 |  4  |
//      +-------+-----+
//      | 28-31 |  4  |
//      +-------+-----+
//      |  32   |  1  |
//      +-------+-----+
//      | 33-38 |  2  |
//      +-------+-----+
//      | 39-52 |  3  |
//      +-------+-----+
//      | 53-76 |  4  |
//      +-------+-----+
always @*
begin
  if ((preambleType == 4'b0010) || (preambleType == 4'b0011))
  // In case of HT rate.
  begin
    case (mcsIndex[6:0])
      7'd0,7'd1,7'd2,7'd3,7'd4,7'd5,7'd6,7'd7          : nSS_comb = 4'd1;
      7'd8,7'd9,7'd10,7'd11,7'd12,7'd13,7'd14,7'd15    : nSS_comb = 4'd2;
      7'd16,7'd17,7'd18,7'd19,7'd20,7'd21,7'd22,7'd23  : nSS_comb = 4'd3;
      7'd24,7'd25,7'd26,7'd27,7'd28,7'd29,7'd30,7'd31  : nSS_comb = 4'd4;
      7'd32                                            : nSS_comb = 4'd1;
      7'd33,7'd34,7'd35,7'd36,7'd37,7'd38              : nSS_comb = 4'd2;
      7'd39,7'd40,7'd41,7'd42,7'd43,7'd44,7'd45,7'd46,
      7'd47,7'd48,7'd49,7'd50,7'd51,7'd52              : nSS_comb = 4'd3;
      7'd53,7'd54,7'd55,7'd56,7'd57,7'd58,7'd59,7'd60,
      7'd61,7'd62,7'd63,7'd64,7'd65,7'd66,7'd67,7'd68,
      7'd69,7'd70,7'd71,7'd72,7'd73,7'd74,7'd75,7'd76  : nSS_comb = 4'd4;
    // Disable coverage on the default state because it cannot be reached.
    // pragma coverage block = off 
      default                                          : nSS_comb = 4'd1;
    // pragma coverage block = on 
    endcase
  end
  else if(preambleType == 4'b0100 | // VHT
          preambleType == 4'b0101 | // HE-SU
          preambleType == 4'b0111 | // HE-ER-SU
          preambleType == 4'b0110 | // HE-MU
          preambleType == 4'b1000)  // HE-TB
    // VHT or HE Rate
    nSS_comb = {1'b0,mcsIndex[6:4]} + 4'd1; // nSS is set from 0 to 7 in mcsIndex[6:4] (where 0 means 1 nSS)
  else
    nSS_comb = 4'd1;
end

// 6*NES selection for nBit calculcation
always @*
begin
  if((preambleType == 4'b0010) || (preambleType == 4'b0011))
  // HT case
  begin
    nES = (nBitPSymb_comb < 15'd1296) ? 24'd6 : 24'd12;
  end
  else if(preambleType == 4'b0100)
  begin
    // 20MHz and  40MHz Bandwidth
    if(chBw < 2'd2)
    begin
      nES = (nBitPSymb_comb <= 15'd2160) ? 24'd6 : (nBitPSymb_comb <= 15'd4320) ? 24'd12 : 24'd18;
    end
    // 80MHz Bandwidth
    else if(chBw == 2'd2)
    begin
      if((nSS_comb == 4'd7) && (nBitPSymb_comb == 15'd8190))
      begin
        nES = 24'd36;
      end
      else if((nSS_comb == 4'd7) && (nBitPSymb_comb == 15'd2457))
      begin
        nES = 24'd18;
      end
      else
      begin
        nES = (nBitPSymb_comb <= 15'd2106) ? 24'd6 : (nBitPSymb_comb <= 15'd4212) ? 24'd12 : (nBitPSymb_comb <= 15'd6318) ? 24'd18 : (nBitPSymb_comb <= 15'd8424) ? 24'd24 : 24'd36;
      end
    end
    // 160MHz Bandwidth
    else if(chBw == 2'd3)
    begin
      if( ((nSS_comb == 4'd4) && (nBitPSymb_comb == 15'd9360)) || ((nSS_comb == 4'd7) && (nBitPSymb_comb == 15'd9828)) )
      begin
        nES = 24'd36;
      end
      else if((nBitPSymb_comb == 15'd14040) && ((nSS_comb == 4'd5) || (nSS_comb == 4'd6)) )
      begin
        nES = 24'd48;
      end
      else if((nSS_comb == 4'd7) && (nBitPSymb_comb == 15'd16380))
      begin
        nES = 24'd54;
      end
      else
      begin
        nES = (nBitPSymb_comb <= 15'd2106)  ? 24'd6  :
              (nBitPSymb_comb <= 15'd4212)  ? 24'd12 :
              (nBitPSymb_comb <= 15'd6318)  ? 24'd18 :
              (nBitPSymb_comb <= 15'd8424)  ? 24'd24 :
              (nBitPSymb_comb <= 15'd10530) ? 24'd30 :
              (nBitPSymb_comb <= 15'd12636) ? 24'd36 :
              (nBitPSymb_comb <= 15'd14742) ? 24'd42 :
              (nBitPSymb_comb <= 15'd16848) ? 24'd48 :
              (nBitPSymb_comb <= 15'd18720) ? 24'd54 :
                                              24'd72 ;
      end
    end
    else
    begin
      nES = 24'd6; // 6*1
    end
  end
  else
  begin
    nES = 24'd6; // 6*1
  end
end

// Set nBit with the total number of bit of the ppdu. 
// Note that for OFDM frame, 22 bits have to be added (16 for service field and 6 for tails)
assign nBit = ((mcsIndex_ff1 < 7'd4) && (preambleType[3:1] == 3'b000)) ? {length,4'b0} :
              (length==20'd0)                                          ? 24'd0         :
                                                                         {1'b0,length,3'b0}+24'd16+nES;


// DCM is not used in conjunction with STBC
always @ *
begin
  if (stbc > 2'd0)
    nBitPSymb  = {nBitPSymb_comb[13:0],1'd0};
  else if (dcm)
    nBitPSymb  = {1'd0,nBitPSymb_comb[14:1]};
  else  
    nBitPSymb  = nBitPSymb_comb;
end

// Delay mcsIndex of one clock cycle.
always @ (posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    mcsIndex_ff1  <= 7'b0;
  else
    mcsIndex_ff1  <= mcsIndex[6:0];
end


// Set data14Duration with the duration of a 14bytes PPDU based on the rate
// Set data14Duration with the duration of a 14bytes PPDU based on the rate
always @ *
begin
  // In case of legacy rate.
  if (isLegacy)
  begin
    case (mcsIndex_ff1[3:0])
             4'd0 : data14Duration = 16'd112; //  1Mbps
             4'd1 : data14Duration =  16'd56; //  2Mbps
             4'd2 : data14Duration =  16'd21; //5.5Mbps
             4'd3 : data14Duration =  16'd11; // 11Mbps
             4'd4 : data14Duration =  16'd24; //  6Mbps
             4'd5 : data14Duration =  16'd16; //  9Mbps
             4'd6 : data14Duration =  16'd12; // 12Mbps
             4'd7 : data14Duration =   16'd8; // 18Mbps
             4'd8 : data14Duration =   16'd8; // 24Mbps
             4'd9 : data14Duration =   16'd4; // 36Mbps
            4'd10 : data14Duration =   16'd4; // 48Mbps
            4'd11 : data14Duration =   16'd4; // 54Mbps
       // Disable coverage on the default state because it cannot be reached.
       // pragma coverage block = off 
       default : data14Duration = 16'd112;    
       // pragma coverage block = on 
    endcase                           
  end
  else if (preambleType <= 4'b0100)
  // In case of HT or VHT rate.
  begin
    // only one symbol since 14*8 + 22 = 134 bits
    if(nBitPSymb_comb >= 15'd134)
    begin
      // If STBC then the SYMBOL is transmitted twice
      // Short Guard interval does not matter since 3.6 or 7.2 is round up to 4 or 8
      data14Duration = (stbc != 2'd0) ? 16'd8 : 16'd4;
    end
    // If two symbols 134/2 = 67
    // HT  NDBPS(78 104 108)
    // VHT NDBPS(78 104 108 117)
    else if(nBitPSymb_comb >= 15'd67)
    begin
      // STBC does not matter since both symbol are send twice in parrallel
      // Short GI does not matter since 3.6*2 = 7.4 = 8
      data14Duration = 16'd8;
    end
    // If three symbols 134/3 = 44.6 = 45
    // HT  NDBPS(52 54)
    // VHT NDBPS(52 54)
    else if(nBitPSymb_comb >= 15'd45)
    begin
      // STBC add 1 more symbol
      // Short GI is taken in the calculation
      data14Duration = (stbc != 2'd0) ? 16'd16 : 16'd12;
    end
    // If four symbols 134/4 = 33.5 = 34 : Ignored since there is no nBitPSymb_comb between 34 and 45
    // If five symbols 134/5 = 26.8 = 27 : Ignored since there is no nBitPSymb_comb between 27 and 34
    // If six  symbols 134/6 = 22.3 = 23
    // HT  NDBPS(26 24)
    // VHT NDBPS(26)
    else if(nBitPSymb_comb > 15'd23)
    begin
      // STBC does not matter
      // Short GI is taken in the calculation
      data14Duration = 16'd24;
    end
    else
    begin
      data14Duration = 16'd4; // 1 symbol
    end
  end
  else
  // In case of HE rate
  begin
    data14Duration = {8'd0,fastHeLtfAndData} + {11'b0,packetExtensionDur};
  end
end

// Set data20Duration with the duration of a 20bytes PPDU based on the rate
always @ *
begin
  // In case of legacy rate.
  if (isLegacy)
  begin
    case (mcsIndex_ff1[3:0])
             4'd0 : data20Duration = 16'd160; //  1Mbps
             4'd1 : data20Duration =  16'd80; //  2Mbps
             4'd2 : data20Duration =  16'd30; //5.5Mbps
             4'd3 : data20Duration =  16'd15; // 11Mbps
             4'd4 : data20Duration =  16'd32; //  6Mbps
             4'd5 : data20Duration =  16'd24; //  9Mbps
             4'd6 : data20Duration =  16'd16; // 12Mbps
             4'd7 : data20Duration =  16'd12; // 18Mbps
             4'd8 : data20Duration =   16'd8; // 24Mbps
             4'd9 : data20Duration =   16'd8; // 36Mbps
            4'd10 : data20Duration =   16'd4; // 48Mbps
            4'd11 : data20Duration =   16'd4; // 54Mbps
       // Disable coverage on the default state because it cannot be reached.
       // pragma coverage block = off 
       default : data20Duration = 16'd128;    
       // pragma coverage block = on 
    endcase
  end
  // HT and VHT mode
  else if (preambleType <= 4'b0100)
  begin
    // Test if it has only 1 symbol (20*8 + 22) = 182
    if(nBitPSymb_comb >= 15'd182)
    begin
      // If STBC then the SYMBOL is transmitted twice
      // Short Guard interval does not matter since 3.6 or 7.2 is round up to 4 or 8
      data20Duration = (stbc != 2'd0) ? 16'd8 : 16'd4;
    end
    // If 2 symbols 182/2 = 91
    // HT  NDBPS(104 156 108 162)
    // VHT NDBPS(104 156 130 108 162 117)
    else if(nBitPSymb_comb >= 15'd91)  
    begin
      // STBC does not matter since both symbol are send twice in parrallel
      // Short GI does not matter since 3.6*2 = 7.4 = 8
      data20Duration = 16'd8;
    end
    // If 3 symbols 182/3 = 60.66 = 61
    // HT  NDBPS(78)
    // VHT NDBPS(78)
    else if(nBitPSymb_comb >= 15'd61)  
    begin
      // STBC add 1 more symbol
      // Short GI is taken in the calculation
      data20Duration = (stbc != 2'd0) ? 16'd16 : 16'd12;
    end
    // If 4 symbols 182/4 = 45.5 = 46
    // HT  NDBPS(52 54)
    // VHT NDBPS(52 54)
    else if(nBitPSymb_comb >= 15'd46)
    begin
      // STBC does not matter
      // Short GI is taken in the calculation
      data20Duration = 16'd16;
    end
    // If 5 symbols 182/5 = 36.4 = 37 : Ignored since there is no nBitPSymb_comb between 37 and 46
    // If 6 symbols 182/6 = 30.3 = 31 : Ignored since there is no nBitPSymb_comb between 31 and 37
    // If 7 symbols 182/7 = 26
    // HT  NDBPS(26)
    // VHT NDBPS(26)
    else if(nBitPSymb_comb >= 15'd26)
    begin
      // STBC add 1 more symbol
      // Short GI is taken in the calculation
      data20Duration = (stbc != 2'd0) ? 16'd32 : 16'd28;
    end
    // If 8 symbols 182/7 = 22.75 = 23
    // HT  NDBPS(24)
    else if(nBitPSymb_comb >= 15'd23)
    begin
      // STBC does not matter
      // Short GI is taken in the calculation
      data20Duration = 16'd32;
    end
    else
    begin
      data20Duration = 16'd4;  // 1 symbol
    end
  end
  else
  // In case of HE rate
  begin
    data20Duration = {8'd0,fastHeLtfAndData} + {11'b0,packetExtensionDur};
  end
end

// Table 22-6: For VHT modulated fields, NSTS,total is the total number of space-time streams in a
// PPDU.
assign nSTSTotal = priNSTS
`ifdef  RW_MUMIMO_TX_EN
                 + {1'b0,secNSTS}
`endif//RW_MUMIMO_TX_EN
                 ;

// Number of space-time streams used by the primary user.
// nSTS0 = 2*nSS when STBC is used
// nSTS0 = nSS   when STBC is not used
assign priNSTS = (stbc == 2'd1) ? {nSS_comb[2:0],1'b0} : nSS_comb;



// HE Fast computation parameters, for Length of 14/20 bytes
//   Fast computation is expected for HE-SU & HE-ER-SU (non-OFDMA)
//   1x1 20MHz / 1x1 40MHz / 1x1 80MHz supported
//      -> Data rate for non-OFDMA 20MHz/40MHz, Nss=1 are used.
always @*
begin
   if (length == 20'd14 & stbc == 2'b00)
      case (nBitPSymb)
      15'd25:  {fastHeNSym,fastHeAInit} = {4'd6, 2'd2}; // Nsym=6, ainit=2
      15'd51:  {fastHeNSym,fastHeAInit} = {4'd3, 2'd3}; // Nsym=3, ainit=3
      15'd58:  {fastHeNSym,fastHeAInit} = {4'd3, 2'd2}; // Nsym=3, ainit=2
      15'd117: {fastHeNSym,fastHeAInit} = {4'd2, 2'd1}; // Nsym=2, ainit=1
      15'd234: {fastHeNSym,fastHeAInit} = {4'd1, 2'd3}; // Nsym=1, ainit=3
      15'd245: {fastHeNSym,fastHeAInit} = {4'd1, 2'd3}; // Nsym=1, ainit=3
      15'd351: {fastHeNSym,fastHeAInit} = {4'd1, 2'd2}; // Nsym=1, ainit=2
      15'd468: {fastHeNSym,fastHeAInit} = {4'd1, 2'd2}; // Nsym=1, ainit=2
      15'd490: {fastHeNSym,fastHeAInit} = {4'd1, 2'd2}; // Nsym=1, ainit=2
      default: {fastHeNSym,fastHeAInit} = {4'd1, 2'd1}; // Nsym=1, ainit=1
      endcase
   else if (length == 20'd14 & stbc != 2'b00)
      case (nBitPSymb[14:1])
      14'd51:  {fastHeNSym,fastHeAInit} = {4'd4, 2'd2}; // Nsym=4, ainit=2
      14'd117: {fastHeNSym,fastHeAInit} = {4'd2, 2'd3}; // Nsym=2, ainit=3
      14'd234: {fastHeNSym,fastHeAInit} = {4'd2, 2'd2}; // Nsym=2, ainit=2
      default: {fastHeNSym,fastHeAInit} = {4'd2, 2'd1}; // Nsym=2, ainit=1
      endcase
   else if (length == 20'd20 & stbc == 2'b00)
      case (nBitPSymb)
      15'd25:  {fastHeNSym,fastHeAInit} = {4'd8, 2'd2}; // Nsym=8, ainit=2
      15'd51:  {fastHeNSym,fastHeAInit} = {4'd4, 2'd3}; // Nsym=4, ainit=3
      15'd58:  {fastHeNSym,fastHeAInit} = {4'd4, 2'd1}; // Nsym=4, ainit=1
      15'd117: {fastHeNSym,fastHeAInit} = {4'd2, 2'd3}; // Nsym=2, ainit=3
      15'd234: {fastHeNSym,fastHeAInit} = {4'd1, 2'd0}; // Nsym=1, ainit=4
      15'd245: {fastHeNSym,fastHeAInit} = {4'd1, 2'd0}; // Nsym=1, ainit=4
      15'd351: {fastHeNSym,fastHeAInit} = {4'd1, 2'd3}; // Nsym=1, ainit=3
      15'd468: {fastHeNSym,fastHeAInit} = {4'd1, 2'd2}; // Nsym=1, ainit=2
      15'd490: {fastHeNSym,fastHeAInit} = {4'd1, 2'd2}; // Nsym=1, ainit=2
      15'd702: {fastHeNSym,fastHeAInit} = {4'd1, 2'd2}; // Nsym=1, ainit=2
      default: {fastHeNSym,fastHeAInit} = {4'd1, 2'd1}; // Nsym=1, ainit=1
      endcase
   else if (length == 20'd20 & stbc != 2'b00)
      case (nBitPSymb[14:1])
      14'd51:  {fastHeNSym,fastHeAInit} = {4'd4, 2'd0}; // Nsym=4, ainit=4
      14'd117: {fastHeNSym,fastHeAInit} = {4'd2, 2'd0}; // Nsym=2, ainit=4
      14'd234: {fastHeNSym,fastHeAInit} = {4'd2, 2'd2}; // Nsym=2, ainit=2
      14'd351: {fastHeNSym,fastHeAInit} = {4'd2, 2'd2}; // Nsym=2, ainit=2
      default: {fastHeNSym,fastHeAInit} = {4'd2, 2'd1}; // Nsym=2, ainit=1
      endcase
   else // length=0
      {fastHeNSym,fastHeAInit} = {4'd0, 2'd0}; // Nsym=0, ainit=N/A
end


// Quick computation of number of HE LTF minus one.
// Number of HE LTF is 2 when Nss is 2 or when STBC is used. In others case, the
// Number of HE LTF is 1 
assign fastNumHeLtf = mcsIndex[6:4]!=3'b0 | stbc!=2'b0;


// Quick computation of Float part of the HE frame (The-ltf-sym + Nsym*Tsym)
//    Nma is always '0', the Maximum number of Data symbol is 4
always @*
begin
   if (woPreamble)
   begin
      case ({fastHeNSym,giType})
      {4'd1,2'd0}:      fastHeLtfAndData = 8'd14;  // Nsym=1, giType= 0.8 => 13.6
      {4'd1,2'd1}:      fastHeLtfAndData = 8'd15;  // Nsym=1, giType= 1.6 => 14.4
      {4'd1,2'd2}:      fastHeLtfAndData = 8'd16;  // Nsym=1, giType= 3.2 => 16.0
      {4'd2,2'd0}:      fastHeLtfAndData = 8'd28;  // Nsym=2, giType= 0.8 => 27.2
      {4'd2,2'd1}:      fastHeLtfAndData = 8'd29;  // Nsym=2, giType= 1.6 => 28.8
      {4'd2,2'd2}:      fastHeLtfAndData = 8'd32;  // Nsym=2, giType= 3.2 => 32.0
      {4'd3,2'd0}:      fastHeLtfAndData = 8'd41;  // Nsym=3, giType= 0.8 => 40.8
      {4'd3,2'd1}:      fastHeLtfAndData = 8'd44;  // Nsym=3, giType= 1.6 => 43.2
      {4'd3,2'd2}:      fastHeLtfAndData = 8'd48;  // Nsym=3, giType= 3.2 => 48.0
      {4'd4,2'd0}:      fastHeLtfAndData = 8'd55;  // Nsym=4, giType= 0.8 => 54.4
      {4'd4,2'd1}:      fastHeLtfAndData = 8'd58;  // Nsym=4, giType= 1.6 => 57.6
      {4'd4,2'd2}:      fastHeLtfAndData = 8'd64;  // Nsym=4, giType= 3.2 => 64.0
      {4'd6,2'd0}:      fastHeLtfAndData = 8'd82;  // Nsym=6, giType= 0.8 => 81.6
      {4'd6,2'd1}:      fastHeLtfAndData = 8'd87;  // Nsym=6, giType= 1.6 => 86.4
      {4'd6,2'd2}:      fastHeLtfAndData = 8'd96;  // Nsym=6, giType= 3.2 => 96.0
      {4'd8,2'd0}:      fastHeLtfAndData = 8'd109; // Nsym=8, giType= 0.8 =>108.8
      {4'd8,2'd1}:      fastHeLtfAndData = 8'd116; // Nsym=8, giType= 1.6 =>115.2
      {4'd8,2'd2}:      fastHeLtfAndData = 8'd128; // Nsym=8, giType= 3.2 =>128.0
      default:          fastHeLtfAndData = 8'd0;   // Nsym=0
      endcase
   end
   else // WithPreamble
   begin
      case ({fastHeNSym,fastNumHeLtf,giType,heLtfType})
      // Nsym=0
      {4'd0,1'b0,2'd0,2'd0}: fastHeLtfAndData = 8'd4;   // giType= 0.8, heLtfType= 3.2 => 4.0
      {4'd0,1'b0,2'd0,2'd1}: fastHeLtfAndData = 8'd8;   // giType= 0.8, heLtfType= 6.4 => 7.2
      {4'd0,1'b0,2'd0,2'd2}: fastHeLtfAndData = 8'd14;  // giType= 0.8, heLtfType=12.8 => 13.6
      {4'd0,1'b0,2'd1,2'd0}: fastHeLtfAndData = 8'd5;   // giType= 1.6, heLtfType= 3.2 => 4.8
      {4'd0,1'b0,2'd1,2'd1}: fastHeLtfAndData = 8'd8;   // giType= 1.6, heLtfType= 6.4 => 8.0
      {4'd0,1'b0,2'd2,2'd2}: fastHeLtfAndData = 8'd16;  // giType= 3.2, heLtfType=12.8 => 16.0
      {4'd0,1'b1,2'd0,2'd0}: fastHeLtfAndData = 8'd8;   // giType= 0.8, heLtfType= 3.2 => 8.0 
      {4'd0,1'b1,2'd0,2'd1}: fastHeLtfAndData = 8'd15;  // giType= 0.8, heLtfType= 6.4 => 14.4
      {4'd0,1'b1,2'd0,2'd2}: fastHeLtfAndData = 8'd28;  // giType= 0.8, heLtfType=12.8 => 27.2
      {4'd0,1'b1,2'd1,2'd0}: fastHeLtfAndData = 8'd10;  // giType= 1.6, heLtfType= 3.2 => 9.6 
      {4'd0,1'b1,2'd1,2'd1}: fastHeLtfAndData = 8'd16;  // giType= 1.6, heLtfType= 6.4 => 16.0
      {4'd0,1'b1,2'd2,2'd2}: fastHeLtfAndData = 8'd32;  // giType= 3.2, heLtfType=12.8 => 32.0
      // Nsym=1
      {4'd1,1'b0,2'd0,2'd0}: fastHeLtfAndData = 8'd18;  // giType= 0.8, heLtfType= 3.2 => 17.6
      {4'd1,1'b0,2'd0,2'd1}: fastHeLtfAndData = 8'd21;  // giType= 0.8, heLtfType= 6.4 => 20.8
      {4'd1,1'b0,2'd0,2'd2}: fastHeLtfAndData = 8'd28;  // giType= 0.8, heLtfType=12.8 => 27.2
      {4'd1,1'b0,2'd1,2'd0}: fastHeLtfAndData = 8'd20;  // giType= 1.6, heLtfType= 3.2 => 19.2
      {4'd1,1'b0,2'd1,2'd1}: fastHeLtfAndData = 8'd23;  // giType= 1.6, heLtfType= 6.4 => 22.4
      {4'd1,1'b0,2'd2,2'd2}: fastHeLtfAndData = 8'd32;  // giType= 3.2, heLtfType=12.8 => 32.0
      {4'd1,1'b1,2'd0,2'd0}: fastHeLtfAndData = 8'd22;  // giType= 0.8, heLtfType= 3.2 => 21.6
      {4'd1,1'b1,2'd0,2'd1}: fastHeLtfAndData = 8'd28;  // giType= 0.8, heLtfType= 6.4 => 28.0
      {4'd1,1'b1,2'd0,2'd2}: fastHeLtfAndData = 8'd41;  // giType= 0.8, heLtfType=12.8 => 40.8
      {4'd1,1'b1,2'd1,2'd0}: fastHeLtfAndData = 8'd24;  // giType= 1.6, heLtfType= 3.2 => 24.0
      {4'd1,1'b1,2'd1,2'd1}: fastHeLtfAndData = 8'd31;  // giType= 1.6, heLtfType= 6.4 => 30.4
      {4'd1,1'b1,2'd2,2'd2}: fastHeLtfAndData = 8'd48;  // giType= 3.2, heLtfType=12.8 => 48.0
      // Nsym=2
      {4'd2,1'b0,2'd0,2'd0}: fastHeLtfAndData = 8'd32;  // giType= 0.8, heLtfType= 3.2 => 31.2
      {4'd2,1'b0,2'd0,2'd1}: fastHeLtfAndData = 8'd35;  // giType= 0.8, heLtfType= 6.4 => 34.4
      {4'd2,1'b0,2'd0,2'd2}: fastHeLtfAndData = 8'd41;  // giType= 0.8, heLtfType=12.8 => 40.8
      {4'd2,1'b0,2'd1,2'd0}: fastHeLtfAndData = 8'd34;  // giType= 1.6, heLtfType= 3.2 => 33.6
      {4'd2,1'b0,2'd1,2'd1}: fastHeLtfAndData = 8'd37;  // giType= 1.6, heLtfType= 6.4 => 36.8
      {4'd2,1'b0,2'd2,2'd2}: fastHeLtfAndData = 8'd48;  // giType= 3.2, heLtfType=12.8 => 48.0
      {4'd2,1'b1,2'd0,2'd0}: fastHeLtfAndData = 8'd36;  // giType= 0.8, heLtfType= 3.2 => 35.2
      {4'd2,1'b1,2'd0,2'd1}: fastHeLtfAndData = 8'd42;  // giType= 0.8, heLtfType= 6.4 => 41.6
      {4'd2,1'b1,2'd0,2'd2}: fastHeLtfAndData = 8'd55;  // giType= 0.8, heLtfType=12.8 => 54.4
      {4'd2,1'b1,2'd1,2'd0}: fastHeLtfAndData = 8'd39;  // giType= 1.6, heLtfType= 3.2 => 38.4
      {4'd2,1'b1,2'd1,2'd1}: fastHeLtfAndData = 8'd45;  // giType= 1.6, heLtfType= 6.4 => 44.8
      {4'd2,1'b1,2'd2,2'd2}: fastHeLtfAndData = 8'd64;  // giType= 3.2, heLtfType=12.8 => 64.0
      // Nsym=3
      {4'd3,1'b0,2'd0,2'd0}: fastHeLtfAndData = 8'd45;  // giType= 0.8, heLtfType= 3.2 => 44.8
      {4'd3,1'b0,2'd0,2'd1}: fastHeLtfAndData = 8'd48;  // giType= 0.8, heLtfType= 6.4 => 48.0
      {4'd3,1'b0,2'd0,2'd2}: fastHeLtfAndData = 8'd55;  // giType= 0.8, heLtfType=12.8 => 54.4
      {4'd3,1'b0,2'd1,2'd0}: fastHeLtfAndData = 8'd48;  // giType= 1.6, heLtfType= 3.2 => 48.0
      {4'd3,1'b0,2'd1,2'd1}: fastHeLtfAndData = 8'd52;  // giType= 1.6, heLtfType= 6.4 => 51.2
      {4'd3,1'b0,2'd2,2'd2}: fastHeLtfAndData = 8'd64;  // giType= 3.2, heLtfType=12.8 => 64.0
      {4'd3,1'b1,2'd0,2'd0}: fastHeLtfAndData = 8'd49;  // giType= 0.8, heLtfType= 3.2 => 48.8
      {4'd3,1'b1,2'd0,2'd1}: fastHeLtfAndData = 8'd56;  // giType= 0.8, heLtfType= 6.4 => 55.2
      {4'd3,1'b1,2'd0,2'd2}: fastHeLtfAndData = 8'd68;  // giType= 0.8, heLtfType=12.8 => 68.0
      {4'd3,1'b1,2'd1,2'd0}: fastHeLtfAndData = 8'd53;  // giType= 1.6, heLtfType= 3.2 => 52.8
      {4'd3,1'b1,2'd1,2'd1}: fastHeLtfAndData = 8'd60;  // giType= 1.6, heLtfType= 6.4 => 59.2
      {4'd3,1'b1,2'd2,2'd2}: fastHeLtfAndData = 8'd80;  // giType= 3.2, heLtfType=12.8 => 80.0
      // Nsym=4
      {4'd4,1'b0,2'd0,2'd0}: fastHeLtfAndData = 8'd59;  // giType= 0.8, heLtfType= 3.2 => 58.4
      {4'd4,1'b0,2'd0,2'd1}: fastHeLtfAndData = 8'd62;  // giType= 0.8, heLtfType= 6.4 => 61.6
      {4'd4,1'b0,2'd0,2'd2}: fastHeLtfAndData = 8'd68;  // giType= 0.8, heLtfType=12.8 => 68.0
      {4'd4,1'b0,2'd1,2'd0}: fastHeLtfAndData = 8'd63;  // giType= 1.6, heLtfType= 3.2 => 62.4
      {4'd4,1'b0,2'd1,2'd1}: fastHeLtfAndData = 8'd66;  // giType= 1.6, heLtfType= 6.4 => 65.6
      {4'd4,1'b0,2'd2,2'd2}: fastHeLtfAndData = 8'd80;  // giType= 3.2, heLtfType=12.8 => 80.0
      {4'd4,1'b1,2'd0,2'd0}: fastHeLtfAndData = 8'd63;  // giType= 0.8, heLtfType= 3.2 => 62.4
      {4'd4,1'b1,2'd0,2'd1}: fastHeLtfAndData = 8'd69;  // giType= 0.8, heLtfType= 6.4 => 68.8
      {4'd4,1'b1,2'd0,2'd2}: fastHeLtfAndData = 8'd82;  // giType= 0.8, heLtfType=12.8 => 81.6
      {4'd4,1'b1,2'd1,2'd0}: fastHeLtfAndData = 8'd68;  // giType= 1.6, heLtfType= 3.2 => 67.2
      {4'd4,1'b1,2'd1,2'd1}: fastHeLtfAndData = 8'd74;  // giType= 1.6, heLtfType= 6.4 => 73.6
      {4'd4,1'b1,2'd2,2'd2}: fastHeLtfAndData = 8'd96;  // giType= 3.2, heLtfType=12.8 => 96.0
      // Nsym=6
      {4'd6,1'b0,2'd0,2'd0}: fastHeLtfAndData = 8'd86;  // giType= 0.8, heLtfType= 3.2 => 85.6
      {4'd6,1'b0,2'd0,2'd1}: fastHeLtfAndData = 8'd89;  // giType= 0.8, heLtfType= 6.4 => 88.8
      {4'd6,1'b0,2'd0,2'd2}: fastHeLtfAndData = 8'd96;  // giType= 0.8, heLtfType=12.8 => 95.2
      {4'd6,1'b0,2'd1,2'd0}: fastHeLtfAndData = 8'd92;  // giType= 1.6, heLtfType= 3.2 => 91.2
      {4'd6,1'b0,2'd1,2'd1}: fastHeLtfAndData = 8'd95;  // giType= 1.6, heLtfType= 6.4 => 94.4
      {4'd6,1'b0,2'd2,2'd2}: fastHeLtfAndData = 8'd112; // giType= 3.2, heLtfType=12.8 =>112.0
      {4'd6,1'b1,2'd0,2'd0}: fastHeLtfAndData = 8'd90;  // giType= 0.8, heLtfType= 3.2 => 89.6 
      {4'd6,1'b1,2'd0,2'd1}: fastHeLtfAndData = 8'd96;  // giType= 0.8, heLtfType= 6.4 => 96.0 
      {4'd6,1'b1,2'd0,2'd2}: fastHeLtfAndData = 8'd109; // giType= 0.8, heLtfType=12.8 =>108.8
      {4'd6,1'b1,2'd1,2'd0}: fastHeLtfAndData = 8'd96;  // giType= 1.6, heLtfType= 3.2 => 96.0 
      {4'd6,1'b1,2'd1,2'd1}: fastHeLtfAndData = 8'd103; // giType= 1.6, heLtfType= 6.4 =>102.4
      {4'd6,1'b1,2'd2,2'd2}: fastHeLtfAndData = 8'd128; // giType= 3.2, heLtfType=12.8 =>128.0  
      // Nsym=8
      {4'd8,1'b0,2'd0,2'd0}: fastHeLtfAndData = 8'd113; // giType= 0.8, heLtfType= 3.2 =>112.8
      {4'd8,1'b0,2'd0,2'd1}: fastHeLtfAndData = 8'd116; // giType= 0.8, heLtfType= 6.4 =>116.0
      {4'd8,1'b0,2'd0,2'd2}: fastHeLtfAndData = 8'd123; // giType= 0.8, heLtfType=12.8 =>122.4
      {4'd8,1'b0,2'd1,2'd0}: fastHeLtfAndData = 8'd120; // giType= 1.6, heLtfType= 3.2 =>120.0
      {4'd8,1'b0,2'd1,2'd1}: fastHeLtfAndData = 8'd124; // giType= 1.6, heLtfType= 6.4 =>123.2
      {4'd8,1'b0,2'd2,2'd2}: fastHeLtfAndData = 8'd144; // giType= 3.2, heLtfType=12.8 =>144.0
      {4'd8,1'b1,2'd0,2'd0}: fastHeLtfAndData = 8'd117; // giType= 0.8, heLtfType= 3.2 =>116.8
      {4'd8,1'b1,2'd0,2'd1}: fastHeLtfAndData = 8'd124; // giType= 0.8, heLtfType= 6.4 =>123.2
      {4'd8,1'b1,2'd0,2'd2}: fastHeLtfAndData = 8'd136; // giType= 0.8, heLtfType=12.8 =>136.0
      {4'd8,1'b1,2'd1,2'd0}: fastHeLtfAndData = 8'd125; // giType= 1.6, heLtfType= 3.2 =>124.8
      {4'd8,1'b1,2'd1,2'd1}: fastHeLtfAndData = 8'd132; // giType= 1.6, heLtfType= 6.4 =>131.2
      {4'd8,1'b1,2'd2,2'd2}: fastHeLtfAndData = 8'd160; // giType= 3.2, heLtfType=12.8 =>160.0
      // Disable coverage on the default state because it cannot be reached.
      // pragma coverage block = off 
      default:               fastHeLtfAndData = 8'd0;
      // pragma coverage block = on 
      endcase
   end
end


// Computation of packetExtensionDur (HE Frame only)
assign aInit = (quickDivEnable) ? fastHeAInit : slowHeAInit;

always @*
begin
   // In case of HE-TB NDP feedback: packetExtension is 0
   // In case of HE-TB frame with TRS trigger method: packetExtension[2:0] is the default PE
   // duration
   // In case of HE-SU or HE-ER-SU: packetExtension is the nominal packet padding

   if (isHETB & ~isHeTbTrs & length==20'd0)
      packetExtensionDur = 5'd0;
   else if (isHeTbTrs)
      packetExtensionDur = {packetExtension,2'b00};
   else
      case ({aInit,packetExtension})
      {2'd0, 3'd1} : packetExtensionDur = 5'd4;  // aInit=4, Nominal Packet Padding= 4us
      //
      {2'd3, 3'd2} : packetExtensionDur = 5'd4;  // aInit=3, Nominal Packet Padding= 8us
      {2'd0, 3'd2} : packetExtensionDur = 5'd8;  // aInit=4, Nominal Packet Padding= 8us
      //
      {2'd2, 3'd3} : packetExtensionDur = 5'd4;  // aInit=2, Nominal Packet Padding=12us
      {2'd3, 3'd3} : packetExtensionDur = 5'd8;  // aInit=3, Nominal Packet Padding=12us
      {2'd0, 3'd3} : packetExtensionDur = 5'd12; // aInit=4, Nominal Packet Padding=12us
      //
      {2'd1, 3'd4} : packetExtensionDur = 5'd4;  // aInit=1, Nominal Packet Padding=16us
      {2'd2, 3'd4} : packetExtensionDur = 5'd8;  // aInit=2, Nominal Packet Padding=16us
      {2'd3, 3'd4} : packetExtensionDur = 5'd12; // aInit=3, Nominal Packet Padding=16us
      {2'd0, 3'd4} : packetExtensionDur = 5'd16; // aInit=4, Nominal Packet Padding=16us
      //
      default:       packetExtensionDur = 5'd0;
      endcase
end


// PreFec Padding factor computation (aInit)
//               | 4
//   aInit = min |                                             or 4 if Nexcess=0
//               | ceil( nExcess / (mSTBC * Ndbps,short) )
always @(posedge macCoreClk or negedge macCoreClkHardRst_n) 
begin
  if (macCoreClkHardRst_n == 1'b0)  // Asynchronous Reset
    slowHeAInit <= 2'b0;
  else if (dividerControlFSMCs==COMPUTE_NSYMB & divDone_p)
  begin
     if (remainderOut==14'd0 | remainderOut>nBitPSymbShort3x)
        slowHeAInit <= 2'd0; // 4
     else if (remainderOut>nBitPSymbShort2x)
        slowHeAInit <= 2'd3;
     else if (remainderOut>nBitPSymbShort1x)
        slowHeAInit <= 2'd2;
     else
        slowHeAInit <= 2'd1;
  end
end
assign nBitPSymbShort3x = nBitPSymbShort2x + nBitPSymbShort1x;
assign nBitPSymbShort2x = {nBitPSymbShort1x[12:0],1'b0};
assign nBitPSymbShort1x = stbc!=2'd0 ? {1'b0,nBitPSymbShort,1'b0} :
                                       {2'b0,nBitPSymbShort}      ;


// Compute HE-LTF time (Preamble + Midamble)
//
// preamble HE-LTF =       Nhe-ltf * The-ltf-sym
// midamble HE-LTF = Nma * Nhe-ltf * The-ltf-sym
//
// The-ltf-sym = The-ltf + Tgi
//
// Possible value of Nhe-ltf: 1 / 2 / 4 / 6 / 8
// Possible value of The-ltf: 3.2us / 6.4us / 12.8us
// Possible value of Tgi    : 0.8us / 1.6us /  3.2us 
//
//           | 0
// Nma = max |
//           | ceil((Nsym-1)/Mma))-1
//
// HE-LTF time is computed in 0.8us unit.

// Compute Number of HE-LTF Field = (Nma+1) * Nhe-ltf
always @*
begin
   if (isHETB)
      nHeLtf = {1'b0,numHeLtf}+4'd1;
   else
      nHeLtf = vhtltf;

   if (woPreamble & (~doppler | computednSymb[5:0]==6'd0))
      nHeLtfField = 6'd0;
   else if (!woPreamble & (~doppler | computednSymb[5:0]==6'd0))
      nHeLtfField = 6'd1;
   else if (woPreamble & doppler)
      nHeLtfField = computednSymb[5:0]-6'd1;
   else//if (~woPreamble & doppler)
      nHeLtfField = computednSymb[5:0];

   nHeLtfField1x = {3'b0,nHeLtfField};      // nHeLtfField * 1 
   nHeLtfField2x = {2'b0,nHeLtfField,1'b0}; // nHeLtfField * 2
   nHeLtfField4x = {1'b0,nHeLtfField,2'b0}; // nHeLtfField * 4
   nHeLtfField8x = {     nHeLtfField,3'b0}; // nHeLtfField * 8

   case (nHeLtf)
   4'd1:    nHeLtfSymb = nHeLtfField1x;
   4'd2:    nHeLtfSymb = nHeLtfField2x;
   4'd4:    nHeLtfSymb = nHeLtfField4x;
   4'd6:    nHeLtfSymb = nHeLtfField4x + nHeLtfField2x;
   4'd8:    nHeLtfSymb = nHeLtfField8x;
   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default: nHeLtfSymb = nHeLtfField1x;
   // pragma coverage block = on 
   endcase
end

// Compute (Nma+1) * Nhe-ltf * The-Ltf
always @*
begin
   case (heLtfType)
   2'd0:    nHeLtfSymbTheLtfDot8 = {2'd0,nHeLtfSymb,2'd0}; // * 3.2  / 0.8
   2'd1:    nHeLtfSymbTheLtfDot8 = {1'd0,nHeLtfSymb,3'b0}; // * 6.4  / 0.8
   2'd2:    nHeLtfSymbTheLtfDot8 = {     nHeLtfSymb,4'b0}; // * 12.8 / 0.8
   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default: nHeLtfSymbTheLtfDot8 = {     nHeLtfSymb,4'b0}; // * 12.8 / 0.8
   // pragma coverage block = on 
   endcase
end

// Compute (Nma+1) * Nhe-ltf * Tgi
always @*
begin
   case (giType)
   2'd0:    nHeLtfSymbTgiDot8 = {2'd0,nHeLtfSymb};      // * 0.8 / 0.8
   2'd1:    nHeLtfSymbTgiDot8 = {1'd0,nHeLtfSymb,1'b0}; // * 1.6 / 0.8
   2'd2:    nHeLtfSymbTgiDot8 = {     nHeLtfSymb,2'b0}; // * 3.2 / 0.8
   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default: nHeLtfSymbTgiDot8 = {     nHeLtfSymb,2'b0}; // * 3.2 / 0.8
   // pragma coverage block = on 
   endcase
end

// Compute preMidHeLtf = (Nma+1) * Nhe-ltf * (The-Ltf+Tgi)
assign preMidHeLtfDot8 = nHeLtfSymbTheLtfDot8 + {2'd0,nHeLtfSymbTgiDot8};

// Number of Data Symbol in the HE frame
assign heTbLength = {1'b0,ppduHeTbLengthMC}+6'd1;
assign heNSymCapt = (isHeTbTrs) ? {3'd0,heTbLength} : computednSymbCapt[8:0];

// Compute Nsym * Tsym in 0.8us unit.
//
// Tsym = Tdft,he + Tgi
//
// Tdft,he = 12.8us
// Possible value of Tgi = 0.8us / 1.6us / 3.2us
always @*
begin
   // Nsym * Tdft,he
   nSymTDftHeDot8 = {heNSymCapt,4'b0};             // *12.8 / 0.8

   // Nsym * Tgi
   case (giType)
   2'd0:    nSymTgiDot8 = {4'b0,heNSymCapt[8:0]};      // * 0.8 / 0.8
   2'd1:    nSymTgiDot8 = {3'b0,heNSymCapt[8:0],1'b0}; // * 1.6 / 0.8
   2'd2:    nSymTgiDot8 = {2'b0,heNSymCapt[8:0],2'b0}; // * 3.2 / 0.8
   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default: nSymTgiDot8 = {2'b0,heNSymCapt[8:0],2'b0}; // * 3.2 / 0.8
   // pragma coverage block = on 
   endcase
end

// Nsym * Tsym
assign nSymTSymDot8 = nSymTDftHeDot8 + nSymTgiDot8;

// Compute HE-LTF + Data Symbol time.
assign heLtfAndDataSymbDot8 = preMidHeLtfDot8 + nSymTSymDot8;


// Debug Port
assign debugPortTxTimeCalculator = {
   initCnt[3:0],
   busy,
   swProcessing,
   quickDivStart_p,
   divDone_p,
   startDiv_p,
   timeOnAirValid,
   computeDuration,
   timeOnAirValidMC,
   startComputationMC_p,
   dividerControlFSMCs[2:0]};


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////



`ifdef RW_SIMU_ON
// Disable coverage on the default state because it cannot be reached.
// pragma coverage block = off 


// dividerControlFSM FSM states displayed in a string to easy simulation and debug
always @*
begin
   case (dividerControlFSMCs)
   IDLE           : dividerControlFSMCs_str = {"IDLE           "};
   COMPUTE_NSYMB  : dividerControlFSMCs_str = {"COMPUTE_NSYMB  "};
   COMPUTE_SHORTGI: dividerControlFSMCs_str = {"COMPUTE_SHORTGI"};
   COMPUTE_NMA    : dividerControlFSMCs_str = {"COMPUTE_NMA    "};
   COMPUTE_DOT8   : dividerControlFSMCs_str = {"COMPUTE_DOT8   "};
   default        : dividerControlFSMCs_str = {"XXX            "};
   endcase
end
// pragma coverage block = on

`endif // RW_SIMU_ON


// System Verilog Assertions
////////////////////////////
`ifdef RW_ASSERT_ON

// The number of Data HT-LTFs is denoted NDLTF. The number of extension HT-LTFs is denoted NELTF. The
// total number of HT-LTFs is:
// NLTF = NDLTF + NELTF       (20-22)
// shall not exceed 5. Table 20-11 shows the determination of the number of space time streams from the
// MCS and STBC fields in the HT-SIG. Table 20-12 shows the number of Data HT-LTFs as a function of the
// number of space time streams (NSTS). Table 20-13 shows the number of extension HT-LTFs as a function of
// the number of extension spatial streams (NESS).NSTS plus NESS is less than or equal to 4. In the case where
// NSTS equals 3, NESS cannot exceed one; if NESS equals one in this case then NLTF equals 5.

// assertion (nSS_comb+stbc+numExtnSS) <= 5 $todo



// $rw_sva Checks if the short preamble is not selection with legRate = 1Mbps
// property noShortPreambleWith1Mbps_prop;
// @(posedge macCoreClk)
//   disable iff(!macCoreClkHardRst_n)
//     !woPreambleMC && (mcsIndex == 0) && (busy) |-> (preambleType == 3'b001);
// endproperty
// noShortPreambleWith1Mbps: assert property (noShortPreambleWith1Mbps_prop); 

// $rw_sva Checks if the short Guard Interval is not selected with legacy rate
// property noShortGIWithLegRate_prop;
// @(posedge macCoreClk)
//   disable iff(!macCoreClkHardRst_n)
//     !woPreambleMC && (mcsIndex <128) && (busy) |-> (shortGI == 1'b0);
// endproperty
// noShortGIWithLegRate: assert property (noShortGIWithLegRate_prop); 

// $rw_sva Checks if the preamble type is compatible with legRate
// property noHTPreambleWithLegRate_prop;
// @(posedge macCoreClk)
//   disable iff(!macCoreClkHardRst_n)
//     !woPreambleMC && (mcsIndex <128) && (busy) |-> (preambleType < 2);
// endproperty
// noHTPreambleWithLegRate: assert property (noHTPreambleWithLegRate_prop); 

// $rw_sva Checks if the preamble type is compatible with HT-Rate
property noLegPreambleWithHTRate_prop;
@(posedge macCoreClk)
  disable iff(!macCoreClkHardRst_n)
    !woPreambleMC && (mcsIndex >=128) && (busy) |-> (preambleType >= 2);
endproperty
noLegPreambleWithHTRate: assert property (noLegPreambleWithHTRate_prop); 

`endif // RW_ASSERT_ON

endmodule

