//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Downstream Physical Channel AHBIF
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module downstream_ahbif
(
  /*****************************************************************************
  * System
  *****************************************************************************/
  input  wire                     clk,
  input  wire                     rst_n,

  /*****************************************************************************
  * control      
  *****************************************************************************/
  input  wire                     start,
  output reg                      done,

  /*****************************************************************************
  * Static parameters
  *****************************************************************************/
  input  wire              [31:0] dst_addr,
  input  wire              [11:0] byte_length,

  /*****************************************************************************
  * AHB interface
  *****************************************************************************/
  input  wire                     hready,
  output reg               [31:0] haddr,
  output reg                [1:0] htrans,
  output reg                [1:0] hsize,
  output reg               [31:0] hwdata,
  input  wire               [1:0] hresp,

  /*****************************************************************************
  * aligner interface
  *****************************************************************************/
  input  wire              [63:0] datai,
  input  wire                     datai_en,
  output wire                     datai_rdy
);


  /*****************************************************************************
  * declarations
  *****************************************************************************/
  localparam  S_IDLE   =3'b000,
              S_W_PAUSE=3'b001,
              S_W_A    =3'b010,
              S_W_AD   =3'b011,
              S_W_D    =3'b100;
  

  reg   [2:0] state;
  reg  [11:0] count;
  
  reg   [2:0] byte_nbr;
  wire [11:0] next_count;
  wire [31:0] next_haddr;
  wire [31:0] next_hwdata;
  
  reg  [63:0] wbuffer_data;
  reg         wbuffer_empty;
  wire        wbuffer_almost_empty;


  /*****************************************************************************
  * assignments
  *****************************************************************************/
  always @*
  begin
    case (hsize)
    2'b00:   byte_nbr = 3'd1;
    2'b01:   byte_nbr = 3'd2;
    default: byte_nbr = 3'd4;
    endcase
  end

  assign next_count  = count - { 9'd0,byte_nbr};

  assign next_haddr  = haddr + {29'd0,byte_nbr};

  assign next_hwdata = (haddr[2]) ? wbuffer_data[63:32] : wbuffer_data[31:0];
  

  /*****************************************************************************
  * FSM
  *****************************************************************************/
  always @(posedge clk,negedge rst_n)
  begin
    if(rst_n==1'b0)
    begin
      haddr       <= 32'b0;
      htrans      <= 2'b00;
      hsize       <= 2'b00;
      hwdata      <= 32'h0;
      
      count       <= 12'd0;
      done        <= 1'b0;
      
      state       <= S_IDLE;
    end
    else
    begin
    
      case(state)
      /*************************************************************************
      * IDLE
      *************************************************************************/
      default:
      begin
      
        if(~done & start)
        begin
          /* display first bus address */
          haddr    <= dst_addr[31:0];
          htrans   <= 2'b00;

          if (dst_addr[0] || byte_length<12'd2)
            hsize  <= 2'b00;
          else if (dst_addr[1] || byte_length<12'd4)
            hsize  <= 2'b01;
          else
            hsize  <= 2'b10;

          count    <= byte_length;
          
          state    <= S_W_PAUSE;
        end
        else if(done & ~start)
        begin
          done <= 1'b0;
        end
      
      end
      
      /*************************************************************************
      * WRITE PAUSE
      *************************************************************************/
      S_W_PAUSE:
      begin
        if(datai_en && datai_rdy)
        begin
          htrans <= 2'b10;
          state  <= S_W_A;
        end
      end
      
      /*************************************************************************
      * WRITE A
      *************************************************************************/
      S_W_A:
      begin
      
        if(hready)
        begin
          count     <= next_count;
          haddr     <= next_haddr;
          hwdata    <= next_hwdata;

          if (next_haddr[0] || next_count<12'd2)
            hsize <= 2'b00;
          else if (next_haddr[1] || next_count<12'd4)
            hsize <= 2'b01;
          else
            hsize <= 2'b10;
          
          if(next_count==12'd0)
          begin
            htrans <= 2'b00;
            state  <= S_W_D;
          end
          else
          begin
            if(datai_en && datai_rdy || !wbuffer_almost_empty)
            begin
              htrans <= 2'b10;
              state  <= S_W_AD;
            end  
            else
            begin
              htrans <= 2'b00;
              state  <= S_W_D;
            end
          end
        end
      end
      
      /*************************************************************************
      * WRITE AD
      *************************************************************************/
      S_W_AD:
      begin
      
        if(hready)
        begin
          count  <= next_count;
          haddr  <= next_haddr;
          hwdata <= next_hwdata;

          if (next_haddr[0] || next_count<12'd2)
            hsize <= 2'b00;
          else if (next_haddr[1] || next_count<12'd4)
            hsize <= 2'b01;
          else
            hsize <= 2'b10;
          
          if(next_count==12'd0)
          begin
            htrans    <= 2'b00;
            state     <= S_W_D;
          end
          else
          begin
            if(datai_en && datai_rdy || !wbuffer_almost_empty)
            begin
              htrans  <= 2'b10;
              state   <= S_W_AD;
            end
            else
            begin
              htrans  <= 2'b00;
              state   <= S_W_D;
            end
          end
        end
      end
      
      /*************************************************************************
      * WRITE D
      *************************************************************************/
      S_W_D:
      begin
        if(hready)
        begin
        
          if(count==12'd0)
          begin
            done     <= 1'b1;
            state    <= S_IDLE;
          end
          else
          begin
            if(datai_en && datai_rdy || !wbuffer_empty)
            begin
              htrans <= 2'b10;
              state  <= S_W_A;
            end
            else
            begin
              state  <= S_W_PAUSE;
            end
          end
        end
      end
      endcase
    end
  end
  
    
  /*****************************************************************************
  * write buffer
  *****************************************************************************/
  assign datai_rdy = (state!=S_IDLE)    &&    // hold on if bus if not started yet
                     (wbuffer_empty || wbuffer_almost_empty);

  assign wbuffer_almost_empty = (hready==1'b1 && (state==S_W_A || state==S_W_AD) &&
                                ((hsize==2'b10 && haddr[2:0]==3'b100) || 
                                 (hsize==2'b01 && haddr[2:0]==3'b110) ||
                                 (hsize==2'b00 && haddr[2:0]==3'b111)));
  
  always @(posedge clk, negedge rst_n)
  begin
    if(rst_n==1'b0)
    begin
      wbuffer_data  <= 64'b0;
      wbuffer_empty <=  1'b0;
    end
    else
    begin
      if(start==1'b0)
      begin
        wbuffer_empty <= 1'b1;
      end
      else
      begin
        if(datai_en==1'b1 && datai_rdy==1'b1)
        begin
          wbuffer_data  <= datai;
          wbuffer_empty <= 1'b0;
        end
        else if(wbuffer_almost_empty==1'b1)
          wbuffer_empty <= 1'b1;
      end
    end
  end


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// pragma coverage block = off 
  reg [16*8-1:0] state_str;
  
  always @(*)
  begin
    case (state)
    S_IDLE:    state_str = "S_IDLE";
    S_W_PAUSE: state_str = "S_W_PAUSE";       
    S_W_A:     state_str = "S_W_A";   
    S_W_AD:    state_str = "S_W_AD";   
    S_W_D:     state_str = "S_W_D";   
    default:   state_str = "XXX";
    endcase    
  end
// pragma coverage block = on 
`endif


endmodule
`default_nettype wire
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////



