//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Downstream AXI Interface
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module downstream_axi
(
  //****************************************************************************
  //$port_g clock and reset
  //****************************************************************************
  input wire                      clk,                     // clock
  input wire                      rst_n,                   // asynchronous reset

  //****************************************************************************
  //$port_g Configuration from register
  //****************************************************************************
  input  wire               [3:0] axi_burst_length_limit,  // AXI Burst Length limitation

  //****************************************************************************
  //$port_g Channel Control
  //****************************************************************************
  input  wire                     channel_sel,             // request selection
  input  wire                     channel_req,             // request
  input  wire              [31:0] channel_saddr,           // request source address
  input  wire              [31:0] channel_daddr,           // request destination address
  input  wire               [3:0] channel_tag,             // request tag
  input  wire              [15:0] channel_length,          // request length (in bytes)
  output wire                     channel_busy,            // request is busy
  output reg                      channel_ack,             // acknowledge
  output reg                [3:0] channel_ack_tag,         // acknowledge tag
  output reg                      channel_ack_error,       // acknowledge error
  input  wire                     channel_ack_ack,         // acknowledge acknowledge

  //****************************************************************************
  //$port_g Downstream Control
  //****************************************************************************
  output reg                      start,                   // Start transfert
  output wire               [2:0] saddr,                   // transfer source address offset
  output wire              [31:0] daddr,                   // transfer destination address
  output reg               [11:0] length,                  // transfer length (in bytes)
  input  wire                     done,                    // transfert is done

  //****************************************************************************
  //$port_g AXI interface
  //****************************************************************************
  output reg               [ 3:0] arid,                    // Read address ID
  output reg               [31:0] araddr,                  // Read address
  output reg               [ 7:0] arlen,                   // Burst Length
  output wire              [ 2:0] arsize,                  // Burst size
  output wire              [ 1:0] arburst,                 // Burst type
  output reg                      arvalid,                 // Read address valid
  input  wire                     arready,                 // Read address ready

  input  wire              [ 3:0] rid,                     // Read data ID
  input  wire              [63:0] rdata,                   // Read data
  input  wire              [ 1:0] rresp,                   // Read response
  input  wire                     rlast,                   // Read last
  input  wire                     rvalid,                  // Read valid
  output wire                     rready,                  // Read ready

  //****************************************************************************
  //$port_g Aligner interface
  //****************************************************************************
  output reg               [63:0] datao,                   // Data
  output reg                      datao_last,              // Last Data
  output reg                      datao_en,                // Data valid
  input  wire                     datao_rdy,               // Data ready

  //****************************************************************************
  //$port_g Debug interface
  //****************************************************************************
  output wire              [15:0] diag_port_downstream_axi0, // Diag Port
  output wire              [15:0] diag_port_downstream_axi1, // Diag Port
  output wire              [15:0] diag_port_downstream_axi2  // Diag Port
);


  /*****************************************************************************
  * parameters  
  *****************************************************************************/
  // Read Address FSM states definition
  //$fsm_sd ar_state
  localparam
             AR_IDLE    = 2'd0, 
             AR_REQUEST = 2'd1;

  // Read Data FSM states definition
  //$fsm_sd r_state
  localparam
             R_IDLE     = 3'd0, 
             R_INIT     = 3'd1,
             R_DATA     = 3'd2,
             R_DONE     = 3'd3,
             R_SAVE     = 3'd4,
             R_ACK      = 3'd5,
             R_ERROR    = 3'd6;


  /*****************************************************************************
  * declarations
  *****************************************************************************/
  reg    [1:0] ar_state;              // Read Address FSM
  reg    [2:0] r_state;               // Read Data FSM

  // Context registers
  //  Save the 16 DMA channel context
  reg   [31:0] context_daddr[15:0];   // Destination address
  reg   [10:0] context_saddr[15:0];   // Source address
  reg   [15:0] context_remain[15:0];  // Data bytes remains per channel
  reg          context_error[15:0];   // Error flag
  reg    [3:0] context_tag;           // Pointer to context table
  wire  [15:0] next_context_remain;
  wire  [31:0] next_context_daddr;
  wire         next_context_error;

  wire   [2:0] qwords_reliquat;
  wire  [16:0] nbr_qwordsx8;          // Number of qwords in channel request
  wire   [8:0] qwords_to_boundary;    // Data qwords from source address to burst boundary

  reg   [31:0] next_araddr;
  reg    [7:0] next_arlen;

  reg   [13:0] ar_remain;             // Data qwords remains in channel request
  reg   [13:0] next_ar_remain;        // Next value of ar_remain

  reg   [11:0] length_to_boundary;    // Data Bytes from destination address to burst boundary

  reg    [8:0] boundary;              // boundary in qwords
  reg    [8:0] boundary_mask;         // boundary mask

  reg          error;                 // AXI response error flag
  

  /*****************************************************************************
  * assignment
  *****************************************************************************/
  assign channel_busy = ar_state!=AR_IDLE;


  /*****************************************************************************
  * computes the number of qwords involved in the DMA transfer
  *****************************************************************************/
  assign qwords_reliquat = channel_length[2:0] + channel_saddr[2:0];
  assign nbr_qwordsx8    = { 1'b0,channel_length[15:0]} +
                           {14'b0,channel_saddr[2:0]}   +
                           {13'b0,|qwords_reliquat,3'b0};

  /*****************************************************************************
  * computes burst boundary signals
  *****************************************************************************/
  always @*
  begin
     case (axi_burst_length_limit)
     4'd0:    boundary = 9'd1<<0;
     4'd1:    boundary = 9'd1<<1;
     4'd2:    boundary = 9'd1<<2;
     4'd3:    boundary = 9'd1<<3;
     4'd4:    boundary = 9'd1<<4;
     4'd5:    boundary = 9'd1<<5;
     4'd6:    boundary = 9'd1<<6;
     4'd7:    boundary = 9'd1<<7;
     default: boundary = 9'd1<<8;
     endcase

     boundary_mask = boundary-9'd1;
  end

  //////////////////////////////////////////////////////////////////////////////
  //
  //  Request
  //
  //////////////////////////////////////////////////////////////////////////////
  always @(posedge clk, negedge rst_n)
  begin
    if (rst_n==1'b0)
    begin
      arid          <= 4'h0;
      araddr        <= 32'h0;
      arlen         <= 8'h0;
      arvalid       <= 1'b0;
      ar_remain     <= 14'h0;
      ar_state      <= AR_IDLE;
    end
    else
    begin
      case (ar_state)
      /**************************************************************************
      * IDLE
      **************************************************************************/
      AR_IDLE:
      //$fsm_s In AR_IDLE state, the state machine waits until request is received
      begin
        if (channel_sel==1'b1 && channel_req==1'b1)
        begin
          // First Request
          arid           <= channel_tag;
          araddr         <= next_araddr;
          arlen          <= next_arlen;
          arvalid        <= 1'b1;
          ar_remain      <= next_ar_remain;
          //$fsm_t When request is received, the state machine goes to AR_REQUEST state.
          ar_state       <= AR_REQUEST;
        end
      end
      /**************************************************************************
      * REQUEST
      **************************************************************************/
      default: //AR_REQUEST:
      //$fsm_s In AR_REQUEST state, the state machine waits until last request is done
      begin
        if (arvalid==1'b1 && arready==1'b1)
        begin
          if (next_ar_remain==14'd0)
          begin
            //$fsm_t When last request is done, the state machine goes to AR_IDLE state.
            arvalid     <= 1'b0;
            ar_state    <= AR_IDLE;
          end
          else
          begin
            ar_remain    <= next_ar_remain;
            araddr       <= next_araddr;
            arlen        <= next_arlen;
          end
        end
      end
      endcase
    end
  end

  /*****************************************************************************
  * next AXI request
  *****************************************************************************/
  assign qwords_to_boundary = boundary - ({1'b0,next_araddr[10:3]}&boundary_mask);

  always @*
  begin
    if (ar_state==AR_IDLE)
    begin
      next_araddr       = {channel_saddr[31:3],3'b000};
      next_ar_remain    = nbr_qwordsx8[16:3];
    end
    else
    begin
      next_araddr[31:3] = araddr[31:3]+{21'h0,arlen}+{29'h1};
      next_araddr[2:0]  = 3'b0;
      next_ar_remain    = ar_remain-{6'h0,arlen}-14'h1;
    end
      
    if (next_ar_remain>{5'b0,qwords_to_boundary})
      next_arlen        = qwords_to_boundary[7:0]-8'h1;
    else
      next_arlen        = next_ar_remain[7:0]-8'h1;
  end

  /*****************************************************************************
  * Fixed AXI Value
  *****************************************************************************/
  assign arsize  = 3'b011;
  assign arburst = 2'b01;


  //////////////////////////////////////////////////////////////////////////////
  //
  //  Data
  //
  //////////////////////////////////////////////////////////////////////////////
  always @(posedge clk, negedge rst_n)
  begin
    if (rst_n==1'b0)
    begin
      start              <= 1'b0;
      error              <= 1'b0;

      datao              <= 64'h0;
      datao_en           <= 1'b0;
      datao_last         <= 1'b0;

      channel_ack        <= 1'b0;
      channel_ack_tag    <= 4'h0;
      channel_ack_error  <= 1'b0;

      context_tag        <= 4'h0;

      r_state            <= R_IDLE;
    end
    else
    begin
      case (r_state)
      /**************************************************************************
      * IDLE
      **************************************************************************/
      default://R_IDLE:
      //$fsm_s In R_IDLE state, the state machine waits until data is ready
      begin
        start              <= 1'b0;
        error              <= 1'b0;
    
        datao              <= 64'h0;
        datao_en           <= 1'b0;
        datao_last         <= 1'b0;
    
        channel_ack        <= 1'b0;
        channel_ack_tag    <= 4'h0;
        channel_ack_error  <= 1'b0;
    
        if (rvalid==1'b1)
        begin
           //$fsm_t When data is ready, the state machine goes to R_INIT state.
           r_state         <= R_INIT;
        end
      end
      /**************************************************************************
      * INIT
      **************************************************************************/
      R_INIT:
      //$fsm_s In R_INIT state, the state machine waits until first data is received
      begin
        if (rvalid==1'b1 && rready==1'b1)
        begin
          context_tag     <= rid;
          datao           <= rdata;
          datao_last      <= rlast;
          datao_en        <= 1'b1;

          start           <= 1'b1;

          if (rresp!=0)
            error         <= 1'b1;

          if (rlast==1'b1)
            //$fsm_t When first&last data is received, the state machine goes to R_DONE state.
            r_state       <= R_DONE;
          else
            //$fsm_t When first data is received, the state machine goes to R_DATA state.
            r_state       <= R_DATA;
        end
      end
      /**************************************************************************
      * DATA
      **************************************************************************/
      R_DATA:
      //$fsm_s In R_DATA state, the state machine waits until last data is recevied
      begin
        if (rvalid==1'b1 && rready==1'b1)
        begin
          datao      <= rdata;
          datao_last <= rlast;
          datao_en   <= 1'b1;

          if (rresp!=0)
            error    <= 1'b1;
        end
        else if (datao_rdy==1'b1)
        begin
          datao_en   <= 1'b0;
        end
    
        if (done==1'b1 || context_remain[context_tag]==16'h0)
          //$fsm_t When ahbif/busif is early done, the state machine goes to R_ERROR state.
          //$fsm_t When no more data is expected for the current tag, the state machine goes to R_ERROR state.
          r_state    <= R_ERROR;
        else if (rvalid==1'b1 && rready==1'b1 && rid!=context_tag)
          //$fsm_t When AXI wrong ID is received, the state machine goes to R_ERROR state.
          r_state    <= R_ERROR;
        else if (rvalid==1'b1 && rready==1'b1 && rlast==1'b1)
          //$fsm_t When last data is received, the state machine goes to R_DONE state.
          r_state    <= R_DONE;
      end
      /**************************************************************************
      * DONE
      **************************************************************************/
      R_DONE:
      //$fsm_s In R_DONE state, the state machine waits until ahbif/busif transfert is done
      begin
        if (datao_rdy==1'b1)
          datao_en   <= 1'b0;
    
        if (done==1'b1)
        begin
          //$fsm_t When ahbif/busif transfert is done, the state machine goes to R_SAVE state.
          start      <= 1'b0;
          r_state    <= R_SAVE;
        end
      end
      /**************************************************************************
      * SAVE
      **************************************************************************/
      R_SAVE:
      //$fsm_s In R_SAVE state, 
      begin
        // Clear error flag for next data exchange
        error <= 1'b0;

        if (next_context_remain==16'd0)
          //$fsm_t When no more data is expected for the current tag, the state machine goes to R_ACK state.
          r_state    <= R_ACK;
        else if (rvalid==1'b1)
          //$fsm_t When new data is ready, the state machine goes to R_INIT state.
          r_state    <= R_INIT;
        else
          //$fsm_t When no data is ready, the state machine goes to R_IDLE state.
          r_state    <= R_IDLE;
      end
      /**************************************************************************
      * ACK
      **************************************************************************/
      R_ACK:
      //$fsm_s In R_ACK state, the state machine waits until acknowledge is received
      begin
        channel_ack        <= 1'b1;
        channel_ack_tag    <= context_tag;
        channel_ack_error  <= context_error[context_tag];
    
        if (channel_ack_ack==1'b1)
        begin
          channel_ack        <= 1'b0;
    
          if (rvalid==1'b1)
            //$fsm_t When acknowledge is received and new data is ready, the state machine goes to R_INIT state.
            r_state          <= R_INIT;
          else
            //$fsm_t When acknowledge is received and no data is ready, the state machine goes to R_IDLE state.
            r_state          <= R_IDLE;
        end
      end
      /**************************************************************************
      * ERROR
      **************************************************************************/
      R_ERROR:
      //$fsm_s In R_ERROR state, the state machine waits until acknowledge is received
      begin
        datao_en           <= 1'b0;
        datao_last         <= 1'b0;

        start              <= 1'b0;
        channel_ack        <= 1'b1;
        channel_ack_tag    <= context_tag;
        channel_ack_error  <= 1'b1;
    
        if (channel_ack_ack==1'b1)
        begin
          //$fsm_t When acknowledge is received, the state machine goes to R_IDLE state.
          channel_ack      <= 1'b0;
          r_state          <= R_IDLE;
        end
    
      end
      endcase
    end
  end

  assign rready = ((datao_rdy==1'b1 && r_state==R_DATA) || r_state==R_INIT) ? 1'b1 : 1'b0;


  //////////////////////////////////////////////////////////////////////////////
  //
  //  Context
  //
  //////////////////////////////////////////////////////////////////////////////
  always @(posedge clk, negedge rst_n)
  begin: b_context
    integer v_i;

    if (rst_n==1'b0)
    begin
      for (v_i=0;v_i<16;v_i=v_i+1)
      begin
        context_daddr[v_i]  <= 32'h0;
        context_saddr[v_i]  <= 11'h0;
        context_remain[v_i] <= 16'h0;
        context_error[v_i]  <= 16'h0;
      end
    end
    else
    begin
      // Context Init
      if (channel_sel==1'b1 && channel_req==1'b1)
      begin
        context_daddr[channel_tag]  <= channel_daddr;
        context_saddr[channel_tag]  <= channel_saddr[10:0];
        context_remain[channel_tag] <= channel_length;
        context_error[channel_tag]  <= 1'b0;
      end
      // Context Update
      if (r_state==R_SAVE)
      begin
        context_daddr[context_tag]  <= next_context_daddr;
        context_saddr[context_tag]  <= 11'h0;
        context_remain[context_tag] <= next_context_remain;
        context_error[context_tag]  <= next_context_error;
      end
      // Context Clear
      if (r_state==R_ERROR)
      begin
        context_remain[context_tag] <= 16'h0;
        context_error[context_tag]  <= 1'b1;
      end
    end
  end

  assign next_context_daddr   = context_daddr[context_tag] +{20'h0,length};
  assign next_context_remain  = context_remain[context_tag]-{ 4'h0,length};
  assign next_context_error   = context_error[context_tag] | error;


  /*****************************************************************************
  * Downstream Control
  *****************************************************************************/
  assign saddr   = context_saddr[context_tag][2:0];
  assign daddr   = context_daddr[context_tag];

  always @*
  begin
    length_to_boundary = {boundary,3'b0} - ({1'b0,context_saddr[context_tag]}&{boundary_mask,3'b111});

    if (context_remain[context_tag]>{4'h0,length_to_boundary})
      length = length_to_boundary;
    else
      length = context_remain[context_tag][11:0];
  end


  //////////////////////////////////////////////////////////////////////////////
  //
  //  Debug Port
  //
  //////////////////////////////////////////////////////////////////////////////
  assign diag_port_downstream_axi2 = {
     1'b0,
     arid,   
     arvalid,
     arready,
     rid,    
     rresp,  
     rlast,  
     rvalid, 
     rready
  }; 
  assign diag_port_downstream_axi1 = context_remain[context_tag];
  assign diag_port_downstream_axi0 = {
     error,
     channel_tag,
     channel_sel,
     channel_req,
     context_tag,
     ar_state,
     r_state
  };

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// pragma coverage block = off 
  reg [16*8-1:0] ar_state_str;
  reg [16*8-1:0] r_state_str;
  
  always @(*)
  begin
    case (ar_state)
    AR_IDLE:    ar_state_str = "AR_IDLE";
    AR_REQUEST: ar_state_str = "AR_REQUEST";
    default:    ar_state_str = "XXX";
    endcase    
  end
  
  always @(*)
  begin
    case (r_state)
    R_IDLE:  r_state_str = "R_IDLE";
    R_INIT:  r_state_str = "R_INIT";
    R_DATA:  r_state_str = "R_DATA";
    R_DONE:  r_state_str = "R_DONE";
    R_SAVE:  r_state_str = "R_SAVE";
    R_ACK:   r_state_str = "R_ACK";
    R_ERROR: r_state_str = "R_ERROR";
    default: r_state_str = "XXX";
    endcase    
  end
// pragma coverage block = on 
`endif

endmodule
`default_nettype wire
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
