//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Midstream Physical Channel AHB Interface
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module midstream_ahbif
(
  /*****************************************************************************
  * System
  *****************************************************************************/
  input  wire                     clk,
  input  wire                     rst_n,

  /*****************************************************************************
  * control      
  *****************************************************************************/
  input  wire                     start,
  output reg                      done,

  /*****************************************************************************
  * Static parameters
  *****************************************************************************/
  input  wire                     write,
  input  wire              [31:0] src_addr,
  input  wire              [31:0] dst_addr,
  input  wire              [15:0] byte_length,

  /*****************************************************************************
  * AHB interface
  *****************************************************************************/
  input  wire                     hready,
  output reg               [31:0] haddr,
  output reg                [1:0] htrans,
  output reg                      hwrite,
  output reg                [1:0] hsize,
  input  wire              [31:0] hrdata,
  output reg               [31:0] hwdata,
  input  wire               [1:0] hresp,

  /*****************************************************************************
  * aligner interface
  *****************************************************************************/
  input  wire              [63:0] datai,
  input  wire                     datai_en,
  output wire                     datai_rdy,
  output wire              [63:0] datao,
  output wire                     datao_en,
  output wire                     datao_last,
  input  wire                     datao_rdy 
);


  /*****************************************************************************
  * declarations
  *****************************************************************************/
  localparam  S_IDLE   =4'b0000,
              S_R_A    =4'b0001,
              S_R_AD   =4'b0010,
              S_R_D    =4'b0011,
              S_R_PAUSE=4'b0100,
              S_W_PAUSE=4'b0101,
              S_W_A    =4'b0110,
              S_W_AD   =4'b0111,
              S_W_D    =4'b1000;
  

  reg   [3:0] state;
  reg  [16:0] count;
  
  reg   [2:0] byte_nbr;
  wire [16:0] next_count;
  wire [31:0] next_haddr;
  wire [31:0] next_hwdata;

  wire  [2:0] addr;
  wire  [2:0] reliquat;
  wire [16:0] nbr_qwordsx8;

  wire        wbuffer_rden;

  reg   [1:0] buffer_rd_ptr;
  reg   [1:0] buffer_wr_ptr;
  reg  [63:0] buffer[1:0];
  wire        buffer_empty;
  wire        buffer_last;
  wire        buffer_full;
  reg         nwrite;


  /*****************************************************************************
  * assignments
  *****************************************************************************/
  always @*
  begin
    case (hsize)
    2'b00:   byte_nbr = 3'd1;
    2'b01:   byte_nbr = 3'd2;
    default: byte_nbr = 3'd4;
    endcase
  end

  assign next_count  = count - {14'd0,byte_nbr};

  assign next_haddr  = haddr + {29'd0,byte_nbr};
  
  assign next_hwdata = (haddr[2]) ? buffer[buffer_rd_ptr[0]][63:32] :
                                    buffer[buffer_rd_ptr[0]][31:0];


  /*****************************************************************************
  * computes the number of qwords involved in the transfer
  *****************************************************************************/
  assign addr         = (write) ? dst_addr[2:0] : src_addr[2:0];
  assign reliquat     = addr[2:0] + byte_length[2:0];
  assign nbr_qwordsx8 = { 1'b0,byte_length[15:0]} +
                        {14'b0,addr[2:0]} +
                        {13'b0,|reliquat,3'b0};


  /*****************************************************************************
  * FSM
  *****************************************************************************/
  always @(posedge clk,negedge rst_n)
  begin
    if(rst_n==1'b0)
    begin
      haddr       <= 32'b0;
      htrans      <= 2'b00;
      hwrite      <= 1'b0;
      hsize       <= 2'b00;
      hwdata      <= 32'h0;
      
      count       <= 17'd0;
      done        <= 1'b0;
      
      state       <= S_IDLE;
    end
    else
    begin
    
      case(state)
      /*************************************************************************
      * IDLE
      *************************************************************************/
      default:
      begin
      
        if(~done & start)
        begin
          if (write)
          begin
            /* display first bus address */
            haddr    <= dst_addr[31:0];
            htrans   <= 2'b00;
            hwrite   <= 1'b1;

            if (dst_addr[0] || byte_length<16'd2)
              hsize <= 2'b00;
            else if (dst_addr[1] || byte_length<16'd4)
              hsize <= 2'b01;
            else
              hsize <= 2'b10;

            count    <= {1'b0,byte_length};
            
            state    <= S_W_PAUSE;
          end
          else
          begin
            /* display first bus address */
            haddr    <= {src_addr[31:3],3'b0};
            htrans   <= 2'b10;
            hwrite   <= 1'b0;
            hsize    <= 2'b10;
           
            count    <= {nbr_qwordsx8[16:3],3'b0};
            
            state    <= S_R_A;
          end
        end
        else if(done & ~start)
        begin
        
          done <= 1'b0;
        
        end
      
      end
      
      /*************************************************************************
      * READ A
      *************************************************************************/
      S_R_A:
      begin
        if(hready)
        begin
          count <= next_count;
          
          if(next_count==17'd0)
          begin
            htrans <= 2'b00;
            state  <= S_R_D;
          end
          else
          begin
            if(buffer_empty || datao_rdy)
            begin
              haddr   <= next_haddr;
              htrans  <= 2'b11;
              state   <= S_R_AD; 
            end  
            else
            begin
              htrans  <= 2'b00;
              state   <= S_R_D;
            end
          end
        end
      end
      
      /*************************************************************************
      * READ D
      *************************************************************************/
      S_R_D:
      begin
        if(hready)
        begin
          
          if(count==17'd0)
          begin
            done       <= 1'b1;
            state      <= S_IDLE;
          end
          else
          begin
            if(buffer_empty || datao_rdy)
            begin
              haddr  <= next_haddr;
              htrans <= 2'b10;
              state  <= S_R_A;
            end
            else
            begin
              state <= S_R_PAUSE;
            end
          end
        end
      end
      
      /*************************************************************************
      * READ AD
      *************************************************************************/
      S_R_AD:
      begin
        if(hready)
        begin
         
          count <= next_count;

          if(count==17'd0)
          begin
            done       <= 1'b1;
            state      <= S_IDLE;
          end
          
          if(buffer_empty || datao_rdy)
          begin
            if(next_count==17'd0)
            begin
              htrans <= 2'b00;
              state  <= S_R_D;
            end
            else
            begin
              haddr  <= next_haddr;
              htrans <= 2'b11;
            end
          end
          else
          begin
            htrans <= 2'b00;
            state  <= S_R_D;
          end
        end
      end
      
      /*************************************************************************
      * READ PAUSE
      *************************************************************************/
      S_R_PAUSE:
      begin
        if(buffer_empty || datao_rdy)                  
        begin                      
          haddr  <= next_haddr;
          htrans <= 2'b10;       
          state  <= S_R_A;    
        end                        
      end

      /*************************************************************************
      * WRITE PAUSE
      *************************************************************************/
      S_W_PAUSE:
      begin
        if(datai_en && datai_rdy)
        begin
          htrans <= 2'b10;
          state  <= S_W_A;
        end
      end
      
      /*************************************************************************
      * WRITE A
      *************************************************************************/
      S_W_A:
      begin
      
        if(hready)
        begin
          count     <= next_count;
          haddr     <= next_haddr;
          hwdata    <= next_hwdata;

          if (next_haddr[0] || next_count<17'd2)
            hsize <= 2'b00;
          else if (next_haddr[1] || next_count<17'd4)
            hsize <= 2'b01;
          else
            hsize <= 2'b10;
          
          if(next_count==17'd0)
          begin
            htrans <= 2'b00;
            state  <= S_W_D;
          end
          else
          begin
            if(datai_en && datai_rdy || buffer_full || !wbuffer_rden)
            begin
              htrans <= 2'b10;
              state  <= S_W_AD;
            end  
            else
            begin
              htrans <= 2'b00;
              state  <= S_W_D;
            end
          end
        end
      end
      
      /*************************************************************************
      * WRITE AD
      *************************************************************************/
      S_W_AD:
      begin
      
        if(hready)
        begin
          count  <= next_count;
          haddr  <= next_haddr;
          hwdata <= next_hwdata;

          if (next_haddr[0] || next_count<17'd2)
            hsize <= 2'b00;
          else if (next_haddr[1] || next_count<17'd4)
            hsize <= 2'b01;
          else
            hsize <= 2'b10;
          
          if(next_count==17'd0)
          begin
            htrans    <= 2'b00;
            state     <= S_W_D;
          end
          else
          begin
            if(datai_en && datai_rdy || buffer_full || !wbuffer_rden)
            begin
              htrans  <= 2'b10;
              state   <= S_W_AD;
            end
            else
            begin
              htrans  <= 2'b00;
              state   <= S_W_D;
            end
          end
        end
      end
      
      /*************************************************************************
      * WRITE D
      *************************************************************************/
      S_W_D:
      begin
        if(hready)
        begin
        
          if(count==17'd0)
          begin
            done     <= 1'b1;
            state    <= S_IDLE;
          end
          else
          begin
            if(datai_en && datai_rdy || !buffer_empty)
            begin
              htrans <= 2'b10;
              state  <= S_W_A;
            end
            else
            begin
              state  <= S_W_PAUSE;
            end
          end
        end
      end
      endcase
    end
  end
  

  /*****************************************************************************
  * read buffer
  *****************************************************************************/
  always @(posedge clk, negedge rst_n)
  begin
    if(rst_n==1'b0)
    begin
      nwrite        <= 1'b0;
      buffer_rd_ptr <= 2'd0;
      buffer_wr_ptr <= 2'd0;
      buffer[0]     <= 64'h0;
      buffer[1]     <= 64'h0;
    end
    else
    begin
      // IDLE
      if (start==1'b0)
      begin
        nwrite        <= 1'b0;
        buffer_rd_ptr <= 2'd0;
        buffer_wr_ptr <= 2'd0;
      end
      else

      // READ BUFFER
      if (write==1'b0)
      begin
        if((state==S_R_AD || state==S_R_D) && hready)
        begin
          nwrite <= nwrite + 1'b1;
        
          if(nwrite==1'd1)
            buffer[buffer_wr_ptr[0]][63:32] <= hrdata;
          else
            buffer[buffer_wr_ptr[0]][31:0]  <= hrdata;
        
          if(nwrite==1'b1 || done==1'b1)
            buffer_wr_ptr                   <= buffer_wr_ptr + 2'd1;
        end
    
        if(datao_en==1'b1)
          buffer_rd_ptr                     <= buffer_rd_ptr + 2'd1;
      end

      // WRITE BUFFER
      else
      begin
        if(datai_en==1'b1 && datai_rdy==1'b1)
        begin
          buffer[buffer_wr_ptr[0]]          <= datai;
          buffer_wr_ptr                     <= buffer_wr_ptr + 2'd1;
        end

        if(wbuffer_rden==1'b1)
          buffer_rd_ptr                     <= buffer_rd_ptr + 2'd1;
      end
    end
  end

  assign buffer_last   =  (buffer_rd_ptr[0]!=buffer_wr_ptr[0])  ? 1'b1 : 1'b0;
  assign buffer_empty  =  (buffer_rd_ptr   ==buffer_wr_ptr   )  ? 1'b1 : 1'b0;
  assign buffer_full   = ((buffer_rd_ptr[1]!=buffer_wr_ptr[1]) &&
                          (buffer_rd_ptr[0]==buffer_wr_ptr[0])) ? 1'b1 : 1'b0;

  assign wbuffer_rden = (hready==1'b1 && (state==S_W_A || state==S_W_AD) &&
                        ((hsize==2'b10 && haddr[2:0]==3'b100) || 
                         (hsize==2'b01 && haddr[2:0]==3'b110) ||
                         (hsize==2'b00 && haddr[2:0]==3'b111)));

  /*****************************************************************************
  * Data out
  *****************************************************************************/
  assign datao      = buffer[buffer_rd_ptr[0]];
  assign datao_en   = !buffer_empty && datao_rdy;
  assign datao_last = buffer_last && done;


  /*****************************************************************************
  * Data In
  *****************************************************************************/
  assign datai_rdy = (state!=S_IDLE) &&    // hold on if bus if not started yet
                     (!buffer_full);


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// pragma coverage block = off 
  reg [16*8-1:0] state_str;
  
  always @(*)
  begin
    case (state)
    S_IDLE:    state_str = "S_IDLE";
    S_R_A:     state_str = "S_R_A";
    S_R_AD:    state_str = "S_R_AD";   
    S_R_D:     state_str = "S_R_D"; 
    S_R_PAUSE: state_str = "S_R_PAUSE";
    S_W_PAUSE: state_str = "S_W_PAUSE";       
    S_W_A:     state_str = "S_W_A";   
    S_W_AD:    state_str = "S_W_AD";   
    S_W_D:     state_str = "S_W_D";   
    default:   state_str = "XXX";
    endcase    
  end
// pragma coverage block = on 
`endif


endmodule
`default_nettype wire
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////



