//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Midstream Physical Channel BusIF
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module midstream_busif
(
  /*****************************************************************************
  * System
  *****************************************************************************/
  input  wire                     clk,
  input  wire                     rst_n,

  /*****************************************************************************
  * control      
  *****************************************************************************/
  input  wire                     start,
  output reg                      done,

  /*****************************************************************************
  * Static parameters
  *****************************************************************************/
  input  wire                     write,
  input  wire              [31:0] src_addr,
  input  wire              [31:0] dst_addr,
  input  wire              [15:0] byte_length,

  /*****************************************************************************
  * bus interface
  *****************************************************************************/
  output reg               [31:0] bus_addr,
  output reg                      bus_trans,
  output reg                      bus_write,
  input  wire              [63:0] bus_rdata,
  output reg               [63:0] bus_wdata,
  output reg                [7:0] bus_we,
  input  wire                     bus_ready,

  /*****************************************************************************
  * aligner interface
  *****************************************************************************/
  input  wire              [63:0] datai,
  input  wire                     datai_en,
  output wire                     datai_rdy,
  output reg               [63:0] datao,
  output reg                      datao_en,
  output reg                      datao_last,
  input  wire                     datao_rdy
);

  /*****************************************************************************
  * declarations
  *****************************************************************************/
  localparam  S_IDLE   =3'b000,
              S_R_A    =3'b001,
              S_R_D    =3'b010,
              S_R_PAUSE=3'b011,
              S_W_PAUSE=3'b100,
              S_W_A    =3'b101,
              S_W_AD   =3'b110,
              S_W_D    =3'b111;
  

  reg   [2:0] state;
  reg  [13:0] count;
  
  wire [13:0] next_count;
  wire [31:0] next_bus_addr;
  
  wire  [2:0] addr;
  wire  [2:0] reliquat;
  wire [16:0] nbr_qwordsx8;

  reg  [63:0] wbuffer_data;
  reg         wbuffer_state;
  wire        wbuffer_rden;

  reg         is_first;
  
  reg   [7:0] first_be;
  reg   [7:0] last_be;
 

  /*****************************************************************************
  * assignments
  *****************************************************************************/
  assign next_count    = count - 14'd1;
  assign next_bus_addr = {bus_addr[31:3],3'b0} + 32'd8;
  
  assign datai_rdy = (state!=S_IDLE) &&    // hold on if bus if not started yet
                     (wbuffer_state==1'b0 || wbuffer_rden==1'b1);


  /*****************************************************************************
  * computes the number of qwords involved in the transfer
  *****************************************************************************/
  assign addr         = (write) ? dst_addr[2:0] : src_addr[2:0];
  assign reliquat     = addr[2:0] + byte_length[2:0];
  assign nbr_qwordsx8 = { 1'b0,byte_length[15:0]} +
                        {14'b0,addr[2:0]}         +
                        {13'b0,|reliquat,3'b0};

  always @(*)
  begin:b_byte_enables
  
    case(dst_addr[2:0])
    
      3'b000:  first_be = 8'b11111111;
      3'b001:  first_be = 8'b11111110;
      3'b010:  first_be = 8'b11111100;
      3'b011:  first_be = 8'b11111000;
      3'b100:  first_be = 8'b11110000;
      3'b101:  first_be = 8'b11100000;
      3'b110:  first_be = 8'b11000000;
      default: first_be = 8'b10000000;
    
    endcase

    case(reliquat)
    
      3'b000:  last_be = 8'b11111111;
      3'b001:  last_be = 8'b00000001;
      3'b010:  last_be = 8'b00000011;
      3'b011:  last_be = 8'b00000111;
      3'b100:  last_be = 8'b00001111;
      3'b101:  last_be = 8'b00011111;
      3'b110:  last_be = 8'b00111111;
      default: last_be = 8'b01111111;
    
    endcase
    
  end


  /*****************************************************************************
  * FSM
  *****************************************************************************/
  always @(posedge clk,negedge rst_n)
  begin
    if(rst_n==1'b0)
    begin
      bus_addr    <= 32'b0;
      bus_trans   <= 1'b0;
      bus_write   <= 1'b0;
      bus_wdata   <= 64'b0;
      bus_we      <= 8'b0;
      
      datao       <= 64'b0;
      datao_en    <= 1'b0;
      datao_last  <= 1'b0;
    
      count       <= 14'b0;
      done        <= 1'b0;
      
      is_first    <= 1'b0;

      state       <= S_IDLE;
    end
    else
    begin
      datao_en   <= 1'b0;
      //datao_last <= 1'b0;
    
      case(state)
      
      /*************************************************************************
      * IDLE
      *************************************************************************/
      S_IDLE:
      begin
      
        datao      <= 64'b0;
        datao_last <= 1'b0;
                  
        if(~done & start)
        begin
          if (write)
          begin
            bus_addr   <= dst_addr;
            bus_write  <= 1'b1;
            is_first   <= 1'b1;
            count      <= nbr_qwordsx8[16:3];
            state      <= S_W_PAUSE;
          end
          else
          begin
            bus_addr   <= {src_addr[31:3],3'b0};
            bus_write  <= 1'b0;
            bus_trans  <= 1'b1;
            count      <= nbr_qwordsx8[16:3];
            state      <= S_R_A;
          end
          
        end
        else if(done & ~start)
        begin
          done <= 1'b0;
        end
      
      end
      
      /*************************************************************************
      * READ A
      *************************************************************************/
      S_R_A:
      begin
      
        if(bus_ready)
        begin
        
          count     <= next_count;
          
          bus_trans <= 1'b0;
          state     <= S_R_D;
          
        end
      
      end
      
      /*************************************************************************
      * READ D
      *************************************************************************/
      S_R_D:
      begin
      
        if(bus_ready)
        begin
        
          datao    <= bus_rdata;
          datao_en <= 1'b1;
          
          if(count==14'd0)
          begin
          
            if(datao_rdy)
            begin

              datao_last <= 1'b1;
              done       <= 1'b1;
              state      <= S_IDLE;

            end
            else
            begin

              datao_last <= 1'b1;
              state      <= S_R_PAUSE;

            end
            
          end
          else
          begin
          
            if(~datao_rdy)
            begin
            
              state <= S_R_PAUSE;
            
            end
            else
            begin
            
              bus_addr  <= next_bus_addr;
              bus_trans <= 1'b1;
              state     <= S_R_A;
              
            end
          
          end
          
        end
      
      end
      
      /*************************************************************************
      * READ PAUSE
      *************************************************************************/
      S_R_PAUSE:
      begin
      
        datao_en  <= 1'b1;

        if(datao_rdy)                  
        begin                      
          datao_en  <= 1'b0;
                                   
          if(count==14'd0)
          begin
          
            done       <= 1'b1;
            state      <= S_IDLE;
            
          end
          else
          begin

            bus_addr  <= next_bus_addr;
            bus_trans <= 1'b1;       
            state     <= S_R_A;    

          end
                                   
        end                        
          
      end

      /*************************************************************************
      * WRITE PAUSE
      *************************************************************************/
      S_W_PAUSE:
      begin
      
        if(datai_en)
        begin
        
          bus_trans <= 1'b1;
          state     <= S_W_A;
        
        end
        
      end
      
      /*************************************************************************
      * WRITE A
      *************************************************************************/
      S_W_A:
      begin
      
        if(bus_ready)
        begin
        
          is_first  <= 1'b0;
          count     <= next_count;
          bus_addr  <= next_bus_addr;
          bus_wdata <= wbuffer_data;
          
          if(count==14'd1)
          begin
          
            if(is_first)
            
              bus_we <= first_be & last_be;
              
            else
            
              bus_we <= last_be;
            
            bus_trans <= 1'b0;
            state     <= S_W_D;
            
          end
          else
          begin
          
            if(is_first)
            
              bus_we <= first_be;
              
            else
            
              bus_we <= {8{1'b1}};
            
            if(datai_en)
            begin
            
              bus_trans <= 1'b1;
              state     <= S_W_AD;
              
            end  
            else
            begin
            
              bus_trans <= 1'b0;
              state     <= S_W_D;
              
            end
          
          end
          
        end
      
      end
      
      /*************************************************************************
      * WRITE AD
      *************************************************************************/
      S_W_AD:
      begin
      
        if(bus_ready)
        begin
        
          count     <= next_count;
          bus_addr  <= next_bus_addr;
          bus_wdata <= wbuffer_data;
          
          if(count==14'd1)
          begin
          
            bus_trans <= 1'b0;
            bus_we    <= last_be;
            state     <= S_W_D;
            
          end
          else
          begin
          
            bus_we <= {8{1'b1}};
          
            if(datai_en)
            begin
            
              bus_trans <= 1'b1;
              state     <= S_W_AD;
              
            end
            else
            begin
            
              bus_trans <= 1'b0;
              state     <= S_W_D;
              
            end
          
          end
          
        end
      
      end
      
      /*************************************************************************
      * WRITE D
      *************************************************************************/
      S_W_D:
      begin
      
        if(bus_ready)
        begin
        
          if(count==14'd0)
          begin
          
            done  <= 1'b1;
            state <= S_IDLE;
            
          end
          else
          begin
          
            if(datai_en==1'b1)
            begin
            
              bus_trans <= 1'b1;
              state     <= S_W_A;
              
            end
            else
            begin
            
              state <= S_W_PAUSE;
            
            end
            
          end
          
        end
        else
        begin
        
          if(count!=14'd0)
          begin
          
            if(datai_en==1'b1)
            begin
            
              bus_trans <= 1'b1;
              state     <= S_W_A;
              
            end
            
          end
          
        end
        
      end
      endcase
    
    end
    
  end
  
    
  /*****************************************************************************
  * stall buffer
  *****************************************************************************/
  assign wbuffer_rden = (bus_ready==1'b1 && (state==S_W_A || state==S_W_AD)) |
                        state==S_W_PAUSE;
  
  always @(posedge clk, negedge rst_n)
  begin
  
    if(rst_n==1'b0)
    begin
    
      wbuffer_data  <= 64'b0;
      wbuffer_state <= 1'b0;
    
    end
    else
    begin
    
      if(start==1'b0 || write==1'b0)
      begin
      
        wbuffer_state <= 1'b0;
      
      end
      else
      begin
      
        if(wbuffer_state==1'b0)
        begin
          
          if(datai_en==1'b1)
          begin
          
            wbuffer_data  <= datai;
            wbuffer_state <= 1'b1;
            
          end
          
        end
        else
        begin
      
          if(wbuffer_rden==1'b1)
          begin
          
            if(datai_en==1'b1)
            
              wbuffer_data <= datai;
              
            else
            
              wbuffer_state <= 1'b0;
            
          end
      
        end
        
      end
      
    end
  
  end


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// pragma coverage block = off 
  reg [16*8-1:0] state_str;
  
  always @(*)
  begin
    case (state)
    S_IDLE:    state_str = "S_IDLE";
    S_R_A:     state_str = "S_R_A";
    S_R_D:     state_str = "S_R_D"; 
    S_R_PAUSE: state_str = "S_R_PAUSE";
    S_W_PAUSE: state_str = "S_W_PAUSE";       
    S_W_A:     state_str = "S_W_A";   
    S_W_AD:    state_str = "S_W_AD";   
    S_W_D:     state_str = "S_W_D";   
    default:   state_str = "XXX";
    endcase    
  end
// pragma coverage block = on 
`endif


endmodule
`default_nettype wire
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////


