//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Midstream Physical Channel BusIF Read
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module midstream_busif_rd
(
  /*****************************************************************************
  * System
  *****************************************************************************/
  input  wire                     clk,
  input  wire                     rst_n,

  /*****************************************************************************
  * control      
  *****************************************************************************/
  input  wire                     start,
  output reg                      done,

  /*****************************************************************************
  * Static parameters
  *****************************************************************************/
  input  wire              [31:0] src_addr,
  input  wire              [15:0] byte_length,

  /*****************************************************************************
  * bus interface
  *****************************************************************************/
  output reg               [31:0] bus_addr,
  output reg                      bus_trans,
  input  wire              [63:0] bus_rdata,
  input  wire                     bus_ready,

  /*****************************************************************************
  * aligner interface
  *****************************************************************************/
  output reg               [63:0] datao,
  output reg                      datao_en,
  output reg                      datao_last,
  input  wire                     datao_rdy
);

  /*****************************************************************************
  * declarations
  *****************************************************************************/
  localparam  S_IDLE   =2'b00,
              S_R_A    =2'b01,
              S_R_D    =2'b10,
              S_R_PAUSE=2'b11;
  

  reg   [1:0] state;
  reg  [13:0] count;
  
  wire [13:0] next_count;
  wire [31:0] next_bus_addr;
  
  wire  [2:0] reliquat;
  wire [16:0] nbr_qwordsx8;


  /*****************************************************************************
  * assignments
  *****************************************************************************/
  assign next_count    = count - 14'd1;
  assign next_bus_addr = {bus_addr[31:3],3'b0} + 32'd8;
  

  /*****************************************************************************
  * computes the number of qwords involved in the transfer
  *****************************************************************************/
  assign reliquat     = src_addr[2:0] + byte_length[2:0];
  assign nbr_qwordsx8 = { 1'b0,byte_length[15:0]} +
                        {14'b0,src_addr[2:0]}     +
                        {13'b0,|reliquat,3'b0};


  /*****************************************************************************
  * FSM
  *****************************************************************************/
  always @(posedge clk,negedge rst_n)
  begin
    if(rst_n==1'b0)
    begin
      bus_addr    <= 32'b0;
      bus_trans   <= 1'b0;
      
      datao       <= 64'b0;
      datao_en    <= 1'b0;
      datao_last  <= 1'b0;
    
      count       <= 14'b0;
      done        <= 1'b0;
      
      state       <= S_IDLE;
    end
    else
    begin
      datao_en   <= 1'b0;
      //datao_last <= 1'b0;
    
      case(state)
      
      /*************************************************************************
      * IDLE
      *************************************************************************/
      S_IDLE:
      begin
      
        datao      <= 64'b0;
        datao_last <= 1'b0;
                  
        if(~done & start)
        begin
          bus_addr   <= {src_addr[31:3],3'b0};
          bus_trans  <= 1'b1;
          count      <= nbr_qwordsx8[16:3];
          state      <= S_R_A;
        end
        else if(done & ~start)
        begin
          done <= 1'b0;
        end
      
      end
      
      /*************************************************************************
      * READ A
      *************************************************************************/
      S_R_A:
      begin
      
        if(bus_ready)
        begin
        
          count     <= next_count;
          
          bus_trans <= 1'b0;
          state     <= S_R_D;
          
        end
      
      end
      
      /*************************************************************************
      * READ D
      *************************************************************************/
      S_R_D:
      begin
      
        if(bus_ready)
        begin
        
          datao    <= bus_rdata;
          datao_en <= 1'b1;
          
          if(count==14'd0)
          begin
          
            if(datao_rdy)
            begin

              datao_last <= 1'b1;
              done       <= 1'b1;
              state      <= S_IDLE;

            end
            else
            begin

              datao_last <= 1'b1;
              state      <= S_R_PAUSE;

            end
            
          end
          else
          begin
          
            if(~datao_rdy)
            begin
            
              state <= S_R_PAUSE;
            
            end
            else
            begin
            
              bus_addr  <= next_bus_addr;
              bus_trans <= 1'b1;
              state     <= S_R_A;
              
            end
          
          end
          
        end
      
      end
      
      /*************************************************************************
      * READ PAUSE
      *************************************************************************/
      S_R_PAUSE:
      begin
      
        datao_en  <= 1'b1;

        if(datao_rdy)                  
        begin                      
          datao_en  <= 1'b0;
                                   
          if(count==14'd0)
          begin
          
            done       <= 1'b1;
            state      <= S_IDLE;
            
          end
          else
          begin

            bus_addr  <= next_bus_addr;
            bus_trans <= 1'b1;       
            state     <= S_R_A;    

          end
                                   
        end                        
          
      end

      endcase
    
    end
    
  end
  
    
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// pragma coverage block = off 
  reg [16*8-1:0] state_str;
  
  always @(*)
  begin
    case (state)
    S_IDLE:    state_str = "S_IDLE";
    S_R_A:     state_str = "S_R_A";
    S_R_D:     state_str = "S_R_D"; 
    S_R_PAUSE: state_str = "S_R_PAUSE";
    default:   state_str = "XXX";
    endcase    
  end
// pragma coverage block = on 
`endif


endmodule
`default_nettype wire
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////


