//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Midstream Physical Channel Top
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module midstream_channel
(
  /*****************************************************************************
  * System
  *****************************************************************************/
  input wire                      clk,
  input wire                      rst_n,

  /*****************************************************************************
  * Channel Control
  *****************************************************************************/
  input  wire                     channel_sel,
  input  wire                     channel_req,
  input  wire              [31:0] channel_saddr,
  input  wire              [31:0] channel_daddr,
  input  wire               [3:0] channel_tag,
  input  wire              [15:0] channel_length,
  output wire                     channel_busy,
  output wire                     channel_ack,
  output wire               [3:0] channel_ack_tag,
  output wire                     channel_ack_error,
  input  wire                     channel_ack_ack,

  /*****************************************************************************
  * Bus interfaces 0 (Shared SRAM)
  *****************************************************************************/
  output wire              [31:0] p0_addr,
  output wire                     p0_trans,
  input  wire              [63:0] p0_rdata,
  input  wire                     p0_ready,

  /*****************************************************************************
  * Bus interfaces 1 (Shared SRAM)
  *****************************************************************************/
  output wire              [31:0] p1_addr,
  output wire                     p1_trans,
  output wire              [63:0] p1_wdata,
  output wire               [7:0] p1_we,
  input  wire                     p1_ready,

  /*****************************************************************************
  * AHB master (IP)
  *****************************************************************************/
  input  wire                     p2_hready,
  output wire              [31:0] p2_haddr,
  output wire               [1:0] p2_htrans,
  output wire                     p2_hwrite,
  output wire               [1:0] p2_hsize,
  input  wire              [31:0] p2_hrdata,
  output wire              [31:0] p2_hwdata,
  input  wire               [1:0] p2_hresp 
);

  /*****************************************************************************
  * declarations
  *****************************************************************************/
  // Controller
  wire        start;
  wire [31:0] src_addr;
  wire [31:0] dst_addr;
  wire [15:0] byte_length;
  reg         done;

  // Aligner
  reg  [63:0] aligner_datai;
  reg         aligner_datai_en;
  reg         aligner_datai_last;
  wire [63:0] aligner_datao;
  wire        aligner_datao_en;

  // BUSIF read Port
  wire [63:0] p0_datao;
  wire        p0_datao_en;
  reg         p0_datao_rdy;
  wire        p0_datao_last;
  wire        p0_start;
  wire        p0_done;

  // BUSIF write Port
  wire [63:0] p1_datai;
  wire        p1_datai_en;
  wire        p1_datai_rdy;
  wire        p1_start;
  wire        p1_done;

  // AHB Port
  wire [63:0] p2_datai;
  wire        p2_datai_en;
  wire        p2_datai_rdy;
  wire [63:0] p2_datao;
  wire        p2_datao_en;
  wire        p2_datao_rdy;
  wire        p2_datao_last;
  wire        p2_start;
  wire        p2_write;
  wire        p2_done;

  //////////////////////////////////////////////////////////////////////////////
  //
  //  CONTROL
  //
  //////////////////////////////////////////////////////////////////////////////
  midstream_control u_midstream_control
  (
    /***************************************************************************
    * System
    ***************************************************************************/
    .clk                (clk                ),
    .rst_n              (rst_n              ),

    /***************************************************************************
    * Channel Control
    ***************************************************************************/
    .channel_sel        (channel_sel        ),
    .channel_req        (channel_req        ),
    .channel_saddr      (channel_saddr      ),
    .channel_daddr      (channel_daddr      ),
    .channel_tag        (channel_tag        ),
    .channel_length     (channel_length     ),
    .channel_busy       (channel_busy       ),
    .channel_ack        (channel_ack        ),
    .channel_ack_tag    (channel_ack_tag    ),
    .channel_ack_error  (channel_ack_error  ),
    .channel_ack_ack    (channel_ack_ack    ),

    /***************************************************************************
    * fragment parameters
    ***************************************************************************/
    .start              (start              ),
    .src_addr           (src_addr           ),
    .dst_addr           (dst_addr           ),
    .byte_length        (byte_length        ),
    .done               (done               )
  );

  //////////////////////////////////////////////////////////////////////////////
  //
  //  Bus IF Read
  //
  //////////////////////////////////////////////////////////////////////////////
  midstream_busif_rd u_midstream_busif_rd
  (
    /***************************************************************************
    * System
    ***************************************************************************/
    .clk         (clk               ),
    .rst_n       (rst_n             ),

    /***************************************************************************
    * control      
    ***************************************************************************/
    .start       (p0_start          ),
    .done        (p0_done           ),

    /***************************************************************************
    * Static parameters
    ***************************************************************************/
    .src_addr    (src_addr          ),
    .byte_length (byte_length       ),

    /***************************************************************************
    * bus interface
    ***************************************************************************/
    .bus_addr    (p0_addr           ),
    .bus_trans   (p0_trans          ),
    .bus_rdata   (p0_rdata          ),
    .bus_ready   (p0_ready          ),

    /***************************************************************************
    * aligner interface
    ***************************************************************************/
    .datao       (p0_datao          ),
    .datao_en    (p0_datao_en       ),
    .datao_last  (p0_datao_last     ),
    .datao_rdy   (p0_datao_rdy      )
  );

  //////////////////////////////////////////////////////////////////////////////
  //
  //  Bus IF Write
  //
  //////////////////////////////////////////////////////////////////////////////
  midstream_busif_wr u_midstream_busif_wr
  (
    /***************************************************************************
    * System
    ***************************************************************************/
    .clk         (clk               ),
    .rst_n       (rst_n             ),

    /***************************************************************************
    * control      
    ***************************************************************************/
    .start       (p1_start          ),
    .done        (p1_done           ),

    /***************************************************************************
    * Static parameters
    ***************************************************************************/
    .dst_addr    (dst_addr          ),
    .byte_length (byte_length       ),

    /***************************************************************************
    * bus interface
    ***************************************************************************/
    .bus_addr    (p1_addr           ),
    .bus_trans   (p1_trans          ),
    .bus_wdata   (p1_wdata          ),
    .bus_we      (p1_we             ),
    .bus_ready   (p1_ready          ),

    /***************************************************************************
    * aligner interface
    ***************************************************************************/
    .datai       (p1_datai          ),
    .datai_en    (p1_datai_en       ),
    .datai_rdy   (p1_datai_rdy      )
  );

  //////////////////////////////////////////////////////////////////////////////
  //
  //  AHB IF
  //
  //////////////////////////////////////////////////////////////////////////////
  midstream_ahbif u_midstream_ahbif
  (
    /***************************************************************************
    * System
    ***************************************************************************/
    .clk         (clk           ),
    .rst_n       (rst_n         ),

    /***************************************************************************
    * control/state
    ***************************************************************************/
    .start       (p2_start      ),
    .write       (p2_write      ),
    .done        (p2_done       ),

    /***************************************************************************
    * Static parameters
    ***************************************************************************/
    .src_addr    (src_addr      ),
    .dst_addr    (dst_addr      ),
    .byte_length (byte_length   ),

    /***************************************************************************
    * AHB interface
    ***************************************************************************/
    .hready      (p2_hready     ),
    .haddr       (p2_haddr      ),
    .htrans      (p2_htrans     ),
    .hwrite      (p2_hwrite     ),
    .hsize       (p2_hsize      ),
    .hrdata      (p2_hrdata     ),
    .hwdata      (p2_hwdata     ),
    .hresp       (p2_hresp      ),

    /***************************************************************************
    * aligner interface
    ***************************************************************************/
    .datai       (p2_datai      ),
    .datai_en    (p2_datai_en   ),
    .datai_rdy   (p2_datai_rdy  ),
    .datao       (p2_datao      ),
    .datao_en    (p2_datao_en   ),
    .datao_rdy   (p2_datao_rdy  ),
    .datao_last  (p2_datao_last )
  );

  //////////////////////////////////////////////////////////////////////////////
  //
  //  Aligner
  //
  //////////////////////////////////////////////////////////////////////////////
  aligner u_midstream_aligner
  (
    /***************************************************************************
    * System
    ***************************************************************************/
    .clk        (clk                ),
    .rst_n      (rst_n              ),

    /***************************************************************************
    *
    ***************************************************************************/
    .start      (start              ),
    .src_addr   (src_addr[2:0]      ),
    .dst_addr   (dst_addr[2:0]      ),

    /***************************************************************************
    *
    ***************************************************************************/
    .datai      (aligner_datai      ),
    .datai_en   (aligner_datai_en   ),
    .datai_last (aligner_datai_last ),

    /***************************************************************************
    *
    ***************************************************************************/
    .datao      (aligner_datao      ),
    .datao_en   (aligner_datao_en   )
  );

  /*****************************************************************************
  * Direction Mux
  *
  * Address[24:23]==0 : Shared Ram
  * Address[24:23]!=0 : AHB
  *
  *****************************************************************************/
  // Port Controls
  assign p0_start = src_addr[24:23]==2'b0 & start; // Source is Shared SRAM
  assign p1_start = dst_addr[24:23]==2'b0 & start; // Destination is Shared SRAM
  assign p2_start = src_addr[24:23]!=2'b0 & start| // Source is AHB
                    dst_addr[24:23]!=2'b0 & start; // Destination is AHB
  assign p2_write = dst_addr[24:23]!=2'b0;         // Destination is AHB

  // Source Mux
  always @*
  begin
     if (src_addr[24:23]==2'b0)
     // Source is Shared SRAM
     begin
        aligner_datai      = p0_datao;
        aligner_datai_en   = p0_datao_en & p0_datao_rdy;
        aligner_datai_last = p0_datao_last;
     end
     else
     // Source is AHB
     begin
        aligner_datai      = p2_datao;
        aligner_datai_en   = p2_datao_en & p2_datao_rdy;
        aligner_datai_last = p2_datao_last;
     end
  end

  // Destination Mux
  always @*
  begin
     if (dst_addr[24:23]==2'b0)
     // Destination is Shared SRAM
     begin
        p0_datao_rdy = p1_datai_rdy;

        // Transfert is done when p1 write is done
        done         = p1_done;
     end
     else
     // Destination is AHB
     begin
        p0_datao_rdy = p2_datai_rdy;

        // Transfert is done when p2 write is done
        done         = p2_done;
     end
  end

  // BusIf Write Port
  assign p1_datai     = aligner_datao;
  assign p1_datai_en  = aligner_datao_en;

  // AHB Port
  assign p2_datai     = aligner_datao;
  assign p2_datai_en  = aligner_datao_en;
  assign p2_datao_rdy = p1_datai_rdy;

endmodule
`default_nettype wire
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
