//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: lvalla $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: 22021 $
// $Date: 2015-10-19 11:14:46 +0200 (Mon, 19 Oct 2015) $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : upstream bus interface
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: https://dpereira@svn.frso.rivierawaves.com/svn/rw_wlan_nx/branches/Projects/WLAN_HE_REF_IP/HW/WLAN_HE_REF_IP_20_40MHZ/IPs/HW/TOP11ax/MACSUBSYS/rw_host_dma/verilog/rtl/upstream_ahbif.v $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module upstream_ahbif
(
  /*****************************************************************************
  * System
  *****************************************************************************/
  input  wire           clk,
  input  wire           rst_n,

  /*****************************************************************************
  * control/state
  *****************************************************************************/
  input  wire           start,
  input  wire           pause,
  output reg            done,

  /*****************************************************************************
  * Static parameters
  *****************************************************************************/
  input  wire [31:0]    src_addr,
  input  wire [11:0]    src_length,
 
  /*****************************************************************************
  * AHB interface
  *****************************************************************************/
  output reg  [31:0]    haddr,
  output reg  [ 1:0]    htrans,
  input  wire [31:0]    hrdata,
  input  wire           hready,

  /*****************************************************************************
  * aligner interface
  *****************************************************************************/
  output wire [63:0]    data,
  output wire           data_en,
  output wire           data_last
);

  /*****************************************************************************
  * declarations
  *****************************************************************************/
  localparam   S_IDLE =3'b000,
               S_A    =3'b001,
               S_AD   =3'b010,
               S_D    =3'b011,
               S_PAUSE=3'b100;

  reg  [ 2:0]  state;
  reg   [9:0]  count;
  reg          nread;
  reg  [31:0]  data_buf;
  
  wire  [9:0]  new_count;
  wire [31:0]  new_addr;
  
  wire [ 2:0]  reliquat;
  wire [11:0]  nbr_qwordsx8;

  /*****************************************************************************
  * FSM
  *****************************************************************************/
  assign new_count    = count - 10'b1;
  assign new_addr     = {haddr[31:2],2'b0} + 32'h4;
  
  assign reliquat     = src_addr[2:0] + src_length[2:0];
  assign nbr_qwordsx8 = src_length[11:0]     +
                        {9'b0,src_addr[2:0]} +
                        {8'b0,|reliquat,3'b0};

  always @(posedge clk,negedge rst_n)
  begin
  
    if(rst_n==1'b0)
    begin
    
      haddr       <= 32'b0;
      htrans      <= 2'b0;
      
      data_buf    <= 32'h0;
      count       <= 10'b0;
      done        <= 1'b0;
      nread       <= 1'b0;
      
      state       <= S_IDLE;
      
    end
    else
    begin

      case(state)
      
        /***********************************************************************
        * IDLE
        ***********************************************************************/
        default:
        begin
        
          nread     <= 1'b0;
                    
          if(~done & start)
          begin
          
            /* display first bus address */
            
            count    <= {nbr_qwordsx8[11:3],1'b0};
          
            haddr    <= {src_addr[31:3],3'b0};
            htrans   <= 2'b10;
           
            state    <= S_A;
            
          end
          else if(done & ~start)
          begin
          
            done <= 1'b0;
          
          end
        
        end
        
        /***********************************************************************
        * A
        ***********************************************************************/
        S_A:
        begin
        
          if(hready)
          begin
          
            count <= new_count;
            
            if(count==10'b1)
            begin
            
              htrans <= 2'b00;
              state  <= S_D;
            
            end
            else
            begin
            
              if(pause)
              begin
              
                htrans  <= 2'b00;
                state   <= S_D;
              
              end  
              else
              begin
              
                haddr   <= new_addr;
                htrans  <= 2'b11;
                state   <= S_AD; 
            
              end
            
            end
            
          end
        
        end
        
        /***********************************************************************
        * D
        ***********************************************************************/
        S_D:
        begin
        
          if(hready)
          begin
            nread     <= nread + 1'b1;
          
            if(nread == 1'd1)
            begin
              nread       <= 1'd0;
            end
            else
            begin
              data_buf    <= hrdata;
            end
            
            if(count==10'b0)
            begin
            
              done      <= 1'b1;
              state     <= S_IDLE;
              
            end
            else
            begin
            
              if(pause)
              begin
              
                state <= S_PAUSE;
              
              end
              else
              begin
              
                haddr  <= new_addr;
                htrans <= 2'b10;
                state  <= S_A;
                
              end
            
            end
            
          end
        
        end
        
        /***********************************************************************
        * AD
        ***********************************************************************/
        S_AD:
        begin
        
          if(hready)
          begin
          
            nread     <= nread + 1'b1;
            if(nread == 1'd1)
            begin
              nread       <= 1'd0;
            end
            else
            begin
              data_buf    <= hrdata;
            end
           
            count   <= new_count;

            if(count==10'b0)
            begin
            
              done      <= 1'b1;
              state     <= S_IDLE;
              
            end
            
            if(pause)
            begin
            
              htrans <= 2'b00;
              state  <= S_D;
              
            end
            else
            begin
            
              if(count==10'b1)
              begin
              
                htrans <= 2'b00;
                state  <= S_D;
              
              end
              else
              begin
              
                haddr <= new_addr;
                htrans <= 2'b11;
              
              end
            
            end
            
          end
        
        end
       
        /***********************************************************************
        * PAUSE
        ***********************************************************************/
        S_PAUSE:
        begin
        
          if(~pause)                  
          begin                      
                                     
            haddr  <= new_addr;
            htrans <= 2'b10;       
            state  <= S_A;    
                                     
          end                        
            
        end
        
      endcase
    
    end
    
  end

  assign data      = {hrdata,data_buf};
  assign data_en   = hready && (state==S_D || state==S_AD) && nread;
  assign data_last = hready && (state==S_D) && (count==10'b0);
  
  `ifdef RW_SIMU_ON
  
    reg [16*8-1:0] state_str;
  
    always @(*)
    
      case (state)
      
        S_IDLE:  state_str = "S_IDLE";
        S_A:     state_str = "S_A";
        S_AD:    state_str = "S_AD";   
        S_D:     state_str = "S_D"; 
        S_PAUSE: state_str = "S_PAUSE";
        default: state_str = "UNDEF";
     
      endcase    
      
  `endif
  
endmodule  
    
  

