//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Upstream AXI Interface
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module upstream_axi
(
  /*****************************************************************************
  * System
  *****************************************************************************/
  input wire                      clk,
  input wire                      rst_n,

  /*****************************************************************************
  * Configuration from register
  *****************************************************************************/
  input  wire               [3:0] axi_burst_length_limit,

  /*****************************************************************************
  * Channel Control
  *****************************************************************************/
  input  wire                     channel_sel,
  input  wire                     channel_req,
  input  wire              [31:0] channel_saddr,
  input  wire              [31:0] channel_daddr,
  input  wire               [3:0] channel_tag,
  input  wire              [15:0] channel_length,
  output wire                     channel_busy,
  output reg                      channel_ack,
  output reg                [3:0] channel_ack_tag,
  output reg                      channel_ack_error,
  input  wire                     channel_ack_ack,

  /*****************************************************************************
  * Upstream Control
  *****************************************************************************/
  output reg                      start,
  output reg               [31:0] saddr,
  output wire               [2:0] daddr,
  output reg               [11:0] length,

  /*****************************************************************************
  * AXI interface
  *****************************************************************************/
  output reg               [ 3:0] awid,
  output reg               [31:0] awaddr,
  output reg               [ 7:0] awlen,
  output wire              [ 2:0] awsize,
  output wire              [ 1:0] awburst,
  output reg               [11:0] awuser,
  output reg                      awvalid,
  input  wire                     awready,

  output wire              [ 3:0] wid,
  output reg               [63:0] wdata,
  output reg               [ 7:0] wstrb,
  output reg                      wlast,
  output reg                      wvalid,
  input  wire                     wready,

  input  wire               [3:0] bid,
  input  wire               [1:0] bresp,
  input  wire                     bvalid,
  output reg                      bready,

  /*****************************************************************************
  * Aligner interface
  *****************************************************************************/
  input  wire              [63:0] datai,
  input  wire                     datai_en,
  output wire                     datai_rdy
);


  /*****************************************************************************
  * parameters
  *****************************************************************************/
  // Write Address & Data FSM states definition
  //$fsm_sd aw_state
  localparam AW_IDLE       = 3'd0,
             AW_ADDR       = 3'd1,
             AW_ADDR_WAIT  = 3'd2,
             AW_DATA_FIRST = 3'd3,
             AW_DATA       = 3'd4,
             AW_DATA_WAIT  = 3'd5;

  // Write response FSM states definition
  //$fsm_sd b_state
  localparam B_IDLE        = 2'd0,
             B_INIT        = 2'd1,
             B_ACK         = 2'd2,
             B_ERROR       = 2'd3;


  /*****************************************************************************
  * declarations
  *****************************************************************************/
  reg    [2:0] aw_state;              // Write Address/Data FSM
  reg    [1:0] b_state;               // Write Response FSM

  wire   [2:0] qwords_reliquat;
  wire  [11:0] nbr_qwordsx8;          // Number of qwords in DMA request

  reg    [7:0] first_strb;            // Write strobes of the first data
  reg    [7:0] last_strb;             // Write strobes of the last data

  reg   [11:0] bytes_to_boundary;     // Data byte from destination address to burst boundary
  reg   [15:0] bytes_remain;          // Data byte remains in DMA request

  reg    [8:0] count;                 // qwords write counter

  // Context registers: Save the 16 DMA channel context
  reg   [13:0] context_req[15:0];     // Number of Request
  reg   [13:0] context_ack[15:0];     // Number of Acknowledge
  reg    [3:0] context_tag;           // Pointer To context table

  reg    [8:0] boundary;              // boundary in qwords
  reg    [8:0] boundary_mask;         // boundary mask

  /*****************************************************************************
  * assignment
  *****************************************************************************/
  assign channel_busy = aw_state!=AW_IDLE;


  /*****************************************************************************
  * computes burst boundary signals
  *****************************************************************************/
  always @*
  begin
     case (axi_burst_length_limit)
     4'd0:    boundary = 9'd1<<0;
     4'd1:    boundary = 9'd1<<1;
     4'd2:    boundary = 9'd1<<2;
     4'd3:    boundary = 9'd1<<3;
     4'd4:    boundary = 9'd1<<4;
     4'd5:    boundary = 9'd1<<5;
     4'd6:    boundary = 9'd1<<6;
     4'd7:    boundary = 9'd1<<7;
     default: boundary = 9'd1<<8;
     endcase

     boundary_mask = boundary-9'd1;
  end


  /*****************************************************************************
  * computes the number of byte involved in the AXI Burst
  *****************************************************************************/
  always @*
  begin
    bytes_to_boundary = {boundary,3'b000} - ({1'b0,awaddr[10:0]}&{boundary_mask,3'b111});

    if (bytes_remain>{4'b0,bytes_to_boundary})
      length = bytes_to_boundary;
    else
      length = bytes_remain[11:0];
  end

  /*****************************************************************************
  * computes the number of qwords involved in the AXI Burst
  *****************************************************************************/
  assign qwords_reliquat = length[2:0] + awaddr[2:0];
  assign nbr_qwordsx8    = length[11:0]       +
                           {9'b0,awaddr[2:0]} +
                           {8'b0,|qwords_reliquat,3'b0};


  /*****************************************************************************
  * computes leading/trailing byte enables
  *****************************************************************************/
  always @(*)
  begin
    case(daddr[2:0])
    3'b000:  first_strb = 8'b11111111;
    3'b001:  first_strb = 8'b11111110;
    3'b010:  first_strb = 8'b11111100;
    3'b011:  first_strb = 8'b11111000;
    3'b100:  first_strb = 8'b11110000;
    3'b101:  first_strb = 8'b11100000;
    3'b110:  first_strb = 8'b11000000;
    default: first_strb = 8'b10000000;
    endcase

    case(qwords_reliquat)
    3'b000:  last_strb  = 8'b11111111;
    3'b001:  last_strb  = 8'b00000001;
    3'b010:  last_strb  = 8'b00000011;
    3'b011:  last_strb  = 8'b00000111;
    3'b100:  last_strb  = 8'b00001111;
    3'b101:  last_strb  = 8'b00011111;
    3'b110:  last_strb  = 8'b00111111;
    default: last_strb  = 8'b01111111;
    endcase
    
    if(nbr_qwordsx8[11:3]==9'b1)
    begin
      first_strb = first_strb & last_strb;
      last_strb  = first_strb;
    end
  end
      

  //////////////////////////////////////////////////////////////////////////////
  //
  //  Address / Data
  //
  //////////////////////////////////////////////////////////////////////////////
  always @(posedge clk, negedge rst_n)
  begin
    if (rst_n==1'b0)
    begin
      start         <= 1'b0;
      saddr         <= 32'h0;

      awid          <= 4'h0;
      awaddr        <= 32'h0;
      awlen         <= 8'h0;
      awuser        <= 12'h0;
      awvalid       <= 1'b0;

      wdata         <= 64'h0;
      wstrb         <= 8'h0;
      wvalid        <= 1'b0;
      wlast         <= 1'b0;

      count         <= 9'h0;

      bytes_remain  <= 16'h0;
      aw_state      <= AW_IDLE;
    end
    else
    begin
      case (aw_state)
      /**************************************************************************
      * IDLE
      **************************************************************************/
      AW_IDLE:
      //$fsm_s In AW_IDLE state, the state machine waits until request is received
      begin
        if (channel_sel==1'b1 && channel_req==1'b1)
        begin
          start          <= 1'b1;
          saddr          <= channel_saddr;
    
          awid           <= channel_tag;
          awaddr         <= channel_daddr[31:0];
          awvalid        <= 1'b0;

          bytes_remain   <= channel_length;
    
          //$fsm_t When request is received, the state machine goes to AW_ADDR state.
          aw_state       <= AW_ADDR;
        end
      end
      /**************************************************************************
      * ADDR
      **************************************************************************/
      AW_ADDR:
      //$fsm_s In AW_ADDR state, the state machine goes to AW_ADDR_WAIT state
      begin
        start            <= 1'b1;
        awlen            <= nbr_qwordsx8[11:3]-9'd1;
        awuser           <= length;
        awvalid          <= 1'b1;

        //$fsm_t the state machine goes to AW_ADDR_WAIT state.
        aw_state         <= AW_ADDR_WAIT;
      end
      /**************************************************************************
      * ADDR WAIT
      **************************************************************************/
      AW_ADDR_WAIT:
      //$fsm_s In AW_ADDR_WAIT state, the state machine waits until address channel is ready
      begin
        count            <= nbr_qwordsx8[11:3];
    
        if (awready==1'b1)
        begin
          //$fsm_t When address channel is ready, the state machine goes to AW_DATA_FIRST state.
          awvalid        <= 1'b0;
          aw_state       <= AW_DATA_FIRST;
        end
      end
      /**************************************************************************
      * DATA FIRST
      **************************************************************************/
      AW_DATA_FIRST:
      //$fsm_s In AW_DATA_FIRST state, the state machine waits until first data is received
      begin
        if (datai_en==1'b1 && datai_rdy==1'b1)
        begin
          count          <= count-9'b1;
                         
          wdata          <= datai;
          wstrb          <= first_strb;
          wvalid         <= 1'b1;
    
          if (count==9'd1)
          begin
            //$fsm_t When first&last data is received, the state machine goes to AW_DATA_WAIT state.
            wlast        <= 1'b1;
            aw_state     <= AW_DATA_WAIT;
          end            
          else           
          begin          
            //$fsm_t When first data is received, the state machine goes to AW_DATA state.
            wlast        <= 1'b0;
            aw_state     <= AW_DATA;
          end
        end
      end
      /**************************************************************************
      * DATA
      **************************************************************************/
      AW_DATA:
      //$fsm_s In AW_DATA state, the state machine waits until last data is received
      begin
        if (datai_en==1'b1 && datai_rdy==1'b1)
        begin
          count          <= count-9'd1;
                         
          wdata          <= datai;
          wlast          <= 1'b0;
          wvalid         <= 1'b1;
    
          if (count==9'd1)
          begin
            //$fsm_t When last data is received, the state machine goes to AW_DATA_WAIT state.
            wstrb        <= last_strb;
            wlast        <= 1'b1;
            aw_state     <= AW_DATA_WAIT;
          end            
          else           
          begin          
            //$fsm_t When last data is not received, the state machine stays in AW_DATA state.
            wstrb        <= 8'hFF;
            aw_state     <= AW_DATA;
          end
        end
        else if (datai_en==1'b0 && datai_rdy==1'b1)
        begin
          wvalid         <= 1'b0;
        end
      end
      /**************************************************************************
      * DATA WAIT
      **************************************************************************/
      default://AW_DATA_WAIT:
      //$fsm_s In AW_DATA_WAIT state, the state machine waits until last data is sent
      begin
        start            <= 1'b0;
    
        if (wready==1'b1)
        begin
          saddr          <= saddr[31:0] +{20'h0,length};

          awaddr         <= awaddr[31:0]+{20'h0,length};

          wvalid         <= 1'b0;
    
          if (bytes_remain=={4'b0,length})
          begin
            //$fsm_t When last data is sent and no more transfert is requiered,
            // the state machine goes in AW_IDLE state.
            aw_state     <= AW_IDLE;
          end
          else
          begin
            //$fsm_t When last data is sent and new transfert is requiered,
            // the state machine goes in AW_ADDR state.
            bytes_remain <= bytes_remain-{4'b0,length};
            aw_state     <= AW_ADDR;
          end
        end
      end
      endcase
    end
  end

  assign daddr     = awaddr[2:0];

  assign awsize    = 3'b011;
  assign awburst   = 2'b01;

  assign wid       = awid;

  assign datai_rdy = (aw_state==AW_DATA_FIRST) || (aw_state==AW_DATA && wready);



  //////////////////////////////////////////////////////////////////////////////
  //
  //  Response
  //
  //////////////////////////////////////////////////////////////////////////////
  always @(posedge clk, negedge rst_n)
  begin
    if (rst_n==1'b0)
    begin
      channel_ack        <= 1'b0;
      channel_ack_tag    <= 4'h0;
      channel_ack_error  <= 1'b0;

      context_tag        <= 4'h0;

      bready             <= 1'b0;

      b_state            <= B_IDLE;
    end
    else
    begin
      case (b_state)
      /**************************************************************************
      * IDLE
      **************************************************************************/
      default://B_IDLE:
      //$fsm_s In B_IDLE state, the state machine waits until response is ready
      begin
        channel_ack        <= 1'b0;
        channel_ack_tag    <= 4'h0;
        channel_ack_error  <= 1'b0;
    
        if (bvalid==1'b1)
        begin
           //$fsm_t When response is ready, the state machine goes to B_INIT state.
           bready          <= 1'b1;
           context_tag     <= bid;
           b_state         <= B_INIT;
        end
      end
      /**************************************************************************
      * INIT
      **************************************************************************/
      B_INIT:
      //$fsm_s In B_INIT state, the state machine waits until response is received
      begin
        if (bvalid==1'b1 && bready==1'b1)
        begin
          bready                   <= 1'b0;
          if (bresp!=2'b00)
            //$fsm_t When error response is received, the state machine goes to B_ERROR state.
            b_state                <= B_ERROR;
          else if ((context_ack[context_tag]+14'd1) == context_req[context_tag])
            //$fsm_t When last response is received, the state machine goes to B_ACK state.
            b_state                <= B_ACK;
          else
            //$fsm_t When not last response is received, the state machine goes to B_IDLE state.
            b_state                <= B_IDLE;
        end
      end
      /**************************************************************************
      * ACK
      **************************************************************************/
      B_ACK:
      //$fsm_s In B_ACK state, the state machine waits until acknowledge is received
      begin
        channel_ack          <= 1'b1;
        channel_ack_tag      <= context_tag;
        channel_ack_error    <= 1'b0;
    
        if (channel_ack_ack==1'b1)
        begin
          //$fsm_t When acknowledge is received, the state machine goes to B_IDLE state.
          channel_ack        <= 1'b0;
          b_state            <= B_IDLE;
        end
      end
      /**************************************************************************
      * ERROR
      **************************************************************************/
      B_ERROR:
      //$fsm_s In B_ERROR state, the state machine waits until acknowledge is received
      begin
        channel_ack        <= 1'b1;
        channel_ack_tag    <= context_tag;
        channel_ack_error  <= 1'b1;
    
        if (channel_ack_ack==1'b1)
        begin
          //$fsm_t When acknowledge is received, the state machine goes to B_IDLE state.
          channel_ack      <= 1'b0;
          b_state          <= B_IDLE;
        end
      end
      endcase
    end
  end


  //////////////////////////////////////////////////////////////////////////////
  //
  //  Context
  //
  //////////////////////////////////////////////////////////////////////////////
  always @(posedge clk, negedge rst_n)
  begin: b_context
    integer v_i;

    if (rst_n==1'b0)
    begin
      for (v_i=0;v_i<16;v_i=v_i+1)
      begin
        context_req[v_i]    <= 14'h0;
        context_ack[v_i]    <= 14'h0;
      end
    end
    else
    begin
      // Context Init
      if (channel_sel==1'b1 && channel_req==1'b1)
      begin
        context_req[channel_tag]    <= 14'h0;
        context_ack[channel_tag]    <= 14'h0;
      end

      // Context Request Update
      if (aw_state==AW_ADDR)
      begin
        context_req[awid]           <= context_req[awid]+14'd1;

        `ifdef RW_SIMU_ON
          if (&context_req[awid])
          begin
            $display("error: context_req overflow detected !");
            $display("  context_req[%x] = %d",awid,context_req[awid]);
            $stop();
          end
        `endif // RW_SIMU_ON
      end

      // Context Acknowledge Update
      if (bvalid==1'b1 && bready==1'b1)
      begin
        context_ack[context_tag]    <= context_ack[context_tag]+14'd1;
      end
    end
  end


////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// pragma coverage block = off 
  reg [16*8-1:0] aw_state_str;
  reg [16*8-1:0] b_state_str;
  
  always @(*)
  begin
    case (aw_state)
    AW_IDLE:       aw_state_str = "AW_IDLE";
    AW_ADDR:       aw_state_str = "AW_ADDR";
    AW_ADDR_WAIT:  aw_state_str = "AW_ADDR_WAIT";
    AW_DATA_FIRST: aw_state_str = "AW_DATA_FIRST";
    AW_DATA:       aw_state_str = "AW_DATA";
    AW_DATA_WAIT:  aw_state_str = "AW_DATA_WAIT";
    default:       aw_state_str = "XXX";
    endcase    
  end
  
  always @(*)
  begin
    case (b_state)
    B_IDLE:        b_state_str  = "B_IDLE";
    B_INIT:        b_state_str  = "B_INIT";
    B_ACK:         b_state_str  = "B_ACK";
    B_ERROR:       b_state_str  = "B_ERROR";
    default:       b_state_str  = "XXX";
    endcase    
  end
// pragma coverage block = on 
`endif

endmodule
`default_nettype wire
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////

