////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of RivieraWaves Hardware Security Unit.
//                    This module is used to compute TKIP MIC and AES-128-CMAC.
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module rw_hsu( 
   //$port_g Clock and reset
   input  wire          clk,                     // Clock
   input  wire          rst_n,                   // Asynchronous Reset
 
   //$port_g Master AHB interface
   output wire   [31:0] m_haddr,                 // Master AHB haddr
   output wire   [ 1:0] m_htrans,                // Master AHB htrans
   output wire          m_hwrite,                // Master AHB hwrite
   output wire   [ 1:0] m_hsize,                 // Master AHB hsize
   output wire   [31:0] m_hwdata,                // Master AHB hwdata
   input  wire   [31:0] m_hrdata,                // Master AHB hrdata
   input  wire   [ 1:0] m_hresp,                 // Master AHB hresp
   input  wire          m_hready,                // Master AHB hready

   //$port_g Slave AHB interface
   input  wire          s_hready_in,             // Slave AHB hready_in
   input  wire          s_hsel,                  // Slave AHB hsel
   input  wire   [31:0] s_haddr,                 // Slave AHB haddr
   input  wire   [ 1:0] s_htrans,                // Slave AHB htrans
   input  wire          s_hwrite,                // Slave AHB hwrite
   input  wire   [ 1:0] s_hsize,                 // Slave AHB hsize
   input  wire   [31:0] s_hwdata,                // Slave AHB hwdata
   output wire   [31:0] s_hrdata,                // Slave AHB hrdata
   output wire   [ 1:0] s_hresp,                 // Slave AHB hresp
   output wire          s_hready,                // Slave AHB hready

`ifdef RW_HSU_RSA_EN
   output reg   [`RW_RSA_RAM_ADDWIDTH-1:0] sram_addr, // Sram address
   output reg  [`RW_RSA_RAM_DATAWIDTH-1:0] sram_din,  // Sram data input
   output reg           sram_rd_en,              // Sram read enable
   output reg     [3:0] sram_wr_en,              // Sram write enable
   input  wire [`RW_RSA_RAM_DATAWIDTH-1:0] sram_dout, // Sram data output
`endif // RW_HSU_RSA_EN

   //$port_g Interrupt
   output wire          done_irq,                // Interrupt (active high)

   //$port_g Debug Port
   output wire   [31:0] debug                    // Debug Port
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
localparam VERSION = 16'h0009;

localparam 
            HMAC1 = 4'd6,
            HMAC256 = 4'd7,
            HMAC224 = 4'd8,
            HMAC512 = 4'd11,
            HMAC384 = 4'd12;

`ifdef RW_HSU_SHA_EN
localparam  HSU_SHA_EN = 1'b1;
`else
localparam  HSU_SHA_EN = 1'b0;
`endif

`ifdef RW_HSU_IP_CHK_EN
localparam  HSU_IP_CHK_EN = 1'b1;
`else
localparam  HSU_IP_CHK_EN = 1'b0;
`endif

`ifdef RW_HSU_SHA512_EN
localparam  HSU_SHA512_EN = 1'b1;
`else
localparam  HSU_SHA512_EN = 1'b0;
`endif

`ifdef RW_HSU_RSA_EN
localparam  HSU_RSA1024_EN = 1'b1;
`else
localparam  HSU_RSA1024_EN = 1'b0;
`endif

`ifdef RW_RSA_4096
  localparam  HSU_RSA4096_EN = 1'b1;
  localparam  HSU_RSA2048_EN = 1'b1;
`else
  localparam  HSU_RSA4096_EN = 1'b0;
  `ifdef RW_RSA_2048
    localparam  HSU_RSA2048_EN = 1'b1;
  `else
    localparam  HSU_RSA2048_EN = 1'b0;
  `endif
`endif


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////

// Registers
wire         start;
wire         start_in_valid;
wire         last_buffer;
wire         first_buffer;
wire   [3:0] mode_sel;
wire   [4:0] mode;
wire  [31:0] source_addr;
wire  [15:0] length;
wire [127:0] key;
wire  [63:0] mic;
wire  [63:0] mic_in;
wire         mic_in_valid;
wire         done_set;
wire         done_clear;
wire   [1:0] remaining_length;
wire   [1:0] remaining_length_in;
wire         remaining_length_in_valid;
wire   [7:0] remaining_byte0;
wire   [7:0] remaining_byte0_in;
wire         remaining_byte0_in_valid;
wire   [7:0] remaining_byte1;
wire   [7:0] remaining_byte1_in;
wire         remaining_byte1_in_valid;
wire   [7:0] remaining_byte2;
wire   [7:0] remaining_byte2_in;
wire         remaining_byte2_in_valid;
`ifdef RW_HSU_SHA_EN
wire [255:0] sha_hash_value;
`endif
// Bus interface
wire         bus_enable;
wire         bus_pause;
wire  [31:0] bus_data;
wire   [2:0] bus_size;
wire         bus_last;
wire         bus_valid;

// TKIP MIC interface
wire         tkip_init;
wire  [63:0] tkip_key;
wire  [31:0] tkip_data;
wire   [3:0] tkip_data_byte_en;
wire         tkip_data_last;
wire         tkip_data_valid;
wire         tkip_data_ready;
wire  [63:0] tkip_mic;
wire         tkip_mic_valid;

// CMAC interface
wire         cmac_init;
wire         cmac_busy;
wire [127:0] cmac_data;
wire  [15:0] cmac_data_byte_en;
wire         cmac_data_last;
wire         cmac_data_valid;
wire         cmac_data_ready;
wire [127:0] cmac_mic;
wire         cmac_mic_valid;

`ifdef RW_HSU_IP_CHK_EN
// TCP interface
wire         ip_init;
wire  [31:0] ip_data;
wire   [3:0] ip_data_byte_en;
wire         ip_data_last;
wire         ip_data_valid;
wire         ip_data_ready;
wire  [15:0] ip_checksum;
wire         ip_checksum_valid;
`endif // RW_HSU_IP_CHK_EN

`ifdef RW_HSU_SHA_EN
// SHA1 interface
wire         hmac_init;
wire   [2:0] sha_mode;
wire	     sha_init;
wire  [31:0] sha_data;
wire   [3:0] sha_data_byte_en;
wire         sha_last_data;
wire         sha_data_valid;
wire  [63:0] sha_length;
wire         sha_data_ready;
wire [255:0] sha_hash;
wire         sha_hash_valid;
wire   [9:0] hmac_key_len;

wire   [2:0] sha_mode_mux;
wire	     sha_init_mux;
wire  [31:0] sha_data_mux;
wire   [3:0] sha_data_byte_en_mux;
wire         sha_last_data_mux;
wire         sha_data_valid_mux;
wire  [63:0] sha_length_mux;
wire         sha_data_ready_mux;
wire [255:0] sha_hash_mux;
wire         sha_hash_valid_mux;

wire   [2:0] sha_mode_hmac;
wire	     sha_init_hmac;
wire  [31:0] sha_data_hmac;
wire   [3:0] sha_last_data_len_hmac;
wire         sha_last_data_hmac;
wire         sha_data_valid_hmac;
wire  [63:0] sha_length_hmac;
wire         sha_data_ready_hmac;
wire [255:0] sha_hash_hmac;
wire         sha_hash_valid_hmac;
wire         hmac_en;
`ifdef RW_HSU_SHA512_EN
wire [255:0] sha_hash_value2;
wire [255:0] sha_hash2;
wire [255:0] sha_hash2_hmac;
wire  [63:0] sha_data_512;
wire [255:0] sha_hash2_mux;
wire  [63:0] sha_data_512_mux;
wire  [63:0] sha_data_512_hmac;
`endif //RW_HSU_SHA512_EN
`endif //RW_HSU_SHA_EN

// AES interface
wire [127:0] aes_data_out;
wire         aes_data_out_valid;
wire [127:0] aes_data_in;
wire         aes_data_in_valid;

`ifdef RW_HSU_RSA_EN
// RSA interface
wire  [`RW_RSA_RAM_ADDWIDTH-1:0] rsa_addr; 
wire [`RW_RSA_RAM_DATAWIDTH-1:0] rsa_din;
wire                             rsa_rd_en;
wire                       [3:0] rsa_wr_en;
reg  [`RW_RSA_RAM_DATAWIDTH-1:0] rsa_dout;

// HSU_CTRL output
wire  [`RW_RSA_RAM_ADDWIDTH-1:0] hsu_ctrl_addr; 
wire [`RW_RSA_RAM_DATAWIDTH-1:0] hsu_ctrl_din;
wire                       [3:0] hsu_ctrl_wr_en;

// HSU_AHB_output
wire                             hsu_ahb_rd_en;
wire  [`RW_RSA_RAM_ADDWIDTH-1:0] hsu_ahb_addr;
reg  [`RW_RSA_RAM_DATAWIDTH-1:0] hsu_ahb_dout;
wire         ahb_write_end;
wire         rsa_res_valid;
wire   [1:0] rsa_mode;
wire         rsa_init_ready;
wire         rsa_init;
reg          ahb_to_sram;
`endif //RW_HSU_RSA_EN

`ifdef  RW_HSU_RSA_EN
wire  [31:0] dest_addr;
`endif//RW_HSU_RSA_EN

`ifdef RW_HSU_SHA_EN
//signals for changing endianness
wire [31:0] sha0_reg;
wire [31:0] sha1_reg;
wire [31:0] sha2_reg;
wire [31:0] sha3_reg;
wire [31:0] sha4_reg;
wire [31:0] sha5_reg;
wire [31:0] sha6_reg;
wire [31:0] sha7_reg;

assign sha0_reg = {sha_hash_value[231:224], sha_hash_value[239:232], sha_hash_value[247:240], sha_hash_value[255:248]};
assign sha1_reg = {sha_hash_value[199:192], sha_hash_value[207:200], sha_hash_value[215:208], sha_hash_value[223:216]};
assign sha2_reg = {sha_hash_value[167:160], sha_hash_value[175:168], sha_hash_value[183:176], sha_hash_value[191:184]};
assign sha3_reg = {sha_hash_value[135:128], sha_hash_value[143:136], sha_hash_value[151:144], sha_hash_value[159:152]};
assign sha4_reg = {sha_hash_value[103: 96], sha_hash_value[111:104], sha_hash_value[119:112], sha_hash_value[127:120]};
assign sha5_reg = {sha_hash_value[ 71: 64], sha_hash_value[ 79: 72], sha_hash_value[ 87: 80], sha_hash_value[ 95: 88]};
assign sha6_reg = {sha_hash_value[ 39: 32], sha_hash_value[ 47: 40], sha_hash_value[ 55: 48], sha_hash_value[ 63: 56]};
assign sha7_reg = {sha_hash_value[  7:  0], sha_hash_value[ 15:  8], sha_hash_value[ 23: 16], sha_hash_value[ 31: 24]};
`ifdef RW_HSU_SHA512_EN
wire [31:0] sha8_reg;
wire [31:0] sha9_reg;
wire [31:0] sha10_reg;
wire [31:0] sha11_reg;
wire [31:0] sha12_reg;
wire [31:0] sha13_reg;
wire [31:0] sha14_reg;
wire [31:0] sha15_reg;
assign sha8_reg  = {sha_hash_value2[231:224], sha_hash_value2[239:232], sha_hash_value2[247:240], sha_hash_value2[255:248]};
assign sha9_reg  = {sha_hash_value2[199:192], sha_hash_value2[207:200], sha_hash_value2[215:208], sha_hash_value2[223:216]};
assign sha10_reg = {sha_hash_value2[167:160], sha_hash_value2[175:168], sha_hash_value2[183:176], sha_hash_value2[191:184]};
assign sha11_reg = {sha_hash_value2[135:128], sha_hash_value2[143:136], sha_hash_value2[151:144], sha_hash_value2[159:152]};
assign sha12_reg = {sha_hash_value2[103: 96], sha_hash_value2[111:104], sha_hash_value2[119:112], sha_hash_value2[127:120]};
assign sha13_reg = {sha_hash_value2[ 71: 64], sha_hash_value2[ 79: 72], sha_hash_value2[ 87: 80], sha_hash_value2[ 95: 88]};
assign sha14_reg = {sha_hash_value2[ 39: 32], sha_hash_value2[ 47: 40], sha_hash_value2[ 55: 48], sha_hash_value2[ 63: 56]};
assign sha15_reg = {sha_hash_value2[  7:  0], sha_hash_value2[ 15:  8], sha_hash_value2[ 23: 16], sha_hash_value2[ 31: 24]};
`endif //RW_HSU_SHA512_EN

`endif //RW_HSU_SHA_EN

////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

//******************************************************************************
// Registers
//******************************************************************************
rw_hsu_reg u_rw_hsu_reg(
   // Clock and reset
   .ahb_clk                   (clk                       ),
   .ahb_nrst                  (rst_n                     ),
   // Registers
   .version                   (VERSION                   ),
   .start_in                  (1'b0                      ),
   .start_in_valid            (start_in_valid            ),
   .done_set                  (done_set                  ),
   .mic0                      (mic[31:0]                 ),
   .mic1                      (mic[63:32]                ),
   .mic0_in                   (mic_in[31:0]              ),
   .mic1_in                   (mic_in[63:32]             ),

`ifdef  RW_HSU_SHA_EN
   .sha0                      (sha0_reg                  ),
   .sha1                      (sha1_reg                  ),
   .sha2                      (sha2_reg                  ),
   .sha3                      (sha3_reg                  ),
   .sha4                      (sha4_reg                  ),
   .sha5                      (sha5_reg                  ),
   .sha6                      (sha6_reg                  ),
   .sha7                      (sha7_reg                  ),
`ifdef  RW_HSU_SHA512_EN
   .sha8                      (sha8_reg                  ),
   .sha9                      (sha9_reg                  ),
   .sha10                     (sha10_reg                 ),
   .sha11                     (sha11_reg                 ),
   .sha12                     (sha12_reg                 ),
   .sha13                     (sha13_reg                 ),
   .sha14                     (sha14_reg                 ),
   .sha15                     (sha15_reg                 ),
`else //RW_HSU_SHA512_EN
   .sha8                      (32'b0                     ),
   .sha9                      (32'b0                     ),
   .sha10                     (32'b0                     ),
   .sha11                     (32'b0                     ),
   .sha12                     (32'b0                     ),
   .sha13                     (32'b0                     ),
   .sha14                     (32'b0                     ),
   .sha15                     (32'b0                     ),
`endif//RW_HSU_SHA512_EN
`endif//RW_HSU_SHA_EN
   .rsa_4096                  (HSU_RSA4096_EN            ),
   .rsa_2048                  (HSU_RSA2048_EN            ),
   .rsa_1024                  (HSU_RSA1024_EN            ),
   .hmac_sha512_sha384        (HSU_SHA512_EN             ),
   .hmac_sha256_sha224        (HSU_SHA_EN                ),
   .hmac_sha1                 (HSU_SHA_EN                ),
   .sha_512_384               (HSU_SHA512_EN             ),
   .sha_256_224               (HSU_SHA_EN                ),
   .sha_1                     (HSU_SHA_EN                ),
   .ip_chk                    (HSU_IP_CHK_EN             ),
   .aes_128_cmac              (1'b1                      ),
   .tkip_mic                  (1'b1                      ),
   .mic0_in_valid             (mic_in_valid              ),
   .mic1_in_valid             (mic_in_valid              ),
   .remaining_length          (remaining_length          ),
   .remaining_length_in       (remaining_length_in       ),
   .remaining_length_in_valid (remaining_length_in_valid ),
   .remaining_byte0           (remaining_byte0           ),
   .remaining_byte0_in        (remaining_byte0_in        ),
   .remaining_byte0_in_valid  (remaining_byte0_in_valid  ),
   .remaining_byte1           (remaining_byte1           ),
   .remaining_byte1_in        (remaining_byte1_in        ),
   .remaining_byte1_in_valid  (remaining_byte1_in_valid  ),
   .remaining_byte2           (remaining_byte2           ),
   .remaining_byte2_in        (remaining_byte2_in        ),
   .remaining_byte2_in_valid  (remaining_byte2_in_valid  ),
   .last_buffer               (last_buffer               ),
   .first_buffer              (first_buffer              ),
   .mode                      (mode                      ),
   .start                     (start                     ),
   .done_clear                (done_clear                ),
   .done_clear_in             (1'b0                      ),
   .done_clear_in_valid       (done_clear                ),
   .key0                      (key[31:0]                 ),
   .key1                      (key[63:32]                ),
   .key2                      (key[95:64]                ),
   .key3                      (key[127:96]               ),
   .source_addr               (source_addr               ),
`ifdef  RW_HSU_RSA_EN
   .dest_addr                 (dest_addr                 ),
`endif//RW_HSU_RSA_EN
   .length                    (length                    ),
   // AHB Slave interface
   .hready_in                 (s_hready_in               ),
   .hsel                      (s_hsel                    ),
   .haddr                     (s_haddr[6:0]              ),
   .htrans                    (s_htrans                  ),
   .hwrite                    (s_hwrite                  ),
   .hwdata                    (s_hwdata                  ),
   .hrdata                    (s_hrdata                  ),
   .hresp                     (s_hresp                   ),
   .hready                    (s_hready                  )
);


//******************************************************************************
// Master AHB Interface
//******************************************************************************
rw_hsu_ahb u_rw_hsu_ahb( 
   //Clock and reset
   .clk                       (clk                       ),
   .rst_n                     (rst_n                     ),
   //Master AHB interface                       
   .m_haddr                   (m_haddr                   ),
   .m_htrans                  (m_htrans                  ),
   .m_hwrite                  (m_hwrite                  ),
   .m_hsize                   (m_hsize                   ),
   .m_hwdata                  (m_hwdata                  ),
   .m_hrdata                  (m_hrdata                  ),
   .m_hresp                   (m_hresp                   ),
   .m_hready                  (m_hready                  ),

`ifdef RW_HSU_RSA_EN
   //$port_g SRAM and RSA interface
   .ren                       (hsu_ahb_rd_en             ),
   .addr                      (hsu_ahb_addr              ),
   .rdata                     (hsu_ahb_dout              ),
   .ahb_write_end             (ahb_write_end             ),
   .rsa_res_valid             (rsa_res_valid             ),
   .dest_addr                 (dest_addr                 ),
`endif // RW_HSU_RSA_EN

   //Bus interface                              
   .bus_enable                (bus_enable                ),
   .bus_pause                 (bus_pause                 ),
   .bus_data                  (bus_data                  ),
   .bus_size                  (bus_size                  ),
   .bus_last                  (bus_last                  ),
   .bus_valid                 (bus_valid                 ),
   //Register                                   
   .source_addr               (source_addr               ),
   .length                    (length                    ) 
);


//******************************************************************************
// Control
//******************************************************************************
rw_hsu_ctrl u_rw_hsu_ctrl( 
   //Clock and reset
   .clk                       (clk                       ),
   .rst_n                     (rst_n                     ),
   //Bus interface
   .bus_enable                (bus_enable                ),
   .bus_pause                 (bus_pause                 ),
   .bus_data                  (bus_data                  ),
   .bus_size                  (bus_size                  ),
   .bus_last                  (bus_last                  ),
   .bus_valid                 (bus_valid                 ),
   //TKIP MIC interface
   .tkip_init                 (tkip_init                 ),
   .tkip_key                  (tkip_key                  ),
   .tkip_data                 (tkip_data                 ),
   .tkip_data_byte_en         (tkip_data_byte_en         ),
   .tkip_data_last            (tkip_data_last            ),
   .tkip_data_valid           (tkip_data_valid           ),
   .tkip_data_ready           (tkip_data_ready           ),
   .tkip_mic                  (tkip_mic                  ),
   .tkip_mic_valid            (tkip_mic_valid            ),
   //CMAC interface
   .cmac_init                 (cmac_init                 ),
   .cmac_busy                 (cmac_busy                 ),
   .cmac_data                 (cmac_data                 ),
   .cmac_data_byte_en         (cmac_data_byte_en         ),
   .cmac_data_last            (cmac_data_last            ),
   .cmac_data_valid           (cmac_data_valid           ),
   .cmac_data_ready           (cmac_data_ready           ),
   .cmac_mic                  (cmac_mic[63:0]            ),
   .cmac_mic_valid            (cmac_mic_valid            ),
`ifdef RW_HSU_IP_CHK_EN
   //TCP interface
   .ip_init                   (ip_init                   ),
   .ip_data                   (ip_data                   ),
   .ip_data_byte_en           (ip_data_byte_en           ),
   .ip_data_valid             (ip_data_valid             ),
   .ip_data_last              (ip_data_last              ),
   .ip_data_ready             (ip_data_ready             ),
   .ip_checksum               (ip_checksum               ),
   .ip_checksum_valid         (ip_checksum_valid         ),
`endif // RW_HSU_IP_CHK_EN
`ifdef RW_HSU_SHA_EN
   //SHA1 interface
   .sha_mode                  (sha_mode                  ),
   .sha_init                  (sha_init                  ),
   .sha_data                  (sha_data                  ),
   .sha_data_byte_en          (sha_data_byte_en          ),
   .sha_data_valid            (sha_data_valid            ),
   .sha_last_data             (sha_last_data             ),
   .sha_data_ready            (sha_data_ready_mux        ),
   .sha_length                (sha_length                ),
   .sha_hash_valid            (sha_hash_valid_mux        ),
   .sha_hash		          (sha_hash_mux              ),
   .sha_hash_value            (sha_hash_value            ),
   .hmac_key_len              (hmac_key_len              ),
`ifdef RW_HSU_SHA512_EN
   .sha_data_512              (sha_data_512              ),
   .sha_hash2                 (sha_hash2_mux             ),
   .sha_hash_value2           (sha_hash_value2           ),
`endif // RW_HSU_SHA512_EN
`endif // RW_HSU_SHA_EN
`ifdef RW_HSU_RSA_EN
   .rsa_sram_addr             (hsu_ctrl_addr             ),
   .rsa_wr_en                 (hsu_ctrl_wr_en            ),
   .rsa_din                   (hsu_ctrl_din              ),
   .rsa_mode                  (rsa_mode                  ),
   .rsa_init_ready            (rsa_init_ready            ),
   .rsa_init                  (rsa_init                  ),
   .rsa_valid                 (ahb_write_end             ),
`endif //RW_HSU_RSA_EN
   //Registers
   .start                     (start                     ),
   .start_in_valid            (start_in_valid            ),
   .mode                      (mode                      ),
   .first_buffer              (first_buffer              ),
   .last_buffer               (last_buffer               ),
   .length                    (length                    ),
   .done_set                  (done_set                  ),
   .done_clear                (done_clear                ),
   .key                       (key                       ),
   .mic                       (mic                       ),
   .mic_in                    (mic_in                    ),
   .mic_in_valid              (mic_in_valid              ),
   .remaining_byte0           (remaining_byte0           ),
   .remaining_byte0_in        (remaining_byte0_in        ),
   .remaining_byte0_in_valid  (remaining_byte0_in_valid  ),
   .remaining_byte1           (remaining_byte1           ),
   .remaining_byte1_in        (remaining_byte1_in        ),
   .remaining_byte1_in_valid  (remaining_byte1_in_valid  ),
   .remaining_byte2           (remaining_byte2           ),
   .remaining_byte2_in        (remaining_byte2_in        ),
   .remaining_byte2_in_valid  (remaining_byte2_in_valid  ),
   .remaining_length          (remaining_length          ),
   .remaining_length_in       (remaining_length_in       ),
   .remaining_length_in_valid (remaining_length_in_valid )
);


//******************************************************************************
// TKIP MIC
//******************************************************************************
tkip_mic u_tkip_mic( 
   //Clock and reset
   .clk                       (clk                       ),
   .rst_n                     (rst_n                     ),
   //Control interface
   .tkip_init                 (tkip_init                 ),
   .tkip_key                  (tkip_key                  ),
   .tkip_data                 (tkip_data                 ),
   .tkip_data_byte_en         (tkip_data_byte_en         ),
   .tkip_data_last            (tkip_data_last            ),
   .tkip_data_valid           (tkip_data_valid           ),
   .tkip_data_ready           (tkip_data_ready           ),
   .tkip_mic                  (tkip_mic                  ),
   .tkip_mic_valid            (tkip_mic_valid            )
);


//******************************************************************************
// CMAC
//******************************************************************************
cmac u_cmac( 
   //Clock and reset
   .clk                       (clk                       ),
   .hard_rst_n                (rst_n                     ),
   .soft_rst_n                (1'b1                      ),
   //Control interface
   .cmac_init                 (cmac_init                 ),
   .cmac_busy                 (cmac_busy                 ),
   .cmac_data                 (cmac_data                 ),
   .cmac_data_byte_en         (cmac_data_byte_en         ),
   .cmac_data_last            (cmac_data_last            ),
   .cmac_data_valid           (cmac_data_valid           ),
   .cmac_data_ready           (cmac_data_ready           ),
   .cmac_mic                  (cmac_mic                  ),
   .cmac_mic_valid            (cmac_mic_valid            ),
   .cmac_mic_ready            (1'b1                      ),
   //AES interface
   .aes_data_out              (aes_data_out              ),
   .aes_data_out_valid        (aes_data_out_valid        ),
   .aes_data_in               (aes_data_in               ),
   .aes_data_in_valid         (aes_data_in_valid         ),
   //Context interface
   .context_in                (256'h0                    ),
   .context_in_valid          (1'b0                      ),
   .context_out               (/*Open*/                  )
);


//******************************************************************************
// AES-128
//******************************************************************************
aesCore u_aesCore (
   //Clocks and Resets
   .pClk                      (clk                       ),
   .nPRst                     (rst_n                     ),
   .nSRst                     (1'b1                      ),
   //ccmCntl
   .abort_p                   (1'b0                      ),
   .aesKey                    ({128'h0,key}              ),
   .aesKeyLen                 (2'b00/*128bits*/          ),
   .aesInData                 (aes_data_in               ),
   .aesInValid                (aes_data_in_valid         ),
   .aesOutData                (aes_data_out              ),
   .aesOutValid_p             (aes_data_out_valid        )
);

`ifdef RW_HSU_IP_CHK_EN
//******************************************************************************
// IP checksum * the same as TCP checksum
//******************************************************************************
ip_checksum u_ip_checksum(
   //Clock and reset
   .clk                       (clk                       ),
   .rst_n                     (rst_n                     ),
   //Control interface
   .ip_init                   (ip_init                   ),
   .ip_data                   (ip_data                   ),
   .ip_byte_en                (ip_data_byte_en           ),
   .ip_data_valid             (ip_data_valid             ),
   .ip_data_last              (ip_data_last              ),
   .ip_data_ready             (ip_data_ready             ),
   .ip_checksum               (ip_checksum               ),
   .ip_checksum_valid         (ip_checksum_valid         )
);
`endif // RW_HSU_IP_CHK_EN

assign mode_sel = mode[3:0];

`ifdef RW_HSU_SHA_EN

//HMAC-SHA or SHA selection
assign hmac_en = (mode_sel==HMAC1 || mode_sel==HMAC256 || mode_sel==HMAC224 || mode_sel==HMAC512 || mode_sel==HMAC384) ? 1'b1 : 1'b0;
assign sha_mode_mux = hmac_en ? sha_mode_hmac : sha_mode;
assign sha_init_mux = hmac_en ? sha_init_hmac : sha_init;
assign sha_data_mux = hmac_en ? sha_data_hmac : sha_data;
assign sha_data_byte_en_mux = hmac_en ? sha_last_data_len_hmac : sha_data_byte_en;
assign sha_data_valid_mux = hmac_en ? sha_data_valid_hmac : sha_data_valid;
assign sha_last_data_mux = hmac_en ? sha_last_data_hmac : sha_last_data;
assign sha_length_mux = hmac_en ? sha_length_hmac : sha_length;
assign sha_data_ready_mux = hmac_en ? sha_data_ready_hmac : sha_data_ready;
assign sha_hash_mux = hmac_en ? sha_hash_hmac : sha_hash;
assign sha_hash_valid_mux = hmac_en ? sha_hash_valid_hmac : sha_hash_valid;
`ifdef RW_HSU_SHA512_EN 
assign sha_hash2_mux = hmac_en ? sha_hash2_hmac : sha_hash2;
assign sha_data_512_mux = hmac_en ? sha_data_512_hmac : sha_data_512;
`endif //RW_HSU_SHA512_EN

//******************************************************************************
// SHA1 hash value
//******************************************************************************
sha u_sha(
   //Clock and reset
   .clk                       (clk                       ),
   .rst_n                     (rst_n                     ),
   //Control interface
   .mode                      (sha_mode_mux              ),
   .sha_init                  (sha_init_mux              ),
   .sha_data                  (sha_data_mux              ),
   .sha_data_byte_en          (sha_data_byte_en_mux      ),
   .sha_data_valid            (sha_data_valid_mux        ),
   .sha_last_data             (sha_last_data_mux         ),
   .sha_data_ready            (sha_data_ready            ),
   .sha_length                (sha_length_mux            ),
   .sha_hash_valid            (sha_hash_valid            ),
`ifdef RW_HSU_SHA512_EN 
   .sha_data_512              (sha_data_512_mux          ),
   .sha_hash2                 (sha_hash2                 ),
`endif //RW_HSU_SHA512_EN
   .sha_hash                  (sha_hash                  )

);

//******************************************************************************
// HMAC SHA hash value
//******************************************************************************
assign hmac_init = hmac_en ? sha_init : 1'b0;
hmac U_hmac (
    .clk                      (clk                       ),
    .rst_n                    (rst_n                     ),
    .hmac_mode                (sha_mode                  ),
    .hmac_init                (hmac_init                 ),
    .hmac_data                (sha_data                  ),
    .hmac_last_data_len       (sha_data_byte_en          ),
    .hmac_last_data           (sha_last_data             ),
    .hmac_data_valid          (sha_data_valid            ),
    .hmac_length              (sha_length                ),
    .hmac_key_len             (hmac_key_len              ),
`ifdef RW_HSU_SHA512_EN
    .hmac_data_512            (sha_data_512              ),
    .hmac_hash2               (sha_hash2_hmac            ),
`endif
    .hmac_data_ready          (sha_data_ready_hmac       ),
    .hmac_hash                (sha_hash_hmac             ),
    .hmac_hash_valid          (sha_hash_valid_hmac       ),
//sha interface
    .sha_mode                 (sha_mode_hmac             ),
    .sha_init                 (sha_init_hmac             ),
    .sha_data                 (sha_data_hmac             ),
    .sha_last_data            (sha_last_data_hmac        ),
    .sha_data_valid           (sha_data_valid_hmac       ),
    .sha_last_data_len        (sha_last_data_len_hmac    ),
    .sha_length               (sha_length_hmac           ),
`ifdef RW_HSU_SHA512_EN
    .sha_data_512             (sha_data_512_hmac         ),
    .sha_hash2                (sha_hash2                 ),
`endif
    .sha_data_ready           (sha_data_ready            ),
    .sha_hash                 (sha_hash                  ),
    .sha_hash_valid           (sha_hash_valid            )
);
`endif //RW_HSU_SHA_EN

`ifdef RW_HSU_RSA_EN
// RSA Instanciation
rsa U_rsa (
    //$port_g Clock and reset
    .clk                      (clk                       ),  
    .rst_n                    (rst_n                     ),

    //$port_g Memory interface
    .ren                      (rsa_rd_en                 ),
    .wen                      (rsa_wr_en                 ),
    .addr                     (rsa_addr                  ),
    .wdata                    (rsa_din                   ),
    .rdata                    (rsa_dout                  ),

    //$port_g Control interface
    .rsa_res_valid            (rsa_res_valid             ),
    .rsa_init                 (rsa_init                  ),
    .rsa_mode                 (rsa_mode                  )
);


always @*
begin
    // Write Only
    if((rsa_init_ready==1'b0) && (ahb_to_sram==1'b0))
    begin
        sram_din = hsu_ctrl_din;
        sram_addr = hsu_ctrl_addr;
        sram_rd_en = 1'b0;
        sram_wr_en = hsu_ctrl_wr_en;
        rsa_dout = {`RW_RSA_RAM_DATAWIDTH{1'b0}};
        hsu_ahb_dout = {`RW_RSA_RAM_DATAWIDTH{1'b0}};
    end
    // Write and Read
    else if((rsa_init_ready==1'b1) && (ahb_to_sram==1'b0))
    begin
        sram_din = rsa_din;
        sram_addr = rsa_addr;
        sram_rd_en = rsa_rd_en;
        sram_wr_en = rsa_wr_en;
        rsa_dout = sram_dout;
        hsu_ahb_dout = {`RW_RSA_RAM_DATAWIDTH{1'b0}};
    end
    // Rread Only
    else
    begin
        sram_din = {`RW_RSA_RAM_DATAWIDTH{1'b0}};
        sram_addr = hsu_ahb_addr;
        sram_rd_en = hsu_ahb_rd_en;
        sram_wr_en = 4'b0;
        rsa_dout = {`RW_RSA_RAM_DATAWIDTH{1'b0}};
        hsu_ahb_dout = sram_dout;
    end
end

always @(posedge clk or negedge rst_n)
begin
  if (rst_n==1'b0)
     ahb_to_sram <= 1'b0;
  else
     if(bus_enable)
        ahb_to_sram <= 1'b0;
     else if (ahb_to_sram==1'b0 && rsa_res_valid)
        ahb_to_sram <= 1'b1;
end


`endif //RW_HSU_RSA_EN

//******************************************************************************
// Interrupt
//******************************************************************************
assign done_irq = done_set;


//******************************************************************************
// Debug Port
//******************************************************************************
assign debug = 32'h0;


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
`endif // RW_SIMU_ON


///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef RW_ASSERT_ON
`endif // RW_ASSERT_ON

endmodule
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
