////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : HSU master AHB interface.
//                    This module read data from AHB and provide it to the 
//                    control module.
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module rw_hsu_ahb( 
   //$port_g Clock and reset
   input  wire          clk,                        // Clock
   input  wire          rst_n,                      // Asynchronous Reset
 
   //$port_g Master AHB interface
   output reg    [31:0] m_haddr,                    // Master AHB haddr
   output reg    [ 1:0] m_htrans,                   // Master AHB htrans
   output wire          m_hwrite,                   // Master AHB hwrite
   output wire   [ 1:0] m_hsize,                    // Master AHB hsize
   output wire   [31:0] m_hwdata,                   // Master AHB hwdata
   input  wire   [31:0] m_hrdata,                   // Master AHB hrdata
   input  wire   [ 1:0] m_hresp,                    // Master AHB hresp
   input  wire          m_hready,                   // Master AHB hready

   //$port_g Bus interface
   input  wire          bus_enable,                 // Read enable
   input  wire          bus_pause,                  // Read Pause
   output reg    [31:0] bus_data,                   // Read Data
   output reg     [2:0] bus_size,                   // Read size in bytes
   output reg           bus_last,                   // Read data is the last
   output reg           bus_valid,                  // Read Data is valid

`ifdef RW_HSU_RSA_EN
   //$port_g SRAM and RSA interface
   output wire          ren,                        // Sram enable
   output wire          ahb_write_end,              // RSA result finish
   input  wire          rsa_res_valid,              // RSA result
   output reg     [`RW_RSA_RAM_ADDWIDTH-1:0] addr,  // Sram address
   input  wire   [`RW_RSA_RAM_DATAWIDTH-1:0] rdata, // Sram data to read
   
   //$port_g register
   input  wire   [31:0] dest_addr,                  // Destination addresss of RSA results
`endif // RW_HSU_RSA_EN

   //$port_g register
   input  wire   [31:0] source_addr,                // Base address of the buffer
   input  wire   [15:0] length                      // Number of bytes in the buffer
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
// FSM states definition
//$fsm_sd ahb
localparam 
   AHB_IDLE  = 3'd0,
   AHB_ADDR  = 3'd1,
   AHB_DATA  = 3'd2,
   AHB_PAUSE = 3'd3,
   AHB_DONE  = 3'd4;


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
reg   [2:0] ahb_cs;              // AHB FSM current state
reg   [2:0] ahb_ns;              // AHB FSM next state   

reg   [1:0] m_haddr10;           // Read address offset

wire [16:0] byte_aligned;        // Number of byte to read, included first unused byte
wire [14:0] word_nbr;            // Number of word to read (32bits)

reg  [14:0] count;               // Word counter
wire [14:0] next_count;          // Incremented word counter

`ifdef RW_HSU_RSA_EN
wire [`RW_RSA_RAM_ADDWIDTH-1:0]
            sram_res_addr;       // SRAM results initial address
wire [`RW_RSA_RAM_DATAWIDTH-1:0]
            sram_rdata_shift;    // SRAM shifted data
wire [15:0] res_addr_begin;      // To calculate the address of result in SRAM
  `ifdef RW_RSA_RAM_DATAWIDTH_256
  wire  [4:0] count_1;           // Counter flag
  `elsif RW_RSA_RAM_DATAWIDTH_128 
  wire  [3:0] count_1;           // Counter flag
  `else
  wire  [2:0] count_1;           // Counter flag
  `endif
reg   [1:0] m_hwsize_tmp;        // Size of data to be written
reg  [31:0] m_hwdata_tmp;        // Data to be written temporary
reg  [14:0] ahb_write_count;     // Counter used when writing
reg  [23:0] ahb_wdata_buf;       // Data buffer to write
reg         ahb_write_en;        // Ahb write enable
`endif // RW_HSU_RSA_EN

////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////

//******************************************************************************
// Compute number of word to read (32bits)
//******************************************************************************
assign byte_aligned = {1'b0,length} + {15'h0,source_addr[1:0]};
`ifdef RW_HSU_RSA_EN
assign word_nbr     = (ahb_write_en==1'b1) ? length[14:0] : byte_aligned[16:2]+{14'h0,|byte_aligned[1:0]};
`else
assign word_nbr     = byte_aligned[16:2]+{14'h0,|byte_aligned[1:0]};
`endif

//******************************************************************************
// AHB FSM current state logic
//******************************************************************************
always @(posedge clk or negedge rst_n) 
begin
  if (rst_n==1'b0)
     ahb_cs <= AHB_IDLE;
  else
     ahb_cs <= ahb_ns;
end


//******************************************************************************
// AHB FSM next state logic
//******************************************************************************
always @* 
begin
   case(ahb_cs)
   AHB_IDLE:
   begin
      //$fsm_s In AHB_IDLE state, the state machine waits the start trigger
`ifdef RW_HSU_RSA_EN
      if (bus_enable || ahb_write_en)
`else
      if (bus_enable)
`endif
         //$fsm_t When start trigger is received, the state machine goes to 
         //AHB_ADDR state
         ahb_ns = AHB_ADDR;
      else
         //$fsm_t While start trigger is not received, the state machine stays
         //in AHB_IDLE state
         ahb_ns = AHB_IDLE;
   end

   AHB_ADDR:
   begin
      //$fsm_s In AHB_ADDR state, the state machine waits hready
      if (m_hready)
         //$fsm_t When hready is high, the state machine goes to AHB_DATA state
         ahb_ns = AHB_DATA;
      else
         //$fsm_t While hready is low, the state machine stays in AHB_ADDR state
         ahb_ns = AHB_ADDR;
   end

   AHB_DATA:
   begin
      //$fsm_s In AHB_DATA state, the state machine waits hready
      if (m_hready)
      begin
`ifdef RW_HSU_RSA_EN
         if (((next_count==word_nbr) && (ahb_write_en==1'b0)) || 
                        ((next_count==length[14:0]) && (ahb_write_en==1'b1)))
`else
         if (next_count==word_nbr)
`endif //RW_HSU_RSA_EN
            //$fsm_t When stop is requested, the state machine goes to AHB_DONE
            //state
            ahb_ns = AHB_DONE;
         else if (bus_pause)
            //$fsm_t When pause is requested, the state machine goes to AHB_PAUSE
            //state
            ahb_ns = AHB_PAUSE;
         else
            //$fsm_t When hready is high, the state machine goes to AHB_ADDR
            //state
            ahb_ns = AHB_ADDR;
      end
      else
         //$fsm_t While heady is low, the state machine stays in AHB_DATA state
         ahb_ns = AHB_DATA;
   end

   AHB_PAUSE:
   begin
      //$fsm_s In AHB_PAUSE state, the state machine waits end of pause
      if (!bus_pause)
         //$fsm_t When end of pause, the state machine goes to AHB_ADDR state
         ahb_ns = AHB_ADDR;
      else
         //$fsm_t While pause is requested, the state machine stays in AHB_PAUSE
         //state
         ahb_ns = AHB_PAUSE;
   end

   AHB_DONE:
   begin
      //$fsm_s In AHB_DONE state, the state machine waits processing is done
      if (!bus_enable)
         //$fsm_t When processing is done, the state machine goes to AHB_IDLE
         //state
         ahb_ns = AHB_IDLE;
      else
         //$fsm_t While processing is not done, the state machine stays in
         //AHB_DONE state
         ahb_ns = AHB_DONE;
   end

   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default:   
      ahb_ns = AHB_IDLE;
   // pragma coverage block = on
   endcase
end


//******************************************************************************
// AHB interface registered output
//******************************************************************************

`ifdef RW_HSU_RSA_EN
  //******************************************************************************
  // Compute the initial address of RSA encryption result
  //******************************************************************************
  `ifdef RW_RSA_RAM_DATAWIDTH_256
    assign count_1 = next_count[4:0] - 5'd1;
    assign res_addr_begin = (length >> 5) + (length >> 4);
    assign sram_rdata_shift = rdata >> {count_1[4:2], 5'b0};
  `elsif RW_RSA_RAM_DATAWIDTH_128
    assign count_1 = next_count[3:0] - 4'd1;
    assign res_addr_begin = (length >> 4) + (length >> 3);
    assign sram_rdata_shift = rdata >> {count_1[3:2], 5'b0};
  `else
    assign count_1 = next_count[2:0] - 3'd1;
    assign res_addr_begin = (length >> 3) + (length >> 2);
    assign sram_rdata_shift = rdata >> {count_1[2], 5'b0};
  `endif
  assign sram_res_addr = res_addr_begin[`RW_RSA_RAM_ADDWIDTH-1:0];

  //******************************************************************************
  // AHB write enable
  //******************************************************************************
  always @(posedge clk or negedge rst_n)
  begin
    if (rst_n==1'b0)
       ahb_write_en <= 1'b0;
    else
       if (ahb_write_en==1'b0 && rsa_res_valid)
          ahb_write_en <= 1'b1;
       else if (ahb_cs==AHB_DONE || bus_enable==1'b1)
          ahb_write_en <= 1'b0;
  end
  
  //******************************************************************************
  // Data buffer 
  //******************************************************************************
  always @(posedge clk or negedge rst_n)
  begin
    if (rst_n==1'b0)
       ahb_wdata_buf <= 24'b0;
    else begin
       if(ahb_cs==AHB_IDLE)
           ahb_wdata_buf <= 24'b0;
       else if (ahb_cs==AHB_DATA && ahb_ns!=AHB_DATA)
           ahb_wdata_buf <= sram_rdata_shift[31:8];
    end
  end

  //******************************************************************************
  // Write the RSA results to Slave AHB
  //******************************************************************************
  always @*
  begin
     if(ahb_write_en==1'b1 && ahb_cs==AHB_DATA) begin
        case(dest_addr[1:0])
           2'b00:   m_hwdata_tmp = sram_rdata_shift[31:0];
           2'b01:   m_hwdata_tmp = {sram_rdata_shift[23:0], ahb_wdata_buf[23:16]};
           2'b10:   m_hwdata_tmp = {sram_rdata_shift[15:0], ahb_wdata_buf[23:8]};
           default:   m_hwdata_tmp = {sram_rdata_shift[7:0], ahb_wdata_buf[23:0]};
        endcase
     end
     else
         m_hwdata_tmp = 32'h0;
  end
  assign m_hwdata = m_hwdata_tmp;
  assign m_hsize = (ahb_write_en==1'b1) ? m_hwsize_tmp : 2'b10;
  assign m_hwrite = (ahb_write_en==1'b1 && (ahb_cs==AHB_DATA || ahb_cs==AHB_ADDR)) ? 1'b1 : 1'b0; // Write

  //******************************************************************************
  // SRAM read
  //******************************************************************************
  // Read enable
  //assign ren = (ahb_ns==AHB_DATA || ahb_ns==AHB_ADDR) ? 1'b1 : 1'b0;
  assign ren = (ahb_cs==AHB_DATA || ahb_cs==AHB_ADDR) ? 1'b1 : 1'b0;
  // Read address control
  always @(posedge clk or negedge rst_n)
  begin
     if (rst_n==1'b0)
        addr <= {`RW_RSA_RAM_ADDWIDTH{1'b0}};
     else
     begin
        if (ahb_cs==AHB_IDLE && ahb_ns==AHB_ADDR)
        begin
           // First Address
           addr <= sram_res_addr;
        end
        else if (ahb_cs==AHB_DATA && ahb_ns!=AHB_DATA)
        begin
        `ifdef RW_RSA_RAM_DATAWIDTH_256
           if(count_1>= 5'b11100 && count<{length[14:0]-15'd7})
        `elsif RW_RSA_RAM_DATAWIDTH_128
           if(count_1>=4'b1100 && count<{length[14:0]-15'd7})
        `else 
           if(count_1>=3'b100 && count<{length[14:0]-15'd7})
        `endif // RW_RSA_RAM_DATAWIDTH
               addr <= addr + {{(`RW_RSA_RAM_ADDWIDTH-1){1'b0}}, 1'b1};
        end
     end
  end
  // AHB write finished signal
  assign ahb_write_end = (ahb_write_en==1'b1 && ahb_cs==AHB_DONE) ? 1'b1 : 1'b0;
`else
  assign m_hwdata = 32'h0; // Read Only
  assign m_hwrite = 1'b0;  // Read Only
  assign m_hsize  = 2'b10; // Always read 32bits  
`endif // RW_HSU_RSA_EN

always @(posedge clk or negedge rst_n)
begin
   if (rst_n==1'b0)
   begin
      m_htrans  <= 2'b00;
      m_haddr   <= 32'h0;
      m_haddr10 <= 2'b00;
   end
   else
   begin
      if (ahb_cs==AHB_IDLE && ahb_ns==AHB_ADDR)
      begin
         // First Address
         m_htrans  <= 2'b10;
`ifdef RW_HSU_RSA_EN
         m_haddr   <= (ahb_write_en==1'b1) ? dest_addr : {source_addr[31:2],2'b0};
`else
         m_haddr   <= {source_addr[31:2],2'b0};
`endif // RW_HSU_RSA_EN
         m_haddr10 <= source_addr[1:0];
      end
      else if (ahb_cs!=AHB_ADDR && ahb_ns==AHB_ADDR)
      begin
         // Increment Address
         m_htrans  <= 2'b10;
`ifdef RW_HSU_RSA_EN
         if (ahb_write_en==1'b1)
             m_haddr <= dest_addr + {17'b0, next_count};
         else
             m_haddr <= m_haddr + 32'd4;
`else
         m_haddr   <= m_haddr+32'd4;
`endif // RW_HSU_RSA_EN
         m_haddr10 <= 2'b00;
      end
      else if (ahb_cs==AHB_ADDR && ahb_ns!=AHB_ADDR)
      begin
         m_htrans  <= 2'b00;
      end
   end
end


//******************************************************************************
// Bus interface
//******************************************************************************
always @*
begin
   case (m_haddr10)
   2'b00  : bus_data = m_hrdata;
   2'b01  : bus_data = m_hrdata>>8;
   2'b10  : bus_data = m_hrdata>>16;
   default: bus_data = m_hrdata>>24;
   endcase

   if (word_nbr==15'd1)
      // Only 1 read
      bus_size = length[2:0];
   else if (word_nbr==next_count)
      // Last read
      bus_size = {~|byte_aligned[1:0],byte_aligned[1:0]};
   else
      // Intermediate read
      bus_size = 3'd4-{1'b0,m_haddr10};
`ifdef RW_HSU_RSA_EN
   bus_valid = (ahb_cs==AHB_DATA && ahb_ns!=AHB_DATA && ahb_write_en==1'b0) ? 1'b1 : 1'b0;
`else
   bus_valid = (ahb_cs==AHB_DATA && ahb_ns!=AHB_DATA) ? 1'b1 : 1'b0;
`endif // RW_HSU_RSA_EN
   bus_last  = (ahb_cs==AHB_DATA && ahb_ns==AHB_DONE) ? 1'b1 : 1'b0;
end


//******************************************************************************
// Word counter
//******************************************************************************
`ifdef RW_HSU_RSA_EN
always @*
begin
    if ((m_haddr[0]==1'b1) || (length[14:0]-count<=15'd3) || (ahb_write_en==1'b0)) begin
        ahb_write_count = count + 15'd1;
        m_hwsize_tmp = 2'b0;
    end
    else if (m_haddr[1:0]==2'b10) begin
        ahb_write_count = count + 15'd2;
        m_hwsize_tmp = 2'b01;
    end
    else begin
        ahb_write_count = count + 15'd4;
        m_hwsize_tmp = 2'b10;
    end
end
assign next_count = ahb_write_count;
`else
assign next_count = count + 15'd1;
`endif

always @(posedge clk or negedge rst_n)
begin
   if (rst_n==1'b0)
      count <= 15'h0;
   else
   begin
      if (ahb_cs==AHB_IDLE && ahb_ns!=AHB_IDLE)
         count <= 15'h0;
      else if (ahb_cs==AHB_DATA && ahb_ns!=AHB_DATA)
         count <= next_count;
   end
end


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// FSM states displayed in a string to easy simulation and debug
reg  [20*8:0] ahb_cs_str;
always @*
begin
   case (ahb_cs)
   AHB_IDLE  : ahb_cs_str = {"AHB_IDLE"};
   AHB_ADDR  : ahb_cs_str = {"AHB_ADDR"};
   AHB_DATA  : ahb_cs_str = {"AHB_DATA"};
   AHB_PAUSE : ahb_cs_str = {"AHB_PAUSE"};
   AHB_DONE  : ahb_cs_str = {"AHB_DONE"};
   default   : ahb_cs_str = {"XXX"};
   endcase
end
`endif // RW_SIMU_ON


///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef RW_ASSERT_ON

`endif // RW_ASSERT_ON

endmodule
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
