////////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//------------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//------------------------------------------------------------------------------
// $Revision: $
// $Date: $
// -----------------------------------------------------------------------------
// Dependencies     : None
// Description      : HSU control module.
//                    This module interface the master AHB with the TKIP MIC and
//                    the CMAC modules.
//                    message.
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// -----------------------------------------------------------------------------
//
//
////////////////////////////////////////////////////////////////////////////////
`default_nettype none

module rw_hsu_ctrl( 
   //$port_g Clock and reset
   input  wire          clk,                     // Clock
   input  wire          rst_n,                   // Asynchronous Reset
 
   //$port_g Bus interface
   output wire          bus_enable,              // Read enable
   output wire          bus_pause,               // Read Pause
   input  wire   [31:0] bus_data,                // Read Data
   input  wire    [2:0] bus_size,                // Read size in bytes
   input  wire          bus_last,                // Read data is the last
   input  wire          bus_valid,               // Read Data is valid

   //$port_g TKIP MIC interface
   output reg           tkip_init,               // Initialisation
   output reg    [63:0] tkip_key,                // TKIP Key
   output reg    [31:0] tkip_data,               // Message word
   output reg     [3:0] tkip_data_byte_en,       // Message word byte enable
   output reg           tkip_data_last,          // Last data
   output reg           tkip_data_valid,         // Data is valid
   input  wire          tkip_data_ready,         // Ready to load data
   input  wire   [63:0] tkip_mic,                // MIC
   input  wire          tkip_mic_valid,          // MIC qualifier

   //$port_g CMAC interface
   output reg           cmac_init,               // Initialisation
   input  wire          cmac_busy,               // CMAC is busy
   output reg   [127:0] cmac_data,               // Message word
   output reg    [15:0] cmac_data_byte_en,       // Message word byte enable
   output reg           cmac_data_last,          // Last data
   output reg           cmac_data_valid,         // Data is valid
   input  wire          cmac_data_ready,         // Ready to load data
   input  wire   [63:0] cmac_mic,                // MIC
   input  wire          cmac_mic_valid,          // MIC qualifier
`ifdef RW_HSU_IP_CHK_EN
    //$port_g IP checksum interface
   output reg           ip_init,                 // Initialisation
   output reg    [31:0] ip_data,                 // Message word
   output reg     [3:0] ip_data_byte_en,         // Message word byte enable
   output reg           ip_data_last,            // Last data tp compute
   output reg           ip_data_valid,           // Data is valid
   input  wire          ip_data_ready,           // Ready to load data
   input  wire   [15:0] ip_checksum,             // Checksum
   input  wire          ip_checksum_valid,       // Checksum qualifier
`endif // RW_HSU_IP_CHK_EN

`ifdef RW_HSU_SHA_EN
   //$port_g SHA1 interface
   output reg     [2:0] sha_mode,                // SHA-1 or SHA-2 selection
   output reg           sha_init,		         // Initialisation
   output reg    [31:0] sha_data,		         // Message Word
   output reg     [3:0] sha_data_byte_en,	     // Message word byte enable
   output reg           sha_last_data,		     // Last data to compute
   output reg           sha_data_valid,	         // Data is valid
   output wire   [63:0] sha_length,	             // Data length
   output reg     [9:0] hmac_key_len,            // Key length of HMAC
   input  wire          sha_data_ready, 	     // Ready to load data
   input  wire  [255:0] sha_hash,                // sha hash value
   input  wire          sha_hash_valid,          // hash value is valid
`ifdef RW_HSU_SHA512_EN
   output reg    [63:0] sha_data_512,            // Message Word for SHA512 or SHA384
   input  wire  [255:0] sha_hash2,               // Supplementary hash values for SHA512 or SHA384
`endif // RW_HSU_SHA512_EN
`endif // RW_HSU_SHA_EN

`ifdef RW_HSU_RSA_EN
   output reg   [`RW_RSA_RAM_ADDWIDTH-1:0] rsa_sram_addr,           // RSA SRAM addresss
   output reg     [3:0] rsa_wr_en,               // RSA SRAN Write enable
   output reg    [`RW_RSA_RAM_DATAWIDTH-1:0] rsa_din,                 // Data input
   output reg     [1:0] rsa_mode,                // RSA Mode select
   output reg           rsa_init_ready,          // RSA initialization ready
   output reg           rsa_init,                // RSA initialization init delay
   input  wire          rsa_valid,               // RSA results are valid
`endif //RW_HSU_RSA_EN

   //$port_g Registers
   input  wire          start,                   // Trigger security processing
   output wire          start_in_valid,          // Used to clear start trigger
   input  wire    [4:0] mode,                    // Security mode [4..0]
                                                 // (0:TKIP, 1:CMAC)
   input  wire          first_buffer,            // First buffer
   input  wire          last_buffer,             // Last buffer
   input  wire   [15:0] length,                  // Number of bytes in the buffer
   output reg           done_set,                // Done flag
   input  wire          done_clear,              // Done flag has to be clear
   input  wire  [127:0] key,                     // Key
   input  wire   [63:0] mic,                     // MIC
   output reg    [63:0] mic_in,                  // MIC
`ifdef RW_HSU_SHA_EN
   output reg   [255:0] sha_hash_value,          // SHA HASH VALUE
`ifdef RW_HSU_SHA512_EN
   output reg   [255:0] sha_hash_value2,         // Supplementary hash values for SHA512 or SHA384
`endif // RW_HSU_SHA512_EN
`endif //RW_HSU_SHA_EN
   output reg           mic_in_valid,            // MIC
   input  wire    [7:0] remaining_byte0,         // Remaining byte0
   output reg     [7:0] remaining_byte0_in,      // Remaining byte0
   output reg           remaining_byte0_in_valid,// Remaining byte0
   input  wire    [7:0] remaining_byte1,         // Remaining byte1
   output reg     [7:0] remaining_byte1_in,      // Remaining byte1
   output reg           remaining_byte1_in_valid,// Remaining byte1
   input  wire    [7:0] remaining_byte2,         // Remaining byte2
   output reg     [7:0] remaining_byte2_in,      // Remaining byte2
   output reg           remaining_byte2_in_valid,// Remaining byte2
   input  wire    [1:0] remaining_length,        // Remaining bytes number
   output reg     [1:0] remaining_length_in,     // Remaining bytes number
   output reg           remaining_length_in_valid// Remaining bytes number
);


////////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
////////////////////////////////////////////////////////////////////////////////
// FSM states definition
//$fsm_sd control
localparam 
   CTRL_IDLE     = 3'd0,
   CTRL_INIT     = 3'd1,
   CTRL_READ     = 3'd2,
   CTRL_FLUSH    = 3'd3,
   CTRL_WAIT_END = 3'd4,
   CTRL_DONE     = 3'd5;

localparam
   TKIP          = 4'd0,
   CMAC          = 4'd1,
   IPCHK         = 4'd2,
   SHA1	         = 4'd3,
   SHA256        = 4'd4,
   SHA224        = 4'd5,
   HMAC1         = 4'd6,
   HMAC256       = 4'd7,
   HMAC224       = 4'd8,
   SHA512        = 4'd9,
   SHA384        = 4'd10,
   HMAC512       = 4'd11,
   HMAC384       = 4'd12,
   RSA1024       = 4'd13,
   RSA2048       = 4'd14,
   RSA4096       = 4'd15;


////////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
////////////////////////////////////////////////////////////////////////////////
reg    [2:0] ctrl_cs;                  // CTRL FSM current state
reg    [2:0] ctrl_ns;                  // CTRL FSM next state   
                                     
reg    [4:0] byte_per_word_;           // Number of byte in each word
wire   [4:0] byte_per_word;            // Number of byte in each word
wire   [3:0] mode_sel;                 // Security mode_sel selection                                    
reg          computation_in_progress;  // Indicate that a computation is in
                                       // progress

reg   [22:0] buffer_load_mask;         // Select buffer bytes to load
reg   [31:0] buffer_load_data;         // 4 bytes pattern to load in the buffer
reg  [183:0] buffer;                   // Data buffer
wire         buffer_shift;             // Indicate that buffer should be shifted
reg  [183:0] buffer_after_shift;       // Data buffer after the requiered shift
                                       
reg    [4:0] byte_avail;               // Number of bytes available in the buffer
reg    [4:0] byte_avail_sub;           // byte_avail decrement value 
reg    [2:0] byte_avail_add;           // byte_avail increment value 
reg    [4:0] byte_avail_after_shift;   // byte_avail after the required shift
reg    [4:0] next_byte_avail;          // Next value of byte_avail
reg   [15:0] byte_enable;              // Buffer byte enable mask

reg    [7:0] key_len_curr;             // The length of the key has been read

`ifdef RW_HSU_RSA_EN
reg    [8:0] rsa_addr_cnt;             // The address counter of rsa
reg          rsa_data_valid;           // RSA data is valid
wire         rsa_data_ready;           // Ready to load data
reg          rsa_data_last;            // Last data for RSA
`endif //RW_HSU_RSA_EN

////////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
////////////////////////////////////////////////////////////////////////////////
//******************************************************************************
// Mode
//******************************************************************************
assign mode_sel = mode[3:0];

//******************************************************************************
// Control FSM
//******************************************************************************
always @(posedge clk or negedge rst_n)
begin
  if (rst_n==1'b0)
     ctrl_cs <= CTRL_IDLE;
  else
     ctrl_cs <= ctrl_ns;
end

always @* 
begin
   case(ctrl_cs)
   CTRL_IDLE:
   begin
      //$fsm_s In CTRL_IDLE state, the state machine waits the start trigger
      if (start)
      begin
         if (first_buffer || mode_sel==TKIP
`ifdef RW_HSU_IP_CHK_EN
            || mode_sel==IPCHK
`endif // RW_HSU_IP_CHK
            )
            //$fsm_t When first buffer or TKIP mode_sel, the state machine goes to 
            //CTRL_INIT state
            ctrl_ns = CTRL_INIT;
         else
            //$fsm_t When intermediate or last buffer, the state machine goes to 
            //CTRL_READ state
            ctrl_ns = CTRL_READ;
      end
      else
         //$fsm_t While start trigger is not received, the state machine stays
         //in CTRL_IDLE state
         ctrl_ns = CTRL_IDLE;
   end

   CTRL_INIT:
   begin
      //$fsm_s In CTRL_INIT state, initialization is done

      if (length==16'h0)
         //$fsm_t After one clock cycle, if length is null, the state machine
         //goes to CTRL_FLUSH state
         ctrl_ns = CTRL_FLUSH;
      else
         //$fsm_t After one clock cycle, if length is not null, the state machine
         //goes to CTRL_READ state
         ctrl_ns = CTRL_READ;
   end

   CTRL_READ:
   begin
      //$fsm_s In CTRL_READ state, the state machine waits the last bytes
      if (bus_last && bus_valid)
      begin
         if (next_byte_avail<byte_per_word && !last_buffer && mode_sel<SHA1 )
            //$fsm_t When last bytes is received, the state machine goes to 
            //CTRL_WAIT_END state
            ctrl_ns = CTRL_WAIT_END;
         else
            //$fsm_t When last bytes is received, the state machine goes to 
            //CTRL_FLUSH state
            ctrl_ns = CTRL_FLUSH;
      end
      else
         //$fsm_t While last bytes are not received, the state machine stays
         //in CTRL_READ state
         ctrl_ns = CTRL_READ;
   end

   CTRL_FLUSH:
   begin
      //$fsm_s In CTRL_FLUSH state, the state machine waits last data pushed
      if ((buffer_shift && next_byte_avail==5'd0) ||
          (buffer_shift && next_byte_avail<byte_per_word && !last_buffer))
         //$fsm_t When all bytes has been processed, the state machine goes to 
         //CTRL_WAIT_END state
         ctrl_ns = CTRL_WAIT_END;
      else
         //$fsm_t While  bytes are not received, the state machine stays
         //in CTRL_FLUSH state
         ctrl_ns = CTRL_FLUSH;
   end

   CTRL_WAIT_END:
   begin
      //$fsm_s In CTRL_WAIT_END state, the state machine waits processing done
      if (!computation_in_progress)
         //$fsm_t When processing is done, the state machine goes to CTRL_DONE
         //state
         ctrl_ns = CTRL_DONE;
      else
         //$fsm_t While processing is not done, the state machine stays in
         //CTRL_WAIT_END state
         ctrl_ns = CTRL_WAIT_END;
   end

   CTRL_DONE:
   begin
      //$fsm_s In CTRL_DONE state, processing results are stored.

      //$fsm_t After one clock cycle, the state machine goes to CTRL_IDLE state
      ctrl_ns = CTRL_IDLE;
   end

   // Disable coverage on the default state because it cannot be reached.
   // pragma coverage block = off 
   default:   
      ctrl_ns = CTRL_IDLE;
   // pragma coverage block = on
   endcase
end

`ifdef RW_HSU_SHA_EN
//******************************************************************************
// SHA1 data length
//******************************************************************************
assign sha_length = {45'b0, length, 3'b0};
`endif //RW_HSU_SHA_EN

//******************************************************************************
// Bus Interface
//******************************************************************************
assign bus_enable = (ctrl_cs==CTRL_READ) ? 1'b1 : 1'b0;
assign bus_pause  = byte_avail[4];


//******************************************************************************
// Number of byte in each word of the selected mode_sel.
// Others: 4 bytes (32bits)
// SHA512/SHA384/HMACSHA512/HMACSHA384/RSA1024/RSA2048/RSA4096 : 8 bytes (64bits)
// CMAC: 16 bytes(128bits)
//******************************************************************************
always @*
begin
    if (mode_sel==CMAC)
        byte_per_word_ = 5'd16;
`ifdef RW_HSU_SHA512_EN
    else if (mode_sel==HMAC512 || mode_sel==SHA512 || mode_sel==HMAC384 || mode_sel==SHA384)
        byte_per_word_ = 5'd8;
`endif // RW_HSU_SHA512_EN
`ifdef RW_HSU_RSA_EN
    else if (mode_sel==RSA4096 || mode_sel==RSA2048 || mode_sel==RSA1024)
        byte_per_word_ = 5'd8;
`endif // RW_HSU_RSA_EN
    else
       byte_per_word_ = 5'd4;
end

assign byte_per_word = byte_per_word_;

`ifdef RW_HSU_SHA_EN
always @(posedge clk or negedge rst_n)
begin
    if (rst_n==1'b0)
        hmac_key_len <= 10'b0;
    else
    begin
        if (ctrl_cs == CTRL_INIT)
            hmac_key_len <= length[9:0];
    end
end
`endif //RW_HSU_SHA_EN


//******************************************************************************
// Computation_in_progress
//   set   when computation is requested
//   reset when computation is done
//******************************************************************************
always @(posedge clk or negedge rst_n)
begin
   if (rst_n==1'b0)
      computation_in_progress <= 1'b0;
   else
   begin
      if (buffer_shift==1'b1)
         computation_in_progress <= 1'b1;
      else if ((mode_sel==TKIP && tkip_mic_valid) || (mode_sel==CMAC && !cmac_busy)
`ifdef RW_HSU_SHA_EN
		            || ((mode_sel==SHA1 || mode_sel==SHA256 || mode_sel==SHA224 ) && sha_hash_valid)
                    || ((mode_sel==HMAC1 || mode_sel==HMAC256 || mode_sel==HMAC224) && ((key_len_curr>=hmac_key_len[7:0]) || (sha_hash_valid)))
`ifdef RW_HSU_SHA512_EN
                    || ((mode_sel==SHA512 || mode_sel==SHA384) && sha_hash_valid)
                    || ((mode_sel==HMAC512 || mode_sel==HMAC384) && ((key_len_curr>=hmac_key_len[7:0]) || (sha_hash_valid)))
`endif //RW_HSU_SHA512_EN
`endif //RW_HSU_SHA_EN

`ifdef RW_HSU_IP_CHK_EN
		    || (mode_sel==IPCHK && (ip_data_ready || ip_checksum_valid))
`endif // RW_HSU_IP_CHK_EN

`ifdef RW_HSU_RSA_EN
            || (mode_sel>=RSA1024 && (rsa_valid || (!last_buffer && rsa_data_ready) ))
`endif //RW_HSU_RSA_EN
            )
         computation_in_progress <= 1'b0;
   end
end


//******************************************************************************
// Buffer (23 bytes: 16 Data, 4 bus latency, 3 unaligned address)
//******************************************************************************
always @*
begin : BUFFER_LOAD
   // Local Variable
   reg [22:0] v_mask;

   case (bus_size)
   3'd0:    v_mask = 23'h0;
   3'd1:    v_mask = 23'h1;
   3'd2:    v_mask = 23'h3;
   3'd3:    v_mask = 23'h7;
   default: v_mask = 23'hF;
   endcase

   if (byte_avail_after_shift[4]) v_mask = v_mask <<16;
   if (byte_avail_after_shift[3]) v_mask = v_mask <<8;
   if (byte_avail_after_shift[2]) v_mask = v_mask <<4;
   if (byte_avail_after_shift[1]) v_mask = v_mask <<2;
   if (byte_avail_after_shift[0]) v_mask = v_mask <<1;

   if (bus_valid)
      buffer_load_mask = v_mask;
   else
      buffer_load_mask = 23'h0;

   case (byte_avail[1:0])
   2'b00:   buffer_load_data = {bus_data};
   2'b01:   buffer_load_data = {bus_data[23:0],bus_data[31:24]};
   2'b10:   buffer_load_data = {bus_data[15:0],bus_data[31:16]};
   default: buffer_load_data = {bus_data[ 7:0],bus_data[31: 8]};
   endcase
end


always @(posedge clk or negedge rst_n)
begin: DATA_BUFFER
   // Local variable
   integer v_byte;
   integer v_bit;

   if (rst_n==1'b0)
   begin
      buffer <= 184'h0;
      key_len_curr <= 8'b0;
   end
   else
   begin
      if ((ctrl_cs==CTRL_INIT) && (first_buffer==1'b1))
      begin
         buffer[183:0] <= 184'h0;
         key_len_curr <= 8'b0;
      end
      else if ((ctrl_cs==CTRL_INIT) && (first_buffer==1'b0))
      begin
         buffer[7:0]    <= remaining_byte0;
         buffer[15:8]   <= remaining_byte1;
         buffer[23:16]  <= remaining_byte2;
         buffer[183:24] <= 160'h0;
      end
      else if (bus_valid || buffer_shift)
      begin
         for (v_byte=0; v_byte<23; v_byte=v_byte+1)
            for (v_bit=0; v_bit<8; v_bit=v_bit+1)
            begin
               buffer[v_byte*8+v_bit] <= buffer_load_mask[v_byte] ?
                                         buffer_load_data[(v_byte*8+v_bit)%32] :
                                         buffer_after_shift[v_byte*8+v_bit];
            end
         if ((buffer_shift==1'b1) && (first_buffer==1'b1))
            key_len_curr <= key_len_curr + {3'b0, byte_per_word};
      end
      else if (ctrl_cs==CTRL_DONE)
        key_len_curr <= 8'b0;
   end
end

assign buffer_shift = ((mode_sel==TKIP && tkip_data_valid && tkip_data_ready) ||
                       (mode_sel==CMAC && cmac_data_valid && cmac_data_ready) 
`ifdef RW_HSU_IP_CHK_EN
                    || (mode_sel==IPCHK && ip_data_valid && ip_data_ready)
`endif // RW_HSU_IP_CHK_EN
`ifdef RW_HSU_RSA_EN
                    || (mode_sel>=RSA1024 && rsa_data_valid && rsa_data_ready)
`endif //RW_HSU_RSA_EN
`ifdef RW_HSU_SHA_EN
		            || (sha_data_valid && sha_data_ready)
`endif //RW_HSU_SHA_EN
									    ) ? 1'b1 : 1'b0;

always @*
begin : BUFFER_UNLOAD
   // Local variable
   reg [183:0] v_shifted;

   v_shifted = buffer;
   if (byte_per_word[4]) v_shifted = v_shifted>>128;
   if (byte_per_word[3]) v_shifted = v_shifted>>64;
   if (byte_per_word[2]) v_shifted = v_shifted>>32;
   if (byte_per_word[1]) v_shifted = v_shifted>>16;
   if (byte_per_word[0]) v_shifted = v_shifted>>8;

   if (buffer_shift)
      buffer_after_shift = v_shifted;
   else
      buffer_after_shift = buffer;
end


//******************************************************************************
// Byte available counter
//   Incremented when data are received from bus interface
//   Decremented when data are provided to processing module
//******************************************************************************
always @*
begin
   // Decrement
   if (!buffer_shift)
      byte_avail_sub = 5'd0;
   else if (byte_avail>byte_per_word)
      byte_avail_sub = byte_per_word;
   else
      byte_avail_sub = byte_avail;

   // Increment
   if (bus_valid)
      byte_avail_add = bus_size;
   else
      byte_avail_add = 3'd0;

   // Byte Available After shift
   byte_avail_after_shift = byte_avail-byte_avail_sub;

   // Next value
   next_byte_avail = byte_avail_after_shift+{2'h0,byte_avail_add};
end

always @(posedge clk or negedge rst_n)
begin
   if (rst_n==1'b0)
      byte_avail <= 5'h0;
   else
   begin
      if ((ctrl_cs==CTRL_INIT) && (first_buffer==1'b1))
         byte_avail <= 5'h0;
      else if ((ctrl_cs==CTRL_INIT) && (first_buffer==1'b0))
         byte_avail <= {3'h0,remaining_length};
      else
         byte_avail <= next_byte_avail;
   end
end

// Convert Byte available to byte enable
always @*
begin
   byte_enable = 16'h0;
   if (byte_avail[4]) byte_enable = {16{1'b1}};
   if (byte_avail[3]) byte_enable = {byte_enable[ 7:0],{8{1'b1}}};
   if (byte_avail[2]) byte_enable = {byte_enable[11:0],{4{1'b1}}};
   if (byte_avail[1]) byte_enable = {byte_enable[13:0],{2{1'b1}}};
   if (byte_avail[0]) byte_enable = {byte_enable[14:0],{1{1'b1}}};
end


//******************************************************************************
// TKIP Control
//   First  buffer, the TKIP is initialized with the TKIP KEY
//   Others buffer, the TKIP is initialized with the intermediate MIC
//******************************************************************************
always @*
begin
   tkip_data = buffer[31:0];
   tkip_key  = (first_buffer==1'b1) ? key[63:0] : mic;
   if (mode_sel==TKIP)
   begin
      tkip_init = (ctrl_cs==CTRL_INIT) ? 1'b1 : 1'b0;

      if (ctrl_cs==CTRL_FLUSH && last_buffer && byte_avail<=byte_per_word)
      begin
         tkip_data_byte_en = byte_enable[3:0];
         tkip_data_last    = 1'b1;
         tkip_data_valid   = 1'b1;
      end
      else
      begin
         tkip_data_byte_en = 4'b1111;
         tkip_data_last    = 1'b0;
         tkip_data_valid   = |byte_avail[4:2];
      end
   end
   else
   begin
      tkip_init         = 1'b0;
      tkip_data_byte_en = 4'b0000;
      tkip_data_last    = 1'b0;
      tkip_data_valid   = 1'b0;
   end
end


//******************************************************************************
// CMAC Control
//******************************************************************************
always @*
begin
   cmac_data = buffer[127:0];
   if (mode_sel==CMAC)
   begin
      cmac_init = (ctrl_cs==CTRL_INIT) ? 1'b1 : 1'b0;

      if (ctrl_cs==CTRL_FLUSH && last_buffer && byte_avail<=byte_per_word)
      begin
         cmac_data_byte_en = byte_enable;
         cmac_data_last    = 1'b1;
         cmac_data_valid   = 1'b1;
      end
      else
      begin
         cmac_data_byte_en = 16'hFFFF;
         cmac_data_last    = 1'b0;
         cmac_data_valid   = byte_avail[4];
      end
   end
   else
   begin
      cmac_init         = 1'b0;
      cmac_data_byte_en = 16'h0;
      cmac_data_last    = 1'b0;
      cmac_data_valid   = 1'b0;
   end
end


`ifdef RW_HSU_IP_CHK_EN
//******************************************************************************
// IP Control
//******************************************************************************
always @*
begin
   ip_data = buffer[31:0];
   if (mode_sel==IPCHK)
   begin
      ip_init = (ctrl_cs==CTRL_INIT) ? 1'b1 : 1'b0;

      if (ctrl_cs==CTRL_FLUSH && last_buffer && byte_avail<=byte_per_word)
      begin
         ip_data_byte_en = byte_enable[3:0];
         ip_data_last    = 1'b1;
         ip_data_valid   = 1'b1;
      end
      else
      begin
         ip_data_byte_en = 4'b1111;
         ip_data_last    = 1'b0;
         ip_data_valid   = |byte_avail[4:2];
      end
   end
   else
   begin
      ip_init         = 1'b0;
      ip_data_byte_en = 4'b0000;
      ip_data_last    = 1'b0;
      ip_data_valid   = 1'b0;
   end
end
`endif // RW_HSU_IP_CHK_EN

`ifdef RW_HSU_SHA_EN
//******************************************************************************
// SHA or HMAC Control
//******************************************************************************
always @*
begin
   if (mode_sel==SHA1 || mode_sel==HMAC1)
        sha_mode = 3'd0;
   else if (mode_sel==SHA256 || mode_sel==HMAC256)
        sha_mode = 3'd1;
   `ifdef RW_HSU_SHA512_EN
   else if (mode_sel==SHA512 || mode_sel==HMAC512)
        sha_mode = 3'd3;
   else if (mode_sel==SHA384 || mode_sel==HMAC384)
        sha_mode = 3'd4;
   `endif // RW_HSU_SHA512_EN
   else
        sha_mode = 3'd2;
   //Change little-endian to big-endian accoding to SHA standard
   sha_data = {buffer[7:0], buffer[15:8], buffer[23:16], buffer[31:24]};
`ifdef RW_HSU_SHA512_EN
   sha_data_512 = {buffer[7:0], buffer[15:8], buffer[23:16], buffer[31:24], 
                   buffer[39:32], buffer[47:40], buffer[55:48], buffer[63:56]};
`endif // RW_HSU_SHA512_EN
   if (mode_sel==SHA1 || mode_sel==SHA256 || mode_sel==SHA224 || mode_sel==SHA512 || mode_sel==SHA384)
   begin
      sha_init = (ctrl_cs==CTRL_INIT) ? 1'b1 : 1'b0;

      if (ctrl_cs==CTRL_FLUSH && byte_avail<=byte_per_word)
      begin
   `ifdef RW_HSU_SHA512_EN
         if(mode_sel==SHA512 || mode_sel==SHA384)
            sha_data_byte_en = byte_avail[3:0];
         else
   `endif // RW_HSU_SHA512_EN
            sha_data_byte_en = byte_enable[3:0];
         sha_last_data    = 1'b1;
         sha_data_valid   = 1'b1;
      end
      else
      begin
   `ifdef RW_HSU_SHA512_EN
         if(mode_sel==SHA512 || mode_sel==SHA384)
            sha_data_byte_en = 4'b1000;
         else
   `endif // RW_HSU_SHA512_EN
            sha_data_byte_en = 4'b1111;
         sha_last_data    = 1'b0;
   `ifdef RW_HSU_SHA512_EN
         if(mode_sel==SHA512 || mode_sel==SHA384)
            sha_data_valid = |byte_avail[4:3];
         else
   `endif // RW_HSU_SHA512_EN
            sha_data_valid   = |byte_avail[4:2];
      end
   end
   else if (mode_sel==HMAC1 || mode_sel==HMAC256 || mode_sel==HMAC224 || mode_sel==HMAC512 || mode_sel==HMAC384)
   begin
       sha_init = (ctrl_cs==CTRL_INIT) ? 1'b1 : 1'b0;

       if (ctrl_cs==CTRL_FLUSH && last_buffer && byte_avail<=byte_per_word)
       begin
`ifdef RW_HSU_SHA512_EN
           if(mode_sel==HMAC512 || mode_sel==HMAC384)
               sha_data_byte_en = byte_avail[3:0];
           else
`endif // RW_HSU_SHA512_EN
               sha_data_byte_en = byte_enable[3:0];
           sha_last_data    = 1'b1;
           sha_data_valid   = 1'b1;
       end
       else if ((ctrl_cs==CTRL_FLUSH) && first_buffer && (key_len_curr + {3'b0, byte_per_word} >= hmac_key_len[7:0]))
       begin
`ifdef RW_HSU_SHA512_EN
           if(mode_sel==HMAC512 || mode_sel==HMAC384)
               sha_data_byte_en = byte_avail[3:0];
           else
`endif // RW_HSU_SHA512_EN
               sha_data_byte_en = byte_enable[3:0];
           sha_last_data    = 1'b0;
           sha_data_valid   = 1'b1;
       end
       else
       begin
`ifdef RW_HSU_SHA512_EN
           if(mode_sel==HMAC512 || mode_sel==HMAC384)
               sha_data_byte_en = 4'b1000;
           else
`endif // RW_HSU_SHA512_EN
               sha_data_byte_en = 4'b1111;
           sha_last_data    = 1'b0;
`ifdef RW_HSU_SHA512_EN
           if(mode_sel==HMAC512 || mode_sel==HMAC384)
               sha_data_valid = |byte_avail[4:3];
           else
`endif // RW_HSU_SHA512_EN
               sha_data_valid   = |byte_avail[4:2];
       end
   end
   else
   begin
      sha_init         = 1'b0;
      sha_data_byte_en = 4'b0000;
      sha_last_data    = 1'b0;
      sha_data_valid   = 1'b0;
   end
end
`endif //RW_HSU_SHA_EN


//******************************************************************************
// RSA Control
//******************************************************************************
`ifdef RW_HSU_RSA_EN
  `ifdef RW_RSA_RAM_DATAWIDTH_64
  always @*
  begin
     rsa_din = buffer[63:0];
     rsa_sram_addr = rsa_addr_cnt[`RW_RSA_RAM_ADDWIDTH-1:0];
     if (mode_sel>=RSA1024)
     begin
        if (ctrl_cs==CTRL_FLUSH && last_buffer && byte_avail<=byte_per_word)
        begin
           rsa_data_last   = 1'b1;
           rsa_data_valid  = 1'b1;
           rsa_wr_en       = 4'b1111;
        end
        else
        begin
           rsa_data_last   = 1'b0;
           rsa_data_valid  = |byte_avail[4:3];
           rsa_wr_en       = (|byte_avail[4:3] == 1'b1) ? 4'b1111 : 4'b0000;
        end
     end
     else
     begin
        rsa_data_last   = 1'b0;
        rsa_data_valid  = 1'b0;
        rsa_wr_en       = 4'b0000;
     end 
  end
  `elsif RW_RSA_RAM_DATAWIDTH_128
  always @*
  begin
     rsa_din = {64'b0,buffer[63:0]} << {rsa_addr_cnt[0], 6'b0};
     rsa_sram_addr = rsa_addr_cnt[`RW_RSA_RAM_ADDWIDTH:1];
     if (mode_sel>=RSA1024)
     begin
        if (ctrl_cs==CTRL_FLUSH && last_buffer && byte_avail<=byte_per_word)
        begin
           rsa_data_last   = 1'b1;
           rsa_data_valid  = 1'b1;
           rsa_wr_en       = 4'b1100;
        end
        else
        begin
           rsa_data_last   = 1'b0;
           rsa_data_valid  = |byte_avail[4:3];
           if(|byte_avail[4:3] == 1'b1) begin
              if(rsa_addr_cnt[0]==1'b0)
                  rsa_wr_en = 4'b0011;
              else
                  rsa_wr_en = 4'b1100;
           end
           else begin
              rsa_wr_en = 4'b0000;
           end
        end
     end
     else
     begin
        rsa_data_last   = 1'b0;
        rsa_data_valid  = 1'b0;
        rsa_wr_en       = 4'b0000;
     end 
  end
  `elsif RW_RSA_RAM_DATAWIDTH_256
  always @*
  begin
     rsa_din = {192'b0,buffer[63:0]} << {rsa_addr_cnt[1:0], 6'b0};
     rsa_sram_addr = rsa_addr_cnt[`RW_RSA_RAM_ADDWIDTH+1:2];
     if (mode_sel>=RSA1024)
     begin
        if (ctrl_cs==CTRL_FLUSH && last_buffer && byte_avail<=byte_per_word)
        begin
           rsa_data_last   = 1'b1;
           rsa_data_valid  = 1'b1;
           rsa_wr_en       = 4'b1000;
        end
        else
        begin
           rsa_data_last   = 1'b0;
           rsa_data_valid  = |byte_avail[4:3];
           if(|byte_avail[4:3]==1'b0)
              rsa_wr_en = 4'b0000;
           else
              rsa_wr_en = (4'b0001 << rsa_addr_cnt[1:0]);
        end
     end
     else
     begin
        rsa_data_last   = 1'b0;
        rsa_data_valid  = 1'b0;
        rsa_wr_en       = 4'b0000;
     end 
  end
  `endif //RW_RSA_RAM_DATAWIDTH_256

// RSA address Counter
always @(posedge clk or negedge rst_n)
begin
   if (rst_n==1'b0)
      rsa_addr_cnt <= 9'b0;
   else
   begin
      if ((ctrl_cs==CTRL_INIT) && (first_buffer==1'b1))
         rsa_addr_cnt <= 9'b0;
      else if (buffer_shift)
         rsa_addr_cnt <= rsa_addr_cnt + 9'd1;
   end
end

  // RSA ready to load data
  `ifdef RW_RSA_ENCRYPTION_ONLY
  assign rsa_data_ready = (rsa_addr_cnt < length[10:2]) ? 1'b1 : 1'b0;
  `else
  assign rsa_data_ready = (rsa_addr_cnt < (length[10:2]+length[11:3])) ? 1'b1 : 1'b0;
  `endif //RW_RSA_ENCRYPTION_ONLY

// RSA mode select
always @*
begin
    if(mode_sel==RSA1024)
        rsa_mode = 2'd0;
    else if (mode_sel==RSA2048)
        rsa_mode = 2'd1;
    else
        rsa_mode = 2'd2;
end

// RSA initialization signal ready
always @(posedge clk or negedge rst_n)
begin
    if (rst_n==1'b0) begin
      rsa_init_ready <= 1'b0;
      rsa_init <= 1'b0;
   end
   else
   begin
      rsa_init <= rsa_data_last;
      if (ctrl_cs==CTRL_INIT)
         rsa_init_ready <= 1'b0;
      else if (rsa_init_ready==1'b0 && rsa_data_last==1'b1)
         rsa_init_ready <= 1'b1;
   end
end
`endif //RW_HSU_RSA_EN

//******************************************************************************
// Save remaining bytes & length
//******************************************************************************
always @*
begin
   remaining_byte0_in  = buffer[7:0];
   remaining_byte1_in  = buffer[15:8];
   remaining_byte2_in  = buffer[23:16];
   remaining_length_in = byte_avail[1:0];

   if (mode_sel==TKIP)
      mic_in = tkip_mic[63:0];
`ifdef RW_HSU_IP_CHK_EN
   else if(mode_sel == IPCHK)
      mic_in = {48'b0, ip_checksum};
`endif // RW_HSU_IP_CHK_EN
   else
      mic_in = cmac_mic[63:0];

`ifdef RW_HSU_SHA_EN
   if (mode_sel>=SHA1)
      sha_hash_value = sha_hash;
   else
      sha_hash_value = 256'b0;
`ifdef RW_HSU_SHA512_EN
   if (mode_sel>=SHA512)
      sha_hash_value2 = sha_hash2;
   else
      sha_hash_value2 = 256'b0;
`endif //RW_HSU_SHA512_EN
`endif //RW_HSU_SHA_EN

   if (mode_sel==TKIP && ctrl_cs==CTRL_DONE)
   begin
      mic_in_valid              = 1'b1;
                                
      remaining_byte0_in_valid  = 1'b1;
      remaining_byte1_in_valid  = 1'b1;
      remaining_byte2_in_valid  = 1'b1;
      remaining_length_in_valid = 1'b1;
   end
   else if (mode_sel==CMAC && ctrl_cs==CTRL_DONE)
   begin
      mic_in_valid              = 1'b1;
                                
      remaining_byte0_in_valid  = 1'b0;
      remaining_byte1_in_valid  = 1'b0;
      remaining_byte2_in_valid  = 1'b0;
      remaining_length_in_valid = 1'b0;
   end
`ifdef RW_HSU_IP_CHK_EN
   else if (mode_sel==IPCHK && ctrl_cs==CTRL_DONE)
   begin
      mic_in_valid              = 1'b1;
                                
      remaining_byte0_in_valid  = 1'b1;
      remaining_byte1_in_valid  = 1'b1;
      remaining_byte2_in_valid  = 1'b1;
      remaining_length_in_valid = 1'b1;
   end
`endif // RW_HSU_IP_CHK_EN
   else
   begin
      mic_in_valid              = 1'b0;
                                
      remaining_byte0_in_valid  = 1'b0;
      remaining_byte1_in_valid  = 1'b0;
      remaining_byte2_in_valid  = 1'b0;
      remaining_length_in_valid = 1'b0;
   end
end


//******************************************************************************
// Registers Control
//******************************************************************************
assign start_in_valid = (ctrl_cs==CTRL_DONE) ? 1'b1 : 1'b0;

always @(posedge clk or negedge rst_n)
begin
   if (rst_n==1'b0)
      done_set <= 1'b0;
   else if (ctrl_cs==CTRL_DONE)
      done_set <= 1'b1;
   else if (done_clear==1'b1)
      done_set <= 1'b0;
end


////////////////////////////////////////////////////////////////////////////////
// Additional Code to ease verification
////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
// Display FSM State in string for RTL simulation waveform
//////////////////////////////////////////////////////////
`ifdef RW_SIMU_ON
// FSM states displayed in a string to easy simulation and debug
reg  [20*8:0] ctrl_cs_str;
always @*
begin
   case (ctrl_cs)
   CTRL_IDLE     : ctrl_cs_str = {"CTRL_IDLE"};
   CTRL_INIT     : ctrl_cs_str = {"CTRL_INIT"};
   CTRL_READ     : ctrl_cs_str = {"CTRL_READ"};
   CTRL_FLUSH    : ctrl_cs_str = {"CTRL_FLUSH"};
   CTRL_WAIT_END : ctrl_cs_str = {"CTRL_WAIT_END"};
   CTRL_DONE     : ctrl_cs_str = {"CTRL_DONE"};
   default       : ctrl_cs_str = {"XXX"};
   endcase
end
`endif // RW_SIMU_ON


///////////////////////////////////////
// System Verilog Assertions
///////////////////////////////////////
`ifdef RW_ASSERT_ON

`endif // RW_ASSERT_ON

endmodule
////////////////////////////////////////////////////////////////////////////////
// End of file
////////////////////////////////////////////////////////////////////////////////
