//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of rw_pta module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module rw_pta( 
   ////////////////////////////////
   //$port_g Clock And Reset
   ////////////////////////////////
   input  wire               pta_nrst,                // Asynchronous Reset (Active Low)
   input  wire               pta_clk,                 // PTA Clock
   input  wire               ahb_nrst,                // Asynchronous Reset (Active Low)
   input  wire               ahb_clk,                 // AHB Clock
   ////////////////////////////////
   //$port_g WLAN Coexistence Interface
   ////////////////////////////////
   input  wire               coex_wlan_tx,            // WLAN Transmission On-going
   input  wire               coex_wlan_rx,            // WLAN Reception    On-going
   output wire               coex_wlan_tx_abort,      // WLAN Transmission Abort Request
   output wire               coex_wlan_rx_abort,      // WLAN Reception    Abort Request
   input  wire         [3:0] coex_wlan_pti,           // WLAN Packet Traffic Information
   input  wire               coex_wlan_pti_tog,       // WLAN Packet Traffic Information Qualifier (Toggle when PTI change)
   input  wire         [6:0] coex_wlan_chan_freq,     // WLAN Primary Channel Center Frequency (12-84MHz)
   input  wire               coex_wlan_chan_bw,       // WLAN Channel Bandwidth (0:20MHz, 1:40MHz)
   input  wire               coex_wlan_chan_offset,   // WLAN Secondary Channel Offset (0:Lower, 1:Upper)
   ////////////////////////////////
   //$port_g Bluetooth Coexistence Interface
   ////////////////////////////////
   input  wire               coex_bt_tx,              // BT Transmission On-going
   input  wire               coex_bt_rx,              // BT Reception    On-going
   input  wire               coex_bt_event,           // BT Event        On-going
   output wire               coex_bt_tx_abort,        // BT Transmission Abort Request
   output wire               coex_bt_rx_abort,        // BT Reception    Abort Request
   input  wire         [3:0] coex_bt_pti,             // BT Packet Traffic Information
   input  wire         [6:0] coex_bt_channel,         // BT Channel (0-78)
   input  wire               coex_bt_bw,              // BT Bandwidth (0:1MHz, 1:2MHz)
   ////////////////////////////////
   //$port_g Slave AHB interface
   ////////////////////////////////
   input  wire               hready_in,
   input  wire               hsel,
   input  wire         [8:0] haddr,
   input  wire         [1:0] htrans,
   input  wire               hwrite,
   input  wire        [31:0] hwdata,
   output wire        [31:0] hrdata,
   output wire         [1:0] hresp,
   output wire               hready
);


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////



//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
// WLAN Coexistence Interface Resynchronized on PTA Clock Domain
wire               coex_wlan_tx_sync;       
wire               coex_wlan_rx_sync;       
wire         [3:0] coex_wlan_pti_sync;      
wire         [6:0] coex_wlan_chan_freq_sync;
wire               coex_wlan_chan_bw_sync;       
wire               coex_wlan_chan_offset_sync;

// Bluetooth Coexistence Interface Resynchronized on PTA Clock Domain
wire               coex_bt_tx_sync;     
wire               coex_bt_rx_sync;     
wire               coex_bt_event_sync;  
wire               coex_bt_event_masked;    // BT Event after register mask  
wire         [3:0] coex_bt_pti_sync;    
wire         [6:0] coex_bt_channel_sync;
wire               coex_bt_bw_sync;     

// Arbiter
reg                priority_sel;            // Current Priority (0:BT, 1:WLAN)
reg                bt_tx_abort;             // BT   Transmission Abort Request
reg                bt_rx_abort;             // BT   Reception    Abort Request
reg                wlan_tx_abort;           // WLAN Transmission Abort Request
reg                wlan_rx_abort;           // WLAN Reception    Abort Request
                                            
// Channel Overlap Detection                
wire         [6:0] bt_chan_freq;            // Bluetooth Channel Center Frequency
reg                chan_overlap;            // Channel Overlap Detected
wire         [6:0] chan_freq_dist_sgn;      // Channel Center Distance Signed
wire         [6:0] chan_freq_dist;          // Channel Center Distance
reg          [6:0] chan_freq_dist_min;      // Channel Center Distance Minimum 
                                            
// AHB Register                             
wire               pta_enable;              // Packet Traffic Arbiter enable
wire        [31:0] pta_revision;            // Packet Traffic Arbiter Revision
wire               basic_priority;          // Default Priority (0:BT, 1:WLAN)
wire               pti_enable;              // Use PTI Information
wire               no_sim_tx;               // Simultaneous TX Not Allowed
wire               no_sim_rx;               // Simultaneous RX Not Allowed
wire               chan_enable;             // Use Channel Info to detect frequency overlap
wire               abortrx_when_tx;         // Abort reception if a transmission is granted on the other side
wire               sw_wlan_rx_abort;        // WLAN Software Controlled RX abort
wire               sw_wlan_tx_abort;        // WLAN Software Controlled TX abort
wire               sw_bt_rx_abort;          // BT Software Controlled RX abort
wire               sw_bt_tx_abort;          // BT Software Controlled TX abort
wire         [5:0] chan_margin;             // Minimum space between WLAN & BT frequency boundary (0-63MHz)
wire               wlan_pti_mode;           // if 1, the WLAN PTI is used when WLAN is IDLE
wire               bt_event_mask;           // coex_bt_event signal mask enable
                                            
// PTA Status                               
reg                coex_bt_tx_sync_ff1;     // BT   Transmission Delayed
reg                coex_bt_rx_sync_ff1;     // BT   Reception    Delayed
reg                coex_wlan_tx_sync_ff1;   // WLAN Transmission Delayed
reg                coex_wlan_rx_sync_ff1;   // WLAN Reception    Delayed
reg                bt_tx_abort_ff1;         // BT   Transmission Abort Request Delayed
reg                bt_rx_abort_ff1;         // BT   Reception    Abort Request Delayed
reg                wlan_tx_abort_ff1;       // WLAN Transmission Abort Request Delayed
reg                wlan_rx_abort_ff1;       // WLAN Reception    Abort Request Delayed
reg                coex_bt_tx_sync_pos;     // BT   Transmission Posedge
reg                coex_bt_rx_sync_pos;     // BT   Reception    Posedge
reg                coex_wlan_tx_sync_pos;   // WLAN Transmission Posedge
reg                coex_wlan_rx_sync_pos;   // WLAN Reception    Posedge
wire               bt_tx_abort_pos;         // BT   Transmission Abort Request Posedge
wire               bt_rx_abort_pos;         // BT   Reception    Abort Request Posedge
wire               wlan_tx_abort_pos;       // WLAN Transmission Abort Request Posedge
wire               wlan_rx_abort_pos;       // WLAN Reception    Abort Request Posedge
wire               bt_tx_aborted_set_p;     // BT   Transmission Aborted Set Pulse
wire               bt_rx_aborted_set_p;     // BT   Reception    Aborted Set Pulse
wire               wlan_tx_aborted_set_p;   // WLAN Transmission Aborted Set Pulse
wire               wlan_rx_aborted_set_p;   // WLAN Reception    Aborted Set Pulse
wire        [31:0] stat_bt_tx;              // BT   TX       Counter: Current Value
wire        [31:0] stat_bt_tx_nxt;          // BT   TX       Counter: Next Value
wire               stat_bt_tx_inc;          // BT   TX       Counter: Increment Enable
wire        [31:0] stat_bt_tx_abort;        // BT   TX Abort Counter: Current Value
wire        [31:0] stat_bt_tx_abort_nxt;    // BT   TX Abort Counter: Next Value
wire               stat_bt_tx_abort_inc;    // BT   TX Abort Counter: Increment Enable
wire        [31:0] stat_bt_rx;              // BT   RX       Counter: Current Value
wire        [31:0] stat_bt_rx_nxt;          // BT   RX       Counter: Next Value
wire               stat_bt_rx_inc;          // BT   RX       Counter: Increment Enable
wire        [31:0] stat_bt_rx_abort;        // BT   RX Abort Counter: Current Value
wire        [31:0] stat_bt_rx_abort_nxt;    // BT   RX Abort Counter: Next Value
wire               stat_bt_rx_abort_inc;    // BT   RX Abort Counter: Increment Enable
wire        [31:0] stat_wlan_tx;            // WLAN TX       Counter: Current Value
wire        [31:0] stat_wlan_tx_nxt;        // WLAN TX       Counter: Next Value
wire               stat_wlan_tx_inc;        // WLAN TX       Counter: Increment Enable
wire        [31:0] stat_wlan_tx_abort;      // WLAN TX Abort Counter: Current Value
wire        [31:0] stat_wlan_tx_abort_nxt;  // WLAN TX Abort Counter: Next Value
wire               stat_wlan_tx_abort_inc;  // WLAN TX Abort Counter: Increment Enable
wire        [31:0] stat_wlan_rx;            // WLAN RX       Counter: Current Value
wire        [31:0] stat_wlan_rx_nxt;        // WLAN RX       Counter: Next Value
wire               stat_wlan_rx_inc;        // WLAN RX       Counter: Increment Enable
wire        [31:0] stat_wlan_rx_abort;      // WLAN RX Abort Counter: Current Value
wire        [31:0] stat_wlan_rx_abort_nxt;  // WLAN RX Abort Counter: Next Value
wire               stat_wlan_rx_abort_inc;  // WLAN RX Abort Counter: Increment Enable

//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// WLAN IF Resynchronisation
//////////////////////////////////////////////////////////////////////////////
rw_pta_coex_wlan_resync u_rw_pta_coex_wlan_resync(
   // Clock And Reset
   .pta_nrst                     (pta_nrst                     ),
   .pta_clk                      (pta_clk                      ),
   // WLAN Coexistence Input
   .coex_wlan_tx                 (coex_wlan_tx                 ),
   .coex_wlan_rx                 (coex_wlan_rx                 ),
   .coex_wlan_pti                (coex_wlan_pti                ),
   .coex_wlan_pti_tog            (coex_wlan_pti_tog            ),
   .coex_wlan_chan_freq          (coex_wlan_chan_freq          ),
   .coex_wlan_chan_bw            (coex_wlan_chan_bw            ),
   .coex_wlan_chan_offset        (coex_wlan_chan_offset        ),
   // WLAN Coexistence Resynchronized
   .coex_wlan_tx_sync            (coex_wlan_tx_sync            ),
   .coex_wlan_rx_sync            (coex_wlan_rx_sync            ),
   .coex_wlan_pti_sync           (coex_wlan_pti_sync           ),
   .coex_wlan_chan_freq_sync     (coex_wlan_chan_freq_sync     ),
   .coex_wlan_chan_bw_sync       (coex_wlan_chan_bw_sync       ),
   .coex_wlan_chan_offset_sync   (coex_wlan_chan_offset_sync   ) 
);


//////////////////////////////////////////////////////////////////////////////
// BT IF Resynchronisation
//////////////////////////////////////////////////////////////////////////////
rw_pta_coex_bt_resync u_rw_pta_coex_bt_resync(
   .pta_clk             (pta_clk             ),
   .pta_nrst            (pta_nrst            ),
   // Input
   .coex_bt_tx          (coex_bt_tx          ),
   .coex_bt_rx          (coex_bt_rx          ),
   .coex_bt_event       (coex_bt_event       ),
   .coex_bt_pti         (coex_bt_pti         ),
   .coex_bt_channel     (coex_bt_channel     ),
   .coex_bt_bw          (coex_bt_bw          ),
   // Output
   .coex_bt_tx_sync     (coex_bt_tx_sync     ),
   .coex_bt_rx_sync     (coex_bt_rx_sync     ),
   .coex_bt_event_sync  (coex_bt_event_sync  ),
   .coex_bt_pti_sync    (coex_bt_pti_sync    ),
   .coex_bt_channel_sync(coex_bt_channel_sync),
   .coex_bt_bw_sync     (coex_bt_bw_sync     )
);
assign coex_bt_event_masked = coex_bt_event_sync & ~bt_event_mask;


//////////////////////////////////////////////////////////////////////////////
// Output assignment
//////////////////////////////////////////////////////////////////////////////
assign coex_bt_tx_abort   = bt_tx_abort;
assign coex_bt_rx_abort   = bt_rx_abort;
assign coex_wlan_tx_abort = wlan_tx_abort;
assign coex_wlan_rx_abort = wlan_rx_abort;


//////////////////////////////////////////////////////////////////////////////
// Priority Arbiter
//////////////////////////////////////////////////////////////////////////////
always @*
begin
   // Direct Mode
   if (pti_enable == 1'b0)
      priority_sel = basic_priority;

   // PTI Mode: Same Priority
   else if (coex_wlan_pti_sync == coex_bt_pti_sync)
      priority_sel = basic_priority;

   // PTI Mode: BT Higher Priority
   else if (coex_wlan_pti_sync < coex_bt_pti_sync)
      priority_sel = 1'b0; // BT

   // PTI Mode: WLAN Higher Priority
   else
      priority_sel = 1'b1; // WLAN
end


//////////////////////////////////////////////////////////////////////////////
// Frequency Overlap Detection
//////////////////////////////////////////////////////////////////////////////
assign bt_chan_freq = coex_bt_channel_sync + 7'h2;

// Channel Center Frequency Distance
assign chan_freq_dist_sgn = coex_wlan_chan_freq_sync - bt_chan_freq;
assign chan_freq_dist     = (chan_freq_dist_sgn[6]==1'b1) ? (~chan_freq_dist_sgn)+7'b1 : // 2nd complement
                                                              chan_freq_dist_sgn;
// Channel Center Frequency Minimum Distance
always @*
begin
   case ({coex_wlan_chan_bw_sync, coex_bt_bw_sync, coex_wlan_chan_offset_sync, chan_freq_dist_sgn[6]})
   4'b0010, 4'b0000, // WLAN 20MHz, BT 1MHz, secondary n/a  , (bt<wlan)
   4'b0011, 4'b0001, // WLAN 20MHz, BT 1MHz, secondary n/a  , (bt>wlan)
   4'b1010,          // WLAN 40MHz, BT 1MHz, secondary upper, (bt<wlan)
   4'b1001:          // WLAN 40MHz, BT 1MHz, secondary lower, (bt>wlan)
       chan_freq_dist_min = {1'b0,chan_margin} + 7'd12;

   4'b0110, 4'b0100, // WLAN 20MHz, BT 2MHz, secondary n/a  , (bt<wlan)
   4'b0111, 4'b0101, // WLAN 20MHz, BT 2MHz, secondary n/a  , (bt>wlan)
   4'b1110,          // WLAN 40MHz, BT 2MHz, secondary upper, (bt<wlan)
   4'b1101:          // WLAN 40MHz, BT 2MHz, secondary lower, (bt>wlan)
       chan_freq_dist_min = {1'b0,chan_margin} + 7'd13;

   4'b1000,          // WLAN 40MHz, BT 1MHz, secondary lower, (bt<wlan)
   4'b1011:          // WLAN 40MHz, BT 1MHz, secondary upper, (bt>wlan)
       chan_freq_dist_min = {1'b0,chan_margin} + 7'd32;

   4'b1100,          // WLAN 40MHz, BT 2MHz, secondary lower, (bt<wlan)
   4'b1111:          // WLAN 40MHz, BT 2MHz, secondary upper, (bt>wlan)
       chan_freq_dist_min = {1'b0,chan_margin} + 7'd33;

   // pragma coverage block = off
   default :
   begin
   end
   // pragma coverage block = on
   endcase
end

// Overlap Detection
always @*
begin
   if ((chan_enable == 1'b0) || (chan_freq_dist < chan_freq_dist_min))
      chan_overlap = 1'b1;
   else
      chan_overlap = 1'b0;
end

//////////////////////////////////////////////////////////////////////////////
// Bluetooth Abort
//////////////////////////////////////////////////////////////////////////////
always @(posedge pta_clk or negedge pta_nrst)
begin
   // Asynchronous Reset
   if (pta_nrst == 1'b0)
   begin
      bt_tx_abort   <= 1'b0;
      bt_rx_abort   <= 1'b0;
   end
   //
   else
   begin
      if (pta_enable == 1'b0)
      begin
         bt_tx_abort <= sw_bt_tx_abort;
         bt_rx_abort <= sw_bt_rx_abort;
      end
      else if ((priority_sel == 1'b1)/* && (pta_enable == 1'b1)*/)
      begin
         case ({coex_wlan_tx_sync, coex_wlan_rx_sync, coex_bt_tx_sync, coex_bt_rx_sync, coex_bt_event_masked})
         // WLAN TX, BT IDLE
         5'b10000:
         begin
            bt_tx_abort   <= 1'b0;
            bt_rx_abort   <= 1'b0;
         end
         // WLAN RX, BT IDLE
         5'b01000:
         begin
            bt_tx_abort   <= 1'b0;
            bt_rx_abort   <= 1'b0;
         end

         // WLAN TX, BT EVENT
         5'b10001:
         begin
            bt_tx_abort   <= 1'b0;
            bt_rx_abort   <= 1'b0;
         end
         // WLAN RX, BT EVENT
         5'b01001:
         begin
            bt_tx_abort   <= 1'b0;
            bt_rx_abort   <= 1'b0;
         end

         // WLAN TX, BT TX
         5'b10100, 5'b10101:
         begin
            bt_tx_abort   <= chan_overlap & no_sim_tx;
            bt_rx_abort   <= chan_overlap & abortrx_when_tx;
         end
         // WLAN TX, BT RX
         5'b10010, 5'b10011:
         begin
            bt_tx_abort   <= chan_overlap & no_sim_tx;
            bt_rx_abort   <= chan_overlap & abortrx_when_tx;
         end

         // WLAN RX, BT TX
         5'b01100, 5'b01101:
         begin
            bt_tx_abort   <= chan_overlap;
            bt_rx_abort   <= chan_overlap & no_sim_rx;
         end
         // WLAN RX, BT RX
         5'b01010, 5'b01011:
         begin
            bt_tx_abort   <= chan_overlap;
            bt_rx_abort   <= chan_overlap & no_sim_rx;
         end
         //
         default:
         begin
            bt_tx_abort   <= 1'b0;
            bt_rx_abort   <= 1'b0;
         end
         endcase
      end
      else
      begin
         bt_tx_abort   <= 1'b0;
         bt_rx_abort   <= 1'b0;
      end
   end
end

//////////////////////////////////////////////////////////////////////////////
// WLAN Abort
//////////////////////////////////////////////////////////////////////////////
always @(posedge pta_clk or negedge pta_nrst)
begin
   // Asynchronous Reset
   if (pta_nrst == 1'b0)
   begin
      wlan_tx_abort <= 1'b0;
      wlan_rx_abort <= 1'b0;
   end
   //
   else
   begin
      if (pta_enable == 1'b0)
      begin
         wlan_tx_abort <= sw_wlan_tx_abort;
         wlan_rx_abort <= sw_wlan_rx_abort;
      end
      else if ((priority_sel == 1'b0)/* && (pta_enable == 1'b1)*/)
      begin
         case ({coex_wlan_tx_sync, coex_wlan_rx_sync, coex_bt_tx_sync, coex_bt_rx_sync, coex_bt_event_masked})
         // BT TX, WLAN IDLE
         5'b00100, 5'b00101:
         begin
            wlan_tx_abort <= (!pti_enable | wlan_pti_mode) & chan_overlap & no_sim_tx;
            wlan_rx_abort <= chan_overlap & abortrx_when_tx;
         end
         // BT TX, WLAN TX
         5'b10100, 5'b10101:
         begin
            wlan_tx_abort <= chan_overlap & no_sim_tx;
            wlan_rx_abort <= chan_overlap & abortrx_when_tx;
         end
         // BT TX, WLAN RX
         5'b01100, 5'b01101:
         begin
            wlan_tx_abort <= chan_overlap & no_sim_tx;
            wlan_rx_abort <= chan_overlap & abortrx_when_tx;
         end

         // BT RX, WLAN IDLE
         5'b00010, 5'b00011:
         begin
            wlan_tx_abort <= (!pti_enable | wlan_pti_mode) & chan_overlap;
            wlan_rx_abort <= (!pti_enable | wlan_pti_mode) & chan_overlap & no_sim_rx;
         end
         // BT RX, WLAN TX
         5'b10010, 5'b10011:
         begin
            wlan_tx_abort <= chan_overlap;
            wlan_rx_abort <= chan_overlap & no_sim_rx;
         end
         // BT RX, WLAN RX
         5'b01010, 5'b01011:
         begin
            wlan_tx_abort <= chan_overlap;
            wlan_rx_abort <= chan_overlap & no_sim_rx;
         end

         // BT EVENT, WLAN IDLE
         5'b00001:
         begin
            wlan_tx_abort <= (!pti_enable | wlan_pti_mode) & chan_overlap;
            wlan_rx_abort <= 1'b0;
         end
         // BT EVENT, WLAN TX
         5'b10001:
         begin
            wlan_tx_abort <= (!pti_enable | wlan_pti_mode) & chan_overlap;
            wlan_rx_abort <= 1'b0;
         end
         // BT EVENT, WLAN RX
         5'b01001:
         begin
            wlan_tx_abort <= (!pti_enable | wlan_pti_mode) & chan_overlap;
            wlan_rx_abort <= 1'b0;
         end
         //
         default:
         begin
            wlan_tx_abort <= 1'b0;
            wlan_rx_abort <= 1'b0;
         end
         endcase
      end
      else
      begin
         wlan_tx_abort <= 1'b0;
         wlan_rx_abort <= 1'b0;
      end
   end
end

//////////////////////////////////////////////////////////////////////////////
// PTA Status
//////////////////////////////////////////////////////////////////////////////
// Posedge Detection
always @(posedge pta_clk or negedge pta_nrst)
begin
   // Asynchronous Reset
   if (pta_nrst == 1'b0)
   begin
      coex_bt_tx_sync_ff1   <= 1'b0;
      coex_bt_rx_sync_ff1   <= 1'b0;
      coex_wlan_tx_sync_ff1 <= 1'b0;
      coex_wlan_rx_sync_ff1 <= 1'b0;

      bt_tx_abort_ff1       <= 1'b0;
      bt_rx_abort_ff1       <= 1'b0;
      wlan_tx_abort_ff1     <= 1'b0;
      wlan_rx_abort_ff1     <= 1'b0;

      coex_bt_tx_sync_pos   <= 1'b0;
      coex_bt_rx_sync_pos   <= 1'b0;
      coex_wlan_tx_sync_pos <= 1'b0;
      coex_wlan_rx_sync_pos <= 1'b0;
   end
   //
   else
   begin
      coex_bt_tx_sync_ff1   <= coex_bt_tx_sync;
      coex_bt_rx_sync_ff1   <= coex_bt_rx_sync;
      coex_wlan_tx_sync_ff1 <= coex_wlan_tx_sync;
      coex_wlan_rx_sync_ff1 <= coex_wlan_rx_sync;

      bt_tx_abort_ff1       <= bt_tx_abort;
      bt_rx_abort_ff1       <= bt_rx_abort;
      wlan_tx_abort_ff1     <= wlan_tx_abort;
      wlan_rx_abort_ff1     <= wlan_rx_abort;
      
      coex_bt_tx_sync_pos   <= coex_bt_tx_sync   & !coex_bt_tx_sync_ff1;
      coex_bt_rx_sync_pos   <= coex_bt_rx_sync   & !coex_bt_rx_sync_ff1;
      coex_wlan_tx_sync_pos <= coex_wlan_tx_sync & !coex_wlan_tx_sync_ff1;
      coex_wlan_rx_sync_pos <= coex_wlan_rx_sync & !coex_wlan_rx_sync_ff1;
   end
end

assign bt_tx_abort_pos       = bt_tx_abort       & !bt_tx_abort_ff1;
assign bt_rx_abort_pos       = bt_rx_abort       & !bt_rx_abort_ff1;
assign wlan_tx_abort_pos     = wlan_tx_abort     & !wlan_tx_abort_ff1;
assign wlan_rx_abort_pos     = wlan_rx_abort     & !wlan_rx_abort_ff1;


// Abort Detection
assign bt_tx_aborted_set_p   = (coex_bt_tx_sync_pos & bt_tx_abort) || (bt_tx_abort_pos & coex_bt_tx_sync);
assign bt_rx_aborted_set_p   = (coex_bt_rx_sync_pos & bt_rx_abort) || (bt_rx_abort_pos & coex_bt_rx_sync);
assign wlan_tx_aborted_set_p = (coex_wlan_tx_sync_pos & wlan_tx_abort) || (wlan_tx_abort_pos & coex_wlan_tx_sync);
assign wlan_rx_aborted_set_p = (coex_wlan_rx_sync_pos & wlan_rx_abort) || (wlan_rx_abort_pos & coex_wlan_rx_sync);


// Status Register Controls
assign stat_bt_tx_nxt         =   stat_bt_tx         + 32'h1;
assign stat_bt_rx_nxt         =   stat_bt_rx         + 32'h1;                                   
assign stat_wlan_tx_nxt       =   stat_wlan_tx       + 32'h1;
assign stat_wlan_rx_nxt       =   stat_wlan_rx       + 32'h1;
assign stat_bt_tx_abort_nxt   =   stat_bt_tx_abort   + 32'h1;
assign stat_bt_rx_abort_nxt   =   stat_bt_rx_abort   + 32'h1;                                   
assign stat_wlan_tx_abort_nxt =   stat_wlan_tx_abort + 32'h1;                                   
assign stat_wlan_rx_abort_nxt =   stat_wlan_rx_abort + 32'h1;                                   

assign stat_bt_tx_inc         = ((stat_bt_tx   != 32'hFFFFFFFF) && (coex_bt_tx_sync_pos   == 1'b1)) ? 1'b1 : 1'b0;
assign stat_bt_rx_inc         = ((stat_bt_rx   != 32'hFFFFFFFF) && (coex_bt_rx_sync_pos   == 1'b1)) ? 1'b1 : 1'b0;
assign stat_wlan_tx_inc       = ((stat_wlan_tx != 32'hFFFFFFFF) && (coex_wlan_tx_sync_pos == 1'b1)) ? 1'b1 : 1'b0;
assign stat_wlan_rx_inc       = ((stat_wlan_rx != 32'hFFFFFFFF) && (coex_wlan_rx_sync_pos == 1'b1)) ? 1'b1 : 1'b0;
assign stat_bt_tx_abort_inc   = ((stat_bt_tx   != 32'hFFFFFFFF) && (bt_tx_aborted_set_p   == 1'b1)) ? 1'b1 : 1'b0;
assign stat_bt_rx_abort_inc   = ((stat_bt_rx   != 32'hFFFFFFFF) && (bt_rx_aborted_set_p   == 1'b1)) ? 1'b1 : 1'b0;
assign stat_wlan_tx_abort_inc = ((stat_wlan_tx != 32'hFFFFFFFF) && (wlan_tx_aborted_set_p == 1'b1)) ? 1'b1 : 1'b0;
assign stat_wlan_rx_abort_inc = ((stat_wlan_rx != 32'hFFFFFFFF) && (wlan_rx_aborted_set_p == 1'b1)) ? 1'b1 : 1'b0;


//////////////////////////////////////////////////////////////////////////////
// AHB Registers
//////////////////////////////////////////////////////////////////////////////
assign pta_revision = 32'h00010001;

rw_pta_reg u_rw_pta_reg (
   // Clock and reset
   .ahb_nrst                          (ahb_nrst                          ),
   .ahb_clk                           (ahb_clk                           ),
   // Registers
   // REVISION register.
   .revision                          (pta_revision                      ),
   // STAT_BT_TX register.
   .stat_bt_tx_in                     (stat_bt_tx_nxt                    ),
   .stat_bt_tx_in_valid               (stat_bt_tx_inc                    ),
   // STAT_BT_TX_ABORT register.                                       
   .stat_bt_tx_abort_in               (stat_bt_tx_abort_nxt              ),
   .stat_bt_tx_abort_in_valid         (stat_bt_tx_abort_inc              ),
   // STAT_BT_RX register.                                             
   .stat_bt_rx_in                     (stat_bt_rx_nxt                    ),
   .stat_bt_rx_in_valid               (stat_bt_rx_inc                    ),
   // STAT_BT_RX_ABORT register.                                       
   .stat_bt_rx_abort_in               (stat_bt_rx_abort_nxt              ),
   .stat_bt_rx_abort_in_valid         (stat_bt_rx_abort_inc              ),
   // STAT_WLAN_TX register.                                           
   .stat_wlan_tx_in                   (stat_wlan_tx_nxt                  ),
   .stat_wlan_tx_in_valid             (stat_wlan_tx_inc                  ),
   // STAT_WLAN_TX_ABORT register.                                     
   .stat_wlan_tx_abort_in             (stat_wlan_tx_abort_nxt            ),
   .stat_wlan_tx_abort_in_valid       (stat_wlan_tx_abort_inc            ),
   // STAT_WLAN_RX register.                                           
   .stat_wlan_rx_in                   (stat_wlan_rx_nxt                  ),
   .stat_wlan_rx_in_valid             (stat_wlan_rx_inc                  ),
   // STAT_WLAN_RX_ABORT register.                                     
   .stat_wlan_rx_abort_in             (stat_wlan_rx_abort_nxt            ),
   .stat_wlan_rx_abort_in_valid       (stat_wlan_rx_abort_inc            ),
   //
   // CONFIG register.
   .abortrx_when_tx                   (abortrx_when_tx                   ),
   .sw_wlan_rx_abort                  (sw_wlan_rx_abort                  ),
   .sw_wlan_tx_abort                  (sw_wlan_tx_abort                  ),
   .sw_bt_rx_abort                    (sw_bt_rx_abort                    ),
   .sw_bt_tx_abort                    (sw_bt_tx_abort                    ),
   .chan_margin                       (chan_margin                       ),
   .wlan_pti_mode                     (wlan_pti_mode                     ),
   .bt_event_mask                     (bt_event_mask                     ),
   .chan_enable                       (chan_enable                       ),
   .pti_enable                        (pti_enable                        ),
   .no_sim_tx                         (no_sim_tx                         ),
   .no_sim_rx                         (no_sim_rx                         ),
   .basic_priority                    (basic_priority                    ),
   .pta_enable                        (pta_enable                        ),
   .stat_bt_tx                        (stat_bt_tx                        ),
   .stat_bt_tx_abort                  (stat_bt_tx_abort                  ),
   .stat_bt_rx                        (stat_bt_rx                        ),
   .stat_bt_rx_abort                  (stat_bt_rx_abort                  ),
   .stat_wlan_tx                      (stat_wlan_tx                      ),
   .stat_wlan_tx_abort                (stat_wlan_tx_abort                ),
   .stat_wlan_rx                      (stat_wlan_rx                      ),
   .stat_wlan_rx_abort                (stat_wlan_rx_abort                ),
   // Bus interface
   .hready_in                         (hready_in                         ),
   .hsel                              (hsel                              ),
   .haddr                             (haddr[8:0]                        ),
   .htrans                            (htrans                            ),
   .hwrite                            (hwrite                            ),
   .hwdata                            (hwdata                            ),
   .hrdata                            (hrdata                            ),
   .hresp                             (hresp                             ),
   .hready                            (hready                            ) 
);

endmodule
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
