//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission 
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : RW PTA Bluetooth Coexistence Interface Resynchronisation Block
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module rw_pta_coex_bt_resync(
   ////////////////////////////////
   //$port_g Clock And Reset
   ////////////////////////////////
   input  wire           pta_nrst,            // Asynchronous Reset (Active Low)
   input  wire           pta_clk,             // PTA and AHB Clock
   ////////////////////////////////
   //$port_g BT Coexistence Input
   ////////////////////////////////
   input  wire           coex_bt_tx,          // BT Transmission On-going
   input  wire           coex_bt_rx,          // BT Reception    On-going
   input  wire           coex_bt_event,       // BT Event        On-going
   input  wire     [3:0] coex_bt_pti,         // BT Packet Traffic Information
   input  wire     [6:0] coex_bt_channel,     // BT Channel (0-78)
   input  wire           coex_bt_bw,          // BT Bandwidth (0:1MHz, 1:2MHz)
   ////////////////////////////////
   //$port_g BT Coexistence Resynchronized
   ////////////////////////////////
   output wire           coex_bt_tx_sync,     //
   output wire           coex_bt_rx_sync,     //
   output wire           coex_bt_event_sync,  //
   output reg      [3:0] coex_bt_pti_sync,    //
   output reg      [6:0] coex_bt_channel_sync,//
   output reg            coex_bt_bw_sync      //
);


//////////////////////////////////////////////////////////////////////////////
// Parameter Definitions
//////////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
wire coex_bt_tx_ff2;     // BT TX After    2nd Resync Flop
wire coex_bt_rx_ff2;     // BT RX After    2nd Resync Flop
wire coex_bt_event_ff2;  // BT EVENT After 2nd Resync Flop
reg  coex_bt_tx_ff3;     // BT TX After    3rd Resync Flop
reg  coex_bt_rx_ff3;     // BT RX After    3rd Resync Flop
reg  coex_bt_event_ff3;  // BT EVENT After 3rd Resync Flop
wire coex_bt_tx_edge;    // BT TX Edge Detected
wire coex_bt_rx_edge;    // BT RX Edge Detected


//////////////////////////////////////////////////////////////////////////////
// Begining of Logic part
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// Output Assignment
//////////////////////////////////////////////////////////////////////////////
assign coex_bt_tx_sync    = coex_bt_tx_ff3;
assign coex_bt_rx_sync    = coex_bt_rx_ff3;
assign coex_bt_event_sync = coex_bt_event_ff3;


//////////////////////////////////////////////////////////////////////////////
// Double Resynchronisation Flop
//////////////////////////////////////////////////////////////////////////////
rw_pta_bit_resync u_coex_bt_tx_resync(
   .dst_clk  (pta_clk             ),
   .dst_nrst (pta_nrst            ),
   .src_bit  (coex_bt_tx          ),
   .dst_bit  (coex_bt_tx_ff2      )
);

rw_pta_bit_resync u_coex_bt_rx_resync(
   .dst_clk  (pta_clk             ),
   .dst_nrst (pta_nrst            ),
   .src_bit  (coex_bt_rx          ),
   .dst_bit  (coex_bt_rx_ff2      )
);

rw_pta_bit_resync u_coex_bt_event_resync(
   .dst_clk  (pta_clk             ),
   .dst_nrst (pta_nrst            ),
   .src_bit  (coex_bt_event       ),
   .dst_bit  (coex_bt_event_ff2   )
);


//////////////////////////////////////////////////////////////////////////////
// 3rd Synchronization flop
//    Used to aligne signal with the PTI & Channel Information resynchronized
//////////////////////////////////////////////////////////////////////////////
always @(posedge pta_clk or negedge pta_nrst)
begin
   // Asynchronous Reset
   if (pta_nrst == 1'b0)
   begin
      coex_bt_tx_ff3 <= 1'b0;
      coex_bt_rx_ff3 <= 1'b0;
      coex_bt_event_ff3 <= 1'b0;
   end
   //
   else 
   begin
      coex_bt_tx_ff3    <= coex_bt_tx_ff2;
      coex_bt_rx_ff3    <= coex_bt_rx_ff2;
      coex_bt_event_ff3 <= coex_bt_event_ff2;
   end
end


//////////////////////////////////////////////////////////////////////////////
// TX/RX Edge Detection
//////////////////////////////////////////////////////////////////////////////
assign coex_bt_tx_edge = coex_bt_tx_ff3 ^ coex_bt_tx_ff2;
assign coex_bt_rx_edge = coex_bt_rx_ff3 ^ coex_bt_rx_ff2;


//////////////////////////////////////////////////////////////////////////////
// PTI & Channel Information Resynchronisation
//   Channel Information & PTI are latched on BT TX/RX edge
//////////////////////////////////////////////////////////////////////////////
always @(posedge pta_clk or negedge pta_nrst)
begin
   // Asynchronous Reset
   if (pta_nrst == 1'b0)
   begin
      coex_bt_pti_sync     <= 4'h0;
      coex_bt_channel_sync <= 7'h0;
      coex_bt_bw_sync      <= 1'b0;
   end
   //
   else 
   begin
      if ((coex_bt_tx_edge == 1'b1) || (coex_bt_rx_edge == 1'b1))
      begin
         coex_bt_pti_sync     <= coex_bt_pti;
         coex_bt_channel_sync <= coex_bt_channel;
         coex_bt_bw_sync      <= coex_bt_bw;
      end
   end
end

endmodule
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
