//////////////////////////////////////////////////////////////////////////
//
//     Module           : accu
//     Full Module Name : accu accumulator block.
//     Architecture     : rtl
//     Block Name       : 
//     Project          : 
//     Designer(s)      : Aedan Coffey aedan.coffey@ceva-dsp.com
//     Current Version  : $Revision: 1.2 $
//     Current Date     : $Date: 2010-02-03 20:39:26+02 $
//     Simulator        : 
//     Synthesiser      : 
//     Coding Standard  : 
//
//     Description      : This block computes the sum of the SNRs of all subcarriers in the group.
//
///////////////////////////////////////////////////////////////////////////
//
//            Copyright (c) Ceva Inc.
//
// This code is confidential and proprietary product of Ceva.
// Any unauthorized use, reproduction or transfer of this 
// code is strictly prohibited.
//
//////////////////////////////////////////////////////////////////////////

`default_nettype none

module accu (  
  input  wire                  BFRModemClk,      // 120 MHz clock.
  input  wire                  nBFRModemRst,     // Async reset, active low.
  input  wire signed    [10:0] cfgSnrFactor,
  input  wire                  tctlStart,        // Asserted to start a computation sequence.
  input  wire                  tctlStop,         // Asserted to stop a computation or initialise everything.
  input  wire            [7:0] snrNsnr,          // Unsigned 8 bit number, SNR.
  input  wire                  snrNsnrActive,  
  input  wire                  tctlAdvance,      // Asserted when the pipeline is to move on a stage.
  input  wire            [7:0] tctlNs,           // The Ns constant. The number of subcarriers we are processing.
  input  wire                  tctlNsValid,
  output reg             [8:0] accuSubAvg,       // Unsigned. A version of the average snr that needs to be subtracted from all values in macif.
  output reg  signed     [7:0] accuAvgSnr        // Signed 8 bit number, average SNR.
  );
  
  reg         [15:0] Sum;          // Unsigned. The summation register, all incoming values are summed into this.
  wire        [23:0] V1;           // Intermediate value used in calculations.
  wire signed [11:0] IntAvgSnr;    // Internal version of the output, pre saturation.
  reg         [14:0] MultFactor, MultFactor_1t, MultFactor_2t;
  

  //
  // The summation register, every time tctlStart is asserted it gets cleared, then when tctlAdvance is asserted
  // the current snrNsnr gets read in if this pipeline stage is active.
  //
  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
      if (!nBFRModemRst)
        begin
          Sum <= 16'd0;
        end
      else
        begin
          if (tctlStart || tctlStop)
            begin
              Sum <= 16'd0;
            end
          else if (tctlAdvance && snrNsnrActive)
            begin
              Sum <= Sum + {8'd0,snrNsnr};
            end
        end
    end
    
  //
  // Perform the exit calculations and register the output values.
  // accuAvgSnr = Sum*(NsK/2^16)-cfgSnrFactor 
  // Use an intermediate value (V1) = Sum*Nsk
  // Sum is 16 bits (max 255*Ns)
  // SubAvg is the value that needs to be subtracted from the values put into the
  // Mac Dpram by the macif block. 
  //
  // Derive NsK from Ns. NsK is 2^16/tctlNs, ie 65536/tctlNs.
  //
  always @(*)
    begin
      case (tctlNs)
        8'd234 : MultFactor = 15'd280;    // Channel width = 80MHz, Grouping = 1.
        8'd122 : MultFactor = 15'd537;    // Channel width = 80MHz, Grouping = 2.
        8'd108 : MultFactor = 15'd607;    // Channel width = 40MHz, Grouping = 1.
        8'd64  : MultFactor = 15'd1024;
        8'd62  : MultFactor = 15'd1057;   // Channel width = 80MHz, Grouping = 4.
        8'd58  : MultFactor = 15'd1130;   // Channel width = 40MHz, Grouping = 2.
        8'd57  : MultFactor = 15'd1150;
        8'd52  : MultFactor = 15'd1260;   // Channel width = 20MHz, Grouping = 1.
        8'd50  : MultFactor = 15'd1311;
        8'd44  : MultFactor = 15'd1489;
        8'd43  : MultFactor = 15'd1524;
        8'd37  : MultFactor = 15'd1771;
        8'd36  : MultFactor = 15'd1820;
        8'd30  : MultFactor = 15'd2185;   // Channel width = 20MHz, Grouping = 2.
        8'd28  : MultFactor = 15'd2341;
        8'd24  : MultFactor = 15'd2731;
        8'd23  : MultFactor = 15'd2849;
        8'd22  : MultFactor = 15'd2979;
        8'd21  : MultFactor = 15'd3121;
        8'd20  : MultFactor = 15'd3277;
        8'd18  : MultFactor = 15'd3641;
        8'd17  : MultFactor = 15'd3855;
        8'd16  : MultFactor = 15'd4096;   // Channel width = 20MHz, Grouping = 4. (tctlNs=16)
        8'd15  : MultFactor = 15'd4369;
        8'd14  : MultFactor = 15'd4681;
        8'd13  : MultFactor = 15'd5041;
        8'd12  : MultFactor = 15'd5461;
        8'd11  : MultFactor = 15'd5958;
        8'd10  : MultFactor = 15'd6554;
        8'd9   : MultFactor = 15'd7282;
        8'd8   : MultFactor = 15'd8192;
        8'd7   : MultFactor = 15'd9362;
        8'd6   : MultFactor = 15'd10923;
        8'd5   : MultFactor = 15'd13107;
        8'd4   : MultFactor = 15'd16384;
        // additional for 40 Mhz HE
        8'd116 : MultFactor = 15'd565;
        8'd110 : MultFactor = 15'd596;
        8'd109 : MultFactor = 15'd601;
        8'd103 : MultFactor = 15'd636;
        8'd102 : MultFactor = 15'd643;
        8'd96  : MultFactor = 15'd683;
        8'd95  : MultFactor = 15'd690;
        8'd89  : MultFactor = 15'd736;
        8'd83  : MultFactor = 15'd790;
        8'd82  : MultFactor = 15'd799;
        8'd76  : MultFactor = 15'd862;
        8'd75  : MultFactor = 15'd874;
        8'd69  : MultFactor = 15'd950;
        8'd68  : MultFactor = 15'd964;
        8'd63  : MultFactor = 15'd1040;
        8'd61  : MultFactor = 15'd1074;
        8'd56  : MultFactor = 15'd1170;
        8'd55  : MultFactor = 15'd1192;
        8'd49  : MultFactor = 15'd1337;
        8'd48  : MultFactor = 15'd1365;
        8'd42  : MultFactor = 15'd1560;
        8'd41  : MultFactor = 15'd1598;
        8'd35  : MultFactor = 15'd1872;
        8'd34  : MultFactor = 15'd1928;
        8'd32  : MultFactor = 15'd2048;
        8'd31  : MultFactor = 15'd2114;
        8'd29  : MultFactor = 15'd2260;
        8'd27  : MultFactor = 15'd2427;
        8'd26  : MultFactor = 15'd2521;
        8'd25  : MultFactor = 15'd2621;
        8'd19  : MultFactor = 15'd3449;
        8'd3   : MultFactor = 15'd21845;
        // additional for 80 Mhz HE
        8'd250 : MultFactor = 15'd262;
        8'd244 : MultFactor = 15'd269;
        8'd238 : MultFactor = 15'd275;
        8'd237 : MultFactor = 15'd277;
        8'd231 : MultFactor = 15'd284;
        8'd230 : MultFactor = 15'd285;
        8'd224 : MultFactor = 15'd293;
        8'd223 : MultFactor = 15'd294;
        8'd217 : MultFactor = 15'd302;
        8'd211 : MultFactor = 15'd311;
        8'd210 : MultFactor = 15'd312;
        8'd204 : MultFactor = 15'd321;
        8'd203 : MultFactor = 15'd322;
        8'd197 : MultFactor = 15'd333;
        8'd196 : MultFactor = 15'd334;
        8'd191 : MultFactor = 15'd343;
        8'd190 : MultFactor = 15'd345;
        8'd184 : MultFactor = 15'd356;
        8'd183 : MultFactor = 15'd358;
        8'd170 : MultFactor = 15'd386;
        8'd176 : MultFactor = 15'd372;
        8'd177 : MultFactor = 15'd370;
        8'd164 : MultFactor = 15'd400;
        8'd163 : MultFactor = 15'd402;
        8'd157 : MultFactor = 15'd417;
        8'd156 : MultFactor = 15'd420;
        8'd150 : MultFactor = 15'd437;
        8'd149 : MultFactor = 15'd440;
        8'd144 : MultFactor = 15'd455;
        8'd143 : MultFactor = 15'd458;
        8'd137 : MultFactor = 15'd478;
        8'd136 : MultFactor = 15'd482;
        8'd130 : MultFactor = 15'd504;
        8'd129 : MultFactor = 15'd508;
        8'd123 : MultFactor = 15'd533;
        8'd115 : MultFactor = 15'd570;
        8'd117 : MultFactor = 15'd560;
        8'd97  : MultFactor = 15'd676;
        8'd90  : MultFactor = 15'd728;
        8'd88  : MultFactor = 15'd745;
        8'd81  : MultFactor = 15'd809;
        8'd70  : MultFactor = 15'd936;
        8'd60  : MultFactor = 15'd1092;
        8'd59  : MultFactor = 15'd1111;
        8'd54  : MultFactor = 15'd1214;
        8'd53  : MultFactor = 15'd1237;
        8'd51  : MultFactor = 15'd1285;
        8'd47  : MultFactor = 15'd1394;
        8'd46  : MultFactor = 15'd1425;
        8'd45  : MultFactor = 15'd1456;
        8'd40  : MultFactor = 15'd1638;
        8'd39  : MultFactor = 15'd1680;
        8'd38  : MultFactor = 15'd1725;
        8'd33  : MultFactor = 15'd1986;
        default: MultFactor = 15'd0;
//        begin
//          if (svdtop_tb.uutI.uut.SVDDone==1'b1) begin
//            $display ("ERROR! unlisted Ns %0d", tctlNs);
//            $stop;
//          end
//        end
      endcase
    end

  // Retiming
  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
      if (!nBFRModemRst)
        begin
          MultFactor_1t <= 16'd0;
          MultFactor_2t <= 16'd0;
        end
      else
        begin
          if (tctlNsValid)
            MultFactor_1t <= MultFactor;
          MultFactor_2t   <= MultFactor_1t;
        end
    end

  assign V1 = {8'd0,Sum} * {9'd0,MultFactor_2t};
  
  assign IntAvgSnr = {3'b000,accuSubAvg} - {cfgSnrFactor[10],cfgSnrFactor}; 
    
    
  //
  // Saturation calculations.
  // Assign accuSubAvg
  //   
  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
      if (!nBFRModemRst)
        begin
          accuAvgSnr <= 8'd0;
          accuSubAvg <= 9'd0;
        end
      else
        begin
          accuSubAvg <= V1[23:15]; // max value 255 * Ns * Nsk fits on 24 bits  
          //
          // Saturate IntAvgSnr to 8 bits.
          //
          if (IntAvgSnr > 12'sd127)      // Number too big.
            begin
              accuAvgSnr <= 8'd127;      // Positive saturation.
            end
          else if (IntAvgSnr < 12'shF80) // Number too small, 12'shF80 is -128.
            begin
              accuAvgSnr <= 8'h80;       // Negative saturation.
            end
          else
            begin
              accuAvgSnr <= IntAvgSnr[7:0];  // No saturation, just leave it alone.
            end
        end
    end
    
endmodule
