//  Copyright (C) by CEVA.
//  This module is a confidential and proprietary property of CEVA
//  and a possession or use of this module requires written permission 
//  from CEVA.
//----------------------------------------------------------------------------
// $Author: $
// Company          : CEVA
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : This block scales the hermitian matrix M and provides
//                    a parallel output. 
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
// $HeadURL: $
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none


module hermitianScale (
            ///////////////////////////////////////////////
            //$port_g Modem clock inputs
            ///////////////////////////////////////////////
            input  wire           BFRModemClk,
            input  wire           nBFRModemRst,
            
            ///////////////////////////////////////////////
            //$port_g Sub-carrier matrix input
            ///////////////////////////////////////////////
            input wire [26:0]     Mxy,
            input wire            MxyVld,
            input wire [1:0]      MxyId,

            ///////////////////////////////////////////////
            //$port_g Scaled hermitian matrix output.
            ///////////////////////////////////////////////
            
            output reg [11:0]    M11,
`ifdef RW_TXRX_2X2
            output reg [11:0]    M22,
            output reg [12:0]    MRe12,
            output reg [12:0]    MIm12,
`endif            
            output reg [3:0]     MScale
            );

reg [3:0]      MxyScaleCurrent;
reg [13:0]     MxyPreScaleCurrent;
reg [3:0]      largestScale;

reg [12:0]     MPreScale11;
reg [3:0]      MScale11;

reg [3:0]		M11ScaleDiff;

reg [11:0]     MRounded11;

`ifdef RW_TXRX_2X2
reg [12:0]     MPreScale22;
reg [3:0]      MScale22;
reg [13:0]     MRePreScale12;
reg [3:0]      MReScale12;
reg [13:0]     MImPreScale12Comb;

reg [11:0]     MRounded22;
reg [12:0]     MReRounded12;
reg [12:0]     MImRounded12;

reg [3:0]		M22ScaleDiff;
reg [3:0]		MRe12ScaleDiff;
reg [3:0]		MIm12ScaleDiff;
`endif

///////////////////////////////////////////////
// Control logic
///////////////////////////////////////////////
always @(posedge BFRModemClk or negedge nBFRModemRst)
begin
  if (nBFRModemRst == 1'b0)
  begin
      largestScale    <= 4'h0;
      MScale        <= 4'h0;
  end
  else if (MxyVld == 1'b1)
  begin
      case (MxyId)
      2'h0: begin
          largestScale    <= MxyScaleCurrent;
        end
      2'h3: begin
          if (largestScale < MxyScaleCurrent)
          begin
              largestScale    <= MxyScaleCurrent;
              MScale          <= MxyScaleCurrent; // put a + 10 here if scale should range from 11 to 25.
          end
          else
          begin
              MScale          <= largestScale; // put a + 10 here if scale should range from 11 to 25.
          end
        end
      default: begin
          if (largestScale < MxyScaleCurrent)
          begin
              largestScale    <= MxyScaleCurrent;
          end
        end
      endcase
  end
end


///////////////////////////////////////////////
// Generate generate current Mxy scale 
///////////////////////////////////////////////
always @* 
begin
  casex (Mxy[26:12])
    15'b000_0000_0000_0000: MxyScaleCurrent  = 4'd0;
    15'b000_0000_0000_0001: MxyScaleCurrent  = 4'd1;
    15'b000_0000_0000_001x: MxyScaleCurrent  = 4'd2;
    15'b000_0000_0000_01xx: MxyScaleCurrent  = 4'd3;
    15'b000_0000_0000_1xxx: MxyScaleCurrent  = 4'd4;
    15'b000_0000_0001_xxxx: MxyScaleCurrent  = 4'd5;
    15'b000_0000_001x_xxxx: MxyScaleCurrent  = 4'd6;
    15'b000_0000_01xx_xxxx: MxyScaleCurrent  = 4'd7;
    15'b000_0000_1xxx_xxxx: MxyScaleCurrent  = 4'd8;
    15'b000_0001_xxxx_xxxx: MxyScaleCurrent  = 4'd9;
    15'b000_001x_xxxx_xxxx: MxyScaleCurrent  = 4'd10;
    15'b000_01xx_xxxx_xxxx: MxyScaleCurrent  = 4'd11;
    15'b000_1xxx_xxxx_xxxx: MxyScaleCurrent  = 4'd12;
    15'b001_xxxx_xxxx_xxxx: MxyScaleCurrent  = 4'd13;
    15'b01x_xxxx_xxxx_xxxx: MxyScaleCurrent  = 4'd14;
    15'b111_1111_1111_1111: MxyScaleCurrent  = 4'd0;
    15'b111_1111_1111_1110: MxyScaleCurrent  = 4'd1;
    15'b111_1111_1111_110x: MxyScaleCurrent  = 4'd2;
    15'b111_1111_1111_10xx: MxyScaleCurrent  = 4'd3;
    15'b111_1111_1111_0xxx: MxyScaleCurrent  = 4'd4;
    15'b111_1111_1110_xxxx: MxyScaleCurrent  = 4'd5;
    15'b111_1111_110x_xxxx: MxyScaleCurrent  = 4'd6;
    15'b111_1111_10xx_xxxx: MxyScaleCurrent  = 4'd7;
    15'b111_1111_0xxx_xxxx: MxyScaleCurrent  = 4'd8;
    15'b111_1110_xxxx_xxxx: MxyScaleCurrent  = 4'd9;
    15'b111_110x_xxxx_xxxx: MxyScaleCurrent  = 4'd10;
    15'b111_10xx_xxxx_xxxx: MxyScaleCurrent  = 4'd11;
    15'b111_0xxx_xxxx_xxxx: MxyScaleCurrent  = 4'd12;
    15'b110_xxxx_xxxx_xxxx: MxyScaleCurrent  = 4'd13;
    15'b10x_xxxx_xxxx_xxxx: MxyScaleCurrent  = 4'd14;
    default:                MxyScaleCurrent  = 4'd14;
  endcase
end

always @* 
begin
  case (MxyScaleCurrent)
    4'd0:     MxyPreScaleCurrent  = {Mxy[12:0], 1'b0};
    4'd1:     MxyPreScaleCurrent  = Mxy[13:0];
    4'd2:     MxyPreScaleCurrent  = Mxy[14:1];
    4'd3:     MxyPreScaleCurrent  = Mxy[15:2];
    4'd4:     MxyPreScaleCurrent  = Mxy[16:3];
    4'd5:     MxyPreScaleCurrent  = Mxy[17:4];
    4'd6:     MxyPreScaleCurrent  = Mxy[18:5];
    4'd7:     MxyPreScaleCurrent  = Mxy[19:6];
    4'd8:     MxyPreScaleCurrent  = Mxy[20:7];
    4'd9:     MxyPreScaleCurrent  = Mxy[21:8];
    4'd10:    MxyPreScaleCurrent  = Mxy[22:9];
    4'd11:    MxyPreScaleCurrent  = Mxy[23:10];
    4'd12:    MxyPreScaleCurrent  = Mxy[24:11];
    4'd13:    MxyPreScaleCurrent  = Mxy[25:12];
    4'd14:    MxyPreScaleCurrent  = Mxy[26:13];
    default:  MxyPreScaleCurrent  = {Mxy[12:0], 1'b0};
  endcase
end


///////////////////////////////////////////////
// Capturing Mxy pre scaled to it's own range
///////////////////////////////////////////////
always @(posedge BFRModemClk or negedge nBFRModemRst)
begin
  if (nBFRModemRst == 1'b0)
  begin
      MPreScale11   <= 13'h0;
      MScale11      <= 4'h0;
`ifdef RW_TXRX_2X2
      MPreScale22   <= 13'h0;
      MScale22      <= 4'h0;
      MRePreScale12 <= 14'h0;
      MReScale12    <= 4'h0;
`endif
  end
  else if (MxyVld == 1'b1)
  begin
      case (MxyId)
      2'h0: begin
          MPreScale11   <= MxyPreScaleCurrent[12:0];
          MScale11      <= MxyScaleCurrent;
        end
`ifdef RW_TXRX_2X2
      2'h1: begin
          MPreScale22   <= MxyPreScaleCurrent[12:0];
          MScale22      <= MxyScaleCurrent;
        end
      2'h2: begin
          MRePreScale12 <= MxyPreScaleCurrent;
          MReScale12    <= MxyScaleCurrent;
        end
`endif
      default: begin
        end
      endcase
  end
end

///////////////////////////////////////////////
// Capturing Mxy pre scaled to it's own range
///////////////////////////////////////////////
always @(posedge BFRModemClk or negedge nBFRModemRst)
begin
  if (nBFRModemRst == 1'b0)
  begin
      M11   <= 12'h0;
`ifdef RW_TXRX_2X2
      M22   <= 12'h0;
      MRe12 <= 13'h0;
      MIm12 <= 13'h0;
`endif //  RW_TXRX_2X2
  end
  else if (MxyVld == 1'b1 && MxyId == 2'h3) begin
    M11   <= MRounded11;
`ifdef RW_TXRX_2X2
    M22   <= MRounded22;
    MRe12 <= MReRounded12;
    MIm12 <= MImRounded12;
`endif //  RW_TXRX_2X2
  end
end

always @* 
begin
    if (largestScale < MxyScaleCurrent)
    begin
        M11ScaleDiff = MxyScaleCurrent-MScale11;
    end
    else
    begin
        M11ScaleDiff = largestScale-MScale11;
    end
end

always @* 
begin
    case (M11ScaleDiff)
    4'h0: begin
        if (MPreScale11[12:1] == 12'hFFF)
            MRounded11[11:0] = MPreScale11[12:1];
        else if (MPreScale11[0] == 1'b1)
            MRounded11[11:0] = MPreScale11[12:1] + 12'd1;
        else
            MRounded11[11:0] = MPreScale11[12:1];
      end
    4'h1: begin
        if (MPreScale11[1] == 1'b1)
            MRounded11[11:0] = {1'b0,MPreScale11[12:2]} + 12'd1;
        else
            MRounded11[11:0] = {1'b0,MPreScale11[12:2]};
      end
    4'h2: begin
        if (MPreScale11[2] == 1'b1)
            MRounded11[11:0] = {2'd0,MPreScale11[12:3]} + 12'd1;
        else
            MRounded11[11:0] = {2'd0,MPreScale11[12:3]};
      end
    4'h3: begin
        if (MPreScale11[3] == 1'b1)
            MRounded11[11:0] = {3'd0,MPreScale11[12:4]} + 12'd1;
        else
            MRounded11[11:0] = {3'd0,MPreScale11[12:4]};
      end
    4'h4: begin
        if (MPreScale11[4] == 1'b1)
            MRounded11[11:0] = {4'd0,MPreScale11[12:5]} + 12'd1;
        else
            MRounded11[11:0] = {4'd0,MPreScale11[12:5]};
      end
    4'h5: begin
        if (MPreScale11[5] == 1'b1)
            MRounded11[11:0] = {5'd0,MPreScale11[12:6]} + 12'd1;
        else
            MRounded11[11:0] = {5'd0,MPreScale11[12:6]};
      end
    4'h6: begin
        if (MPreScale11[6] == 1'b1)
            MRounded11[11:0] = {6'd0,MPreScale11[12:7]} + 12'd1;
        else
            MRounded11[11:0] = {6'd0,MPreScale11[12:7]};
      end
    4'h7: begin
        if (MPreScale11[7] == 1'b1)
            MRounded11[11:0] = {7'd0,MPreScale11[12:8]} + 12'd1;
        else
            MRounded11[11:0] = {7'd0,MPreScale11[12:8]};
      end
    4'h8: begin
        if (MPreScale11[8] == 1'b1)
            MRounded11[11:0] = {8'd0,MPreScale11[12:9]} + 12'd1;
        else
            MRounded11[11:0] = {8'd0,MPreScale11[12:9]};
      end
    4'h9: begin
        if (MPreScale11[9] == 1'b1)
            MRounded11[11:0] = {9'd0,MPreScale11[12:10]} + 12'd1;
        else
            MRounded11[11:0] = {9'd0,MPreScale11[12:10]};
      end
    4'hA: begin
        if (MPreScale11[10] == 1'b1)
            MRounded11[11:0] = {10'd0,MPreScale11[12:11]} + 12'd1;
        else
            MRounded11[11:0] = {10'd0,MPreScale11[12:11]};
      end
    4'hB: begin
        if (MPreScale11[11] == 1'b1)
            MRounded11[11:0] = {11'd0,MPreScale11[12]} + 12'd1;
        else
            MRounded11[11:0] = {11'd0,MPreScale11[12]};
      end
    4'hC: begin
        MRounded11[11:0] = {11'd0,MPreScale11[12]};
      end
    default: begin
        MRounded11 = 12'h0;
      end
    endcase
end

`ifdef RW_TXRX_2X2
always @* 
begin
    if (largestScale < MxyScaleCurrent)
    begin
        M22ScaleDiff = MxyScaleCurrent-MScale22;
    end
    else
    begin
        M22ScaleDiff = largestScale-MScale22;
    end
end

always @* 
begin
    case (M22ScaleDiff)
    4'h0: begin
        if (MPreScale22[12:1] == 12'hFFF)
            MRounded22[11:0] = MPreScale22[12:1];
        else if (MPreScale22[0] == 1'b1)
            MRounded22[11:0] = MPreScale22[12:1] + 12'd1;
        else
            MRounded22[11:0] = MPreScale22[12:1];
      end
    4'h1: begin
        if (MPreScale22[1] == 1'b1)
            MRounded22[11:0] = {1'b0,MPreScale22[12:2]} + 12'd1;
        else
            MRounded22[11:0] = {1'b0,MPreScale22[12:2]};
      end
    4'h2: begin
        if (MPreScale22[2] == 1'b1)
            MRounded22[11:0] = {2'd0,MPreScale22[12:3]} + 12'd1;
        else
            MRounded22[11:0]  = {2'd0,MPreScale22[12:3]};
      end
    4'h3: begin
        if (MPreScale22[3] == 1'b1)
            MRounded22[11:0] = {3'd0,MPreScale22[12:4]} + 12'd1;
        else
            MRounded22[11:0] = {3'd0,MPreScale22[12:4]};
      end
    4'h4: begin
        if (MPreScale22[4] == 1'b1)
            MRounded22[11:0] = {4'd0,MPreScale22[12:5]} + 12'd1;
        else
            MRounded22[11:0] = {4'd0,MPreScale22[12:5]};
      end
    4'h5: begin
        if (MPreScale22[5] == 1'b1)
            MRounded22[11:0] = {5'd0,MPreScale22[12:6]} + 12'd1;
        else
            MRounded22[11:0] = {5'd0,MPreScale22[12:6]};
      end
    4'h6: begin
        if (MPreScale22[6] == 1'b1)
            MRounded22[11:0] = {6'd0,MPreScale22[12:7]} + 12'd1;
        else
            MRounded22[11:0] = {6'd0,MPreScale22[12:7]};
      end
    4'h7: begin
        if (MPreScale22[7] == 1'b1)
            MRounded22[11:0] = {7'd0,MPreScale22[12:8]} + 12'd1;
        else
            MRounded22[11:0] = {7'd0,MPreScale22[12:8]};
      end
    4'h8: begin
        if (MPreScale22[8] == 1'b1)
            MRounded22[11:0] = {8'd0,MPreScale22[12:9]} + 12'd1;
        else
            MRounded22[11:0] = {8'd0,MPreScale22[12:9]};
      end
    4'h9: begin
        if (MPreScale22[9] == 1'b1)
            MRounded22[11:0] = {9'd0,MPreScale22[12:10]} + 12'd1;
        else
            MRounded22[11:0] = {9'd0,MPreScale22[12:10]};
      end
    4'hA: begin
        if (MPreScale22[10] == 1'b1)
            MRounded22[11:0] = {10'd0,MPreScale22[12:11]} + 12'd1;
        else
            MRounded22[11:0] = {10'd0,MPreScale22[12:11]};
      end
    4'hB: begin
        if (MPreScale22[11] == 1'b1)
            MRounded22[11:0] = {11'd0,MPreScale22[12]} + 12'd1;
        else
            MRounded22[11:0] = {11'd0,MPreScale22[12]};
      end
    4'hC: begin
        MRounded22[11:0] = {11'd0,MPreScale22[12]};
      end
    default: begin
        MRounded22 = 12'h0;
      end
    endcase
end

always @* 
begin
    if (largestScale < MxyScaleCurrent)
    begin
        MRe12ScaleDiff = MxyScaleCurrent-MReScale12;
    end
    else
    begin
        MRe12ScaleDiff = largestScale-MReScale12;
    end
end

always @* 
begin
    case (MRe12ScaleDiff)
    4'h0: begin
        if (MRePreScale12[13:1] == 13'hFFF)  // saturation, no rounding.
            MReRounded12[12:0] = MRePreScale12[13:1];
        else if (MRePreScale12[0] == 1'b1)
            MReRounded12[12:0] = MRePreScale12[13:1] + 13'd1;
        else
            MReRounded12[12:0] = MRePreScale12[13:1];
      end
    4'h1: begin
        if (MRePreScale12[13] == 1'b1 && MRePreScale12[1] == 1'b1)
            MReRounded12[12:0] = {{1'h1, MRePreScale12[13:2]} + 13'd1};
        else if (MRePreScale12[13] == 1'b1)
            MReRounded12[12:0] = {1'h1, MRePreScale12[13:2]};
        else if (MRePreScale12[1] == 1'b1)
            MReRounded12[12:0] = {2'd0,MRePreScale12[12:2]} + 13'd1;
        else
            MReRounded12[12:0] = {2'd0,MRePreScale12[12:2]};
      end
    4'h2: begin
        if (MRePreScale12[13] == 1'b1 && MRePreScale12[2] == 1'b1)
            MReRounded12[12:0] = {{2'h3, MRePreScale12[13:3]} + 13'd1};
        else if (MRePreScale12[13] == 1'b1)
            MReRounded12[12:0] = {2'h3, MRePreScale12[13:3]};
        else if (MRePreScale12[2] == 1'b1)
            MReRounded12[12:0] = {3'd0,MRePreScale12[12:3]} + 13'd1;
        else
            MReRounded12[12:0]  = {3'd0,MRePreScale12[12:3]};
      end
    4'h3: begin
        if (MRePreScale12[13] == 1'b1 && MRePreScale12[3] == 1'b1)
            MReRounded12[12:0] = {{3'h7, MRePreScale12[13:4]} + 13'd1};
        else if (MRePreScale12[13] == 1'b1)
            MReRounded12[12:0] = {3'h7, MRePreScale12[13:4]};
        else if (MRePreScale12[3] == 1'b1)
            MReRounded12[12:0] = {4'd0,MRePreScale12[12:4]} + 13'd1;
        else
            MReRounded12[12:0] = {4'd0,MRePreScale12[12:4]};
      end
    4'h4: begin
        if (MRePreScale12[13] == 1'b1 && MRePreScale12[4] == 1'b1)
            MReRounded12[12:0] = {{4'hF, MRePreScale12[13:5]} + 13'd1};
        else if (MRePreScale12[13] == 1'b1)
            MReRounded12[12:0] = {4'hF, MRePreScale12[13:5]};
        else if (MRePreScale12[4] == 1'b1)
            MReRounded12[12:0] = {5'd0,MRePreScale12[12:5]} + 13'd1;
        else
            MReRounded12[12:0] = {5'd0,MRePreScale12[12:5]};
      end
    4'h5: begin
        if (MRePreScale12[13] == 1'b1 && MRePreScale12[5] == 1'b1)
            MReRounded12[12:0] = {{5'h1F, MRePreScale12[13:6]} + 13'd1};
        else if (MRePreScale12[13] == 1'b1)
            MReRounded12[12:0] = {5'h1F, MRePreScale12[13:6]};
        else if (MRePreScale12[5] == 1'b1)
            MReRounded12[12:0] = {6'd0,MRePreScale12[12:6]} + 13'd1;
        else
            MReRounded12[12:0] = {6'd0,MRePreScale12[12:6]};
      end
    4'h6: begin
        if (MRePreScale12[13] == 1'b1 && MRePreScale12[6] == 1'b1)
            MReRounded12[12:0] = {{6'h3F, MRePreScale12[13:7]} + 13'd1};
        else if (MRePreScale12[13] == 1'b1)
            MReRounded12[12:0] = {6'h3F, MRePreScale12[13:7]};
        else if (MRePreScale12[6] == 1'b1)
            MReRounded12[12:0] = {7'd0,MRePreScale12[12:7]} + 13'd1;
        else
            MReRounded12[12:0] = {7'd0,MRePreScale12[12:7]};
      end
    4'h7: begin
        if (MRePreScale12[13] == 1'b1 && MRePreScale12[7] == 1'b1)
            MReRounded12[12:0] = {{7'h7F, MRePreScale12[13:8]} + 13'd1};
        else if (MRePreScale12[13] == 1'b1)
            MReRounded12[12:0] = {7'h7F, MRePreScale12[13:8]};
        else if (MRePreScale12[7] == 1'b1)
            MReRounded12[12:0] = {8'd0,MRePreScale12[12:8]} + 13'd1;
        else
            MReRounded12[12:0] = {8'd0,MRePreScale12[12:8]};
      end
    4'h8: begin
        if (MRePreScale12[13] == 1'b1 && MRePreScale12[8] == 1'b1)
            MReRounded12[12:0] = {{8'hFF, MRePreScale12[13:9]} + 13'd1};
        else if (MRePreScale12[13] == 1'b1)
            MReRounded12[12:0] = {8'hFF, MRePreScale12[13:9]};
        else if (MRePreScale12[8] == 1'b1)
            MReRounded12[12:0] = {9'd0,MRePreScale12[12:9]} + 13'd1;
        else
            MReRounded12[12:0] = {9'd0,MRePreScale12[12:9]};
      end
    4'h9: begin
        if (MRePreScale12[13] == 1'b1 && MRePreScale12[9] == 1'b1)
            MReRounded12[12:0] = {{9'h1FF, MRePreScale12[13:10]} + 13'd1};
        else if (MRePreScale12[13] == 1'b1)
            MReRounded12[12:0] = {9'h1FF, MRePreScale12[13:10]};
        else if (MRePreScale12[9] == 1'b1)
            MReRounded12[12:0] = {10'd0,MRePreScale12[12:10]} + 13'd1;
        else
            MReRounded12[12:0] = {10'd0,MRePreScale12[12:10]};
      end
    4'hA: begin
        if (MRePreScale12[13] == 1'b1 && MRePreScale12[10] == 1'b1)
            MReRounded12[12:0] = {{10'h3FF, MRePreScale12[13:11]} + 13'd1};
        else if (MRePreScale12[13] == 1'b1)
            MReRounded12[12:0] = {10'h3FF, MRePreScale12[13:11]};
        else if (MRePreScale12[10] == 1'b1)
            MReRounded12[12:0] = {11'd0,MRePreScale12[12:11]} + 13'd1;
        else
            MReRounded12[12:0] = {11'd0,MRePreScale12[12:11]};
      end
    4'hB: begin
        if (MRePreScale12[13] == 1'b1 && MRePreScale12[11] == 1'b1)
            MReRounded12[12:0] = {{11'h7FF, MRePreScale12[13:12]} + 13'd1};
        else if (MRePreScale12[13] == 1'b1)
            MReRounded12[12:0] = {11'h7FF, MRePreScale12[13:12]};
        else if (MRePreScale12[11] == 1'b1)
            MReRounded12[12:0] = {12'd0,MRePreScale12[12]} + 13'd1;
        else
            MReRounded12[12:0] = {12'd0,MRePreScale12[12]};
      end
    4'hC: begin
        if (MRePreScale12[13:12] == 2'b11)
            MReRounded12 = 13'h0;
        else if (MRePreScale12[13] == 1'b1)
            MReRounded12 = 13'h1FFF;
        else
            MReRounded12 = {12'd0,MRePreScale12[12]};
      end
    default: begin
        if (MRePreScale12[13:12] == 2'b11)      // 11xx
          if (MRePreScale12[11:0] == 12'h800)        // 110x
            MReRounded12 = 13'h1FFF;
          else 
            MReRounded12 = 13'h0;
        else if (MRePreScale12[13:12] == 2'b01) // 01xx
          MReRounded12 = 13'd0;
        else
          MReRounded12 = 13'd0;                 // 10xx and 00xx 
      end
    endcase
end

always @* 
begin
    if (largestScale < MxyScaleCurrent)
    begin
        MIm12ScaleDiff = 4'h0;
    end
    else
    begin
        MIm12ScaleDiff = largestScale - MxyScaleCurrent;
    end
    MImPreScale12Comb = MxyPreScaleCurrent;
end

always @* 
begin
    MImRounded12 = 13'h0;
    case (MIm12ScaleDiff)
    4'h0: begin
        if (MImPreScale12Comb[13:1] == 13'hFFF)  // saturation, no rounding.
            MImRounded12[12:0] = MImPreScale12Comb[13:1];
        else if (MImPreScale12Comb[0] == 1'b1)
            MImRounded12[12:0] = MImPreScale12Comb[13:1] + 13'd1;
        else
            MImRounded12[12:0] = MImPreScale12Comb[13:1];
      end
    4'h1: begin
        if (MImPreScale12Comb[13] == 1'b1 && MImPreScale12Comb[1] == 1'b1)
            MImRounded12[12:0] = {{1'h1, MImPreScale12Comb[13:2]} + 13'd1};
        else if (MImPreScale12Comb[13] == 1'b1)
            MImRounded12[12:0] = {1'h1, MImPreScale12Comb[13:2]};
        else if (MImPreScale12Comb[1] == 1'b1)
            MImRounded12[12:0] = {2'd0,MImPreScale12Comb[12:2]} + 13'd1;
        else
            MImRounded12[12:0] = {2'd0,MImPreScale12Comb[12:2]};
      end
    4'h2: begin
        if (MImPreScale12Comb[13] == 1'b1 && MImPreScale12Comb[2] == 1'b1)
            MImRounded12[12:0] = {{2'h3, MImPreScale12Comb[13:3]} + 13'd1};
        else if (MImPreScale12Comb[13] == 1'b1)
            MImRounded12[12:0] = {2'h3, MImPreScale12Comb[13:3]};
        else if (MImPreScale12Comb[2] == 1'b1)
            MImRounded12[12:0] = {3'd0,MImPreScale12Comb[12:3]} + 13'd1;
        else
            MImRounded12[12:0] = {3'd0,MImPreScale12Comb[12:3]};
      end
    4'h3: begin
        if (MImPreScale12Comb[13] == 1'b1 && MImPreScale12Comb[3] == 1'b1)
            MImRounded12[12:0] = {{3'h7, MImPreScale12Comb[13:4]} + 13'd1};
        else if (MImPreScale12Comb[13] == 1'b1)
            MImRounded12[12:0] = {3'h7, MImPreScale12Comb[13:4]};
        else if (MImPreScale12Comb[3] == 1'b1)
            MImRounded12[12:0] = {4'd0,MImPreScale12Comb[12:4]} + 13'd1;
        else
            MImRounded12[12:0] = {4'd0,MImPreScale12Comb[12:4]};
      end
    4'h4: begin
        if (MImPreScale12Comb[13] == 1'b1 && MImPreScale12Comb[4] == 1'b1)
            MImRounded12[12:0] = {{4'hF, MImPreScale12Comb[13:5]} + 13'd1};
        else if (MImPreScale12Comb[13] == 1'b1)
            MImRounded12[12:0] = {4'hF, MImPreScale12Comb[13:5]};
        else if (MImPreScale12Comb[4] == 1'b1)
            MImRounded12[12:0] = {5'd0,MImPreScale12Comb[12:5]} + 13'd1;
        else
            MImRounded12[12:0] = {5'd0,MImPreScale12Comb[12:5]};
      end
    4'h5: begin
        if (MImPreScale12Comb[13] == 1'b1 && MImPreScale12Comb[5] == 1'b1)
            MImRounded12[12:0] = {{5'h1F, MImPreScale12Comb[13:6]} + 13'd1};
        else if (MImPreScale12Comb[13] == 1'b1)
            MImRounded12[12:0] = {5'h1F, MImPreScale12Comb[13:6]};
        else if (MImPreScale12Comb[5] == 1'b1)
            MImRounded12[12:0] = {6'd0,MImPreScale12Comb[12:6]} + 13'd1;
        else
            MImRounded12[12:0] = {6'd0,MImPreScale12Comb[12:6]};
      end
    4'h6: begin
        if (MImPreScale12Comb[13] == 1'b1 && MImPreScale12Comb[6] == 1'b1)
            MImRounded12[12:0] = {{6'h3F, MImPreScale12Comb[13:7]} + 13'd1};
        else if (MImPreScale12Comb[13] == 1'b1)
            MImRounded12[12:0] = {6'h3F, MImPreScale12Comb[13:7]};
        else if (MImPreScale12Comb[6] == 1'b1)
            MImRounded12[12:0] = {7'd0,MImPreScale12Comb[12:7]} + 13'd1;
        else
            MImRounded12[12:0] = {7'd0,MImPreScale12Comb[12:7]};
      end
    4'h7: begin
        if (MImPreScale12Comb[13] == 1'b1 && MImPreScale12Comb[7] == 1'b1)
            MImRounded12[12:0] = {{7'h7F, MImPreScale12Comb[13:8]} + 13'd1};
        else if (MImPreScale12Comb[13] == 1'b1)
            MImRounded12[12:0] = {7'h7F, MImPreScale12Comb[13:8]};
        else if (MImPreScale12Comb[7] == 1'b1)
            MImRounded12[12:0] = {8'd0,MImPreScale12Comb[12:8]} + 13'd1;
        else
            MImRounded12[12:0] = {8'd0,MImPreScale12Comb[12:8]};
      end
    4'h8: begin
        if (MImPreScale12Comb[13] == 1'b1 && MImPreScale12Comb[8] == 1'b1)
            MImRounded12[12:0] = {{8'hFF, MImPreScale12Comb[13:9]} + 13'd1};
        else if (MImPreScale12Comb[13] == 1'b1)
            MImRounded12[12:0] = {8'hFF, MImPreScale12Comb[13:9]};
        else if (MImPreScale12Comb[8] == 1'b1)
            MImRounded12[12:0] = {9'd0,MImPreScale12Comb[12:9]} + 13'd1;
        else
            MImRounded12[12:0] = {9'd0,MImPreScale12Comb[12:9]};
      end
    4'h9: begin
        if (MImPreScale12Comb[13] == 1'b1 && MImPreScale12Comb[9] == 1'b1)
            MImRounded12[12:0] = {{9'h1FF, MImPreScale12Comb[13:10]} + 13'd1};
        else if (MImPreScale12Comb[13] == 1'b1)
            MImRounded12[12:0] = {9'h1FF, MImPreScale12Comb[13:10]};
        else if (MImPreScale12Comb[9] == 1'b1)
            MImRounded12[12:0] = {10'd0,MImPreScale12Comb[12:10]} + 13'd1;
        else
            MImRounded12[12:0] = {10'd0,MImPreScale12Comb[12:10]};
      end
    4'hA: begin
        if (MImPreScale12Comb[13] == 1'b1 && MImPreScale12Comb[10] == 1'b1)
            MImRounded12[12:0] = {{10'h3FF, MImPreScale12Comb[13:11]} + 13'd1};
        else if (MImPreScale12Comb[13] == 1'b1)
            MImRounded12[12:0] = {10'h3FF, MImPreScale12Comb[13:11]};
        else if (MImPreScale12Comb[10] == 1'b1)
            MImRounded12[12:0] = {11'd0,MImPreScale12Comb[12:11]} + 13'd1;
        else
            MImRounded12[12:0] = {11'd0,MImPreScale12Comb[12:11]};
      end
    4'hB: begin
        if (MImPreScale12Comb[13] == 1'b1 && MImPreScale12Comb[11] == 1'b1)
            MImRounded12[12:0] = {{11'h7FF, MImPreScale12Comb[13:12]} + 13'd1};
        else if (MImPreScale12Comb[13] == 1'b1)
            MImRounded12[12:0] = {11'h7FF, MImPreScale12Comb[13:12]};
        else if (MImPreScale12Comb[11] == 1'b1)
            MImRounded12[12:0] = {12'd0,MImPreScale12Comb[12]} + 13'd1;
        else
            MImRounded12[12:0] = {12'd0,MImPreScale12Comb[12]};
      end
    4'hC: begin
        if (MImPreScale12Comb[13:12] == 2'b11)
            MImRounded12 = 13'h0;
        else if (MImPreScale12Comb[13] == 1'b1)
            MImRounded12 = 13'h1FFF;
        else 
            MImRounded12 = {12'd0,MImPreScale12Comb[12]};
      end
    default: begin
        if (MImPreScale12Comb[13:12] == 2'b11)
          if (MImPreScale12Comb[11:0] == 12'h800)        // 110x
            MImRounded12 = 13'h1FFF;
          else
            MImRounded12 = 13'h0;
        else if (MImPreScale12Comb[13:12] == 2'b01) // 01xx
          MImRounded12 = 13'd0;
        else 
          MImRounded12 = 13'd0;                     // 10xx and 00xx
      end
    endcase
end
`endif

endmodule

//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////



