//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : 
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none
module RamIf # (
  parameter RAM_AWIDTH =  10,
  parameter RAM_DWIDTH =  128,
  parameter SKIP_ADDR = 11'd1171
)( 
  /* system */
  input  wire                   nRst,
  input  wire                   Clk,

  /* control */
  input  wire                   Enable,
  input  wire                   Start,
  input  wire [RAM_AWIDTH-1:0]  SkipAddress,
  input  wire [RAM_AWIDTH-1:0]  StopAddress,

  /* Memory interface */
  output reg  [RAM_AWIDTH-1:0]  RAMAddr,
  output reg                    RAMWrEn,
  output reg  [RAM_DWIDTH-1:0]  RAMWrData,
  output wire                   RAMRdEn,
  input  wire [RAM_DWIDTH-1:0]  RAMRdData,
  
  /* Data interface */
  input  wire                   DataWrEn,
  input  wire [RAM_AWIDTH-1:0]  DataWrAddr,
  input  wire [RAM_DWIDTH-1:0]  DataWrData,

  output wire [RAM_DWIDTH-1:0]  DataOut,
  output wire                   DataOutLastSkip,
  output wire                   DataOutLast,
  output wire                   DataOutValid,
  input  wire                   DataOutReady

  );

  /******************************************************************************
  * declarations
  ******************************************************************************/
  localparam   [RAM_AWIDTH-1:0] SIG_ONE_DSP_AWIDTH_PARAM = 'b1;

  // Internal data stages
  wire [RAM_DWIDTH-1:0] s1Data;
  wire nS1Valid, loadStall, s2StallReady, s3Ready;
  wire [RAM_DWIDTH-1:0] nS2Data, nS2StallData, nS3Data;
  wire nS2Valid, nS2StallValid, nS2Ready;
  wire nS3Valid;
  wire loadS3withStall, loadS3withS2;
  reg  [RAM_DWIDTH-1:0] s2Data, s2StallData, s3Data;
  wire [RAM_DWIDTH-1:0] ns2ors3Data;
  reg  s1Valid, s2Valid, s2StallValid, s3Valid;
  reg  s2Ready;
  wire s1LastSkip,nS2LastSkip,nS2StallLastSkip,ns2ors3LastSkip,nS3LastSkip;
  reg  s2LastSkip,s2StallLastSkip,s3LastSkip;
  wire s1Last,nS2Last,nS2StallLast,ns2ors3Last,nS3Last;
  reg  s2Last,s2StallLast,s3Last;
  // ensure that after enable=0, the block does not start before Start=1
  reg  busy;
  wire LastRdAccess, LastRdAccessSkip;
  reg  LastRdData, LastRdDataSkip;
  
  reg  RAMRdEnInt;
  reg  StartD;

  /******************************************************************************
  * Output
  ******************************************************************************/
  assign DataOutValid = s3Valid && !Start;   
  assign DataOutLast  = s3Last;
  assign DataOutLastSkip  = s3LastSkip;
  assign DataOut      = s3Data;


  /******************************************************************************
  * Data processing
  ******************************************************************************/
  /* The block is organized as a pipeline 
   * - Stage 1 mimics the RAM data port. Only Valid is implemented here. 
   * - Stage 2 is used to immediately register the RAM data. As it is not possible to flow-control 
   * the RAM interface, Stage 2 also contains a stall buffer to store the data in case Stage 3
   * is not ready to accept it.
   * - Stage 3 is the output stage. 
   * A ready/valid interface is used through the stages, except that the RAM interface can't be flow controlled,
   * so the stall buffer in stage 2 is used instead. */
  /******************************************************************************
  * Stage 1
  ******************************************************************************/
  // s1Ready not used as RAM interface can't be flow controlled
  assign {s1Last,s1LastSkip,s1Data} = {LastRdData,LastRdDataSkip,RAMRdData}; 
  // s1Data is saved at each cycle in s2Data or s2StallData. When an adress stays several clock 
  // cycles because of s3Ready, the data is valid only until it is loaded in the next stage.
  assign nS1Valid = RAMRdEnInt && (s3Ready || !s2Valid || !s2StallValid);

  /******************************************************************************
  * Stage 2
  ******************************************************************************/
  /* Stage 2 is ready to accept data when 
   * - stage 2 data can be transferred out, either because stage 3 is empty (s3Valid=0)
   * or because stage 3 data can be updated (s3Ready=1). Note that stage 3 is fed in priority with
   * data from the stall buffer, but in this case the stall buffer itself is freed to accept stage 2 data.
   * - stage 2 is empty (s2Valid=0) */
  assign nS2Ready = s3Ready || !s3Valid || !s2Valid;

  assign {nS2Last,nS2LastSkip,nS2Data}  = (s1Valid && s2Ready) ? {s1Last,s1LastSkip,s1Data} : {s2Last,s2LastSkip,s2Data};
  assign nS2Valid = s2Ready ? s1Valid : s2Valid;

  /* As it is not possible to forward the ready to the RAM interface, implement a stall buffer
   * to save the data that can't be used at once */
  // Load stall buffer when there is valid data inside the stage, and 
  // - stage 3 is not ready to accept it.
  // - stage 3 is ready to accept data, but older data is valid inside the stall buffer, so stage 2 
  // data won't be sent to stage 3 at this cycle and must be stalled
  assign loadStall      = s2Valid && ( (!s3Ready && s3Valid)|| (s2StallValid && s3Ready) );
  // The buffer content must be frozen when it contains valid data, and stage 3 is not ready to accept it
  assign s2StallReady   = ! (s2StallValid && !s3Ready);
  assign {nS2StallLast,nS2StallLastSkip,nS2StallData} = (loadStall && s2StallReady) ?
                                                        {s2Last,s2LastSkip,s2Data} : {s2StallLast,s2StallLastSkip,s2StallData};
  assign nS2StallValid = s2StallReady ? (loadStall && !(s2StallValid && s2Ready)) : s2StallValid;

  /******************************************************************************
  * Stage 3
  ******************************************************************************/
  assign s3Ready  = DataOutReady;   // assign DataOutReady to s3Ready for readability
  // Stage 3 is loaded with data from Stage 2, using in priority oldest data, i.e.
  // data from stall buffer.
  assign loadS3withStall = (s2StallValid && s3Ready) || (s2StallValid && !s3Ready && !s3Valid);
  assign loadS3withS2    = (s2Valid       && s3Ready) || (s2Valid       && !s3Ready && !s3Valid);
  assign {ns2ors3Last,ns2ors3LastSkip,ns2ors3Data} = loadS3withS2 ? {s2Last,s2LastSkip,s2Data} : {s3Last,s3LastSkip,s3Data};
  assign {nS3Last,nS3LastSkip,nS3Data} = loadS3withStall ? {s2StallLast,s2StallLastSkip,s2StallData}  : {ns2ors3Last,ns2ors3LastSkip,ns2ors3Data}; 
assign nS3Valid    = (loadS3withStall || loadS3withS2) ? (s2StallValid || s2Valid) : s3Valid;

  /******************************************************************************
  * Registers
  ******************************************************************************/
  always @(posedge Clk, negedge nRst)
    if(!nRst) begin
      s1Valid         <= 1'd0;
      s2Data          <= {RAM_DWIDTH{1'b0}};
      s2Last          <= 1'd0;
      s2LastSkip      <= 1'd0;
      s2Valid         <= 1'd0;
      s2Ready         <= 1'd1;
      s2StallData     <= {RAM_DWIDTH{1'b0}};
      s2StallLast     <= 1'd0;
      s2StallLastSkip <= 1'd0;
      s2StallValid    <= 1'd0;
      s3Data          <= {RAM_DWIDTH{1'b0}};
      s3Last          <= 1'd0;
      s3LastSkip      <= 1'd0;
      s3Valid         <= 1'd0;
    end
    else if(!Enable) begin
      s1Valid         <= 1'd0;
      s2Data          <= {RAM_DWIDTH{1'b0}};
      s2Last          <= 1'd0;
      s2LastSkip      <= 1'd0;
      s2Valid         <= 1'd0;
      s2Ready         <= 1'd1;
      s2StallData     <= {RAM_DWIDTH{1'b0}};
      s2StallLast     <= 1'd0;
      s2StallLastSkip <= 1'd0;
      s2StallValid    <= 1'd0;
      s3Data          <= {RAM_DWIDTH{1'b0}};
      s3Last          <= 1'd0;
      s3LastSkip      <= 1'd0;
      s3Valid         <= 1'd0;
    end else begin
      if(Start) begin
        s1Valid         <= 1'd0;
        s2Data          <= {RAM_DWIDTH{1'b0}};
        s2Last          <= 1'd0;
        s2LastSkip      <= 1'd0;
        s2Valid         <= 1'd0;
        s2Ready         <= 1'd1;
        s2StallData     <= {RAM_DWIDTH{1'b0}};
        s2StallLast     <= 1'd0;
        s2StallLastSkip <= 1'd0;
        s2StallValid    <= 1'd0;
        s3Data          <= {RAM_DWIDTH{1'b0}};
        s3Last          <= 1'd0;
        s3LastSkip      <= 1'd0;
        s3Valid         <= 1'd0;
      end else begin
        s1Valid         <= nS1Valid;
        s2Data          <= nS2Data;
        s2Last          <= nS2Last;
        s2LastSkip      <= nS2LastSkip;
        s2Valid         <= nS2Valid;
        s2Ready         <= nS2Ready;
        s2StallData     <= nS2StallData;
        s2StallLast     <= nS2StallLast;
        s2StallLastSkip <= nS2StallLastSkip;
        s2StallValid    <= nS2StallValid;
        s3Data          <= nS3Data;
        s3Last          <= nS3Last;
        s3LastSkip      <= nS3LastSkip;
        s3Valid         <= nS3Valid;
      end
    end

  /******************************************************************************
  * control
  ******************************************************************************/
  assign LastRdAccess     = (RAMAddr==StopAddress) && RAMRdEnInt;
  assign LastRdAccessSkip = (RAMAddr==SkipAddress) && RAMRdEnInt;

  /* Registers */
  always @(posedge Clk, negedge nRst)
    if(!nRst) begin
      RAMWrEn            <= 1'b0;
      RAMWrData          <= {RAM_DWIDTH{1'b0}};
      RAMAddr            <= {RAM_AWIDTH{1'b0}};
      RAMRdEnInt         <= 1'b0;
      busy               <= 1'b0;
      LastRdData         <= 1'b0;
      LastRdDataSkip     <= 1'b0;
      StartD             <= 1'b0;
    end else if(!Enable) begin // write operation
      RAMWrEn            <= DataWrEn;
      RAMWrData          <= DataWrData;
      RAMAddr            <= DataWrAddr;
      RAMRdEnInt         <= 1'b0;
      busy               <= 1'b0;
      LastRdData         <= 1'b0;
      LastRdDataSkip     <= 1'b0;
      StartD             <= 1'b0;
    end else begin
      StartD             <= Start;
      LastRdData         <= LastRdAccess;
      LastRdDataSkip     <= LastRdAccessSkip;

      if (Start) begin
        RAMAddr          <= {RAM_DWIDTH{1'b0}};
        RAMRdEnInt       <= 1'b1;
        busy             <= 1'b1;
        LastRdData       <= 1'b0;
        LastRdDataSkip   <= 1'b0;
      end else if (busy) begin
        if(s3Ready || !s3Valid) begin
          if (RAMAddr==StopAddress) begin
            RAMRdEnInt   <= 1'b0;
            RAMAddr      <= {RAM_AWIDTH{1'b0}};
            busy         <= 1'b0;
          end else begin
            RAMRdEnInt   <= 1'b1;
            if (RAMAddr == SkipAddress) 
              RAMAddr  <= SKIP_ADDR;
            else 
              RAMAddr  <= RAMAddr + SIG_ONE_DSP_AWIDTH_PARAM;
          end
        end
      end
      
    end
    
    // Gate RAMRdEn when read data not needed, ungate it one cycle before read data needed
    // so that RAMRdEn does not stay high while output is stalled
    assign RAMRdEn = RAMRdEnInt && (s1Valid || StartD || s3Ready);

endmodule
                 
`default_nettype wire
