//
// Mac interface, incorporating the MAC memory interface, the angles buffer and SNR buffer.
//

`default_nettype none

module macif # (
  parameter SNR_BASE_ADDR = 10'd585,  // The address of the first Snr entry in the RAM.
  parameter AWIDTH        = 10        // The address width of the RAM.
  ) (
  ///////////////////////////////////////////////
  //$port_g Modem Clock and reset
  ///////////////////////////////////////////////
  input  wire                  BFRModemClk,        // BFR Modem Clock
  input  wire                  BFRModemGClk,       // BFR Modem Gated Clock
  input  wire                  nBFRModemRst,       // Active Low Reset(Modem Domain)

  ///////////////////////////////////////////////
  //$port_g Control signals.
  ///////////////////////////////////////////////
  input  wire                  tctlStart,          // Start processing, send the angles.
  input  wire                  tctlStop,           // Stop/abort memory write.
  input  wire                  tctlAdvance,        // Asserted when the pipeline is to advance.
  input  wire                  SVDDone,            // Asserted when the SVD computation phase has completed.  
  output wire                  ReportWritten,      // Indicates that all the report has been stored in the Mdm/Mac memory
  output reg                   ReportRead,         // Asserted for one tick when the Mac has sent its last data.
 
  ///////////////////////////////////////////////
  //$port_g configuration
  ///////////////////////////////////////////////
  input  wire  [1:0]           cfgChBw,           
  input  wire  [1:0]           cfgNg,
  input  wire  [1:0]           cfgNc,
  input  wire                  cfgFeedbackType,
  input  wire                  cfgHe,

  ///////////////////////////////////////////////
  //$port_g Input data and valid signals.
  ///////////////////////////////////////////////
  input wire [31:0]            packerWord,      // Angles, packed in words.
  input wire                   packerValid,     // Asserted when packerWord is valid.
  input wire                   packerLast,
  input wire  [5:0]            packerLastNum,
  input wire  [7:0]            snrNsnr0,        // per sub-carrier SNR for nc=1
  input wire  [7:0]            snrNsnr1,        // per sub-carrier SNR for nc=2
  input wire                   snrNsnrActive,
  input wire                   snrNsnrLast,
  input wire  [8:0]            accuSubAvg0,     // The value that needs to be subtracted from each snr value before being sent to the MAC.
  input wire  [8:0]            accuSubAvg1,     // accuSubAvg = accuAvgSnr before scaling
  input wire  [7:0]            accuAvgSnr0,
  input wire  [7:0]            accuAvgSnr1,

   ///////////////////////////////////////////////
  //$port_g Interface with the mac.
  ///////////////////////////////////////////////
  input wire                   MacDataOutReady,
  output reg [31:0]            MacDataOut,
  output reg  [3:0]            MacDataOutValid,
  
  ///////////////////////////////////////////////
  //$port_g Read Ram interface.
  ///////////////////////////////////////////////
  output wire [AWIDTH-1:0]     RamAddress,
  output wire                  RamWe,
  output wire [31:0]           RamWData,
  output wire                  RamRe,
  input  wire [31:0]           RamRData
  );
 
  reg  [AWIDTH-1:0]   DpRamWAddress;
  reg  [31:0]         DpRamWData;
  reg                 DpRamWe;

  reg  [2:0]          MacIfWriteState; // State to write Report RAM
  reg  [2:0]          MacIfReadState;  // State to read the Report RAM

  reg  [AWIDTH-1:0]   SnrAddress;
  reg  [AWIDTH-1:0]   AngleAddress;
  reg  [AWIDTH-1:0]   LastAngleAddr;
  reg  [AWIDTH-1:0]   LastDSNRAddr;

  // Compute constants from configuration values
  wire                LastAngleData;

  wire [31:0]  PackedSnr;       // delta SNR values filtered to keep relevant sc per Nc and packed to 32 bits
  wire         PackedSnrValid;  // Asserted when PackedSnr contains valid data to write to the RAM
  wire         PackedSnrLast;   // Asserted with the last PackedSnr data to write to the RAM
  wire  [3:0]  PackedSnrLastEn;
 
  // Detect SVDDone rising edge
  reg                 dSVDDone;
  reg                 SvdFinished; 

  // Data2Ram If
  reg                 StartRd;
  reg                 RdEnable;
  wire [AWIDTH-1:0]   StopAddress;
  wire [31:0]         SVDDataOut;
  wire                SVDDataOutLast;
  wire                SVDDataOutLastSkip;
  wire                SVDDataOutValid;
  wire                SVDDataOutReady;
  wire                SVDDataOutNotReady;
  
  // Compute DSNR values
  wire snrNValid;
  wire snrNLast;
  wire [15:0] MacDSnrOutNc1, MacDSnrOutNc2, MacDSnrOut;

  // 
  // Parameters for configuration values
  //
  localparam NC1 = 2'd0, NC2 = 2'd1; // Nc index = Nc-1
  localparam FBACK_SU = 1'd0, FBACK_MU = 1'd1;

  //
  // Declare the states used in the Write state machine.
  //
  localparam MIWS_IDLE        = 3'd0; // Idle
  localparam MIWS_ANGLES      = 3'd1; // Write angle value
  localparam MIWS_DSNR        = 3'd2; // Write per sub-carrier SNR values for MU report
  localparam MIWS_LAST_ANGLE  = 3'd3; // Write remaining word of packed angles
  localparam MIWS_DONE        = 3'd4; // Last angle or SNR written
 
  //
  // Declare the states used in the Read State Machine.
  //
  localparam MIRS_IDLE        = 3'd0;
  localparam MIRS_AVGSNR      = 3'd1;  // Send the Average SNR to the MAC.
  localparam MIRS_ANGLES      = 3'd2;  // Read the Angles values from the ram and send them out.
  localparam MIRS_LAST_ANGLE  = 3'd3;  // Read the last Angles value from the ram and send them out.
  localparam MIRS_DSNR        = 3'd4;  // Read the delta snr values from the RAM and send them out.
  localparam MIRS_LAST_DSNR   = 3'd5;  // Send the last delta snr values to the Mac.
  localparam MIRS_DONE        = 3'd6;  // Write done.

  //
  // A function to implement min(max(floor(...))) and some other logic and return a 4 bit result.
  // This slightly odd implementation was necessary to match Matlab.
  //
  function signed [3:0] MinMaxFloor4;
    input  [7:0] Snr;
    input  [8:0] Avg;
    reg   signed [9:0] A;
    reg   signed [7:0] B;
    begin
      A = {1'b0,Snr,1'b0}-{1'b0,Avg};
      //
      // Divide by four.
      //
      B = A[9:2];
      //
      // Work out the min/max part.
      // The range of a 4 bit signed number is -8 to 7.
      //
      if (!B[7] && (B > 8'd7))
        begin
          MinMaxFloor4 = 4'd7;
        end
      else if (B[7] && (B < -8'd8))
        begin
          MinMaxFloor4 = 4'b1000; // -8
        end
      else
        begin
          MinMaxFloor4 = B[3:0];
        end
        
    end
  endfunction 
   
 
 //
 // State machine to control the sequence of data being written to the memory.
 //
 always @(posedge BFRModemGClk or negedge nBFRModemRst)
   begin
   if (!nBFRModemRst)
       begin
         MacIfWriteState <= MIWS_IDLE;
       end
     else
       begin
         //
         // Overriding condition first for safety.
         // 
         if (tctlStart || tctlStop)
           begin
             MacIfWriteState <= MIWS_IDLE;
           end
         else
           begin
             //
             // Do the state transitions here.
             //
             case (MacIfWriteState)
                 //
                 // Sit in the idle state until there is something to do.
                 //
                 MIWS_IDLE:
                   begin
                     if (PackedSnrValid)       // There are Snr values to be written.
                       begin
                         MacIfWriteState <= MIWS_DSNR;
                       end
                     else if (packerValid)     // Angles have arrived, deal with them first as they only stay for one tick each.
                       begin
                         if (packerLast==1'b1) // Finished,
                           MacIfWriteState <= MIWS_LAST_ANGLE; 
                         else 
                           MacIfWriteState <= MIWS_ANGLES;
                       end
                   end
                 //
                 // When in the Angles state stay there until there are no more angles to be processed.
                 //
                 MIWS_ANGLES:
                   begin
                     if (packerValid)
                       MacIfWriteState <= MIWS_ANGLES;
                     else
                       MacIfWriteState <= MIWS_IDLE;
                   end
                 //
                 // When in the Snr state, always go back to idle.
                 //
                 MIWS_DSNR :
                   begin
                     MacIfWriteState <= MIWS_IDLE;
                   end
                 //
                 // Write last angle in memory and go to Done
                 //
                 MIWS_LAST_ANGLE :
                   begin
                     MacIfWriteState <= MIWS_DONE;
                   end  
                 //
                 // When in Done state move to Idle always.
                 //
                 default: // MIWS_DONE
                   begin
                     MacIfWriteState <= MIWS_IDLE;
                   end
             endcase
           end               
       end    
   end              
 
   //
   // Control the memory write address.
   //
   always @(posedge BFRModemGClk or negedge nBFRModemRst)
     begin
       if (!nBFRModemRst)
         begin
           SnrAddress    <= {AWIDTH{1'b0}};
           AngleAddress  <= {AWIDTH{1'b0}};
           LastAngleAddr <= {AWIDTH{1'b0}};
           LastDSNRAddr  <= {AWIDTH{1'b0}};
         end
       else
         begin
           //
           // SnrAddress: At the start set the SnrAddress to SNR_BASE_ADDR, as that's where the Snr values start.
           // Then increase at each SNR written.
           //
           if (tctlStart)
             SnrAddress <= SNR_BASE_ADDR;
           else if (MacIfWriteState == MIWS_DSNR)  
             SnrAddress <= SnrAddress + {{(AWIDTH-1){1'b0}},1'b1};
           //
           // AngleAddress: At the start set the AngleAddress to 0, as that's where the Angle values start.
           // Then increase at each Angle written.
           //
           if (tctlStart)
             AngleAddress <= {AWIDTH{1'b0}};
           else if (packerValid) 
             AngleAddress <= AngleAddress + {{(AWIDTH-1){1'b0}},1'b1};
           //
           // LastAngleAddr: Store address of the last angle written to skip to SNR address when reaching it during read.
           // This is useful when using HE Start/End indexes because Ns value is unknown at start.
           //
           if (MacIfWriteState==MIWS_LAST_ANGLE)
             LastAngleAddr <= RamAddress;
           //
           // LastDSNRAddr: Store address of the last DSNR written to know when to stop read.
           // This is useful when using HE Start/End indexes because Ns value is unknown at start.
           //
           if ((MacIfWriteState==MIWS_DSNR) && (PackedSnrLast==1'b1)) // PackedSnrLast stays high
             LastDSNRAddr <= DpRamWAddress; // not same delay on Angle and DSR write, use DpRamWAddress
         end
     end
    
    always @(*)
      begin
        //
        // Mux the address onto the DpRamWAddress bus.
        //
        case (MacIfWriteState)
          MIWS_ANGLES   : DpRamWAddress = AngleAddress;        
          MIWS_DSNR     : DpRamWAddress = SnrAddress; 
          default       : DpRamWAddress = AngleAddress;
        endcase

        //
        // Mux the data onto the DpRamWData bus.
        //
        case (MacIfWriteState)
          MIWS_ANGLES   : DpRamWData = packerWord;
          MIWS_DSNR     : DpRamWData = PackedSnr;
          default       : DpRamWData = packerWord;
        endcase
        //
        // Control the DpRamWe write enable signal.
        //
        case (MacIfWriteState)
          MIWS_IDLE       : DpRamWe = packerValid;
          MIWS_ANGLES     : DpRamWe = packerValid;
          MIWS_LAST_ANGLE : DpRamWe = 1'b0;
          MIWS_DSNR       : DpRamWe = 1'b1;
          MIWS_DONE       : DpRamWe = 1'b0;
          default         : DpRamWe = 1'b0;
        endcase
      end

   assign ReportWritten = (MacIfWriteState==MIWS_DONE);
   

   //
   // Instantiate the snr filter block, this block decides whether the current snr value is to be put into the 
   // ram or not, and packs the SNR for Nc=1 or 2 on 32 bits.
   //
   assign snrNValid = tctlAdvance && snrNsnrActive && (cfgFeedbackType == FBACK_MU);
   assign snrNLast = snrNValid && snrNsnrLast;
   macif_snr_filter imacif_snr_filter (
     .nRst(nBFRModemRst),
     .Clk(BFRModemClk),
     .tctlStart(tctlStart),
     .tctlStop(tctlStop),
     .snr0(snrNsnr0),
     .snr1(snrNsnr1),     
     .snrValid(snrNValid),
     .snrLast(snrNLast),
     .cfgNc(cfgNc),
     .cfgNg(cfgNg),
     .cfgChBw(cfgChBw),
     .cfgHe(cfgHe),
     .PackedSnrValid(PackedSnrValid),
     .PackedSnrLast(PackedSnrLast),
     .PackedSnrLastEn(PackedSnrLastEn),
     .PackedSnr(PackedSnr)
     );



 // -------------------------------------------------------------------------------------------------------------------------------
 // Ungated Clock Domain 
 // -------------------------------------------------------------------------------------------------------------------------------
  // LastAngleData flag: when reaching the address of the last Angle value in memory, + pipe length.
  assign LastAngleData = (cfgFeedbackType == FBACK_SU) ? SVDDataOutLast : SVDDataOutLastSkip;
 
  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
      if (nBFRModemRst == 1'b0)
        begin
          dSVDDone  <= 1'b0;
        end  
      else
        begin
          dSVDDone  <= SVDDone;
        end
    end   

  always @(*)
  begin
    if(SVDDone && !dSVDDone)
      begin
        SvdFinished = 1'b1;
      end
    else 
      begin
        SvdFinished = 1'b0;
      end 
  end    

  //
  // This module reads the data from the RAM, implementing a buffer to save/restore 
  // the read value when the macif is not ready to accept it. It also forwards and registers
  // the write accesses.
  
  // In SU, read up to last angle; in MU, read up to last SNR
  assign StopAddress         = (cfgFeedbackType == FBACK_SU) ? LastAngleAddr : LastDSNRAddr;

  RamIf # (
    .RAM_AWIDTH(AWIDTH),
    .RAM_DWIDTH(32),
    .SKIP_ADDR(SNR_BASE_ADDR)
    ) iRamIf (
     // Clock and reset
    .nRst             (nBFRModemRst),
    .Clk              (BFRModemClk),
    // controls
    .Enable           (RdEnable),
    .Start            (StartRd),
    .StopAddress      (StopAddress),
    .SkipAddress      (LastAngleAddr), // In MU mode, skip from LastAngleAddr to SNR_BASE_ADDR.
    // RAM interface
    .RAMAddr          (RamAddress),
    .RAMWrEn          (RamWe),
    .RAMWrData        (RamWData),
    .RAMRdEn          (RamRe),
    .RAMRdData        (RamRData),
    // Data interface
    .DataWrEn         (DpRamWe),
    .DataWrAddr       (DpRamWAddress),
    .DataWrData       (DpRamWData),
    
    .DataOut          (SVDDataOut),
    .DataOutLast      (SVDDataOutLast),
    .DataOutLastSkip  (SVDDataOutLastSkip),
    .DataOutValid     (SVDDataOutValid),
    .DataOutReady     (SVDDataOutReady)
  );
  
  // SVDDataOutNotReady si set high to freeze the RamIf block.
  assign SVDDataOutNotReady    = (MacIfReadState==MIRS_AVGSNR) ||  // RamIf enabled but output data frozen while SNR from register is sent out
                                 !MacDataOutReady;                 // Stop RamIf when MAC if is stopped

  assign SVDDataOutReady       =  !SVDDataOutNotReady;

  //
  // A state machine to read the report from the RAM and send it out.
  // The state is aligned with SVDDataOut.
  //
  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
      if (!nBFRModemRst)
        begin
          MacIfReadState      <= MIRS_IDLE;
          StartRd             <= 1'd0;
          RdEnable            <= 1'd0;
        end
      else
        begin
          StartRd             <= 1'd0; // RTZ
          if (tctlStop)
            begin
              MacIfReadState      <= MIRS_IDLE;
              StartRd             <= 1'd0;
              RdEnable            <= 1'd0;
            end
          else
            begin
              case (MacIfReadState)

                // Idle. Wait for end of SVD operation to start transfer to the MAC
                MIRS_IDLE:
                  begin
                    if (SvdFinished)
                      begin
                        // Send the Average SNR values from register
                        MacIfReadState      <= MIRS_AVGSNR;
                        // Start RamIf block at address 0 (angles)
                        // Pipe will fill while avg SNR are sent out
                        StartRd             <= 1'd1;
                        RdEnable            <= 1'd1;
                      end
                  end

                // Send Average SNR value from register (up to 16 bits, one word out for all Nc values)
                MIRS_AVGSNR:
                  begin
                    if (MacDataOutReady)
                      MacIfReadState      <= MIRS_ANGLES;
                  end

                // Send Angle value from RAM, up to last angle
                MIRS_ANGLES:
                  begin
                    if (SVDDataOutValid && MacDataOutReady && LastAngleData)
                      MacIfReadState      <= MIRS_LAST_ANGLE;
                  end

                // Wait for MAC ready before going to delta SNR or Done 
                MIRS_LAST_ANGLE:
                  begin

                    // SU mode: transfer is done. Disable RAMIf read.
                    if (cfgFeedbackType == FBACK_SU) begin
                      if (MacDataOutReady) begin
                        MacIfReadState      <= MIRS_DONE;
                        RdEnable            <= 1'd0;
                      end

                    // MU mode: transfer delta SNR values
                    end else begin
                      if (SVDDataOutValid && MacDataOutReady) begin
                        // In some Start/End RU HE configurations, Ns=4 and all DSNR are sent in one cycle
                        if (SVDDataOutLast) begin
                          MacIfReadState      <= MIRS_LAST_DSNR;
                          RdEnable            <= 1'd0; // stop read, just flush pipe
                        end else begin
                          MacIfReadState      <= MIRS_DSNR;
                        end
                      end
                    end

                  end

                // Send delta SNR values from RAM
                MIRS_DSNR:
                  begin
                    if (SVDDataOutLast && SVDDataOutValid && MacDataOutReady) begin
                      MacIfReadState      <= MIRS_LAST_DSNR;
                      RdEnable            <= 1'd0; // stop read, just flush pipe
                    end
                  end

                // Wait for MAC ready before going to Done 
                MIRS_LAST_DSNR:
                  begin
                    if (MacDataOutReady) begin // Last SNR sent to MAC.
                      MacIfReadState      <= MIRS_DONE;
                      // Disable RamIf read
                      RdEnable            <= 1'd0;
                    end
                  end

                // Last state of the state machine, used to set ReportRead flag
                MIRS_DONE:
                  begin
                    MacIfReadState      <= MIRS_IDLE;
                  end
                  
                // The default case, should never happen.
                default:
                  begin
                    MacIfReadState      <= MIRS_IDLE;
                    StartRd             <= 1'd0;
                    RdEnable            <= 1'd0;
                end
              endcase
            end
        end
    end
   

  //
  // Assert ReportRead
  //
  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
      if (nBFRModemRst == 1'b0)  begin
        ReportRead   <=  1'd0;
      end else begin
        ReportRead   <=  1'd0; // RTZ
        if (MacIfReadState == MIRS_DONE)
          ReportRead <=  1'd1;
      end
    end

  // Compute delta SNR value based on cfgNc and RAM read data
  assign MacDSnrOutNc1 = {MinMaxFloor4(SVDDataOut[31:24],accuSubAvg0), MinMaxFloor4(SVDDataOut[23:16],accuSubAvg0),
                          MinMaxFloor4(SVDDataOut[15:8], accuSubAvg0), MinMaxFloor4(SVDDataOut[7:0],  accuSubAvg0)};
  assign MacDSnrOutNc2 = {MinMaxFloor4(SVDDataOut[31:24],accuSubAvg1), MinMaxFloor4(SVDDataOut[23:16],accuSubAvg0),
                          MinMaxFloor4(SVDDataOut[15:8], accuSubAvg1), MinMaxFloor4(SVDDataOut[7:0],  accuSubAvg0)};
  assign MacDSnrOut    = (cfgNc==NC1) ? MacDSnrOutNc1 : MacDSnrOutNc2;              


reg [7:0] MacDataShift;
reg [2:0] MacDataShiftNum;
    
  //
  // load the MacDataOut and control the MacDataOutValid signal.
  //
  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
      if (nBFRModemRst == 1'b0)
        begin
          MacDataOut      <=  32'd0;
          MacDataOutValid <=  4'b0;
          MacDataShift    <=  8'd0;
          MacDataShiftNum <=  3'd0;
        end  
      else
        begin
          // From any state, in case of abort, immediately invalidate data out
          if (tctlStop==1'b1) begin
            MacDataOutValid <=  4'b0;
          end else begin
          
            case (MacIfReadState)

              // Idle - reset MAC outputs
              MIRS_IDLE:
              begin
                MacDataOut      <=  32'd0;
                MacDataOutValid <=  4'b0;
              end

              // Average SNR
              MIRS_AVGSNR:
              begin
                if (MacDataOutReady) begin // wait for ready so that no need to handle Valid going back to 0
                  if (cfgNc==NC1)
                    MacDataOutValid <=  4'b0001; // one byte of SNR is valid
                  else
                    MacDataOutValid <=  4'b0011; // two bytes of SNR are valid
                  MacDataOut      <=  {16'd0,accuAvgSnr1,accuAvgSnr0};
                end
              end

              // Angles from RAM
              MIRS_ANGLES:
              begin
                if (MacDataOutReady) begin
                  if (SVDDataOutValid) begin
                    if (LastAngleData==1'b1) begin
                      MacDataShiftNum <= packerLastNum[2:0];
                      case (packerLastNum)
                        6'd32 : MacDataOutValid <= 4'b1111;
                        6'd24 : MacDataOutValid <= 4'b0111;
                        6'd16 : MacDataOutValid <= 4'b0011;
                        6'd8  : MacDataOutValid <= 4'b0001;
                        // When dataout is not an entire number of bytes:
                        // - if no SNR report follows, send all data out
                        // - if an SNR report follows, send complete bytes out and keep remaining bits to be packed with 
                        // SNR report
                        default : begin
                          if (packerLastNum<6'd8) begin
                            MacDataOutValid <= {3'b000,~cfgFeedbackType};
                            MacDataShift    <= SVDDataOut[7:0];
                          end else if (packerLastNum<6'd16) begin
                            MacDataOutValid <= {2'b00,~cfgFeedbackType,1'b1};
                            MacDataShift    <= SVDDataOut[15:8];
                          end else if (packerLastNum<6'd24) begin
                            MacDataOutValid <= {1'b0,~cfgFeedbackType,2'b11};
                            MacDataShift    <= SVDDataOut[23:16];
                          end else begin
                            MacDataOutValid <= {~cfgFeedbackType,3'b111};
                            MacDataShift    <= SVDDataOut[31:24];
                          end
                        end
                      endcase
                    end else begin
                      MacDataOutValid <= 4'b1111;
                    end
                    MacDataOut      <= SVDDataOut;
                  end else begin
                    MacDataOutValid <=  4'd0; //Reset once previous value (average SNR) sampled, to wait for RamIf
                  end
                end
              end

              // Last Angle or Delta SNR value
              MIRS_LAST_ANGLE:
              begin
                if (MacDataOutReady) begin
                  // SU: reset signals as state goes to Idle
                  if (cfgFeedbackType==FBACK_SU) begin
                    MacDataOut      <= 32'd0;
                    MacDataOutValid <=  4'd0;
                    
                  // MU: Prepare first delta SNR - can also be last with HE configurations (Ns=3)
                  end else begin
                    if (SVDDataOutLast) begin// Force unused SNR bits to zero
                      MacDataOutValid <= {2'b0,|PackedSnrLastEn[3:2],|PackedSnrLastEn[1:0]};
                      case (PackedSnrLastEn)
                        4'b0001 : MacDataOut <= {28'd0,MacDSnrOut[3:0]};
                        4'b0011 : MacDataOut <= {24'd0,MacDSnrOut[7:0]};
                        4'b0111 : MacDataOut <= {20'd0,MacDSnrOut[11:0]};
                        default : MacDataOut <= {16'd0,MacDSnrOut};
                      endcase
                    end else begin
                      MacDataOutValid <= 4'b0011; // 4 SNR values (2 bytes) packed per memory word
                      // if any, pack last bits of angles
                      case (MacDataShiftNum)
                        3'd1   : begin
                          MacDataOut      <= {16'd0,MacDSnrOut[14:0],MacDataShift[0]};
                          MacDataShift    <= {7'd0,MacDSnrOut[15]};
                        end
                        3'd2   : begin
                          MacDataOut      <= {16'd0,MacDSnrOut[13:0],MacDataShift[1:0]};
                          MacDataShift    <= {6'd0,MacDSnrOut[15:14]};
                        end
                        3'd3   : begin
                          MacDataOut      <= {16'd0,MacDSnrOut[12:0],MacDataShift[2:0]};
                          MacDataShift    <= {5'd0,MacDSnrOut[15:13]};
                        end
                        3'd4   : begin
                          MacDataOut      <= {16'd0,MacDSnrOut[11:0],MacDataShift[3:0]};
                          MacDataShift    <= {4'd0,MacDSnrOut[15:12]};
                        end
                        3'd5   : begin
                          MacDataOut      <= {16'd0,MacDSnrOut[10:0],MacDataShift[4:0]};
                          MacDataShift    <= {3'd0,MacDSnrOut[15:11]};
                        end
                        3'd6   : begin
                          MacDataOut      <= {16'd0,MacDSnrOut[9:0],MacDataShift[5:0]};
                          MacDataShift    <= {2'd0,MacDSnrOut[15:10]};
                        end
                        3'd7   : begin
                          MacDataOut      <= {16'd0,MacDSnrOut[8:0],MacDataShift[6:0]};
                          MacDataShift    <= {1'd0,MacDSnrOut[15:9]};
                        end
                        default: begin
                          MacDataOut      <= {16'd0,MacDSnrOut};
                        end
                      endcase
                    end
                  end
                end
              end
              
              // Last Delta SNR value
              MIRS_LAST_DSNR:
              begin
                // Reset signals as state goes to Idle
                if (MacDataOutReady) begin
                  MacDataOut      <= 32'd0;
                  MacDataOutValid <=  4'd0;
                end
              end
              
              // Send delta SNR to MAC
              MIRS_DSNR:
              begin
                if ((SVDDataOutValid==1'b1) && ( (MacDataOutReady==1'b1) || (MacDataOutValid==4'd0) )) begin
                  if (SVDDataOutLast) begin// Force unused SNR bits to zero
                    case ({MacDataShiftNum,PackedSnrLastEn})
                      {3'd0,4'b0001}   : begin
                        MacDataOut      <= {28'd0,MacDSnrOut[3:0]};
                        MacDataOutValid <= 4'b0001;
                      end
                      {3'd0,4'b0011}   : begin
                        MacDataOut      <= {24'd0,MacDSnrOut[7:0]};
                        MacDataOutValid <= 4'b0001;
                      end
                      {3'd0,4'b0111}   : begin
                        MacDataOut      <= {20'd0,MacDSnrOut[11:0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd0,4'b1111}   : begin
                        MacDataOut      <= {16'd0,MacDSnrOut};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd1,4'b0001}   : begin
                        MacDataOut      <= {27'd0,MacDSnrOut[3:0],MacDataShift[0]};
                        MacDataOutValid <= 4'b0001;
                      end
                      {3'd1,4'b0011}   : begin
                        MacDataOut      <= {23'd0,MacDSnrOut[7:0],MacDataShift[0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd1,4'b0111}   : begin
                        MacDataOut      <= {19'd0,MacDSnrOut[11:0],MacDataShift[0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd1,4'b1111}   : begin
                        MacDataOut      <= {15'd0,MacDSnrOut,MacDataShift[0]};
                        MacDataOutValid <= 4'b0111;
                      end
                      {3'd2,4'b0001}   : begin
                        MacDataOut      <= {26'd0,MacDSnrOut[3:0],MacDataShift[1:0]};
                        MacDataOutValid <= 4'b0001;
                      end
                      {3'd2,4'b0011}   : begin
                        MacDataOut      <= {22'd0,MacDSnrOut[7:0],MacDataShift[1:0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd2,4'b0111}   : begin
                        MacDataOut      <= {18'd0,MacDSnrOut[11:0],MacDataShift[1:0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd2,4'b1111}   : begin
                        MacDataOut      <= {14'd0,MacDSnrOut,MacDataShift[1:0]};
                        MacDataOutValid <= 4'b0111;
                      end
                      {3'd3,4'b0001}   : begin
                        MacDataOut      <= {25'd0,MacDSnrOut[3:0],MacDataShift[2:0]};
                        MacDataOutValid <= 4'b0001;
                      end
                      {3'd3,4'b0011}   : begin
                        MacDataOut      <= {21'd0,MacDSnrOut[7:0],MacDataShift[2:0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd3,4'b0111}   : begin
                        MacDataOut      <= {17'd0,MacDSnrOut[11:0],MacDataShift[2:0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd3,4'b1111}   : begin
                        MacDataOut      <= {13'd0,MacDSnrOut,MacDataShift[2:0]};
                        MacDataOutValid <= 4'b0111;
                      end
                      {3'd4,4'b0001}   : begin
                        MacDataOut      <= {24'd0,MacDSnrOut[3:0],MacDataShift[3:0]};
                        MacDataOutValid <= 4'b0001;
                      end
                      {3'd4,4'b0011}   : begin
                        MacDataOut      <= {20'd0,MacDSnrOut[7:0],MacDataShift[3:0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd4,4'b0111}   : begin
                        MacDataOut      <= {16'd0,MacDSnrOut[11:0],MacDataShift[3:0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd4,4'b1111}   : begin
                        MacDataOut      <= {12'd0,MacDSnrOut,MacDataShift[3:0]};
                        MacDataOutValid <= 4'b0111;
                      end
                      {3'd5,4'b0001}   : begin
                        MacDataOut      <= {23'd0,MacDSnrOut[3:0],MacDataShift[4:0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd5,4'b0011}   : begin
                        MacDataOut      <= {19'd0,MacDSnrOut[7:0],MacDataShift[4:0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd5,4'b0111}   : begin
                        MacDataOut      <= {15'd0,MacDSnrOut[11:0],MacDataShift[4:0]};
                        MacDataOutValid <= 4'b0111;
                      end
                      {3'd5,4'b1111}   : begin
                        MacDataOut      <= {11'd0,MacDSnrOut,MacDataShift[4:0]};
                        MacDataOutValid <= 4'b0111;
                      end
                      {3'd6,4'b0001}   : begin
                        MacDataOut      <= {22'd0,MacDSnrOut[3:0],MacDataShift[5:0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd6,4'b0011}   : begin
                        MacDataOut      <= {18'd0,MacDSnrOut[7:0],MacDataShift[5:0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd6,4'b0111}   : begin
                        MacDataOut      <= {14'd0,MacDSnrOut[11:0],MacDataShift[5:0]};
                        MacDataOutValid <= 4'b0111;
                      end
                      {3'd6,4'b1111}   : begin
                        MacDataOut      <= {10'd0,MacDSnrOut,MacDataShift[5:0]};
                        MacDataOutValid <= 4'b0111;
                      end
                      {3'd7,4'b0001}   : begin
                        MacDataOut      <= {21'd0,MacDSnrOut[3:0],MacDataShift[6:0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd7,4'b0011}   : begin
                        MacDataOut      <= {17'd0,MacDSnrOut[7:0],MacDataShift[6:0]};
                        MacDataOutValid <= 4'b0011;
                      end
                      {3'd7,4'b0111}   : begin
                        MacDataOut      <= {13'd0,MacDSnrOut[11:0],MacDataShift[6:0]};
                        MacDataOutValid <= 4'b0111;
                      end
                      {3'd7,4'b1111}   : begin
                        MacDataOut      <= {9'd0,MacDSnrOut,MacDataShift[6:0]};
                        MacDataOutValid <= 4'b0111;
                      end
                      default: begin
                        MacDataOut      <= {16'd0,MacDSnrOut};
                        MacDataOutValid <= {2'b0,|PackedSnrLastEn[3:2],|PackedSnrLastEn[1:0]};
                      end
                    endcase

                  end else begin
                    MacDataOutValid <= 4'b0011; // 4 SNR values (2 bytes) packed per memory word
                    // if any, shift including  last bits of angles
                    case (MacDataShiftNum)
                      3'd1   : begin
                        MacDataOut      <= {16'd0,MacDSnrOut[14:0],MacDataShift[0]};
                        MacDataShift    <= {7'd0,MacDSnrOut[15]};
                      end
                      3'd2   : begin
                        MacDataOut      <= {16'd0,MacDSnrOut[13:0],MacDataShift[1:0]};
                        MacDataShift    <= {6'd0,MacDSnrOut[15:14]};
                      end
                      3'd3   : begin
                        MacDataOut      <= {16'd0,MacDSnrOut[12:0],MacDataShift[2:0]};
                        MacDataShift    <= {5'd0,MacDSnrOut[15:13]};
                      end
                      3'd4   : begin
                        MacDataOut      <= {16'd0,MacDSnrOut[11:0],MacDataShift[3:0]};
                        MacDataShift    <= {4'd0,MacDSnrOut[15:12]};
                      end
                      3'd5   : begin
                        MacDataOut      <= {16'd0,MacDSnrOut[10:0],MacDataShift[4:0]};
                        MacDataShift    <= {3'd0,MacDSnrOut[15:11]};
                      end
                      3'd6   : begin
                        MacDataOut      <= {16'd0,MacDSnrOut[9:0],MacDataShift[5:0]};
                        MacDataShift    <= {2'd0,MacDSnrOut[15:10]};
                      end
                      3'd7   : begin
                        MacDataOut      <= {16'd0,MacDSnrOut[8:0],MacDataShift[6:0]};
                        MacDataShift    <= {1'd0,MacDSnrOut[15:9]};
                      end
                      default: begin
                        MacDataOut      <= {16'd0,MacDSnrOut};
                      end
                    endcase
                  end
                end
              end

              // Done: reset signals
              MIRS_DONE:
              begin
                MacDataOut      <= 32'd0;
                MacDataOutValid <= 4'd0;
              end

              // keep values
              default:
              begin
                MacDataOut         <=  MacDataOut;
                MacDataOutValid    <=  MacDataOutValid;
              end

            endcase

          end // tctlStop
        end
      end   


   //
   // To help debug
   // 
   `ifdef RW_SIMU_ON
   reg [(16*8)-1:0] MacIfWriteStateStr;
   always @(*)
   begin
     case (MacIfWriteState)
       MIWS_IDLE       : MacIfWriteStateStr = "WrIdle";
       MIWS_ANGLES     : MacIfWriteStateStr = "WrAngles";
       MIWS_LAST_ANGLE : MacIfWriteStateStr = "WrLastAngle";
       MIWS_DSNR       : MacIfWriteStateStr = "WrDSNR";
       MIWS_DONE       : MacIfWriteStateStr = "WrDone";
       default         : MacIfWriteStateStr = "UNKNOWN";
     endcase
   end
   
   
   reg [3:0] DSNR0Val, DSNR1Val, DSNR2Val, DSNR3Val;
   always @(*)
   begin
     if (cfgNc==NC1) begin
       DSNR3Val = MinMaxFloor4(SVDDataOut[31:24], accuSubAvg0);
       DSNR2Val = MinMaxFloor4(SVDDataOut[23:16], accuSubAvg0);
       DSNR1Val = MinMaxFloor4(SVDDataOut[15:8],  accuSubAvg0);
       DSNR0Val = MinMaxFloor4(SVDDataOut[7:0],   accuSubAvg0);
     end else begin
       DSNR3Val = MinMaxFloor4(SVDDataOut[31:24], accuSubAvg1);
       DSNR2Val = MinMaxFloor4(SVDDataOut[23:16], accuSubAvg0);
       DSNR1Val = MinMaxFloor4(SVDDataOut[15:8],  accuSubAvg1);
       DSNR0Val = MinMaxFloor4(SVDDataOut[7:0],   accuSubAvg0);
     end
   end


   reg [(16*8)-1:0] MacIfReadStateStr;
   always @(*)
   begin
     case (MacIfReadState)
       MIRS_IDLE       : MacIfReadStateStr = "RdIdle";
       MIRS_AVGSNR     : MacIfReadStateStr = "RdAvgSnr";
       MIRS_ANGLES     : MacIfReadStateStr = "RdAngles";
       MIRS_LAST_ANGLE : MacIfReadStateStr = "RdLastAngle";
       MIRS_DSNR       : MacIfReadStateStr = "RdDSNR";
       MIRS_LAST_DSNR  : MacIfReadStateStr = "RdLastDSNR";
       MIRS_DONE       : MacIfReadStateStr = "RdDone";
       default         : MacIfReadStateStr = "UNKNOWN";
     endcase
   end
   `endif
endmodule
