//
// Block to decide if a particular snr value is to be put into the RAM or dropped.
// The selected values are packed into 32bit words before being sent out.
//

`default_nettype none

module macif_snr_filter (
  // clock and reset
  input  wire         nRst,
  input  wire         Clk,
  // control
  input  wire         tctlStart,
  input  wire         tctlStop,
  // configuration
  input  wire [1:0]   cfgNc,
  input  wire [1:0]   cfgNg,
  input  wire [1:0]   cfgChBw,
  input  wire         cfgHe,
  // input data
  input  wire [7:0]   snr0,
  input  wire [7:0]   snr1,
  input  wire         snrValid,
  input  wire         snrLast,
  // output data
  output reg          PackedSnrValid,
  output reg          PackedSnrLast,
  output reg   [3:0]  PackedSnrLastEn,
  output reg  [31:0]  PackedSnr
  );


  // parameters
  localparam BW20 = 2'd0, BW40 = 2'd1, BW80 = 2'd2, BW_UNSUPP = 2'd3;
  localparam GR1 = 2'd0, GR2 = 2'd1, GR4 = 2'd2;
  localparam NC1 = 2'd0, NC2 = 2'd1; // Nc index = Nc-1

  // declarations
  reg [7:0]   SnrNo;
  reg snrSelected;
  reg [1:0] PackedSnrCount;
  
  // logic starts here
  
   // Count SNR values coming in
   always @(posedge Clk or negedge nRst)
   begin
     if (!nRst) begin
       SnrNo <= 8'd0;
     end else begin
       if (tctlStart)
         SnrNo <= 8'd1;
       else if (snrValid)
         SnrNo <= SnrNo + 8'd1;
     end
   end
   
  
  //
  // Examine the configuration parameters and the number of the current snr sample. If this particular sample is to
  // be written out to memory then assert snrSelected, if not deassert it.
  //
  // The numbers are cut and pasted from the Matlab code.
  //
  always @(*)
    begin
      if (cfgHe==1'b1) begin
        snrSelected = 1'b1; // all SNR of angle sub-carriers are selected
      end else begin  
      case (cfgChBw)
        BW20:             // Channel bandwidth = 20MHz.
          begin
            case (cfgNg)
              GR1:       // Ng = 0
                begin
                  case (SnrNo)
                    8'd1,8'd3,8'd5,8'd7,8'd8,8'd10,8'd12,8'd14,8'd16,8'd18,8'd20,8'd21,8'd23,8'd25,8'd26,8'd27: snrSelected = 1'b1;
                    8'd28,8'd30,8'd32,8'd33,8'd35,8'd37,8'd39,8'd41,8'd43,8'd45,8'd46,8'd48,8'd50,8'd52:        snrSelected = 1'b1;
                    default:  snrSelected=1'b0;
                  endcase
                end
              GR2:       // Ng = 1
                begin
                  case (SnrNo)
                    8'd1,8'd3,8'd5,8'd7,8'd9,8'd11,8'd13,8'd15,8'd16,8'd18,8'd20,8'd22,8'd24,8'd26,8'd28,8'd30:   snrSelected = 1'b1;
                    default:  snrSelected=1'b0;
                  endcase
                end
              default:       // Ng = 2 and others.
                begin
                  case (SnrNo)
                    8'd1,8'd3,8'd5,8'd7,8'd8,8'd9,8'd10,8'd12,8'd14,8'd16 : snrSelected = 1'b1;
                    default:  snrSelected=1'b0;
                  endcase
                end
            endcase
          end
        BW40:             // Channel bandwidth = 40MHz.
          begin
            case (cfgNg)
              GR1:       // Ng = 0
                begin
                  case (SnrNo)
                    8'd1,8'd3,8'd5,8'd6,8'd8,8'd10,8'd12,8'd14,8'd16,8'd18,8'd20,8'd22,8'd24,8'd26,8'd28,8'd30,8'd32,8'd33,8'd35,8'd37,8'd39,8'd41,8'd43,8'd45,8'd46,8'd48,8'd50,8'd52,8'd54,8'd55,8'd57:   snrSelected = 1'b1;
                    8'd59,8'd61,8'd63,8'd64,8'd66,8'd68,8'd70,8'd72,8'd74,8'd76,8'd77,8'd79,8'd81,8'd83,8'd85,8'd87,8'd89,8'd91,8'd93,8'd95,8'd97,8'd99,8'd101,8'd103,8'd104,8'd106,8'd108:     snrSelected = 1'b1;
                    default:  snrSelected = 1'b0;
                  endcase
                end
              GR2:       // Ng = 1
                begin
                  case (SnrNo)
                    8'd1,8'd3,8'd5,8'd7,8'd9,8'd11,8'd13,8'd15,8'd17,8'd19,8'd21,8'd23,8'd25,8'd27,8'd29,8'd30,8'd32,8'd34,8'd36,8'd38,8'd40,8'd42,8'd44,8'd46:   snrSelected = 1'b1;
                    8'd48,8'd50,8'd52,8'd54,8'd56,8'd58 :     snrSelected = 1'b1;
                    default:  snrSelected = 1'b0;
                  endcase
                end
              default:      // Ng = 2 and others.
                begin
                  case (SnrNo)                
                    8'd1,8'd3,8'd5,8'd7,8'd9,8'd11,8'd13,8'd15,8'd16,8'd18,8'd20,8'd22,8'd24,8'd26,8'd28,8'd30: snrSelected = 1'b1;
                    default:  snrSelected = 1'b0;
                  endcase
                end
            endcase
          end
        BW80:             // Channel bandwidth = 80MHz.
          begin
            case (cfgNg)
              GR1:       // Ng = 0
                begin
                  case (SnrNo)
                    8'd1,8'd3,8'd5,8'd7,8'd9,8'd11,8'd13,8'd15,8'd17,8'd19,8'd20,8'd22,8'd24,8'd26,8'd28,8'd30,8'd32,8'd34,8'd36,8'd38,8'd40,8'd42,8'd44,8'd46,8'd47,8'd49,8'd51,8'd53,8'd55,8'd57:     snrSelected = 1'b1; 
                    8'd59,8'd61,8'd63,8'd65,8'd67,8'd69,8'd71,8'd73,8'd75,8'd77,8'd79,8'd81,8'd82,8'd84,8'd86,8'd88,8'd90,8'd92,8'd94,8'd96,8'd98,8'd100,8'd102,8'd104,8'd106,8'd108,8'd109:   snrSelected = 1'b1;
                    8'd111,8'd113,8'd115,8'd117,8'd118,8'd120,8'd122,8'd124,8'd126,8'd127,8'd129,8'd131,8'd133,8'd135,8'd137,8'd139,8'd141,8'd143,8'd145,8'd147,8'd149,8'd151:  snrSelected = 1'b1;
                    8'd153,8'd154,8'd156,8'd158,8'd160,8'd162,8'd164,8'd166,8'd168,8'd170,8'd172,8'd174,8'd176,8'd178,8'd180,8'd182,8'd184,8'd186,8'd188,8'd189,8'd191,8'd193:  snrSelected = 1'b1;
                    8'd195,8'd197,8'd199,8'd201,8'd203,8'd205,8'd207,8'd209,8'd211,8'd213,8'd215,8'd216,8'd218,8'd220,8'd222,8'd224,8'd226,8'd228,8'd230,8'd232,8'd234:      snrSelected = 1'b1;
                    default:    snrSelected = 1'b0;
                  endcase
                end
              GR2:       // Ng = 1
                begin
                  case (SnrNo)
                    8'd1,8'd3,8'd5,8'd7,8'd9,8'd11,8'd13,8'd15,8'd17,8'd19,8'd21,8'd23,8'd25,8'd27,8'd29,8'd31,8'd33,8'd35,8'd37,8'd39,8'd41,8'd43,8'd45,8'd47,8'd49,8'd51,8'd53:     snrSelected = 1'b1;
                    8'd55,8'd57,8'd59,8'd61,8'd62,8'd64,8'd66,8'd68,8'd70,8'd72,8'd74,8'd76,8'd78,8'd80,8'd82,8'd84,8'd86,8'd88,8'd90,8'd92,8'd94,8'd96,8'd98,8'd100,8'd102:    snrSelected = 1'b1;
                    8'd104,8'd106,8'd108,8'd110,8'd112,8'd114,8'd116,8'd118,8'd120,8'd122:                                         snrSelected = 1'b1;
                    default:    snrSelected = 1'b0;
                  endcase
                end
              default:       // Ng = 2 and others.
                begin
                  case (SnrNo)
                    8'd1,8'd3,8'd5,8'd7,8'd9,8'd11,8'd13,8'd15,8'd17,8'd19,8'd21,8'd23,8'd25,8'd27,8'd29,8'd31,8'd32,8'd34,8'd36,8'd38,8'd40,8'd42,8'd44:     snrSelected = 1'b1;
                    8'd46,8'd48,8'd50,8'd52,8'd54,8'd56,8'd58,8'd60,8'd62:                                          snrSelected = 1'b1;
                    default:    snrSelected = 1'b0;
                  endcase
                end
            endcase
          end
        default:
          begin
            snrSelected = 1'b0;
          end
      endcase  
      end
    end
    
   // Pack SNR values selected with snrSelected into PackedSnr (32 bits)
   always @(posedge Clk or negedge nRst)
   begin

     if (!nRst) begin
       PackedSnrCount      <= 2'd0;
       PackedSnr           <= 32'd0;
       PackedSnrValid      <= 1'b0;
       PackedSnrLast       <= 1'b0;
       PackedSnrLastEn     <= 4'd0;

     end else begin

       PackedSnrValid <= 1'b0;

       // Start / abort
       if ((tctlStart) || (tctlStop)) begin
         PackedSnrCount      <= 2'd0;
         PackedSnr           <= 32'd0;
         PackedSnrValid      <= 1'b0;
         PackedSnrLast       <= 1'b0;
         PackedSnrLastEn     <= 4'd0;

       // Selected valid SNR
       end else if (snrValid && snrSelected) begin
         // NC=1: shift snr0 into PackedSnr, increase PackedSnrCount by 1
         if (cfgNc==NC1) begin
           // on last SNR, align the content of PackedSnr with bit 0.
           if (snrLast) begin
             PackedSnrValid <= 1'b1; // valid even if less than 32 bits used
             PackedSnrLast  <= 1'b1; 
             case (PackedSnrCount)
               2'd0    :  begin
                 PackedSnr           <= {24'd0,snr0};
                 PackedSnrLastEn     <= 4'b0001; // 1 SNR value
               end
               2'd1    :  begin
                 PackedSnr           <= {16'd0,snr0,PackedSnr[31:24]};
                 PackedSnrLastEn     <= 4'b0011; // 2 SNR values
               end
               2'd2    :  begin
                 PackedSnr           <= {8'd0,snr0,PackedSnr[31:16]};
                 PackedSnrLastEn     <= 4'b0111; // 3 SNR values
               end
               default :  begin
                 PackedSnr           <= {snr0,PackedSnr[31:8]};
                 PackedSnrLastEn     <= 4'b1111; // 4 SNR values
               end
             endcase
           end else begin
             PackedSnrCount <= PackedSnrCount+2'd1;
             PackedSnr      <= {snr0,PackedSnr[31:8]};
             if (PackedSnrCount==2'd3) begin
               PackedSnrValid <= 1'b1; // valid when 32 bits used
               PackedSnrLast  <= 1'b0; 
             end
           end

         // NC=2: shift snr0 and snr1 into PackedSnr, increase PackedSnrCount by 2
         end else begin // cfgNc==NC2
           // on last SNR, align the cotent of PackedSnr with bit 0.
           if (snrLast) begin
             PackedSnrValid <= 1'b1; // valid even if less than 32 bits used
             PackedSnrLast  <= 1'b1; 
             case (PackedSnrCount)
               2'd0    :  begin
                 PackedSnr           <= {16'd0,snr1,snr0};
                 PackedSnrLastEn     <= 4'b0011; // 2 SNR values
               end  
               default :    begin
                 PackedSnr           <= {snr1,snr0,PackedSnr[31:16]}; // should be 2'd2
                 PackedSnrLastEn     <= 4'b1111; // 4 SNR values
               end  
             endcase
           end else begin
             PackedSnrCount <= PackedSnrCount+2'd2;
             PackedSnr      <= {snr1,snr0,PackedSnr[31:16]};
             if (PackedSnrCount==2'd2) begin
               PackedSnrValid <= 1'b1; // valid when 32 bits used
               PackedSnrLast  <= 1'b0; 
             end
           end
         end
       end
     end
   end

endmodule
