//
// This is the angle packer block of the svd system.
//

`default_nettype none

module packer # (
  parameter PIPE_DEPTH = 10  // Used to know how long to run for once the memory interface has finished reading.
) (
  ///////////////////////////////////////////////
  //$port_g Clock and reset
  ///////////////////////////////////////////////
  input  wire                 BFRModemClk,          // BFR Modem Clock
  input  wire                 nBFRModemRst,         // Active Low Reset(Modem Domain)

  ///////////////////////////////////////////////
  //$port_g Control
  ///////////////////////////////////////////////
  input wire                  tctlStart,
  input wire                  tctlStop,

  ///////////////////////////////////////////////
  //$port_g configuration inputs
  ///////////////////////////////////////////////
  input  wire                 cfgFeedbackType,      // Configuration parameter, 0=SU mode 1-MU mode.
  input  wire                 cfgCodebook,          // Configuration parameter, codebook. 
  input  wire           [1:0] cfgNr,
  input  wire           [1:0] cfgNc,

  ///////////////////////////////////////////////
  //$port_g phi and psi inputs.
  ///////////////////////////////////////////////
  input wire           [11:0] phiPhi11,
  input wire           [11:0] phiPsi21,
//`ifdef RW_MUMIMO_RX_EN  
  input wire           [11:0] phiPhi21,
  input wire           [11:0] phiPhi31,
  input wire           [11:0] phiPsi31,
  input wire           [11:0] phiPsi41,
`ifdef RW_TXRX_2X2
  input wire           [11:0] phiPhi22,
  input wire           [11:0] phiPhi32,
  input wire           [11:0] phiPsi32,
  input wire           [11:0] phiPsi42,
`endif  
//`endif  
  input wire [PIPE_DEPTH-1:0] tctlActiveStages,
  input wire                  tctlAdvance,
  input wire            [2:0] tctlBitsPerPsi,        // Calculated constant, bits per psi value.
  input wire            [3:0] tctlBitsPerPhi,        // Calculated constant, bits per phi value.    
  input wire            [3:0] tctl12MinusBitsPerPhi,
  input wire            [3:0] tctl11MinusBitsPerPhi,

  ///////////////////////////////////////////////
  //$port_g packer outputs.
  ///////////////////////////////////////////////
  output wire [31:0]          packerWord,           // Bytes to be written to memory.
  output reg                  packerValid,          // Asserted when packerWord is valid.
  output reg                  packerLast,
  output reg   [5:0]          packerLastNum
  ); 
   
  localparam NR_UNSUPPORTED = 2'd0, NR2 = 2'd1, NR3 = 2'd2, NR4 = 2'd3; // Nr index = Nr-1
  localparam NC1 = 2'd0, NC2 = 2'd1; // Nc index = Nc-1
  localparam CB0 = 1'b0, CB1 = 1'b1;
  localparam FBACK_SU = 1'd0, FBACK_MU = 1'd1;

  //
  // Function to saturate an unsigned number to 9 bits.
  //
  function [8:0] SaturateUsgn9;
    input  [9:0] X;
    input  [3:0] NBits; // Max value possible is 9.
    
    begin
      if (X > (2**NBits - 1)) // do not cast 2**NBits on less than 16 bits else result is 0
        begin
          SaturateUsgn9 = 2**NBits - 1;
        end
      else
        begin
          SaturateUsgn9 = X[8:0];
        end
    end 
  endfunction
  
  //
  // Function to saturate an unsigned number to 7 bits.
  //
  function [6:0] SaturateUsgn7;
    input  [9:0] X7;
    input  [2:0] NBits7; // Max value possible is 7.
    
    begin
      if (X7 > (2**NBits7 - 1)) // do not cast 2**NBits on less than 16 bits else result is 0
        begin
          SaturateUsgn7 = 2**NBits7 - 1;
        end
      else
        begin
          SaturateUsgn7 = X7[6:0];
        end
    end 
  endfunction

  
  reg  [15:0] angle;
  reg  [47:0] anglereg;
  reg   [4:0] bitsvalid;
  reg   [5:0] angleregvalid;
  reg   [2:0] PackerCount;
  reg   [2:0] EndPackerCount;

  reg  [11:0] pPhiPsi21;   // Incoming data delayed by one pipeline tick.
  reg  [11:0] pPhiPhi11;
//`ifdef RW_MUMIMO_RX_EN  
  reg  [11:0] pPhiPhi21;
  reg  [11:0] pPhiPsi31;
  reg  [11:0] pPhiPsi41;
  reg  [11:0] pPhiPhi31;
`ifdef RW_TXRX_2X2
  reg  [11:0] pPhiPhi22;
  reg  [11:0] pPhiPhi32;
  reg  [11:0] pPhiPsi32;
  reg  [11:0] pPhiPsi42;
`endif  
//`endif  

  reg  [11:0] phirt11;  // Interim versions.
  reg  [11:0] psirt21;
//`ifdef RW_MUMIMO_RX_EN  
  reg  [11:0] phirt21;
  reg  [11:0] phirt31;
  reg  [11:0] psirt31;
  reg  [11:0] psirt41;
`ifdef RW_TXRX_2X2
  reg  [11:0] phirt22;
  reg  [11:0] phirt32;
  reg  [11:0] psirt32;
  reg  [11:0] psirt42;
`endif  
//`endif  


  reg   [8:0] pPhir11;  // Delayed by one pipeline delay.
  reg   [6:0] pPsir21;
//`ifdef RW_MUMIMO_RX_EN
  reg   [8:0] pPhir21;
  reg   [8:0] pPhir31;
  reg   [6:0] pPsir31;
  reg   [6:0] pPsir41;
`ifdef RW_TXRX_2X2
  reg   [8:0] pPhir22;
  reg   [8:0] pPhir32;
  reg   [6:0] pPsir32;
  reg   [6:0] pPsir42;
`endif
//`endif
  
  reg   [8:0] ppPhir11;  // Delayed by two pipeline delays.
  reg   [6:0] ppPsir21;
//`ifdef RW_MUMIMO_RX_EN
  reg   [8:0] ppPhir21;
  reg   [8:0] ppPhir31;
  reg   [6:0] ppPsir31;
  reg   [6:0] ppPsir41;
`ifdef RW_TXRX_2X2
  reg   [8:0] ppPhir22;
  reg   [8:0] ppPhir32;
  reg   [6:0] ppPsir32;
`endif  
//`endif  
  
  reg   [8:0] pppPhir11; // Delayed by three pipeline delays.
  reg   [6:0] pppPsir21;
//`ifdef RW_MUMIMO_RX_EN
  reg   [8:0] pppPhir21;
  reg   [8:0] pppPhir31;
  reg   [6:0] pppPsir31;
  reg   [6:0] pppPsir41;
`ifdef RW_TXRX_2X2
  reg   [8:0] pppPhir22;
  reg   [8:0] pppPhir32;
`endif  
//`endif  

`ifdef RW_TXRX_2X2
  reg   [8:0] p4Phir11;  // Delayed by four pipeline delays.
  reg   [6:0] p4Psir21;
//`ifdef RW_MUMIMO_RX_EN
  reg   [8:0] p4Phir21;
  reg   [8:0] p4Phir31;
  reg   [6:0] p4Psir31;
  reg   [8:0] p4Phir22;
//`endif  

  reg   [8:0] p5Phir11;  // Delayed by five pipeline delays.
  reg   [6:0] p5Psir21;
//`ifdef RW_MUMIMO_RX_EN
  reg   [8:0] p5Phir21;
  reg   [8:0] p5Phir31;
//`endif  
`endif  


  
  wire  [8:0] phir11;  // The versions used internally, should be lined up correctly in time.
  wire  [6:0] psir21;
//`ifdef RW_MUMIMO_RX_EN
  wire  [8:0] phir21;
  wire  [8:0] phir31;
  wire  [6:0] psir31;
  wire  [6:0] psir41;
`ifdef RW_TXRX_2X2
  wire  [8:0] phir22;
  wire  [8:0] phir32;
  wire  [6:0] psir32;
  wire  [6:0] psir42;
`endif  
//`endif  


  //
  // For timing reasons register the incoming phiPhi and phiPsi busses.
  //
  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
      if (!nBFRModemRst)
        begin
          pPhiPsi21 <= 12'd0;
          pPhiPhi11 <= 12'd0;
//`ifdef RW_MUMIMO_RX_EN
          pPhiPhi21 <= 12'd0;
          pPhiPsi31 <= 12'd0;
          pPhiPsi41 <= 12'd0;
          pPhiPhi31 <= 12'd0;
`ifdef RW_TXRX_2X2
          pPhiPhi22 <= 12'd0;
          pPhiPhi32 <= 12'd0;
          pPhiPsi32 <= 12'd0;
          pPhiPsi42 <= 12'd0;
`endif         
//`endif         
        end
      else
        begin
          if (tctlAdvance)
            begin
              pPhiPsi21 <= phiPsi21;
              pPhiPhi11 <= phiPhi11;
//`ifdef RW_MUMIMO_RX_EN
              pPhiPhi21 <= phiPhi21;
              pPhiPsi31 <= phiPsi31;
              pPhiPsi41 <= phiPsi41;
              pPhiPhi31 <= phiPhi31;
`ifdef RW_TXRX_2X2
              pPhiPsi32 <= phiPsi32;
              pPhiPsi42 <= phiPsi42;
              pPhiPhi22 <= phiPhi22;
              pPhiPhi32 <= phiPhi32;
`endif             
//`endif             
            end
        end
    end
  
  
  //
  // Round and saturate phi and psi to get phir and psir.
  // The *t versions are intermediate values prior to unsigned saturation.
  //
  always @(*)
    begin
      phirt11 = (pPhiPhi11 >> tctl12MinusBitsPerPhi) + {11'd0,pPhiPhi11[tctl11MinusBitsPerPhi]};
      psirt21 = (pPhiPsi21 >> tctl12MinusBitsPerPhi) + {11'd0,pPhiPsi21[tctl11MinusBitsPerPhi]};
//`ifdef RW_MUMIMO_RX_EN     
      phirt21 = (pPhiPhi21 >> tctl12MinusBitsPerPhi) + {11'd0,pPhiPhi21[tctl11MinusBitsPerPhi]};
      phirt31 = (pPhiPhi31 >> tctl12MinusBitsPerPhi) + {11'd0,pPhiPhi31[tctl11MinusBitsPerPhi]};
      psirt31 = (pPhiPsi31 >> tctl12MinusBitsPerPhi) + {11'd0,pPhiPsi31[tctl11MinusBitsPerPhi]};
      psirt41 = (pPhiPsi41 >> tctl12MinusBitsPerPhi) + {11'd0,pPhiPsi41[tctl11MinusBitsPerPhi]};
`ifdef RW_TXRX_2X2
      phirt22 = (pPhiPhi22 >> tctl12MinusBitsPerPhi) + {11'd0,pPhiPhi22[tctl11MinusBitsPerPhi]};
      phirt32 = (pPhiPhi32 >> tctl12MinusBitsPerPhi) + {11'd0,pPhiPhi32[tctl11MinusBitsPerPhi]};
      psirt32 = (pPhiPsi32 >> tctl12MinusBitsPerPhi) + {11'd0,pPhiPsi32[tctl11MinusBitsPerPhi]};
      psirt42 = (pPhiPsi42 >> tctl12MinusBitsPerPhi) + {11'd0,pPhiPsi42[tctl11MinusBitsPerPhi]};
`endif
//`endif
    end
    
  //
  // At this stage all the phir and psir signals are arriving at different times, delay each one by a different number of
  // pipeline delays to even them all up again. A minimium of one register is required as the inputs are only stable for
  // one tick in some cases.
  //
  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
      if (!nBFRModemRst)
        begin
          //
          // First pipeline delay stage.
          //
          pPhir11 <= 9'd0;
          pPsir21 <= 7'd0;
//`ifdef RW_MUMIMO_RX_EN
          pPhir21 <= 9'd0;
          pPhir31 <= 9'd0;
          pPsir31 <= 7'd0;
          pPsir41 <= 7'd0;
`ifdef RW_TXRX_2X2
          pPhir22 <= 9'd0;
          pPhir32 <= 9'd0;
          pPsir32 <= 7'd0;
          pPsir42 <= 7'd0;
`endif              
//`endif              
          //
          // Second pipeline delay stage.
          //
          ppPhir11 <= 9'd0;
          ppPsir21 <= 7'd0;
//`ifdef RW_MUMIMO_RX_EN
          ppPhir21 <= 9'd0;
          ppPhir31 <= 9'd0;
          ppPsir31 <= 7'd0;
          ppPsir41 <= 7'd0;
`ifdef RW_TXRX_2X2
          ppPhir22 <= 9'd0;
          ppPhir32 <= 9'd0;
          ppPsir32 <= 7'd0;
`endif              
//`endif              
          //
          // Third pipeline delay stage.
          //
          pppPhir11 <= 9'd0;
          pppPsir21 <= 7'd0;
//`ifdef RW_MUMIMO_RX_EN
          pppPhir21 <= 9'd0;
          pppPhir31 <= 9'd0;
          pppPsir31 <= 7'd0;
          pppPsir41 <= 7'd0;
`ifdef RW_TXRX_2X2
          pppPhir22 <= 9'd0;
          pppPhir32 <= 9'd0;
`endif          
//`endif          

`ifdef RW_TXRX_2X2
          //
          // Fourth pipeline delay stage.
          //
          p4Phir11 <= 9'd0;
          p4Psir21 <= 7'd0;
//`ifdef RW_MUMIMO_RX_EN
          p4Phir21 <= 9'd0; 
          p4Phir31 <= 9'd0;
          p4Psir31 <= 7'd0;
          p4Phir22 <= 9'd0;
//`endif
          //
          // Fifth pipeline delay stage.
          //
          p5Phir11 <= 9'd0;
          p5Psir21 <= 7'd0;
//`ifdef RW_MUMIMO_RX_EN
          p5Phir21 <= 9'd0;
          p5Phir31 <= 9'd0;
//`endif          
`endif
         end
      else
        begin
          if (tctlAdvance)
            begin  
              //
              // Capture the incoming data, and first pipeline delay stage.
              //
              pPhir11 <= SaturateUsgn9(phirt11[9:0],tctlBitsPerPhi); // *rt* signals on 12 bits for lint, only 10 used
              pPsir21 <= SaturateUsgn7(psirt21[9:0],tctlBitsPerPsi);
//`ifdef RW_MUMIMO_RX_EN
              pPhir21 <= SaturateUsgn9(phirt21[9:0],tctlBitsPerPhi);
              pPhir31 <= SaturateUsgn9(phirt31[9:0],tctlBitsPerPhi);
              pPsir31 <= SaturateUsgn7(psirt31[9:0],tctlBitsPerPsi);
              pPsir41 <= SaturateUsgn7(psirt41[9:0],tctlBitsPerPsi);
`ifdef RW_TXRX_2X2
              pPhir22 <= SaturateUsgn9(phirt22[9:0],tctlBitsPerPhi);
              pPhir32 <= SaturateUsgn9(phirt32[9:0],tctlBitsPerPhi);
              pPsir32 <= SaturateUsgn7(psirt32[9:0],tctlBitsPerPsi);
              pPsir42 <= SaturateUsgn7(psirt42[9:0],tctlBitsPerPsi);
`endif              
//`endif              
              //
              // Second pipeline delay stage.
              //
              ppPhir11 <= pPhir11;
              ppPsir21 <= pPsir21;
//`ifdef RW_MUMIMO_RX_EN
              ppPhir21 <= pPhir21;
              ppPhir31 <= pPhir31;
              ppPsir31 <= pPsir31;
              ppPsir41 <= pPsir41;
`ifdef RW_TXRX_2X2
              ppPhir22 <= pPhir22;
              ppPhir32 <= pPhir32;
              ppPsir32 <= pPsir32;
`endif              
//`endif              
              //
              // Third pipeline delay stage.
              //
              pppPhir11 <= ppPhir11;
              pppPsir21 <= ppPsir21;
//`ifdef RW_MUMIMO_RX_EN
              pppPhir21 <= ppPhir21;
              pppPhir31 <= ppPhir31;
              pppPsir31 <= ppPsir31;
              pppPsir41 <= ppPsir41;
`ifdef RW_TXRX_2X2
              pppPhir22 <= ppPhir22;
              pppPhir32 <= ppPhir32;
`endif              
//`endif              

`ifdef RW_TXRX_2X2
              //
              // Fourth pipeline delay stage.
              //
              p4Phir11 <= pppPhir11;
              p4Psir21 <= pppPsir21;
//`ifdef RW_MUMIMO_RX_EN
              p4Phir21 <= pppPhir21;
              p4Phir31 <= pppPhir31;
              p4Psir31 <= pppPsir31;
              p4Phir22 <= pppPhir22;
//`endif
              //
              // Fifth pipeline delay stage.
              //
              p5Phir11 <= p4Phir11;
              p5Psir21 <= p4Psir21;              
//`ifdef RW_MUMIMO_RX_EN
              p5Phir21 <= p4Phir21;
              p5Phir31 <= p4Phir31;
//`endif              
`endif
            end
        end
    end

        
`ifdef RW_TXRX_2X2
assign phir11 = p5Phir11;
assign psir21 = p5Psir21;
//`ifdef RW_MUMIMO_RX_EN          
assign phir21 = p5Phir21;
assign phir31 = p5Phir31;
assign psir31 = p4Psir31;
assign psir41 = pppPsir41;
assign phir22 = p4Phir22;
assign phir32 = pppPhir32;
// psi32 does not come from the same pipe level if Nr=3 or Nr=4    
assign psir32 = (cfgNr == NR3) ? pPsir32 : ppPsir32;
assign psir42 = pPsir42;
//`endif        
`else
 // 1x1 2 less level pipes needed
assign phir11 = pppPhir11;
assign psir21 = pppPsir21;
//`ifdef RW_MUMIMO_RX_EN          
assign phir21 = pppPhir21;
assign phir31 = pppPhir31;
assign psir31 = ppPsir31;
assign psir41 = pPsir41;
//`endif        
`endif        

  
  //
  // A very small state machine to control the packing, implemented as a counter
  // rather than with state names as that makes it easier to control.
  // This state machine runs once per sample.
  //
  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
      if (!nBFRModemRst)
        begin
          PackerCount <= 3'd0;
        end
      else
        begin
          if (tctlStop) begin
            PackerCount <= 3'd0;
          end else begin
            case (PackerCount)
              3'd0 : // Idle, stay here until the pipeline advances.
                begin
                  if (tctlAdvance && tctlActiveStages[PIPE_DEPTH-1])
                    begin
                      PackerCount <= PackerCount + 3'd1;
                    end
                end
              default:
                begin
                  if (PackerCount == EndPackerCount)
                    begin
                      PackerCount <= 3'd0;
                    end
                  else
                    begin
                      PackerCount <= PackerCount + 3'd1;
                    end
                end
            endcase
          end
        end
    end
    
  //
  // Calculate EndPackerCount based on the configuration parameters.
  //
  always @(*)
    begin
      case (cfgNr)
        NR2 : EndPackerCount = 3'd2; // 1 packer cycle
        NR3 :
          begin
            if ((cfgNc==NC2) && (cfgFeedbackType==FBACK_MU))
              EndPackerCount = 3'd4; // 3 packer cycles
            else if ({cfgNc,cfgCodebook,cfgFeedbackType}=={NC1,CB0,FBACK_SU})
              EndPackerCount = 3'd2; // 1 packer cycle
            else
              EndPackerCount = 3'd3; // 2 paker cycles
          end
        NR4 : 
          begin
            if (cfgNc==NC1) begin
              if (cfgFeedbackType==FBACK_SU)
                EndPackerCount = 3'd3; // 2 packer cycles
              else
                EndPackerCount = 3'd4; // 3 packer cycles
            end else begin
              case ({cfgCodebook,cfgFeedbackType})
                {CB0,FBACK_SU} : EndPackerCount = 3'd3; // 2 packer cycles
                {CB0,FBACK_MU} : EndPackerCount = 3'd5; // 4 packer cycles
                {CB1,FBACK_SU} : EndPackerCount = 3'd5; // 4 packer cycles
                default        : EndPackerCount = 3'd6; // 5 packer cycles for {CB1,FBACK_MU}
              endcase
            end
          end
        default: EndPackerCount = 3'd6;
      endcase
    end  
        
  //
  // Pack psir and phir into an angles vector and work out how many
  // bits of it are valid.
  //
  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
      if (!nBFRModemRst)
        begin
          angle     <= 16'd0;
          bitsvalid <= 5'd0;
        end
      else
        begin
          case ({cfgNr,cfgNc,cfgCodebook,cfgFeedbackType})
            // CB0, FBACK_SU: bpsi=2 and bphi=4, nbits = Na*3 
            // CB0, FBACK_MU: bpsi=5 and bphi=7, nbits = Na*6 
            // CB1, FBACK_SU: bpsi=4 and bphi=6, nbits = Na*5
            // CB1, FBACK_MU: bpsi=7 and bphi=9, nbits = Na*8 

            {NR2,NC1,CB0,FBACK_SU},{NR2,NC2,CB0,FBACK_SU} :   // Na=2, nbits=6, 1 packer cycle
              begin 
                angle     <= {10'd0,psir21[1:0],phir11[3:0]}; //  6 bits per vector.
                bitsvalid <= 5'd6; 
              end
            {NR2,NC1,CB0,FBACK_MU},{NR2,NC2,CB0,FBACK_MU} :   // Na=2, nbits=12, 1 packer cycle 
              begin
                angle     <= { 4'd0,psir21[4:0],phir11[6:0]}; // 12 bits per vector.
                bitsvalid <= 5'd12;
              end
            {NR2,NC1,CB1,FBACK_SU},{NR2,NC2,CB1,FBACK_SU} :   // Na=2, nbits=10, 1 packer cycle
              begin
                angle     <= { 6'd0,psir21[3:0],phir11[5:0]}; // 10 bits per vector.
                bitsvalid <= 5'd10;
              end
            {NR2,NC1,CB1,FBACK_MU},{NR2,NC2,CB1,FBACK_MU} :   // Na=2, nbits=16, 1 packer cycle 
              begin
                angle     <= {      psir21[6:0],phir11[8:0]}; // 16 bits per vector.
                bitsvalid <= 5'd16;
              end
              
//`ifdef RW_MUMIMO_RX_EN
            {NR3,NC1,CB0,FBACK_SU} : // Na=4, nbits=12, 1 packer cycle
              begin
                angle     <= {4'd0, psir31[1:0],psir21[1:0],phir21[3:0],phir11[3:0]}; // 12 bits per vector.
                bitsvalid <= 5'd12;
              end
            {NR3,NC1,CB0,FBACK_MU} : // Na=4, nbits=24=16+8, 2 packer cycles
              begin
                case (PackerCount)
                  3'd1:
                     begin
                       angle     <= { psir21[1:0],phir21[6:0],phir11[6:0]}; // 16 bits
                       bitsvalid <= 5'd16;
                     end
                  3'd2:
                    begin
                      angle     <= { 8'd0,psir31[4:0],psir21[4:2]}; // 8 bits
                      bitsvalid <= 5'd8;
                    end
                  default:
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
            {NR3,NC1,CB1,FBACK_SU} : // Na=4, nbits=20=16+4, 2 packer cycles
              begin
                case (PackerCount)
                  3'd1:
                     begin
                       angle     <= {psir21[3:0],phir21[5:0],phir11[5:0]}; // 16 bits
                       bitsvalid <= 5'd16;
                     end
                  3'd2:
                    begin
                      angle     <= { 12'd0,psir31[3:0]}; // 4 bits
                      bitsvalid <= 5'd4;
                    end
                  default:
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
            {NR3,NC1,CB1,FBACK_MU} : // Na=4, nbits=32=16+16, 2 packer cycles
              begin
                case (PackerCount)
                  3'd1:
                     begin
                       angle     <= {phir21[6:0],phir11[8:0]}; // 16 bits
                       bitsvalid <= 5'd16;
                     end
                  3'd2:
                    begin
                      angle     <= {psir31[6:0],psir21[6:0],phir21[8:7]}; // 16 bits
                      bitsvalid <= 5'd16;
                    end
                  default:
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
`ifdef RW_TXRX_2X2
            {NR3,NC2,CB0,FBACK_SU} : // Na=6, nbits=18=16+2, 2 packer cycles
              begin
                case (PackerCount)
                  3'd1:
                     begin
                       angle     <= {phir22[3:0], psir31[1:0],psir21[1:0],phir21[3:0],phir11[3:0]}; // 16 bits
                       bitsvalid <= 5'd16;
                     end
                  3'd2:
                    begin
                      angle     <= { 14'd0, psir32[1:0]}; // 2 bits
                      bitsvalid <= 5'd2;
                    end
                  default:
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
            {NR3,NC2,CB0,FBACK_MU} : // Na=6, nbits=36=16+16+4, 3 packer cycles
              begin
                case (PackerCount)
                  3'd1:
                     begin
                       angle     <= { psir21[1:0],phir21[6:0],phir11[6:0]}; // 16 bits
                       bitsvalid <= 5'd16;
                     end
                  3'd2:
                    begin
                      angle     <= { psir32[0],phir22[6:0] ,psir31[4:0],psir21[4:2]}; // 16 bits
                      bitsvalid <= 5'd16;
                    end
                  3'd3:
                    begin
                      angle     <= { 12'd0,psir32[4:1]}; // 4 bits
                      bitsvalid <= 5'd4;
                    end
                  default:
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
            {NR3,NC2,CB1,FBACK_SU} : // Na=6, nbits=30=16+14, 2 packer cycles
              begin
                case (PackerCount)
                  3'd1:
                     begin
                       angle     <= {psir21[3:0],phir21[5:0],phir11[5:0]}; // 16 bits
                       bitsvalid <= 5'd16;
                     end
                  3'd2:
                    begin
                      angle     <= {2'd0, psir32[3:0],phir22[5:0], psir31[3:0]}; // 14 bits
                      bitsvalid <= 5'd14;
                    end
                  default:
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
            {NR3,NC2,CB1,FBACK_MU} : // Na=6, nbits=48=16+16+16, 3 packer cycles
              begin
                case (PackerCount)
                  3'd1:
                     begin
                       angle     <= {phir21[6:0],phir11[8:0]}; // 16 bits
                       bitsvalid <= 5'd16;
                     end
                  3'd2:
                    begin
                      angle     <= {psir31[6:0],psir21[6:0],phir21[8:7]}; // 16 bits
                      bitsvalid <= 5'd16;
                    end
                  3'd3:
                    begin
                      angle     <= {psir32[6:0],phir22[8:0]}; // 16 bits
                      bitsvalid <= 5'd16;
                    end
                  default:
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
`endif
            {NR4,NC1,CB0,FBACK_SU} : // Na=6, nbits=18=16+2, 2 packer cycles - angles are not the same as for 3x2
              begin
                case (PackerCount)
                  3'd1:
                     begin
                       angle     <= { psir31[1:0],psir21[1:0],phir31[3:0],phir21[3:0],phir11[3:0]}; // 16
                       bitsvalid <= 5'd16;
                     end
                  3'd2:
                     begin
                       angle     <= {14'd0, psir41[1:0]}; // 2 bits
                       bitsvalid <= 5'd2;
                     end
                  default:
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
            {NR4,NC1,CB0,FBACK_MU} : // Na=6, nbits=36=16+16+4, 3 packer cycles - angles are no the same as for 3x2
              begin
                case (PackerCount)
                  3'd1 :
                    begin
                      angle <= {phir31[1:0], phir21[6:0], phir11[6:0]};
                      bitsvalid <= 5'd16;
                    end
                  3'd2 : 
                    begin
                      angle <= {  psir41[0], psir31[4:0], psir21[4:0], phir31[6:2]};
                      bitsvalid <= 5'd16;
                    end
                  3'd3 : 
                    begin
                      angle <= {12'd0,psir41[4:1]};
                      bitsvalid <= 5'd4;
                    end
                  default :
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end          
            {NR4,NC1,CB1,FBACK_SU} : // Na=6, nbits=30=16+14, 2 packer cycles - angles are no the same as for 3x2
              begin
                case (PackerCount)
                  3'd1 :
                    begin
                      angle <= {phir31[3:0],phir21[5:0], phir11[5:0]};
                      bitsvalid <= 5'd16;
                    end
                  3'd2 : 
                    begin
                      angle <= {2'd0,psir41[3:0], psir31[3:0], psir21[3:0], phir31[5:4]};
                      bitsvalid <= 5'd14;
                    end
                  default :
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
            {NR4,NC1,CB1,FBACK_MU} : // Na=6, nbits=48=16+16+16, 3 packer cycles - angles are no the same as for 3x2
              begin
                case (PackerCount)
                  3'd1 :
                    begin
                      angle <= {phir21[6:0], phir11[8:0]};
                      bitsvalid <= 5'd16;
                    end
                  3'd2 : 
                    begin
                      angle <= {psir21[4:0],phir31[8:0],phir21[8:7]};
                      bitsvalid <= 5'd16;
                    end
                  3'd3 : 
                    begin
                      angle <= {psir41[6:0], psir31[6:0], psir21[6:5]};
                      bitsvalid <= 5'd16;
                    end
                  default :
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
`ifdef RW_TXRX_2X2
            {NR4,NC2,CB0,FBACK_SU} : // Na=10, nbits=30=16+14, 2 packer cycles
              begin
                case (PackerCount)
                  3'd1 :
                    begin
                      angle <= {psir31[1:0], psir21[1:0], phir31[3:0], phir21[3:0], phir11[3:0]};
                      bitsvalid <= 5'd16;
                    end
                  3'd2 : 
                    begin
                      angle <= {2'd0, psir42[1:0], psir32[1:0], phir32[3:0], phir22[3:0], psir41[1:0]};
                      bitsvalid <= 5'd14;
                    end
                  default :
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
            {NR4,NC2,CB0,FBACK_MU} : // Na=10, nbits=60=16+16+16+12, 4 packer cycles
              begin
                case (PackerCount)
                  3'd1 :
                    begin
                      angle <= {phir31[1:0], phir21[6:0], phir11[6:0]};
                      bitsvalid <= 5'd16;
                    end
                  3'd2 : 
                    begin
                      angle <= {  psir41[0], psir31[4:0], psir21[4:0], phir31[6:2]};
                      bitsvalid <= 5'd16;
                    end
                  3'd3 : 
                    begin
                      angle <= {phir32[4:0], phir22[6:0], psir41[4:1]};
                      bitsvalid <= 5'd16;
                    end
                  3'd4 : 
                    begin
                      angle <= {4'd0, psir42[4:0], psir32[4:0], phir32[6:5]};
                      bitsvalid <= 5'd12;
                    end
                  default :
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
            {NR4,NC2,CB1,FBACK_SU} : // Na=10, nbits=50=16+16+16+2, 4 packer cycles
              begin
                case (PackerCount)
                  3'd1 :
                    begin
                      angle <= {phir31[3:0],phir21[5:0], phir11[5:0]};
                      bitsvalid <= 5'd16;
                    end
                  3'd2 : 
                    begin
                      angle <= {phir22[1:0],psir41[3:0], psir31[3:0], psir21[3:0], phir31[5:4]};
                      bitsvalid <= 5'd16;
                    end
                  3'd3 : 
                    begin
                      angle <= { psir42[1:0],psir32[3:0], phir32[5:0], phir22[5:2] };
                      bitsvalid <= 5'd16;
                    end
                  3'd4 : 
                    begin
                      angle <= {14'd0,psir42[3:2]};
                      bitsvalid <= 5'd2;
                    end
                  default :
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
            {NR4,NC2,CB1,FBACK_MU} : // Na=10, nbits=80=16+16+16+16+16, 5 packer cycles
              begin
                case (PackerCount)
                  3'd1 :
                    begin
                      angle <= {phir21[6:0], phir11[8:0]};
                      bitsvalid <= 5'd16;
                    end
                  3'd2 : 
                    begin
                      angle <= {psir21[4:0], phir31[8:0], phir21[8:7]};
                      bitsvalid <= 5'd16;
                    end
                  3'd3 : 
                    begin
                      angle <= {psir41[6:0], psir31[6:0], psir21[6:5]};
                      bitsvalid <= 5'd16;
                    end
                  3'd4 : 
                    begin
                      angle <= {phir32[6:0], phir22[8:0]};
                      bitsvalid <= 5'd16;
                    end
                  3'd5 :
                    begin
                      angle <= {psir42[6:0], psir32[6:0], phir32[8:7]};
                      bitsvalid <= 5'd16;
                    end
                  default :
                    begin
                      angle <= 16'd0;
                      bitsvalid <= 5'd0;
                    end
                endcase
              end
`endif
//`endif
            default:
              begin
                angle     <= {16'hdead}; //Should never happen.
                bitsvalid <= 5'd16;
              end          
            endcase
        end  
    end
          
  //
  // Depending on the configuration pack the bits into the appropriate vector.
  //
  always @(posedge BFRModemClk or negedge nBFRModemRst)
  begin
    if (!nBFRModemRst)
      begin
        anglereg         <= 48'd0;
        angleregvalid    <= 6'd0; 
        packerValid      <= 1'b0;
        packerLast       <= 1'b0;
        packerLastNum    <= 6'd0;
      end
    else
      begin
        packerLast    <= 1'b0; // pulse
        //
        // Reset everything at the start of a sequence, this ensures that odd things will not happen
        // after reconfiguration due to partially populated bytes being present. Perform the same type of
        // reset when told to stop, just to be sure.
        //
        if (tctlStart || tctlStop)
          begin
          angleregvalid    <= 6'd0;
          packerValid      <= 1'b0;
          anglereg         <= 48'd0;
          packerLast       <= 1'b0;
          packerLastNum    <= 6'd0;
        end
        //
        // If packerValid is asserted but we are no longer processing data then update
        // angleregvalid and deassert packerValid.
        //
        else if (packerValid && (PackerCount == 3'd0))
          begin
            angleregvalid <= angleregvalid - 6'd32;
            anglereg      <= {32'd0,anglereg[47:32]};
            packerValid   <= 1'b0;
          end
        //
        // If PackerCount > 1 then load new data in to the anglereg register to the left of whatever
        // is already loaded.
        //
        else if (PackerCount > 3'd1)
          begin
            //
            // Load the new angles into anglereg
            //
            case (angleregvalid)
              //
              // Cases 0 to 31, no unloading will occur on this tick unless it's the very end.
              // Note that anglereg is always even. 
              //
              6'd0    : anglereg <= {32'd0,angle};
              6'd2    : anglereg <= {30'd0,angle,anglereg[ 1:0]};
              6'd4    : anglereg <= {28'd0,angle,anglereg[ 3:0]};
              6'd6    : anglereg <= {26'd0,angle,anglereg[ 5:0]};
              6'd8    : anglereg <= {24'd0,angle,anglereg[ 7:0]};
              6'd10   : anglereg <= {22'd0,angle,anglereg[ 9:0]};
              6'd12   : anglereg <= {20'd0,angle,anglereg[11:0]};
              6'd14   : anglereg <= {18'd0,angle,anglereg[13:0]};
              6'd16   : anglereg <= {16'd0,angle,anglereg[15:0]};
              6'd18   : anglereg <= {14'd0,angle,anglereg[17:0]};
              6'd20   : anglereg <= {12'd0,angle,anglereg[19:0]};
              6'd22   : anglereg <= {10'd0,angle,anglereg[21:0]};
              6'd24   : anglereg <= { 8'd0,angle,anglereg[23:0]};
              6'd26   : anglereg <= { 6'd0,angle,anglereg[25:0]};
              6'd28   : anglereg <= { 4'd0,angle,anglereg[27:0]};
              6'd30   : anglereg <= { 2'd0,angle,anglereg[29:0]};
              //
              // Cases 32 to 48, 32 bits will be unloaded at the same time as angle is getting
              // loaded into anglereg.
              //
              6'd32   : anglereg <= {32'd0,angle};
              6'd34   : anglereg <= {30'd0,angle,anglereg[33:32]};
              6'd36   : anglereg <= {28'd0,angle,anglereg[35:32]};
              6'd38   : anglereg <= {26'd0,angle,anglereg[37:32]};
              6'd40   : anglereg <= {24'd0,angle,anglereg[39:32]};
              6'd42   : anglereg <= {22'd0,angle,anglereg[41:32]};
              6'd44   : anglereg <= {20'd0,angle,anglereg[43:32]};
              default : anglereg <= {18'd0,angle,anglereg[45:32]}; // 5'd46
            endcase
            //
            // Calculate the new angleregvalid value, if packerValid has been asserted the last time round then
            // 32 bits are coming out this tick.
            //
            if (packerValid)
              begin
                angleregvalid <= angleregvalid + {1'b0,bitsvalid} - 6'd32;
              end
            else
              begin
                angleregvalid <= angleregvalid + {1'b0,bitsvalid};
              end
            //
            // If packerValid is already asserted then keep it asserted if there will be 32 or more bits ready to go out
            // on the next tick.
            //
            if (packerValid && ((angleregvalid + {1'b0,bitsvalid} - 6'd32) >= 6'd32))
              begin
                packerValid <= 1'b1;
                if ((tctlActiveStages[PIPE_DEPTH-1]==1'b0) && (PackerCount==EndPackerCount) &&  ((angleregvalid + {1'b0,bitsvalid} - 6'd32) == 6'd32)) begin
                  packerLast    <= 1'b1;
                  packerLastNum <= 6'd32;
                end
              end
            //
            // If packerValid is not asserted then assert it if there will be 32 or more bits ready to go out on
            // the next tick.
            //
            else if (!packerValid && ((angleregvalid + {1'b0,bitsvalid}) >= 6'd32))
              begin
                packerValid <= 1'b1;
                if ((tctlActiveStages[PIPE_DEPTH-1]==1'b0) && (PackerCount==EndPackerCount) && ((angleregvalid + {1'b0,bitsvalid}) == 6'd32)) begin
                  packerLast    <= 1'b1;
                  packerLastNum <= 6'd32;
                end
              end
            else
              begin
                packerValid <= 1'b0;
              end
          end
        //
        // If we have reached the end and there are less than 16 bits left then send them out.
        //
        else if (!tctlActiveStages[PIPE_DEPTH-1] && (angleregvalid != 6'd0) && (angleregvalid < 6'd32) && (PackerCount == 3'd0))
          begin
            packerValid   <= 1'b1;
            packerLast    <= 1'b1;
            angleregvalid <= 6'd0;
            packerLastNum <= angleregvalid;
          end
        else
          begin
            packerValid <= 1'b0;
          end
      end
    end
       
  assign packerWord = anglereg[31:0];
    
endmodule
