//
// Module to multiply and add two sets of complex numbers as used in complex 2x2 array multiplication.
// Doing this sequentially saves a lot of multipliers and thus a lot of gates.
// This version has two multipliers.
//
// Z = A*B + C*D
//

`default_nettype none

module cmult13 (
  ///////////////////////////////////////////////
  //$port_g clock and reset.
  ///////////////////////////////////////////////
  input  wire                 BFRModemClk,          // BFR Modem Clock
  input  wire                 nBFRModemRst,         // Active Low Reset(Modem Domain)

  ///////////////////////////////////////////////
  //$port_g Configuration and control
  ///////////////////////////////////////////////
  input wire                 tctlAdvance,  
  ///////////////////////////////////////////////
  //$port_g complex inputs.
  ///////////////////////////////////////////////
  input wire signed [12:0]    ARe,
  input wire signed [12:0]    AIm,
  input wire signed [13:0]    BRe,
  input wire signed [13:0]    BIm,
  input wire signed [12:0]    CRe,
  input wire signed [12:0]    CIm,
  input wire signed [13:0]    DRe,
  ///////////////////////////////////////////////
  //$port_g complex outputs.
  ///////////////////////////////////////////////
  output reg   signed [27:0]   ZRe,
  output reg   signed [27:0]   ZIm
  );

  
  reg  signed [13:0] MIn1 ,MAIn2,MBIn2;	// Inputs to the two multipliers.
  wire signed [27:0] MAZ, MBZ; 
  reg          [1:0] StateCount;

  wire signed [13:0]    BImInv;
  
  assign BImInv = ~BIm + 13'd1;

  //
  // A simple counter to act as a state machine.
  //
  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
	  if (!nBFRModemRst)
        begin
          StateCount <= 2'd0;
          ZRe        <= 28'd0;
          ZIm        <= 28'd0;
        end
      else
        begin
          //
          // The state counter, starts when tctlAdvance is asserted and increments till it
          // wraps around, then stop.
          //
          if ((StateCount != 2'd0) || tctlAdvance)
            begin
              StateCount <= StateCount + 2'd1;
            end
          //
          // The two accumulators, for ZRe and ZIm.
          //
          case (StateCount)
            //
            2'd1,2'd2,2'd3: // accumulate
              begin
                ZRe <= ZRe + MAZ;
                ZIm <= ZIm + MBZ;
              end
            default: // 0
              begin
                if (tctlAdvance) begin // reset
                  ZRe <= 28'd0;
                  ZIm <= 28'd0;
                end else begin // keep value
                  ZRe <= ZRe;
                  ZIm <= ZIm;
                end
              end
          endcase
        end
    end  
  
  //
  // Use two multipliers and mux their inputs based on the state machine.
  //
  always @(*)
    begin
      case (StateCount)
        //
        2'd1: // A
          begin 
            MIn1  = {ARe[12],ARe}; 
            MAIn2 = BRe;
            MBIn2 = BIm; 
          end
        //
        2'd2: // B
          begin 
            MIn1  = {AIm[12],AIm}; 
            MAIn2 = BImInv;
            MBIn2 = BRe; 
          end
        //
        default: // C
          begin 
            MIn1  = DRe; 
            MAIn2 = {CRe[12],CRe};
            MBIn2 = {CIm[12],CIm}; 
          end
      endcase
    end
    
  assign MAZ = {{14{MIn1[13]}},MIn1}  * {{14{MAIn2[13]}},MAIn2};
  assign MBZ = {{14{MIn1[13]}},MIn1}  * {{14{MBIn2[13]}},MBIn2};

endmodule
