//
// The Cordic Phase block. 
//

`default_nettype none

module cordicPhase
#(   
  parameter PREPOSTPROC = 0
)
(
            ///////////////////////////////////////////////
            // Clock and reset
            ///////////////////////////////////////////////
            input   wire                          Clk,   // Modem clock.
            input   wire                          Rst,  // Modem active low reset.

            ///////////////////////////////////////////////
            // Control Signals.
            ///////////////////////////////////////////////
            //Control Signals
            input   wire                          Start, //Active HIGH start signal.
            output  reg                           Done,	// Asserted when cordic is finished.
            
            ///////////////////////////////////////////////
            // Data signals.
            //////////////////////////////////////////////
            input   wire   signed     [12:0]      RIn, //Real part of Input
            input   wire   signed     [12:0]      IIn, //Imaginary part of Input
            output  reg    signed     [12:0]      ROut, //Real part of Output
            output  reg    signed     [11:0]      POut 
            );

   
   reg               Active;
   reg         [3:0] K;
   reg signed [11:0] P, aP1, aP2;
   reg signed [14:0] R, aR1, aR2, aR;  
   reg signed [14:0] I, aI1, aI2, aI;
  
   // Pre processing
   wire signed     [12:0]      RPre;
   wire signed     [12:0]      IPre;
   // Post processing
   wire signed     [11:0]      nextPOut; 

   function [11:0] Arctanvalues;
     input [3:0] X;
     begin
       case (X)
            4'd1: Arctanvalues = 12'd511;   
            4'd2: Arctanvalues = 12'd302;   
            4'd3: Arctanvalues = 12'd159;    
            4'd4: Arctanvalues = 12'd81;    
            4'd5: Arctanvalues = 12'd40;    
            4'd6: Arctanvalues = 12'd20;    
            4'd7: Arctanvalues = 12'd10;     
            4'd8: Arctanvalues = 12'd5;     
            4'd9: Arctanvalues = 12'd2;     
            4'd10: Arctanvalues = 12'd1;
           default : Arctanvalues = 12'd0;
       endcase
     end
   endfunction
   
   //
   // Function fo calculate the "floor(It*2^-(K-1));" type function as seen in the
   // MatLab code.
   //
   function signed [14:0] CordicFloor;
     input [14:0] Y;
     input [ 3:0] Z;
     begin
       //
       case (Z)
         4'd0:    CordicFloor = Y;
         4'd1:    CordicFloor = {Y[14],Y[14:1]};
         4'd2:    CordicFloor = {{2{Y[14]}},Y[14:2]};
         4'd3:    CordicFloor = {{3{Y[14]}},Y[14:3]};
         4'd4:    CordicFloor = {{4{Y[14]}},Y[14:4]};
         4'd5:    CordicFloor = {{5{Y[14]}},Y[14:5]};
         4'd6:    CordicFloor = {{6{Y[14]}},Y[14:6]};
         4'd7:    CordicFloor = {{7{Y[14]}},Y[14:7]};
         4'd8:    CordicFloor = {{8{Y[14]}},Y[14:8]};
         default: CordicFloor = {{9{Y[14]}},Y[14:9]};
       endcase
     end
   endfunction 

   //
   // Post-processing
   //
generate
  if (PREPOSTPROC == 1)
  begin
   cordicpre cpre(.InRe(RIn), .InIm(IIn), .R(RPre), .I(IPre));

   reg                         RNeg;
   always @(posedge Clk or negedge Rst)
   begin
     if (!Rst)
       begin
         RNeg   <= 1'b0;
       end
     else
       begin
         if (Start)
           begin
             RNeg <= RIn[12];
           end
       end             
   end           
   
   assign nextPOut = RNeg ? {!aP1[11],aP1[10:0]} : aP1;
   

  end
  else
  begin
   assign RPre = RIn;
   assign IPre = IIn;
   assign nextPOut = aP1;
  end
endgenerate

   //
   // Control the calculations.
   //
   always @(posedge Clk or negedge Rst)
   begin
     if (!Rst)
       begin
         Done   <= 1'b0;
         K      <= 4'd1;
         Active <= 1'b0;
         R      <= 15'd0;
         I      <= 15'd0;
         P      <= 12'd0;
         ROut   <= 13'd0;
         POut   <= 12'd0;
       end
     else
       begin
         //
         // Control the K loop counter, Active and Done signals.
         //
         if (K == 4'd9)
           begin
             K <= 4'd1;
             Active <= 1'b0;
             Done <= 1'b1;
           end
         else if (Start)
           begin
             K <= K + 4'd2;
             Active <= 1'b1;
             Done <= 1'b0;
           end
         else if (Active)
           begin
             K <= K + 4'd2;
             Done <= 1'b0;
           end
         else
           begin
             Done <= 1'b0;
           end
         //
         // Do the actual calculations here.
         //
         if (!Start && !Active)
           begin
             R <= {RPre[12],RPre[12],RPre};
             I <= {IPre[12],IPre[12],IPre};
             P <= 12'd0;
           end
         else
           begin
             R <= aR2;
             I <= aI2;
             P <= aP2;
           end
         //
         // Load the outputs at the correct time.
         //
         if (K == 4'd9)
           begin
             ROut <= aR1[14:2]+ {12'd0,aR1[1]}; // Divide by 4 and round.
             POut <= nextPOut;
           end
       end             
   end           
   
  //
  // Mux between the input data and the registered values, this saves a tick at the start.
  //
  always @(*)
    begin
      if (Start)
        begin
          aR = {RPre[12],RPre[12],RPre};
          aI = {IPre[12],IPre[12],IPre};
        end
      else
        begin
          aR = R;
          aI = I;
        end
    end
 
  //
  // Do the cordic calculations here in an async process so that two iterations can be done per clock
  // cycle.
  //
  always @(*)
    begin
      if (K == 4'd0) // Ie ((!Active) && (!Start))
        begin
          aR1 = {RPre[12],RPre[12],RPre};
          aI1 = {IPre[12],IPre[12],IPre};
          aP1 = 12'd0;
          aR2 = 15'd0;
          aI2 = 15'd0;
          aP2 = 12'd0;
        end
      else		// Active 
        begin
          //
          // First iteration.
          //
          if (!aI[14])         // Ie if (aI >= 0)
            begin
              aR1 = aR + CordicFloor(aI,(K-4'd1));                         
              aI1 = aI - CordicFloor(aR,(K-4'd1));                         // $floor((1.0*aR)/((2**(K-1))));//
              aP1 = P + Arctanvalues(K);
              //$display("First iteration, K=%d aI1=%d",K,aI1);
            end
          else
            begin
              aR1 = aR - CordicFloor(aI,(K-4'd1));
              aI1 = aI + CordicFloor(aR,(K-4'd1));
              aP1 = P - Arctanvalues(K);
            end
          //
          //Second iteration.
          //
          if (!aI1[14])        // Ie if (aI1 >= 0)
            begin
              aR2 = aR1 + CordicFloor(aI1,K);
              aI2 = aI1 - CordicFloor(aR1,K);
              aP2 = aP1 + Arctanvalues(K+4'd1);
            end
          else
            begin
              aR2 = aR1 - CordicFloor(aI1,K);
              aI2 = aI1 + CordicFloor(aR1,K);
              aP2 = aP1 - Arctanvalues(K+4'd1);
            end
        end
    end            
           
endmodule
