//
// Block to perform the Cordic Rot calculations.
//
`default_nettype none

module cordicRot # (
    parameter INWIDTH = 13
  )(
  ///////////////////////////////////////////////
  // Clock and reset
  ///////////////////////////////////////////////
  input   wire                          Clk,   // Modem clock.
  input   wire                          Rst,  // Modem active low reset.

  ///////////////////////////////////////////////
  // Control Signals.
  ///////////////////////////////////////////////
  //Control Signals
  input   wire                          Start, //Active HIGH start signal.
  output  reg                           Done,	// Asserted when cordic is finished.
  ///////////////////////////////////////////////
  //$port_g inputs.
  ///////////////////////////////////////////////
  input  wire signed [INWIDTH-1:0]      RIn, // INWIDTH must be 13 or 14
  input  wire signed [INWIDTH-1:0]      IIn,
  input  wire signed [11:0]             ThetaIn,
  ///////////////////////////////////////////////
  //$port_g outputs
  ///////////////////////////////////////////////
  output reg signed [12:0]              ROut,
  output reg signed [12:0]              IOut
  );
  
  //
  // Function to generate the Arctan values.
  //
  function [11:0] Arctanvalues;
    input [3:0] X;
    begin
      case (X)
           4'd1: Arctanvalues = 12'd511;   
           4'd2: Arctanvalues = 12'd302;   
           4'd3: Arctanvalues = 12'd159;    
           4'd4: Arctanvalues = 12'd81;    
           4'd5: Arctanvalues = 12'd40;    
           4'd6: Arctanvalues = 12'd20;    
           4'd7: Arctanvalues = 12'd10;     
           4'd8: Arctanvalues = 12'd5;     
           4'd9: Arctanvalues = 12'd2;     
           default: Arctanvalues = 12'd1;
      endcase
    end
  endfunction
  
  //
  // Function to calculate the "floor(It*2^-(K-1));" type function as seen in the
  // MatLab code.
  //
  function signed [14:0] CordicFloor;
    input [14:0] Y;
    input [3:0] Z;
    begin
      //
      case (Z)
        4'd0:    CordicFloor = Y;
        4'd1:    CordicFloor = {Y[14],Y[14:1]};
        4'd2:    CordicFloor = {{2{Y[14]}},Y[14:2]};
        4'd3:    CordicFloor = {{3{Y[14]}},Y[14:3]};
        4'd4:    CordicFloor = {{4{Y[14]}},Y[14:4]};
        4'd5:    CordicFloor = {{5{Y[14]}},Y[14:5]};
        4'd6:    CordicFloor = {{6{Y[14]}},Y[14:6]};
        4'd7:    CordicFloor = {{7{Y[14]}},Y[14:7]};
        4'd8:    CordicFloor = {{8{Y[14]}},Y[14:8]};
        default: CordicFloor = {{9{Y[14]}},Y[14:9]};
      endcase
    end
  endfunction 
   
  //
  // Declarations.
  //
  reg signed [11:0] Theta, ThetaTmp, aT1, aT2, aT; 
  reg               Active;
  reg         [3:0] K;
  reg signed [14:0] R, aR1, aR2, aR, naR1;  
  reg signed [14:0] I, aI1, aI2, aI, naI1;
  reg               NegateOutput;
  reg               aNegateOutput;
  
  wire signed [14:0]      RInExt;
  wire signed [14:0]      IInExt;
 
  genvar i;
  generate
    for (i=INWIDTH; i <= 14; i=i+1) begin : EXTEND
      assign RInExt[i] = RIn[INWIDTH-1];
      assign IInExt[i] = IIn[INWIDTH-1];
    end
  endgenerate
  // LSBs
  assign RInExt[INWIDTH-1:0] = RIn;
  assign IInExt[INWIDTH-1:0] = IIn;

  //
  // Theta in has a range of -2048 to 2047, move it to 
  // a range of -1024 to 1023
  //
  always @(*)
    begin
      if (!ThetaIn[11] && (ThetaIn > 12'd1024))
        begin
          ThetaTmp = ThetaIn - 12'd2048;
          aNegateOutput = 1'b1;
        end
      else if (ThetaIn[11] && (ThetaIn < -12'd1024))
        begin
          ThetaTmp = ThetaIn + 12'd2048;
          aNegateOutput = 1'b1;
        end
      else
        begin
          ThetaTmp = ThetaIn[11:0];
          aNegateOutput = 1'b0;
        end
    end
  
  //
  // Control the calculations.
  //
  always @(posedge Clk or negedge Rst)
  begin
    if (!Rst)
      begin
        Done         <= 1'b0;
        K            <= 4'd1;
        Active       <= 1'b0;
        R            <= 15'd0;
        I            <= 15'd0;
        Theta        <= 12'd0;
        ROut         <= 13'd0;
        IOut         <= 13'd0;
        NegateOutput <= 1'b0;
      end
    else
      begin
        //
        // Control the K loop counter, Active and Done signals.
        //
        if (K == 4'd9)
          begin
            K <= 4'd1;
            Active <= 1'b0;
            Done <= 1'b1;
          end
        else if (Start)
          begin
            K <= K + 4'd2;
            Active <= 1'b1;
            Done <= 1'b0;
          end
        else if (Active)
          begin
            K <= K + 4'd2;
            Done <= 1'b0;
          end
        else
          begin
            Done <= 1'b0;
          end
        //
        // Note whether we have to invert the output or not.
        //
        if (Start)
          begin
            NegateOutput <= aNegateOutput;
          end
        //
        // Do the actual calculations here.
        //
        if (!Start && !Active)
          begin
            R            <= RInExt;
            I            <= IInExt;
            Theta        <= ThetaTmp;
          end
        else
          begin
            R <= aR2;
            I <= aI2;
            Theta <= aT2;
          end
        //
        // Load the outputs at the correct time, round and negate as necessary.
        //
        if (K == 4'd9)
          begin
            if (NegateOutput)
              begin
                ROut <= naR1[14:2] + {12'd0,naR1[1]}; // Divide by 4, round and negate.
                IOut <= naI1[14:2] + {12'd0,naI1[1]}; // Divide by 4, round and negate.
              end
            else
              begin
                ROut <= aR1[14:2] + {12'd0,aR1[1]}; // Divide by 4 and round.
                IOut <= aI1[14:2] + {12'd0,aI1[1]}; // Divide by 4 and round.
              end
          end
      end             
  end           
  
  //
  // Mux between the input data and the registered values, this saves a tick at the start.
  //
  always @(*)
    begin
      if (Start)
        begin
          aR = RInExt;
          aI = IInExt;
          aT = ThetaTmp;
        end
      else
        begin
          aR = R;
          aI = I;
          aT = Theta;
        end
    end
  

  //
  // Do the cordic calculations here in an async process so that two iterations can be done per clock
  // cycle.
  //
  always @(*)
    begin
      if (K == 4'd0) // ((~Active) && (~Start))
        begin
          aR1 = RInExt;
          aI1 = IInExt;
          aR2 = 15'd0;
          aI2 = 15'd0;
          aT1 = 12'd0;
          aT2 = 12'd0;
        end
      else		// Active 
        begin
          //
          // First iteration.
          //
          if (aT[11]) // Ie if (aT < 0) 
            begin
              aR1 = aR + CordicFloor(aI,(K-4'd1));                         
              aI1 = aI - CordicFloor(aR,(K-4'd1));                  
              aT1 = aT + Arctanvalues(K);
            end
          else
            begin
              aR1 = aR - CordicFloor(aI,(K-4'd1));
              aI1 = aI + CordicFloor(aR,(K-4'd1));
              aT1 = aT - Arctanvalues(K);
            end
          //
          //Second iteration.
          //
          if (aT1[11])   // Ie if (aT1 < 0)
            begin
              aR2 = aR1 + CordicFloor(aI1,K);
              aI2 = aI1 - CordicFloor(aR1,K);
              aT2 = aT1 + Arctanvalues(K+4'd1);
            end
          else
            begin
              aR2 = aR1 - CordicFloor(aI1,K);
              aI2 = aI1 + CordicFloor(aR1,K);
              aT2 = aT1 - Arctanvalues(K+4'd1);
            end
        end
    end            

  always @(*)
    begin
      naR1 = -aR1;
      naI1 = -aI1;
    end
   
endmodule
