//
// A block to scale and round and saturate a number. The scaling is done by shifting P-11 bits to the right.
// Saturation is done to 13 bits as per the Matlab code.
//
// I = input number
// P = precision
// Z = output
//

`default_nettype none

module roundscale  # (
    parameter WIDTH = 28
) (
  ///////////////////////////////////////////////
  //$port_g inputs.
  ///////////////////////////////////////////////
  input  wire signed [WIDTH-1:0]    I,
  input  wire             [ 4:0]    P,
  ///////////////////////////////////////////////
  //$port_g outputs
  ///////////////////////////////////////////////
  output wire signed     [ 12:0]    Z
  );
  
  reg signed [WIDTH-1:0] Zint;       // Result pre saturation.
  reg signed      [ 4:0] Pshift;	    // No of bits to shift, P-11 >= 0.
  
  //
  // Calculate the number of bits to shift right by
  //
  always @(*)
    begin
      Pshift = P -5'd11;
    end
   
  //
  // Do the shifting and rounding. 
 //
  wire [WIDTH+15:0] ZintNegShift; // for lint
  assign ZintNegShift = ({16'hFFFF,I} >> Pshift) + {{(WIDTH+15){1'b0}},I[Pshift-1]};

  always @(*)
    begin
      //
      // If we dont' have to shift (Pshift=0) then just return the input.
      //
      if (Pshift == 5'd0) begin
        Zint = I;
      end else begin 
        // Shift to the right, ie make the number smaller.

        // If I is a positive number and the MSB of the part that's been dropped off is 1 then round by adding 1.
        if (!I[WIDTH-1]) // Top bit not set, positive number.
          Zint = (I >> Pshift) + {{(WIDTH-1){1'b0}},I[Pshift-1]};

        // If it's a negative number and the MSB of the part that's been dropped off is 1 then round by adding 1.
        // Add 1's in on the left as we shift.
        else // Top bit set, negative number.
          Zint   = ZintNegShift[WIDTH-1:0];
      end
    end
     
generate
if (WIDTH>13) begin : Sat13Gen
  reg signed [12:0] Zgen;

  //
  // Saturate to 13 bits.
  //
  always @(*)
    begin
      //
      // Positive saturation, funny value detection used here to keep lint happy.
      // If the top bit is zero then it's a positive number, then check the lower bits.
      //
      if (!Zint[WIDTH-1] && (Zint[WIDTH-2:12] != {(WIDTH-13){1'b0}}))        // Ie (Zint > 4095)
        begin
          Zgen = 13'd4095;                               // 4095
        end
      else if (Zint[WIDTH-1] && (Zint[WIDTH-2:12] != {(WIDTH-13){1'b1}}))   // Ie (Zint < -4096)
        begin
          Zgen = 13'h1000;                               // -4096
        end
      else
        begin
          Zgen = Zint[12:0];                             // Zint
        end
    end  
  assign Z = Zgen;       

end else begin : noSat13Gen
  assign Z = Zint;
end

endgenerate
    
endmodule
