//////////////////////////////////////////////////////////////////////////
//
//     Module           : log2
//     Full Module Name : Log2 calculation block
//     Architecture     : rtl
//     Block Name       : 
//     Project          : 
//     Designer(s)      : Aedan Coffey aedan.coffey@ceva-dsp.com
//     Current Version  : $Revision: 1.2 $
//     Current Date     : $Date: 2010-02-03 20:39:26+02 $
//     Simulator        : 
//     Synthesiser      : 
//     Coding Standard  : 
//
//     Description      : Calculates the log to the base 2 of the input value.
//                        Result is approximate. Algorithm is based on the document
//                        "Beamforming Report" study, version 1.0. Dated 2014-11-06.
//
///////////////////////////////////////////////////////////////////////////
//
//            Copyright (c) Ceva Inc.
//
// This code is confidential and proprietary product of Ceva.
// Any unauthorized use, reproduction or transfer of this 
// code is strictly prohibited.
//
//////////////////////////////////////////////////////////////////////////

`default_nettype none

module log2 (
  input  wire [12:0] log2in,    // Input value, unsigned.
  output wire [10:0] result16    // Log2 of the input value, multiplied by 16.
  );  
  
  reg [3:0] FSB;         // Position of first set bit that is set in the input vector.
  reg [4:0] FiveBits;    // The next five bits below the FSB, not including it.
  reg [4:0] Fractional;  // The fractional part of the result.
  integer   i;           // Loop counter.

  //
  // First calculate the position of the most significant '1' in the input vector.
  //
  always @(*) // Version using for loop.
    begin
      FSB = 4'd0;
      for (i=12; i>0; i=i-1)
        begin
          if ((log2in[i] == 1'b1) && (FSB == 4'd0))
            begin
              FSB = i;
            end
        end
    end

  //
  // Extract the five bits directly below the FSB. This can be coded with the default case
  // handling all cases from 5 to 63 but then it gets lint errors.
  //
  always @(*)
    begin
      case (FSB)
        4'd0  :  FiveBits = 5'b00000;
        4'd1  :  FiveBits = {log2in[0],4'b0000};
        4'd2  :  FiveBits = {log2in[1:0],3'b000};
        4'd3  :  FiveBits = {log2in[2:0],2'b00};
        4'd4  :  FiveBits = {log2in[3:0],1'b0};
        4'd5  :  FiveBits = log2in[4:0];
        4'd6  :  FiveBits = log2in[5:1];
        4'd7  :  FiveBits = log2in[6:2];
        4'd8  :  FiveBits = log2in[7:3];
        4'd9  :  FiveBits = log2in[8:4];
        4'd10 :  FiveBits = log2in[9:5];
        4'd11 :  FiveBits = log2in[10:6];
        default :FiveBits = log2in[11:7]; // 4'd12
      endcase
    end

          
  //
  // Look up the five bits in a look up table to make the fractional
  // part of the result.
  //
  always @(*)
    begin
      case (FiveBits)
        5'd00 : Fractional = 5'd0;
        5'd01 : Fractional = 5'd1;
        5'd02 : Fractional = 5'd1;
        5'd03 : Fractional = 5'd2;
        5'd04 : Fractional = 5'd3;
        5'd05 : Fractional = 5'd3;
        5'd06 : Fractional = 5'd4;
        5'd07 : Fractional = 5'd5;        
        5'd08 : Fractional = 5'd5;
        5'd09 : Fractional = 5'd6;
        5'd10 : Fractional = 5'd6;
        5'd11 : Fractional = 5'd7;
        5'd12 : Fractional = 5'd7;
        5'd13 : Fractional = 5'd8;
        5'd14 : Fractional = 5'd8;
        5'd15 : Fractional = 5'd9;        
        5'd16 : Fractional = 5'd9;
        5'd17 : Fractional = 5'd10;
        5'd18 : Fractional = 5'd10;
        5'd19 : Fractional = 5'd11;
        5'd20 : Fractional = 5'd11;
        5'd21 : Fractional = 5'd12;
        5'd22 : Fractional = 5'd12;
        5'd23 : Fractional = 5'd13;        
        5'd24 : Fractional = 5'd13;
        5'd25 : Fractional = 5'd13;
        5'd26 : Fractional = 5'd14;
        5'd27 : Fractional = 5'd14;
        5'd28 : Fractional = 5'd15;
        5'd29 : Fractional = 5'd15;
        5'd30 : Fractional = 5'd15;
        5'd31 : Fractional = 5'd16;
      endcase
    end
  
  //
  // Add both parts to make the final result.
  // result = 16*log2(log2in).
  //
  assign result16 = {3'd0,FSB,4'd0} + {6'b0,Fractional};  
  
endmodule                      
