//////////////////////////////////////////////////////////////////////////
//
//     Module           : snr
//     Full Module Name : SNR calculation block
//     Architecture     : rtl
//     Block Name       : 
//     Project          : 
//     Designer(s)      : Aedan Coffey aedan.coffey@ceva-dsp.com
//     Current Version  : $Revision: 1.2 $
//     Current Date     : $Date: 2010-02-03 20:39:26+02 $
//     Simulator        : 
//     Synthesiser      : 
//     Coding Standard  : 
//
//     Description      : Calculates the Signal to Noise ratio based on two
//                      : input signals.
//
///////////////////////////////////////////////////////////////////////////
//
//            Copyright (c) Ceva Inc.
//
// This code is confidential and proprietary product of Ceva.
// Any unauthorized use, reproduction or transfer of this 
// code is strictly prohibited.
//
//////////////////////////////////////////////////////////////////////////

`default_nettype none

module snr (
    input wire         BFRModemClk,     // 120 MHz clock.
    input wire         nBFRModemRst,    // Async reset, active low.
    input wire         tctlAdvance,     // Asserted when the pipeline is to move on a tick.
    input  wire [12:0] lambdaLambda,    // Unsigned 13 bit number.
    input  wire [3:0]  lambdaMscale,    // Unsigned 4 bit number. lambdaMscale=scale-11.
    output reg  [7:0]  snrNsnr          // Unsigned 8 bit number.
    );
    
    wire [10:0] log2lambda16;           // 16*log2(lambdaLambda)
    wire [10:0] l_qdB3;                 // log2lambda16*3.
    reg  [9:0]  l_qdB;                  // l_qdB3 divided by 16 and rounded.
    wire        snrAdvance;
    wire [10:0] snrNsnrX2;

`ifdef RW_TXRX_2X2

    //
    // Delay lambdaLambda by one tick to ease FPGA timing.
    //

    reg  [12:0] dLambdaLambda;      // Delayed version of lambdaLambda.
    reg         dTctlAdvance;       // Delayed version of tctlAdvance;

    always @(posedge BFRModemClk or negedge nBFRModemRst)
      begin
        if (!nBFRModemRst)
          begin
            dLambdaLambda <= 13'd0;
            dTctlAdvance <= 1'b0;
          end
        else
          begin
            dTctlAdvance <= tctlAdvance;
            if (tctlAdvance)
              begin
                dLambdaLambda <= lambdaLambda;
              end
          end
      end
    assign snrAdvance = dTctlAdvance;

`else
    wire  [12:0] dLambdaLambda;
    assign dLambdaLambda = lambdaLambda;
    assign snrAdvance    = tctlAdvance;
`endif
    
    
    //
    // Instantiate the log2 function.
    //    
    log2 i_log2(
        .log2in    (dLambdaLambda),
        .result16  (log2lambda16));

    //
    // Take the output of the log, multiply it by 3 and divide by 4. This should match l_qdB in the MatLab model.
    //
    assign l_qdB3 = (log2lambda16<<1) + log2lambda16;    // Multiply by three.
    //
    always @(*)        // Divide by 4 and round. Force l_qdB to zero if lambdaLambda = 0
      begin
        if (dLambdaLambda == 13'd0)
          begin
            l_qdB = 10'd0;
          end
        else
          begin
            l_qdB = {1'b0,l_qdB3[10:2]}+{9'b0,l_qdB3[1]};
          end
      end


  //
  // Add in the scaling factor as per the MatLab code.
  // Note that lambdaMscale is being used instead of scale, where lambdaMscale is defined to be
  // scale-11, thus saving one bit on scale and a DFF or two.
  // snrNsnr = (l_qdB + 12*lambdaMscale)/2;
  // snrNsnr is equal to p_hdB in the Matlab code.
  //
  assign snrNsnrX2 = {1'b0,l_qdB} + {4'd0,lambdaMscale,3'b000} + {5'd0,lambdaMscale,2'b00};

  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
      if (!nBFRModemRst)
        begin
          snrNsnr <= 8'b0;
        end
      else
        begin
          if (snrAdvance)
            begin
              snrNsnr <= snrNsnrX2[8:1]; // divide by 2 with floor
            end
        end
    end

endmodule
