//
// Top level of the svd block. Not in its final version yet.
//

`default_nettype none

module svdtop # (
`ifdef TEST
  `ifdef RW_NX_CHBW804020 // 80 MHz
    parameter HAWIDTH = 10,
    `ifdef RW_TXRX_2X2
    parameter AWIDTH = 10,
    parameter SNR_BASE_ADDR = 10'd625 // address of the first Snr entry in the RAM.
    `else
    parameter AWIDTH = 9,
    parameter SNR_BASE_ADDR = 10'd375 // address of the first Snr entry in the RAM.
    `endif
  `else
    `ifdef RW_NX_CHBW4020 // 40 MHz
    parameter HAWIDTH = 9,
    `ifdef RW_TXRX_2X2
    parameter AWIDTH = 9,
    parameter SNR_BASE_ADDR = 10'd305 // address of the first Snr entry in the RAM.
    `else
    parameter AWIDTH = 8,
    parameter SNR_BASE_ADDR = 8'd183 // address of the first Snr entry in the RAM.
    `endif
    `else // 20 MHz
    parameter HAWIDTH = 8,
    `ifdef RW_TXRX_2X2
    parameter AWIDTH = 8,
    parameter SNR_BASE_ADDR = 10'd160 // address of the first Snr entry in the RAM.
    `else
    parameter AWIDTH = 7,
    parameter SNR_BASE_ADDR = 8'd96 // address of the first Snr entry in the RAM.
    `endif
    `endif
  `endif // RW_NX_CHBW804020
`else // ac only
  parameter HAWIDTH = 8,
//`ifdef RW_MUMIMO_RX_EN
  `ifdef RW_TXRX_2X2
    parameter AWIDTH = 10,
    parameter SNR_BASE_ADDR = 10'd585
  `else
    parameter AWIDTH = 9,
    parameter SNR_BASE_ADDR = 9'd351
  `endif // RW_TXRX_2X2
//`else
//  parameter AWIDTH = 8,
//  parameter SNR_BASE_ADDR = 9'd117
//`endif // RW_MUMIMO_RX_EN  
`endif
  ) (
  ///////////////////////////////////////////////
  //$port_g Modem Clock and reset
  ///////////////////////////////////////////////
  input  wire              BFRModemClk,       // BFR Modem Clock
  input  wire              BFRModemGClk,      // BFR Modem Gated Clock
  input  wire              nBFRModemRst,      // Active Low Reset

  output wire              BFRModemGClkEn,    // Clock enable for the Modem.
  
  ///////////////////////////////////////////////
  //$port_g Configuration.
  ///////////////////////////////////////////////
  input  wire              cfgHe,                // Indicates an HE packet
  input  wire        [5:0] cfgHeRUStartIndex,    // RU start index for HE
  input  wire        [5:0] cfgHeRUEndIndex,      // RU end index for HE
  input  wire        [1:0] cfgNg,                // Subcarriers grouping factor, (0,1,2)
  input  wire        [1:0] cfgChBw,              // Channel bandwidth, 0=20MHz, 1=40MHz, 2=80 MHz.  
  input  wire              cfgFeedbackType,      // Configuration parameter, 0=SU mode 1-MU mode. 
  input  wire signed [7:0] cfgSigmadB,           // Congiguration constant, sig_dB in Matlab code.
  input  wire signed [9:0] cfgSnrScaling,        // Configuration constant, reg_snr_scaling in Matlab code.
  input  wire              cfgCodebook,          // Configuration constant, 0 or 1.
  input  wire        [1:0] cfgNr,                // Configuration constant. Value = Matlab value -1.
  input  wire        [1:0] cfgNc,                // Configuration constant. Value = Matlab value -1.
  input  wire        [3:0] cfgRxAntSel,
  
  ///////////////////////////////////////////////
  //$port_g H memory interface
  ///////////////////////////////////////////////
  output wire               HMemActive,           // Indicates when the H memory is read
//`ifdef RW_MUMIMO_RX_EN
  `ifdef RW_TXRX_2X2
     input wire [207:0]     HMemRData,            // RW_MUMIMO_RX_EN and RW_TXRX_2X2 
  `else
     input wire [103:0]     HMemRData,            // RW_MUMIMO_RX_EN and not RW_TXRX_2X2
  `endif // RW_TXRX_2X2
//`else
//     input wire [103:0]     HMemRData,            // not RW_MUMIMO_RX_EN (and RW_TXRX_2X2)
//`endif // RW_MUMIMO_RX_EN  
  input  wire               HMemRDataVld,
  output wire [HAWIDTH-1:0] HMemRAddr,
  output wire               HMemREn,            

  ///////////////////////////////////////////////
  //$port_g Ram interface
  ///////////////////////////////////////////////
  output wire [AWIDTH-1:0] RamAddress,
  output wire              RamWe,
  output wire [31:0]       RamWData,
  output wire              RamRe,
  input  wire [31:0]       RamRData,

  ///////////////////////////////////////////////
  //$port_g MAC interface
  ///////////////////////////////////////////////
  input  wire              BFRStart,             // Control input, a rising edge indicates a request to start.
  output wire              SVDDone,
  
  input  wire              MacDataOutReady,
  output wire [31:0]       MacDataOut,
  output wire [ 3:0]       MacDataOutValid,

  ///////////////////////////////////////////////
  //$port_g diagnostic port
  ///////////////////////////////////////////////
  output wire [15:0]       Dbg0
    
  );  

`ifdef RW_TXRX_2X2
  localparam PIPE_DEPTH   = 10;
`else
  localparam PIPE_DEPTH   = 6;
`endif
  
  wire                   HMemDone;
  wire [7:0]             HMemNs;

  wire [11:0]            hermM11;
  wire [3:0]             hermMScale;
  wire [3:0]             lambdaMscale;

`ifdef RW_TXRX_2X2
  wire [11:0]            hermM22;
  wire signed [12:0]     hermMRe12;
  wire signed [12:0]     hermMIm12;
  
  wire [23:0]            eigenDelta2Data;      // Delta2 in MatLab model.
  wire                   eigenDelta2Valid;
  wire [12:0]            eigenBtildeData;      // btilde in MatLab model.
  wire [3:0]             eigenMscale;
  
  wire [11:0]            sqrDelta;
  wire [12:0]            sqrBtildeData; 
  wire [3:0]             sqrMscale;

  wire [12:0]            lambdaLambda2;
`endif
  
  wire [12:0]            lambdaLambda1;
           
  wire  [7:0]            snrNsnr0;
  wire  [7:0]            snrNsnr1;
  wire                   snrNsnrActive;
  wire                   snrNsnrLast;
  
  wire [11:0]            phiPhi11;
  wire [11:0]            phiPsi21;
//`ifdef RW_MUMIMO_RX_EN       
  wire [11:0]            phiPhi21;
  wire [11:0]            phiPhi31;
  wire [11:0]            phiPsi31;
  wire [11:0]            phiPsi41;
`ifdef RW_TXRX_2X2
  wire [11:0]            phiPhi22;
  wire [11:0]            phiPhi32;
  wire [11:0]            phiPsi32;
  wire [11:0]            phiPsi42;
`endif
//`endif    
  
  wire [31:0]            packerWord;
  wire                   packerValid;
  wire  [5:0]            packerLastNum;
  wire                   packerLast;

  wire signed  [7:0]     accuAvgSnr0;          // avp_qdB_re in MatLab model.
  wire signed  [7:0]     accuAvgSnr1;          // avp_qdB_re in MatLab model.
  wire         [8:0]     accuSubAvg0;
  wire         [8:0]     accuSubAvg1;
  wire         [2:0]     tctlBitsPerPsi;       // Calculated constant, bits per psi value.
  wire         [3:0]     tctlBitsPerPhi;       // Calculated constant, bits per phi value.
  wire         [3:0]     tctl12MinusBitsPerPhi;
  wire         [3:0]     tctl11MinusBitsPerPhi;
  wire signed [10:0]     tctlSnrFactor;
 
  
  wire                   HMemIfRdDataVld;
  
  wire        [12:0]     HMemIFGroupedRe11;
  wire        [12:0]     HMemIFGroupedIm11;
  wire        [12:0]     HMemIFGroupedRe12;
  wire        [12:0]     HMemIFGroupedIm12;
//`ifdef RW_MUMIMO_RX_EN
  wire        [12:0]     HMemIFGroupedRe13;
  wire        [12:0]     HMemIFGroupedIm13;
  wire        [12:0]     HMemIFGroupedRe14;
  wire        [12:0]     HMemIFGroupedIm14;
//`endif // RW_MUMIMO_RX_EN
`ifdef RW_TXRX_2X2
  wire        [12:0]     HMemIFGroupedRe21;
  wire        [12:0]     HMemIFGroupedIm21;
  wire        [12:0]     HMemIFGroupedRe22;
  wire        [12:0]     HMemIFGroupedIm22;
//`ifdef RW_MUMIMO_RX_EN
  wire        [12:0]     HMemIFGroupedRe23;
  wire        [12:0]     HMemIFGroupedIm23;
  wire        [12:0]     HMemIFGroupedRe24;
  wire        [12:0]     HMemIFGroupedIm24;
//`endif // RW_MUMIMO_RX_EN
`endif // RW_TXRX_2X2

  wire                   tctlAdvance;         // Asserted to tell the pipeline to advance.
  wire [PIPE_DEPTH-1:0]  tctlActiveStages;    // Bitmap to show which pipeline stages are active.
  wire                   tctlStart;           // Asserted to tell everything to start.
  wire                   tctlStop;            // Asserted to tell everything to stop.
  wire                   ReportRead;          // Asserted for one tick when the report has been read.
  wire                   ReportWritten;       // All report written by SVD
  
  //
  // Internal versions of the cfg* configuration parameters. These are registered in the SVD block at the beginning of each computation
  // cycle in case software changes them while a computation is in progress.
  //
  wire              tctlCfgHe;
  wire        [5:0] tctlHeRUStartIndex;       // RU start index for HE
  wire        [5:0] tctlHeRUEndIndex;         // RU end index for HE
  wire        [1:0] tctlCfgNg;                // Subcarriers grouping factor, (1,2,4)
  wire        [1:0] tctlCfgChBw;              // Channel bandwidth, 20, 40, 80 MHz.  
  wire              tctlCfgFeedbackType;      // Configuration parameter, 0=SU mode 1-MU mode. 
  wire              tctlCfgCodebook;          // Configuration constant, value = codebook value.
  wire        [1:0] tctlCfgNr;                // Configuration constant. Value = Matlab value -1.
  wire        [1:0] tctlCfgNc;                // Configuration constant. Value = Matlab value -1.
  wire        [3:0] tctlCfgRxAntSel;    

  assign Dbg0 = {tctlAdvance,          // 1
                 // MAC interface
                 BFRStart,             // 1
                 SVDDone,              // 1
                 MacDataOutReady,      // 1
                 MacDataOutValid,      // 4
                 // Configuration
                 tctlCfgFeedbackType,  // 1
                 tctlCfgCodebook,      // 1
                 tctlCfgNg,            // 2
                 tctlCfgNr,            // 2
                 tctlCfgNc};           // 2
  //
  // Instantiate the input memory interface.
  //
  hmemif # (
    .HAWIDTH (HAWIDTH)
  ) ihmemif (
     ///////////////////////////////////////////////
     //$port_g Modem clock inputs
     ///////////////////////////////////////////////
     .BFRModemClk(BFRModemGClk),
     .nBFRModemRst(nBFRModemRst),
            
     ///////////////////////////////////////////////
     //$port_g Inputs 
     ///////////////////////////////////////////////
     .cfgNg(tctlCfgNg),
     .cfgChBw(tctlCfgChBw),
     .cfgNr(tctlCfgNr),
     .cfgHe(tctlCfgHe),
     .cfgHeRUStartIndex(tctlHeRUStartIndex),
     .cfgHeRUEndIndex(tctlHeRUEndIndex),

     .HMemRdData(HMemRData),
     .HMemRDataVld(HMemRDataVld),
     .tctlAdvance(tctlAdvance),
     .tctlStart(tctlStart),
     .tctlStop(tctlStop),
     .HMemActive(HMemActive),
                   
     ///////////////////////////////////////////////
     //$port_g Outputs
     ///////////////////////////////////////////////
        
     .HMemRAddr(HMemRAddr),
     .HMemREn(HMemREn),            
     .HMemDone(HMemDone),
     .HMemNs(HMemNs),
     .HMemIfRdDataVld(HMemIfRdDataVld), 
//`ifdef RW_MUMIMO_RX_EN
     .HMemIFRdDataRe14(HMemIFGroupedRe14),
     .HMemIFRdDataIm14(HMemIFGroupedIm14),
     .HMemIFRdDataRe13(HMemIFGroupedRe13),
     .HMemIFRdDataIm13(HMemIFGroupedIm13),
`ifdef RW_TXRX_2X2
     .HMemIFRdDataRe24(HMemIFGroupedRe24),
     .HMemIFRdDataIm24(HMemIFGroupedIm24),
     .HMemIFRdDataRe23(HMemIFGroupedRe23),
     .HMemIFRdDataIm23(HMemIFGroupedIm23),
`endif // RW_TXRX_2X2
//`endif // RW_MUMIMO_RX_EN
`ifdef RW_TXRX_2X2
     .HMemIFRdDataRe22(HMemIFGroupedRe22),
     .HMemIFRdDataIm22(HMemIFGroupedIm22),
     .HMemIFRdDataRe21(HMemIFGroupedRe21),
     .HMemIFRdDataIm21(HMemIFGroupedIm21),
`endif // RW_TXRX_2X2
     .HMemIFRdDataRe12(HMemIFGroupedRe12),
     .HMemIFRdDataIm12(HMemIFGroupedIm12),
     .HMemIFRdDataRe11(HMemIFGroupedRe11),    
     .HMemIFRdDataIm11(HMemIFGroupedIm11)     
      );


  //
  // The Hermitian block.
  //
  hermitianBlock ihermitianBlock (
    ///////////////////////////////////////////////
    //$port_g Modem clock inputs
    ///////////////////////////////////////////////
    .BFRModemClk(BFRModemGClk),
    .nBFRModemRst(nBFRModemRst),

    ///////////////////////////////////////////////
    //$port_g H memory input
    ///////////////////////////////////////////////
    .HMemIFRdDataRe11(HMemIFGroupedRe11),    
    .HMemIFRdDataIm11(HMemIFGroupedIm11),     
    .HMemIFRdDataRe12(HMemIFGroupedRe12),
    .HMemIFRdDataIm12(HMemIFGroupedIm12),
`ifdef RW_TXRX_2X2
    .HMemIFRdDataRe21(HMemIFGroupedRe21),
    .HMemIFRdDataIm21(HMemIFGroupedIm21),
    .HMemIFRdDataRe22(HMemIFGroupedRe22),
    .HMemIFRdDataIm22(HMemIFGroupedIm22),
`endif //  RW_TXRX_2X2
//`ifdef RW_MUMIMO_RX_EN
    .HMemIFRdDataRe13(HMemIFGroupedRe13),
    .HMemIFRdDataIm13(HMemIFGroupedIm13),
    .HMemIFRdDataRe14(HMemIFGroupedRe14),
    .HMemIFRdDataIm14(HMemIFGroupedIm14),
`ifdef RW_TXRX_2X2
    .HMemIFRdDataRe23(HMemIFGroupedRe23),
    .HMemIFRdDataIm23(HMemIFGroupedIm23),
    .HMemIFRdDataRe24(HMemIFGroupedRe24),
    .HMemIFRdDataIm24(HMemIFGroupedIm24),
`endif //  RW_TXRX_2X2
//`endif // RW_MUMIMO_RX_EN
    .HMemRdDataVld(HMemIfRdDataVld),
    .HNrSel(tctlCfgNr),  

    ///////////////////////////////////////////////
    //$port_g Scaled hermitian matrix output.
    ///////////////////////////////////////////////
    .M11(hermM11),
`ifdef RW_TXRX_2X2
    .M22(hermM22),
    .MRe12(hermMRe12),
    .MIm12(hermMIm12),
`endif //  RW_TXRX_2X2
    .MScale(hermMScale)
  );
  

`ifdef RW_TXRX_2X2
  //
  // The Eigenvalues block (NRx=2)
  //
  eigen ieigen (
    ///////////////////////////////////////////////
    //$port_g Modem clock inputs
    ///////////////////////////////////////////////
    .BFRModemClk(BFRModemGClk),
    .nBFRModemRst(nBFRModemRst),

    ///////////////////////////////////////////////
    //$port_g Inputs from  hermitian matrix.
    ///////////////////////////////////////////////
    .hM11(hermM11),
    .hM22(hermM22),
    .hMRe12(hermMRe12),
    .hMIm12(hermMIm12),

    ///////////////////////////////////////////////
    //$port_g Outputs
    ///////////////////////////////////////////////
    .eigenDelta2Data(eigenDelta2Data),
    .eigenBtildeData(eigenBtildeData),
    ///////////////////////////////////////////////
    //$port_g Scale
    ///////////////////////////////////////////////
    .hMVld(tctlAdvance),
    .hMScale(hermMScale),
    .eigenDelta2Valid(eigenDelta2Valid),
    .eigenMscale(eigenMscale)
  );


  //
  // The sqr block (NRx=2)
  // Also delays Btilde and Mscale to keep them in sync with the
  // data they correspond to.
  //
  sqr  isqr (
    .BFRModemClk(BFRModemGClk),
    .nBFRModemRst(nBFRModemRst),
    .eigenDelta2Data(eigenDelta2Data), 
    .eigenBtildeData(eigenBtildeData),          
    .sqrDelta(sqrDelta),              
    .sqrBtildeData(sqrBtildeData),              
    .eigenDelta2Valid(eigenDelta2Valid),
    .eigenMscale(eigenMscale),       
    .sqrMscale(sqrMscale)        
    );
`endif //  RW_TXRX_2X2


  //
  // Lambda calculation.
  // 
`ifdef RW_TXRX_2X2
  assign lambdaLambda1 = sqrBtildeData + {1'b0,sqrDelta};
  assign lambdaLambda2 = sqrBtildeData - {1'b0,sqrDelta};
  assign lambdaMscale  = sqrMscale;
`else
  // 1 tctl tick before NRx=2 case
  assign lambdaLambda1 = {1'b0,hermM11};
  assign lambdaMscale  = hermMScale;
`endif    


  //
  // Lambda outputs are numbered 1 and 2, NSNR are numbered 0 and 1
  // name the snr blocks after NSNR convention, 0 and 1.
  //    
  snr isnr0 (
    .BFRModemClk        (BFRModemGClk),            // 120 MHz clock.
    .nBFRModemRst       (nBFRModemRst),            // Async reset, active low.
    .tctlAdvance        (tctlAdvance),             // Asserted when the pipeline is to move on a stage.
    .lambdaLambda       (lambdaLambda1),
    .lambdaMscale       (lambdaMscale),
    .snrNsnr            (snrNsnr0)
    );
  
`ifdef RW_TXRX_2X2
  snr isnr1 (
    .BFRModemClk        (BFRModemGClk),            // 120 MHz clock.
    .nBFRModemRst       (nBFRModemRst),            // Async reset, active low.
    .tctlAdvance        (tctlAdvance),             // Asserted when the pipeline is to move on a stage.
    .lambdaLambda       (lambdaLambda2),
    .lambdaMscale       (lambdaMscale),
    .snrNsnr            (snrNsnr1)
    );
  // SNR and lambda latencies are not the same if NRx=1 or 2
  assign snrNsnrActive = tctlActiveStages[2];
  assign snrNsnrLast   = ~tctlActiveStages[1];
`else
  // SNR and lambda latencies are not the same if NRx=1 or 2
  assign snrNsnrActive = tctlActiveStages[1];
  assign snrNsnrLast   = ~tctlActiveStages[0];
  assign snrNsnr1      = 8'd0;
`endif    
    
  //
  // Two accu blocks, named 0 and 1 as per diagram in the BFR report.
  //
  accu iaccu0 (  
    .BFRModemClk        (BFRModemGClk),         // 120 MHz clock.
    .nBFRModemRst       (nBFRModemRst),         // Async reset, active low.
    .cfgSnrFactor       (tctlSnrFactor),
    .tctlStart          (tctlStart),            // Asserted to start a computation sequence.
    .tctlStop           (tctlStop),             // Asserted to initialise everything or stop a computation.
    .snrNsnr            (snrNsnr0),             // Unsigned 8 bit number, SNR.
    .snrNsnrActive      (snrNsnrActive),        // Show when SNR is active.
    .tctlAdvance        (tctlAdvance),          // Asserted when the pipeline is to move on a stage.
    .tctlNs             (HMemNs),               // Ns constant, updated at end of hmem read
    .tctlNsValid        (HMemDone),
    .accuSubAvg         (accuSubAvg0),          // The amount that needs to be subtracted from the values in the Report RAM.
    .accuAvgSnr         (accuAvgSnr0)           // Unsigned 9 bit number, average SNR.
    );

`ifdef RW_TXRX_2X2
  accu iaccu1 (  
    .BFRModemClk        (BFRModemGClk),         // 120 MHz clock.
    .nBFRModemRst       (nBFRModemRst),         // Async reset, active low.
    .cfgSnrFactor       (tctlSnrFactor),
    .tctlStart          (tctlStart),            // Asserted to start a computation sequence.
    .tctlStop           (tctlStop),             // Asserted to initialise everything or stop a computation.
    .snrNsnr            (snrNsnr1),             // Unsigned 8 bit number, SNR.
    .snrNsnrActive      (snrNsnrActive),        // Show when SNR is active.
    .tctlAdvance        (tctlAdvance),          // Asserted when the pipeline is to move on a stage.
    .tctlNs             (HMemNs),               // Ns constant, calculated in tctl.
    .tctlNsValid        (HMemDone),
    .accuSubAvg         (accuSubAvg1),          // The amount that needs to be subtracted from the values in the Report RAM.
    .accuAvgSnr         (accuAvgSnr1)           // Unsigned 9 bit number, average SNR.
    );
`else
  assign accuSubAvg1 =  9'd0;
  assign accuAvgSnr1 =  8'd0;
`endif    

  //
  // Instantiate the phi calculation block.
  //
  phi iphi (
    ///////////////////////////////////////////////
    //$port_g Clock and reset
    ///////////////////////////////////////////////
    .BFRModemClk        (BFRModemGClk),            // 120 MHz clock.
    .nBFRModemRst       (nBFRModemRst),            // Async reset, active low.
    ///////////////////////////////////////////////
    //$port_g configuration and control inputs.
    ///////////////////////////////////////////////
    .cfgNr              (tctlCfgNr),
    .tctlAdvance        (tctlAdvance),
`ifdef RW_TXRX_2X2
    ///////////////////////////////////////////////
    //$port_g Hermatian inputs.
    ///////////////////////////////////////////////
    .hermM11            (hermM11),
    .hermMRe12          (hermMRe12),
    .hermMIm12          (hermMIm12),
    ///////////////////////////////////////////////
    //$port_g lambda inputs.
    ///////////////////////////////////////////////
    .lambdaLambda1       (lambdaLambda1),
    .lambdaLambda2       (lambdaLambda2),
`endif  
    ///////////////////////////////////////////////
    //$port_g H vector inputs.
    ///////////////////////////////////////////////
    .HMemIFGroupedRe11        (HMemIFGroupedRe11),
    .HMemIFGroupedIm11        (HMemIFGroupedIm11),
    .HMemIFGroupedRe12        (HMemIFGroupedRe12),
    .HMemIFGroupedIm12        (HMemIFGroupedIm12),
//`ifdef RW_MUMIMO_RX_EN
    .HMemIFGroupedRe13        (HMemIFGroupedRe13),
    .HMemIFGroupedIm13        (HMemIFGroupedIm13),
    .HMemIFGroupedRe14        (HMemIFGroupedRe14),
    .HMemIFGroupedIm14        (HMemIFGroupedIm14),
//`endif  
`ifdef RW_TXRX_2X2
    .HMemIFGroupedRe21        (HMemIFGroupedRe21),
    .HMemIFGroupedIm21        (HMemIFGroupedIm21),
    .HMemIFGroupedRe22        (HMemIFGroupedRe22),
    .HMemIFGroupedIm22        (HMemIFGroupedIm22),
//`ifdef RW_MUMIMO_RX_EN
    .HMemIFGroupedRe23        (HMemIFGroupedRe23),
    .HMemIFGroupedIm23        (HMemIFGroupedIm23),
    .HMemIFGroupedRe24        (HMemIFGroupedRe24),
    .HMemIFGroupedIm24        (HMemIFGroupedIm24),
//`endif  
`endif  
    ///////////////////////////////////////////////
    //$port_g outouts
    ///////////////////////////////////////////////
//`ifdef RW_MUMIMO_RX_EN
    .phiPhi21(phiPhi21),
    .phiPhi31(phiPhi31),
    .phiPsi31(phiPsi31),
    .phiPsi41(phiPsi41),
`ifdef RW_TXRX_2X2
    .phiPsi32(phiPsi32),
    .phiPsi42(phiPsi42),
    .phiPhi22(phiPhi22),
    .phiPhi32(phiPhi32),
`endif  
//`endif  
    .phiPhi11(phiPhi11),
    .phiPsi21(phiPsi21)
  );
   
  //
  // Instantiate the angle packer.
  //
  packer # (
    .PIPE_DEPTH(PIPE_DEPTH)
  ) ipacker (  
    ///////////////////////////////////////////////
    //$port_g Clock and reset
    ///////////////////////////////////////////////
    .BFRModemClk        (BFRModemGClk),            // 120 MHz clock.
    .nBFRModemRst       (nBFRModemRst),            // Async reset, active low.

    ///////////////////////////////////////////////
    //$port_g Control
    ///////////////////////////////////////////////
    .tctlStart(tctlStart),
    .tctlStop(tctlStop),

    ///////////////////////////////////////////////
    //$port_g Configuration.
    ///////////////////////////////////////////////
    .cfgFeedbackType(tctlCfgFeedbackType),   // Configuration parameter, 0=SU mode 1-MU mode. 
    .cfgCodebook(tctlCfgCodebook),           // Configuration constant, value = codebook value.
    .cfgNr(tctlCfgNr),
    .cfgNc(tctlCfgNc),
    
    ///////////////////////////////////////////////
    //$port_g inputs
    ///////////////////////////////////////////////
    .phiPhi11(phiPhi11),
    .phiPsi21(phiPsi21),
//`ifdef RW_MUMIMO_RX_EN  
    .phiPhi21(phiPhi21),
    .phiPhi31(phiPhi31),
    .phiPsi31(phiPsi31),
    .phiPsi41(phiPsi41),
`ifdef RW_TXRX_2X2
    .phiPsi32(phiPsi32),
    .phiPsi42(phiPsi42),
    .phiPhi22(phiPhi22),
    .phiPhi32(phiPhi32),
`endif
//`endif    
    .tctlActiveStages(tctlActiveStages),
    .tctlAdvance(tctlAdvance),
    .tctlBitsPerPsi(tctlBitsPerPsi),              // Calculated constant, bits per psi value.
    .tctlBitsPerPhi(tctlBitsPerPhi),              // Calculated constant, bits per phi value.    
    .tctl12MinusBitsPerPhi(tctl12MinusBitsPerPhi),    
    .tctl11MinusBitsPerPhi(tctl11MinusBitsPerPhi),    

    ///////////////////////////////////////////////
    //$port_g packer outputs
    ///////////////////////////////////////////////
    .packerWord(packerWord),
    .packerValid(packerValid),
    .packerLastNum(packerLastNum),
    .packerLast(packerLast)
   );
   
   //
   // Instantiate the MAC interface block.
   //
   macif # (
     .SNR_BASE_ADDR(SNR_BASE_ADDR),
     .AWIDTH(AWIDTH)
   ) imacif (
     ///////////////////////////////////////////////
     //$port_g Modem Clock and reset
     ///////////////////////////////////////////////
     .BFRModemClk(BFRModemClk),                     // BFR Modem Clock
     .BFRModemGClk(BFRModemGClk),                   // BFR Modem Gated Clock
     .nBFRModemRst(nBFRModemRst),                   // Active Low Reset
     
     ///////////////////////////////////////////////
     //$port_g Configuration signals.
     ///////////////////////////////////////////////
     .cfgChBw(tctlCfgChBw),
     .cfgNg(tctlCfgNg),
     .cfgNc(tctlCfgNc),
     .cfgFeedbackType(tctlCfgFeedbackType),
     .cfgHe(tctlCfgHe),

     ///////////////////////////////////////////////
     //$port_g Overall control signals.
     ///////////////////////////////////////////////
     .tctlStart(tctlStart),                         // Start processing, send the angles.
     .tctlStop(tctlStop),                           // Stop everything in SVD and report write operation.
     .tctlAdvance(tctlAdvance),
     .SVDDone(SVDDone),                             // Asserted when the SVD phase has completed.
     .ReportWritten(ReportWritten),                 // All report written by SVD
     .ReportRead(ReportRead),

     ///////////////////////////////////////////////
     //$port_g Data and data valid signals.
     ///////////////////////////////////////////////
     .packerWord(packerWord),                       // Angles, packed in bytes.
     .packerValid(packerValid),                     // Asserted when packerWord is valid.     
     .packerLast(packerLast),
     .packerLastNum(packerLastNum),
     .snrNsnr0(snrNsnr0),
     .snrNsnr1(snrNsnr1),
     .snrNsnrActive(snrNsnrActive),
     .snrNsnrLast(snrNsnrLast),
     .accuSubAvg0(accuSubAvg0),
     .accuSubAvg1(accuSubAvg1),
     .accuAvgSnr0(accuAvgSnr0),
     .accuAvgSnr1(accuAvgSnr1),
    
     ///////////////////////////////////////////////
     //$port_g Mac interface
     ///////////////////////////////////////////////
     .MacDataOut(MacDataOut),
     .MacDataOutValid(MacDataOutValid),
     .MacDataOutReady(MacDataOutReady),

     ///////////////////////////////////////////////
     //$port_g The Ram interface.
     ///////////////////////////////////////////////
     .RamAddress(RamAddress),
     .RamWe(RamWe),
     .RamWData(RamWData),
     .RamRe(RamRe),
     .RamRData(RamRData)
    );
  
  //
  // Instantiate the timing control block. This block manages the pipeline timing.
  //  
  tctl # (
    .AWIDTH(AWIDTH),
    .PIPE_DEPTH(PIPE_DEPTH)
  ) itctl (
     ///////////////////////////////////////////////
     //$port_g Modem Clock and reset
     ///////////////////////////////////////////////
     .BFRModemClk(BFRModemClk),                    // BFR Modem Clock
     .BFRModemGClk(BFRModemGClk),                  // BFR Modem Gated Clock
     .nBFRModemRst(nBFRModemRst),                  // Active Low Reset
     .BFRModemGClkEn(BFRModemGClkEn),              // Clock enable for the BFRModemClk

     ///////////////////////////////////////////////
     //$port_g Incoming Configuration
     ///////////////////////////////////////////////
     .cfgHe(cfgHe),                                // Indicates an HE report
     .cfgHeRUStartIndex(cfgHeRUStartIndex),
     .cfgHeRUEndIndex(cfgHeRUEndIndex),
     .cfgNg(cfgNg),                                // Subcarriers grouping factor, (1,2,4)
     .cfgChBw(cfgChBw),                            // Channel bandwidth, 0=20 MHz, 1=40 MHz, 2=80 MHz.  
     .cfgFeedbackType(cfgFeedbackType),            // Configuration parameter, 0=SU mode 1-MU mode.
     .cfgSigmadB(cfgSigmadB),                      // Congiguration constant, sig_dB in Matlab code.
     .cfgSnrScaling(cfgSnrScaling),                // Configuration constant, reg_snr_scaling in Matlab code.     
     .cfgCodebook(cfgCodebook),                    // Configuration constant, value = codebook value. 
     .cfgNr(cfgNr),                                // Configuration constant, value = Matlab Nr -1.
     .cfgNc(cfgNc),                                // Configuration constant, value = Matlab Nr -1.
     .cfgRxAntSel(cfgRxAntSel),  

     ///////////////////////////////////////////////
     //$port_g Registered Configuration
     ///////////////////////////////////////////////
     .tctlCfgHe(tctlCfgHe),
     .tctlHeRUStartIndex(tctlHeRUStartIndex),
     .tctlHeRUEndIndex(tctlHeRUEndIndex),
     .tctlCfgNg(tctlCfgNg),                        // Subcarriers grouping factor, (1,2,4)
     .tctlCfgChBw(tctlCfgChBw),                    // Channel bandwidth, 0=20 MHz, 1=40 MHz, 2=80 MHz.  
     .tctlCfgFeedbackType(tctlCfgFeedbackType),    // Configuration parameter, 0=SU mode 1-MU mode.
     .tctlCfgCodebook(tctlCfgCodebook),            // Configuration constant, value = codebook value. 
     .tctlCfgNr(tctlCfgNr),                        // Configuration constant, value = Matlab Nr -1.
     .tctlCfgNc(tctlCfgNc),                        // Configuration constant, value = Matlab Nr -1.
     .tctlCfgRxAntSel(tctlCfgRxAntSel),  

     ///////////////////////////////////////////////
     //$port_g Derived configuration signals, static for the duration of a calculation.
     ///////////////////////////////////////////////
     .tctlBitsPerPsi(tctlBitsPerPsi),              // Calculated constant, bits per psi value.
     .tctlBitsPerPhi(tctlBitsPerPhi),              // Calculated constant, bits per phi value.
     .tctl12MinusBitsPerPhi(tctl12MinusBitsPerPhi),    
     .tctl11MinusBitsPerPhi(tctl11MinusBitsPerPhi),    
     .tctlSnrFactor(tctlSnrFactor),     

     ///////////////////////////////////////////////
     //$port_g Overall control signals.
     ///////////////////////////////////////////////
     .BFRStart(BFRStart),                           // Start generating a BFR when a rising edge is detected on this.
     .tctlStart(tctlStart),                         // Start SVD processing when this is asserted.
     .tctlStop(tctlStop),                           // Stop SVD processing and initialise everything.
     .HMemDone(HMemDone),                           // End processing a while after this is deasserted.
     .HMemActive(HMemActive),
     .SVDDone(SVDDone),                             // Indicates that the SVD processing has been done.
     .ReportRead(ReportRead),                       // Indicates that all the report has been read.
     .ReportWritten(ReportWritten),                 // All report written by SVD
     
     ///////////////////////////////////////////////
     // Timing control signals.
     ///////////////////////////////////////////////
     .tctlActiveStages(tctlActiveStages),          // Bitmap to show which pipeline stages are active.
     .tctlAdvance(tctlAdvance)                     // Advance the pipeline by one step.
     );
    
endmodule
