//
// Timing control block. This block controls all timing and the data progression through the pipeline.
//

`default_nettype none

module tctl  # (
  parameter AWIDTH     = 10, // RAM address width, used to size some constants
  parameter PIPE_DEPTH = 10  // Used to know how long to run for once the memory interface has finished reading.
  ) (
  ///////////////////////////////////////////////
  //$port_g Modem Clock and reset
  ///////////////////////////////////////////////
  input  wire                 BFRModemClk,          // BFR Modem Clock
  input  wire                 BFRModemGClk,         // BFR Modem Gated Clock
  input  wire                 nBFRModemRst,         // Active Low Reset

  output reg                  BFRModemGClkEn,       // Clock enable for the BFRModemClk.
  
  ///////////////////////////////////////////////
  //$port_g Incoming configuration signals
  ///////////////////////////////////////////////
  input  wire                 cfgHe,                // Indicates an HE packet
  input  wire [5:0]           cfgHeRUStartIndex,    // RU start index for HE
  input  wire [5:0]           cfgHeRUEndIndex,      // RU end index for HE
  input  wire [1:0]           cfgNg,                // Subcarriers grouping factor, (1,2,4)
  input  wire [1:0]           cfgChBw,              // Channel bandwidth, 20, 40, 80 MHz.  
  input  wire                 cfgFeedbackType,      // Report feedback type, 0=SU mode 1-MU mode. 
  input  wire [7:0]           cfgSigmadB,           // sig_dB in Matlab code. 
  input  wire [9:0]           cfgSnrScaling,        // reg_snr_scaling in Matlab code.
  input  wire                 cfgCodebook,          // codebook value.
  input  wire [1:0]           cfgNr,                // Matlab Nr value -1.
  input  wire [1:0]           cfgNc,                // Matlab Nc value -1.
  input  wire [3:0]           cfgRxAntSel,

  ///////////////////////////////////////////////
  //$port_g Latched configuration signals, static for the duration of a calculation.
  ///////////////////////////////////////////////
  output reg                  tctlCfgHe,
  output reg  [5:0]           tctlHeRUStartIndex,     // RU start index for HE
  output reg  [5:0]           tctlHeRUEndIndex,       // RU end index for HE
  output reg  [1:0]           tctlCfgNg,               // Subcarriers grouping factor, (1,2,4)
  output reg  [1:0]           tctlCfgChBw,             // Channel bandwidth, 20, 40, 80 MHz.  
  output reg                  tctlCfgFeedbackType,     // Report feedback type, 0=SU mode 1-MU mode. 
  output reg                  tctlCfgCodebook,         // codebook value.
  output reg  [1:0]           tctlCfgNr,               // Matlab Nr value -1.
  output reg  [1:0]           tctlCfgNc,               // Matlab Nc value -1.
  output reg  [3:0]           tctlCfgRxAntSel,

  ///////////////////////////////////////////////
  //$port_g Derived configuration signals, static for the duration of a calculation.
  ///////////////////////////////////////////////
  output reg  [2:0]           tctlBitsPerPsi,          // Calculated constant, bits per psi value.
  output reg  [3:0]           tctlBitsPerPhi,          // Calculated constant, bits per phi value.
  output reg  [3:0]           tctl12MinusBitsPerPhi,
  output reg  [3:0]           tctl11MinusBitsPerPhi,
  output reg [10:0]           tctlSnrFactor,           // Used in SNR computation


  ///////////////////////////////////////////////
  //$port_g Overall control signals.
  ///////////////////////////////////////////////
  input  wire                 BFRStart,             // Start or restart processing when a rising edge on this signal is detected.
  output reg                  SVDDone,              // Indicates that the SVD processing is complete.
  output reg                  tctlStart,            // Start of SVD operation, after setup phase.
  output wire                 tctlStop,             // Abort processing, initialise everything.
  input  wire                 ReportWritten,        // Indicates that all the report has been stored in the memory
  input  wire                 ReportRead,           // Indicates that all the report has been transferred out.
  input  wire                 HMemDone,             // Indicates when reading of the H memory is over.
  output reg                  HMemActive,           // Indicates when reading of the H memory is on-going.

  ///////////////////////////////////////////////
  // Timing control signals.
  ///////////////////////////////////////////////
  output reg [PIPE_DEPTH-1:0] tctlActiveStages,     // Bitmap showing which pipeline stages are currently active.               
  output reg                  tctlAdvance           // Advance the pipeline by one step.
  );
  
  localparam CEVA_TICKSPERPIPELINESTAGE = 4'd10;   // No of ticks per pipeline stage, normally 10, can be increased for debug.
  localparam CEVA_INITIALCOUNTERVALUE   = 4'd10;   // Number of ticks from tctlStart to first tctlAdvance pulse.

  //
  // State definitions for the Mac clock domain state machine.
  //                       
  localparam IDLE_ST         = 4'd0;
  localparam WAIT1_ST        = 4'd1;
  localparam SVD_RUN_ST      = 4'd2;
  localparam SEND_REPORT_ST  = 4'd3;
  localparam SVD_STOP_ST     = 4'd4;
  localparam STOP_READ_ST    = 4'd5;
  localparam WAIT2_ST        = 4'd6;
  localparam WAIT3_ST        = 4'd7;
  localparam WAIT_ACK_ST     = 4'd8;
  localparam WAIT4_ST        = 4'd9;
  
   
  localparam GR1 = 2'd0, GR2 = 2'd1, GR4 = 2'd2, GR_INVALID = 2'd3;
  localparam BW20 = 2'd0, BW40 = 2'd1, BW80 = 2'd2;
  localparam NC1 = 2'd0, NC2 = 2'd1; // Nc index = Nc-1
  localparam NR_UNSUPPORTED = 2'd0, NR2 = 2'd1, NR3 = 2'd2, NR4 = 2'd3; // Nr index = Nr-1
  localparam CB0 = 1'b0, CB1 = 1'b1;
  localparam FBACK_SU = 1'd0, FBACK_MU = 1'd1;
  localparam FORMAT_NONHE = 1'b0, FORMAT_HE = 1'b1;

 
  reg [3:0]                 tctlState; 
  reg                       dBFRStart;
  reg                       dSvdActive;
  reg [3:0]                 TicksCounter;
  reg                       SvdActive; 
  wire                      BFRStartRise;
  wire                      SvdActiveFall;
  reg                       SVDStop;

  // Internal vatiable cfg* are derived from static input parameters cfg*.
  // Their computation is a synthesis false path
  // They are registered in tctl* outputs at BFRStart
  reg  [2:0]               cfgBitsPerPsi;          // Calculated constant, bits per psi value.
  reg  [3:0]               cfgBitsPerPhi;          // Calculated constant, bits per phi value.
  wire [3:0]               cfg12MinusBitsPerPhi;
  wire [3:0]               cfg11MinusBitsPerPhi;
  wire [10:0]              cfgSnrFactor;           // cfgSigmadB*4 - cfgSnrScaling 


  //
  // Make delayed versions of BFRStart and SvdActive to detect edges on them.
  //
  always @(posedge BFRModemClk or negedge nBFRModemRst)
  begin
    if (!nBFRModemRst) begin
      dBFRStart  <= 1'b0;
      dSvdActive <= 1'b0;
    end else begin
      dBFRStart  <= BFRStart;
      dSvdActive <= SvdActive; 
    end
  end
        
  assign BFRStartRise  = BFRStart && !dBFRStart;
  assign SvdActiveFall = !SvdActive && dSvdActive;
  assign tctlStop      = SVDStop || (BFRStartRise && (tctlState==SEND_REPORT_ST)); // No delay on ReadStop so that no unwanted data is sent out
  

  //
  // The main control state machine
  //
  
  always @(posedge BFRModemClk or negedge nBFRModemRst)
  begin
    if (!nBFRModemRst) begin
      tctlState <= IDLE_ST;

      BFRModemGClkEn <= 1'b0;
      HMemActive     <= 1'b0;
      SVDDone        <= 1'b0;
      tctlStart      <= 1'b0;
      SVDStop        <= 1'b0;
    end else begin

      tctlStart      <= 1'b0; // RTZ
      SVDStop        <= 1'b0; // RTZ

      case (tctlState)

        // Sit in Idle until BFRStart rising edge
        IDLE_ST: begin
          // Default state of control signals
          BFRModemGClkEn <= 1'b0;
          HMemActive     <= 1'b0;
          SVDDone        <= 1'b0;

          if (BFRStartRise==1'b1) begin
            tctlState <= WAIT1_ST;
            // Controls
            BFRModemGClkEn <= 1'b1; // Start Modem clock
            HMemActive     <= 1'b1;
          end
        end

        // Wait until the gated clocks are running.
        WAIT1_ST: begin
          tctlState      <= WAIT2_ST;
          BFRModemGClkEn <= 1'b1;
          HMemActive     <= 1'b1;
        end
        WAIT2_ST: begin
          tctlState      <= WAIT3_ST;
          BFRModemGClkEn <= 1'b1;
          HMemActive     <= 1'b1;
        end
        WAIT3_ST: begin
          tctlState      <= WAIT4_ST;
          BFRModemGClkEn <= 1'b1;
          HMemActive     <= 1'b1;
        end
        WAIT4_ST: begin
          tctlState <= SVD_RUN_ST;
          // Controls
          tctlStart      <= 1'b1;
          BFRModemGClkEn <= 1'b1;
          HMemActive     <= 1'b1;
        end

        // Run the SVD block, wait for a falling edge on SvdActive to show it's finished or a falling edge on
        // BFRStart to show that an abort has been requested.
        // Using SvdActive ensures the part running on GClk has reached idle state before the clock is gated
        SVD_RUN_ST: begin

          // Request to abort. Do not test edge, so that abort requests received in WAIT*_ST are also taken into account              
          if (BFRStart==1'b0) begin
            tctlState      <= SVD_STOP_ST;
            // Controls
            SVDStop        <= 1'b1;
            // HMemActive must go low before BFRModemGClkEn, so that hmemif goes to idle while clock is still running
            BFRModemGClkEn <= 1'b1;
            HMemActive     <= 1'b0;

          // Normal end of SVD operation
          end else if (SvdActiveFall==1'b1) begin
            tctlState      <= SEND_REPORT_ST;
            // Controls
            BFRModemGClkEn <= 1'b0;
            HMemActive     <= 1'b0; // should be already 0
            SVDDone        <= 1'b1;
          end
          
          // End of H memory operation (SVD operation will end once pipe is cleared)
          if (HMemDone==1'b1)
            HMemActive  <= 1'b0;
        end

        // Wait until the complete BFR has been sent, then go back to Idle.
        // If a new Start is detected on BFRStart then stop.
        SEND_REPORT_ST: begin
          if (BFRStart==1'b0)               // SVDDone acknowledge
            SVDDone        <= 1'b0;

          if (BFRStartRise==1'b1) begin     // Request to start another computation.
            tctlState      <= WAIT1_ST;
            BFRModemGClkEn <= 1'b1; // Start Modem clock
            HMemActive     <= 1'b1; // Start H memory clock and wait
          end else if (ReportRead==1'b1) begin
            if (BFRStart==1'b1)             // SVDDone not yet ackowledged (should not happen)
              tctlState <= WAIT_ACK_ST;
            else
              tctlState <= IDLE_ST;
          end
        end
        
        // Wait for SVDDone acknowledge
        WAIT_ACK_ST: begin
          if (BFRStart==1'b0) begin         // SVDDone acknowledge
            tctlState <= IDLE_ST;
            SVDDone   <= 1'b0;
          end
        end

        // Stop the SVD computation, wait until SvdActive is deasserted.
        SVD_STOP_ST: begin
          if (SvdActive==1'b0) begin
            tctlState <= IDLE_ST;
            // Controls
            BFRModemGClkEn <= 1'b0;
            HMemActive     <= 1'b0;
          end
        end

        // The default case, should never happen.
        default: begin
          tctlState <= IDLE_ST;
        end

      endcase
    end
  end    
      

  assign cfgSnrFactor = {cfgSigmadB[7],cfgSigmadB,2'b00} + {cfgSnrScaling[9],cfgSnrScaling};

  //
  // Register the configuration parameters at the start of each computation in case they change while the
  // calculations are being performed.
  //
  always @(posedge BFRModemClk or negedge nBFRModemRst)
    begin
      if (!nBFRModemRst)
        begin
          tctlCfgHe                <=  1'b0;
          tctlHeRUStartIndex       <=  6'd0;
          tctlHeRUEndIndex         <=  6'd0;
          tctlCfgNg                <=  2'b00;
          tctlCfgChBw              <=  2'b00;  
          tctlCfgFeedbackType      <=  1'b0; 
          tctlCfgCodebook          <=  1'b0;
          tctlCfgNr                <=  2'b00;
          tctlCfgNc                <=  2'b00; 
          tctlCfgRxAntSel          <=  4'd0;
          tctlBitsPerPsi           <=  3'd0;
          tctlBitsPerPhi           <=  4'd0;
          tctl12MinusBitsPerPhi    <=  4'd0;
          tctl11MinusBitsPerPhi    <=  4'd0;
          tctlSnrFactor            <= 11'd0;
        end
      else
        begin
          if (BFRStartRise)    // Rising edge of BFRStart.
            begin
              tctlCfgHe                <= cfgHe;
              tctlHeRUStartIndex       <= cfgHeRUStartIndex;
              tctlHeRUEndIndex         <= cfgHeRUEndIndex;
              tctlCfgNg                <= cfgNg;
              tctlCfgChBw              <= cfgChBw;  
              tctlCfgFeedbackType      <= cfgFeedbackType; 
              tctlCfgCodebook          <= cfgCodebook;
              tctlCfgNr                <= cfgNr;
              tctlCfgNc                <= cfgNc;
              tctlCfgRxAntSel          <= cfgRxAntSel;
              tctlBitsPerPsi           <= cfgBitsPerPsi;
              tctlBitsPerPhi           <= cfgBitsPerPhi;
              tctl12MinusBitsPerPhi    <= cfg12MinusBitsPerPhi;
              tctl11MinusBitsPerPhi    <= cfg11MinusBitsPerPhi;
              tctlSnrFactor            <= cfgSnrFactor;
            end
        end
    end    


  // --------------------------------- BFRModemGClk section --------------------------------------------------------------------

             
  //
  // Wait for a pulse on tctlStart, then produce timing pulses every timing period. The period is
  // defined in the parameter TICKSPERPIPELINESTAGE.
  // 
  always @(posedge BFRModemGClk or negedge nBFRModemRst)
   begin
     if (!nBFRModemRst)
       begin
         TicksCounter        <= CEVA_INITIALCOUNTERVALUE;
         SvdActive           <= 1'b0;
         tctlActiveStages    <= {PIPE_DEPTH{1'b0}};
       end
     else
       begin
         //
         // Stop everything if SVDStop detected.
         //
         if (SVDStop)
           begin
             TicksCounter        <= CEVA_INITIALCOUNTERVALUE;
             SvdActive           <= 1'b0;
             tctlActiveStages    <= {PIPE_DEPTH{1'b0}};
           end
         //
         // Assert SvdActive once we receive a signal to start and up to report written
         //
         else if (tctlStart)
           begin
             SvdActive         <= 1'b1;
             tctlActiveStages  <= {{(PIPE_DEPTH-1){1'b0}},1'b1};
           end
         else if (ReportWritten==1'd1)
           begin
             SvdActive <= 1'b0;
           end
         //
         // When we are active run the ticks counter. 
         //
         if ((tctlStart || SvdActive) && !SVDStop)
           begin
             if (TicksCounter != 4'd0)
               begin
                 TicksCounter <= TicksCounter - 4'd1;
               end
             else
               begin
                 TicksCounter     <= (CEVA_TICKSPERPIPELINESTAGE-4'd1);
                 tctlActiveStages <= {tctlActiveStages[PIPE_DEPTH-2:0],HMemActive};
               end
           end
         else
           begin
             TicksCounter <= CEVA_INITIALCOUNTERVALUE;
           end
       end
   end

  always @(posedge BFRModemGClk or negedge nBFRModemRst)
   begin
     if (!nBFRModemRst)
       begin
         tctlAdvance        <= 1'd0;
       end
     else
       begin
         tctlAdvance        <= 1'd0; // RTZ
         // Test on SvdActive ensures tctlAdvance goes back to 0 before the gated clock is stopped,
         // to have a correct start value for next SVD operation
         if ((TicksCounter == 4'd1) && (SvdActive==1'b1))
           tctlAdvance        <= 1'd1;
       end
   end


  assign cfg12MinusBitsPerPhi = 4'd12-cfgBitsPerPhi;
  assign cfg11MinusBitsPerPhi = 4'd11-cfgBitsPerPhi;


  // Calculate bpsi and bphi (cf 802.11-2016 Table 9-6), depend on cfgFeedbackType and cfgCodebook
  //     bpsi   = 2*BFParameter.Codebook + 3*BFParameter.Feedback + 2;
  //     bphi   = bpsi + 2;
  // Max value bpsi can thus have is 7, max for bphi is 9.
  always @(*)
    begin
      case ({cfgFeedbackType, cfgCodebook}) 
        {FBACK_SU,CB0}: begin                      
          cfgBitsPerPsi     = 3'd2;
          cfgBitsPerPhi     = 4'd4;
        end
        {FBACK_SU,CB1}: begin                      
          cfgBitsPerPsi     = 3'd4;
          cfgBitsPerPhi     = 4'd6;
        end
        {FBACK_MU,CB0}: begin                      
          cfgBitsPerPsi     = 3'd5;
          cfgBitsPerPhi     = 4'd7;
        end
        default: begin // {FBACK_MU,CB1}
          cfgBitsPerPsi     = 3'd7;
          cfgBitsPerPhi     = 4'd9;
        end
      endcase
    end    

`ifdef RW_SIMU_ON

  reg [25*8:0] StateStr;

  always@(*)
  begin
    case (tctlState)
      IDLE_ST             : StateStr = {"IDLE_ST"};
      WAIT1_ST            : StateStr = {"WAIT1_ST"};
      SVD_RUN_ST          : StateStr = {"SVD_RUN_ST"};
      SEND_REPORT_ST      : StateStr = {"SEND_REPORT_ST"};
      SVD_STOP_ST         : StateStr = {"SVD_STOP_ST"};
      STOP_READ_ST        : StateStr = {"STOP_READ_ST"};
      WAIT2_ST            : StateStr = {"WAIT2_ST"};
      WAIT3_ST            : StateStr = {"WAIT3_ST"};
      WAIT_ACK_ST         : StateStr = {"WAIT_ACK_ST"};
      WAIT4_ST            : StateStr = {"WAIT4_ST"};
      default             : StateStr = {"UNKNOWN"};
    endcase
  end

`endif
     
endmodule
