//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      : Top level of bfr_mpif module
// Simulation Notes : 
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////
`default_nettype none

module bfr_mpif ( 
  input  wire         rst_n,
  input  wire         mpif_clk,
  input  wire         svd_clk,  // SVD clk is aligned with mpif_clk and running at half mpif_clk frequency

  /*****************************************************************************
  * MAC BFMEE IF 
  *****************************************************************************/
  input  wire         mac_bfr_start,          // BFR computation started
  output reg          mac_bfr_done,           // Beamforming Report computation is done

  input  wire [1:0]   mac_bfr_ch_bw,          // Beamforming report Channel Width
  input  wire [1:0]   mac_bfr_grouping,       // Beamforming report Grouping
  input  wire         mac_bfr_codebook,       // Beamforming report Codebook Information
  input  wire         mac_bfr_feedback_type,  // Beamforming report Feedback Type
  input  wire [1:0]   mac_bfr_nr,             // Beamforming report Nr Index
  input  wire [1:0]   mac_bfr_nc,             // Beamforming report Nc Index
  input  wire         mac_bfr_he,             // Beamforming report Format
                                              // (0:VHT or 1:HE)
  input  wire [5:0]   mac_bfr_he_rustartidx,  // Beamforming report RU Start index for HE
  input  wire [5:0]   mac_bfr_he_ruendidx,    // Beamforming report RU end index for HE
  
  output reg  [7:0]   mac_bfr_data,           // Beamforming report byte
  output reg          mac_bfr_data_valid,     // Beamforming report byte valid          
  input  wire         mac_bfr_data_request,   // Beamforming report byte request

  /*****************************************************************************
  * MODEM BFMEE IF 
  *****************************************************************************/
  output reg          bfmee_bfr_start,          // BFR computation started
  input  wire         bfmee_bfr_done,           // Beamforming Report computation is done

  output reg  [1:0]   bfmee_bfr_ch_bw,          // Beamforming report Channel Width
  output reg  [1:0]   bfmee_bfr_grouping,       // Beamforming report Grouping
  output reg          bfmee_bfr_codebook,       // Beamforming report Codebook Information
  output reg          bfmee_bfr_feedback_type,  // Beamforming report Feedback Type
  output reg  [1:0]   bfmee_bfr_nr,             // Beamforming report Nr Index
  output reg  [1:0]   bfmee_bfr_nc,             // Beamforming report Nc Index
  output reg          bfmee_bfr_he,             // Beamforming report HE mode
  output reg  [5:0]   bfmee_bfr_he_rustartidx,  // Beamforming report RU Start index for HE
  output reg  [5:0]   bfmee_bfr_he_ruendidx,    // Beamforming report RU end index for HE

  output reg          bfmee_ready,
  input  wire [31:0]  bfmee_data,
  input  wire [ 3:0]  bfmee_valid,

  output wire [15:0]  dbg1,
  output wire [15:0]  dbg2
);


/////////////////////////////////////////////////////////////////////////////
// Internal Wires declarations
//////////////////////////////////////////////////////////////////////////////
  reg          mac_bfr_start_d;
  wire [11:0]  n_mac_bfrfifo_rq_count;
  reg  [11:0]  mac_bfrfifo_rq_count;
  wire         mac_bfrfifo_wr_req;
  wire         mac_bfrfifo_rd_req;
  reg   [3:0]  mac_bfrfifo_count;             
  reg  [95:0]  mac_bfrfifo;
  reg  [95:0]  n_mac_bfrfifo_wrdata;
  wire         n_bfmee_ready;

//////////////////////////////////////////////////////////////////////////////
// Diag
//////////////////////////////////////////////////////////////////////////////
  assign dbg1 = {mac_bfrfifo_count,   // 15:12
                 mac_bfrfifo_rq_count // 11:0
                };
  assign dbg2 = {(bfmee_valid & {4{bfmee_ready}}), // 15:12
                 mac_bfrfifo_rd_req,    // 11
                 mac_bfrfifo_wr_req,    // 10
                 mac_bfr_data_valid,    // 9
                 mac_bfr_data_request,  // 8
                 mac_bfr_data           // 7:0
                 };
                 

//////////////////////////////////////////////////////////////////////////////
// Logic part - SVD clock
//////////////////////////////////////////////////////////////////////////////

  // Sample controls to SVD block on mac_bfr_start rising edge
  // Register ready output on svd_clk
  always @(posedge svd_clk or negedge rst_n)
  begin
    if (!rst_n) begin
      bfmee_bfr_start          <= 1'b0;
      bfmee_bfr_he             <= 1'b0;
      bfmee_bfr_he_rustartidx  <= 6'd0;
      bfmee_bfr_he_ruendidx    <= 6'd0;
      bfmee_bfr_grouping       <= 2'd0;
      bfmee_bfr_ch_bw          <= 2'd0;
      bfmee_bfr_feedback_type  <= 1'b0; 
      bfmee_bfr_codebook       <= 1'b0;
      bfmee_bfr_nr             <= 2'd0;
      bfmee_bfr_nc             <= 2'd0;
      
      bfmee_ready              <= 1'b0;
    end else begin
      // clocks are aligned: just forward control signals
      bfmee_bfr_start            <= mac_bfr_start;
      bfmee_ready                <= n_bfmee_ready;
      
      if (mac_bfr_start && !bfmee_bfr_start) begin // Rising edge of mac_bfr_start
        bfmee_bfr_he             <= mac_bfr_he;
        bfmee_bfr_he_rustartidx  <= mac_bfr_he_rustartidx;
        bfmee_bfr_he_ruendidx    <= mac_bfr_he_ruendidx;
        bfmee_bfr_grouping       <= mac_bfr_grouping;
        bfmee_bfr_ch_bw          <= mac_bfr_ch_bw;
        bfmee_bfr_feedback_type  <= mac_bfr_feedback_type;
        bfmee_bfr_codebook       <= mac_bfr_codebook;
        bfmee_bfr_nr             <= mac_bfr_nr;
        bfmee_bfr_nc             <= mac_bfr_nc;
      end
    end
  end    


//////////////////////////////////////////////////////////////////////////////
// Logic part - MPIF clock
//////////////////////////////////////////////////////////////////////////////

  // Keep track of the number of bytes requested by / provided to the MAC
  assign n_mac_bfrfifo_rq_count = mac_bfrfifo_rq_count          // Request count
                                + {11'd0, mac_bfr_data_request} // New byte requested
                                - {11'd0, mac_bfr_data_valid};  // Byte provided

  always @(posedge mpif_clk or negedge rst_n)
  begin
    if (!rst_n) begin
      mac_bfr_done         <= 1'b0;
      mac_bfr_start_d      <= 1'b0;
      mac_bfrfifo_rq_count <= 12'd0;
    end else begin
      mac_bfr_start_d      <= mac_bfr_start;
      // clocks are aligned: just forward control signals
      mac_bfr_done         <= bfmee_bfr_done;

      // counter for mac byte requests
      if (mac_bfr_start && !bfmee_bfr_start) begin // Rising edge of mac_bfr_start
        mac_bfrfifo_rq_count <= 12'd0;
      end  else begin
        mac_bfrfifo_rq_count <= n_mac_bfrfifo_rq_count;
      end
      
    end
  end    
  
  // bfmee FIFO with 12 entries: setting registering bfmee_ready on svd_clock can take up to 2 svd clock cycles,
  // i.e. 8 FIFO entries + 4 needed to buffer data to the MAC without delay.

  // Assert n_bfmee_ready when fifo count <= 4 bytes (at least 8 free entries, up to 2 SVD clock cycles)
  assign n_bfmee_ready      = !(|mac_bfrfifo_count[3:2]);
  // Write request  when valid and ready bfmee data. Toggle ensures slower bfmee signals are sampled only once 
  // on fast mpif_clk. Ready ensures that the FIFO has enough free entries.
  assign mac_bfrfifo_wr_req = (bfmee_valid!=4'd0) && bfmee_ready;
  // Read request as soon as data requested by the MAC is available in the FIFO.
  assign mac_bfrfifo_rd_req = (n_mac_bfrfifo_rq_count!=12'd0) && (mac_bfrfifo_count!=3'd0);
 

  // FIFO write data
  always @(*)
  begin
    n_mac_bfrfifo_wrdata = mac_bfrfifo;
    case (bfmee_valid)
      4'b0001 : n_mac_bfrfifo_wrdata[8*mac_bfrfifo_count +:  8] = bfmee_data[7:0];
      4'b0011 : n_mac_bfrfifo_wrdata[8*mac_bfrfifo_count +: 16] = bfmee_data[15:0];
      4'b0111 : n_mac_bfrfifo_wrdata[8*mac_bfrfifo_count +: 24] = bfmee_data[23:0];
      4'b1111 : n_mac_bfrfifo_wrdata[8*mac_bfrfifo_count +: 32] = bfmee_data;
      default : n_mac_bfrfifo_wrdata = mac_bfrfifo;
    endcase
  end

  
  // Register FIFO and output data
  always @(posedge mpif_clk, negedge rst_n)
  begin
    if(!rst_n) begin

      mac_bfr_data_valid <= 1'b0;
      mac_bfr_data       <= 8'b0;
      mac_bfrfifo        <= 96'b0;
      mac_bfrfifo_count  <= 4'b0;

    end else begin
    
      // Reset on mac_bfr_start rising edge
      if (mac_bfr_start && !bfmee_bfr_start) begin
        
        mac_bfr_data_valid <= 1'b0;
        mac_bfr_data       <= 8'b0;
        mac_bfrfifo        <= 96'b0;
        mac_bfrfifo_count  <= 4'd0;
        
      end else begin

        // Read and write requests
        case ({mac_bfrfifo_wr_req,mac_bfrfifo_rd_req})

          // Write, no read
          2'b10   : begin 
            mac_bfr_data_valid <= 1'b0;
            mac_bfrfifo        <= n_mac_bfrfifo_wrdata;
            mac_bfrfifo_count  <= mac_bfrfifo_count + {3'd0,bfmee_valid[0]}
                                                    + {3'd0,bfmee_valid[1]}
                                                    + {3'd0,bfmee_valid[2]}
                                                    + {3'd0,bfmee_valid[3]};
          end

          // Read, no write
          2'b01   : begin 
            mac_bfr_data_valid <= 1'b1;
            mac_bfr_data       <= mac_bfrfifo[7:0];
            mac_bfrfifo        <= {8'd0,mac_bfrfifo[95:8]};
            mac_bfrfifo_count  <= mac_bfrfifo_count - 4'd1;
          end

          // Read and write
          2'b11   : begin 
            mac_bfr_data_valid <= 1'b1;
            mac_bfr_data       <= n_mac_bfrfifo_wrdata[7:0];
            mac_bfrfifo        <= {8'd0,n_mac_bfrfifo_wrdata[95:8]};
            mac_bfrfifo_count  <= mac_bfrfifo_count + {3'd0,bfmee_valid[0]}
                                                    + {3'd0,bfmee_valid[1]}
                                                    + {3'd0,bfmee_valid[2]}
                                                    + {3'd0,bfmee_valid[3]}
                                                    - 4'd1;
          end

          // No read, no write
          default : begin
            mac_bfr_data_valid <=   1'b0;
          end
        endcase
      end
    end
      
  end


endmodule
                 
//////////////////////////////////////////////////////////////////////////////
// End of file
//////////////////////////////////////////////////////////////////////////////
