// This block allows the decoder's programmable registers to be replaced
// by look up tables.

module ldpcDecLu (       
  // Clock and reset
  input  wire          clk,
  input  wire          rst_n,
  // Controls
  input  wire          enable,
  output reg           lutEnable,
  // Packet parameters
  input  wire [2:0]    qamEnum,  // 0:4 is BPSK, QPSK, 16QAM, 64QAM, 256QAM
  input  wire [1:0]    sizeEnum, // zEnum
  input  wire [1:0]    rateEnum, // rEnum
  input  wire          ac,       // 0:11n 1:11ac
  input  wire [1:0]    bw,       // 0:3 is 20/40/80/160MHz
  input  wire [6:0]    mcs,      // 0-9 for 11ac, 0-72 for 11n
  input  wire [1:0]    nssLess1, // e.g. NSS=4 is specified as a 3.
  input  wire [13:0]   nCw,      // packetLen in LDPC blocks
  input  wire [1:0]    nSym,     // num OFDM symbols. clip this to 3.
  // Output parameters not depending on data rate, only on packet size and modulation
  output reg  [14:0]  targetLevel,
  output reg  [10:0]  parityThresh,
  // Address in config RAM to read packet configuration
  output reg  [10:0] CfgAddr
  );

reg              enableD;
reg  [1:0]     bucketIdx;   // selects a range of packet lengths
reg  [1:0]     nbucketIdx;
reg  [4:0]     dataRateIdx; // selects iteration settings according to datarates
reg  [4:0]     ndataRateIdx;

reg  [14:0]    ntargetLevel;
reg  [10:0]    nparityThresh;
reg  [ 10:0]    nCfgAddr;

// TARGET_LEVEL_DATA is to control scale-backs between iterations.
always @(*)
  case (qamEnum)
    3'd0: 
      case (sizeEnum)
        2'd0: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd5190;
            2'd1    : ntargetLevel = 15'd0;
            2'd2    : ntargetLevel = 15'd6170;
            default : ntargetLevel = 15'd0;
          endcase
        2'd1: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd5650;
            2'd1    : ntargetLevel = 15'd0;
            2'd2    : ntargetLevel = 15'd5870;
            default : ntargetLevel = 15'd0;
          endcase
        default: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd4860;
            2'd1    : ntargetLevel = 15'd0;
            2'd2    : ntargetLevel = 15'd3670;
            default : ntargetLevel = 15'd0;
          endcase
      endcase
    3'd1: 
      case (sizeEnum)
        2'd0: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd5190;
            2'd1    : ntargetLevel = 15'd0;
            2'd2    : ntargetLevel = 15'd6170;
            default : ntargetLevel = 15'd0;
          endcase
        2'd1: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd5650;
            2'd1    : ntargetLevel = 15'd0;
            2'd2    : ntargetLevel = 15'd5870;
            default : ntargetLevel = 15'd0;
          endcase
        default: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd4860;
            2'd1    : ntargetLevel = 15'd0;
            2'd2    : ntargetLevel = 15'd3670;
            default : ntargetLevel = 15'd0;
          endcase
      endcase
    3'd2: 
      case (sizeEnum)
        2'd0: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd3200;
            2'd1    : ntargetLevel = 15'd0;
            2'd2    : ntargetLevel = 15'd5200;
            default : ntargetLevel = 15'd0;
          endcase
        2'd1: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd4800;
            2'd1    : ntargetLevel = 15'd0;
            2'd2    : ntargetLevel = 15'd6600;
            default : ntargetLevel = 15'd0;
          endcase
        default: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd3800;
            2'd1    : ntargetLevel = 15'd0;
            2'd2    : ntargetLevel = 15'd5200;
            default : ntargetLevel = 15'd0;
          endcase
      endcase
    3'd3: 
      case (sizeEnum)
        2'd0: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd3200;
            2'd1    : ntargetLevel = 15'd6310;
            2'd2    : ntargetLevel = 15'd5870;
            default : ntargetLevel = 15'd3990;
          endcase
        2'd1: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd4800;
            2'd1    : ntargetLevel = 15'd1970;
            2'd2    : ntargetLevel = 15'd6280;
            default : ntargetLevel = 15'd5920;
          endcase
        default: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd3800;
            2'd1    : ntargetLevel = 15'd2690;
            2'd2    : ntargetLevel = 15'd3690;
            default : ntargetLevel = 15'd6700;
          endcase
      endcase
    default: 
      case (sizeEnum)
        2'd0: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd0;
            2'd1    : ntargetLevel = 15'd0;
            2'd2    : ntargetLevel = 15'd6200;
            default : ntargetLevel = 15'd5200;
          endcase
        2'd1: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd0;
            2'd1    : ntargetLevel = 15'd0;
            2'd2    : ntargetLevel = 15'd3630;
            default : ntargetLevel = 15'd4900;
          endcase
        default: 
          case (rateEnum)
            2'd0    : ntargetLevel = 15'd0;
            2'd1    : ntargetLevel = 15'd0;
            2'd2    : ntargetLevel = 15'd6400;
            default : ntargetLevel = 15'd6800;
          endcase
      endcase
  endcase

// Early termination control register. Used in conjunction with EARLY_TEST_ITERATIONS.
always @(*)
  if (sizeEnum == 2'd2) begin
    case (rateEnum)
      2'd0    : nparityThresh = 11'd100;
      2'd1    : nparityThresh = 11'd80;
      2'd2    : nparityThresh = 11'd45;
      default : nparityThresh = 11'd40;
    endcase
  end else begin
    nparityThresh = 11'd255;
  end

// Define the nCw that marks the start of the next NOM_ITERATIONS bucket.
always @(*) begin
  if (sizeEnum != 2'd2)
    nbucketIdx = 2'd0;
  else
    case (rateEnum)
      2'd0 : nbucketIdx = 
             (nSym < 2'd2) ? 2'd0 :
             (nCw < 14'd12) ? 2'd1 :
             (nCw < 14'd33) ? 2'd2 : 2'd3;
      2'd1 : nbucketIdx = 
             (nSym < 2'd2) ? 2'd0 :
             (nCw < 14'd9) ? 2'd1 :
             (nCw < 14'd25) ? 2'd2 : 2'd3;
      2'd2 : nbucketIdx = 
             (nSym < 2'd2) ? 2'd0 :
             (nCw < 14'd8) ? 2'd1 :
             (nCw < 14'd22) ? 2'd2 : 2'd3;
      default : nbucketIdx = 
             (nSym < 2'd2) ? 2'd0 :
             (nCw < 14'd7) ? 2'd1 :
             (nCw < 14'd20) ? 2'd2 : 2'd3;
    endcase
end

// Iteration control registers (stored in the configuration RAM) determine throughput and
// latency during and at the end of the packet. Decode Bandwidth and MCS into the datarate index
// used to address the RAM.
wire [1:0] nssMasked;
assign nssMasked = ac ? nssLess1 : 2'd0;


// Find the index into the iteration tables (according to bw/mcs/ac/nss).
always @(*) begin
    case ({rateEnum,bw,mcs,ac,nssMasked})
    // Coding Rate 1/2
`ifdef RW_TXRX_2X2
    {2'd0, 2'd2, 7'd3, 1'd1, 2'd1} : ndataRateIdx = 5'd10; // mcs_ac_bw80x2, 260 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd0, 2'd2, 7'd3, 1'd1, 2'd0} : ndataRateIdx = 5'd9; // mcs_ac_bw80x1, 130 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd0, 2'd2, 7'd1, 1'd1, 2'd1} : ndataRateIdx = 5'd9; // mcs_ac_bw80x2, 130 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd0, 2'd1, 7'd11, 1'd0, 2'd0} : ndataRateIdx = 5'd8; // mcs_n_bw40x2, 120 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd0, 2'd1, 7'd3, 1'd1, 2'd1} : ndataRateIdx = 5'd8; // mcs_ac_bw40x2, 120 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd0, 2'd2, 7'd1, 1'd1, 2'd0} : ndataRateIdx = 5'd7; // mcs_ac_bw80x1, 65 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd0, 2'd2, 7'd0, 1'd1, 2'd1} : ndataRateIdx = 5'd7; // mcs_ac_bw80x2, 65 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd0, 2'd1, 7'd3, 1'd0, 2'd0} : ndataRateIdx = 5'd6; // mcs_n_bw40x1, 60 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd0, 2'd1, 7'd9, 1'd0, 2'd0} : ndataRateIdx = 5'd6; // mcs_n_bw40x2, 60 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd0, 2'd1, 7'd3, 1'd1, 2'd0} : ndataRateIdx = 5'd6; // mcs_ac_bw40x1, 60 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd0, 2'd1, 7'd1, 1'd1, 2'd1} : ndataRateIdx = 5'd6; // mcs_ac_bw40x2, 60 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd0, 2'd0, 7'd11, 1'd0, 2'd0} : ndataRateIdx = 5'd5; // mcs_n_bw20x2, 58 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd0, 2'd0, 7'd3, 1'd1, 2'd1} : ndataRateIdx = 5'd5; // mcs_ac_bw20x2, 58 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd0, 2'd2, 7'd0, 1'd1, 2'd0} : ndataRateIdx = 5'd4; // mcs_ac_bw80x1, 33 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd0, 2'd1, 7'd1, 1'd0, 2'd0} : ndataRateIdx = 5'd3; // mcs_n_bw40x1, 30 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd0, 2'd1, 7'd8, 1'd0, 2'd0} : ndataRateIdx = 5'd3; // mcs_n_bw40x2, 30 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd0, 2'd1, 7'd1, 1'd1, 2'd0} : ndataRateIdx = 5'd3; // mcs_ac_bw40x1, 30 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd0, 2'd1, 7'd0, 1'd1, 2'd1} : ndataRateIdx = 5'd3; // mcs_ac_bw40x2, 30 Mbps
`endif 
    {2'd0, 2'd0, 7'd3, 1'd0, 2'd0} : ndataRateIdx = 5'd2; // mcs_n_bw20x1, 29 Mbps
`ifdef RW_TXRX_2X2
    {2'd0, 2'd0, 7'd9, 1'd0, 2'd0} : ndataRateIdx = 5'd2; // mcs_n_bw20x2, 29 Mbps
`endif 
    {2'd0, 2'd0, 7'd3, 1'd1, 2'd0} : ndataRateIdx = 5'd2; // mcs_ac_bw20x1, 29 Mbps
`ifdef RW_TXRX_2X2
    {2'd0, 2'd0, 7'd1, 1'd1, 2'd1} : ndataRateIdx = 5'd2; // mcs_ac_bw20x2, 29 Mbps
`endif 
    {2'd0, 2'd0, 7'd1, 1'd0, 2'd0} : ndataRateIdx = 5'd1; // mcs_n_bw20x1, 15 Mbps
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd0, 2'd1, 7'd0, 1'd0, 2'd0} : ndataRateIdx = 5'd1; // mcs_n_bw40x1, 15 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd0, 2'd0, 7'd8, 1'd0, 2'd0} : ndataRateIdx = 5'd1; // mcs_n_bw20x2, 15 Mbps
`endif 
    {2'd0, 2'd0, 7'd1, 1'd1, 2'd0} : ndataRateIdx = 5'd1; // mcs_ac_bw20x1, 15 Mbps
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd0, 2'd1, 7'd0, 1'd1, 2'd0} : ndataRateIdx = 5'd1; // mcs_ac_bw40x1, 15 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd0, 2'd0, 7'd0, 1'd1, 2'd1} : ndataRateIdx = 5'd1; // mcs_ac_bw20x2, 15 Mbps
`endif 
    {2'd0, 2'd0, 7'd0, 1'd0, 2'd0} : ndataRateIdx = 5'd0; // mcs_n_bw20x1, 8 Mbps
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd0, 2'd1, 7'd32, 1'd0, 2'd0} : ndataRateIdx = 5'd0; // mcs_n_bw40x1, 8 Mbps
`endif 
    {2'd0, 2'd0, 7'd0, 1'd1, 2'd0} : ndataRateIdx = 5'd0; // mcs_ac_bw20x1, 8 Mbps
    // Coding Rate 2/3
`ifdef RW_TXRX_2X2
    {2'd1, 2'd2, 7'd5, 1'd1, 2'd1} : ndataRateIdx = 5'd5; // mcs_ac_bw80x2, 520 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd1, 2'd2, 7'd5, 1'd1, 2'd0} : ndataRateIdx = 5'd4; // mcs_ac_bw80x1, 260 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd1, 2'd1, 7'd13, 1'd0, 2'd0} : ndataRateIdx = 5'd3; // mcs_n_bw40x2, 240 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd1, 2'd1, 7'd5, 1'd1, 2'd1} : ndataRateIdx = 5'd3; // mcs_ac_bw40x2, 240 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd1, 2'd1, 7'd5, 1'd0, 2'd0} : ndataRateIdx = 5'd2; // mcs_n_bw40x1, 120 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd1, 2'd1, 7'd5, 1'd1, 2'd0} : ndataRateIdx = 5'd2; // mcs_ac_bw40x1, 120 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd1, 2'd0, 7'd13, 1'd0, 2'd0} : ndataRateIdx = 5'd1; // mcs_n_bw20x2, 116 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd1, 2'd0, 7'd5, 1'd1, 2'd1} : ndataRateIdx = 5'd1; // mcs_ac_bw20x2, 116 Mbps
`endif 
    {2'd1, 2'd0, 7'd5, 1'd0, 2'd0} : ndataRateIdx = 5'd0; // mcs_n_bw20x1, 58 Mbps
    {2'd1, 2'd0, 7'd5, 1'd1, 2'd0} : ndataRateIdx = 5'd0; // mcs_ac_bw20x1, 58 Mbps
    // Coding Rate 3/4
`ifdef RW_TXRX_2X2
    {2'd2, 2'd2, 7'd8, 1'd1, 2'd1} : ndataRateIdx = 5'd17; // mcs_ac_bw80x2, 780 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd2, 7'd6, 1'd1, 2'd1} : ndataRateIdx = 5'd16; // mcs_ac_bw80x2, 585 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd2, 2'd2, 7'd8, 1'd1, 2'd0} : ndataRateIdx = 5'd15; // mcs_ac_bw80x1, 390 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd2, 7'd4, 1'd1, 2'd1} : ndataRateIdx = 5'd15; // mcs_ac_bw80x2, 390 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd1, 7'd8, 1'd1, 2'd1} : ndataRateIdx = 5'd14; // mcs_ac_bw40x2, 360 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd2, 2'd2, 7'd6, 1'd1, 2'd0} : ndataRateIdx = 5'd13; // mcs_ac_bw80x1, 293 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd1, 7'd14, 1'd0, 2'd0} : ndataRateIdx = 5'd12; // mcs_n_bw40x2, 270 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd1, 7'd6, 1'd1, 2'd1} : ndataRateIdx = 5'd12; // mcs_ac_bw40x2, 270 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd2, 2'd2, 7'd4, 1'd1, 2'd0} : ndataRateIdx = 5'd11; // mcs_ac_bw80x1, 195 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd2, 7'd2, 1'd1, 2'd1} : ndataRateIdx = 5'd11; // mcs_ac_bw80x2, 195 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd1, 7'd12, 1'd0, 2'd0} : ndataRateIdx = 5'd10; // mcs_n_bw40x2, 180 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd2, 2'd1, 7'd8, 1'd1, 2'd0} : ndataRateIdx = 5'd10; // mcs_ac_bw40x1, 180 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd1, 7'd4, 1'd1, 2'd1} : ndataRateIdx = 5'd10; // mcs_ac_bw40x2, 180 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd0, 7'd8, 1'd1, 2'd1} : ndataRateIdx = 5'd9; // mcs_ac_bw20x2, 174 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd2, 2'd1, 7'd6, 1'd0, 2'd0} : ndataRateIdx = 5'd8; // mcs_n_bw40x1, 135 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd2, 2'd1, 7'd6, 1'd1, 2'd0} : ndataRateIdx = 5'd8; // mcs_ac_bw40x1, 135 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd0, 7'd14, 1'd0, 2'd0} : ndataRateIdx = 5'd7; // mcs_n_bw20x2, 130 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd0, 7'd6, 1'd1, 2'd1} : ndataRateIdx = 5'd7; // mcs_ac_bw20x2, 130 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd2, 2'd2, 7'd2, 1'd1, 2'd0} : ndataRateIdx = 5'd6; // mcs_ac_bw80x1, 98 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd2, 2'd1, 7'd4, 1'd0, 2'd0} : ndataRateIdx = 5'd5; // mcs_n_bw40x1, 90 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd1, 7'd10, 1'd0, 2'd0} : ndataRateIdx = 5'd5; // mcs_n_bw40x2, 90 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd2, 2'd1, 7'd4, 1'd1, 2'd0} : ndataRateIdx = 5'd5; // mcs_ac_bw40x1, 90 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd1, 7'd2, 1'd1, 2'd1} : ndataRateIdx = 5'd5; // mcs_ac_bw40x2, 90 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd2, 2'd0, 7'd12, 1'd0, 2'd0} : ndataRateIdx = 5'd4; // mcs_n_bw20x2, 87 Mbps
`endif 
    {2'd2, 2'd0, 7'd8, 1'd1, 2'd0} : ndataRateIdx = 5'd4; // mcs_ac_bw20x1, 87 Mbps
`ifdef RW_TXRX_2X2
    {2'd2, 2'd0, 7'd4, 1'd1, 2'd1} : ndataRateIdx = 5'd4; // mcs_ac_bw20x2, 87 Mbps
`endif 
    {2'd2, 2'd0, 7'd6, 1'd0, 2'd0} : ndataRateIdx = 5'd3; // mcs_n_bw20x1, 65 Mbps
    {2'd2, 2'd0, 7'd6, 1'd1, 2'd0} : ndataRateIdx = 5'd3; // mcs_ac_bw20x1, 65 Mbps
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd2, 2'd1, 7'd2, 1'd0, 2'd0} : ndataRateIdx = 5'd2; // mcs_n_bw40x1, 45 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd2, 2'd1, 7'd2, 1'd1, 2'd0} : ndataRateIdx = 5'd2; // mcs_ac_bw40x1, 45 Mbps
`endif 
    {2'd2, 2'd0, 7'd4, 1'd0, 2'd0} : ndataRateIdx = 5'd1; // mcs_n_bw20x1, 44 Mbps
`ifdef RW_TXRX_2X2
    {2'd2, 2'd0, 7'd10, 1'd0, 2'd0} : ndataRateIdx = 5'd1; // mcs_n_bw20x2, 44 Mbps
`endif 
    {2'd2, 2'd0, 7'd4, 1'd1, 2'd0} : ndataRateIdx = 5'd1; // mcs_ac_bw20x1, 44 Mbps
`ifdef RW_TXRX_2X2
    {2'd2, 2'd0, 7'd2, 1'd1, 2'd1} : ndataRateIdx = 5'd1; // mcs_ac_bw20x2, 44 Mbps
`endif 
    {2'd2, 2'd0, 7'd2, 1'd0, 2'd0} : ndataRateIdx = 5'd0; // mcs_n_bw20x1, 22 Mbps
    {2'd2, 2'd0, 7'd2, 1'd1, 2'd0} : ndataRateIdx = 5'd0; // mcs_ac_bw20x1, 22 Mbps
    // Coding Rate 5/6
`ifdef RW_TXRX_2X2
    {2'd3, 2'd2, 7'd9, 1'd1, 2'd1} : ndataRateIdx = 5'd9; // mcs_ac_bw80x2, 867 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd3, 2'd2, 7'd7, 1'd1, 2'd1} : ndataRateIdx = 5'd8; // mcs_ac_bw80x2, 650 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd3, 2'd2, 7'd9, 1'd1, 2'd0} : ndataRateIdx = 5'd7; // mcs_ac_bw80x1, 434 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd3, 2'd1, 7'd9, 1'd1, 2'd1} : ndataRateIdx = 5'd6; // mcs_ac_bw40x2, 400 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd3, 2'd2, 7'd7, 1'd1, 2'd0} : ndataRateIdx = 5'd5; // mcs_ac_bw80x1, 325 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd3, 2'd1, 7'd15, 1'd0, 2'd0} : ndataRateIdx = 5'd4; // mcs_n_bw40x2, 300 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd3, 2'd1, 7'd7, 1'd1, 2'd1} : ndataRateIdx = 5'd4; // mcs_ac_bw40x2, 300 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd3, 2'd1, 7'd9, 1'd1, 2'd0} : ndataRateIdx = 5'd3; // mcs_ac_bw40x1, 200 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd3, 2'd1, 7'd7, 1'd0, 2'd0} : ndataRateIdx = 5'd2; // mcs_n_bw40x1, 150 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
    {2'd3, 2'd1, 7'd7, 1'd1, 2'd0} : ndataRateIdx = 5'd2; // mcs_ac_bw40x1, 150 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd3, 2'd0, 7'd15, 1'd0, 2'd0} : ndataRateIdx = 5'd1; // mcs_n_bw20x2, 145 Mbps
`endif 
`ifdef RW_TXRX_2X2
    {2'd3, 2'd0, 7'd7, 1'd1, 2'd1} : ndataRateIdx = 5'd1; // mcs_ac_bw20x2, 145 Mbps
`endif 
    {2'd3, 2'd0, 7'd7, 1'd0, 2'd0} : ndataRateIdx = 5'd0; // mcs_n_bw20x1, 73 Mbps
    {2'd3, 2'd0, 7'd7, 1'd1, 2'd0} : ndataRateIdx = 5'd0; // mcs_ac_bw20x1, 73 Mbps
    default: ndataRateIdx = 5'd0;
  endcase
end // always

// Create RAM address based on data rate index
always @(*) begin
    case (rateEnum)
      2'd0 : 
      case (dataRateIdx)
        default : begin // 8 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd0; // AddrIdx=1
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd1;
              2'd1    :   nCfgAddr = 11'd2;
              2'd2    :   nCfgAddr = 11'd3;
              default :   nCfgAddr = 11'd4;
            endcase
          end // if sizeEnum
        end
        5'd1 : begin // 15 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd5; // AddrIdx=2
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd6;
              2'd1    :   nCfgAddr = 11'd7;
              2'd2    :   nCfgAddr = 11'd8;
              default :   nCfgAddr = 11'd9;
            endcase
          end // if sizeEnum
        end
        5'd2 : begin // 29 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd10; // AddrIdx=3
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd11;
              2'd1    :   nCfgAddr = 11'd12;
              2'd2    :   nCfgAddr = 11'd13;
              default :   nCfgAddr = 11'd14;
            endcase
          end // if sizeEnum
        end
`ifdef RW_NX_DERIV_CHBW4020ONLY
        5'd3 : begin // 30 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd250; // AddrIdx=4
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd251;
              2'd1    :   nCfgAddr = 11'd252;
              2'd2    :   nCfgAddr = 11'd253;
              default :   nCfgAddr = 11'd254;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        5'd4 : begin // 33 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd670; // AddrIdx=5
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd671;
              2'd1    :   nCfgAddr = 11'd672;
              2'd2    :   nCfgAddr = 11'd673;
              default :   nCfgAddr = 11'd674;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd5 : begin // 58 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd900; // AddrIdx=6
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd901;
              2'd1    :   nCfgAddr = 11'd902;
              2'd2    :   nCfgAddr = 11'd903;
              default :   nCfgAddr = 11'd904;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
        5'd6 : begin // 60 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd255; // AddrIdx=7
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd256;
              2'd1    :   nCfgAddr = 11'd257;
              2'd2    :   nCfgAddr = 11'd258;
              default :   nCfgAddr = 11'd259;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        5'd7 : begin // 65 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd675; // AddrIdx=8
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd676;
              2'd1    :   nCfgAddr = 11'd677;
              2'd2    :   nCfgAddr = 11'd678;
              default :   nCfgAddr = 11'd679;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd8 : begin // 120 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd905; // AddrIdx=9
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd906;
              2'd1    :   nCfgAddr = 11'd907;
              2'd2    :   nCfgAddr = 11'd908;
              default :   nCfgAddr = 11'd909;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        5'd9 : begin // 130 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd680; // AddrIdx=10
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd681;
              2'd1    :   nCfgAddr = 11'd682;
              2'd2    :   nCfgAddr = 11'd683;
              default :   nCfgAddr = 11'd684;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd10 : begin // 260 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd910; // AddrIdx=11
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd911;
              2'd1    :   nCfgAddr = 11'd912;
              2'd2    :   nCfgAddr = 11'd913;
              default :   nCfgAddr = 11'd914;
            endcase
          end // if sizeEnum
        end
`endif 
      endcase // dataRateIdx
      2'd1 : 
      case (dataRateIdx)
        default : begin // 58 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd15; // AddrIdx=12
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd16;
              2'd1    :   nCfgAddr = 11'd17;
              2'd2    :   nCfgAddr = 11'd18;
              default :   nCfgAddr = 11'd19;
            endcase
          end // if sizeEnum
        end
`ifdef RW_TXRX_2X2
        5'd1 : begin // 116 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd915; // AddrIdx=13
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd916;
              2'd1    :   nCfgAddr = 11'd917;
              2'd2    :   nCfgAddr = 11'd918;
              default :   nCfgAddr = 11'd919;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
        5'd2 : begin // 120 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd260; // AddrIdx=14
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd261;
              2'd1    :   nCfgAddr = 11'd262;
              2'd2    :   nCfgAddr = 11'd263;
              default :   nCfgAddr = 11'd264;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd3 : begin // 240 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd920; // AddrIdx=15
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd921;
              2'd1    :   nCfgAddr = 11'd922;
              2'd2    :   nCfgAddr = 11'd923;
              default :   nCfgAddr = 11'd924;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        5'd4 : begin // 260 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd685; // AddrIdx=16
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd686;
              2'd1    :   nCfgAddr = 11'd687;
              2'd2    :   nCfgAddr = 11'd688;
              default :   nCfgAddr = 11'd689;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd5 : begin // 520 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd925; // AddrIdx=17
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd926;
              2'd1    :   nCfgAddr = 11'd927;
              2'd2    :   nCfgAddr = 11'd928;
              default :   nCfgAddr = 11'd929;
            endcase
          end // if sizeEnum
        end
`endif 
      endcase // dataRateIdx
      2'd2 : 
      case (dataRateIdx)
        default : begin // 22 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd20; // AddrIdx=18
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd21;
              2'd1    :   nCfgAddr = 11'd22;
              2'd2    :   nCfgAddr = 11'd23;
              default :   nCfgAddr = 11'd24;
            endcase
          end // if sizeEnum
        end
        5'd1 : begin // 44 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd25; // AddrIdx=19
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd26;
              2'd1    :   nCfgAddr = 11'd27;
              2'd2    :   nCfgAddr = 11'd28;
              default :   nCfgAddr = 11'd29;
            endcase
          end // if sizeEnum
        end
`ifdef RW_NX_DERIV_CHBW4020ONLY
        5'd2 : begin // 45 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd265; // AddrIdx=20
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd266;
              2'd1    :   nCfgAddr = 11'd267;
              2'd2    :   nCfgAddr = 11'd268;
              default :   nCfgAddr = 11'd269;
            endcase
          end // if sizeEnum
        end
`endif 
        5'd3 : begin // 65 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd30; // AddrIdx=21
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd31;
              2'd1    :   nCfgAddr = 11'd32;
              2'd2    :   nCfgAddr = 11'd33;
              default :   nCfgAddr = 11'd34;
            endcase
          end // if sizeEnum
        end
        5'd4 : begin // 87 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd35; // AddrIdx=22
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd36;
              2'd1    :   nCfgAddr = 11'd37;
              2'd2    :   nCfgAddr = 11'd38;
              default :   nCfgAddr = 11'd39;
            endcase
          end // if sizeEnum
        end
`ifdef RW_NX_DERIV_CHBW4020ONLY
        5'd5 : begin // 90 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd270; // AddrIdx=23
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd271;
              2'd1    :   nCfgAddr = 11'd272;
              2'd2    :   nCfgAddr = 11'd273;
              default :   nCfgAddr = 11'd274;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        5'd6 : begin // 98 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd690; // AddrIdx=24
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd691;
              2'd1    :   nCfgAddr = 11'd692;
              2'd2    :   nCfgAddr = 11'd693;
              default :   nCfgAddr = 11'd694;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd7 : begin // 130 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd930; // AddrIdx=25
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd931;
              2'd1    :   nCfgAddr = 11'd932;
              2'd2    :   nCfgAddr = 11'd933;
              default :   nCfgAddr = 11'd934;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
        5'd8 : begin // 135 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd275; // AddrIdx=26
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd276;
              2'd1    :   nCfgAddr = 11'd277;
              2'd2    :   nCfgAddr = 11'd278;
              default :   nCfgAddr = 11'd279;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd9 : begin // 174 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd935; // AddrIdx=27
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd936;
              2'd1    :   nCfgAddr = 11'd937;
              2'd2    :   nCfgAddr = 11'd938;
              default :   nCfgAddr = 11'd939;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
        5'd10 : begin // 180 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd280; // AddrIdx=28
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd281;
              2'd1    :   nCfgAddr = 11'd282;
              2'd2    :   nCfgAddr = 11'd283;
              default :   nCfgAddr = 11'd284;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        5'd11 : begin // 195 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd695; // AddrIdx=29
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd696;
              2'd1    :   nCfgAddr = 11'd697;
              2'd2    :   nCfgAddr = 11'd698;
              default :   nCfgAddr = 11'd699;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd12 : begin // 270 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd940; // AddrIdx=30
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd941;
              2'd1    :   nCfgAddr = 11'd942;
              2'd2    :   nCfgAddr = 11'd943;
              default :   nCfgAddr = 11'd944;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        5'd13 : begin // 293 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd700; // AddrIdx=31
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd701;
              2'd1    :   nCfgAddr = 11'd702;
              2'd2    :   nCfgAddr = 11'd703;
              default :   nCfgAddr = 11'd704;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd14 : begin // 360 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd945; // AddrIdx=32
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd946;
              2'd1    :   nCfgAddr = 11'd947;
              2'd2    :   nCfgAddr = 11'd948;
              default :   nCfgAddr = 11'd949;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        5'd15 : begin // 390 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd705; // AddrIdx=33
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd706;
              2'd1    :   nCfgAddr = 11'd707;
              2'd2    :   nCfgAddr = 11'd708;
              default :   nCfgAddr = 11'd709;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd16 : begin // 585 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd950; // AddrIdx=34
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd951;
              2'd1    :   nCfgAddr = 11'd952;
              2'd2    :   nCfgAddr = 11'd953;
              default :   nCfgAddr = 11'd954;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd17 : begin // 780 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd955; // AddrIdx=35
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd956;
              2'd1    :   nCfgAddr = 11'd957;
              2'd2    :   nCfgAddr = 11'd958;
              default :   nCfgAddr = 11'd959;
            endcase
          end // if sizeEnum
        end
`endif 
      endcase // dataRateIdx
      default : 
      case (dataRateIdx)
        default : begin // 73 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd40; // AddrIdx=36
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd41;
              2'd1    :   nCfgAddr = 11'd42;
              2'd2    :   nCfgAddr = 11'd43;
              default :   nCfgAddr = 11'd44;
            endcase
          end // if sizeEnum
        end
`ifdef RW_TXRX_2X2
        5'd1 : begin // 145 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd960; // AddrIdx=37
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd961;
              2'd1    :   nCfgAddr = 11'd962;
              2'd2    :   nCfgAddr = 11'd963;
              default :   nCfgAddr = 11'd964;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
        5'd2 : begin // 150 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd285; // AddrIdx=38
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd286;
              2'd1    :   nCfgAddr = 11'd287;
              2'd2    :   nCfgAddr = 11'd288;
              default :   nCfgAddr = 11'd289;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW4020ONLY
        5'd3 : begin // 200 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd290; // AddrIdx=39
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd291;
              2'd1    :   nCfgAddr = 11'd292;
              2'd2    :   nCfgAddr = 11'd293;
              default :   nCfgAddr = 11'd294;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd4 : begin // 300 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd965; // AddrIdx=40
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd966;
              2'd1    :   nCfgAddr = 11'd967;
              2'd2    :   nCfgAddr = 11'd968;
              default :   nCfgAddr = 11'd969;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        5'd5 : begin // 325 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd710; // AddrIdx=41
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd711;
              2'd1    :   nCfgAddr = 11'd712;
              2'd2    :   nCfgAddr = 11'd713;
              default :   nCfgAddr = 11'd714;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd6 : begin // 400 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd970; // AddrIdx=42
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd971;
              2'd1    :   nCfgAddr = 11'd972;
              2'd2    :   nCfgAddr = 11'd973;
              default :   nCfgAddr = 11'd974;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        5'd7 : begin // 434 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd715; // AddrIdx=43
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd716;
              2'd1    :   nCfgAddr = 11'd717;
              2'd2    :   nCfgAddr = 11'd718;
              default :   nCfgAddr = 11'd719;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd8 : begin // 650 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd975; // AddrIdx=44
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd976;
              2'd1    :   nCfgAddr = 11'd977;
              2'd2    :   nCfgAddr = 11'd978;
              default :   nCfgAddr = 11'd979;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_TXRX_2X2
        5'd9 : begin // 867 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd980; // AddrIdx=45
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd981;
              2'd1    :   nCfgAddr = 11'd982;
              2'd2    :   nCfgAddr = 11'd983;
              default :   nCfgAddr = 11'd984;
            endcase
          end // if sizeEnum
        end
`endif 
      endcase // dataRateIdx
    endcase
end // always

  always @(posedge clk, negedge rst_n)
    if(!rst_n) begin
      targetLevel         <= 15'd0;
      parityThresh        <= 11'd0;
      enableD             <= 1'd0;
      lutEnable           <= 1'd0;
      CfgAddr             <= 11'd0;
      dataRateIdx         <= 5'd0;
      bucketIdx           <= 2'd0;
    end else begin
      targetLevel         <= ntargetLevel;
      parityThresh        <= nparityThresh;
      enableD             <= enable;
      lutEnable           <= enableD;
      CfgAddr             <= nCfgAddr;
      dataRateIdx         <= ndataRateIdx;
      bucketIdx           <= nbucketIdx;
    end

endmodule
