// This block allows the decoder's programmable registers to be replaced
// by look up tables.

module ldpcDecLuStbc (       
  // Clock and reset
  input  wire          clk,
  input  wire          rst_n,
  // Packet parameters
  input  wire [2:0]    qamEnum,  // 0:4 is BPSK, QPSK, 16QAM, 64QAM, 256QAM
  input  wire [1:0]    sizeEnum, // zEnum
  input  wire [1:0]    rateEnum, // rEnum
  input  wire          ac,       // 0:11n 1:11ac
  input  wire [1:0]    bw,       // 0:3 is 20/40/80/160MHz
  input  wire [6:0]    mcs,      // 0-9 for 11ac, 0-72 for 11n
  input  wire [1:0]    nssLess1, // e.g. NSS=4 is specified as a 3.
  input  wire [13:0]   nCw,      // packetLen in LDPC blocks
  input  wire [1:0]    nSym,     // num OFDM symbols. clip this to 3.
  // Address in config RAM to read packet configuration
  output reg  [10:0] CfgAddr
  );

reg  [1:0]     bucketIdx;   // selects a range of packet lengths
reg  [1:0]     nbucketIdx;
reg  [3:0]     dataRateIdx; // selects iteration settings according to datarates
reg  [3:0]     ndataRateIdx;

reg  [ 10:0]    nCfgAddr;

// Define the nCw that marks the start of the next NOM_ITERATIONS bucket.
always @(*) begin
  if (sizeEnum != 2'd2)
    nbucketIdx = 2'd0;
  else
    case (rateEnum)
      2'd0 : nbucketIdx = 
             (nSym < 2'd2) ? 2'd0 :
             (nCw < 14'd12) ? 2'd1 :
             (nCw < 14'd33) ? 2'd2 : 2'd3;
      2'd1 : nbucketIdx = 
             (nSym < 2'd2) ? 2'd0 :
             (nCw < 14'd9) ? 2'd1 :
             (nCw < 14'd25) ? 2'd2 : 2'd3;
      2'd2 : nbucketIdx = 
             (nSym < 2'd2) ? 2'd0 :
             (nCw < 14'd8) ? 2'd1 :
             (nCw < 14'd22) ? 2'd2 : 2'd3;
      default : nbucketIdx = 
             (nSym < 2'd2) ? 2'd0 :
             (nCw < 14'd7) ? 2'd1 :
             (nCw < 14'd20) ? 2'd2 : 2'd3;
    endcase
end

// Iteration control registers (stored in the configuration RAM) determine throughput and
// latency during and at the end of the packet. Decode Bandwidth and MCS into the datarate index
// used to address the RAM.
wire [1:0] nssMasked;
assign nssMasked = ac ? nssLess1 : 2'd0;


// Find the index into the iteration tables (according to bw/mcs/ac/nss).
always @(*) begin
    case ({rateEnum,bw,mcs,ac,nssMasked})
    // Coding Rate 1/2
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd0, 2'd2, 7'd3, 1'd1, 2'd0} : ndataRateIdx = 4'd7; // mcs_ac_bw80x1, 130 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd0, 2'd2, 7'd1, 1'd1, 2'd0} : ndataRateIdx = 4'd6; // mcs_ac_bw80x1, 65 Mbps
`endif 
    {2'd0, 2'd1, 7'd3, 1'd0, 2'd0} : ndataRateIdx = 4'd5; // mcs_n_bw40x1, 60 Mbps
    {2'd0, 2'd1, 7'd3, 1'd1, 2'd0} : ndataRateIdx = 4'd5; // mcs_ac_bw40x1, 60 Mbps
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd0, 2'd2, 7'd0, 1'd1, 2'd0} : ndataRateIdx = 4'd4; // mcs_ac_bw80x1, 33 Mbps
`endif 
    {2'd0, 2'd1, 7'd1, 1'd0, 2'd0} : ndataRateIdx = 4'd3; // mcs_n_bw40x1, 30 Mbps
    {2'd0, 2'd1, 7'd1, 1'd1, 2'd0} : ndataRateIdx = 4'd3; // mcs_ac_bw40x1, 30 Mbps
    {2'd0, 2'd0, 7'd3, 1'd0, 2'd0} : ndataRateIdx = 4'd2; // mcs_n_bw20x1, 29 Mbps
    {2'd0, 2'd0, 7'd3, 1'd1, 2'd0} : ndataRateIdx = 4'd2; // mcs_ac_bw20x1, 29 Mbps
    {2'd0, 2'd0, 7'd1, 1'd0, 2'd0} : ndataRateIdx = 4'd1; // mcs_n_bw20x1, 15 Mbps
    {2'd0, 2'd1, 7'd0, 1'd0, 2'd0} : ndataRateIdx = 4'd1; // mcs_n_bw40x1, 15 Mbps
    {2'd0, 2'd0, 7'd1, 1'd1, 2'd0} : ndataRateIdx = 4'd1; // mcs_ac_bw20x1, 15 Mbps
    {2'd0, 2'd1, 7'd0, 1'd1, 2'd0} : ndataRateIdx = 4'd1; // mcs_ac_bw40x1, 15 Mbps
    {2'd0, 2'd0, 7'd0, 1'd0, 2'd0} : ndataRateIdx = 4'd0; // mcs_n_bw20x1, 8 Mbps
    {2'd0, 2'd1, 7'd32, 1'd0, 2'd0} : ndataRateIdx = 4'd0; // mcs_n_bw40x1, 8 Mbps
    {2'd0, 2'd0, 7'd0, 1'd1, 2'd0} : ndataRateIdx = 4'd0; // mcs_ac_bw20x1, 8 Mbps
    // Coding Rate 2/3
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd1, 2'd2, 7'd5, 1'd1, 2'd0} : ndataRateIdx = 4'd2; // mcs_ac_bw80x1, 260 Mbps
`endif 
    {2'd1, 2'd1, 7'd5, 1'd0, 2'd0} : ndataRateIdx = 4'd1; // mcs_n_bw40x1, 120 Mbps
    {2'd1, 2'd1, 7'd5, 1'd1, 2'd0} : ndataRateIdx = 4'd1; // mcs_ac_bw40x1, 120 Mbps
    {2'd1, 2'd0, 7'd5, 1'd0, 2'd0} : ndataRateIdx = 4'd0; // mcs_n_bw20x1, 58 Mbps
    {2'd1, 2'd0, 7'd5, 1'd1, 2'd0} : ndataRateIdx = 4'd0; // mcs_ac_bw20x1, 58 Mbps
    // Coding Rate 3/4
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd2, 2'd2, 7'd8, 1'd1, 2'd0} : ndataRateIdx = 4'd11; // mcs_ac_bw80x1, 390 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd2, 2'd2, 7'd6, 1'd1, 2'd0} : ndataRateIdx = 4'd10; // mcs_ac_bw80x1, 293 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd2, 2'd2, 7'd4, 1'd1, 2'd0} : ndataRateIdx = 4'd9; // mcs_ac_bw80x1, 195 Mbps
`endif 
    {2'd2, 2'd1, 7'd8, 1'd1, 2'd0} : ndataRateIdx = 4'd8; // mcs_ac_bw40x1, 180 Mbps
    {2'd2, 2'd1, 7'd6, 1'd0, 2'd0} : ndataRateIdx = 4'd7; // mcs_n_bw40x1, 135 Mbps
    {2'd2, 2'd1, 7'd6, 1'd1, 2'd0} : ndataRateIdx = 4'd7; // mcs_ac_bw40x1, 135 Mbps
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd2, 2'd2, 7'd2, 1'd1, 2'd0} : ndataRateIdx = 4'd6; // mcs_ac_bw80x1, 98 Mbps
`endif 
    {2'd2, 2'd1, 7'd4, 1'd0, 2'd0} : ndataRateIdx = 4'd5; // mcs_n_bw40x1, 90 Mbps
    {2'd2, 2'd1, 7'd4, 1'd1, 2'd0} : ndataRateIdx = 4'd5; // mcs_ac_bw40x1, 90 Mbps
    {2'd2, 2'd0, 7'd8, 1'd1, 2'd0} : ndataRateIdx = 4'd4; // mcs_ac_bw20x1, 87 Mbps
    {2'd2, 2'd0, 7'd6, 1'd0, 2'd0} : ndataRateIdx = 4'd3; // mcs_n_bw20x1, 65 Mbps
    {2'd2, 2'd0, 7'd6, 1'd1, 2'd0} : ndataRateIdx = 4'd3; // mcs_ac_bw20x1, 65 Mbps
    {2'd2, 2'd1, 7'd2, 1'd0, 2'd0} : ndataRateIdx = 4'd2; // mcs_n_bw40x1, 45 Mbps
    {2'd2, 2'd1, 7'd2, 1'd1, 2'd0} : ndataRateIdx = 4'd2; // mcs_ac_bw40x1, 45 Mbps
    {2'd2, 2'd0, 7'd4, 1'd0, 2'd0} : ndataRateIdx = 4'd1; // mcs_n_bw20x1, 44 Mbps
    {2'd2, 2'd0, 7'd4, 1'd1, 2'd0} : ndataRateIdx = 4'd1; // mcs_ac_bw20x1, 44 Mbps
    {2'd2, 2'd0, 7'd2, 1'd0, 2'd0} : ndataRateIdx = 4'd0; // mcs_n_bw20x1, 22 Mbps
    {2'd2, 2'd0, 7'd2, 1'd1, 2'd0} : ndataRateIdx = 4'd0; // mcs_ac_bw20x1, 22 Mbps
    // Coding Rate 5/6
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd3, 2'd2, 7'd9, 1'd1, 2'd0} : ndataRateIdx = 4'd4; // mcs_ac_bw80x1, 434 Mbps
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
    {2'd3, 2'd2, 7'd7, 1'd1, 2'd0} : ndataRateIdx = 4'd3; // mcs_ac_bw80x1, 325 Mbps
`endif 
    {2'd3, 2'd1, 7'd9, 1'd1, 2'd0} : ndataRateIdx = 4'd2; // mcs_ac_bw40x1, 200 Mbps
    {2'd3, 2'd1, 7'd7, 1'd0, 2'd0} : ndataRateIdx = 4'd1; // mcs_n_bw40x1, 150 Mbps
    {2'd3, 2'd1, 7'd7, 1'd1, 2'd0} : ndataRateIdx = 4'd1; // mcs_ac_bw40x1, 150 Mbps
    {2'd3, 2'd0, 7'd7, 1'd0, 2'd0} : ndataRateIdx = 4'd0; // mcs_n_bw20x1, 73 Mbps
    {2'd3, 2'd0, 7'd7, 1'd1, 2'd0} : ndataRateIdx = 4'd0; // mcs_ac_bw20x1, 73 Mbps
    default: ndataRateIdx = 4'd0;
  endcase
end // always

// Create RAM address based on data rate index
always @(*) begin
    case (rateEnum)
      2'd0 : 
      case (dataRateIdx)
        default : begin // 8 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd335; // AddrIdx=1
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd336;
              2'd1    :   nCfgAddr = 11'd337;
              2'd2    :   nCfgAddr = 11'd338;
              default :   nCfgAddr = 11'd339;
            endcase
          end // if sizeEnum
        end
        4'd1 : begin // 15 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd340; // AddrIdx=2
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd341;
              2'd1    :   nCfgAddr = 11'd342;
              2'd2    :   nCfgAddr = 11'd343;
              default :   nCfgAddr = 11'd344;
            endcase
          end // if sizeEnum
        end
        4'd2 : begin // 29 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd345; // AddrIdx=3
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd346;
              2'd1    :   nCfgAddr = 11'd347;
              2'd2    :   nCfgAddr = 11'd348;
              default :   nCfgAddr = 11'd349;
            endcase
          end // if sizeEnum
        end
        4'd3 : begin // 30 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd585; // AddrIdx=4
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd586;
              2'd1    :   nCfgAddr = 11'd587;
              2'd2    :   nCfgAddr = 11'd588;
              default :   nCfgAddr = 11'd589;
            endcase
          end // if sizeEnum
        end
`ifdef RW_NX_DERIV_CHBW804020ONLY
        4'd4 : begin // 33 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd785; // AddrIdx=5
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd786;
              2'd1    :   nCfgAddr = 11'd787;
              2'd2    :   nCfgAddr = 11'd788;
              default :   nCfgAddr = 11'd789;
            endcase
          end // if sizeEnum
        end
`endif 
        4'd5 : begin // 60 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd590; // AddrIdx=6
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd591;
              2'd1    :   nCfgAddr = 11'd592;
              2'd2    :   nCfgAddr = 11'd593;
              default :   nCfgAddr = 11'd594;
            endcase
          end // if sizeEnum
        end
`ifdef RW_NX_DERIV_CHBW804020ONLY
        4'd6 : begin // 65 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd790; // AddrIdx=7
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd791;
              2'd1    :   nCfgAddr = 11'd792;
              2'd2    :   nCfgAddr = 11'd793;
              default :   nCfgAddr = 11'd794;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        4'd7 : begin // 130 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd795; // AddrIdx=8
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd796;
              2'd1    :   nCfgAddr = 11'd797;
              2'd2    :   nCfgAddr = 11'd798;
              default :   nCfgAddr = 11'd799;
            endcase
          end // if sizeEnum
        end
`endif 
      endcase // dataRateIdx
      2'd1 : 
      case (dataRateIdx)
        default : begin // 58 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd350; // AddrIdx=9
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd351;
              2'd1    :   nCfgAddr = 11'd352;
              2'd2    :   nCfgAddr = 11'd353;
              default :   nCfgAddr = 11'd354;
            endcase
          end // if sizeEnum
        end
        4'd1 : begin // 120 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd595; // AddrIdx=10
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd596;
              2'd1    :   nCfgAddr = 11'd597;
              2'd2    :   nCfgAddr = 11'd598;
              default :   nCfgAddr = 11'd599;
            endcase
          end // if sizeEnum
        end
`ifdef RW_NX_DERIV_CHBW804020ONLY
        4'd2 : begin // 260 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd800; // AddrIdx=11
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd801;
              2'd1    :   nCfgAddr = 11'd802;
              2'd2    :   nCfgAddr = 11'd803;
              default :   nCfgAddr = 11'd804;
            endcase
          end // if sizeEnum
        end
`endif 
      endcase // dataRateIdx
      2'd2 : 
      case (dataRateIdx)
        default : begin // 22 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd355; // AddrIdx=12
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd356;
              2'd1    :   nCfgAddr = 11'd357;
              2'd2    :   nCfgAddr = 11'd358;
              default :   nCfgAddr = 11'd359;
            endcase
          end // if sizeEnum
        end
        4'd1 : begin // 44 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd360; // AddrIdx=13
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd361;
              2'd1    :   nCfgAddr = 11'd362;
              2'd2    :   nCfgAddr = 11'd363;
              default :   nCfgAddr = 11'd364;
            endcase
          end // if sizeEnum
        end
        4'd2 : begin // 45 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd600; // AddrIdx=14
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd601;
              2'd1    :   nCfgAddr = 11'd602;
              2'd2    :   nCfgAddr = 11'd603;
              default :   nCfgAddr = 11'd604;
            endcase
          end // if sizeEnum
        end
        4'd3 : begin // 65 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd365; // AddrIdx=15
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd366;
              2'd1    :   nCfgAddr = 11'd367;
              2'd2    :   nCfgAddr = 11'd368;
              default :   nCfgAddr = 11'd369;
            endcase
          end // if sizeEnum
        end
        4'd4 : begin // 87 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd370; // AddrIdx=16
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd371;
              2'd1    :   nCfgAddr = 11'd372;
              2'd2    :   nCfgAddr = 11'd373;
              default :   nCfgAddr = 11'd374;
            endcase
          end // if sizeEnum
        end
        4'd5 : begin // 90 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd605; // AddrIdx=17
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd606;
              2'd1    :   nCfgAddr = 11'd607;
              2'd2    :   nCfgAddr = 11'd608;
              default :   nCfgAddr = 11'd609;
            endcase
          end // if sizeEnum
        end
`ifdef RW_NX_DERIV_CHBW804020ONLY
        4'd6 : begin // 98 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd805; // AddrIdx=18
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd806;
              2'd1    :   nCfgAddr = 11'd807;
              2'd2    :   nCfgAddr = 11'd808;
              default :   nCfgAddr = 11'd809;
            endcase
          end // if sizeEnum
        end
`endif 
        4'd7 : begin // 135 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd610; // AddrIdx=19
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd611;
              2'd1    :   nCfgAddr = 11'd612;
              2'd2    :   nCfgAddr = 11'd613;
              default :   nCfgAddr = 11'd614;
            endcase
          end // if sizeEnum
        end
        4'd8 : begin // 180 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd615; // AddrIdx=20
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd616;
              2'd1    :   nCfgAddr = 11'd617;
              2'd2    :   nCfgAddr = 11'd618;
              default :   nCfgAddr = 11'd619;
            endcase
          end // if sizeEnum
        end
`ifdef RW_NX_DERIV_CHBW804020ONLY
        4'd9 : begin // 195 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd810; // AddrIdx=21
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd811;
              2'd1    :   nCfgAddr = 11'd812;
              2'd2    :   nCfgAddr = 11'd813;
              default :   nCfgAddr = 11'd814;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        4'd10 : begin // 293 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd815; // AddrIdx=22
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd816;
              2'd1    :   nCfgAddr = 11'd817;
              2'd2    :   nCfgAddr = 11'd818;
              default :   nCfgAddr = 11'd819;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        4'd11 : begin // 390 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd820; // AddrIdx=23
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd821;
              2'd1    :   nCfgAddr = 11'd822;
              2'd2    :   nCfgAddr = 11'd823;
              default :   nCfgAddr = 11'd824;
            endcase
          end // if sizeEnum
        end
`endif 
      endcase // dataRateIdx
      default : 
      case (dataRateIdx)
        default : begin // 73 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd375; // AddrIdx=24
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd376;
              2'd1    :   nCfgAddr = 11'd377;
              2'd2    :   nCfgAddr = 11'd378;
              default :   nCfgAddr = 11'd379;
            endcase
          end // if sizeEnum
        end
        4'd1 : begin // 150 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd620; // AddrIdx=25
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd621;
              2'd1    :   nCfgAddr = 11'd622;
              2'd2    :   nCfgAddr = 11'd623;
              default :   nCfgAddr = 11'd624;
            endcase
          end // if sizeEnum
        end
        4'd2 : begin // 200 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd625; // AddrIdx=26
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd626;
              2'd1    :   nCfgAddr = 11'd627;
              2'd2    :   nCfgAddr = 11'd628;
              default :   nCfgAddr = 11'd629;
            endcase
          end // if sizeEnum
        end
`ifdef RW_NX_DERIV_CHBW804020ONLY
        4'd3 : begin // 325 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd825; // AddrIdx=27
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd826;
              2'd1    :   nCfgAddr = 11'd827;
              2'd2    :   nCfgAddr = 11'd828;
              default :   nCfgAddr = 11'd829;
            endcase
          end // if sizeEnum
        end
`endif 
`ifdef RW_NX_DERIV_CHBW804020ONLY
        4'd4 : begin // 434 Mbps
          if ((sizeEnum == 2'd0) || (sizeEnum == 2'd1)) begin
            nCfgAddr = 11'd830; // AddrIdx=28
          end else begin
            case (bucketIdx)
              2'd0    :   nCfgAddr = 11'd831;
              2'd1    :   nCfgAddr = 11'd832;
              2'd2    :   nCfgAddr = 11'd833;
              default :   nCfgAddr = 11'd834;
            endcase
          end // if sizeEnum
        end
`endif 
      endcase // dataRateIdx
    endcase
end // always

  always @(posedge clk, negedge rst_n)
    if(!rst_n) begin
      CfgAddr             <= 11'd0;
      dataRateIdx         <= 4'd0;
      bucketIdx           <= 2'd0;
    end else begin
      CfgAddr             <= nCfgAddr - 11'd335; // refs #10147: stbc temporary hack
      dataRateIdx         <= ndataRateIdx;
      bucketIdx           <= nbucketIdx;
    end

endmodule
