//------------------------------------------------------------------------------
// ldcpDecLuTop.v
// 
// Description
//   Generates many of the LDPC Decoder registers using a look up table. This
//   allows the decoder to be used in systems without an MCU/DSP. The tables
//   are in the 2 instanced blocks, which are auto-generated. Note that these
//   blocks are stored in the config/verilog directory and have names like
//   8_6_3_2p_480.v. The sole purpose of ldpcDecLuTop is to muxe between
//   the STBC and non-STBC tables.
//
// Parameters:
// Inputs:
//   stbc        0: non STBC, 1: STBC
//   qamEnum     0:4 is BPSK QPSK 16QAM 64QAM 256QAM
//   sizeEnum    zEnum
//   rateEnum    rEnum
//   ac          0:11n 1:11ac
//   bw          0:3 is 20/40/80/160MHz
//   mcs         0-9 for 11ac 0-72 for 11n
//   nssLess1    e.g. NSS=4 is specified as a 3.
//   nCw         packetLen in LDPC blocks
//   nSym        num OFDM symbols. clip this to 3.
// Outputs:
//   Each output maps to an LDPC Decoder register.
//
// 13 Nov 2014 M. Rumsey - Ported from C reference model.
//
// (c) Copyright 2014, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------
`default_nettype none

`ifdef LDEC_SV
`include "ldpcDec.svh"
`else
`include "ldpcDec.vh"
`endif

module ldpcDecLuTop
(
  input  wire                                 clk,
  input  wire                                 rst_n,

  input  wire                                 enable,
  output reg                                  lutEnable,
  
  input  wire                                 stbc,     // 0: non STBC, 1: STBC                    
  input  wire [2:0]                           qamEnum,  // 0:4 is BPSK, QPSK, 16QAM, 64QAM, 256QAM 
  input  wire [1:0]                           sizeEnum, // zEnum                                   
  input  wire [1:0]                           rateEnum, // rEnum                                   
  input  wire [2:0]                           RuType,   // 0:5 is RU26, RU52, RU106, RU242, RU484
  input  wire                                 ac,       // 0:11n 1:11ac                            
  input  wire                                 he,       // 0:11n/11ac  1: he
  input  wire [1:0]                           bw,       // 0:3 is 20/40/80/160MHz                  
  input  wire [6:0]                           mcs,      // 0-9 for 11ac, 0-72 for 11n              
  input  wire [1:0]                           nssLess1, // e.g. NSS=4 is specified as a 3.         
  input  wire [`LDEC_B_BITS-1:0]              nCw,      // packetLen in LDPC blocks                
  input  wire [1:0]                           nSym,     // num OFDM symbols. clip this to 3.       
  output reg [`LDEC_VAR_BITS-1+7:0]           targetLevel,                                         
  output reg [numBits(`LDEC_K_MAX)-1:0]       parityThresh,                                        
  output reg [`RW_LDPCCFGRAM_AWIDTH-1:0]      CfgAddr
);
  
`ifdef LDEC_SV
`include "ldpcDecFuncs.svh"
`else
`include "ldpcDecFuncs.vh"
`endif
  
  wire [`LDEC_VAR_BITS-1+7:0]          targetLevel0;
  wire [numBits(`LDEC_K_MAX)-1:0]      parityThresh0;
  wire [10:0]                          CfgAddr0;
  wire [10:0]                          HeCfgAddr0;
  wire                                 lutEnable0;
  wire [10:0]                          StbcCfgAddr0;
  wire [10:0]                          StbcHeCfgAddr0;

  ldpcDecLu Lu0
    (
     .clk           (clk),
     .rst_n         (rst_n),
     .qamEnum       (qamEnum),
     .sizeEnum      (sizeEnum),
     .rateEnum      (rateEnum),
     .enable        (enable),
     .lutEnable     (lutEnable0),
     .ac            (ac),
     .bw            (bw),
     .mcs           (mcs),
     .nssLess1      (nssLess1),
`ifdef RW_NX_DERIV_CHBW20ONLY
     .nCw           ({5'd0,nCw}),
`endif // RW_NX_DERIV_CHBW20ONLY
`ifdef RW_NX_DERIV_CHBW4020ONLY
     .nCw           ({4'd0,nCw}),
`endif // RW_NX_DERIV_CHBW4020ONLY
`ifdef RW_NX_DERIV_CHBW804020ONLY
     .nCw           ({3'd0,nCw}),
`endif // RW_NX_DERIV_CHBW804020ONLY
     .nSym          (nSym),
     // Outputs
     .targetLevel   (targetLevel0),
     .parityThresh  (parityThresh0),
     .CfgAddr       (CfgAddr0)
  );
     

  HeLdpcDecLu HeLu0
    (
     .clk      (clk),
     .rst_n    (rst_n),
     .qamEnum  (qamEnum),
     .sizeEnum (sizeEnum),
     .rateEnum (rateEnum),
     .RuType   (RuType),
     .mcs      (mcs),
     .nssLess1 (nssLess1),
     .dcm      (1'b0),
`ifdef RW_NX_DERIV_CHBW20ONLY
     .nCw      ({5'd0,nCw}),
`endif // RW_NX_DERIV_CHBW20ONLY
`ifdef RW_NX_DERIV_CHBW4020ONLY
     .nCw      ({4'd0,nCw}),
`endif // RW_NX_DERIV_CHBW4020ONLY
`ifdef RW_NX_DERIV_CHBW804020ONLY
     .nCw      ({3'd0,nCw}),
`endif // RW_NX_DERIV_CHBW804020ONLY
     // Outputs
     .CfgAddr  (HeCfgAddr0)
  );
     

  ldpcDecLuStbc Lu0Stbc
    (
     .clk           (clk),
     .rst_n         (rst_n),
     .qamEnum       (qamEnum),
     .sizeEnum      (sizeEnum),
     .rateEnum      (rateEnum),
     .ac            (ac),
     .bw            (bw),
     .mcs           (mcs),
     .nssLess1      (nssLess1),
`ifdef RW_NX_DERIV_CHBW20ONLY
     .nCw           ({5'd0,nCw}),
`endif // RW_NX_DERIV_CHBW20ONLY
`ifdef RW_NX_DERIV_CHBW4020ONLY
     .nCw           ({4'd0,nCw}),
`endif // RW_NX_DERIV_CHBW4020ONLY
`ifdef RW_NX_DERIV_CHBW804020ONLY
     .nCw           ({3'd0,nCw}),
`endif // RW_NX_DERIV_CHBW804020ONLY
     .nSym          (nSym),
     // Outputs
     .CfgAddr       (StbcCfgAddr0)
  );

     
  HeLdpcDecLuStbc HeLuStbc0
    (
     .clk      (clk),
     .rst_n    (rst_n),
     .qamEnum  (qamEnum),
     .sizeEnum (sizeEnum),
     .rateEnum (rateEnum),
     .RuType   (RuType),
     .mcs      (mcs),
     .nssLess1 (nssLess1),
     .dcm      (1'b0),
`ifdef RW_NX_DERIV_CHBW20ONLY
     .nCw      ({5'd0,nCw}),
`endif // RW_NX_DERIV_CHBW20ONLY
`ifdef RW_NX_DERIV_CHBW4020ONLY
     .nCw      ({4'd0,nCw}),
`endif // RW_NX_DERIV_CHBW4020ONLY
`ifdef RW_NX_DERIV_CHBW804020ONLY
     .nCw      ({3'd0,nCw}),
`endif // RW_NX_DERIV_CHBW804020ONLY
     // Outputs
     .CfgAddr  (StbcHeCfgAddr0)
  );
  
     
  always @(posedge clk, negedge rst_n)
    if(!rst_n) begin
      targetLevel   <= 'd0;
      parityThresh  <= 'd0;
      CfgAddr       <= {`RW_LDPCCFGRAM_AWIDTH{1'b0}};
      lutEnable     <= 'd0;
    end else begin
      targetLevel   <= targetLevel0;
      parityThresh  <= parityThresh0;
      if (stbc==1'b0) begin
        if (he==1'b1)
          CfgAddr   <= HeCfgAddr0[`RW_LDPCCFGRAM_AWIDTH-1:0]; // Depending on define, only first addresses of the RAM are used
        else
          CfgAddr   <= CfgAddr0[`RW_LDPCCFGRAM_AWIDTH-1:0];
      end else begin
        if (he==1'b1)
          CfgAddr   <= StbcHeCfgAddr0[`RW_LDPCCFGRAM_AWIDTH-1:0];
        else
          CfgAddr   <= StbcCfgAddr0[`RW_LDPCCFGRAM_AWIDTH-1:0];
      end
      lutEnable     <= lutEnable0;
    end
    
endmodule
`default_nettype wire
