//------------------------------------------------------------------------------
// ldcpDec.v
// 
// Description
//   Top level wrapper of the LDPC decoder with registers, clocking and
//   (optionally) memories at the top level.
// 
// Inputs:
//   nReset         : Asynchronous reset.
//   clk            : Master clock.
// (From the processor)
//   blkSelect      : Active high select for the block.
//   writeNotRead   : Active high write enable.
//   addr           : Address bus from processor.
//   wrData         : Write data bus from processor.
// (From sender (e.g. equaliser))
//   inStrobe       : High when inData is valid.
//   ipWidth        : The number of LLRs that are valid values.
//   inDataWord     : `IP_WIDTH samples of `IP_BITS.
//   llrUnity       : Integer that represents unity (gets scaled each iter).
// Outputs:
// (To sender (e.g. demapper))
//   rdyToRcv       : Enables the sender to start sending. Stays high until
//                    the expected amount of data has been transferred.
//   lastIpSample   : Synchronous to the last input sample of the LDPC block.
// (To downstream hardware)
//   decodeComplete : A pulse indicating completion of decode.
//   decodeStatus   : 0/1 for fail/pass. Valid on decodeComplete.
//   packetComplete : 1 indicates all blocks have been processed. Set on last
//                    decodeComplete pulse. Cleared when enable reg cleared.
//   packetStatus   : Starts 1 but cleared on first block decode fail.
//   opStrobe       : When 1 the data on opDataWord is valid.
//   opDataWord     : Decoded output data.
// (To the processor)
//   rdDataOut        : Read data bus.
//
// 20 Apr 2010 M. Rumsey. Created.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`include "ldpcDec.vh"

module ldpcDec
(
 input                                    nReset,
 input                                    clk,
 input                                    scanMode,
    // Inputs from the processor
 input [`LDEC_BUS32:0]                    writeEn,
 input                                    readEn,
 input [`LDEC_NUM_ADDR_BITS-1:0]          addr,
 input [`LDEC_NUM_DATA_BITS-1:0]          wrData,
    // Inputs from sender
 input                                    inStrobe,
 input [numBits(`LDEC_IP_WIDTH)-1:0]      ipWidth,
 input [`LDEC_IP_WIDTH*`LDEC_IP_BITS-1:0] inDataWord,
 input [`LDEC_CHK_BITS-1:0]               llrUnity,
 input                                    iterationAbort, 
    // Inputs from output interface
 input                                    clrToSend,
    // Outputs to sender
 output                                   rdyToRcvOut,
 output                                   lastIpSampleOut,
    // Outputs to downstream hardware
 output                                   decodeActiveOut,
 output                                   decodeCompleteOut,
 output                                   decodeStatusOut,
 output                                   packetCompleteOut,
 output                                   packetOpCompleteOut,
 output                                   packetStatusOut,
 output                                   opStrobeOut,
 output [`LDEC_OP_WIDTH-1:0]              opDataWordOut,
 output [`LDEC_OP_WIDTH/8-1:0]            opByteEnOut,
    // Outputs to the processor. Not registered.
 output [`LDEC_NUM_DATA_BITS-1:0]         rdDataOut);

`include "ldpcDecFuncs.vh"

  wire ipClkEn;  // Loading of input data
  wire opClkEn;  
  wire regsClkEn;
  wire decClkEn;
  wire cellClkEn;
  wire cellClkEn2;
  wire vmRam1ClkEn;  
  wire vmRam2ClkEn; 
  wire hdRamClkEn; 
  wire crRamClkEn; 
  wire ctrlClkEn;

  wire ipClk;
  wire opClk;
  wire regsClk;
  wire decClk;
  wire cellClk;
  wire vmRam1Clk;  
  wire vmRam2Clk;  
  wire hdRamClk;  
  wire crRamClk;  
  wire ctrlClk;
                      
  //---------------------------------------------------------------------------
  // Registers programmable by processor.
  //---------------------------------------------------------------------------
                         
  
  wire                                   enable;
  wire                                   enLostTime;
  wire                                   beatTimeLine;
  wire                                   cellAccess;
  wire [numBits(`LDEC_MAX_BLKNUM)-1:0]   packetLen;
  wire [15:0]                            packetBytesLs;
  wire [`LDEC_PB_MS_LEFT:0]              packetBytesMs;
  wire [15:0]                            frameEndByteLs;
  wire [`LDEC_PB_MS_LEFT+1:0]            frameEndByteMs;
  
  wire [numBits(`LDEC_BPS_MAX)-1:0]      bitsPerSymbol;
  // Code characteristics from cont rol.
  wire [numBits(`LDEC_Z_ENUM_MAX)-1:0]   zEnum;
  wire [numBits(`LDEC_R_ENUM_MAX)-1:0]   rEnum;
  // Shortening etc parameters from control.
  wire [numBits(`LDEC_N_MAX-1)-1:0]      nShrtFloor;
  wire [numBits(`LDEC_MAX_BLKNUM)-1:0]   shrtMod;
  wire [numBits(`LDEC_M_MAX-1)-1:0]      nPuncFloor;
  wire [numBits(`LDEC_MAX_BLKNUM)-1:0]   puncMod;
  wire [numBits(`LDEC_NREPS_MAX-1)-1:0]  nRepFloor;
  wire [numBits(`LDEC_MAX_BLKNUM)-1:0]   repMod;
  // Scaling control               
  wire [`LDEC_VAR_BITS-1+7:0]            targetLevel;
  // Iteration control             
  wire [numBits(`LDEC_K_MAX)-1:0]        parityThresh;
  wire [numBits(8*`LDEC_MAX_ITER)-1:0]   nomIterations;
  wire [numBits(`LDEC_MAX_ITER)-1:0]     earlyTestIterations;
  wire [numBits(8*`LDEC_MAX_ITER)-1:0]   maxRunningCount;
  wire [numBits(8*`LDEC_MAX_ITER)-1:0]   endRunningCount;
  wire [`LDEC_CHK_BITS-2:0]              llrUnityReg;
                                   
  // Strobes and busses to allow write/read of the code cell memory.
  wire                                   cellRead;
  wire                                   cellWrite;
  wire [`LDEC_CELL_RAM_W-1:0]            cellWrData;
  
  //---------------------------------------------------------------------------
  // Status signals readable by processor.
  //---------------------------------------------------------------------------

  wire [numBits(`LDEC_MAX_BLKNUM)-1:0]   blkErrs;
  wire [numBits(`LDEC_MAX_BLKNUM)-1:0]   curBlkNum;
  wire [numBits(`LDEC_MAX_ITER)-1:0]     curIteration;
  wire [numBits(`LDEC_PAR_ERRS_MAX)-1:0] prevParityErrs;
  wire [numBits(`LDEC_MAX_ITER)-1:0]     prevIterations;
  wire                                   prevDecodeStatus;
  wire [`LDEC_VAR_BITS+5:0]              varMetricChkSum;
  wire [`LDEC_CELL_RAM_W-1:0]            cellRdData;
  
  //---------------------------------------------------------------------------
  // Memories
  //---------------------------------------------------------------------------

  // Input and output signals to RAMs are passed to the top level with simple
  // concatenated vectors (*RamIps, *RamOps) to simplify interfaces.
  // Top level RAMs are optional and these signals will be
  // hanging if `LDEC_RAMS_AT_TOP defined.
`ifdef LDEC_RAMS_AT_TOP
`ifndef LDEC_USE_CELL_ROM
  wire [`LDEC_CELL_RAM_O-1:0]            cellRamOps;
  wire [`LDEC_CELL_RAM_I-1:0]            cellRamIps;
  wire [`LDEC_CELL_RAM_A-1:0]            cellRamAddrA;
  wire [`LDEC_CELL_RAM_A-1:0]            cellRamAddrB;
  wire [`LDEC_CELL_RAM_W-1:0]            cellRamWrData;
  wire                                   cellRamWrite;
  wire                                   cellRamEnA;
  wire                                   cellRamEnB;
  wire [`LDEC_CELL_RAM_W-1:0]            cellRamRdDataA;
  wire [`LDEC_CELL_RAM_W-1:0]            cellRamRdDataB;
`endif
  // VM RAMs
  wire [`LDEC_VM_RAM_O-1:0]              vmRamOps;
  wire [`LDEC_VM_RAM_I-1:0]              vmRamIps;
  wire [`LDEC_VM_RAM_A-1:0]              vmRamAddrAX;
  wire [`LDEC_VM_RAM_A-1:0]              vmRamAddrBX;
  wire [`LDEC_VM_RAM_W-1:0]              vmRamWrDataX;
  wire [`LDEC_VM_RAM_WE-1:0]             vmRamWriteX;
  wire                                   vmRamEnAX;
  wire [`LDEC_VM_RAM_W-1:0]              vmRamRdDataX;
  wire [`LDEC_VM_RAM_A-1:0]              vmRamAddrAY;
  wire [`LDEC_VM_RAM_A-1:0]              vmRamAddrBY;
  wire [`LDEC_VM_RAM_W-1:0]              vmRamWrDataY;
  wire [`LDEC_VM_RAM_WE-1:0]             vmRamWriteY;
  wire                                   vmRamEnAY;
  wire [`LDEC_VM_RAM_W-1:0]              vmRamRdDataY;
  // HD RAMs
  wire [`LDEC_HD_RAM_O-1:0]              hdRamOps;
  wire [`LDEC_HD_RAM_I-1:0]              hdRamIps;
  wire [`LDEC_HD_RAM_A-1:0]              hdRamAddrAX;
  wire [`LDEC_HD_RAM_A-1:0]              hdRamAddrB;
  wire [`LDEC_HD_RAM_W-1:0]              hdRamWrData;
  wire                                   hdRamWriteX;
  wire                                   hdRamEnAX;
  wire [`LDEC_HD_RAM_W-1:0]              hdRamRdDataX;
  wire [`LDEC_HD_RAM_A-1:0]              hdRamAddrAY;
  wire                                   hdRamWriteY;
  wire                                   hdRamEnAY;
  wire [`LDEC_HD_RAM_W-1:0]              hdRamRdDataY;
  // Cr RAMs
  wire [`LDEC_CR_RAM_O-1:0]              crRamOps;
`ifdef LDEC_2_PORTS
  wire [`LDEC_CR_RAM_I-1:0]              crRamIps;
`else
  wire [`LDEC_CR1_RAM_I-1:0]             cr1RamIps;
`endif
  wire [`LDEC_CR_RAM_A-1:0]              crRamAddrA;
  wire [`LDEC_CR_RAM_A-1:0]              crRamAddrB;
  wire [`LDEC_CR_RAM_W-1:0]              crRamWrData;
  wire                                   crRamEn;
  wire                                   crRamWr;
  wire [`LDEC_CR_RAM_W-1:0]              crRamRdData;
  // Vr RAMs
  wire [`LDEC_VR_RAM_O-1:0]              vrRamOps;
`ifdef LDEC_2_PORTS
  wire [`LDEC_VR_RAM_I-1:0]              vrRamIps;
`else
  wire [`LDEC_VR1_RAM_I-1:0]             vr1RamIps;
`endif
  wire [`LDEC_VR_RAM_A-1:0]              vrRamAddrA;
  wire [`LDEC_VR_RAM_A-1:0]              vrRamAddrB;
  wire [`LDEC_VR_RAM_W-1:0]              vrRamWrData;
  wire                                   vrRamEn;
  wire                                   vrRamWr;
  wire [`LDEC_VR_RAM_W-1:0]              vrRamRdData;
`endif
  // Debug port
  wire [`LDEC_DBG_PORT_WIDTH-1:0]        dbgIp1;
  wire [`LDEC_DBG_PORT_WIDTH-1:0]        dbgIp2;
  wire [`LDEC_DBG_PORT_WIDTH-1:0]        dbgOp;
  wire [`LDEC_DBG_PORT_WIDTH-1:0]        dbgCtrl;

  //---------------------------------------------------------------------------
  // Clocking
  //---------------------------------------------------------------------------

  // Add decClkEn to cellClkEn when `LDEC_USE_ALT_RAMS as only a single clock;
  // used for CellMem in that case.
   assign cellClkEn2 = (`LDEC_USE_ALT_RAMS) ? cellClkEn | decClkEn : cellClkEn;
  
  // Gated clocking
  ldpcDecClkGate clkGate1 (clk, scanMode, ipClkEn, nReset, ipClk);
  ldpcDecClkGate clkGate2 (clk, scanMode, regsClkEn, nReset, regsClk);
  ldpcDecClkGate clkGate3 (clk, scanMode, decClkEn, nReset, decClk);
  ldpcDecClkGate clkGate4 (clk, scanMode, cellClkEn2, nReset, cellClk);
  ldpcDecClkGate clkGate5 (clk, scanMode, vmRam1ClkEn, nReset, vmRam1Clk);
  ldpcDecClkGate clkGate6 (clk, scanMode, vmRam2ClkEn, nReset, vmRam2Clk);
  ldpcDecClkGate clkGate7 (clk, scanMode, ctrlClkEn, nReset, ctrlClk);
  ldpcDecClkGate clkGate8 (clk, scanMode, hdRamClkEn, nReset, hdRamClk);
  ldpcDecClkGate clkGate9 (clk, scanMode, crRamClkEn, nReset, crRamClk);
  ldpcDecClkGate clkGate10 (clk, scanMode, opClkEn, nReset, opClk);
 
  //---------------------------------------------------------------------------
  // Registers && Control
  //---------------------------------------------------------------------------
  
   ldpcDecRegs regs
     (
      // Inputs
      .nReset                 (nReset),
      .clk                    (regsClk),
      // Processor interface inputs.
      .writeEn                (writeEn),
      .readEn                 (readEn),
      .addr                   (addr),
      .wrData                 (wrData),
      // Read only registers that may be monitored by processor.
      .blkErrs                (blkErrs),
      .curBlkNum              (curBlkNum),
      .curIteration           (curIteration),
      .prevParityErrs         (prevParityErrs),
      .prevIterations         (prevIterations),
      .prevDecodeStatus       (prevDecodeStatus),
      .varMetricChkSum        (varMetricChkSum),
      .cellRdData             (cellRdData),
      // Outputs             
      .clkEnOut               (regsClkEn),
      .rdDataOut              (rdDataOut),
      .enableOut              (enable),
      .enLostTimeOut          (enLostTime),
      .beatTimeLineOut        (beatTimeLine),     
      // To mem block to program the code cell information.
      .cellAccessOut          (cellAccess),
      .cellReadOut            (cellRead),
      .cellWriteOut           (cellWrite),
      .cellWrDataOut          (cellWrData),
      // Control registers.
      .zEnumOut               (zEnum),
      .rEnumOut               (rEnum),
      .bitsPerSymbolOut       (bitsPerSymbol),
      .packetLenOut           (packetLen),
      .packetBytesLsOut       (packetBytesLs),
      .packetBytesMsOut       (packetBytesMs),
      .frameEndByteLsOut      (frameEndByteLs),
      .frameEndByteMsOut      (frameEndByteMs),    
      .nShrtFloorOut          (nShrtFloor),
      .shrtModOut             (shrtMod),
      .nPuncFloorOut          (nPuncFloor),
      .puncModOut             (puncMod),
      .nRepFloorOut           (nRepFloor),
      .repModOut              (repMod),
      .targetLevelOut         (targetLevel),
      .parityThreshOut        (parityThresh),
      .nomIterationsOut       (nomIterations),
      .earlyTestIterationsOut (earlyTestIterations),
      .maxRunningCountOut     (maxRunningCount),
      .endRunningCountOut     (endRunningCount),
      .llrUnityRegOut         (llrUnityReg));

  ldpcDecCore core
    (
     .nReset              (nReset),
     // Inputs from sender
     .inStrobe            (inStrobe),
     .inDataWord          (inDataWord),
     .llrUnity            (llrUnity),
     .iterationAbort      (iterationAbort),
     // Inputs from output interface
     .clrToSend           (clrToSend),
     // Outputs to sender
     .rdyToRcvOut         (rdyToRcvOut),
     .lastIpSampleOut     (lastIpSampleOut),
     // Outputs to downstream hardware
     .decodeCompleteOut   (decodeCompleteOut),
     .decodeStatusOut     (decodeStatusOut),
     .packetCompleteOut   (packetCompleteOut),
     .packetOpCompleteOut (packetOpCompleteOut),
     .packetStatusOut     (packetStatusOut),
     .opStrobeOut         (opStrobeOut),
     .opDataWordOut       (opDataWordOut),
     .opByteEnOut         (opByteEnOut),
     //-----------------------------------------
     // Registers
     //-----------------------------------------
     // Master enable to control.
     .enable              (enable),
     .enLostTime          (enLostTime),
     .beatTimeLine        (beatTimeLine),     
     // To mem block to program the code cell information.
     .cellAccess          (cellAccess),
     .cellRead            (cellRead),
     .cellWrite           (cellWrite),
     .cellWrData          (cellWrData),
     // Code characteristics to control.
     .ipWidth             (ipWidth),
     .zEnum               (zEnum),
     .rEnum               (rEnum),
     .bitsPerSymbol       (bitsPerSymbol),
     .packetLen           (packetLen),
     .packetBytesLs       (packetBytesLs),
     .packetBytesMs       (packetBytesMs),
     .frameEndByteLs      (frameEndByteLs),
     .frameEndByteMs      (frameEndByteMs),    
     // Shortening etc parameters to control.
     .nShrtFloor          (nShrtFloor),
     .shrtMod             (shrtMod),
     .nPuncFloor          (nPuncFloor),
     .puncMod             (puncMod),
     .nRepFloor           (nRepFloor),
     .repMod              (repMod),
     .targetLevel         (targetLevel),
     .parityThresh        (parityThresh),
     .nomIterations       (nomIterations),
     .earlyTestIterations (earlyTestIterations),
     .maxRunningCount     (maxRunningCount),
     .endRunningCount     (endRunningCount),
     .llrUnityReg         (llrUnityReg),
     // Status registers
     .blkErrsOut          (blkErrs),
     .curBlkNumOut        (curBlkNum),
     .curIterationOut     (curIteration),
     .prevParityErrsOut   (prevParityErrs),
     .prevIterationsOut   (prevIterations),
     .prevDecodeStatusOut (prevDecodeStatus),
     .varMetricChkSumOut  (varMetricChkSum),
     .cellRdDataOut       (cellRdData),
     //-----------------------------------------
     // Clocking
     //-----------------------------------------
     .regsClk             (regsClk),
     .ipClk               (ipClk),
     .opClk               (opClk),
     .decClk              (decClk),
     .cellClk             (cellClk),
     .vmRam1Clk           (vmRam1Clk),
     .vmRam2Clk           (vmRam2Clk),
     .hdRamClk            (hdRamClk),
     .crRamClk            (crRamClk),
     .ctrlClk             (ctrlClk),
     .ipClkEnOut          (ipClkEn),
     .opClkEnOut          (opClkEn),
     .decClkEnOut         (decClkEn),
     .cellClkEnOut        (cellClkEn),
     .vmRam1ClkEnOut      (vmRam1ClkEn),
     .vmRam2ClkEnOut      (vmRam2ClkEn),
     .hdRamClkEnOut       (hdRamClkEn),
     .crRamClkEnOut       (crRamClkEn),
     .ctrlClkEnOut        (ctrlClkEn),
     //-----------------------------------------------------------------------
     // I/O busses to RAMS (when `LDEC_RAMS_AT_TOP == 1)
     //-----------------------------------------------------------------------
`ifdef LDEC_RAMS_AT_TOP
`ifndef LDEC_USE_CELL_ROM
     .cellRamOps          (cellRamOps),
     .cellRamIps          (cellRamIps),
`endif
     .vmRamOps            (vmRamOps),
     .vmRamIps            (vmRamIps),
     .hdRamOps            (hdRamOps),
     .hdRamIps            (hdRamIps),
`ifdef LDEC_2_PORTS
     .crRamOps            (crRamOps),
     .crRamIps            (crRamIps),
     .vrRamOps            (vrRamOps),
     .vrRamIps            (vrRamIps),
`else
     .cr1RamOps           (crRamOps),
     .cr1RamIps           (cr1RamIps),
     .vr1RamOps           (vrRamOps),
     .vr1RamIps           (vr1RamIps),
`endif
`endif
     //----------------------------
     //  Debug Port
     //----------------------------
     .dbgIp1Out           (dbgIp1),
     .dbgIp2Out           (dbgIp2),
     .dbgOpOut            (dbgOp),
     .dbgCtrlOut          (dbgCtrl));

  assign decodeActiveOut  = dbgCtrl[0];

`ifdef LDEC_RAMS_AT_TOP

  //-------------------------------------------------------------------------
  // Cell RAM
  //-------------------------------------------------------------------------

  // Cell RAM stores the LDPC code definition. You can also use a hardcoded
  // LUT (resident in ldpcDecCore) by defining `LDEC_USE_CELL_ROM
`ifndef LDEC_USE_CELL_ROM
      
  ldpcDecCellRam CellMem 
    (
     // Load/read via regs and read Fwd Cell info.
     .ckA   (cellClk),
     .selA  (cellRamEnA),
     .addrA (cellRamAddrA),
     .dInA  (cellRamWrData),
     .weA   (cellRamWrite),
     .dOutA (cellRamRdDataA),
     // Bwd Cell info (this port is read only).
     .ckB   (decClk),
     .selB  (cellRamEnB),
     .addrB (cellRamAddrB),
     .dInB  ('0),           // Not used.
     .weB   (1'b0),
     .dOutB (cellRamRdDataB));
  
  assign cellRamOps = {cellRamRdDataA, cellRamRdDataB};
  assign {cellRamEnA, cellRamEnB, cellRamWrite, cellRamAddrA,
          cellRamAddrB, cellRamWrData} = cellRamIps;
      
`endif

  //--------------------------------------------------------------------------
  // Cr (Check Response) RAM
  //--------------------------------------------------------------------------

`ifdef LDEC_2_PORTS
      
  ldpcDecCrRam crRam
    (
     .ck   (crRamClk),
     // Write
     .addrA (crRamAddrA),
     .dInA  (crRamWrData),
     .weA   (crRamWr),
     // Read
     .selB  (crRamEn),
     .addrB (crRamAddrB),
     .dOutB (crRamRdData));
  
  assign crRamOps = crRamRdData;
  assign {crRamWr, crRamEn, crRamAddrA, crRamWrData, crRamAddrB} = crRamIps;
      
`else
      
  // Requires read before write behaviour.
  ldpcDecCr1pRam cr1Ram
    (
     .ck   (crRamClk),
     .sel  (crRamEn),
     .addr (crRamAddrA),
     .dIn  (crRamWrData),
     .we   (crRamEn),
     .dOut (crRamRdData));
  
  assign crRamOps = crRamRdData;
  assign {crRamEn, crRamAddrA, crRamWrData} = cr1RamIps;

`endif
    
  //--------------------------------------------------------------------------
  // Vr (Check Response) RAM
  //--------------------------------------------------------------------------

`ifdef LDEC_2_PORTS
      
  ldpcDecVrRam vrRam
    (
     .ck   (decClk),
     // Write
     .addrA (vrRamAddrA),
     .dInA  (vrRamWrData),
     .weA   (vrRamWr),
     // Read
     .selB  (vrRamEn),
     .addrB (vrRamAddrB),
     .dOutB (vrRamRdData));
  
  assign vrRamOps = vrRamRdData;
  assign {vrRamWr, vrRamEn, vrRamAddrA,
          vrRamWrData, vrRamAddrB} = vrRamIps;
  
`else
  // Requires read before write behaviour.
  
  ldpcDecVr1pRam vr1Ram
    (
     .ck   (decClk),
     .sel  (vrRamEn),
     .addr (vrRamAddrA),
     .dIn  (vrRamWrData),
     .we   (vrRamEn),
     .dOut (vrRamRdData));
  
  assign vrRamOps = vrRamRdData;
  assign {vrRamEn, vrRamAddrA, vrRamWrData} = vr1RamIps;
  
`endif
   
  //-------------------------------------------------------------------------
  // VM RAM
  //-------------------------------------------------------------------------

  // VM RAM stores signal metrics, which are initially the input LLRs but
  // are begin 'improved through the decode iterations. After the final
  // iteration they are soft decisions for the final output, however the
  // actual output comes from the HD RAM. There are two RAMs to allow double
  // buffering of the input data.

  ldpcDecVmRam vmRamX
    (
     .ck    (vmRam1Clk),
     // A port always read
     .selA  (vmRamEnAX),
     .addrA (vmRamAddrAX),
     .dOutA (vmRamRdDataX),
     // B port always write
     .addrB (vmRamAddrBX),
     .dInB  (vmRamWrDataX),
     .weB   (vmRamWriteX));

  ldpcDecVmRam vmRamY
    (
     .ck    (vmRam2Clk),
     // A port always read
     .selA  (vmRamEnAY),
     .addrA (vmRamAddrAY),
     .dOutA (vmRamRdDataY),
     // B port always write
     .addrB (vmRamAddrBY),
     .dInB  (vmRamWrDataY),
     .weB   (vmRamWriteY));

  // Unpack what was packed in ldpcDecVmRamMux.
  assign vmRamOps = {vmRamRdDataX, vmRamRdDataY};
  assign {vmRamEnAX, vmRamEnAY, vmRamWriteX, vmRamAddrAX, vmRamAddrBX,
          vmRamWrDataX, vmRamWriteY, vmRamAddrAY, vmRamAddrBY,
          vmRamWrDataY} = vmRamIps;
  
  //-------------------------------------------------------------------------
  // HD RAM
  //-------------------------------------------------------------------------

  // HD RAM duplicates the sign bit of the two VM RAMs. The times of buffer
  // switching are however different (HD follows the output interface, while
  // VM follows the input interface).

  ldpcDecHdRam hdRamX
    (
     .ck    (hdRamClk),
     // A port always read
     .selA  (hdRamEnAX),
     .addrA (hdRamAddrAX),
     .dOutA (hdRamRdDataX),
     // B port always write
     .addrB (hdRamAddrB),
     .dInB  (hdRamWrData),
     .weB   (hdRamWriteX));

  ldpcDecHdRam hdRamY
    (
     .ck    (hdRamClk),
     // A port always read
     .selA  (hdRamEnAY),
     .addrA (hdRamAddrAY),
     .dOutA (hdRamRdDataY),
     // B port always write
     .addrB (hdRamAddrB),
     .dInB  (hdRamWrData),
     .weB   (hdRamWriteY));

  // Unpack what was packed in ldpcDecHdRamMux.
  assign hdRamOps = {hdRamRdDataX, hdRamRdDataY};
  assign {hdRamEnAX, hdRamEnAY, hdRamWriteX, hdRamWriteY,
          hdRamAddrAX, hdRamAddrB, hdRamWrData, hdRamAddrAY} = hdRamIps;
`endif
  
endmodule

