//-----------------------------------------------------------------------------
// ldcpDecBwd.v
// 
// Description
//   The backward section. This handles the backward part of the check metric
//   calculations. While the forward part creates a 'sum' over all variable
//   responses, the backward part will, for each signal (or bit), remove the
//   contribution due to the signal itself. This creates an estimate of
//   the signal based on *the other* variables leveraging properies of the
//   parity check equation. The overall signal metric that combines the variable
//   estimates (responses) from all check equations, plus the initial LLR of the
//   received signal, is also updated.
//
//   Where an input is listed with an _n postfix this means that it is a version
//   of the signal that is n cycles early. The core processing of pChkResp,
//   pvarMetric, pVarMetricSum and pParity are considered to be at T==0.
//   pShort and pShift  generate registered data used by the T==0 so they
//   need _1 inputs.
//
// Inputs
// (Programmable config)
//   shrtBlkMod     : (blkNum < cfg->nShrt.mod)
//   z              : The macro cells size for the current code.
//   k              : Number of data bits in the code word.
//   k2             : Number of data bits after shortening. 
// (From control)
//   col            : The column number to process.
//   rowStart       : Start of row.
//   cycShift       : The cyclical shift for this column.
//   flag           : The flag bits for this column (2 bits)
// (From Fwd)
//   chkMetricReg   : Soft sum of all varible responses for a row. (sgn+abs).
// (From memories)
//   hardDecisionsRd: Read bus from hard decisions.
//   varRespRd      : Variable response generated by Fwd. (sgn+abs).
// Outputs
// (To control)
//   parityErrsUpdate : Strobe following end of row update of parityErrs.
//   parityErrs     : Number of parity errs, updated end of row && accumulated
//                    for the iteration.
// (To IO)
//   cycLast        : Residual cyclical shift to be corrected on output
//                    from the decoder.
// (To memories)
//   varMetricWr    : Write bus of signal metrics.
//   hardDecisionsWr: Write bus of hard decisions.
//   chkRespWr      : Write bus to check response memory. (sgn+abs).
//  
// 8 Apr 2010 M. Rumsey - Ported from C reference model
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//-----------------------------------------------------------------------------


`include "ldpcDec.vh"

module ldpcDecBwd
  #(parameter PE_HI = numBits(`LDEC_PAR_ERRS_MAX)-1,
    Z_HI = numBits(`LDEC_Z_MAX-1)-1)
  (
   input                                    nReset,
   input                                    clk,
   input                                    ctrlClk,
   input                                    enable,
   // From control     
   input [numBits(`LDEC_Z_ENUM_MAX)-1:0]    zEnum,
   input                                    iterStart,
   input                                    newBlk,
   input                                    firstIteration_1,
   input                                    rowStart,
   input `LDEC_shrtCol_P                    shrtColP,
   input [numBits(`LDEC_NCOLS-1)-1:0]       parityStartCol,
   input                                    endIter,
   input [`LDEC_CHK_BITS-3:0]               llrUnityDiv2,
   input `LDEC_varAbsType                   shortDefault,
   // From memories    
   input [numBits(`LDEC_NCOLS_RIGHT)-1:0]   col_1,
   input [numBits(`LDEC_NCOLS_RIGHT)-1:0]   col,
   input [Z_HI:0]                           cycShift_1,
   input [`LDEC_FLAG_BITS-1:0]              flag,
   input [`LDEC_FLAG_BITS-1:0]              flag_1,
   input                                    bwdStall,
   input                                    bwdStall_1,
   input                                    bwdNoOp,
   input                                    bwdNoOp_1,
   input `LDEC_sgnZType                     hardDecisionsRd,
   input `LDEC_varZ_P                       varRespRd,
   // From Fwd block   
   input `LDEC_sgnZType                     chkMetricRegSgn,
   input `LDEC_chkMetricAbsZ_P              chkMetricRegAbsP,
   // To Control       
   output                                   parityErrsUpdateOut,
   output [PE_HI:0]                         parityErrsOut,
   output [`LDEC_VAR_BITS-1+7:0]            varMetricSumOut,
   // To IO            
   output `LDEC_cycShift_P                  cycLastOut,
   // To memories      
   output `LDEC_varZ_P                      varMetricWrOut,
   output `LDEC_sgnZType                    hardDecisionsWrOut,
   output `LDEC_sgnZType                    chkRespWrSgnOut,
   output `LDEC_chkRespAbsZ_P               chkRespWrAbsOut);

`include "ldpcDecFuncs.vh"

  // A mask of 1'b1s marking enabled bits for each Z size.
  wire `LDEC_sgnZType                   Z_MASK          [0:`LDEC_Z_ENUM_MAX];

  // Unpacked
  wire `LDEC_colType                    shrtCol         [0:1];
  wire `LDEC_chkAbs3_P                  chkMetricRegAbs [0:`LDEC_Z_MAX-1];
  
  // Local variables - no registers required
  wire                                  endRow;
  wire                                  metricFirstUse;
  wire                                  metricFirstUse_1;
  wire                                  metricLastUse;
  wire `LDEC_sgnZType                   hardDecisionsRot;
  reg  `LDEC_sgnZType                   parityVec;
  reg  `LDEC_sgnZType                   chkRespSgn;
  reg  `LDEC_chkAbsType                 chkRespAbs      `LDEC_chkRespAbsZ_R;
  wire `LDEC_sgnZType                   varMetricSgn;
  wire `LDEC_sgnZType                   varRespSgn;
  wire                                  bitFlip;
  reg                                   hadBitFlip;
  reg                                   parityUpdate;
  reg                                   parityErrsUpdate;
  // lint: parityCount is used for simulation logging and is not for synthesis
  reg  [Z_HI:0]                         parityCount;
  reg  [PE_HI:0]                        parityErrs;
  reg  `LDEC_varType                    varMetric       `LDEC_varZ_R;
  wire `LDEC_varZ_P                     varMetricP;
  reg  [`LDEC_VAR_BITS-1+7:0]           varMetricSum;
  reg  [`LDEC_VAR_BITS-1+7:0]           varMetricSumReg;
  reg  `LDEC_varAbsType                 varMetricAbs0;
  reg  `LDEC_varAbsType                 varMetricAbs26;
                                  
  // Pipelining signals.           
                                   
  // DFF if `LDEC_PIPE_VR.         
  wire `LDEC_varZ_P                     varRespRdReg;
  wire `LDEC_varType                    varRespRdRegU   `LDEC_varZ_R;
  // These are registers if `LDEC_PIPE_X==1.
  reg  `LDEC_varType                    varResp         `LDEC_varZ_R;
  reg  `LDEC_sgnZType                   hardDecisionsD1;
  wire `LDEC_chkRespAbsZ_P              chkRespAbsD1P;
  reg  `LDEC_chkAbsType                 chkRespAbsD1    `LDEC_chkRespAbsZ_R;
  reg  `LDEC_sgnZType                   chkRespSgnD1;
  reg                                   metricFirstUseD1;
  reg                                   metricLastUseD1;
  reg                                   iterStartD1;
  reg                                   bwdStallD1;
  reg                                   endRowD1;
  reg                                   bwdNoOpD1;
  reg                                   rowStartD1;
  reg  [1:0]                            colShortenedD1;
  reg                                   bwdStall_1D1;
  reg  [Z_HI:0]                         cycShift_1D1;
  wire [numBits(`LDEC_NCOLS_RIGHT)-1:0] col_1D1;
  reg                                   notFirstUse;
  // These are registers if `LDEC_PIPE_VMS==1.
  reg  `LDEC_varAbsType                 varMetricAbs0D2;
  reg  `LDEC_varAbsType                 varMetricAbs26D2;
  reg                                   iterStartD2;
  reg                                   metricLastUseD2;
  // These are registers if `LDEC_PIPE_PAR == 1.
  reg  `LDEC_sgnZType                   hardDecisionsDHD;
  reg  `LDEC_sgnZType                   varMetricSgnDHD;
  reg                                   bwdStallDHD;
  reg                                   endRowDHD;
  reg                                   bwdNoOpDHD;
  reg                                   rowStartDHD;
  reg                                   metricFirstUseDHD;

  // Local registers.
  reg  [Z_HI:0]                         leftShift;
  reg  `LDEC_cycShiftType               cycLast     `LDEC_cycShift_R;
  
  genvar idx1; // For packing

  assign metricFirstUse = flag[`LDEC_FLAG_FIRST_VAR_USE_BIT];
  assign metricFirstUse_1 = flag_1[`LDEC_FLAG_FIRST_VAR_USE_BIT];
  assign metricLastUse = flag[`LDEC_FLAG_LAST_VAR_USE_BIT];
  assign endRow = flag[`LDEC_FLAG_END_ROW_BIT];

  `LDEC_UNPACK(gShrt, shrtColP, shrtCol, `LDEC_COL_BITS, 2)
  // This one is partly unpacked.
  `LDEC_UNPACK(gCm, chkMetricRegAbsP, chkMetricRegAbs,
               (`LDEC_CHK_BITS-1)*(`LDEC_CHK_METRIC_RIGHT+1), `LDEC_Z_MAX)

  //---------------------------------------------------------------------------
  // Z_MASK
  //---------------------------------------------------------------------------

  // Make a mask for each of the Z_ENUM cases. 1's mark valid bits.
  generate
    genvar Ze;
    genvar Zi;

    assign Z_MASK[`LDEC_Z_ENUM_MAX] = {`LDEC_Z_MAX {1'b1}};
    
    for (Ze=0; Ze<=`LDEC_Z_ENUM_MAX-1; Ze=Ze+1)
    begin
      for (Zi=0; Zi<Z_SIZES2(Ze); Zi=Zi+1) begin
       assign Z_MASK[Ze][Zi] = 1'b1;
      end
      for (Zi=Z_SIZES2(Ze); Zi<`LDEC_Z_MAX; Zi=Zi+1) begin
       assign Z_MASK[Ze][Zi] = 1'b0;
      end
    end
   endgenerate
 
  //---------------------------------------------------------------------------
  // varResp fetch
  //---------------------------------------------------------------------------

  // varResp may be registered with the output reg of the varResp memory being
  // brought forward a cycle in ldpcDecCore so there is no functional change.
  // This is helpful when `LDEC_PIPE_X is  not set (`LDEC_PIPE_X already breaks
  // up paths from varResp).

  generate
    
    if (`LDEC_PIPE_VR == 1) begin: gVrReg
      reg `LDEC_varZ_P  varRespRdRegR;

      always @(posedge(clk) `LDEC_RESET_STR_B)
      begin : pVrReg       
        if (!nReset) begin
          varRespRdRegR <= `LDEC_PAD(1'b0, `LDEC_Z_MAX*`LDEC_VAR_BITS-1);       
        end else begin
          varRespRdRegR <= varRespRd;     
        end        
      end //pVrReg
      assign varRespRdReg = varRespRdRegR;
      
    end else begin
      
      assign varRespRdReg = varRespRd;
      
    end
  endgenerate //  gVrReg
      
  //---------------------------------------------------------------------------
  // Check Response Update
  //---------------------------------------------------------------------------

  assign varRespSgn = getSgnV(varRespRdReg);
  `LDEC_UNPACK(gVz, varRespRdReg, varRespRdRegU, `LDEC_VAR_BITS, `LDEC_Z_MAX)

  always @(*) // (chkMetricRegSgn, chkMetricRegAbs, varRespRdReg, varRespSgn, llrUnityDiv2)
  begin : pChkResp
    
    reg `LDEC_chkAbsType  varRespCmAbsV;
    reg `LDEC_chkType     chkClipped;
    reg `LDEC_varType     varRespV;
    integer zI;
      
    for (zI=0; zI<=`LDEC_Z_MAX-1; zI=zI+1) begin
      // Calculate the llrs for the check response using the forward pass
      // accumulations.
      chkRespSgn[zI] = chkMetricRegSgn[zI] ^ varRespSgn[zI];
      varRespV = varRespRdRegU[zI];    
      chkClipped = `LDEC_CLIP_S(varRespV, `LDEC_VAR_BITS, `LDEC_CHK_BITS);
      varRespCmAbsV = `LDEC_ABS1(chkClipped, `LDEC_CHK_BITS);          
      chkRespAbs[zI] = softSub(chkMetricRegAbs[zI], varRespCmAbsV, llrUnityDiv2);
    end
  end //pChkResp

  //---------------------------------------------------------------------------
  // PIPE X
  //---------------------------------------------------------------------------

  // Register Chk Response. When enabled re-timed varResp and HardDecisions
  // are also needed.

  generate
    if (`LDEC_PIPE_X == 1) begin: gPipeX1
      assign col_1D1  = col;
      always @(posedge(clk) `LDEC_RESET_STR)
      begin : pPipe
        if (!nReset) begin
          colShortenedD1 <= 2'b00;
          metricLastUseD1 <= 1'b0;
          iterStartD1 <= 1'b0;
          bwdStallD1 <= 1'b0;
          endRowD1 <= 1'b0;
          bwdNoOpD1 <= 1'b0;
          rowStartD1 <= 1'b0;
          metricFirstUseD1 <= 1'b0;      
          // for pShift
          bwdStall_1D1 <= 1'b0;
          cycShift_1D1 <= `LDEC_PAD(1'b0, Z_HI);
          notFirstUse <= 1'b0;
        end else begin
          
          metricLastUseD1 <= metricLastUse & !(bwdStall | bwdNoOp);
          iterStartD1 <= iterStart;
          bwdStallD1 <= bwdStall;
          endRowD1 <= endRow;
          bwdNoOpD1 <= bwdNoOp;
          rowStartD1 <= rowStart;
          metricFirstUseD1 <= metricFirstUse;
          colShortenedD1 <= 2'b11;
          if (col >= parityStartCol) begin
            colShortenedD1 <= 2'b00;
          end
          else begin
            if (col < shrtCol[0]) begin
              colShortenedD1[0] <= 1'b0;
            end
            if (col < shrtCol[1]) begin
              colShortenedD1[1] <= 1'b0;
            end
          end
          // for pShift
          bwdStall_1D1 <= bwdStall_1 | bwdNoOp_1;
          cycShift_1D1 <= cycShift_1;
          notFirstUse <= !(firstIteration_1 & metricFirstUse_1);

        end
      end
      always @(posedge(clk) `LDEC_RESET_STR_B)
      begin: pPipe2
        integer idx1;
        if (!nReset) begin
          `LDEC_INITQ(varResp, `LDEC_PADS(1'b0, `LDEC_VAR_BITS-1), `LDEC_Z_MAX);
          hardDecisionsD1 <= `LDEC_PAD(1'b0, `LDEC_Z_MAX-1);
          `LDEC_INITQ(chkRespAbsD1, `LDEC_PAD(1'b0, `LDEC_CHK_BITS-2), `LDEC_Z_MAX);
          chkRespSgnD1 <= `LDEC_PAD(1'b0, `LDEC_Z_MAX-1);            
        end else begin
          `LDEC_UNPACKQ(varRespRdReg, varResp, `LDEC_VAR_BITS,  `LDEC_Z_MAX);
          `LDEC_COPYQ(chkRespAbs, chkRespAbsD1, `LDEC_Z_MAX);
          hardDecisionsD1 <= hardDecisionsRd;
          chkRespSgnD1 <= chkRespSgn;               
        end
      end
    end else begin

      always @(*) begin: pPipe3
        integer idx1;
        `LDEC_UNPACKF(varRespRdReg, varResp, `LDEC_VAR_BITS,  `LDEC_Z_MAX);
        `LDEC_COPYQ(chkRespAbs, chkRespAbsD1, `LDEC_Z_MAX);
        hardDecisionsD1 = hardDecisionsRd;
        chkRespSgnD1 = chkRespSgn;   
        metricLastUseD1 = metricLastUse & !(bwdStall | bwdNoOp);
        iterStartD1 = iterStart;
        bwdStallD1 = bwdStall;
        endRowD1 = endRow;
        bwdNoOpD1 = bwdNoOp;
        rowStartD1 = rowStart;
        metricFirstUseD1 = metricFirstUse;
      end
      
      always @(*) // (col, parityStartCol, shrtCol)
      begin : pShrtNoPipe
        colShortenedD1 = 2'b11;
        if (col >= parityStartCol) begin
          colShortenedD1 = 2'b00;
        end
        else begin
          if (col < shrtCol[0]) begin
            colShortenedD1[0] = 1'b0;
          end
          if (col < shrtCol[1]) begin
            colShortenedD1[1] = 1'b0;
          end
        end
      end //pShrtNoPipe
      // for pShift
      always @(*) begin
        bwdStall_1D1 = bwdStall_1 | bwdNoOp_1;
        cycShift_1D1 = cycShift_1;
        notFirstUse = !(firstIteration_1 & metricFirstUse_1);
      end
      assign col_1D1 = col_1;
    end
  endgenerate // gPipeX
  
  //---------------------------------------------------------------------------
  // Cyclical Shifting
  //---------------------------------------------------------------------------   
  
  // The hardDecisions memory is in chkNode order from the last time it was written 
  // and needs to be in chknode order for this macro cell to align with varMetrics.
  // If PipeX is enabled begin hardDecisionsD1 will have been registered and
  // leftShift from pShift gets delayed a cycle.

  always @(posedge(clk) `LDEC_RESET_STR_B)
  begin : pShift
    
    reg signed [numBits(`LDEC_Z_MAX-1):0] shift;
    integer                               idx1;
    
    if (!nReset) begin
      leftShift <= `LDEC_PAD(1'b0, Z_HI);
      `LDEC_INITQ(cycLast, `LDEC_PAD(1'b0, Z_HI), `LDEC_NCOLS);
    end else begin
      if ( (bwdStall_1D1) == 1'b0) begin
        shift = $signed({1'b0, cycShift_1D1});
        if (notFirstUse == 1'b1) begin
          shift = shift - $signed({1'b0, cycLast[col_1D1]});
        end
        leftShift <= modZ(shift, Z_SIZES2(zEnum));
        cycLast[col_1D1] <= cycShift_1D1;
      end
    end

  end //pShift

  // Do the rotate (left shift) using a recursive function.
  assign hardDecisionsRot = ldpcDecRotateDownZ1(hardDecisionsD1, zEnum,
                                          numBits(`LDEC_Z_MAX)-1, 0, leftShift);

  //---------------------------------------------------------------------------
  // Variable Metric Update
  //---------------------------------------------------------------------------
      
  // Update the varMetric by adding the newly calculated check response to varResp. 
  // This is done without cyclical shifting so varMetric is mis-aligned. This will
  // be compensated for next time it is read.
 
  `LDEC_PACK(gCrP, chkRespAbsD1, chkRespAbsD1P, `LDEC_CHK_BITS-1, `LDEC_Z_MAX)
    
  always @(*) // (chkRespAbsD1, chkRespSgnD1, varResp)
  begin : pVarMetric
    
    reg signed [`LDEC_VAR_BITS:0]   vM;  // 1 bit headroom
    reg signed [`LDEC_CHK_BITS-1:0] cR;
    integer                         zI;
    
    for (zI=0; zI<=`LDEC_Z_MAX-1; zI=zI+1) begin
      
      cR = `LDEC_SGNMAG_TO_ONES(chkRespSgnD1[zI], chkRespAbsD1[zI]);
      
      vM = $signed({$signed(varResp[zI][`LDEC_VAR_BITS-1]), varResp[zI]}) +
           $signed({$signed({ (`LDEC_VAR_BITS+1-`LDEC_CHK_BITS) {cR[`LDEC_CHK_BITS-1]}}), cR});
      varMetric[zI] = `LDEC_CLIP_S(vM, `LDEC_VAR_BITS+1, `LDEC_VAR_BITS);      
    end    
  end //pVarMetric
  
  `LDEC_PACK(gVm, varMetric, varMetricP, `LDEC_VAR_BITS, `LDEC_Z_MAX)

  //---------------------------------------------------------------------------
  // varMetric Scaling Support
  //---------------------------------------------------------------------------
  
  // Zero the varMetrics used for scaling if shortened.
  always @(*) // (varMetric, colShortenedD1, shortDefault)
  begin : pVmShrt
    if (colShortenedD1[0] == 1'b1) begin
      varMetricAbs0 = shortDefault;
    end
    else begin
      varMetricAbs0 = `LDEC_ABS1(varMetric[0], `LDEC_VAR_BITS);
    end
    if (colShortenedD1[1] == 1'b1) begin
      varMetricAbs26 = shortDefault;
    end
    else begin
      varMetricAbs26 = `LDEC_ABS1(varMetric[26], `LDEC_VAR_BITS);
    end
  end //pVmShrt
  
  // Without the optional pipelines a combinatorial varMetricSum is passed to
  // the control block, however when there is pipelining before the use of the
  // scale flag in the fwd block (Pipe A, B or `LDEC_BWD_FWD_LATENCY > 1).
  // allowed extra cycle(s) and this is introduced by registering varMetricAbs
  // so that the calculation if the the varMetricSum does !stress the timing
  // in the primary datapath.

  generate
    if (`LDEC_PIPE_VMS > 0) begin: gPipeVms1
      always @(posedge(clk) `LDEC_RESET_STR)
      begin
        if (!nReset) begin
          iterStartD2 <= 1'b1;
          metricLastUseD2 <= 1'b0;
        end else begin
          iterStartD2 <= iterStartD1;
          metricLastUseD2 <= metricLastUseD1;
        end
      end
      always @(posedge(clk) `LDEC_RESET_STR_B)
      begin
        if (!nReset) begin
          varMetricAbs0D2 <= `LDEC_PAD(1'b0, `LDEC_VAR_BITS-2);
          varMetricAbs26D2 <= `LDEC_PAD(1'b0, `LDEC_VAR_BITS-2);          
        end else begin
          varMetricAbs0D2 <= varMetricAbs0;
          varMetricAbs26D2 <= varMetricAbs26;
        end
      end
    end
  endgenerate //  gPipeVms1
  
  generate
    if (`LDEC_PIPE_VMS == 0) begin: gPipeVms0
      always @(*) begin
        varMetricAbs0D2 = varMetricAbs0;
        varMetricAbs26D2 = varMetricAbs26;
        iterStartD2 = iterStartD1;
        metricLastUseD2 = metricLastUseD1;
      end
    end
  endgenerate //  gPipeVms0  
  
  always @(*) //(varMetricAbs0D2, varMetricAbs26D2, varMetricSumReg,
              // iterStartD2, metricLastUseD2)
  begin : pVarMetricSum
    
    reg [`LDEC_VAR_BITS-1+7:0] vMSum;
    
    vMSum = varMetricSumReg;   
    // Monitor varMetric mean levels. Sub sample with two samples per macro col. 
    // This sum will be used to decide if scaling is required on the next iter. 
    if (iterStartD2 == 1'b1) begin
      vMSum = `LDEC_PAD(1'b0, `LDEC_VAR_BITS-1+7);
    end
    if (metricLastUseD2 == 1'b1) begin
      vMSum = vMSum + `LDEC_PAD(varMetricAbs0D2, 8) +
                      `LDEC_PAD(varMetricAbs26D2, 8);
    end
    varMetricSum = vMSum;
  end //pVarMetricSum

  always @(posedge(clk) `LDEC_RESET_STR_B)
  begin : pVarMetricSumReg
    if (!nReset) begin
      varMetricSumReg <= `LDEC_PAD(1'b0, `LDEC_VAR_BITS-1+7);
    end else begin
      if (iterStartD2 == 1'b1 || metricLastUseD2 == 1'b1) begin
        varMetricSumReg <= varMetricSum;
      end
    end
  end //pVarMetricSumReg

  //---------------------------------------------------------------------------
  // Parity Check
  //---------------------------------------------------------------------------
           
  // Negative metric means logic 1 (LLR is log(Pr(0)/Pr(1))), but -ve sign;
  // denoted by a 1 (like in 2's complement). So varMetric gives hard-decision.

  // Compare new hard decision with previous for the current cell. To improve
  // timing we can register hardDecisions and varMetricSgn. Pipelining
  // here is independent of the main pipelines because N cycles latency in the
  // parity update only adds N cycles to the overall decode. The only difficult
  // bit is re-aligning results for logging.

  assign varMetricSgn = getSgnV(varMetricP);
  generate
    if (`LDEC_PIPE_PAR == 1) begin: gRegHd1
      always @(posedge(clk) `LDEC_RESET_STR_B)
      begin : pRegHd
        if (!nReset) begin
          hardDecisionsDHD <= `LDEC_PAD(1'b0, `LDEC_Z_MAX-1);
          varMetricSgnDHD <= `LDEC_PAD(1'b0, `LDEC_Z_MAX-1);
          bwdStallDHD <= 1'b0;
          endRowDHD <= 1'b0;
          bwdNoOpDHD <= 1'b0;
          rowStartDHD <= 1'b0;
          metricFirstUseDHD <= 1'b0;
        end else begin
          hardDecisionsDHD <= hardDecisionsRot;
          varMetricSgnDHD <= varMetricSgn;
          bwdStallDHD <= bwdStallD1;
          endRowDHD <= endRowD1;
          bwdNoOpDHD <= bwdNoOpD1;
          rowStartDHD <= rowStartD1;
          metricFirstUseDHD <= metricFirstUseD1;
        end
      end //pRegHd
    end
  endgenerate //  gRegHd1
  generate
    if (`LDEC_PIPE_PAR == 0) begin: gRegHd0
      always @(*) begin
        hardDecisionsDHD = hardDecisionsRot;
        varMetricSgnDHD = varMetricSgn;
        bwdStallDHD = bwdStallD1;
        endRowDHD = endRowD1;
        bwdNoOpDHD = bwdNoOpD1;
        rowStartDHD = rowStartD1;
        metricFirstUseDHD = metricFirstUseD1;
      end
    end
  endgenerate //  gRegHd0

  // Or reduce all the hard decision checks
  assign bitFlip = | ((hardDecisionsDHD ^ varMetricSgnDHD) & Z_MASK[zEnum]);

  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pParity
    
    reg  hadBitFlipV;
    reg [`LDEC_Z_MAX-1:0] parityVecV;
    
    if (!nReset) begin
      parityVec <= `LDEC_PAD(1'b0, `LDEC_Z_MAX-1);
      parityUpdate <= 1'b0;
      hadBitFlip <= 1'b0;
      // if (`LDEC_RESET_ALL) begin (Different from VHDL)
        parityVecV = `LDEC_PAD(1'b0, `LDEC_Z_MAX-1);
      // end
    end else begin
      parityUpdate <= endRowDHD & !bwdStallDHD;
      hadBitFlipV = hadBitFlip;
      if (enable == 1'b1 && (bwdStallDHD || bwdNoOpDHD) == 1'b0) begin
  
        // Detect && latch a bit-flip
        
        if (rowStartDHD == 1'b1) begin
          hadBitFlipV = 1'b0;
        end    
        // If the state of a bit has changed begin previous parity check 
        // passes that used that bit are probably wrong. Catch any bit flips
        // across the row.
        if (metricFirstUseDHD == 1'b0 && bitFlip == 1'b1) begin
          hadBitFlipV = 1'b1;
        end
        hadBitFlip <= hadBitFlipV;
  
        // Accumulate Parity for each of Z check equations.
        
        parityVecV = parityVec;
        if (rowStartDHD == 1'b1) begin
          parityVecV = `LDEC_PAD(1'b0, `LDEC_Z_MAX-1);
        end
        parityVecV = (varMetricSgnDHD ^ parityVecV) & Z_MASK[zEnum];
        parityVec <= parityVecV;
      end                           // enabled && !stalled
    end

  end //pParity

  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pParityErrs
    
    reg  [Z_HI:0]                parityCountV;
    reg                          armedForEndIterV;
    reg                          clrParityErrsV;
    
    if (!nReset) begin
      armedForEndIterV = 1'b0;
      parityErrs <= `LDEC_PAD(1'b0, PE_HI);
      parityCount <= `LDEC_PAD(1'b0, Z_HI);
      parityErrsUpdate <= 1'b0;
      clrParityErrsV = 1'b0;
    end else begin
      parityErrsUpdate <= 1'b0;
      parityCount <= `LDEC_PAD(1'b0, Z_HI);
      if (enable == 1'b0) begin
        armedForEndIterV = 1'b0;
        parityErrs <= `LDEC_PAD(1'b0, PE_HI);
        clrParityErrsV = 1'b0;
      end
      // Clear parity errs reg once it is sampled.
      if (clrParityErrsV == 1'b1 || newBlk == 1'b1) begin
        parityErrs <= `LDEC_PAD(1'b0, PE_HI);
        clrParityErrsV = 1'b0;
      end
      if (endIter == 1'b1) begin
        // Catch the end of iteration signal that occurs the cycle before
        // the parity update, which has T+1 timing.
        armedForEndIterV = 1'b1;
      end
      if (parityUpdate == 1'b1) begin
        parityErrsUpdate <= 1'b1;
        // Count num parity errs in this Z cell.
        if (`LDEC_EARLY_TERM_SUPPORT) begin
          parityCountV = countBits2(parityVec);
          parityCount <= parityCountV;
          // Add this cell's error count into parityErrs.
          parityErrs <= parityErrs + `LDEC_PAD(parityCountV, PE_HI-Z_HI) +
                        `LDEC_PAD(hadBitFlip, PE_HI);
        end
        else begin
          // Just count number of failing macro cells
          if (parityVec != `LDEC_PAD(1'b0, `LDEC_Z_MAX-1)) begin
            parityCount <= `LDEC_PAD(1'b0, Z_HI);
          end
          if (parityVec != `LDEC_PAD(1'b0, `LDEC_Z_MAX-1) || hadBitFlip == 1'b1) begin
            parityErrs <= parityErrs + `LDEC_PAD(1'b1, PE_HI);
          end
        end
        // We only get the newBlk = '1' case as a result of
        // abortIterations signal. In this case we need to
        // ensure parityErrs is cleared down.
        if (armedForEndIterV || newBlk) begin     
          armedForEndIterV = 1'b0;
          clrParityErrsV = 1'b1;
        end
      end
    end

  end //pParityErrs

`ifdef BRC_SIMU_ON
//synthesis translate_off
//synopsys translate_off
  ldpcDecBwdLog bwdLog
    (
     .clk              (ctrlClk),
     .zEnum            (zEnum),
     .enable           (enable),
     .chkRespSgn       (chkRespSgnD1),
     .chkRespAbs       (chkRespAbsD1P),
     .varMetricSgn     (varMetricSgn),
     .varMetric        (varMetricWrOut),
     .bwdStall         (bwdStallD1),
     .bwdNoOp          (bwdNoOpD1),
     .rowStart         (rowStartD1),
     .newBlk           (newBlk),
     .varMetricSum     (varMetricSum),
     .hadBitFlip       (hadBitFlip),
     .parityErrs       (parityErrs),
     .parityCount      (parityCount));
//synopsys translate_on
//synthesis translate_on
`endif //BRC_SIMU_ON
 
  assign hardDecisionsWrOut = varMetricSgn;
  assign chkRespWrSgnOut = chkRespSgnD1;
  assign chkRespWrAbsOut = chkRespAbsD1P;
  assign parityErrsOut = parityErrs;
  assign parityErrsUpdateOut = parityErrsUpdate;
  
  `LDEC_PACK(gCyc, cycLast, cycLastOut, Z_HI+1, `LDEC_NCOLS)
  `LDEC_PACK(gvMp, varMetric, varMetricWrOut, `LDEC_VAR_BITS, `LDEC_Z_MAX)

  // Put out a registered version of varMetricSum if timing allows.
  // Must be received in the ctrl block the cycle before start of fwd
  // iteration. With no pipelining a combinatorial path is required.
  // Note: `LDEC_PIPE_X and `LDEC_PIPE_VMS delay the calc of varmMetricSum.
  assign varMetricSumOut = (`LDEC_PIPE_ABCXL - `LDEC_PIPE_VMS - `LDEC_PIPE_X > 0 ) ?
                           varMetricSumReg : varMetricSum;
  
endmodule

 
