//-----------------------------------------------------------------------------
// ldcpDecBwdLog.v
// 
// Description
//   Logging of results to a file.
//  
// 30 Sel 2010 M. Rumsey - Moved from ldpcDecBwd
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//-----------------------------------------------------------------------------

`include "ldpcDec.vh"

module ldpcDecBwdLog
(
 input                                   clk,
 input [numBits(`LDEC_Z_ENUM_MAX)-1:0]   zEnum,
 input                                   enable,
 input `LDEC_sgnZType                    chkRespSgn,
 input `LDEC_chkRespAbsZ_P               chkRespAbs,
 input `LDEC_sgnZType                    varMetricSgn,
 input `LDEC_varZ_P                      varMetric,
 input                                   bwdStall,
 input                                   bwdNoOp,
 input                                   rowStart,
 input                                   newBlk,
 input [`LDEC_VAR_BITS-1+7:0]            varMetricSum,
 input                                   hadBitFlip,
 input [numBits(`LDEC_PAR_ERRS_MAX)-1:0] parityErrs,
 input [numBits(`LDEC_Z_MAX-1)-1:0]      parityCount);
  

`include "ldpcDecFuncs.vh"

  reg                        enableD;
  reg `LDEC_sgnZType         chkRespSgnD;
  reg `LDEC_chkAbsType       chkRespAbsD     `LDEC_chkRespAbsZ_R;
  reg `LDEC_sgnZType         varMetricSgnD;
  reg `LDEC_varType          varMetricD      `LDEC_varZ_R;
  reg                        bwdStallD;
  reg                        bwdStallD2;
  reg                        newBlkD1;
  reg                        bwdNoOpD;
  reg                        rowStartD;
  reg [`LDEC_VAR_BITS-1+7:0] varMetricSumD;
  reg                        hadBitFlipD;

`ifdef BRC_SIMU_ON
//synthesis translate_off
//synopsys translate_off
  integer vectorNum  = 0;

  //---------------------------------------------------------------------------
  // Re-time signals to a common point of reference.
  //---------------------------------------------------------------------------

  // The signals to log are generated at differing parts of a pipeline, however
  // they will be compared with vectors that are un-pipelined. The end of the
  // pipeline is parityErrs. The following lists out the
  // latencies of the logged signals.
  //                                 Latency          Dly Required
  // chkResp                         0                2 + `LDEC_PIPE_PAR
  // varMetric                       0                2 + `LDEC_PIPE_PAR
  // bwdStall, bwdNoOp, rowStart, En 0                2 + `LDEC_PIPE_PAR
  // varMetricSum                    `LDEC_PIPE_VMS        2 + `LDEC_PIPE_PAR - `LDEC_PIPE_VMS
  // hadBitFlip                      1 + `LDEC_PIPE_PAR    1
  // parityErrs, Count               2 + `LDEC_PIPE_PAR    0

  parameter DLY2P = (2 + `LDEC_PIPE_PAR);
  parameter DLY2PV = (2 + `LDEC_PIPE_PAR - `LDEC_PIPE_VMS);

  always @(posedge(clk))
  begin : pDly
    reg `LDEC_sgnZType         chkRespSgnDly     [1:DLY2P];
    reg `LDEC_chkRespAbsZ_P    chkRespAbsDly     [1:DLY2P];
    reg `LDEC_sgnZType         varMetricSgnDly   [1:DLY2P];
    reg `LDEC_varZ_P           varMetricDly      [1:DLY2P];
    reg                        bwdNoOpDly        [1:DLY2P];
    reg                        rowStartDly       [1:DLY2P];
    reg                        enableDly         [1:DLY2P];
    reg [`LDEC_VAR_BITS-1+7:0] varMetricSumDly   [1:DLY2PV];
    reg                        bwdStallDly       [1:DLY2P];
    integer                    idx;
    integer                    idx1;

    hadBitFlipD   <= hadBitFlip;

    for (idx = 2; idx<=DLY2P; idx = idx+1) begin
      chkRespSgnDly[idx]   = chkRespSgnDly[idx-1];  
      chkRespAbsDly[idx]   = chkRespAbsDly[idx-1];
      varMetricSgnDly[idx] = varMetricSgnDly[idx-1];
      varMetricDly[idx]    = varMetricDly[idx-1];
      bwdNoOpDly[idx]      = bwdNoOpDly[idx-1];
      rowStartDly[idx]     = rowStartDly[idx-1];
      enableDly[idx]       = enableDly[idx-1];
      bwdStallDly[idx]     = bwdStallDly[idx-1];
    end
    for (idx = 2; idx<=DLY2PV; idx = idx+1) begin    
      varMetricSumDly[idx] = varMetricSumDly[idx-1];
    end

    chkRespSgnDly[1]   = chkRespSgn;
    chkRespAbsDly[1]   = chkRespAbs;
    varMetricSgnDly[1] = varMetricSgn;
    varMetricDly[1]    = varMetric;
    bwdNoOpDly[1]      = bwdNoOp;
    rowStartDly[1]     = rowStart;
    enableDly[1]       = enable;
    varMetricSumDly[1] = varMetricSum;
    if (`LDEC_PIPE_X == 0) begin
      bwdStallDly[1]   = bwdStall;
    end else begin
      bwdStallDly[1]   = (bwdStall | newBlkD1);
    end
    
    chkRespSgnD   <= chkRespSgnDly[DLY2P];
    varMetricSgnD <= varMetricSgnDly[DLY2P];
    bwdNoOpD      <= bwdNoOpDly[DLY2P];
    rowStartD     <= rowStartDly[DLY2P];
    enableD       <= enableDly[DLY2P];
    varMetricSumD <= varMetricSumDly[DLY2PV];    
    bwdStallD     <= bwdStallDly[DLY2P];

    //chkRespAbsD   <= chkRespAbsDly[DLY2P];
    //varMetricD    <= varMetricDly[DLY2P];
   `LDEC_UNPACKQ(chkRespAbsDly[DLY2P], chkRespAbsD, `LDEC_CHK_BITS-1, `LDEC_Z_MAX);
   `LDEC_UNPACKQ(varMetricDly[DLY2P], varMetricD, `LDEC_VAR_BITS, `LDEC_Z_MAX);

  end
  
  // See ldpcDecVmMem notBwdStallD1, which controls updates to VM mem.
  // Here we need to emulate the timing and gating with newBlk.
  generate
    if (`LDEC_PIPE_X != 0) begin: gPipeX
      always @(posedge(clk))
      begin : pStall
        bwdStallD2 <= bwdStall;
        newBlkD1 <= newBlk;
      end //pStall
    end 
  endgenerate // gPipeX
  
  //---------------------------------------------------------------------------
  // Straightforward Logging of the re-timed information.
  //---------------------------------------------------------------------------
  
  always @(posedge(clk))
  begin : pLog
    
    integer wrInt;    
    integer z;
    integer i;
    reg     maskLogging;    
    
    if (enableD == 1'b0) begin
      maskLogging = 1'b1;
    end
    if (rowStart == 1'b1) begin
      maskLogging = 1'b0;
    end
    if (enableD == 1'b1 && bwdStallD == 1'b0 && maskLogging == 1'b0) begin
      if (ldpcDecGlobal.bwdLogFile != 0) begin
        vectorNum <= vectorNum + 1;
        $fwrite(ldpcDecGlobal.bwdLogFile, "%3d%2d%5d%7d", parityCount, hadBitFlipD,
               parityErrs, varMetricSumD);
        
        z = Z_SIZES2(zEnum);
        // Z wide data to memories
        for (i=0; i<=z-1; i=i+1) begin
          wrInt = $signed(varMetricD[i]);         
          if (bwdNoOpD == 1'b1) begin
            wrInt = 0;
          end
          $fwrite(ldpcDecGlobal.bwdLogFile, "%5d", wrInt);         
        end
        for (i=0; i<=z-1; i=i+1) begin
          // This is hardDecision logging.
          if (bwdNoOpD == 1'b1) begin
            $fwrite(ldpcDecGlobal.bwdLogFile, "%2d", 0);            
          end
          else begin
            $fwrite(ldpcDecGlobal.bwdLogFile, "%2d", varMetricSgnD[i] ? 1 : 0);            
          end
        end
        for (i=0; i<=z-1; i=i+1) begin
          wrInt = chkRespAbsD[i];
          if (chkRespSgnD[i] == 1'b1) begin
            wrInt = 0 - wrInt;
          end
          if (bwdNoOpD == 1'b1) begin
            wrInt = 0;
          end
          $fwrite(ldpcDecGlobal.bwdLogFile, "%4d", wrInt);
        end
        $fwrite(ldpcDecGlobal.bwdLogFile, "\n");
      end
    end
  end //pLog
//synthesis translate_on
//synopsys translate_on
`endif //BRC_SIMU_ON

endmodule
