//------------------------------------------------------------------------------
// ldcpDecCM.v
// 
// Description
//   Check Metric processing.
//
// Parameters:
// Inputs:
// (Programmable config)
// (From control)
//   fwdStall       : Fwd processing is stalled.
//   rowStart       : Start of row.
//   llrUnity       : Integer that represents unity (gets scaled each iter).
//   latchChkMetric : Latch the check metric results (end of fwd & bwd rows).
// (From ldpcDecFwd)
//   varRespCm      : The signal response clipped to chkBits-1.
//   varResp        : the full signal response ** For logging only **
// Outputs:
// (To Bwd)
//   chkMetricReg   : Soft sum of all varible responses for a row. (sgn+abs).
//
// 2 Apr 2010 M. Rumsey - Ported from C reference model.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`include "ldpcDec.vh"

module ldpcDecCM
( input                              clk,
  input                              nReset,
  input                              enable,
  input  [numBits(`LDEC_Z_MAX)-1:0]  z,           // not used in synth
  input                              fwdStall,
  input                              rowStart,
  input  `LDEC_chkAbsType            llrUnity,
  input                              latchChkMetric,
  input  `LDEC_sgnZType              varRespZSgn,
  input  `LDEC_chkRespAbsZ_P         varRespZCmAbsP, 
`ifdef BRC_SIMU_ON
//synthesis translate_off
//synopsys translate_off
  input  `LDEC_varZ_P                varRespZP, 
//synthesis translate_on
//synopsys translate_on
`endif //BRC_SIMU_ON
  output reg `LDEC_sgnZType          chkMetricRegSgnOut,
  output reg `LDEC_chkMetricAbsZ_P   chkMetricRegAbsOut);

`include "ldpcDecFuncs.vh"

  parameter PIPE_ABC = `LDEC_PIPE_A+`LDEC_PIPE_B+`LDEC_PIPE_C;

  // Packing/UnPacking
  wire  `LDEC_chkAbsType    varRespZCmAbs    `LDEC_chkRespAbsZ_R;
  reg   `LDEC_sgnZType      chkMetricSumSgn;
  reg   `LDEC_chkAbsType    chkMetricSumAbs  `LDEC_chkMetricAbsZ_R;
  genvar idx1; // packing

`LDEC_UNPACK(gVarRespCm, varRespZCmAbsP, varRespZCmAbs, `LDEC_CHK_BITS-1, `LDEC_Z_MAX)
  
`ifdef BRC_SIMU_ON
//synthesis translate_off
//synopsys translate_off
  wire  `LDEC_varType  varRespZ    `LDEC_varZ_R;  
  integer  vectorNum                   = 0;
  integer  zDly      [0:PIPE_ABC];
  reg      started                     = 1'b0;

  `LDEC_UNPACK(gVarResp, varRespZP, varRespZ, `LDEC_VAR_BITS, `LDEC_Z_MAX)
  
  // We have to delay z to match the logging pipelining. This is
  // relevant when the decoder switches zEnum.
  generate 
    if (PIPE_ABC == 0) begin: gZDly
      always @(*) begin
         zDly[0] = z;
      end
    end else begin

      always @(posedge(clk))
      begin : pDly
        integer I;
        for (I=0; I<=PIPE_ABC-1; I=I+1) begin
          if (I == PIPE_ABC-1) begin
            zDly[I] <= z;
          end
          else begin
            zDly[I] <= zDly[I+1];
          end
        end
      end //pDly
    end
  endgenerate //  gZDly
//synopsys translate_on
//synthesis translate_on
`endif //BRC_SIMU_ON
  
  always @(posedge(clk) `LDEC_RESET_STR_B)
  begin : pChkMetric
    
    reg `LDEC_chkAbs3_P      cAbs;
    reg `LDEC_sgnZType       chkMetricSumSgnV;
    reg `LDEC_chkAbsType     chkMetricSumAbsV    `LDEC_chkMetricAbsZ_R;
    integer zI;
    integer idx1;
    integer i;

    if (nReset == 1'b0) begin
      chkMetricSumSgn <= `LDEC_PAD(1'b0, `LDEC_Z_MAX-1);
      chkMetricRegSgnOut <= `LDEC_PAD(1'b0, `LDEC_Z_MAX-1);
      `LDEC_INIT2Q(chkMetricSumAbs, `LDEC_PAD(1'b0, `LDEC_CHK_BITS-1),
                   `LDEC_Z_MAX, `LDEC_CHK_METRIC_RIGHT+1);
      chkMetricRegAbsOut <= `LDEC_PAD(1'b0, `LDEC_Z_MAX*(`LDEC_CHK_METRIC_RIGHT+1)*
                                      (`LDEC_CHK_BITS-1)-1);
    end else begin

`ifdef BRC_SIMU_ON
//synthesis translate_off
//synopsys translate_off   
      // Just for logging, so we know when to start.
      if (enable == 1'b0) begin
        started = 1'b0;
      end
      else if (rowStart == 1'b1) begin
        started = 1'b1;
      end
      // Log some control signals
      if (fwdStall == 1'b0 && started == 1'b1 && ldpcDecGlobal.fwdLogFile != 0)
      begin
        $fwrite(ldpcDecGlobal.fwdLogFile, "%1d", rowStart);
      end
//synopsys translate_on
//synthesis translate_on
`endif //BRC_SIMU_ON
      
      // Process signs
      chkMetricSumSgnV = chkMetricSumSgn;  // default
      if (fwdStall == 1'b0) begin
        if (rowStart == 1'b0) begin
          chkMetricSumSgnV = varRespZSgn ^ chkMetricSumSgn;
        end
        else begin
          chkMetricSumSgnV = varRespZSgn;
        end
      end
      chkMetricSumSgn <= chkMetricSumSgnV;
  
      // Process Abs values
      
      // default value if stalled.
      `LDEC_COPY2(chkMetricSumAbs, chkMetricSumAbsV, `LDEC_Z_MAX,
                  `LDEC_CHK_METRIC_RIGHT+1);
      for (zI=0; zI<=`LDEC_Z_MAX-1; zI=zI+1) begin
        if (fwdStall == 1'b0) begin

          `LDEC_PACKF(chkMetricSumAbs[zI], cAbs, `LDEC_CHK_BITS-1,
                      `LDEC_CHK_METRIC_RIGHT+1);
          // Create the check response forward pass sum.
//          if (`LDEC_MINSUM_ALG > 1) begin
            cAbs = softAdd(cAbs, varRespZCmAbs[zI], rowStart, llrUnity);
//          end else begin
//            cAbs = softAddB(cAbs, varRespZCmAbs[zI], rowStart);            
//          end
          `LDEC_UNPACKF(cAbs, chkMetricSumAbsV[zI], `LDEC_CHK_BITS-1,
                        `LDEC_CHK_METRIC_RIGHT+1);
          
`ifdef BRC_SIMU_ON
//synthesis translate_off
//synopsys translate_off   
          // Log the chk Metric
          if (zI < zDly[0] && ldpcDecGlobal.fwdLogFile != 0 && started == 1'b1) begin
  
            // Without correction the 3rd element is the same as the first
            // so is redundant.
            $fwrite(ldpcDecGlobal.fwdLogFile, "%2d%3d%3d%3d",
                   chkMetricSumSgnV[zI], cAbs[0], cAbs[1],
                    (`LDEC_CHK_METRIC_RIGHT == 2) ? cAbs[2] :  cAbs[0]);          
  
          end
//synopsys translate_on
//synthesis translate_on
`endif //BRC_SIMU_ON
          
        end  
      end
      `LDEC_COPY2Q(chkMetricSumAbsV, chkMetricSumAbs, `LDEC_Z_MAX,
                   `LDEC_CHK_METRIC_RIGHT+1);
      
      // Register to pass to backward block.
      if (latchChkMetric ==1'b1) begin       // n.b. may occur during stall.
        chkMetricRegSgnOut <= chkMetricSumSgnV;
        `LDEC_PACK2Q(chkMetricSumAbsV, chkMetricRegAbsOut,
                     `LDEC_CHK_BITS-1, `LDEC_Z_MAX, `LDEC_CHK_METRIC_RIGHT+1);
      end
      
`ifdef BRC_SIMU_ON
//synthesis translate_off
//synopsys translate_off
      if ((ldpcDecGlobal.fwdLogFile != 0) && (fwdStall == 1'b0 && started == 1'b1)) begin
        vectorNum <= vectorNum + 1;
        for (i=0; i<=zDly[0]-1; i=i+1) begin
          $fwrite(ldpcDecGlobal.fwdLogFile, "%5d", $signed(varRespZ[i]));
        end
        $fwrite(ldpcDecGlobal.fwdLogFile, "\n");
      end
//synopsys translate_on
//synthesis translate_on
`endif //BRC_SIMU_ON
    end

  end //pChkMetric

endmodule
