//------------------------------------------------------------------------------
// ldcpDecCore.v
// 
// Description
//   Core functionality for the LDPC decoder. Registers, clock control and
//   (optionally) memory are provided by the parent wrapper.
// 
// Inputs:
//   nReset         : Asynchronous reset.
// (From the processor)
//   blkSelect      : Active high select for the block.
//   writeNotRead   : Active high write enable.
//   addr           : Address bus from processor.
//   wrData         : Write data bus from processor.
// (From sender (e.g. equaliser))
//   inStrobe       : High when inData is valid.
//   ipWidth        : The number of LLRs that are valid values.
//   inDataWord     : `IP_WIDTH samples of `IP_BITS.
//   llrUnity       : Integer that represents unity (gets scaled each iter).
// Outputs:
// (To sender (e.g. demapper))
//   rdyToRcv       : Enables the sender to start sending. Stays high until
//                    the expected amount of data has been transferred.
//   lastIpSample   : Synchronous to the last input sample of the LDPC block.
// (To downstream hardware)
//   decodeComplete : A pulse indicating completion of decode.
//   decodeStatus   : 0/1 for fail/pass. Valid on decodeComplete.
//   packetComplete : 1 indicates all blocks have been processed. Set on last
//                    decodeComplete pulse. Cleared when enable reg cleared.
//   packetStatus   : Starts 1 but cleared on first block decode fail.
//   opStrobe       : When 1 the data on opDataWord is valid.
//   opDataWord     : Decoded output data.
// (To the processor)
//   rdDataOut      : Read data bus.
// (To wrapper)
//   registers
//   clocking
//   memory
//
// 20 Apr 2010 M. Rumsey. Created.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`default_nettype wire
`include "ldpcDec.vh"

module ldpcDecCore # (
 parameter NUM_USERS = 1 )
(
 input                                    nReset,
 // Inputs from sender
 input                                    inStrobe,
 input [numBits(`LDEC_IP_WIDTH)-1:0]      ipWidth,
 input [`LDEC_IP_WIDTH*`LDEC_IP_BITS-1:0] inDataWord,
 input                                    ipSkipBlock,
 output                                   ipLoaded,
 output [numBits(`LDEC_MAX_IPBITS)-1:0]   numIpBits,
 output                                   numIpBitsValid,
 input [`LDEC_CHK_BITS-1:0]               llrUnity,
 input                                    iterationAbort,
 input [4:0]                              inUser,
 output [4:0]                             opUser,
 // Inputs from output interface
 input                                    clrToSend,
 // Outputs to sender
 output                                   rdyToRcvOut,
 // Outputs to downstream hardware
 output                                   decodeCompleteOut,
 output                                   decodeStatusOut,
 output                                   packetCompleteOut,
 output                                   packetOpCompleteOut,
 output                                   packetStatusOut,
 output                                   opDoneOut,
 output                                   opStrobeOut,
 output [`LDEC_OP_WIDTH-1:0]              opDataWordOut,
 output [`LDEC_OP_WIDTH/8-1:0]            opByteEnOut,
 output [numBits(`LDEC_OP_WIDTH)-1:0]     opBitEnOut,
 //-------------------------------------------------------------------------
 // Registers
 //-------------------------------------------------------------------------
 // Master enable to control.
 input [NUM_USERS-1:0]                    enabledUsers,
 input                                    enableIp,
 input                                    enLostTime,                        // !!!!! not included in MU mode
 input                                    beatTimeLine,                      // !!!!! not included in MU mode
 // To mem block to program the code cell information.
 input                                    cellAccess,                        // !!!!! not included in MU mode
 input                                    cellRead,                          // !!!!! not included in MU mode
 input                                    cellWrite,                         // !!!!! not included in MU mode
 input [`LDEC_CELL_RAM_W-1:0]             cellWrData,                        // !!!!! not included in MU mode
 // Code characteristics to control.
 input [numBits(`LDEC_Z_ENUM_MAX)-1:0]    zEnumU0,                           // Per user
 input [numBits(`LDEC_R_ENUM_MAX)-1:0]    rEnumU0,                           // Per user
 input [numBits(`LDEC_BPS_MAX)-1:0]       bitsPerSymbolU0,                   // Per user
 // Packet Management
 input [numBits(`LDEC_MAX_BLKNUM)-1:0]    RegNcwU0,                          // Per user
 input [15:0]                             packetBytesLs,                     // !!!!! not included in MU mode
 input [`LDEC_PB_MS_LEFT:0]               packetBytesMs,                     // !!!!! not included in MU mode
 input [15:0]                             frameEndByteLs,                    // !!!!! not included in MU mode
 input [`LDEC_PB_MS_LEFT+1:0]             frameEndByteMs,                    // !!!!! not included in MU mode
 // Shortening etc parameters to control.
 input [numBits(`LDEC_N_MAX-1)-1:0]       nShrtFloorU0,                      // Per user
 input [numBits(`LDEC_MAX_BLKNUM)-1:0]    shrtModU0,                         // Per user
 input [numBits(`LDEC_M_MAX-1)-1:0]       nPuncFloorU0,                      // Per user
 input [numBits(`LDEC_MAX_BLKNUM)-1:0]    puncModU0,                         // Per user
 input [numBits(`LDEC_NREPS_MAX)-1:0]     nRepFloorU0,                       // Per user
 input [numBits(`LDEC_MAX_BLKNUM)-1:0]    repModU0,                          // Per user
 input [`LDEC_TARGET_LEVEL_LEFT:0]        targetLevel,                       // !!!!! not included in MU mode
 input [numBits(`LDEC_K_MAX)-1:0]         parityThresh,                      // !!!!! not included in MU mode
 input [numBits(8*`LDEC_MAX_ITER)-1:0]    nomIterations,                     // !!!!! not included in MU mode
 input [numBits(`LDEC_MAX_ITER)-1:0]      earlyTestIterations,               // !!!!! not included in MU mode
 input [numBits(8*`LDEC_MAX_ITER)-1:0]    maxRunningCount,                   // !!!!! not included in MU mode
 input [numBits(8*`LDEC_MAX_ITER)-1:0]    endRunningCount,                   // !!!!! not included in MU mode
 // Status registers
 output [numBits(`LDEC_MAX_BLKNUM)-1:0]   blkErrsOut,
 output [numBits(`LDEC_MAX_ITER)-1:0]     curIterationOut,
 output [numBits(`LDEC_PAR_ERRS_MAX)-1:0] prevParityErrsOut,
 output [numBits(`LDEC_MAX_ITER)-1:0]     prevIterationsOut,
 output                                   prevDecodeStatusOut,
 output [`LDEC_VAR_BITS+5:0]              varMetricChkSumOut,
 output [`LDEC_CELL_RAM_W-1:0]            cellRdDataOut,
 //-------------------------------------------------------------------------
 // Clocking
 //-------------------------------------------------------------------------
 input                                    freeClk, // RW modification
 input                                    regsClk,
 input                                    ipClk,
 input                                    opClk,
 input                                    decClk,
 input                                    cellClk,
 input                                    vmRam1Clk, 
 input                                    vmRam2Clk, 
 input                                    hdRamClk, 
 input                                    crRamClk, 
 input                                    ctrlClk,
 output                                   ipClkEnOut,
 output                                   opClkEnOut, 
 output                                   decClkEnOut,
 output                                   cellClkEnOut,
 output                                   vmRam1ClkEnOut, 
 output                                   vmRam2ClkEnOut, 
 output                                   hdRamClkEnOut, 
 output                                   crRamClkEnOut, 
 output                                   ctrlClkEnOut,
`ifdef LDEC_RAMS_AT_TOP
 // Memories (when `LDEC_RAMS_AT_TOP)
`ifndef LDEC_USE_CELL_ROM
 input [`LDEC_CELL_RAM_O-1:0]             cellRamOps,
 output [`LDEC_CELL_RAM_I-1:0]            cellRamIps,
`endif
 input [`LDEC_VM_RAM_O-1:0]               vmRamOps,
 output [`LDEC_VM_RAM_I-1:0]              vmRamIps,
 input [`LDEC_HD_RAM_O-1:0]               hdRamOps,
 output [`LDEC_HD_RAM_I-1:0]              hdRamIps,
 // For the cr && vr RAMS there are 2 port && 1 port options with slightly
 // different signal lists, hence signals are provided for both cases though
 // only one will be used at a time.
`ifdef LDEC_2_PORTS
 input [`LDEC_CR_RAM_O-1:0]               crRamOps,
 output [`LDEC_CR_RAM_I-1:0]              crRamIps,
 input [`LDEC_VR_RAM_O-1:0]               vrRamOps,
 output [`LDEC_VR_RAM_I-1:0]              vrRamIps,
`else
 input [`LDEC_CR1_RAM_O-1:0]              cr1RamOps,
 output [`LDEC_CR1_RAM_I-1:0]             cr1RamIps,
 input [`LDEC_VR1_RAM_O-1:0]              vr1RamOps,
 output [`LDEC_VR1_RAM_I-1:0]             vr1RamIps,
`endif
`endif
 //----------------------------
 //  Debug Port
 //----------------------------
 // RW MODIFIED
 output [15:0]                            dbgIter,
 output [15:0]                            dbgIp,
 // END RW MODIFIED
 output [`LDEC_DBG_PORT_WIDTH-1:0]        dbgIp2Out,
 output [`LDEC_DBG_PORT_WIDTH-1:0]        dbgOpOut,
 output [`LDEC_DBG_PORT_WIDTH-1:0]        dbgCtrlOut);
    
`include "ldpcDecFuncs.vh"

 localparam WROFFSET_MAX = maximum(ceilDiv(`LDEC_Z_MAX, `LDEC_DEC_RAM_IP_WIDTH)-1, 1);
 localparam frameEndByteMsLeft = `LDEC_PB_MS_LEFT+1;
 localparam dbgIp1OutLeft = `LDEC_DBG_PORT_WIDTH-1;
  
  //---------------------------------------------------------------------------
  // Status signals readable by processor.
  //---------------------------------------------------------------------------

  wire [numBits(`LDEC_MAX_BLKNUM)-1:0]   blkErrs;
  wire [numBits(`LDEC_MAX_ITER)-1:0]     curIteration;
  wire [numBits(`LDEC_PAR_ERRS_MAX)-1:0] prevParityErrs;
  wire [numBits(`LDEC_MAX_ITER)-1:0]     prevIterations;
  wire                                   prevDecodeStatus;
  wire [`LDEC_VAR_BITS+5:0]              varMetricChkSum;

  //---------------------------------------------------------------------------
  // Controller outputs (that aren't mentioned elsewhere)
  //---------------------------------------------------------------------------

  wire                              blkPingPongE;  // toggles at decode end
  wire                              blkPingPongS;   // toggles at decode start
  wire                              decodeActive;
  wire                              decodeActive_1;
  wire                              decodeActive_2;
  wire                              decodeActive_2D;
  wire                              disabling;
  // To Ip
  wire                              setRdyToRcv;
  wire                              decodeStart;
  wire                              firstBlk;
  // To Op
  wire                              newBlkPulse;
  wire                              packetComplete;
  // To Fwd 
  wire                              fwdStall;
  wire                              fwdStall_1;
  wire                              fwdStall2;
  wire                              fwdStall2_1;
  wire                              fwdNoOp;
  wire                              fwdNoOp_1;
  wire                              fwdFirstIteration;
  wire                              fwdFirstIteration_1;
  wire                              fwdRowStart;
  wire                              scaleFlag;
  wire `LDEC_chkAbsType             fwdLlrUnity;
  wire                              metricVeryFirstUse;
  // lint: in sime configurations this is not used.
  wire                              metricVeryFirstUse_1;
  wire                              latchChkMetric;
  // To Bwd
  wire                              bwdStall;
  wire                              bwdStall_1;
  wire                              stallVr;
  wire                              bwdStall2;
  wire                              bwdStall2_1;
  wire                              bwdNoOp;
  wire                              bwdNoOp_1;
  wire                              bwdCellPrime;
  wire                              bwdIterStart;
  wire                              fwdIterStart_1;
  wire                              bwdFirstIteration_1;
  wire                              bwdRowStart;
  wire                              bwdRowStart_1;
  // Lint: this signal is only read in some configurations of the hardware.
  wire                              bwdRowStart_2;
  wire                              bwdEndIter;
  wire [`LDEC_CHK_BITS-3:0]         bwdLlrUnityDiv2;
  // To Mem
  wire [numBits(`LDEC_NCELLS-1)-1:0] fwdCellIdx;
  wire [numBits(`LDEC_NCELLS-1)-1:0] bwdCellIdx;
  wire                              holdFwdCell_1;
  wire                              holdBwdCell_1;

  // Pipeline delayed signals
  reg                               crRdStartD1;
  reg                               crStallD1;
  reg                               fwdRowStartD1;
`ifndef LDEC_2_PORTS
  reg                               decodeActiveD1;
  reg                               decodeActiveD2;
`endif
  reg                               fwdRowStartD2;
  reg                               fwdStallD1;
  reg                               fwdStallD2;
  reg                               bwdIterStartD1;
  reg                               bwdStallD1;
  wire                              bwdRowStart_VR;

  //---------------------------------------------------------------------------
  // Ip Outputs
  //---------------------------------------------------------------------------
  
  wire `LDEC_chkAbsType             llrUnityInitial0;
  wire `LDEC_chkAbsType             llrUnityInitial1;
  wire                              ipAlmostLoaded; // RW modification
  // To mem block
  wire [`LDEC_DEC_RAM_IP_WIDTH*`LDEC_VAR_BITS-1:0] ipWrData;

  wire [numBits(WROFFSET_MAX)-1:0]       ipWrOffset;
  wire [numBits(`LDEC_NCOLS-1)-1:0]      ipWrAddr;
  wire                                   ipWrEnable;
  // From mem block
  wire [`LDEC_VM_RAM_W-1:0]              ipRdData;
  wire [numBits(`LDEC_NCOLS-1)-1:0]      ipRdAddr;
  wire                                   ipRdEnable;
  // To Clocking
  wire                                   ipClkEn;
  // To Control
  wire                                   lastBlkOfSym;
  wire                                   waitFirstWordOfSym;
  
  //---------------------------------------------------------------------------
  // Op Outputs
  //---------------------------------------------------------------------------

  wire                                   hdRamSel;
  wire [`LDEC_VM_RAM_A-1:0]              hdRamAddr;
  wire                                   opBufferLocked;
  
  //---------------------------------------------------------------------------
  // Bwd Outputs
  //---------------------------------------------------------------------------

  // To Ctrl
  wire                                   parityErrsUpdate;
  wire [numBits(`LDEC_PAR_ERRS_MAX)-1:0] parityErrs;
  wire [`LDEC_VAR_BITS-1+7:0]            varMetricSum;
  // To IO           
  wire `LDEC_cycShift_P                  cycLast;
  wire                                   decodeComplete;
  // To memories     
  wire `LDEC_varSlvZType                 varMetricWrSlv;
  wire `LDEC_sgnZType                    hardDecisionsWr;
  wire `LDEC_chkRespSlvZType             chkRespWr;
  wire `LDEC_sgnZType                    chkRespWrSgn;
  wire `LDEC_chkRespAbsZ_P               chkRespWrAbs;
  
  //---------------------------------------------------------------------------
  // Fwd Outputs
  //---------------------------------------------------------------------------

  wire `LDEC_sgnZType                    chkMetricRegSgn;
  wire `LDEC_chkMetricAbsZ_P             chkMetricRegAbs;
  wire `LDEC_shrtCol_P                   shrtCol;
  // To memories 
  wire `LDEC_varSlvZType                 varRespWrSlv;
  
  //---------------------------------------------------------------------------
  // VmMem (Variable Metric) Block Outputs
  //---------------------------------------------------------------------------

  // To Ctrl && fwd/bwd     
  wire [`LDEC_FLAG_BITS-1:0]             fwdFlag;
  wire [`LDEC_FLAG_BITS-1:0]             bwdFlag;
  wire [`LDEC_FLAG_BITS-1:0]             fwdFlag_1;
  wire [`LDEC_FLAG_BITS-1:0]             bwdFlag_1;
  wire [numBits(`LDEC_NCELLS)-1:0]       numMacroCells;
  // To Fwd                              
  wire `LDEC_varSlvZType                 varMetricRdSlv;
  wire [numBits(`LDEC_NCOLS_RIGHT)-1:0]  fwdCol_1;
  wire [numBits(`LDEC_Z_MAX-1)-1:0]      fwdCycShift_1;
  wire [numBits(`LDEC_NCOLS_RIGHT)-1:0]  fwdCol;
  wire [numBits(`LDEC_Z_MAX-1)-1:0]      fwdCycShift;
  // To Bwd                              
  wire `LDEC_sgnZType                    hardDecisionsRd;
  wire [numBits(`LDEC_NCOLS_RIGHT)-1:0]  bwdCol_1;
  wire [numBits(`LDEC_NCOLS_RIGHT)-1:0]  bwdCol;
  wire [numBits(`LDEC_Z_MAX-1)-1:0]      bwdCycShift_1;
  // To Op                               
  wire `LDEC_sgnZType                    hdRamData;

  //---------------------------------------------------------------------------
  // CrMem (Check Response) Block Outputs
  //---------------------------------------------------------------------------

  wire `LDEC_chkRespSlvZType             chkRespRd;
  wire `LDEC_sgnZType                    chkRespRdSgn;
  wire `LDEC_chkRespAbsZ_P               chkRespRdAbs;
  
  //---------------------------------------------------------------------------
  // VrMem (Variable Response) Block Outputs
  //---------------------------------------------------------------------------

  wire `LDEC_varSlvZType                 varRespRdSlv;

  // Other 
  wire                                   shrtBlkModDec;
  wire                                   shrtBlkModOp;
  wire [numBits(`LDEC_K_MAX)-1:0]        k2;
  
  //---------------------------------------------------------------------------
  // Values that are looked up based on ENUMS
  //---------------------------------------------------------------------------

  localparam K_BITS = numBits(`LDEC_K_MAX);
  localparam N_BITS = numBits(`LDEC_N_MAX);
  wire [numBits(`LDEC_Z_MAX)-1:0]                                  zArray[0:NUM_USERS-1];
  wire [numBits(ceilDiv(`LDEC_Z_MAX, `LDEC_DEC_RAM_IP_WIDTH))-1:0] numWrPerZArray[0:NUM_USERS-1];  
  wire [numBits(`LDEC_N_MAX)-1:0]                                  nArray[0:NUM_USERS-1];
  wire [numBits(`LDEC_K_MAX)-1:0]                                  kArray[0:NUM_USERS-1];
  wire [numBits(`LDEC_NCOLS-1)-1:0]                                parityStartColArray[0:NUM_USERS-1];

  //---------------------------------------------------------------------------
  //Other
  //---------------------------------------------------------------------------

  reg `LDEC_varAbsType shortDefault;
  reg  [`LDEC_PB_MS_LEFT+16:0] frameEndByte;
  
  //---------------------------------------------------------------------------
  // User Control
  //---------------------------------------------------------------------------
  wire [4:0] ctrlUserDec;
  wire [4:0] ctrlUserOp;
  wire       enableAnyUser;
  wire       enableOp;
  wire       blockOpComplete;

  //---------------------------------------------------------------------------
  // Debug port
  //---------------------------------------------------------------------------

  // Local copies of outputs.
  wire lastIpSample;
  wire rdyToRcv;
  wire [`LDEC_OP_WIDTH-1:0] opDataWord;
  wire [`LDEC_OP_WIDTH/8-1:0] opByteEn;
  genvar                      idx1;
  
  assign enableAnyUser = |enabledUsers;
  
  // Map per user registers to arraies
  wire [numBits(`LDEC_Z_ENUM_MAX)-1:0]    zEnumArray[0:NUM_USERS-1];          
  wire [numBits(`LDEC_R_ENUM_MAX)-1:0]    rEnumArray[0:NUM_USERS-1];          
  wire [numBits(`LDEC_BPS_MAX)-1:0]       bitsPerSymbolArray[0:NUM_USERS-1];  
  wire [numBits(`LDEC_MAX_BLKNUM)-1:0]    RegNcwArray[0:NUM_USERS-1];         
  wire [numBits(`LDEC_N_MAX-1)-1:0]       nShrtFloorArray[0:NUM_USERS-1];     
  wire [numBits(`LDEC_MAX_BLKNUM)-1:0]    shrtModArray[0:NUM_USERS-1];        
  wire [numBits(`LDEC_M_MAX-1)-1:0]       nPuncFloorArray[0:NUM_USERS-1];     
  wire [numBits(`LDEC_MAX_BLKNUM)-1:0]    puncModArray[0:NUM_USERS-1];        
  wire [numBits(`LDEC_NREPS_MAX)-1:0]     nRepFloorArray[0:NUM_USERS-1];      
  wire [numBits(`LDEC_MAX_BLKNUM)-1:0]    repModArray[0:NUM_USERS-1];         
  
  assign zEnumArray[0]         = zEnumU0;        
  assign rEnumArray[0]         = rEnumU0;        
  assign bitsPerSymbolArray[0] = bitsPerSymbolU0;
  assign RegNcwArray[0]        = RegNcwU0;       
  assign nShrtFloorArray[0]    = nShrtFloorU0;   
  assign shrtModArray[0]       = shrtModU0;      
  assign nPuncFloorArray[0]    = nPuncFloorU0;   
  assign puncModArray[0]       = puncModU0;      
  assign nRepFloorArray[0]     = nRepFloorU0;    
  assign repModArray[0]        = repModU0;       

  // Lookup various parameters based on enums.
genvar lu_idx;
generate
for (lu_idx=0;lu_idx<NUM_USERS;lu_idx=lu_idx+1) begin : gLu

   ldpcDecLookup lookup
     (
      .nReset            (nReset),
      .clk               (regsClk),
      .zEnum             (zEnumArray[lu_idx]),
      .rEnum             (rEnumArray[lu_idx]),
      .zOut              (zArray[lu_idx]),
      .numWrPerZOut      (numWrPerZArray[lu_idx]),
      .nOut              (nArray[lu_idx]),
      .kOut              (kArray[lu_idx]),
      .parityStartColOut (parityStartColArray[lu_idx]));

end
endgenerate     

  
  // TODO NOT INCLUDED IN MU MODE
  // Capture frameEndByte, which is written as 2 regs with the top
  // of the second reg used to indicate a change.
  generate
    if (`LDEC_SPECIAL_AGG_SUPPORT) begin: gAggMode
      always @(posedge(ctrlClk) `LDEC_RESET_STR)
      begin : pFrameEnd
        reg  lastTopBitV;
        if (nReset == 1'b0) begin
          lastTopBitV = 1'b0;
          frameEndByte <= `LDEC_PAD(1'b0, `LDEC_PB_MS_LEFT+16);
        end else begin                               
          if (enableAnyUser == 1'b0) begin
            lastTopBitV = 1'b0;
            frameEndByte <= `LDEC_PAD(1'b0, `LDEC_PB_MS_LEFT+16);          
          end
          else if (frameEndByteMs[frameEndByteMsLeft] != lastTopBitV || `LDEC_BUS32) begin
            lastTopBitV = frameEndByteMs[frameEndByteMsLeft];
            frameEndByte <= {frameEndByteMs[frameEndByteMsLeft-1:0],
                            frameEndByteLs};
          end
        end    
      end //pFrameEnd
    end
  endgenerate // gAggMode
  // END OF TODO NOT INCLUDED IN MU MODE
  
   wire [7:0] dbgBlkNum;
   
   ldpcDecCtrl # (
     .NUM_USERS               (NUM_USERS)
   ) ctrl (
      .nReset                 (nReset),
      .clk                    (ctrlClk),
      // Master enable from Regs.
      .enable                 (enableAnyUser),
      .enLostTime             (enLostTime),
      .beatTimeLine           (beatTimeLine),
      // User control
      .ctrlUserIn             (inUser),
      .ctrlUserDec            (ctrlUserDec),
      .ctrlUserOp             (ctrlUserOp),
      .enabledUsers           (enabledUsers),
      .enableIp               (enableIp),
      .enableOp               (enableOp),
      .ipSkipBlock            (ipSkipBlock),
      .blockOpComplete        (blockOpComplete),
      // Code characteristics from Regs.
      .packetLenU0            (RegNcwU0),           // only one user supported for now
      .packetBytesLs          (packetBytesLs),
      .packetBytesMs          (packetBytesMs),
      .frameEndByte           (frameEndByte),
      .k                      (kArray[0]),
      .nShrtFloor             (nShrtFloorArray[0]),
      .shrtModU0              (shrtModU0),          // only one user supported for now
      // From Ip block
      .lastBlkOfSym           (lastBlkOfSym),
      .waitFirstWordOfSym     (waitFirstWordOfSym),        
      // Shortening etc parameters from Regs.
      .parityThresh           (parityThresh),
      .targetLevel            (targetLevel),
      .llrUnity0              (llrUnityInitial0),
      .llrUnity1              (llrUnityInitial1),
      .nomIterations          (nomIterations),
      .earlyTestIterations    (earlyTestIterations),
      .maxRunningCount        (maxRunningCount),
      .endRunningCount        (endRunningCount),
      // External override to force current block to stop
      .iterationAbort         (iterationAbort),
      // From Bwd
      .parityErrsUpdate       (parityErrsUpdate),
      .parityErrs             (parityErrs),
      .varMetricSum           (varMetricSum),
      // From Mem
      .fwdFlag                (fwdFlag),
      .bwdFlag                (bwdFlag),
      .fwdFlag_1              (fwdFlag_1),
      .bwdFlag_1              (bwdFlag_1),
      .fwdCol_1               (fwdCol_1),
      .bwdCol_1               (bwdCol_1),
      .numMacroCells          (numMacroCells),
      // From Ip
      .ipLoaded               (ipLoaded),
      // From Op
      .opBufferLocked         (opBufferLocked),
      .packetOpComplete       (packetOpCompleteOut),
      // To Clocking
      .clkEnOut               (ctrlClkEnOut),
      .disablingOut           (disabling),
      // To Ip
      .setRdyToRcvOut         (setRdyToRcv),
      .decodeStartOut         (decodeStart),      
      .firstBlkOut            (firstBlk),
      // To decoding
      .shrtBlkModDec          (shrtBlkModDec),
      // To Fwd
      .fwdStallOut            (fwdStall),
      .fwdStallOut_1          (fwdStall_1),
      .fwdNoOpOut             (fwdNoOp),
      .fwdNoOpOut_1           (fwdNoOp_1),
      .fwdFirstIterationOut   (fwdFirstIteration),
      .fwdFirstIterationOut_1 (fwdFirstIteration_1),
      .fwdIterStartOut_1      (fwdIterStart_1),
      .metricVeryFirstUseOut  (metricVeryFirstUse),
      .metricVeryFirstUseOut_1 (metricVeryFirstUse_1),
      .fwdRowStartOut         (fwdRowStart),
      .scaleFlagOut           (scaleFlag),
      .fwdLlrUnityOut         (fwdLlrUnity),
      .latchChkMetricOut      (latchChkMetric),
      // To Bwd
      .bwdStallOut            (bwdStall),
      .bwdStallOut_1          (bwdStall_1),
      .stallVrOut             (stallVr),
      .bwdNoOpOut             (bwdNoOp),
      .bwdNoOpOut_1           (bwdNoOp_1),
      .bwdCellPrimeOut        (bwdCellPrime),
      .bwdIterStartOut        (bwdIterStart),
      .bwdFirstIterationOut_1 (bwdFirstIteration_1),
      .bwdRowStartOut         (bwdRowStart),
      .bwdRowStartOut_1       (bwdRowStart_1),
      .bwdRowStartOut_2       (bwdRowStart_2),
      .bwdEndIterOut          (bwdEndIter),
      .bwdLlrUnityDiv2        (bwdLlrUnityDiv2),
      // To Mem (these may change)
      .decodeActiveOut        (decodeActive),
      .decodeActiveOut_1      (decodeActive_1),
      .decodeActiveOut_2      (decodeActive_2),
      .decodeActiveOut_2D     (decodeActive_2D),
      .fwdCellIdxOut          (fwdCellIdx),
      .bwdCellIdxOut          (bwdCellIdx),
      .holdFwdCellOut_1       (holdFwdCell_1),
      .holdBwdCellOut_1       (holdBwdCell_1),
      .blkPingPongSOut        (blkPingPongS),
      .blkPingPongEOut        (blkPingPongE),
      // To Op
      .decodeCompleteOut      (decodeComplete),
      .newBlkPulseOut         (newBlkPulse),
      .shrtBlkModOp           (shrtBlkModOp),
      // RW MODIFIED
      .dbgIter                (dbgIter),
      .dbgBlkNum              (dbgBlkNum),
      // END RW MODIFIED
      // To Regs for status registers.
      .blkErrsOut             (blkErrs),
      .packetStatusOut        (packetStatusOut),
      .packetCompleteOut      (packetComplete),
      .curIterationOut        (curIteration),
      .prevParityErrsOut      (prevParityErrs),
      .prevIterationsOut      (prevIterations),
      .prevDecodeStatusOut    (prevDecodeStatus),
      .varMetricChkSumOut     (varMetricChkSum));

  assign decodeStatusOut  = prevDecodeStatus;
  assign decodeCompleteOut  = decodeComplete;
  
  // A stall is where fwd/bwd parts are totally non operational.
  // A NoOp is where there is a dummy cell. Control processing
  // is as normal but datapath is disabled. These are used in NUM_PIPES==1
  // mode. Some blocks don't distinguish && the two can be combined.
  assign fwdStall2  = fwdStall | fwdNoOp;
  assign fwdStall2_1  = fwdStall_1 | fwdNoOp_1;
  assign bwdStall2  = bwdStall | bwdNoOp;
  assign bwdStall2_1  = bwdStall_1 | bwdNoOp_1;

  // Debug: Control block
  assign dbgCtrlOut  = {newBlkPulse, fwdIterStart_1, fwdRowStart,
    varMetricSum,  // 14 bits
    scaleFlag, parityErrsUpdate, parityErrs,
    decodeComplete, decodeActive};
 
  //---------------------------------------------------------------------------
  // Input interface
  //---------------------------------------------------------------------------
  wire [5:0] dbgIpInt;
  
  ldpcDecIp # (
     .NUM_USERS       (NUM_USERS)
  ) ip (
     .nReset          (nReset),
     .clk             (ipClk),
     .clkEnOut        (ipClkEn),
     // From control
     .bitsPerSymbol   (bitsPerSymbolArray[inUser]),
     .firstBlk        (firstBlk),// From ctrl
     .nShrtFloor      (nShrtFloorArray[inUser]),
     .shrtMod         (shrtModArray[inUser]),
     .nPuncFloor      (nPuncFloorArray[inUser]),
     .puncMod         (puncModArray[inUser]),
     .nRepFloor       (nRepFloorArray[inUser]),
     .repMod          (repModArray[inUser]),
     .zEnum           (zEnumArray[inUser]),
     .n               (nArray[inUser]),
     .k               (kArray[inUser]),
     .parityStartCol  (parityStartColArray[inUser]),
     .numWrPerZ       (numWrPerZArray[inUser]),
     .enabledUsers    (enabledUsers),
     .enable          (enableIp),
     .inUser          (inUser),
     .disabling       (disabling),
     .setRdyToRcv     (setRdyToRcv),
     .decodeStart     (decodeStart),        
     // From sender  
     .inStrobe        (inStrobe),
     .ipWidth         (ipWidth),
     .inDataWord      (inDataWord),
     .ipSkipBlock     (ipSkipBlock),
     .numIpBits       (numIpBits),
     .numIpBitsValid  (numIpBitsValid),
     .llrUnity        ($signed(llrUnity)),
     // To sender    
     .rdyToRcvOut     (rdyToRcv),
     .lastIpSampleOut (lastIpSample),
     // To control   
     .llrUnity0Out    (llrUnityInitial0),
     .llrUnity1Out    (llrUnityInitial1),
     .ipLoadedOut     (ipLoaded),
     .ipAlmostLoadedOut (ipAlmostLoaded), // RW modification
     // dbg
     .dbgIp           (dbgIpInt),
     // From memory  
     .rdData          (ipRdData),
     // To memory    
     .wrDataOut       (ipWrData),
     .wrOffsetOut     (ipWrOffset),
     .wrAddrOut       (ipWrAddr),
     .wrEnableOut     (ipWrEnable),
     .rdAddrOut       (ipRdAddr),
     .rdEnableOut     (ipRdEnable),
     .lastBlkOfSymOut (lastBlkOfSym),
     .waitFirstWordOfSymOut (waitFirstWordOfSym));


  // Debug: Input control signals.
  assign dbgIp  = {2'd0, dbgIpInt, dbgBlkNum};
  // lint: warning expected. We rely on automatic extension.
  assign dbgIp2Out  = {llrUnityInitial0, llrUnity,
                       ipRdEnable, ipRdAddr,
                       ipRdEnable, ipWrAddr,
                       ipWrOffset, ipWrEnable, lastIpSample,
                       ipLoaded, firstBlk, inStrobe, rdyToRcv, enableAnyUser};
  
  //---------------------------------------------------------------------------
  // Memories
  //---------------------------------------------------------------------------

   ldpcDecVmMem vmMem
     (
      .nReset              (nReset),
      .ipClkEn             (ipClkEn),
      .vmRam1Clk           (vmRam1Clk),
      .vmRam2Clk           (vmRam2Clk),
      .cellClk             (cellClk),
      .decClk              (decClk),
      .freeClk             (freeClk), // RW modification
      .hdRamClk            (hdRamClk),
      // From regs to load Cell Info RAM
      .enable              (enableAnyUser),
      .disabling           (disabling),
      .decodeActive        (decodeActive),
      .decodeActive_1      (decodeActive_1),
      .decodeActive_2      (decodeActive_2),
      .cellAccess          (cellAccess),
      .cellRead            (cellRead),
      .cellWrite           (cellWrite),
      .cellWrData          (cellWrData),
      .zEnum               (zEnumArray[ctrlUserDec]),
      .rEnum               (rEnumArray[ctrlUserDec]),
      // From Ip (Initialisation of varMetrics)
      .wrData              (ipWrData),
      .wrOffset            (ipWrOffset),
      .wrAddr              (ipWrAddr),
      .wrEnable            (ipWrEnable),
      .rdAddr              (ipRdAddr),
      .rdEnable            (ipRdEnable),
      // To memory
      .rdData              (ipRdData),
      // From Ctrl
      .blkPingPongS        (blkPingPongS),
      .blkPingPongE        (blkPingPongE),
      .newBlk              (newBlkPulse),
      .fwdCellIdx          (fwdCellIdx),
      .bwdCellIdx          (bwdCellIdx),
      .bwdStall            (bwdStall2),
      .bwdStall_1          (bwdStall2_1),
      .fwdStall_1          (fwdStall2_1),
      .bwdNoOp             (bwdNoOp),
      .fwdNoOp_1           (fwdNoOp_1),
      .bwdCellPrime        (bwdCellPrime),
      .holdFwdCell_1       (holdFwdCell_1),
      .holdBwdCell_1       (holdBwdCell_1),
      .metricVeryFirstUse  (metricVeryFirstUse),
      // From Bwd
      .varMetricWr         (varMetricWrSlv),
      .hardDecisionsWr     (hardDecisionsWr),
      // From Op
      .opBufferSel         (hdRamSel),
      .opBufferAddr        (hdRamAddr),
      // To clocking
      .cellClkEnOut        (cellClkEnOut),
      .vmRam1ClkEnOut      (vmRam1ClkEnOut),
      .vmRam2ClkEnOut      (vmRam2ClkEnOut),
      .hdRamClkEnOut       (hdRamClkEnOut),
      // To Regs
      .cellRdDataOut       (cellRdDataOut),
      // To Ctrl && fwd/bwd
      .fwdFlagOut          (fwdFlag),
      .bwdFlagOut          (bwdFlag),
      .fwdFlagOut_1        (fwdFlag_1),
      .bwdFlagOut_1        (bwdFlag_1),
      .numMacroCellsOut    (numMacroCells),
      // To Fwd           
      .varMetricRdOut      (varMetricRdSlv),
      .fwdColOut_1         (fwdCol_1),
      .fwdCycShiftOut_1    (fwdCycShift_1),
      .fwdColOut           (fwdCol),
      .fwdCycShiftOut      (fwdCycShift),
      // To Bwd           
      .hardDecisionsRdOut  (hardDecisionsRd),
      .bwdColOut_1         (bwdCol_1),
      .bwdColOut           (bwdCol),
      .bwdCycShiftOut_1    (bwdCycShift_1),
      // To Op
      .opBufferDataOut     (hdRamData)
      // Memories (when `LDEC_RAMS_AT_TOP)
`ifdef LDEC_RAMS_AT_TOP
      ,
`ifndef LDEC_USE_CELL_ROM
      .cellRamOps          (cellRamOps),  
      .cellRamIps          (cellRamIps),
`endif
      .vmRamOps            (vmRamOps),  
      .vmRamIps            (vmRamIps),
      .hdRamOps            (hdRamOps),  
      .hdRamIps            (hdRamIps)
`endif
      );

  // Pipelining in fwd block adds latency to varResp so the
  // write control signals need to be delayed correspondingly.
  
  generate
    if (`LDEC_PIPE_A == 0) begin: gPipeAVr0
      always @(*) begin
`ifndef LDEC_2_PORTS
        decodeActiveD1 = decodeActive;
`endif
        fwdRowStartD1 = fwdRowStart;
        fwdStallD1 = fwdStall;
      end
    end
  endgenerate //  gPipeAVr0
  
  generate
    if (`LDEC_PIPE_A == 1) begin: gPipeAVr1
      always @(posedge(decClk) `LDEC_RESET_STR)
      begin : pPipeAVr
        if (nReset == 1'b0) begin
`ifndef LDEC_2_PORTS
          decodeActiveD1 <=1'b0;
`endif
          fwdRowStartD1 <= 1'b0;
          fwdStallD1 <= 1'b0;
        end else begin
`ifndef LDEC_2_PORTS
          decodeActiveD1 <= decodeActive;
`endif
          fwdRowStartD1 <= fwdRowStart;
          fwdStallD1 <= fwdStall;
        end
      end //pPipeAVr
    end
  endgenerate //  gPipeAvr1
  
  generate
    if (`LDEC_PIPE_B == 0) begin: gPipeBVr0
      always @(*) begin
`ifndef LDEC_2_PORTS
        decodeActiveD2 = decodeActiveD1;
`endif
        fwdRowStartD2 = fwdRowStartD1;
        fwdStallD2 = fwdStallD1;
      end
    end
  endgenerate //  gPipeBVr0
  
  generate
    if (`LDEC_PIPE_B == 1) begin: gPipeBVr1
      always @(posedge(decClk) `LDEC_RESET_STR)
      begin : pPipeBVr
        if (nReset == 1'b0) begin
`ifndef LDEC_2_PORTS
          decodeActiveD2 <=1'b0;
`endif
          fwdRowStartD2 <= 1'b0;
          fwdStallD2 <= 1'b0;
        end else begin
`ifndef LDEC_2_PORTS
          decodeActiveD2 <= decodeActiveD1;
`endif
          fwdRowStartD2 <= fwdRowStartD1;
          fwdStallD2 <= fwdStallD1;
        end
      end //pPipeBVr
    end
  endgenerate //  gPipeBvr1

  // `LDEC_PIPE_VR in bwd block requires start signal to be pulled forward.
  // Pull forward of bwdStall is done in ctrl block.
  generate
    if (`LDEC_PIPE_VR == 0) begin: gPipeVrReg0
      assign bwdRowStart_VR = bwdRowStart_1;
    end
  endgenerate //  gPipeVrReg0

  generate
    if (`LDEC_PIPE_VR == 1) begin: gPipeVrReg1
      assign bwdRowStart_VR = bwdRowStart_2;
    end
  endgenerate //  gPipeVrReg1
  
`ifdef LDEC_2_PORTS
  ldpcDecVrFifo vrFifo
    (
     .nReset       (nReset),
     .clk          (decClk),
     .wrRowStart   (fwdRowStartD2),
     .wrStall      (fwdStallD2),
     .rdRowStart_2 (bwdRowStart_VR),
     .rdStall_2    (stallVr),
     .dIn          (varRespWrSlv),
     .dOut         (varRespRdSlv)
`ifdef LDEC_RAMS_AT_TOP
     ,
     .vrRamOps     (vrRamOps),  
     .vrRamIps     (vrRamIps)
`endif
     );
`endif
  

`ifndef LDEC_2_PORTS
  ldpcDecVrFifo1p vrFifo
    (
     .nReset       (nReset),
     .clk          (decClk),
     .en           (decodeActiveD2),
     .wrRowStart   (fwdRowStartD2),
     .rdRowStart_2 (bwdRowStart_VR),
     .dIn          (varRespWrSlv),
     .dOut         (varRespRdSlv)
`ifdef LDEC_RAMS_AT_TOP
     ,
     .vrRamOps     (vr1RamOps),  
     .vrRamIps     (vr1RamIps)
`endif
     );
`endif
  
  // Pipe line option A (which is mostly implemented in fwd block) needs
  // increased latency on check response reads, which is achieved here.
  // Write delays are handled in ctrl block by delaying bwdIterStart.
  // PIPE B would have needed this as well, except the pipelining;
  // done on the check responses after they have been read.
  
  generate
    if (`LDEC_PIPE_A == 0) begin: gPipeA0
      always @(*) begin
        crRdStartD1 = fwdIterStart_1 &
                           !fwdFirstIteration_1 & !fwdStall_1;
        crStallD1  = fwdStall_1;
      end
    end
  endgenerate // gPipeA0

  generate
    if (`LDEC_PIPE_A == 1) begin: gPipeA1
      always @(posedge(decClk) `LDEC_RESET_STR)
      begin : pPipeA
        if (nReset == 1'b0) begin
          crRdStartD1 <= 1'b0;
          crStallD1 <= 1'b0;
        end else begin
          crRdStartD1 <= fwdIterStart_1 &
                         !fwdFirstIteration_1 & !fwdStall_1;
          crStallD1 <= fwdStall_1;
        end
      end //pPipeA     
    end
  endgenerate //  gPipeA1

  // Pipe line option X (which is mostly implemented in bwd block) needs
  // increased latency on check response writes, which is achieved here.
  
  generate
    if (`LDEC_PIPE_X == 0) begin: gPipeX0
      always @(*) begin
        bwdIterStartD1 = bwdIterStart;
        bwdStallD1 = bwdStall;
      end
    end
  endgenerate //  gPipeX0
  generate
    if (`LDEC_PIPE_X == 1) begin: gPipeX1
      always @(posedge(decClk) `LDEC_RESET_STR)
      begin : pPipeX
        if (nReset == 1'b0) begin
          bwdIterStartD1 <= 1'b0;
          bwdStallD1 <= 1'b0;
        end else begin
          bwdIterStartD1 <= bwdIterStart;
          bwdStallD1 <= bwdStall;
        end
      end //pPipeX     
    end
  endgenerate //  gPipeX1
  
    
`ifdef LDEC_2_PORTS
  ldpcDecCrFifo crFifo
    (
     .nReset        (nReset),
     .clk           (crRamClk),
     .en            (decodeActive_2),
     .disabling     (disabling),
     .wrIterStart   (bwdIterStartD1),
     .rdIterStart_1 (crRdStartD1),
     .rdStall_1     (crStallD1),
     .wrStall       (bwdStallD1),
     .dIn           (chkRespWr),
     .clkEn         (crRamClkEnOut),
     .dOut          (chkRespRd)
`ifdef LDEC_RAMS_AT_TOP
     ,
     .crRamOps      (crRamOps),  
     .crRamIps      (crRamIps)
`endif
     );
`endif

`ifndef LDEC_2_PORTS
  assign crRamClkEnOut = decodeActive | disabling;
  ldpcDecCrFifo1p crFifo
    (
     .nReset        (nReset),
     .clk           (crRamClk),
     .en            (enableAnyUser),
     .wrIterStart   (bwdIterStartD1),
     .rdIterStart_1 (crRdStartD1),
     .dIn           (chkRespWr),
     .dOut          (chkRespRd)
`ifdef LDEC_RAMS_AT_TOP
     ,
     .crRamOps      (cr1RamOps),  
     .crRamIps      (cr1RamIps)
`endif
     );
`endif
  
  //---------------------------------------------------------------------------
  // Fwd/Bwd Signal Processing
  //---------------------------------------------------------------------------

  // RW modification: add ipAlmostLoaded so that decClkEnOut is available BEFORE actual need of first decClk edge
  // to account for delay in resynchronization cell
  assign decClkEnOut = decodeActive_2 | decodeActive_2D | disabling | ipAlmostLoaded;

  // From linear vector to 2D array
  assign chkRespRdSgn = getSgnC(chkRespRd);
  assign chkRespRdAbs = getAbsC(chkRespRd);

  always @(targetLevel)
  begin : pShortClip
    reg  [`LDEC_TARGET_LEVEL_LEFT-5:0] dfltV;
    // The default signal value for a shortened bit is targetLevel/32. 
    dfltV = targetLevel[`LDEC_TARGET_LEVEL_LEFT : 5];
    shortDefault = `LDEC_CLIP_U(dfltV, `LDEC_TARGET_LEVEL_LEFT-4, `LDEC_VAR_BITS-1);
  end //pShortClip
  
  ldpcDecFwd fwd 
    (
     .nReset             (nReset),
     .clk                (decClk),
     .enable             (enableAnyUser),
     .nShrtFloor         (nShrtFloorArray[ctrlUserDec]),
     .shrtBlkMod_1       (shrtBlkModDec),
     // From control    
     .zEnum              (zEnumArray[ctrlUserDec]),
     .z                  (zArray[ctrlUserDec]),
     .k                  (kArray[ctrlUserDec]),
     .firstIteration     (fwdFirstIteration),
     .firstIteration_1   (fwdFirstIteration_1),
     .col_1              (fwdCol_1),
     .col                (fwdCol),
     .rowStart           (fwdRowStart),
     .cycShift_1         (fwdCycShift_1),
     .cycShift           (fwdCycShift),
     .flag               (fwdFlag),
     .flag_1             (fwdFlag_1),
     .fwdStall           (fwdStall2),
     .fwdStall_1         (fwdStall2_1),      
     .scaleFlag          (scaleFlag),
     .llrUnity           (fwdLlrUnity),
     .latchChkMetric     (latchChkMetric),
     // From memories   
     .varMetricRd        (varMetricRdSlv),
     .chkRespRdSgn       (chkRespRdSgn),
     .chkRespRdAbsP      (chkRespRdAbs),
     // To Bwd block
     .chkMetricRegSgnOut (chkMetricRegSgn),
     .chkMetricRegAbsOut (chkMetricRegAbs),
     .shrtColOut         (shrtCol),
     // To memories 
     .varRespWrOut       (varRespWrSlv));

  ldpcDecBwd bwd
    (
     .nReset             (nReset),
     .clk                (decClk),
     .ctrlClk            (ctrlClk),
     .enable             (enableAnyUser),
     // From control
     .zEnum              (zEnumArray[ctrlUserDec]),
     .iterStart          (bwdIterStart),
     .firstIteration_1   (bwdFirstIteration_1),
     .rowStart           (bwdRowStart),
     .shrtColP           (shrtCol),
     .parityStartCol     (parityStartColArray[ctrlUserDec]),
     .newBlk             (newBlkPulse),
     .endIter            (bwdEndIter),
     .llrUnityDiv2       (bwdLlrUnityDiv2),
     .shortDefault       (shortDefault),
     // From memories
     .col_1              (bwdCol_1),
     .col                (bwdCol),
     .cycShift_1         (bwdCycShift_1),
     .flag               (bwdFlag),
     .flag_1             (bwdFlag_1),
     .bwdStall           (bwdStall),
     .bwdStall_1         (bwdStall_1),
     .bwdNoOp            (bwdNoOp),
     .bwdNoOp_1          (bwdNoOp_1),
     .hardDecisionsRd    (hardDecisionsRd),
     .varRespRd          (varRespRdSlv),
     // From Fwd block
     .chkMetricRegSgn    (chkMetricRegSgn),
     .chkMetricRegAbsP   (chkMetricRegAbs),
     // To Control
     .parityErrsUpdateOut (parityErrsUpdate),
     .parityErrsOut      (parityErrs),
     .varMetricSumOut    (varMetricSum),
     // To IO
     .cycLastOut         (cycLast),
     // To memories
     .varMetricWrOut     (varMetricWrSlv),
     .hardDecisionsWrOut (hardDecisionsWr),
     .chkRespWrSgnOut    (chkRespWrSgn),
     .chkRespWrAbsOut    (chkRespWrAbs));

  assign chkRespWr  = combineZ(chkRespWrSgn, chkRespWrAbs);

  //---------------------------------------------------------------------------
  // Output Interface
  //---------------------------------------------------------------------------

  assign k2  = kArray[ctrlUserOp]-nShrtFloorArray[ctrlUserOp] - {{ (K_BITS-1) {1'b0} }, shrtBlkModOp};
  
  ldpcDecOp op
    (
     .nReset              (nReset),
     .opClk               (opClk),
     .enable              (enableAnyUser), // !!! MU mode not compatible with CONCAT_OP=1
     .BlockEnable         (enableOp),
     .disabling           (disabling),
     .zEnum               (zEnumArray[ctrlUserOp]),
     .packetBytesLs       (packetBytesLs),
     .packetBytesMs       (packetBytesMs),
     .parityStartCol      (parityStartColArray[ctrlUserOp]),
     .k2                  (k2),
     .z                   (zArray[ctrlUserOp]),
     .cycLastP            (cycLast),
     .clrToSend           (clrToSend),
     .hdRamData           (hdRamData),
     .newBlk              (newBlkPulse),
     .packetComplete      (packetComplete),
     .decodeComplete      (decodeComplete),
     .opStrobeOut         (opStrobeOut),
     .opDataWordOut       (opDataWord),
     .opByteEnOut         (opByteEn),
     .hdRamSelOut         (hdRamSel),
     .hdRamAddrOut        (hdRamAddr),
     .opBufferLockedOut   (opBufferLocked),
     .opClkEnOut          (opClkEnOut),
     .opBitEnOut          (opBitEnOut),
     .opDoneOut           (opDoneOut),
     .blockOpCompleteOut  (blockOpComplete),
     .packetOpCompleteOut (packetOpCompleteOut)
     );
     
  assign dbgOpOut = {hdRamAddr, hdRamSel, decodeComplete, opBufferLocked, clrToSend,
    opDataWord, opByteEn};
    
  assign packetCompleteOut = packetComplete;
  assign blkErrsOut = blkErrs;
  assign curIterationOut = curIteration;
  assign prevParityErrsOut = prevParityErrs;
  assign prevIterationsOut = prevIterations;
  assign prevDecodeStatusOut = prevDecodeStatus;
  assign varMetricChkSumOut = varMetricChkSum;
  assign ipClkEnOut = ipClkEn;
  assign rdyToRcvOut = rdyToRcv;
  assign opDataWordOut = opDataWord;
  assign opByteEnOut = opByteEn;
  assign opUser = ctrlUserOp;
  
endmodule

