//------------------------------------------------------------------------------
// ldcpDecCrFifo.v
// 
// Description:
//   Variable Response FIFO. Requires a read-before-write RAM. Depth of the FIFO
//   is 2 more than the RAM size due to registering && the read-before-write
//   operation.
//
// Inputs:
//   en    : Enable both reading && writing
//   wrIterStart :  Write data for the 1st entry of iter is on dIn.
//   rdIterStart :  Read data for 1st entry required on dOut in 1 cycles time.
//   rdStall     :  Stall with T-1 timing prevents rdAddr increment.
//   wrStall     :  Hold write address.
//   dIn         :  Data to write.
//   dOut        :  Data to read.
//
// 9 June 2010 M. Rumsey. Created.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//
//-----------------------------------------------------------------------------

`include "ldpcDec.vh"

module ldpcDecCrFifo
( 
  input                           nReset,
  input                           clk,
  input                           en,
  input                           disabling,
  input                           wrIterStart,
  input                           rdIterStart_1,
  input                           rdStall_1,
  input                           wrStall,
  input [`LDEC_CR_RAM_W-1:0]      dIn,
  output                          clkEn,
  output [`LDEC_CR_RAM_W-1:0]     dOut
`ifdef LDEC_RAMS_AT_TOP
  ,
  input [`LDEC_CR_RAM_O-1:0]      crRamOps,
  output [`LDEC_CR_RAM_I-1:0]     crRamIps
`endif
  );    

`include "ldpcDecFuncs.vh"

  localparam [`LDEC_CR_RAM_A-1:0] CR_RAM_D1 = `LDEC_CR_RAM_D-1;
  
  wire wrEn;
  reg  hadWrRowStart;
  reg  [`LDEC_CR_RAM_A-1:0] rdAddr;
  reg  [`LDEC_CR_RAM_A-1:0] wrAddr;
  reg  [`LDEC_CR_RAM_A-1:0] rdAddrReg;
  reg  [`LDEC_CR_RAM_A-1:0] wrAddrReg;

  assign wrEn = en & !wrStall;
  
  // Hold off the clock until the backward processing starts writing.
  // Clock begin stays on until en is dropped (at the end of the packet).
  assign clkEn = (en & (wrIterStart | hadWrRowStart)) | disabling;

  // Address generation.
  
  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pAddrReg
    if (nReset == 1'b0) begin
      hadWrRowStart <= 1'b0;
      rdAddrReg <= `LDEC_PAD(1'b0, `LDEC_CR_RAM_A-1);
      wrAddrReg <= `LDEC_PAD(1'b0, `LDEC_CR_RAM_A-1);
    end else begin
      if (en == 1'b1) begin
        if (rdStall_1 == 1'b0) begin
          if (rdAddr < CR_RAM_D1) begin
            rdAddrReg <= rdAddr + `LDEC_PAD(1'b1, `LDEC_CR_RAM_A-1);
          end
        end
        if (wrStall == 1'b0) begin
          if (wrAddr < CR_RAM_D1 && wrEn == 1'b1) begin
            wrAddrReg <= wrAddr + `LDEC_PAD(1'b1, `LDEC_CR_RAM_A-1);
          end
        end
        if (wrIterStart == 1'b1) begin
          hadWrRowStart <= 1'b1;
        end 
      end else begin
        hadWrRowStart <= 1'b0;
      end
    end
  end //pAddrReg

  // The address presented to the RAM must be combinatorially derived
  // to avoid having to bring the rdIterStart signal yet another cycle
  // earlier (it is already T-2).
  always @(rdAddrReg, rdIterStart_1)
  begin : pRdAddr
    if (rdIterStart_1 == 1'b1) begin
      rdAddr = `LDEC_PAD(1'b0, `LDEC_CR_RAM_A-1);
    end
    else begin
      rdAddr = rdAddrReg;
    end
  end //pRdAddr

  always @(wrAddrReg, wrIterStart)
  begin : pWrAddr
    if (wrIterStart == 1'b1) begin
      wrAddr = `LDEC_PAD(1'b0, `LDEC_CR_RAM_A-1);
    end
    else begin
      wrAddr = wrAddrReg;
    end
  end //pWrAddr
 
  // The RAM

`ifndef LDEC_RAMS_AT_TOP
  
  ldpcDecCrRam crRam
    (
     .ck    (clk),
     // Write
     .addrA (wrAddr),
     .dInA  (dIn),
     .weA   (wrEn),
     // Read
     .selB  (en),
     .addrB (rdAddr),
     .dOutB (dOut));

`else
  
  assign crRamIps = {wrEn, en, wrAddr, dIn, rdAddr};
  assign dOut = crRamOps;

`endif
  
endmodule

    
    

