//------------------------------------------------------------------------------
// ldcpDecCrFifo1p.v
// 
// Description:
//   Variable Response FIFO. Requires a read-before-write single port RAM. Depth
//   of the FIFO is 2 more than the RAM size due to registering && the
//   read-before-write operation.
//
//   en          : Enable both reading and writing. 
//   wrRowStart  : Write data for the 1st entry of row is on dIn.
//   wrStall     : May be raised when there is a write stall at the end of a row
//                 if all rows have stalls at the end (these are 'pipeline' stalls).
//   rdRowStart  : Read data required on dOut in 2 cycles time. This means 2
//                 unstalled cycles.
//
//
// 1 June 2010 M. Rumsey. Created.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//
//-----------------------------------------------------------------------------

`include "ldpcDec.vh"

module ldpcDecCrFifo1p
(  
   input                           nReset,
   input                           clk,
   input                           en,
   input                           wrIterStart,
   input                           rdIterStart_1,
   input [`LDEC_CR_RAM_W-1:0]      dIn,
   output [`LDEC_CR_RAM_W-1:0]     dOut
`ifdef LDEC_RAMS_AT_TOP
   ,
   input [`LDEC_CR1_RAM_O-1:0]     crRamOps,
   output [`LDEC_CR1_RAM_I-1:0]    crRamIps
`endif
   );

`include "ldpcDecFuncs.vh"

  reg [numBits(`LDEC_CR_RAM_D-1)-1:0] addr;
  reg [numBits(`LDEC_CR_RAM_D-1)-1:0] addrLast;
  reg [numBits(`LDEC_CR_RAM_D-1)-1:0] wrapAddr;
  reg [numBits(`LDEC_CR_RAM_D-1)-1:0] iterStartAddr;
  reg [numBits(`LDEC_CR_RAM_D-1)-1:0] addrReg;
  reg                                 firstReadSeen;
 
  // Address generation. This gets set back to start of row on rdIterStart where
  // start of row is defined as the address registered when wrIterStart was high.

  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pAddrReg
    reg  firstWriteSeenV;
    
    if (nReset == 1'b0) begin
      addrReg <= 0;
      iterStartAddr <= 0;
      wrapAddr <= `LDEC_CR_RAM_D-1;
      firstReadSeen <= 1'b0;
      firstWriteSeenV = 1'b0;
    end else begin
      if (en == 1'b0) begin
        // synchronous reset at end of block
        addrReg <= 0;
        iterStartAddr <= 0;
        wrapAddr <= `LDEC_CR_RAM_D-1;
        firstReadSeen <= 1'b0;
        firstWriteSeenV = 1'b0;
      end
      else   begin
        addrLast <= addr;
        if (wrIterStart == 1'b1) begin
          firstWriteSeenV = 1'b1;
          iterStartAddr <= addr;
        end
        if (addr == wrapAddr) begin
          addrReg <= 0;
        end
        else if (firstWriteSeenV == 1'b1) begin
          addrReg <= addr + 1;
        end
        if ( (rdIterStart_1 && !firstReadSeen) == 1'b1) begin
          firstReadSeen <= 1'b1;
          wrapAddr <= addrLast;
        end
      end
    end
    
  end //pAddrReg

  always @(addrReg, iterStartAddr, rdIterStart_1)
  begin : pAddr
    if (rdIterStart_1 == 1'b1) begin
      addr <= iterStartAddr;
    end
    else begin
      addr <= addrReg;
    end
  end //pAddr
 
  // The RAM


`ifdef LDEC_RAMS_AT_TOP

  assign crRamIps  = {en, addr, dIn};
  assign dOut = crRamOps;

`else
  
  ldpcDecCr1pRam crRam
    (
     .ck   (clk),
     .sel  (en),
     .addr (addr),
     .dIn  (dIn),
     .we   (en),
     .dOut (dOut));
  
`endif
  
endmodule

    
    

