// Disable coverage on this file
// pragma coverage block = off 
//------------------------------------------------------------------------------
// ldcpDecFuncs.sv
// 
// Description:
//   Function file, included where needed.
//
// 15 Oct 2012 M. Rumsey. Created.
//
// (c) Copyright 2012, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`include "ldpcDec.vh"

//------------------------------------------------------------------------------
// Lookup tables using zEnum and rEnum.
//------------------------------------------------------------------------------
// Parameter array form is replaced by functions because Cadence linting tool
// does not support them.

//localparam bit [numBits(`LDEC_N_MAX)-1:0] N_SIZES[0:2] = '{648, 1296, 1944};
function automatic [10:0] N_SIZES (input [1:0] zE);
  reg [10:0] y;
  begin
    case (zE)
      2'd0 : y = 11'd648;
      2'd1 : y = 11'd1296;
      default : y = 11'd1944;
    endcase
    N_SIZES = y;
  end
endfunction

//localparam bit [numBits(`LDEC_NCOLS-1)-1:0]  PARITY_START_COL[0:3] = '{12, 16, 18, 20};
function automatic [4:0] PARITY_START_COL (input [1:0] rE);
  reg [4:0] y;
  begin
    case (rE)
      2'd0 : y = 5'd12;
      2'd1 : y = 5'd16;
      2'd2 : y = 5'd18;
      default : y = 5'd20;
    endcase
    PARITY_START_COL = y;
  end
endfunction

//localparam bit [numBits(`LDEC_K_MAX)-1:0] K_SIZES[0:2][0:3] = '{'{324, 432, 486, 540},
//                                                                '{648, 864, 972, 1080},
//                                                                '{972, 1296, 1458, 1620}};
function automatic [10:0] K_SIZES (input [1:0] zE, input [1:0] rE);
  reg [10:0] y;
  begin
    case (zE)
      2'd0 :
        case (rE)
          2'd0 : y = 11'd324;
          2'd1 : y = 11'd432;
          2'd2 : y = 11'd486;
          default : y = 11'd540;
        endcase
      2'd1 : 
        case (rE)
          2'd0 : y = 11'd648;
          2'd1 : y = 11'd864;
          2'd2 : y = 11'd972;
          default : y = 11'd1080;
        endcase
      default :
        case (rE)
          2'd0 : y = 11'd972;
          2'd1 : y = 11'd1296;
          2'd2 : y = 11'd1458;
          default : y = 11'd1620;
        endcase
    endcase
    K_SIZES = y;
  end
endfunction

//localparam bit [numBits(`LDEC_Z_MAX)-1:0] Z_SIZES[0:2] = '{27, 27, 27};
//function automatic [6:0] Z_SIZES (input [1:0] zE);
//  reg [6:0] y;
//  begin
//    y = 7'd27;
//    Z_SIZES = y;
//  end
//endfunction

//localparam bit [numBits(`LDEC_Z_MAX)-1:0] Z_SIZES2[0:2] = '{27, 54, 81};
function automatic [6:0] Z_SIZES2 (input [1:0] zE);
  reg [6:0] y;
  begin
    case (zE)
      2'd0 : y = 7'd27;
      2'd1 : y = 7'd54;
      default : y = 7'd81;
    endcase
    Z_SIZES2 = y;
  end
endfunction

//localparam bit [numBits(`LDEC_Z_MAX)-1:0] Z_SIZES3[0:2] = '{0, 27, 54};
function automatic [6:0] Z_SIZES3 (input [1:0] zE);
  reg [6:0] y;
  begin
    case (zE)
      2'd0 : y = 7'd0;
      2'd1 : y = 7'd27;
      default : y = 7'd54;
    endcase
    Z_SIZES3 = y;
  end
endfunction

//localparam bit [numBits(`LDEC_Z_MAX)-1:0] Z_SIZES2[0:2] = '{27, 54, 81};
function automatic integer Z_SIZES2_INT (input integer zE);
  integer y;
  begin
    case (zE)
      0 : y = 27;
      1 : y = 54;
      default : y = 81;
    endcase
    Z_SIZES2_INT = y;
  end
endfunction

localparam integer ZE_HI = numBits(`LDEC_Z_ENUM_MAX)-1;

localparam MUX_MAX = ceilDiv(`LDEC_Z_MAX, `LDEC_MEM_FETCH_SIZE)-1;
localparam MUX_HI = numBits(MUX_MAX)-1;

//localparam bit [MUX_HI:0] MEM_MUX_MAX[0:2] = '{ceilDiv(27, `LDEC_MEM_FETCH_SIZE)-1,
//                                               ceilDiv(54, `LDEC_MEM_FETCH_SIZE)-1,
//                                               ceilDiv(81, `LDEC_MEM_FETCH_SIZE)-1};
// Used in ldpcDecOp
function automatic [MUX_HI:0] MEM_MUX_MAX (input [1:0] zE);
  reg [MUX_HI:0] y;
  begin
    case (zE)
      2'd0 :
          y = $unsigned(27/`LDEC_MEM_FETCH_SIZE - 1);
      2'd1 : 
          y = $unsigned(54/`LDEC_MEM_FETCH_SIZE - 1);
      default :
          y = $unsigned(81/`LDEC_MEM_FETCH_SIZE - 1);
    endcase
    MEM_MUX_MAX = y;
  end
endfunction
   

// NUM_WR_PER_Z
localparam NWZ_HI = numBits(ceilDiv(`LDEC_Z_MAX, `LDEC_DEC_RAM_IP_WIDTH))-1;

//localparam bit [NWZ_HI:0] NUM_WR_PER_Z [0:2] =
//                    '{ceilDiv(27, `LDEC_DEC_RAM_IP_WIDTH),
//                      ceilDiv(54, `LDEC_DEC_RAM_IP_WIDTH),
//                      ceilDiv(81, `LDEC_DEC_RAM_IP_WIDTH)};

// Used in ldpcDecIp
function automatic [NWZ_HI:0] NUM_WR_PER_Z (input [1:0] zE);
  reg [NWZ_HI:0] y;
  begin
    case (zE)
      2'd0 :
          y = $unsigned(27 / `LDEC_DEC_RAM_IP_WIDTH);
      2'd1 : 
          y = $unsigned(54 / `LDEC_DEC_RAM_IP_WIDTH);
      default :
          y = $unsigned(81 / `LDEC_DEC_RAM_IP_WIDTH);
    endcase
    NUM_WR_PER_Z = y;
  end
endfunction

//---------------------------------------------------------------------------
//  Various integer functions (for constant use)
//---------------------------------------------------------------------------

// If select; 1 (for true) begin return the sel1 value otherwise the sel0.
function automatic integer sel2(input integer selvar,
                     input integer sel1,
                     input integer sel0);
  begin
    if (selvar > 0) begin
      sel2 = sel1;
    end
    else begin
      sel2 = sel0;
    end
  end
endfunction // integer sel

// Number of bits needed to represent a number. Not for synthesis!
function automatic [31:0] numBits (input integer a);
//function automatic integer numBits (input integer a);
  begin: func
    //numBits = $clog2(a+1);
    integer a2;
    a2 = a;
    numBits = 32'b0;
    while (a2>0) begin
      numBits = numBits+1;
      a2 = $signed($unsigned(a2) >> 1);
    end
  end
endfunction // numBits

function automatic integer maximum (input integer a,
                                    input integer b);
  begin
    if (a > b) begin
      maximum = a;
    end
    else begin
      maximum = b;
    end
  end
endfunction // maximum

function automatic integer minimum (input integer a,
                                    input integer b);
  begin
    if (a < b) begin
      minimum = a;
    end
    else begin
      minimum = b;
    end    
  end
endfunction // minimum

// Divide a by b with round up. Not for synthesis!
function automatic integer ceilDiv (input integer a,
                                    input integer b);
  integer y;
  begin
    y = a / b;
    if ((a % b) != 0) begin
      y = y + 1;
    end
    ceilDiv = y;    
  end
endfunction // ceilDiv

//---------------------------------------------------------------------------
//  Various rounding type functions
//---------------------------------------------------------------------------

// Round off 1 bit but add an msb for overflow. This is for unsigned only.
function automatic [`LDEC_CHK_BITS-2:0] roundNoClipC (input [`LDEC_CHK_BITS-2:0]  a);
  reg [`LDEC_CHK_BITS-2:0] y;
  begin
    y = {1'b0, a[`LDEC_CHK_BITS-2:1]};    // 1 bit for overflow
    if (a[0] == 1'b1) begin
      y = y + {{(`LDEC_CHK_BITS-2) {1'b0}}, 1'b1};
    end
    roundNoClipC = y;
  end
endfunction // roundNoClip

// Round off 1 bit but add an msb for overflow. This is for signed only.
// Operates on VAR_BITS+1 input width. Rounds away fm zero.
function automatic signed [`LDEC_VAR_BITS:0] roundNoClipV1 (input signed [`LDEC_VAR_BITS:0]  a);
  reg signed [`LDEC_VAR_BITS:0] y;
  begin
    y = $signed({a[`LDEC_VAR_BITS], a[`LDEC_VAR_BITS:1]});    // 1 bit for overflow
    if ((a[`LDEC_VAR_BITS] == 1'b0) && (a[0] == 1'b1)) begin
      y = y + $signed({{`LDEC_VAR_BITS {1'b0}}, 1'b1});
    end
    roundNoClipV1 = y;
  end
endfunction // roundNoClip

//---------------------------------------------------------------------------
//  Mod with folding of negatives
//---------------------------------------------------------------------------

function automatic [numBits(`LDEC_Z_MAX-1)-1:0] modZ (input signed [numBits(`LDEC_Z_MAX-1):0]    a,
                                                      input        [numBits(`LDEC_Z_MAX-1)-1:0]  z);
  reg signed [numBits(`LDEC_Z_MAX-1):0] y;
  begin
    if (z == `LDEC_PAD(1'b0, numBits(`LDEC_Z_MAX-1)-1)) begin
      y = `LDEC_PADS(1'b0, numBits(`LDEC_Z_MAX-1));
    end else if (a < `LDEC_PADS(1'sb0, numBits(`LDEC_Z_MAX-1))) begin
      y = a + $signed({1'b0, z});
    end else if (a >= $signed({1'b0, z})) begin
      y = a - $signed({1'b0, z});
    end else begin
      y = a;
    end
    modZ = y[numBits(`LDEC_Z_MAX-1)-1:0]; // top bits is stripped
  end
endfunction // modZ

`ifdef ALLOW_RECURSIVE

// Rotate a, which is of length Z_MAX*VAR_BITS, by 2^level if shift(level)===1.
//  Called recursively to rotate for all bits of shift.

function automatic `LDEC_varZType        ldpcDecRotateDownZ
  (input `LDEC_varZType                  a,
   input [numBits(`LDEC_Z_ENUM_MAX)-1:0] zEnum,
   input integer                         level,
   input integer                         bottom,
   input [numBits(`LDEC_Z_MAX-1)-1:0]    shift);
                      
  reg `LDEC_varZType               y;
  //integer                          offset;
  integer                          opIdx;
  integer                          ipIdx;
  integer                          zE;
    
  begin
    
    y = a;                             // Default for no shift
    // Shift for this bit (level) of the shift vector. Do a straight
    // through if level is less than bottom.
    if (level >= bottom) begin
      //offset = 2**level;
      if (2**level < Z_SIZES2(zEnum)) begin
        if (shift[level] == 1'b1) begin
          for (zE=0; zE<=`LDEC_Z_ENUM_MAX; zE=zE+1) begin
            for (opIdx=0; opIdx<Z_SIZES2(zE); opIdx=opIdx+1) begin
              if (zE == zEnum) begin
                ipIdx = opIdx + 2**level;
                if (ipIdx > Z_SIZES2(zE)-1) begin
                  ipIdx = ipIdx - Z_SIZES2(zE);
                end
                if (ipIdx < Z_SIZES2(zE)) begin
                  y[opIdx] = a[ipIdx];
                end
              end
            end
          end
        end
      end
    end
    
    // Do shifts recursively for lower levels.
    if (level > bottom) begin
      y = ldpcDecRotateDownZ(y, zEnum, level-1, bottom, shift);
    end
    return y;
  end
endfunction

// Rotate a, which is of length Z_MAX, by 2^level if shift(level)===1. Called
// recursively to rotate for all bits of shift.

function automatic [`LDEC_Z_MAX-1:0] ldpcDecRotateDownZ1
  (input [`LDEC_Z_MAX-1:0]               a,
   input [numBits(`LDEC_Z_ENUM_MAX)-1:0] zEnum,
   input integer                    level,
   input integer                    bottom,
   input [numBits(`LDEC_Z_MAX-1)-1:0]    shift);
                      
  reg [`LDEC_Z_MAX-1:0]            y;
  //integer                     offset;
  integer                     opIdx;
  integer                     ipIdx;
  integer                     zE;
    
  begin
    
    y = a;                             // Default for no shift
    // Shift for this bit (level) of the shift vector. Do a straight
    // through if level is less than bottom.
    if (level >= bottom) begin
      // offset = 2**level;
      if (2**level < Z_SIZES2(zEnum)) begin
        if (shift[level] == 1'b1) begin
          for (zE=0; zE<=`LDEC_Z_ENUM_MAX; zE=zE+1) begin
            for (opIdx=0; opIdx<Z_SIZES2(zE); opIdx=opIdx+1) begin
              if (zE == zEnum) begin
                ipIdx = opIdx + 2**level;
                if (ipIdx > Z_SIZES2(zE)-1) begin
                  ipIdx = ipIdx - Z_SIZES2(zE);
                end
                if (ipIdx < Z_SIZES2(zE)) begin
                  y[opIdx] = a[ipIdx];
                end
              end
            end
          end
        end
      end
    end 

    // Do shifts recursively for lower levels.
    if (level > bottom) begin
      y = ldpcDecRotateDownZ1(y, zEnum, level-1, bottom, shift);
    end
    return y;
  end
endfunction
  
//---------------------------------------------------------------------------
// ShiftDown (not cyclical)
//---------------------------------------------------------------------------

// Shift down a by 2^level if shift(level)===1. Called recursively to rotate
// for all bits of shift.
function automatic [`LDEC_CR_HI:0]`LDEC_ip2DType ldpcDecShiftDown
  (input [`LDEC_CR_HI:0]`LDEC_ip2DType              a,
   input integer                          level,
   input integer                          bottom,
   input [numBits(`LDEC_DEC_RAM_IP_WIDTH)-1:0] shift);
  
    reg     [`LDEC_CR_HI:0]`LDEC_ip2DType y;
    //integer offset;
    integer opIdx;
  
  begin
    // default for no shift
    for (opIdx=0; opIdx<=`LDEC_CR_HI; opIdx=opIdx+1) begin
      y[opIdx] = a[opIdx];
    end
    // Shift for this bit (level) of the shift vector. Do a straight
    // through if level is less than bottom.
    if (level >= bottom) begin
      //offset =  2**level;
      // If a shift goes ahead it is by offset, this means the highest
      // bit we care about is $high(a)-offset, higher bits are left untouched on
      // the assuption that some other function loads new data into them.
      for (opIdx=0; opIdx<=`LDEC_CR_HI-2**level; opIdx=opIdx+1) begin
        if (shift[level] == 1'b1) begin        
          y[opIdx] = a[opIdx+2**level];
        end
      end
    end
    // Do shifts recursively for lower levels.
    if (level > bottom) begin
      y = ldpcDecShiftDown(y, level-1, bottom, shift);
    end
    ldpcDecShiftDown = y;
  end
endfunction

`else

// Non recursive versions with duplicated functions for each level.
`include "ldpcDecShiftDown.vh"
`include "ldpcDecRotateDownZ.vh"
`include "ldpcDecRotateDownZ1.vh"

`endif

//------------------------------------------------------------------------------
// Soft add/subtract
//------------------------------------------------------------------------------

// Performs the soft Add function.
function automatic  `LDEC_chkAbs3_P softAdd (input `LDEC_chkAbs3_P   aP,
                                             input `LDEC_chkAbsType  b,
                                             input                   init,
                                             input `LDEC_chkAbsType  llrUnity);

  reg `LDEC_chkAbsType              a            `LDEC_chkAbs3_R;
  reg `LDEC_chkAbsType              bClip;
  reg [`LDEC_CHK_BITS-3:0]          aHalf;
  reg [`LDEC_CHK_BITS-3:0]          bHalf;
  reg `LDEC_chkAbsType              y           `LDEC_chkAbs3_R;
  reg `LDEC_chkAbsType              corr        `LDEC_chkAbs2_R;
  reg [`LDEC_CHK_BITS-2:0]          abDiff2; 
  reg  signed [`LDEC_CHK_BITS-2:0]  abDiff2S;
  localparam SEL_IDX = (`LDEC_MINSUM_ALG > 1) ? 2'd2 : 2'd0;
  integer                           idx1; // packing
  integer                           i;
  
  begin

    `LDEC_UNPACKF(aP, a, `LDEC_CHK_BITS-1, `LDEC_CHK_METRIC_RIGHT+1);

    bClip = b; // Already clipped. clip(b, CHK_BITS-1);
     
    if (init == 1'b1) begin       
      // 'a' is not initialised so b is the min.
      y[0] = bClip;                                   // chkMetric 
      y[1] = {`LDEC_CHK_BITS-1 {1'b1}};               // chkMetricMgn
      if (`LDEC_CHK_METRIC_RIGHT == 2) begin
        y[`LDEC_CHK_METRIC_RIGHT] = bClip;            // lowest varMetric so far
      end
    end
    else begin
      // Get the corrections.
      corr[0] = `LDEC_PAD(1'b0, `LDEC_CHK_BITS-2);
      corr[1] = `LDEC_PAD(1'b0, `LDEC_CHK_BITS-2);
      if (`LDEC_MINSUM_ALG >= 2) begin
        bHalf = bClip[`LDEC_CHK_BITS-2:1];
        for (i=0; i<=`LDEC_MINSUM_ALG-2 ; i=i+1) begin
          aHalf = a[i][`LDEC_CHK_BITS-2:1];
          abDiff2S = $signed({1'b0, aHalf}) - $signed({1'b0, bHalf});
          abDiff2 = `LDEC_ABS2(abDiff2S, `LDEC_CHK_BITS-1);
         
          if (abDiff2 < llrUnity) begin
            corr[i] = {1'b0, llrUnity[`LDEC_CHK_BITS-2:1]};
          end
        end
      end
     
      // Updates for marginalised metric.
      `LDEC_COPY(a, y, `LDEC_CHK_METRIC_RIGHT+1);
      if (bClip <= a[SEL_IDX]) begin
        // if b is lowest b so far it becomes the new a(0) and
        // old a(0) becomes a(1) (the 2nd lowest a).
        if (`LDEC_CHK_METRIC_RIGHT == 2) begin
          y[`LDEC_CHK_METRIC_RIGHT] = bClip;
        end        
        y[1] = a[0];
      end
      else begin
        if (a[1] > bClip) begin
          y[1] = bClip; // else defaults to a(1)     
        end
        if (`LDEC_MINSUM_ALG > 2) begin
          if (y[1] > corr[1]) begin
            y[1] = y[1] - corr[1];
          end
          else begin
            y[1] = `LDEC_PAD(1'b0, `LDEC_CHK_BITS-1);
          end
        end
      end
         
      // Corrections for metric.
      if (a[0] > bClip) begin
        // if b is less than old a(0) it becomes the new a(0).
        y[0] = bClip; // else defaults to a(0)
      end
      if (`LDEC_MINSUM_ALG >= 2) begin
        if (y[0] > corr[0]) begin
          y[0] = y[0] - corr[0];
        end
        else begin
          y[0] = `LDEC_PAD(1'b0, `LDEC_CHK_BITS-1);
        end
      end
          
    end
    `LDEC_PACKF(y, softAdd, `LDEC_CHK_BITS-1, `LDEC_CHK_METRIC_RIGHT+1);
    
  end
endfunction // softAdd

// A pruned down version of above, for clarity
function automatic `LDEC_chkAbs3_P softAddB (input `LDEC_chkAbs3_P   aP,
                                             input `LDEC_chkAbsType  b,
                                             input                   init);
  
  reg `LDEC_chkAbsType   a   `LDEC_chkAbs3_R;
  reg `LDEC_chkAbsType   y   `LDEC_chkAbs3_R;
  integer                idx1; // packing
  
  begin
   
    `LDEC_UNPACKF(aP, a, `LDEC_CHK_BITS-1, `LDEC_CHK_METRIC_RIGHT+1);

      if (init == 1'b1) begin       
      // 'a' is not initialised so b is the min.
      y[0] = b;                           // chkMetric 
      y[1] = {`LDEC_CHK_BITS-1 {1'b1}};   // chkMetricMgn
    end
    else begin
      `LDEC_COPY(a, y, `LDEC_CHK_METRIC_RIGHT+1);
      if (b <= a[0]) begin
        // if b is lowest b so far it becomes the new a(0) and
        // old a(0) becomes a(1) (the 2nd lowest a).
        y[0] = b;
        y[1] = a[0];
      end
      else begin
        if (a[1] > b) begin
          y[1] = b; // else defaults to a(1)     
        end
      end          
    end
    `LDEC_PACKF(y, softAddB, `LDEC_CHK_BITS-1, `LDEC_CHK_METRIC_RIGHT+1);
  end
endfunction // softAddB

// Performs the soft Sub function.
function automatic `LDEC_chkAbsType softSub (input  `LDEC_chkAbs3_P       yP,
                                             input  `LDEC_chkAbsType      b,
                                             input  [`LDEC_CHK_BITS-3:0]  llrUnityDiv2);
  
  reg `LDEC_chkAbsType   a;
  reg `LDEC_chkAbsType   y   `LDEC_chkAbs3_R;
  integer                idx1; // packing
  
  begin

    `LDEC_UNPACKF(yP, y, `LDEC_CHK_BITS-1, `LDEC_CHK_METRIC_RIGHT+1);

     a = `LDEC_PAD(1'b0, `LDEC_CHK_BITS-2); // default
    
    if (`LDEC_CHK_METRIC_RIGHT == 2) begin
      if (y[`LDEC_CHK_METRIC_RIGHT] == b) begin
        a = y[1];
      end else begin
        a = y[0];
      end
    end else begin
      if (b == y[0]) begin
        a = y[1];
      end else begin
        a = y[0];
      end     
    end

    // Offset MIN_SUM algorithm
    if (`LDEC_MINSUM_ALG == 1) begin
      if (a <= {1'b0, llrUnityDiv2}) begin
        a = `LDEC_PAD(1'b0, `LDEC_CHK_BITS-2);
      end else begin
        a = a - {1'b0, llrUnityDiv2};
      end
    end
    
    softSub = a;
    
  end
endfunction // softSub

//------------------------------------------------------------------------------
// getSgn
//------------------------------------------------------------------------------

function automatic `LDEC_sgnZType getSgnV(input `LDEC_varZ_P varRespP);
  // lint: sign bits are extracted, others are unused.
  reg [`LDEC_VAR_BITS-1:0] varResp `LDEC_varZ_R;
  reg `LDEC_sgnZType       y;
  integer idx1; // packing
  integer zI;
  begin
    `LDEC_UNPACKF(varRespP, varResp, `LDEC_VAR_BITS, `LDEC_Z_MAX);
    for (zI=0; zI<`LDEC_Z_MAX; zI=zI+1) begin
      y[zI] = varResp[zI][`LDEC_VAR_BITS-1];
    end
    getSgnV = y;
  end
endfunction // getSgnV

function automatic `LDEC_sgnZType getSgnC(input `LDEC_chkRespSlvZType a);
  reg `LDEC_sgnZType y;
  integer zI;
  begin
    for (zI=0; zI<`LDEC_Z_MAX; zI=zI+1) begin
      y[zI] = a[(zI+1)*`LDEC_CHK_BITS-1];
    end
    getSgnC = y;
  end
endfunction // getSgnC

//------------------------------------------------------------------------------
// getAbsC
//------------------------------------------------------------------------------

// Extract abs bits from concatenated sgn/abs samples.
function automatic `LDEC_chkRespAbsZ_P getAbsC (input `LDEC_chkRespSlvZType a);
  reg `LDEC_chkAbsType  y  `LDEC_chkRespAbsZ_R;
  integer base;
  integer i;
  integer idx1; // packing
  begin
    for (i=0; i<=`LDEC_Z_MAX-1; i=i+1) begin
      base = `LDEC_CHK_BITS*i;
      y[i] = a[base+:`LDEC_CHK_BITS-1];
    end
    `LDEC_PACKF(y, getAbsC, `LDEC_CHK_BITS-1, `LDEC_Z_MAX);
  end
endfunction

//------------------------------------------------------------------------------
// 2D to 1D vector mapping
//------------------------------------------------------------------------------

// Use pack and unpack instead.
//function automatic `LDEC_varZ_P varFmSlv (input `LDEC_varSlvZType a);
//function automatic `LDEC_varSlvZType varToSlv (input `LDEC_varZType a);

//------------------------------------------------------------------------------
// combineZ (pack sgn/abs)
//------------------------------------------------------------------------------

function automatic `LDEC_chkRespSlvZType combineZ (input `LDEC_sgnZType      sgnZ,
                                                   input `LDEC_chkRespAbsZ_P absZP);
  reg     `LDEC_chkRespSlvZType  y;
  reg     `LDEC_chkAbsType       absZ  `LDEC_chkRespAbsZ_R;
  integer                        base;
  integer                        i;
  integer                        idx1;
  begin  
    `LDEC_UNPACKF(absZP, absZ, `LDEC_CHK_BITS-1, `LDEC_Z_MAX);
    for (i=0; i<=`LDEC_Z_MAX-1; i=i+1) begin
      base = `LDEC_CHK_BITS*i;
      y[base+:`LDEC_CHK_BITS-1] = absZ[i];
      y[base+`LDEC_CHK_BITS-1] = sgnZ[i];
    end
    combineZ = y;
  end
endfunction // combineZ


//------------------------------------------------------------------------------
// countBits2
//------------------------------------------------------------------------------

// Count the number of bits in a vector, guiding the synthesis to use a
// two level tree.
function automatic [numBits(`LDEC_Z_MAX)-1:0] countBits2 (input [`LDEC_Z_MAX-1:0] a);
  
  reg [numBits(`LDEC_Z_SLICE2)-1:0] sliceCnt;
  reg [numBits(`LDEC_Z_MAX)-1:0]    count;
  integer                           c1, c2;
  begin
    // A two level counter whereby we have a number of counters each counting
    // bits in their own slice of the input && begin have another set of
    // counters counting the results. For FPGA a slice size of 4 is envisioned
    // to result in the use of 4 input LUTs at the first level of counters.
    count = `LDEC_PAD(1'b0, numBits(`LDEC_Z_MAX)-1);
    for (c1=0; c1<=ceilDiv(`LDEC_Z_MAX, `LDEC_Z_SLICE2) - 1; c1=c1+1) begin
      // Count the set bits in our slice.
      sliceCnt = `LDEC_PAD(1'b0, numBits(`LDEC_Z_SLICE2)-1);
      for (c2=0; c2<=minimum(`LDEC_Z_SLICE2, `LDEC_Z_MAX-c1*`LDEC_Z_SLICE2)-1; c2=c2+1) begin
        if (a[c1*`LDEC_Z_SLICE2 + c2] == 1'b1) begin
          sliceCnt = sliceCnt + `LDEC_PAD(1'b1, numBits(`LDEC_Z_SLICE2)-1);
        end
      end
      // Sum the slices.
      count = count + `LDEC_PAD(sliceCnt, numBits(`LDEC_Z_MAX)-numBits(`LDEC_Z_SLICE2));
    end
    countBits2 = count;
  end
endfunction // countBits2
// pragma coverage block = on 
