//------------------------------------------------------------------------------
// ldcpDecFwd.v
// 
// Description
//   Check Response forward pass including Var Response backward step.
//
//   Processes short and shift run a cycle ahead of the main datapath
//   && generate registered signals used by the datapath so T-1 input signals
//   are used (denoted by _1 postfix).
//
//   Pipelining with the decoder is mostly here. Additional support comes
//   in ctrl, which needs to inject stalls due to the additional latency,
//   and in crFifo, which is bigger, again because of additional latency.
//
// Inputs:
// (Programmable config)
//   nShrtFloor     : Floor of number of shorting bits.
//   shrtBlkMod     : 1 when blkNum < shrtMod.
//   z              : The macro cells size for the current code.
//   k              : Data-part size for the code.
// (From control)
//   firstIteration : first Iteration of the decode.
//   col            : The column number to process.
//   rowStart       : Start of row.
//   cycShift       : The cyclical shift for this column.
//   flag           : The flag bits for this column (2 bits).
//   scaleFlag      : True when LLRs are to be scaled back to reduce growth.
//   llrUnity       : Integer that represents unity (gets scaled each iter).
//   latchChkMetric : Latch the check metric results (end of fwd & bwd rows).
// (From memories)
//   varMetricRd    : Read bus of signal metrics.
//   chkRespRd      : Check response generated by Bwd. (sgn+abs).
// Outputs:
// (To Bwd)
//   chkMetric2Reg  : Soft sum of all varible responses for a row. (sgn+abs).
//   shrtCol        : macro col numbers where the two micro columns used for
//                    varMetricSum (0 & 26) are shortened.
// (To memories)
//   varRespWr      : Write bus to var response memory. (sgn+abs).
//  
// 2 Apr 2010 M. Rumsey - Ported from C reference model.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//------------------------------------------------------------------------------

`include "ldpcDec.vh"

module ldpcDecFwd
  #(parameter
    NC_HI = numBits(`LDEC_NCOLS_RIGHT)-1,
    Z_HI  = numBits(`LDEC_Z_MAX-1)-1,
    K_HI  = numBits(`LDEC_K_MAX-1)-1)
  (
   input                                  nReset,
   input                                  clk,
   input                                  enable,
   input [numBits(`LDEC_N_MAX-1)-1:0]     nShrtFloor,
   input                                  shrtBlkMod_1,
   // From control    
   input [numBits(`LDEC_Z_ENUM_MAX)-1:0]  zEnum,
   input [numBits(`LDEC_Z_MAX)-1:0]       z,
   input [numBits(`LDEC_K_MAX)-1:0]       k,
   input                                  firstIteration,
   input                                  firstIteration_1,
   input [NC_HI:0] col_1,
   input [NC_HI:0] col,
   input                                  rowStart,
   input [Z_HI:0]                         cycShift_1,
   input [Z_HI:0]                         cycShift,
   input [`LDEC_FLAG_BITS-1:0]            flag,
   input [`LDEC_FLAG_BITS-1:0]            flag_1,
   input                                  fwdStall,
   input                                  fwdStall_1,
   input                                  scaleFlag,
   input `LDEC_chkAbsType                 llrUnity,
   input                                  latchChkMetric,
   // From memories   
   input `LDEC_varZ_P                     varMetricRd,
   input `LDEC_sgnZType                   chkRespRdSgn,
   input `LDEC_chkRespAbsZ_P              chkRespRdAbsP,
   // To Bwd block
   output `LDEC_sgnZType                  chkMetricRegSgnOut,
   output `LDEC_chkMetricAbsZ_P           chkMetricRegAbsOut,
   output `LDEC_shrtCol_P                 shrtColOut,
   // To memories 
   output `LDEC_varZ_P                    varRespWrOut);

`include "ldpcDecFuncs.vh"

  // Unpacked types
  wire `LDEC_chkAbsType             chkRespRdAbs       `LDEC_chkRespAbsZ_R;
  
  // Local variables - no registers required
  wire                              metricFirstUse;
  wire                              metricFirstUse_1;
  reg  `LDEC_sgnZType               varRespZSgn;
  reg  `LDEC_varType                varRespZ           `LDEC_varZ_R;
  reg  `LDEC_chkAbsType             varRespZCmAbsU     `LDEC_chkRespAbsZ_R;
  reg  `LDEC_chkRespAbsZ_P          varRespZCmAbs;
  reg  `LDEC_varP1Type              varRespP1Z         `LDEC_varP1Z_R;
  wire `LDEC_sgnZType               chkRespZSgn;
  reg  `LDEC_chkAbsType             chkRespZAbs        `LDEC_chkRespAbsZ_R;
  wire [numBits(`LDEC_K_MAX)-1:0]   k2;

  // Optionally pipelined signals
  // Pipe A
  wire `LDEC_varZ_P                 varMetricDA;
  reg  `LDEC_varZ_P                 varMetricMidDA;
  wire `LDEC_varZ_P                 varMetricMid;
  reg  [Z_HI:0]                     leftShiftDA;
  reg                               firstIterationDA;
  reg                               metricFirstUseDA;
  reg                               fwdStallDA;
  reg                               rowStartDA;
  reg                               latchChkMetricDA;
  reg [NC_HI:0 ]                    colDA;
  // Pipe B                         
  reg  `LDEC_chkType                chkRespZDB         `LDEC_chkZ_R;
  reg  `LDEC_varZ_P                 varMetricDB;
  wire `LDEC_varType                varMetricDBU       `LDEC_varZ_R;
  
  reg                               fwdStallDB;
  reg                               rowStartDB;
  reg                               latchChkMetricDB;
  reg `LDEC_chkAbsType              llrUnityDB;
  reg                               scaleDB;
  reg [NC_HI:0]                     colDB;
  //  -- Pipe C                     
`ifdef BRC_SIMU_ON                  
//synthesis translate_off           
//synopsys translate_off            
  reg  `LDEC_varZ_P                 varRespZDC;
//synthesis translate_on            
//synopsys translate_on             
`endif //BRC_SIMU_ON                
  reg `LDEC_sgnZType                varRespZSgnDC;
  reg `LDEC_chkRespAbsZ_P           varRespZCmAbsDC;
  reg                               fwdStallDC;
  reg                               rowStartDC;
  reg                               latchChkMetricDC;
  reg `LDEC_chkAbsType              llrUnityDC;
                                    
  // Local registers                
  reg [Z_HI:0]                      leftShift;
  wire [`LDEC_Z_MAX-1:0]            shorted;
  reg `LDEC_cycShift_L              cycLast            `LDEC_cycShift_R;

  genvar idx1;

  
  assign metricFirstUse  = flag[`LDEC_FLAG_FIRST_VAR_USE_BIT];
  assign metricFirstUse_1  = flag_1[`LDEC_FLAG_FIRST_VAR_USE_BIT];

  //---------------------------------------------------------------------------
  // Unpack inputs
  //---------------------------------------------------------------------------
  
  // VarMetric stays packed for pipeling re-timing and is then unpacked.
  `LDEC_UNPACK(gVM, varMetricDB, varMetricDBU, `LDEC_VAR_BITS, `LDEC_Z_MAX)
  `LDEC_UNPACK(gCR, chkRespRdAbsP, chkRespRdAbs, `LDEC_CHK_BITS-1, `LDEC_Z_MAX)

  //---------------------------------------------------------------------------
  // Shortening
  //---------------------------------------------------------------------------

  generate
    
    if (`LDEC_SHRT_SUPPORT) begin: gShrt

      reg `LDEC_colType shrtCol [1:0];
      
      assign k2  = k-nShrtFloor-`LDEC_PAD(shrtBlkMod_1, K_HI);
      
      ldpcDecShort short1
        (
         .nReset       (nReset),
         .clk          (clk),
         .zEnum        (zEnum),
         .z            (z),
         .k            (k),
         .k2           (k2),
         .col_1        (col_1),
         .stall_1      (fwdStall_1),
         .stall        (fwdStall),
         .cycShift_1   (cycShift_1),
         .cycShift     (cycShift),
         .shortedOut   (shorted));
  
      // Capture the macro columns where shortening starts for the two
      // varMetrics that are sub-sampled for varMetricSum. shorted is timed
      // to pipeline B. The final shrtCol must be co-incident with chkMetricRegs,
      // which means it is latched with latchChkMetricDC.
      always @(posedge(clk) `LDEC_RESET_STR_B)
      begin: pShrtCol
        reg `LDEC_colType shrtColV      `LDEC_shrtCol_R;
        reg `LDEC_colType shrtColLatchV `LDEC_shrtCol_R;
        integer idx1;
        
        if (nReset == 1'b0) begin
          {shrtCol[1], shrtCol[0]} <= {2 {`LDEC_PAD(1'b0, NC_HI)}};
          {shrtColLatchV[1], shrtColLatchV[0]} = {2 {`LDEC_PAD(1'b0, NC_HI)}};
          {shrtColV[1], shrtColV[0]} = {2 {`LDEC_PAD(1'b0, NC_HI)}};
        end else begin
          if (rowStartDB == 1'b1) begin
            shrtColV[0] = `LDEC_NCOLS_RIGHT;
            shrtColV[1] = `LDEC_NCOLS_RIGHT;
          end
          if ((shorted[0] == 1'b1) && (colDB < shrtColV[0])) begin
            shrtColV[0] = colDB;
          end
          if ((shorted[26] == 1'b1) && (colDB < shrtColV[1])) begin
            shrtColV[1] = colDB;
          end
          if (latchChkMetricDB == 1'b1) begin
            `LDEC_COPY(shrtColV, shrtColLatchV, 2);
          end
          if (latchChkMetricDC == 1'b1) begin
            `LDEC_COPYQ(shrtColLatchV, shrtCol, 2);
          end
        end
      end // pShrtCol
      assign shrtColOut = {shrtCol[1], shrtCol[0]}; // packing
    end else begin // else generate
      assign shrtColOut = {2 {`LDEC_PAD(1'b0, NC_HI)}}; // packing
      assign shorted = `LDEC_PAD(1'b0, `LDEC_Z_MAX-1);
    end
  endgenerate // gShrt
  
  //---------------------------------------------------------------------------
  // Cyclical Shifting
  //---------------------------------------------------------------------------   

  
  // The logical left shift is 'cycShift', however the varMetric was previously
  // stored without a required right shift of +cycLast, which needs to be done
  // as well. See "Vectorisation Using Cyclical Codes" in BRC005.
  always @(posedge(clk) `LDEC_RESET_STR)
  begin: pShift
    reg signed [Z_HI+1:0] shift;
    
    if (nReset == 1'b0) begin
      leftShift <= `LDEC_PAD(1'b0, Z_HI);
    end else begin
      if (fwdStall_1 == 1'b0) begin
        shift = $signed({1'b0, cycShift_1});
        if ((firstIteration_1 && metricFirstUse_1) == 1'b0) begin
          shift = shift - $signed({1'b0, cycLast[col_1]});
        end       
        leftShift <= modZ(shift, Z_SIZES2(zEnum));
      end
    end
  end
  
  always @(posedge(clk) `LDEC_RESET_STR_B)
  begin: pShift2
    reg signed [numBits(`LDEC_Z_MAX-1):0] shift;
    integer                               idx1;
    
    if (nReset == 1'b0) begin
      `LDEC_INITQ(cycLast, `LDEC_PAD(1'b0, Z_HI), `LDEC_NCOLS);
    end else begin
      if (fwdStall_1 == 1'b0) begin
        cycLast[col_1] <= cycShift_1;
      end
    end
  end //pShift


  // Cyclical shifter. Each call is able to handle multiple bits of the shift but
  // it is split into two parts so that registering may be added in the middle.

  // Shift by the bottom few bits of leftshift.
  assign varMetricMid = ldpcDecRotateDownZ(varMetricRd, zEnum, 
                                           `LDEC_RDZ_LEVEL, 0, leftShift);
  
  // Shift by the upper bits of leftshift after optionally registering
  // the 'bot' signals.
  assign varMetricDA = ldpcDecRotateDownZ(varMetricMidDA, zEnum,
                                          numBits(`LDEC_Z_MAX)-1,
                                          `LDEC_RDZ_LEVEL+1, leftShiftDA);
 
  // Optionally register the var metrics and all downstream control signals.
  // Note that chkResp is delayed in the CrFifo and ldpcDecShort handles
  // delay of shorted. PipeA causes all bwd processing and the
  // cyclical shifter to share a cycle.
  generate
    if (`LDEC_PIPE_A == 1) begin: gPipeA
      always @(posedge(clk) `LDEC_RESET_STR)
      begin: pPipe
        if (nReset == 1'b0) begin
          varMetricMidDA <= `LDEC_PAD(1'b0, `LDEC_VAR_BITS*`LDEC_Z_MAX-1);
          
          firstIterationDA <= 1'b1;
          metricFirstUseDA <= 1'b1;
          fwdStallDA <= 1'b0;
          rowStartDA <= 1'b0;
          colDA <= `LDEC_PAD(1'b0, NC_HI);
          latchChkMetricDA <= 1'b0;
          // if (`LDEC_RESET_ALL) begin (Different to VHDL)
          leftShiftDA <= `LDEC_PAD(1'b0, Z_HI);
          // end
        end else begin
          varMetricMidDA <= varMetricMid;
          firstIterationDA <= firstIteration;
          metricFirstUseDA <= metricFirstUse;
          fwdStallDA <= fwdStall;
          rowStartDA <= rowStart;
          colDA <= col;
          latchChkMetricDA <= latchChkMetric;
          leftShiftDA <= leftShift;
          // Note that ScaleFlag and llrUnity receive retiming in
          // ldpcDecCtrl so no further retiming required.
        end
      end //pPipe
    end else begin
      always @(*) begin
        varMetricMidDA = varMetricMid;
        firstIterationDA = firstIteration;
        metricFirstUseDA = metricFirstUse;
        fwdStallDA = fwdStall;
        rowStartDA = rowStart;
        colDA = col;
        latchChkMetricDA = latchChkMetric;
        leftShiftDA = leftShift;
      end
    end
  endgenerate //  gPipeA

  //---------------------------------------------------------------------------
  // chkResp Manipulation
  //---------------------------------------------------------------------------

  // Set chkResp to 0 on first iteration to force varResp=varMetric. Scale it
  // on occasions when varMetric has already been scaled. Note that `LDEC_PIPE_A does
  // not add registering on this path, instead a balancing delay is added in
  // the FIFO so only control signals need to be delayed if `LDEC_PIPE_A is active.
  //always @(chkRespRdAbs, firstIterationDA, scaleFlag, metricFirstUseDA)
  always @(*)
  begin: pChkResp
    integer zI;
    
    for (zI=0; zI<`LDEC_Z_MAX; zI=zI+1) begin     
      if (firstIterationDA == 1'b0) begin
        if ((scaleFlag && !metricFirstUseDA) == 1'b1) begin
          chkRespZAbs[zI] = roundNoClipC(chkRespRdAbs[zI]);
        end
        else begin
          chkRespZAbs[zI] = chkRespRdAbs[zI];
        end
      end else begin
        chkRespZAbs[zI] = `LDEC_PAD(1'b0, `LDEC_CHK_BITS-2);
      end
    end
  end //pChkResp

  assign chkRespZSgn = (firstIterationDA == 1'b0 ) ? chkRespRdSgn :
                       `LDEC_PAD(1'b0, `LDEC_Z_MAX);

  //---------------------------------------------------------------------------
  // Pipe B 
  //---------------------------------------------------------------------------

  // Here we register varmetric, the output of the barrel shifter. This;
  // equivalent to PIPEA with RDZ Level == 6, so normally PIPEA would be moved
  // earlier (e.g. `LDEC_RDZ_LEVEL == -1). chkResp is also registered AND we also
  // do a conversion to 2's complement.
  
  generate
    if (`LDEC_PIPE_B == 1) begin: gPipeB
      
      always @(posedge(clk) `LDEC_RESET_STR)
      begin: pPipeB
        integer zI;        
        if (nReset == 1'b0) begin
          fwdStallDB <= 1'b0;
          rowStartDB <= 1'b0;
          colDB <= `LDEC_PAD(1'b0, NC_HI);
          latchChkMetricDB <= 1'b0;
          scaleDB <= 1'b0;        
        end else begin
          fwdStallDB <= fwdStallDA;
          rowStartDB <= rowStartDA;
          colDB <= colDA;
          latchChkMetricDB <= latchChkMetricDA;
          // Here signals got A pipelining elsewhere.
          scaleDB <= metricFirstUseDA & scaleFlag;
        end
      end
      
      always @(posedge(clk) `LDEC_RESET_STR_B)
      begin: pPipeB2
        integer zI;
        integer idx1;
        
        if (nReset == 1'b0) begin
          `LDEC_INITQ(chkRespZDB, `LDEC_PADS(1'b0, `LDEC_CHK_BITS-1), `LDEC_Z_MAX);
          varMetricDB <= `LDEC_PAD(1'b0, `LDEC_VAR_BITS*`LDEC_Z_MAX-1);
          llrUnityDB <= `LDEC_PAD(1'b0, `LDEC_CHK_BITS-2);          
        end else begin
          for (zI=0; zI<`LDEC_Z_MAX; zI=zI+1) begin     
            chkRespZDB[zI] <= `LDEC_SGNMAG_TO_ONES(chkRespZSgn[zI], $signed(chkRespZAbs[zI]));
          end
          varMetricDB <= varMetricDA;
          // Here signals got A pipelining elsewhere.
          llrUnityDB <= llrUnity;
        end
      end //pPipeB
      
    end else begin
      
      always @(*) // (chkRespZSgn, chkRespZAbs)
      begin: pPipeB
        integer zI;
        
        for (zI=0; zI<`LDEC_Z_MAX; zI=zI+1) begin     
          chkRespZDB[zI][`LDEC_CHK_BITS-1:0] =
            `LDEC_SGNMAG_TO_ONES(chkRespZSgn[zI], $signed(chkRespZAbs[zI]));
        end
      end //pPipeB

      always @(*) begin
        varMetricDB = varMetricDA;
        fwdStallDB = fwdStallDA;
        rowStartDB = rowStartDA;
        colDB = colDA;
        latchChkMetricDB = latchChkMetricDA;
        llrUnityDB = llrUnity;
        scaleDB = metricFirstUseDA & scaleFlag;
      end
      
    end
  endgenerate //  gPipeBF

  //---------------------------------------------------------------------------
  // Generate varResp
  //---------------------------------------------------------------------------

  always @(*) //  @(chkRespZDB, varMetricDB, shorted)
  begin: pvarResp
    integer zI;
    
    for (zI=0; zI<`LDEC_Z_MAX; zI=zI+1) begin     
      if ((shorted[zI] == 1'b1) && `LDEC_SHRT_SUPPORT) begin
        varRespP1Z[zI] = $signed({1'b0, {(`LDEC_VAR_BITS) {1'b1}}});
      end
      else begin
        // Calculate varResp == varMetric - chkResp       
        varRespP1Z[zI] =
          $signed({varMetricDBU[zI][`LDEC_VAR_BITS-1], varMetricDBU[zI]}) -
          $signed({{`LDEC_VAR_BITS+1-`LDEC_CHK_BITS{chkRespZDB[zI][`LDEC_CHK_BITS-1]}}, chkRespZDB[zI]});
      end
    end
  end //pvarResp

  // Final scale and clip. 

  always @(*) // @(varRespP1Z, scaleDB)
  begin: pVarResp2
    reg signed [`LDEC_VAR_BITS:0]   varRespP1;
    reg signed [`LDEC_VAR_BITS-1:0] varRespV;
    reg signed [`LDEC_CHK_BITS-1:0] clipped;
    integer                         zI;
    integer                         idx1;

    for (zI=0; zI<`LDEC_Z_MAX; zI=zI+1) begin
      varRespP1 = varRespP1Z[zI];
      if (scaleDB == 1'b1) begin
        varRespP1 = roundNoClipV1(varRespP1);
      end
      varRespV = `LDEC_CLIP_S(varRespP1, `LDEC_VAR_BITS+1, `LDEC_VAR_BITS);
      varRespZ[zI] = varRespV;
      // Clip as required for the check metric.
      clipped = `LDEC_CLIP_S(varRespV, `LDEC_VAR_BITS, `LDEC_CHK_BITS);
      varRespZCmAbsU[zI] = `LDEC_ABS1(clipped, `LDEC_CHK_BITS);
    end
    `LDEC_PACKF(varRespZCmAbsU, varRespZCmAbs, `LDEC_CHK_BITS-1, `LDEC_Z_MAX);
  end //pVarResp2
  

  // Get the sign bit
  always @(*) // @(varRespZ)
  begin: pvarRespZSgn
    integer i;   
    for (i=0; i<`LDEC_Z_MAX; i=i+1) begin
      varRespZSgn[i] = varRespZ[i][`LDEC_VAR_BITS-1];
    end    
  end
  
  // Optionally register varResp as used for the chkMetric.
  generate
    if (`LDEC_PIPE_C == 1) begin: gPipeCT
      
      always @(posedge(clk) `LDEC_RESET_STR)
      begin: pPipeC
        if (nReset == 1'b0) begin
          fwdStallDC <= 1'b0;
          rowStartDC <= 1'b0;
          latchChkMetricDC <= 1'b0;
        end else begin
          fwdStallDC <= fwdStallDB;
          rowStartDC <= rowStartDB;
          latchChkMetricDC <= latchChkMetricDB;
        end
      end
      
      always @(posedge(clk) `LDEC_RESET_STR_B)
      begin: pVrDc
        integer idx1;
        if (nReset == 1'b0) begin
          varRespZSgnDC <= `LDEC_PAD(1'b0, `LDEC_Z_MAX);
          varRespZCmAbsDC <= `LDEC_PAD(1'b0, `LDEC_Z_MAX*(`LDEC_CHK_BITS-1)-1);
          llrUnityDC <= `LDEC_PAD(1'b0, `LDEC_CHK_BITS-2);
        end else begin
          varRespZSgnDC <= varRespZSgn;
          varRespZCmAbsDC <= varRespZCmAbs;
`ifdef BRC_SIMU_ON
//synthesis translate_off
//synopsys translate_off
          // for logging
          `LDEC_PACKQ(varRespZ, varRespZDC, `LDEC_VAR_BITS, `LDEC_Z_MAX);         
//synopsys translate_on
//synthesis translate_on
`endif //BRC_SIMU_ON
          llrUnityDC <= llrUnityDB;
        end
      end //pPipeC
      
    end else begin

      always @(*)
      begin: pVrDc2
        integer idx1;        
        varRespZSgnDC = varRespZSgn;
        varRespZCmAbsDC = varRespZCmAbs;
`ifdef BRC_SIMU_ON
//synthesis translate_off
//synopsys translate_off
        `LDEC_PACKF(varRespZ, varRespZDC, `LDEC_VAR_BITS, `LDEC_Z_MAX);
//synopsys translate_on
//synthesis translate_on
`endif //BRC_SIMU_ON
        fwdStallDC = fwdStallDB;
        rowStartDC = rowStartDB;
        latchChkMetricDC = latchChkMetricDB;
        llrUnityDC = llrUnityDB;
      end
    end
  endgenerate //  gPipeCF

  //-------------------------------------------------------------------------
  // Generate ChkMetric
  //-------------------------------------------------------------------------

   ldpcDecCM chkMetric
(
      .clk                (clk),
      .nReset             (nReset),
      .enable             (enable),
      .z                  (z),      
      .fwdStall           (fwdStallDC),
      .rowStart           (rowStartDC),
      .llrUnity           (llrUnityDC),
      .latchChkMetric     (latchChkMetricDC),
      .varRespZSgn        (varRespZSgnDC),
      .varRespZCmAbsP     (varRespZCmAbsDC),
`ifdef BRC_SIMU_ON
//synthesis translate_off
//synopsys translate_off
      .varRespZP          (varRespZDC), // Logging only
//synthesis translate_on
//synopsys translate_on
`endif //BRC_SIMU_ON
      .chkMetricRegSgnOut (chkMetricRegSgnOut),
      .chkMetricRegAbsOut (chkMetricRegAbsOut));

  `LDEC_PACK(gVr, varRespZ, varRespWrOut, `LDEC_VAR_BITS, `LDEC_Z_MAX)
  
endmodule
