//------------------------------------------------------------------------------
// ldpcDecHdRamMux.v
// 
// Description:
//   RAM wrapper for the HardDecion RAM. This is ping-ponged to allow output
//   data to be fetched during the following decode. Furthermore
//   it is dual ported so that the bwd part can read previous
//   estimates && test if there has been a bit inversion.
//
// 7 June 2010 M. Rumsey. Created.
//
// (c) Copyright 2010, Blue Rum Consulting Limited, All Rights Reserved.
//
//-----------------------------------------------------------------------------

`include "ldpcDec.vh"

module ldpcDecHdRamMux
( 
  input                         pp,
    // Comon clock to avoid muxing
  input                         ck,
  input                         freeck,
  input                         nReset,
    // A prt used for reads only
  input                         selA,
  input [`LDEC_HD_RAM_A-1:0]    addrA,
  output [`LDEC_HD_RAM_W-1:0]   dOutA,
    // B prt used for write only
  input [`LDEC_HD_RAM_A-1:0]    addrB,
  input [`LDEC_HD_RAM_W-1:0]    dInB,
  input                         weB,
    // PingPong A port used for reads
  input                         selA2,
  input [`LDEC_HD_RAM_A-1:0]    addrA2,
  output [`LDEC_HD_RAM_W-1:0]   dOutA2
`ifdef LDEC_RAMS_AT_TOP
  ,
  input [`LDEC_HD_RAM_O-1:0]    hdRamOps,
  output [`LDEC_HD_RAM_I-1:0]   hdRamIps
`endif
  );

`include "ldpcDecFuncs.vh"

  wire                        selAX;
  wire                        selAY;
  wire [`LDEC_HD_RAM_A-1:0]   addrAX;
  wire [`LDEC_HD_RAM_A-1:0]   addrAY;
  wire [`LDEC_HD_RAM_W-1:0]   dOutAX;
  wire [`LDEC_HD_RAM_W-1:0]   dOutAY; 

  wire weBX;
  wire weBY;

  // Here are the two RAMs. Since they are ping-ponged they are
  // totally symmetrical. We only write on the B port && read
  // on the A port.
  
`ifdef LDEC_RAMS_AT_TOP
  
  assign hdRamIps  = {selAX, selAY, weBX, weBY,
                      addrAX, addrB, dInB, addrAY};
  assign {dOutAX, dOutAY} = hdRamOps;

`else
  
  ldpcDecHdRam hdRamX
    (
     .ck    (ck),
     // A port always read
     .selA  (selAX),
     .addrA (addrAX),
     .dOutA (dOutAX),
     // B port always write
     .addrB (addrB),
     .dInB  (dInB),
     .weB   (weBX));
  
  ldpcDecHdRam hdRamY
    (
     .ck    (ck),
     // A port always read
     .selA  (selAY),
     .addrA (addrAY),
     .dOutA (dOutAY),
     // B port always write
     .addrB (addrB),
     .dInB  (dInB),
     .weB   (weBY));

`endif
  
  wire [`LDEC_HD_RAM_W-1:0] dOutAYhold;
  reg  [`LDEC_HD_RAM_W-1:0] dOutAYreg;
  reg  selAYD;
  wire [`LDEC_HD_RAM_W-1:0] dOutAXhold;
  reg  [`LDEC_HD_RAM_W-1:0] dOutAXreg;
  reg  selAXD;
  
  // Hold the read data when Sel is deasserted
  always @(posedge(freeck), negedge(nReset))
  begin
    if (nReset==1'b0)
    begin
      dOutAXreg  <= {`LDEC_HD_RAM_W{1'b0}};
      selAXD     <= 1'b0;
    end else begin
      if(selAXD)
        dOutAXreg <= dOutAX;
      selAXD      <= selAX;
    end
  end
  assign dOutAXhold = !selAXD ? dOutAXreg : dOutAX;
  
  always @(posedge(freeck), negedge(nReset))
  begin
    if (nReset==1'b0)
    begin
      dOutAYreg  <= {`LDEC_HD_RAM_W{1'b0}};
      selAYD     <= 1'b0;
    end else begin
      if(selAYD)
        dOutAYreg <= dOutAY;
      selAYD      <= selAY;
    end
  end
  assign dOutAYhold = !selAYD ? dOutAYreg : dOutAY;
  
  // Multiplex the inputs.

   assign selAX = (pp == 1'b0 ) ? selA : selA2;
   assign selAY = (pp== 1'b1 ) ? selA : selA2;

   assign weBX = (pp == 1'b0 ) ? weB : 1'b0;
   assign weBY = (pp == 1'b1 ) ? weB : 1'b0;

   assign addrAX = (pp == 1'b0 ) ? addrA : addrA2;
   assign addrAY = (pp == 1'b1 ) ? addrA : addrA2;

  // Multiplex the outputs

   assign dOutA = (pp == 1'b0 ) ? dOutAX : dOutAY;
   assign dOutA2 = (pp == 1'b1 ) ? dOutAXhold : dOutAYhold;

endmodule

    
    

