//----------------------------------------------------------------------------
// ldcpDecIp.v
// 
// Description
//   The input controller. This registers the ipData word and generates addresses
//   for where the data should be stored.
//
// Inputs:
// (From control)
//   firstBlk       : Pulse on first block synchronised to setRdyToRcv.
//   nShrtFloor     : Floor of number of shortening bits.
//   shrtMod        : 1 when blkNum < nShrtMod.
//   nPuncFloor     : Floor of number of shortening bits.
//   puncMod        : 1 when blkNum < nPuncMod.
//   nRepFloor      : Floor of number of repetition bits.
//   repMod         : 1 when blkNum < nRepMod.
//   z              : The macro cells size for the current code.
//   k              : Data-part size for the code.
//   parityStartCol : The macro column number (rel 0) where parity bits start.
//   numWrPerZ      : ceil(Z/DEC_RAM_IP_WIDTH).
//   enable         : Decoder has been configured so is ready for decode.
//   setRdyToRcv    : Pulse at end of decode || on first enable.
// (From sender)
//   inStrobe       : High when inData is valid
//   ipWidth        : The number of LLRs that are valid values.
//   inDataWord     : `IP_WIDTH samples of `IP_BITS.
//   llrUnity       : Integer that represents unity (gets scaled each iter).
// (From Op block)
// Outputs:
// (To Op Block)
// (To sender)
//   rdyToRcv       : Enables the sender to start sending. Stays high until
//                    the expected amount of data has been transferred.
// (To control)
//   ipLoaded       : Pulse when a complete ip block has been loaded.
// (To memories)
//   wrData         : Write data of length $size(inData).
//   wrOffset       : Write is to the bits starting at wrOffset*$size(inData).
//   wrAddr         : Write address - actually the macro column number.
//   wrEnable       : Memory write enable.
//  
// 13 Apr 2010 M. Rumsey - Ported from C reference model.
// 27 Sep 2010 M. Rumsey - rework of pBitNum for timing improvement.
// 07 Oct 2010 M. Rumsey - Added preprocessor.
//
// (c) Copyright 2010-11, Blue Rum Consulting Limited, All Rights Reserved.
//----------------------------------------------------------------------------

`include "ldpcDec.vh"

// Width of data read from varMetricsRam
`define LDEC_SLICE_LEN `LDEC_DEC_RAM_IP_WIDTH*`LDEC_VAR_BITS

module ldpcDecIp # (
  // also defined in ldpcDecFuncs.svh as NWZ_HI
  parameter NWZ_HI2 = numBits(ceilDiv(`LDEC_Z_MAX, `LDEC_DEC_RAM_IP_WIDTH))-1,
  parameter NUM_USERS = 1
) (
  // Clocking/reset
  input                                    nReset,
  input                                    clk,
  output                                   clkEnOut,
  // User control
  input [NUM_USERS-1:0]                    enabledUsers,
  input [4:0]                              inUser, 
  input                                    ipSkipBlock,
  // From control
  input [numBits(`LDEC_BPS_MAX)-1:0]       bitsPerSymbol,
  input                                    firstBlk,
  input [numBits(`LDEC_N_MAX-1)-1:0]       nShrtFloor,
  input [numBits(`LDEC_MAX_BLKNUM)-1:0]    shrtMod,
  input [numBits(`LDEC_M_MAX)-1:0]         nPuncFloor,
  input [numBits(`LDEC_MAX_BLKNUM)-1:0]    puncMod,
  input [numBits(`LDEC_NREPS_MAX)-1:0]     nRepFloor,
  input [numBits(`LDEC_MAX_BLKNUM)-1:0]    repMod,
  input [numBits(`LDEC_Z_ENUM_MAX)-1:0]    zEnum,
  input [numBits(`LDEC_N_MAX)-1:0]         n,
  input [numBits(`LDEC_K_MAX)-1:0]         k,
  input [numBits(`LDEC_NCOLS-1)-1:0]       parityStartCol,
  input                                    enable,
  input                                    disabling,
  input                                    setRdyToRcv,
  input                                    decodeStart,

  input [NWZ_HI2:0]                        numWrPerZ,
                                             
  // From sender                             
  input                                    inStrobe, 
  input [numBits(`LDEC_IP_WIDTH)-1:0]      ipWidth,
  input [`LDEC_IP_WIDTH*`LDEC_IP_BITS-1:0] inDataWord,
  input signed [`LDEC_CHK_BITS-1:0]        llrUnity,
  output wire [numBits(`LDEC_MAX_IPBITS)-1:0] numIpBits,
  output wire                              numIpBitsValid,
                                             
  // From memory (Var Metrics readback for repeat combining)
  input [`LDEC_VM_RAM_W-1:0]               rdData,
                                         
  // To sender                           
  output                                   rdyToRcvOut, 
  output                                   lastIpSampleOut, 
                                         
  // To control                          
  output reg `LDEC_chkAbsType              llrUnity0Out,
  output reg `LDEC_chkAbsType              llrUnity1Out,
  output                                   ipLoadedOut,
  output wire                              ipAlmostLoadedOut, // RW modification
  
  // Dbg
  output wire [5:0]                        dbgIp, 

  // To memory
  output [`LDEC_SLICE_LEN-1:0]             wrDataOut,
  output [NWZ_HI2:0]                       wrOffsetOut,
  output [numBits(`LDEC_NCOLS-1)-1:0]      wrAddrOut,
  output                                   wrEnableOut,
  output [numBits(`LDEC_NCOLS-1)-1:0]      rdAddrOut,
  output                                   rdEnableOut,
  output                                   lastBlkOfSymOut,
  output reg                               waitFirstWordOfSymOut);

`include "ldpcDecFuncs.vh"
 
  //typedef enum bit [1:0] {dataState, parityState, puncState} captureStateType;
  localparam [1:0] dataState = 0;
  localparam [1:0] parityState = 1;
  localparam [1:0] puncState = 2;

  localparam NUM_OFFSETS = ceilDiv(`LDEC_Z_MAX, `LDEC_DEC_RAM_IP_WIDTH);
  localparam NO_HI = numBits(NUM_OFFSETS)-1;
  localparam N_HI = numBits(`LDEC_N_MAX)-1;
  localparam M_HI = numBits(`LDEC_M_MAX)-1;
  localparam BC_HI = numBits(`LDEC_MAX_IPBITS)-1;
  localparam IDW_HI = `LDEC_IP_WIDTH*`LDEC_IP_BITS-1;
  localparam BN_HI = numBits(`LDEC_MAX_BLKNUM)-1;
  localparam NC_HI = numBits(`LDEC_NCOLS-1)-1;
  localparam REP_HI = numBits(`LDEC_NREPS_MAX)-1;
  localparam [NC_HI:0] VM_RAM_D = `LDEC_VM_RAM_D;
  localparam [NC_HI:0] VM_RAM_D1 = `LDEC_VM_RAM_D-1;
  
  // Simple register of the input data.
  reg [IDW_HI:0]                             inDataWordReg;
  // The input data collected into `LDEC_DEC_RAM_IP_WIDTH samples suited to the VM RAM.
  wire [`LDEC_DEC_RAM_IP_WIDTH*`LDEC_IP_BITS-1:0] inDataWordPre;
  
  // The input data further expanded with each sample now `LDEC_VAR_BITS wide.
  reg [`LDEC_SLICE_LEN-1:0]              dataWordExt;
  reg [numBits(`LDEC_IP_WIDTH)-1:0]      ipWidthReg;
  reg  [BC_HI:0]                         numIpBitsNxt;  
  reg  [BC_HI:0]                         ipBitCount;
  wire [`LDEC_LB_HI:0]                   loadBase;
  reg                                    lastIpSample;
  reg  [NO_HI:0]                         rowIdx;
                                         
  wire                                   clkEn;
  wire                                   captureEn;
  reg                                    capturing;
  reg                                    capturingNxt;
  reg  [N_HI:0]                          currentBitNum;
  reg                                    inStrobeReg;
  wire                                   inStrobePre;
  wire                                   inStrobePre_1;
  reg                                    inStrobePre_1Last;
  wire                                   ipStall;
  reg                                    setRdyToRcvD1;
  reg  [numBits(`LDEC_NCOLS)-1:0]        addr;
                                         
  reg  [NO_HI:0]                         offset;
  reg  [numBits(`LDEC_NCOLS)-1:0]        wrAddr;
  reg  [NO_HI:0]                         wrOffset;
  wire [numBits(`LDEC_NCOLS)-1:0]        rdAddr;
  wire [NO_HI:0]                         rdOffset;
                                         
  reg                                    rdyToRcv;
  reg                                    rdyToRcvD1;
  reg                                    ipLoaded;
  reg                                    punctured;
  reg                                    puncturedNxt;
  reg [1:0]                              captureState;    // captureStateType
  reg [1:0]                              captureStateNxt; // captureStateType
                                         
  // New signals for repeat combining    
  wire [`LDEC_SLICE_LEN-1:0]             rdDataWord;
  reg [`LDEC_SLICE_LEN-1:0]              dataWordComb;
  wire [numBits(`LDEC_K_MAX)-1:0]        endData;
  wire [N_HI:0]                          endParity;
  wire [REP_HI:0]                        numReps;
  reg [REP_HI:0]                         repsRemaining;
  
  reg                                    repeatMode;
  reg                                    repeatModeNxt;
  reg                                    ipRepeating;
  reg                                    isEndCapture;
  reg                                    isEndCaptureNxt;
  reg                                    isLastWord;
  reg                                    isEndData;
  reg                                    isEndParity;

  // optional re-timing
  reg [`LDEC_SLICE_LEN-1:0]              dataWordCombD1;
  reg [numBits(`LDEC_NCOLS)-1:0]         wrAddrD1;
  reg [NO_HI:0]                          wrOffsetD1;
  reg                                    wrEnableD1;  

  wire                                   puncturing;
  wire                                   puncModComp;
  reg  [BN_HI:0]                         blkNumArray[0:NUM_USERS-1];
  reg  [NUM_USERS-1:0]                   blkPingPongArray;
  reg                                    countingStalls;

  // OFDM symbol tracking. Note that MAX_IPBITS is marginally larger
  // than 1 OFDM/STBC symbol.
  reg  [BC_HI:0]                         symbolBitRunningCount;
  reg  lastBlkOfSym;
  
  // Extract a word from a concatenated set of words
  function automatic [`LDEC_SLICE_LEN-1:0] deMuxA(input [`LDEC_VM_RAM_W-1:0] word,
                                        input [NO_HI:0] offset);
    reg [`LDEC_SLICE_LEN-1:0]  result;
    integer                    i;
    begin
      result = `LDEC_PAD(1'b0, `LDEC_SLICE_LEN-1);
      for (i=0; i<NUM_OFFSETS; i=i+1) begin
        if (i[NO_HI:0] == offset) begin
          result = word[i*`LDEC_SLICE_LEN+:`LDEC_SLICE_LEN]; 
        end
      end
      deMuxA = result;
    end
  endfunction
  
  // DBG
  assign dbgIp = {ipBitCount[4:0],blkNumArray[0][0]};
  
  // Enable && clock enable for capturing input. 
  assign captureEn  = inStrobe | inStrobeReg | inStrobePre |
                      capturing | setRdyToRcv | setRdyToRcvD1 | decodeStart |
                      ipLoaded | countingStalls;
  assign clkEn  = captureEn | disabling | ipSkipBlock;

  assign puncModComp = (blkNumArray[inUser] < puncMod) ? 1'b1 : 1'b0;
  assign puncturing = (puncModComp == 1'b1 ||
                       (nPuncFloor != `LDEC_PAD(1'b0, M_HI))) ? 1'b1 : 1'b0;

  // This is the number of ip bits in the next block.
  always @(*)
  begin : pNumIpBitsNext
   reg  [BC_HI:0] b; 
    b = numIpBits;
    // If the next block is on a boundary where shrt etc change begin adjust.
    if (numIpBitsValid) begin


      if (blkNumArray[inUser] + `LDEC_PAD(1'b1, BN_HI) == shrtMod) begin
        b = b + `LDEC_PAD(1'b1, BC_HI);
      end
      if (blkNumArray[inUser] + `LDEC_PAD(1'b1, BN_HI) == puncMod) begin
        b = b + `LDEC_PAD(1'b1, BC_HI);
      end
      if (blkNumArray[inUser]+ + `LDEC_PAD(1'b1, BN_HI) == repMod) begin
        b = b - `LDEC_PAD(1'b1, BC_HI);
      end
    end
    numIpBitsNxt = b;
  end //pNumIpBitsNext
  

`ifdef BRC_SIMU_ON
//synthesis translate_off
//synopsys translate_off

  always @(posedge(clk))
  begin
    if ((~enable & inStrobe & rdyToRcv))
      $error("Attempted input to decoder when not enabled.");
  end

//synthesis translate_on
//synopsys translate_on
`endif //BRC_SIMU_ON

  //-------------------------------------------------------------------------
  // Capture input data to assure good timing.
  //-------------------------------------------------------------------------
  
  // Capture data on the input interface
  
  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pCapture
    reg [BC_HI:0] ipBitCountV;
    
    // Asynchronous reset.
    if (nReset == 1'b0) begin
      rdyToRcv <= 1'b0;
      rdyToRcvD1 <= 1'b0;
      inStrobeReg <= 1'b0;
      ipBitCount <= `LDEC_PAD(1'b0, BC_HI);
      // if (`LDEC_RESET_ALL) begin (Different to VHDL)
      inDataWordReg <= `LDEC_PAD(1'b0, IDW_HI);
      ipWidthReg <= `LDEC_PAD(1'b0, numBits(`LDEC_IP_WIDTH)-1);      
      // end
    end else begin
      if ((ipStall == 1'b0) && (clkEn == 1'b1 || `LDEC_CLK_GATING == 1)) begin
        ipBitCountV = ipBitCount;
        rdyToRcvD1 <= rdyToRcv;
        if (rdyToRcv == 1'b1) begin
          // retime the strobe
          inStrobeReg <= inStrobe;
          ipWidthReg <= ipWidth;
          // register the data
          if (inStrobe == 1'b1) begin
            inDataWordReg <= inDataWord;            
            ipBitCountV = ipBitCountV +
                          `LDEC_PAD(ipWidth, BC_HI-(numBits(`LDEC_IP_WIDTH)-1));
            // On last sample drop rdyToRcv
            if (lastIpSample == 1'b1) begin
              rdyToRcv <= 1'b0;
            end
          end
        end
        // Synchronous reset.
        if (enable == 1'b0 || rdyToRcv == 1'b0) begin
          inStrobeReg <= 1'b0;
          rdyToRcv <= 1'b0;
          rdyToRcvD1 <= 1'b0;
          ipBitCountV = `LDEC_PAD(1'b0, BC_HI);          
        end
        // Count input bits so that we can drop rdyToRcv at the end of a block.
        if (setRdyToRcv && enable) begin
          rdyToRcv <= 1'b1;
          if (`LDEC_CONCAT_IP) begin
            // Initiate bit count with bits that were loaded with the last word
            // of the previous block.
            ipBitCountV = `LDEC_PAD(loadBase, BC_HI-`LDEC_LB_HI);
          end
          else begin
            ipBitCountV =`LDEC_PAD(1'b0, BC_HI) ;
          end
        end
        ipBitCount <= ipBitCountV;
      end
    end
  end //pCapture

  always @(numIpBitsValid, ipBitCount, ipWidth, numIpBits)
  begin
    lastIpSample = 1'b0;        
    // On last but 1 sample set lastIpSample
    if (numIpBitsValid == 1'b1) begin
      if (ipBitCount +
          `LDEC_PAD(ipWidth, BC_HI-(numBits(`LDEC_IP_WIDTH)-1)) >=
          numIpBits) begin
        lastIpSample = 1'b1;
      end
    end
  end
  
  
  //---------------------------------------------------------------------------
  // OFDM/STBC symbol boundary detection
  //---------------------------------------------------------------------------
  
  // symbolBitRunningCount counts bits received.
  // When this exceeds bitsPerSymbol it is decremented by bitsPerSymbol.
  // When it stays at bitsPerSymbol for a few cycles we know the input has
  // stalled on the symbol boundary.
  // There are two cases of symbol size handled in the code. Look for
  // numIpBits >= bitsPerSymbol && numIpBits < bitsPerSymbol;
  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pSymBoundary
    reg  [numBits(`LDEC_MAX_IPBITS+`LDEC_N_MAX)-1:0] bitV;
    reg  [1:0] stallCountV;

    if (nReset == 1'b0) begin
      countingStalls <= 1'b0;
      stallCountV = 2'd0;
      lastBlkOfSym <= 1'b1;
      symbolBitRunningCount <= `LDEC_PAD(1'b0, BC_HI);
      waitFirstWordOfSymOut <= 1'b1;               
    end else begin
      if (enable == 1'b0) begin
        // ctrl block needs this initialisation.
        lastBlkOfSym <= 1'b1;
        symbolBitRunningCount <= `LDEC_PAD(1'b0, BC_HI);
        stallCountV = 2'd0;
        waitFirstWordOfSymOut <= 1'b1;         
        countingStalls <= 1'b0;
      end
      else begin
        countingStalls <= 1'b0;
        // LDPC block is comprised of >= 1 OFDM symbols.
        // Maintain symbolBitRunningCount. This calc may take several cycles.
        if ((numIpBits >= bitsPerSymbol) && (numIpBitsValid)) begin
          if (symbolBitRunningCount > bitsPerSymbol) begin
            countingStalls <= 1'b1;    // make sure we have a clock
            symbolBitRunningCount <= symbolBitRunningCount - bitsPerSymbol;
          end
        end
        // Relating to above, we need a clock. This is a way to get one.
        if (!numIpBitsValid) begin
          countingStalls <= 1'b1;
        end
        // Analyse the block that has just been picked up by the core.
        if (decodeStart) begin
          waitFirstWordOfSymOut <= 1'b0;         
          countingStalls <= 1'b1;    // make sure we have a follow on clock
          stallCountV = 2'd0;
          bitV = symbolBitRunningCount + numIpBits;
          // Determine lastBlkOfSym
          if ((numIpBits >= bitsPerSymbol)  && numIpBitsValid) begin
            // Small symbolcase. Every block is last of the symbol.
            lastBlkOfSym <= 1'b1;
          end else begin
            
            if (bitV > bitsPerSymbol) begin
              // Had a wrap. This is first block of symbol.
              bitV = bitV - bitsPerSymbol;              
            end            
            // If the next block causes a wrap begin the
            //  current block must be last of symbol.
            if (bitV + numIpBitsNxt > bitsPerSymbol) begin
              lastBlkOfSym <= 1'b1;
            end else begin
              lastBlkOfSym <= 1'b0;
            end
          end
          symbolBitRunningCount <= bitV;
          
        end
        else if (lastBlkOfSym == 1'b1) begin
          // Already had last block so now receiving the block that crosses
          // the symbol boundary. Watch out for stalls at the boundary. A
          // counter is used to filter spurious stall. Watch out for
          // the duration of clkEn if extending this.
          if (stallCountV == 2'd3) begin
            waitFirstWordOfSymOut <= 1'b1;         
          end
          // symbolBitRunningCount was made up to end of last block so
          // adding ipBitCount [which applies:this block] provides
          // a dynamic symbol count.
          bitV = symbolBitRunningCount + ipBitCount;
          if (bitV == bitsPerSymbol) begin
            // Have loaded all bits of the symbol. Note that we do !
            // need to check inStrobe because if (it was there) begin ipBitCount
            // would !be static for several cycles.
            if (stallCountV < 2'd3) begin
              stallCountV = stallCountV + 2'b1;
              countingStalls <= 1'b1;
            end
            else
              stallCountV = 2'd0;
          end
          else
            stallCountV = 2'd0;
        end
      end
    end

  end 
  
  //-------------------------------------------------------------------------
  // Preprocessor to map `LDEC_IP_WIDTH bits to `LDEC_DEC_RAM_IP_WIDTH
  //-------------------------------------------------------------------------

  // This preprocessor takes a stream of samples `LDEC_IP_WIDTH wide && remaps 
  // them to RAM_IP_WIDTH wide.
  generate
    if (`LDEC_IP_PREPROC) begin: gPre1
    ldpcDecIpPre ipPre (
      .nReset         (nReset),
      .clk            (clk),
      .clkEn          (clkEn),
      // From control
      .enable         (enable),
      .blkNum         (blkNumArray[inUser]),
      .setRdyToRcv    (setRdyToRcv),
      .setRdyToRcvD1  (setRdyToRcvD1),
      .rdyToRcvD1     (rdyToRcvD1),
      .nShrtFloor     (nShrtFloor),
      .shrtMod        (shrtMod),
      .nPuncFloor     (nPuncFloor),
      .puncMod        (puncMod),
      .nRepFloor      (nRepFloor),
      .repMod         (repMod),
      .n              (n),
      .k              (k),
      // From IP block
      .isLastWord     (isLastWord),
      // From sender 
      .inStrobe       (inStrobeReg),
      .ipWidth        (ipWidthReg),
      .inDataWord     (inDataWordReg),
      // To IP block
      .loadBaseOut    (loadBase),
      .numIpBitsValid (numIpBitsValid),
      .numIpBits      (numIpBits),
      .k2Out          (endData),
      .endParity2Out  (endParity),
      .numRepsOut     (numReps),
      .strobeOut_1    (inStrobePre_1),
      .strobeOut      (inStrobePre),
      .ipStallOut     (ipStall),
      .dataWordOut    (inDataWordPre));
    end
  endgenerate //  gPre1

  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pRegLlr
    reg signed [`LDEC_CHK_BITS:0] llrUnityAddV;
    integer i;
    if (nReset == 1'b0) begin
      blkPingPongArray <= {NUM_USERS{1'b0}};
      llrUnity0Out <= `LDEC_PAD(1'b0, `LDEC_CHK_BITS-2);
      llrUnity1Out <= `LDEC_PAD(1'b0, `LDEC_CHK_BITS-2);
    end else begin
      if ((clkEn == 1'b1) || (`LDEC_CLK_GATING == 1)) begin
        for (i=0;i<NUM_USERS;i=i+1) begin
          if (~enabledUsers[i])
            blkPingPongArray[i] <= 1'b0;
        end
        if (firstBlk)
          blkPingPongArray[inUser] <= 1'b0;
          
        // Capture llrUnity at end of transfer.
        if (isLastWord) begin
          llrUnityAddV = $signed({$signed(llrUnity[`LDEC_CHK_BITS-1]), llrUnity});
          if (llrUnityAddV < `LDEC_PADS(1'sb0, `LDEC_CHK_BITS)) begin
            llrUnityAddV = `LDEC_PADS(1'sb0, `LDEC_CHK_BITS);
          end
          if (blkPingPongArray[inUser]) begin
            llrUnity1Out <= llrUnityAddV[`LDEC_CHK_BITS-2:0];
          end else begin
            llrUnity0Out <= llrUnityAddV[`LDEC_CHK_BITS-2:0];
          end        
          blkPingPongArray[inUser] <= ~blkPingPongArray[inUser];
       end
      end
    end
  end //pRegLlr

  generate
    if (! `LDEC_IP_PREPROC) begin: gPre0
      reg [numBits(`LDEC_K_MAX)-1:0] endDataR;
      reg [N_HI:0]                   endParityR;
      
      wire [numBits(1)-1:0] shrtModComp;
      
      assign inStrobePre = inStrobeReg;
      assign inStrobePre_1 = inStrobe;
      assign inDataWordPre = inDataWordReg;
      assign loadBase = 0;
      assign ipStall = 0;      
      // Establish compare values for end of data && end of parity.
      assign shrtModComp = ((blkNumArray[inUser] < shrtMod) && `LDEC_SHRT_SUPPORT) ? 1 : 0;
      
      always @(posedge(clk) `LDEC_RESET_STR)
      begin : pReg
        if (nReset == 1'b0) begin
          endDataR <= 0;
          endParityR <= 0;
        end else begin
          if (clkEn == 1'b1 || `LDEC_CLK_GATING == 1) begin
            if (setRdyToRcv == 1'b1) begin
              endDataR <= k - (nShrtFloor + shrtModComp); // - 1;
              endParityR <= n - (nPuncFloor + puncModComp); // - 1;
            end
          end
        end
      end //pReg
      // External logic is expected to have performed the repeat combining.
      assign numReps  = 0;
      assign numIpBits = n - (nShrtFloor + shrtModComp) - (nPuncFloor + puncModComp);
      assign numIpBitsValid = 1'b1;
      assign endParity = endParityR;
      assign endData = endDataR;
    end
  endgenerate //  gPre0

  //---------------------------------------------------------------------------
  // Bit counting (excludes punctured bits)
  //---------------------------------------------------------------------------
  
  // Determine how many bits will be written this cycle.
  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pBitNum
    reg [N_HI:0]                                            bitNumV;
    reg [numBits(`LDEC_DEC_RAM_IP_WIDTH)-1:0]               samplesAddedV;
    reg                                                     isEndDataV;

    // VHDL version uses a function to set an array, but as long as
    // DEC_RAM_IP_WIDTH divides into Z, all elements are DEC_RAM_IP_WIDTH.
    localparam [numBits(`LDEC_DEC_RAM_IP_WIDTH)-1:0] SAMPLES_ADDED = `LDEC_DEC_RAM_IP_WIDTH;
    
    if (nReset == 1'b0) begin
      isEndParity <= 1'b0;
      isEndData <= 1'b0;
      isLastWord <= 1'b0;
      currentBitNum <= `LDEC_PAD(1'b0, N_HI);
      setRdyToRcvD1 <= 1'b0;
      ipRepeating <= 1'b0;
      inStrobePre_1Last <= 1'b0;
      rowIdx <= `LDEC_PAD(1'b0, NO_HI);
      // if (`LDEC_RESET_ALL) begin (different to VHDL)
      repsRemaining <= `LDEC_PAD(1'b0, REP_HI);
      // end
    end else begin
      if (clkEn == 1'b1 || `LDEC_CLK_GATING == 1) begin
        bitNumV = currentBitNum;
        setRdyToRcvD1 <= setRdyToRcv;
        // defaults for one pulse flags.
        isEndParity <= 1'b0;
        isLastWord <= 1'b0;
        inStrobePre_1Last <= inStrobePre_1;
        
        // Snapshot the number of reps after it has been updated.
        if (setRdyToRcvD1 == 1'b1) begin
          repsRemaining <= numReps;
        end
        // Initialise
        if (setRdyToRcv == 1'b1) begin
          isEndParity <= 1'b0;
          isEndData <= 1'b0;
          bitNumV = `LDEC_PAD(1'b0, N_HI);
          ipRepeating <= 1'b0;        
          isLastWord <= 1'b0;
          rowIdx <= `LDEC_PAD(1'b0, NO_HI);
        end
        else begin
          // Sample counting
          isEndDataV = isEndData;
          isLastWord <= 1'b0;        
          if ((inStrobePre_1 == 1'b1) &&
              // check for previous inStrobePre_1 being properly followed by inStrobePre.
              // This is because inStrobePre_1 can sometimes be held for
              // multiple cycles from end of one block's input to start of next.
              (inStrobePre_1Last == 1'b0 || inStrobePre == 1'b1))
          begin
            // Get number of samples that will be loaded.
            samplesAddedV = SAMPLES_ADDED;
//            if (rowIdx == (`LDEC_PAD(NUM_WR_PER_Z[zEnum], NO_HI-NWZ_HI) -
//            if (`LDEC_PAD(rowIdx, NWZ_HI-NO_HI) == (NUM_WR_PER_Z[zEnum] -
            if (rowIdx == (NUM_WR_PER_Z(zEnum) - `LDEC_PAD(1'b1, NO_HI))) begin
              rowIdx <= `LDEC_PAD(1'b0, NO_HI);
            end
            else begin
              rowIdx <= rowIdx + `LDEC_PAD(1'b1, NO_HI);
            end
            // Accumulate total number of samples. 
            bitNumV = bitNumV + `LDEC_PAD(samplesAddedV, numBits(`LDEC_N_MAX)-numBits(`LDEC_DEC_RAM_IP_WIDTH));
            
            if (bitNumV >= endData) begin
              isEndDataV = 1'b1;
            end
            else begin
              isEndDataV = 1'b0;
            end
            // Handle shortening.
            if (isEndDataV == 1'b1 && isEndData == 1'b0) begin
              // Need to correct samplesAdded to keep repsRemaining correct.
              samplesAddedV = endData - currentBitNum;
              bitNumV = k;
            end
            // Derived flags
            if (bitNumV >= n - `LDEC_PAD(1'b1, N_HI)) begin
              isEndParity <= 1'b0;
            end
            if (bitNumV >= endParity) begin
              isEndParity <= 1'b1;
              isEndDataV = 1'b0;
              // In the event of puncturing we assume that punctured bits have been
              // zeroed by the sender (or ipPre block), so there is no need to
              // correct bitNum || samplesAdded. In repeat mode there can be no
              // puncturing so bitNum && samplesAdded will be correct assuming
              // `LDEC_DEC_RAM_IP_WIDTH divides into z && again samplesAdded
              // need no correction. bitNum will be n-1 but we reset it ready for repeats.
              bitNumV = `LDEC_PAD(1'b0, N_HI);
              // This is the end when there are not repeats.
              if (numReps == `LDEC_PAD(1'b0, REP_HI)) begin
                isLastWord <= 1'b1;
              end
              else begin
                ipRepeating <= 1'b1;
              end
            end
            isEndData <= isEndDataV;
            // Detect end of transfer in the case of repeats
            if (ipRepeating == 1'b1) begin
              if (`LDEC_PAD(samplesAddedV, REP_HI-(numBits(`LDEC_DEC_RAM_IP_WIDTH)-1)) >= repsRemaining) begin
                repsRemaining <= `LDEC_PAD(1'b0, REP_HI);
                isLastWord <= 1'b1;
              end
              else begin
                repsRemaining <= repsRemaining -
                                 `LDEC_PAD(samplesAddedV,
                                    numBits(`LDEC_NREPS_MAX)-numBits(`LDEC_DEC_RAM_IP_WIDTH));
              end
            end
          end
        end
        currentBitNum <= bitNumV;
      end
    end
  end //pBitNum

  //-------------------------------------------------------------------------
  // Update adddress after each write.
  //-------------------------------------------------------------------------
  // Combinatorial version gives read address (for repeat combining)
  // Registered version is write address.
  
  always @(isEndCapture, captureState, inStrobePre, isLastWord,
           isEndData, isEndParity, parityStartCol, repsRemaining, wrOffset,
           numWrPerZ, wrAddr, repeatMode, punctured, capturing, puncturing)
  begin : pAddr
    reg  incrAddrV;
    reg  startingRepeatV;
    
    // Defaults
    startingRepeatV = 1'b0;
    incrAddrV = 1'b0;
    addr = wrAddr;
    offset = wrOffset;
    capturingNxt = capturing;
    captureStateNxt = captureState;
    repeatModeNxt = repeatMode;
    puncturedNxt = punctured;

    // The capture ends with the last input word except when puncturing.
    isEndCaptureNxt = 1'b0;
    if (puncturing == 1'b0) begin
      if (isEndCapture == 1'b0) begin        // make sure it is a pulse
        isEndCaptureNxt = isLastWord;
      end
    end
    // 3 key modes: loading data, parity && setting punctured parity bits
    if (isEndCapture == 1'b1) begin
      capturingNxt = 1'b0;
      puncturedNxt = 1'b0;
    end
    else begin
      
      case (captureState)
        
        dataState : begin
          if (inStrobePre == 1'b1) begin     
            capturingNxt = 1'b1;           // hold clock on from 1st strobe.
            // Detect the end of the data (normal || shortened).
            if (isEndData == 1'b1) begin
              addr = parityStartCol;
              offset = `LDEC_PAD(1'b0, NO_HI);
              captureStateNxt = parityState;
            end
            else begin
              incrAddrV = 1'b1;       
            end
          end
        end
        parityState : begin
          if (inStrobePre == 1'b1) begin         
            incrAddrV = 1'b1;
            // Detect end of Rx.
            if (isEndParity == 1'b1) begin
              if (repsRemaining > `LDEC_PAD(1'b0, REP_HI)) begin                
                captureStateNxt = dataState;
                repeatModeNxt = 1'b1;
                startingRepeatV = 1'b1;
              end
              else begin
                puncturedNxt = puncturing;
                if (puncturing == 1'b1) begin
                  captureStateNxt = puncState;
                end
                else begin
                  captureStateNxt = dataState;                 
                end
              end
            end
          end
        end
        default: begin // puncState
          if ((wrOffset == (numWrPerZ-`LDEC_PAD(1'b1, NWZ_HI))) && (wrAddr == VM_RAM_D-`LDEC_PAD(1'b1, NC_HI))) begin
            isEndCaptureNxt = 1'b1;
            puncturedNxt = 1'b0;
            captureStateNxt = dataState;
          end
          else begin
            incrAddrV = 1'b1;            
          end
        end
      endcase
    end
    // Inc addr || clear down as directed by the state machine above.
    if ((isEndParity == 1'b1) && (repeatMode == 1'b1 || startingRepeatV == 1'b1)) begin
      offset = `LDEC_PAD(1'b0, NO_HI);
      addr = `LDEC_PAD(1'b0, NC_HI);
    end
    else if (incrAddrV == 1'b1) begin
      if (wrOffset == numWrPerZ-`LDEC_PAD(1'b1, NWZ_HI)) begin
        offset = `LDEC_PAD(1'b0, NO_HI);
        if (wrAddr < VM_RAM_D1) begin
          addr = wrAddr + `LDEC_PAD(1'b1, NC_HI);
        end
        else begin
          // Got to the end of the RAM. In puncture mode this is the end of
          // capture.
          if (puncturing == 1'b1) begin
            isEndCaptureNxt = 1'b1;
            puncturedNxt = 1'b0;            
            captureStateNxt = dataState;
          end
        end
      end else begin
        offset = wrOffset + `LDEC_PAD(1'b1, NO_HI);
      end
    end
  end //pAddr

  // Register the addresses and related control signals
  always @(posedge(clk) `LDEC_RESET_STR)
  begin : pAddrReg
    integer i;
    if (nReset == 1'b0) begin
      wrAddr <= `LDEC_PAD(1'b0, NC_HI);
      wrOffset <= `LDEC_PAD(1'b0, NO_HI);
      capturing <= 1'b0;
      captureState <= dataState;
      ipLoaded <= 1'b0;
      punctured <= 1'b0;
      repeatMode <= 1'b0;     
      isEndCapture <= 1'b0;
      for (i=0;i<NUM_USERS;i=i+1)
        blkNumArray[i] <= `LDEC_PAD(1'b0, BN_HI);
    end else begin
      if (clkEn == 1'b1 || `LDEC_CLK_GATING == 1) begin
  
        // Block counter
        ipLoaded <= isEndCapture;
        for (i=0;i<NUM_USERS;i=i+1) begin
          if (enabledUsers[i] == 1'b0) 
            blkNumArray[i] <= `LDEC_PAD(1'b0, BN_HI);
        end
        if (firstBlk == 1'b1) begin
            blkNumArray[inUser] <= `LDEC_PAD(1'b0, BN_HI);
        end
        else if (isEndCapture == 1'b1) begin
          blkNumArray[inUser] <= blkNumArray[inUser] + `LDEC_PAD(1'b1, BN_HI);
        end
        else if (ipSkipBlock == 1'b1) begin
          blkNumArray[inUser] <= blkNumArray[inUser] + `LDEC_PAD(1'b1, BN_HI);
        end        
        wrAddr <= addr;
        wrOffset <= offset;
        captureState <= captureStateNxt;
        capturing <= capturingNxt;
        isEndCapture <= isEndCaptureNxt;
        repeatMode <= repeatModeNxt;
        punctured <= puncturedNxt;
        
        // Address management
        if (setRdyToRcv == 1'b1) begin
          wrAddr <= `LDEC_PAD(1'b0, NC_HI);
          wrOffset <= `LDEC_PAD(1'b0, NO_HI);
          captureState <= dataState;
          repeatMode <= 1'b0;
          punctured <= 1'b0;
          capturing <= 1'b0;
        end
        if (enable == 1'b0) begin
          capturing <= 1'b0;
          captureState <= dataState;
          ipLoaded <= 1'b0;
          punctured <= 1'b0;
          repeatMode <= 1'b0;     
          isEndCapture <= 1'b0;
        end
        
      end
    end
  end //pAddrReg

  // Read address is the unregistered addr (a cycle earlier than wr Addr).
  assign rdAddr = addr;
  // Read offset is the registered offset because it is a mux used on the
  // data after it has been read from the RAM.
  assign rdOffset = wrOffset;

  always @(inDataWordPre)
  begin : pExt
    integer i, j;
    
    dataWordExt = `LDEC_PAD(1'b0, `LDEC_SLICE_LEN-1);
    for (i=0; i<=`LDEC_DEC_RAM_IP_WIDTH-1; i=i+1) begin
      dataWordExt[i*`LDEC_VAR_BITS+:`LDEC_IP_BITS-1] = inDataWordPre[i*`LDEC_IP_BITS+:`LDEC_IP_BITS-1];   
      // Propogate sign bit.
      for (j=0; j<=`LDEC_VAR_BITS-`LDEC_IP_BITS; j=j+1) begin
        dataWordExt[(i+1)*`LDEC_VAR_BITS-1-j] = inDataWordPre[(i+1)*`LDEC_IP_BITS-1];
      end
    end
  end //pExt
  
  //-------------------------------------------------------------------------
  // Repeat Combining (optional)
  //-------------------------------------------------------------------------

  generate    
    if (`LDEC_REP_SUPPORT) begin: gRepT 
      
      // Demux a slice of VM rdData (this still contains multiple samples).
      assign rdDataWord = deMuxA(rdData, rdOffset);
      
      always @(dataWordExt, repeatMode, rdDataWord, punctured)
      begin : pComb
        reg signed [`LDEC_VAR_BITS:0]               comb;
        reg signed [`LDEC_VAR_BITS-1:0]             d;
        reg signed [`LDEC_VAR_BITS-1:0]             r;
        reg signed [`LDEC_VAR_BITS-1:0]             updateV;
        reg [`LDEC_SLICE_LEN-1:0]                   dataWordCombV;
        integer                                     i;
        
        // default is pass through
        dataWordCombV = dataWordExt;
        // For a repeat do an add and clip.
        if (repeatMode == 1'b1) begin
          // Process the `LDEC_DEC_RAM_IP_WIDTH samples in parallel
          for (i=0; i<=`LDEC_DEC_RAM_IP_WIDTH-1; i=i+1) begin
            d = $signed(`LDEC_DEMUX(dataWordExt, i, `LDEC_VAR_BITS));
            r = $signed(`LDEC_DEMUX(rdDataWord, i, `LDEC_VAR_BITS));
            comb = d + r;
            // Map result to the concatenated output word.
            updateV = `LDEC_CLIP_S(comb, `LDEC_VAR_BITS+1, `LDEC_VAR_BITS);
            dataWordCombV[i*`LDEC_VAR_BITS+:`LDEC_VAR_BITS] = $unsigned(updateV);
          end
        end
        dataWordComb = dataWordCombV;
        if (punctured == 1'b1) begin
          dataWordComb = `LDEC_PAD(1'b0, `LDEC_SLICE_LEN-1);
        end
      end //pComb
    end
  endgenerate //  gRepT
       
  generate
    if (! `LDEC_REP_SUPPORT) begin: gRepF
      always @(*) begin
        if (punctured == 1'b0) begin
          dataWordComb = dataWordExt;
        end else begin
          dataWordComb = `LDEC_PAD(1'b0,`LDEC_SLICE_LEN-1);
        end
      end
    end
  endgenerate //  gRepF   

  // When timing is very challenging, register the combined data to save on
  // routing delay, RAM setup time and RAM muxing. See related easing on
  // similar paths at the end of ldpcDecIpPre (gOpReg).
  // Note that this pipeline stage is independent of PIPE A/B/C, we merely
  // use the amount of pipelining as a guide to how challenging timing is.
  generate
    if (`LDEC_PIPE_ABCX >= 3 && `LDEC_REP_SUPPORT) begin: gRepReg1
      always @(posedge(clk) `LDEC_RESET_STR)
      begin : pRepReg
        if (nReset == 1'b0) begin
          wrEnableD1 <= 1'b0;
        end else begin
          wrEnableD1 <= inStrobePre | punctured;
        end
      end
      always @(posedge(clk) `LDEC_RESET_STR_B)
      begin
        if (nReset == 1'b0) begin
          dataWordCombD1 <= `LDEC_PAD(1'b0, `LDEC_SLICE_LEN-1);
          wrAddrD1 <= `LDEC_PAD(1'b0, NC_HI);
          wrOffsetD1 <= `LDEC_PAD(1'b0, NC_HI);
        end else begin
          dataWordCombD1 <= dataWordComb;
          wrAddrD1 <= wrAddr;        
          wrOffsetD1 <= wrOffset;
        end
      end
    end
  endgenerate //  gRepReg1
  generate
    if (`LDEC_PIPE_ABCX < 3 || !`LDEC_REP_SUPPORT) begin: gRepReg0
      always @(*) begin
        dataWordCombD1 = dataWordComb;
        wrAddrD1 = wrAddr;
        wrOffsetD1 = wrOffset;
        wrEnableD1 = inStrobePre | punctured;
      end
    end
  endgenerate //  gRepReg0
 
  assign wrDataOut  = dataWordCombD1;
  assign wrAddrOut  = wrAddrD1;
  assign wrOffsetOut  = wrOffsetD1;
  assign rdAddrOut  = rdAddr;
  assign wrEnableOut  = wrEnableD1;
//  assign rdEnableOut  = inStrobePre & (repeatMode | isEndParity);
  assign rdEnableOut  = (repeatMode | isEndParity);
  assign rdyToRcvOut  = rdyToRcv & ! ipStall;
  assign ipLoadedOut  = ipLoaded;
  assign lastIpSampleOut  = lastIpSample;
  assign lastBlkOfSymOut  = lastBlkOfSym;
  
  assign clkEnOut  = clkEn;
  
  // RW modification: indicate when IP is almost loaded to start the decoding clock with some cycles of margin
  assign ipAlmostLoadedOut = isEndCapture | isEndCaptureNxt | ipLoaded;
  
endmodule
        
